"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeployAssert = void 0;
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const hash_1 = require("./private/hash");
const providers_1 = require("./providers");
const sdk_1 = require("./sdk");
const DEPLOY_ASSERT_SYMBOL = Symbol.for('@aws-cdk/integ-tests.DeployAssert');
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * Construct that allows for registering a list of assertions
 * that should be performed on a construct
 */
class DeployAssert extends core_2.Construct {
    constructor(scope) {
        super(scope, 'DeployAssert');
        Object.defineProperty(this, DEPLOY_ASSERT_SYMBOL, { value: true });
        this._assertions = [];
        const provider = new providers_1.AssertionsProvider(this, 'ResultsProvider');
        const resource = new core_1.CustomResource(this, 'ResultsCollection', {
            serviceToken: provider.serviceToken,
            properties: {
                assertionResults: core_1.Lazy.list({
                    produce: () => this._assertions.map(a => a.result),
                }),
            },
            resourceType: providers_1.RESULTS_RESOURCE_TYPE,
        });
        // TODO: need to show/store this information
        new core_1.CfnOutput(this, 'Results', {
            value: `\n${resource.getAttString('message')}`,
        }).overrideLogicalId('Results');
    }
    /**
     * Returns whether the construct is a DeployAssert construct
     */
    static isDeployAssert(x) {
        return x !== null && typeof (x) === 'object' && DEPLOY_ASSERT_SYMBOL in x;
    }
    /**
     * Finds a DeployAssert construct in the given scope
     */
    static of(construct) {
        const scopes = constructs_1.Node.of(construct).scopes.reverse();
        const deployAssert = scopes.find(s => DeployAssert.isDeployAssert(s));
        if (!deployAssert) {
            throw new Error('No DeployAssert construct found in scopes');
        }
        return deployAssert;
    }
    /**
     * Query AWS using JavaScript SDK V2 API calls
     */
    queryAws(options) {
        const id = hash_1.md5hash(options);
        return new sdk_1.SdkQuery(this, `SdkQuery${id}`, options);
    }
    /**
     * Register an assertion that should be run as part of the
     * deployment
     */
    registerAssertion(assertion) {
        this._assertions.push(assertion);
    }
}
exports.DeployAssert = DeployAssert;
//# sourceMappingURL=data:application/json;base64,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