"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('Add Step with static ClusterId and Step configuration', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: 'ClusterId',
        name: 'StepName',
        jar: 'Jar',
        actionOnFailure: tasks.ActionOnFailure.CONTINUE,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                },
            },
        },
    });
});
test('Terminate cluster with ClusterId from payload and static Step configuration', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: sfn.Data.stringAt('$.ClusterId'),
        name: 'StepName',
        jar: 'Jar',
        actionOnFailure: tasks.ActionOnFailure.CONTINUE,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            'ClusterId.$': '$.ClusterId',
            'Step': {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                },
            },
        },
    });
});
test('Add Step with static ClusterId and Step Name from payload', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: 'ClusterId',
        name: sfn.Data.stringAt('$.StepName'),
        jar: 'Jar',
        actionOnFailure: tasks.ActionOnFailure.CONTINUE,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                'Name.$': '$.StepName',
                'ActionOnFailure': 'CONTINUE',
                'HadoopJarStep': {
                    Jar: 'Jar',
                },
            },
        },
    });
});
test('Add Step with static ClusterId and Step configuration and FIRE_AND_FORGET integrationPattern', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: 'ClusterId',
        name: 'StepName',
        jar: 'Jar',
        actionOnFailure: tasks.ActionOnFailure.CONTINUE,
        integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                },
            },
        },
    });
});
test('Add Step with static ClusterId and Step configuration with TERMINATE_CLUSTER', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: 'ClusterId',
        name: 'StepName',
        jar: 'Jar',
        actionOnFailure: tasks.ActionOnFailure.TERMINATE_CLUSTER,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'TERMINATE_CLUSTER',
                HadoopJarStep: {
                    Jar: 'Jar',
                },
            },
        },
    });
});
test('Add Step with static ClusterId and Step configuration with Args', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: 'ClusterId',
        name: 'StepName',
        jar: 'Jar',
        args: ['Arg1'],
        actionOnFailure: tasks.ActionOnFailure.CONTINUE,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                    Args: ['Arg1'],
                },
            },
        },
    });
});
test('Add Step with static ClusterId and Step configuration with Properties', () => {
    // WHEN
    const task = new tasks.EmrAddStep(stack, 'Task', {
        clusterId: 'ClusterId',
        name: 'StepName',
        jar: 'Jar',
        properties: {
            PropertyKey: 'PropertyValue',
        },
        actionOnFailure: tasks.ActionOnFailure.CONTINUE,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::elasticmapreduce:addStep.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ClusterId: 'ClusterId',
            Step: {
                Name: 'StepName',
                ActionOnFailure: 'CONTINUE',
                HadoopJarStep: {
                    Jar: 'Jar',
                    Properties: [{
                            Key: 'PropertyKey',
                            Value: 'PropertyValue',
                        }],
                },
            },
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new tasks.EmrAddStep(stack, 'Task', {
            clusterId: 'ClusterId',
            name: 'StepName',
            jar: 'Jar',
            actionOnFailure: tasks.ActionOnFailure.CONTINUE,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
//# sourceMappingURL=data:application/json;base64,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