"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ddb = require("@aws-cdk/aws-dynamodb");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
/**
 *
 * Stack verification steps:
 * * aws stepfunctions start-execution --state-machine-arn <deployed state machine arn> : should return execution arn
 * *
 * * aws stepfunctions describe-execution --execution-arn <execution-arn generated before> --query 'status': should return status as SUCCEEDED
 * * aws stepfunctions describe-execution --execution-arn <execution-arn generated before> --query 'output': should return the number 42
 */
class CallDynamoDBStack extends cdk.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        const TABLE_NAME = 'Messages';
        const MESSAGE_ID = '1234';
        const firstNumber = 18;
        const secondNumber = 24;
        const table = new ddb.Table(this, 'Messages', {
            tableName: TABLE_NAME,
            partitionKey: {
                name: 'MessageId',
                type: ddb.AttributeType.STRING,
            },
            readCapacity: 10,
            writeCapacity: 5,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const putItemTask = new tasks.DynamoPutItem(this, 'PutItem', {
            item: {
                MessageId: tasks.DynamoAttributeValue.fromString(MESSAGE_ID),
                Text: tasks.DynamoAttributeValue.fromString(sfn.Data.stringAt('$.bar')),
                TotalCount: tasks.DynamoAttributeValue.fromNumber(firstNumber),
            },
            table,
        });
        const getItemTaskAfterPut = new tasks.DynamoGetItem(this, 'GetItemAfterPut', {
            key: { MessageId: tasks.DynamoAttributeValue.fromString(MESSAGE_ID) },
            table,
        });
        const updateItemTask = new tasks.DynamoUpdateItem(this, 'UpdateItem', {
            key: { MessageId: tasks.DynamoAttributeValue.fromString(MESSAGE_ID) },
            table,
            expressionAttributeValues: {
                ':val': tasks.DynamoAttributeValue.numberFromString(sfn.Data.stringAt('$.Item.TotalCount.N')),
                ':rand': tasks.DynamoAttributeValue.fromNumber(secondNumber),
            },
            updateExpression: 'SET TotalCount = :val + :rand',
        });
        const getItemTaskAfterUpdate = new tasks.DynamoGetItem(this, 'GetItemAfterUpdate', {
            key: { MessageId: tasks.DynamoAttributeValue.fromString(MESSAGE_ID) },
            table,
            outputPath: sfn.Data.stringAt('$.Item.TotalCount.N'),
        });
        const deleteItemTask = new tasks.DynamoDeleteItem(this, 'DeleteItem', {
            key: { MessageId: tasks.DynamoAttributeValue.fromString(MESSAGE_ID) },
            table,
            resultPath: 'DISCARD',
        });
        const definition = new sfn.Pass(this, 'Start', {
            result: sfn.Result.fromObject({ bar: 'SomeValue' }),
        })
            .next(putItemTask)
            .next(getItemTaskAfterPut)
            .next(updateItemTask)
            .next(getItemTaskAfterUpdate)
            .next(deleteItemTask);
        const stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition,
        });
        new cdk.CfnOutput(this, 'StateMachineArn', {
            value: stateMachine.stateMachineArn,
        });
    }
}
const app = new cdk.App();
new CallDynamoDBStack(app, 'aws-stepfunctions-integ');
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcuY2FsbC1keW5hbW9kYi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImludGVnLmNhbGwtZHluYW1vZGIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw2Q0FBNkM7QUFDN0Msa0RBQWtEO0FBQ2xELHFDQUFxQztBQUNyQyxtQ0FBbUM7QUFFbkM7Ozs7Ozs7R0FPRztBQUNILE1BQU0saUJBQWtCLFNBQVEsR0FBRyxDQUFDLEtBQUs7SUFDdkMsWUFBWSxLQUFjLEVBQUUsRUFBVSxFQUFFLFFBQXdCLEVBQUU7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEIsTUFBTSxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzlCLE1BQU0sVUFBVSxHQUFHLE1BQU0sQ0FBQztRQUMxQixNQUFNLFdBQVcsR0FBRyxFQUFFLENBQUM7UUFDdkIsTUFBTSxZQUFZLEdBQUcsRUFBRSxDQUFDO1FBRXhCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzVDLFNBQVMsRUFBRSxVQUFVO1lBQ3JCLFlBQVksRUFBRTtnQkFDWixJQUFJLEVBQUUsV0FBVztnQkFDakIsSUFBSSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTTthQUMvQjtZQUNELFlBQVksRUFBRSxFQUFFO1lBQ2hCLGFBQWEsRUFBRSxDQUFDO1lBQ2hCLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDekMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxXQUFXLEdBQUcsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDM0QsSUFBSSxFQUFFO2dCQUNKLFNBQVMsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQztnQkFDNUQsSUFBSSxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBQ3ZFLFVBQVUsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQzthQUMvRDtZQUNELEtBQUs7U0FDTixDQUFDLENBQUM7UUFFSCxNQUFNLG1CQUFtQixHQUFHLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDM0UsR0FBRyxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDckUsS0FBSztTQUNOLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDcEUsR0FBRyxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDckUsS0FBSztZQUNMLHlCQUF5QixFQUFFO2dCQUN6QixNQUFNLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLHFCQUFxQixDQUFDLENBQUM7Z0JBQzdGLE9BQU8sRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQzthQUM3RDtZQUNELGdCQUFnQixFQUFFLCtCQUErQjtTQUNsRCxDQUFDLENBQUM7UUFFSCxNQUFNLHNCQUFzQixHQUFHLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDakYsR0FBRyxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDckUsS0FBSztZQUNMLFVBQVUsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztTQUNyRCxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ3BFLEdBQUcsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ3JFLEtBQUs7WUFDTCxVQUFVLEVBQUUsU0FBUztTQUN0QixDQUFDLENBQUM7UUFFSCxNQUFNLFVBQVUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRTtZQUM3QyxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsRUFBRSxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUM7U0FDcEQsQ0FBQzthQUNDLElBQUksQ0FBQyxXQUFXLENBQUM7YUFDakIsSUFBSSxDQUFDLG1CQUFtQixDQUFDO2FBQ3pCLElBQUksQ0FBQyxjQUFjLENBQUM7YUFDcEIsSUFBSSxDQUFDLHNCQUFzQixDQUFDO2FBQzVCLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUV4QixNQUFNLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUM5RCxVQUFVO1NBQ1gsQ0FBQyxDQUFDO1FBRUgsSUFBSSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUN6QyxLQUFLLEVBQUUsWUFBWSxDQUFDLGVBQWU7U0FDcEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBRUQsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxFQUFFLENBQUM7QUFDMUIsSUFBSSxpQkFBaUIsQ0FBQyxHQUFHLEVBQUUseUJBQXlCLENBQUMsQ0FBQztBQUN0RCxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBkZGIgZnJvbSAnQGF3cy1jZGsvYXdzLWR5bmFtb2RiJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyB0YXNrcyBmcm9tICcuLi8uLi9saWInO1xuXG4vKipcbiAqXG4gKiBTdGFjayB2ZXJpZmljYXRpb24gc3RlcHM6XG4gKiAqIGF3cyBzdGVwZnVuY3Rpb25zIHN0YXJ0LWV4ZWN1dGlvbiAtLXN0YXRlLW1hY2hpbmUtYXJuIDxkZXBsb3llZCBzdGF0ZSBtYWNoaW5lIGFybj4gOiBzaG91bGQgcmV0dXJuIGV4ZWN1dGlvbiBhcm5cbiAqICpcbiAqICogYXdzIHN0ZXBmdW5jdGlvbnMgZGVzY3JpYmUtZXhlY3V0aW9uIC0tZXhlY3V0aW9uLWFybiA8ZXhlY3V0aW9uLWFybiBnZW5lcmF0ZWQgYmVmb3JlPiAtLXF1ZXJ5ICdzdGF0dXMnOiBzaG91bGQgcmV0dXJuIHN0YXR1cyBhcyBTVUNDRUVERURcbiAqICogYXdzIHN0ZXBmdW5jdGlvbnMgZGVzY3JpYmUtZXhlY3V0aW9uIC0tZXhlY3V0aW9uLWFybiA8ZXhlY3V0aW9uLWFybiBnZW5lcmF0ZWQgYmVmb3JlPiAtLXF1ZXJ5ICdvdXRwdXQnOiBzaG91bGQgcmV0dXJuIHRoZSBudW1iZXIgNDJcbiAqL1xuY2xhc3MgQ2FsbER5bmFtb0RCU3RhY2sgZXh0ZW5kcyBjZGsuU3RhY2sge1xuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkFwcCwgaWQ6IHN0cmluZywgcHJvcHM6IGNkay5TdGFja1Byb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIGNvbnN0IFRBQkxFX05BTUUgPSAnTWVzc2FnZXMnO1xuICAgIGNvbnN0IE1FU1NBR0VfSUQgPSAnMTIzNCc7XG4gICAgY29uc3QgZmlyc3ROdW1iZXIgPSAxODtcbiAgICBjb25zdCBzZWNvbmROdW1iZXIgPSAyNDtcblxuICAgIGNvbnN0IHRhYmxlID0gbmV3IGRkYi5UYWJsZSh0aGlzLCAnTWVzc2FnZXMnLCB7XG4gICAgICB0YWJsZU5hbWU6IFRBQkxFX05BTUUsXG4gICAgICBwYXJ0aXRpb25LZXk6IHtcbiAgICAgICAgbmFtZTogJ01lc3NhZ2VJZCcsXG4gICAgICAgIHR5cGU6IGRkYi5BdHRyaWJ1dGVUeXBlLlNUUklORyxcbiAgICAgIH0sXG4gICAgICByZWFkQ2FwYWNpdHk6IDEwLFxuICAgICAgd3JpdGVDYXBhY2l0eTogNSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSk7XG5cbiAgICBjb25zdCBwdXRJdGVtVGFzayA9IG5ldyB0YXNrcy5EeW5hbW9QdXRJdGVtKHRoaXMsICdQdXRJdGVtJywge1xuICAgICAgaXRlbToge1xuICAgICAgICBNZXNzYWdlSWQ6IHRhc2tzLkR5bmFtb0F0dHJpYnV0ZVZhbHVlLmZyb21TdHJpbmcoTUVTU0FHRV9JRCksXG4gICAgICAgIFRleHQ6IHRhc2tzLkR5bmFtb0F0dHJpYnV0ZVZhbHVlLmZyb21TdHJpbmcoc2ZuLkRhdGEuc3RyaW5nQXQoJyQuYmFyJykpLFxuICAgICAgICBUb3RhbENvdW50OiB0YXNrcy5EeW5hbW9BdHRyaWJ1dGVWYWx1ZS5mcm9tTnVtYmVyKGZpcnN0TnVtYmVyKSxcbiAgICAgIH0sXG4gICAgICB0YWJsZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGdldEl0ZW1UYXNrQWZ0ZXJQdXQgPSBuZXcgdGFza3MuRHluYW1vR2V0SXRlbSh0aGlzLCAnR2V0SXRlbUFmdGVyUHV0Jywge1xuICAgICAga2V5OiB7IE1lc3NhZ2VJZDogdGFza3MuRHluYW1vQXR0cmlidXRlVmFsdWUuZnJvbVN0cmluZyhNRVNTQUdFX0lEKSB9LFxuICAgICAgdGFibGUsXG4gICAgfSk7XG5cbiAgICBjb25zdCB1cGRhdGVJdGVtVGFzayA9IG5ldyB0YXNrcy5EeW5hbW9VcGRhdGVJdGVtKHRoaXMsICdVcGRhdGVJdGVtJywge1xuICAgICAga2V5OiB7IE1lc3NhZ2VJZDogdGFza3MuRHluYW1vQXR0cmlidXRlVmFsdWUuZnJvbVN0cmluZyhNRVNTQUdFX0lEKSB9LFxuICAgICAgdGFibGUsXG4gICAgICBleHByZXNzaW9uQXR0cmlidXRlVmFsdWVzOiB7XG4gICAgICAgICc6dmFsJzogdGFza3MuRHluYW1vQXR0cmlidXRlVmFsdWUubnVtYmVyRnJvbVN0cmluZyhzZm4uRGF0YS5zdHJpbmdBdCgnJC5JdGVtLlRvdGFsQ291bnQuTicpKSxcbiAgICAgICAgJzpyYW5kJzogdGFza3MuRHluYW1vQXR0cmlidXRlVmFsdWUuZnJvbU51bWJlcihzZWNvbmROdW1iZXIpLFxuICAgICAgfSxcbiAgICAgIHVwZGF0ZUV4cHJlc3Npb246ICdTRVQgVG90YWxDb3VudCA9IDp2YWwgKyA6cmFuZCcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBnZXRJdGVtVGFza0FmdGVyVXBkYXRlID0gbmV3IHRhc2tzLkR5bmFtb0dldEl0ZW0odGhpcywgJ0dldEl0ZW1BZnRlclVwZGF0ZScsIHtcbiAgICAgIGtleTogeyBNZXNzYWdlSWQ6IHRhc2tzLkR5bmFtb0F0dHJpYnV0ZVZhbHVlLmZyb21TdHJpbmcoTUVTU0FHRV9JRCkgfSxcbiAgICAgIHRhYmxlLFxuICAgICAgb3V0cHV0UGF0aDogc2ZuLkRhdGEuc3RyaW5nQXQoJyQuSXRlbS5Ub3RhbENvdW50Lk4nKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGRlbGV0ZUl0ZW1UYXNrID0gbmV3IHRhc2tzLkR5bmFtb0RlbGV0ZUl0ZW0odGhpcywgJ0RlbGV0ZUl0ZW0nLCB7XG4gICAgICBrZXk6IHsgTWVzc2FnZUlkOiB0YXNrcy5EeW5hbW9BdHRyaWJ1dGVWYWx1ZS5mcm9tU3RyaW5nKE1FU1NBR0VfSUQpIH0sXG4gICAgICB0YWJsZSxcbiAgICAgIHJlc3VsdFBhdGg6ICdESVNDQVJEJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IGRlZmluaXRpb24gPSBuZXcgc2ZuLlBhc3ModGhpcywgJ1N0YXJ0Jywge1xuICAgICAgcmVzdWx0OiBzZm4uUmVzdWx0LmZyb21PYmplY3QoeyBiYXI6ICdTb21lVmFsdWUnIH0pLFxuICAgIH0pXG4gICAgICAubmV4dChwdXRJdGVtVGFzaylcbiAgICAgIC5uZXh0KGdldEl0ZW1UYXNrQWZ0ZXJQdXQpXG4gICAgICAubmV4dCh1cGRhdGVJdGVtVGFzaylcbiAgICAgIC5uZXh0KGdldEl0ZW1UYXNrQWZ0ZXJVcGRhdGUpXG4gICAgICAubmV4dChkZWxldGVJdGVtVGFzayk7XG5cbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgc2ZuLlN0YXRlTWFjaGluZSh0aGlzLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbixcbiAgICB9KTtcblxuICAgIG5ldyBjZGsuQ2ZuT3V0cHV0KHRoaXMsICdTdGF0ZU1hY2hpbmVBcm4nLCB7XG4gICAgICB2YWx1ZTogc3RhdGVNYWNoaW5lLnN0YXRlTWFjaGluZUFybixcbiAgICB9KTtcbiAgfVxufVxuXG5jb25zdCBhcHAgPSBuZXcgY2RrLkFwcCgpO1xubmV3IENhbGxEeW5hbW9EQlN0YWNrKGFwcCwgJ2F3cy1zdGVwZnVuY3Rpb25zLWludGVnJyk7XG5hcHAuc3ludGgoKTtcbiJdfQ==