"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssembleWith = exports.SplitType = exports.BatchStrategy = exports.CompressionType = exports.InputMode = exports.RecordWrapperType = exports.S3DataDistributionType = exports.S3DataType = exports.DockerImage = exports.S3Location = void 0;
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.Data.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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