"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InvocationType = exports.RunLambdaTask = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @deprecated Use `LambdaInvoke`
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('lambda', 'invoke', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            },
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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