"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueStartJobRun = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts an AWS Glue job in a Task state
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class GlueStartJobRun extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.getPolicies();
        this.taskMetrics = {
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.props.glueJobName },
        };
    }
    renderTask() {
        var _a, _b;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        return {
            Resource: task_utils_1.integrationResourceArn('glue', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobName: this.props.glueJobName,
                Arguments: (_a = this.props.arguments) === null || _a === void 0 ? void 0 : _a.value,
                Timeout: (_b = this.props.timeout) === null || _b === void 0 ? void 0 : _b.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty,
            }),
            TimeoutSeconds: undefined,
        };
    }
    getPolicies() {
        let iamActions;
        if (this.integrationPattern === sfn.IntegrationPattern.REQUEST_RESPONSE) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun',
            ];
        }
        return [new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'glue',
                        resource: 'job',
                        resourceName: this.props.glueJobName,
                    }),
                ],
                actions: iamActions,
            })];
    }
}
exports.GlueStartJobRun = GlueStartJobRun;
GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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