"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrTerminateCluster = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * A Step Functions Task to terminate an EMR Cluster.
 *
 * @experimental
 */
class EmrTerminateCluster extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrTerminateCluster.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'terminateCluster', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call TerminateCluster.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:DescribeCluster',
                    'elasticmapreduce:TerminateJobFlows',
                ],
                resources: [`arn:aws:elasticmapreduce:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:cluster/*`],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRTerminateJobFlowsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrTerminateCluster = EmrTerminateCluster;
EmrTerminateCluster.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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