"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrAddStep = exports.ActionOnFailure = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The action to take when the cluster step fails.
 * @see https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
 *
 * Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
 *
 * @default CONTINUE
 *
 * @experimental
 */
var ActionOnFailure;
(function (ActionOnFailure) {
    /**
     * Terminate the Cluster on Step Failure
     */
    ActionOnFailure["TERMINATE_CLUSTER"] = "TERMINATE_CLUSTER";
    /**
     * Cancel Step execution and enter WAITING state
     */
    ActionOnFailure["CANCEL_AND_WAIT"] = "CANCEL_AND_WAIT";
    /**
     * Continue to the next Step
     */
    ActionOnFailure["CONTINUE"] = "CONTINUE";
})(ActionOnFailure = exports.ActionOnFailure || (exports.ActionOnFailure = {}));
/**
 * A Step Functions Task to add a Step to an EMR Cluster
 *
 * The StepConfiguration is defined as Parameters in the state machine definition.
 *
 * OUTPUT: the StepId
 *
 * @experimental
 */
class EmrAddStep extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.actionOnFailure = (_a = props.actionOnFailure) !== null && _a !== void 0 ? _a : ActionOnFailure.CONTINUE;
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.createPolicyStatements();
    }
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('elasticmapreduce', 'addStep', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                ClusterId: this.props.clusterId,
                Step: {
                    Name: this.props.name,
                    ActionOnFailure: this.actionOnFailure.valueOf(),
                    HadoopJarStep: {
                        Jar: this.props.jar,
                        MainClass: this.props.mainClass,
                        Args: this.props.args,
                        Properties: (this.props.properties === undefined) ?
                            undefined :
                            Object.entries(this.props.properties).map(kv => ({
                                Key: kv[0],
                                Value: kv[1],
                            })),
                    },
                },
            }),
        };
    }
    /**
     * This generates the PolicyStatements required by the Task to call AddStep.
     */
    createPolicyStatements() {
        const stack = core_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: [
                    'elasticmapreduce:AddJobFlowSteps',
                    'elasticmapreduce:DescribeStep',
                    'elasticmapreduce:CancelSteps',
                ],
                resources: [`arn:aws:elasticmapreduce:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:cluster/*`],
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventForEMRAddJobFlowStepsRule',
                    })],
            }));
        }
        return policyStatements;
    }
}
exports.EmrAddStep = EmrAddStep;
EmrAddStep.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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