"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamoAttributeValue = exports.DynamoProjectionExpression = exports.DynamoReturnValues = exports.DynamoItemCollectionMetrics = exports.DynamoConsumedCapacity = void 0;
const utils_1 = require("./private/utils");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * Represents the data for an attribute.
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 */
class DynamoAttributeValue {
    constructor(value) {
        this.attributeValue = value;
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath
     */
    static fromString(value) {
        return new DynamoAttributeValue({ S: value });
    }
    /**
     * Sets a literal number. For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumber(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberFromString(value) {
        return new DynamoAttributeValue({ N: value.toString() });
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string
     */
    static fromBinary(value) {
        return new DynamoAttributeValue({ B: value });
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    static fromStringSet(value) {
        return new DynamoAttributeValue({ SS: value });
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumberSet(value) {
        return new DynamoAttributeValue({ NS: value.map(String) });
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberSetFromStrings(value) {
        return new DynamoAttributeValue({ NS: value });
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    static fromBinarySet(value) {
        return new DynamoAttributeValue({ BS: value });
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    static fromMap(value) {
        return new DynamoAttributeValue({ M: utils_1.transformAttributeValueMap(value) });
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used
     */
    static mapFromJsonPath(value) {
        if (!value.startsWith('$')) {
            throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
        }
        return new DynamoAttributeValue({ 'M.$': value });
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    static fromList(value) {
        return new DynamoAttributeValue({ L: value.map((val) => val.toObject()) });
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    static fromNull(value) {
        return new DynamoAttributeValue({ NULL: value });
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    static fromBoolean(value) {
        return new DynamoAttributeValue({ BOOL: value });
    }
    /**
     * Returns the DynamoDB attribute value
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
//# sourceMappingURL=data:application/json;base64,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