"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchSubmitJob = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Task to submits an AWS Batch job from a job definition.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
 */
class BatchSubmitJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.RUN_JOB;
        task_utils_1.validatePatternSupported(this.integrationPattern, BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS);
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`attempt duration must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is required since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_b = props.containerOverrides) === null || _b === void 0 ? void 0 : _b.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
        this.taskPolicies = this.configurePolicyStatements();
    }
    renderTask() {
        var _a;
        return {
            Resource: task_utils_1.integrationResourceArn('batch', 'submitJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: (_a = this.props.payload) === null || _a === void 0 ? void 0 : _a.value,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            }),
            TimeoutSeconds: undefined,
        };
    }
    configurePolicyStatements() {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a, _b;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: (_b = containerOverrides.memory) === null || _b === void 0 ? void 0 : _b.toMebibytes(),
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.BatchSubmitJob = BatchSubmitJob;
BatchSubmitJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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