from abc import ABC, abstractmethod
from datetime import datetime
from typing import TYPE_CHECKING, ClassVar, Iterator, NamedTuple, Optional, Tuple, Type

if TYPE_CHECKING:
    from dql.data_storage import AbstractDataStorage


class Bucket(NamedTuple):
    name: str
    uri: str
    created: Optional[datetime]


class Client(ABC):
    name: str
    protocol: ClassVar[str]

    @staticmethod
    def get_implementation(url: str) -> Type["Client"]:
        from .azure import AzureClient
        from .gcs import GCSClient
        from .local import FileClient
        from .s3 import ClientS3

        if url.startswith(ClientS3.PREFIX):
            return ClientS3
        elif url.startswith(GCSClient.PREFIX):
            return GCSClient
        elif url.startswith(AzureClient.PREFIX):
            return AzureClient
        elif url.startswith(FileClient.PREFIX):
            return FileClient
        raise RuntimeError(f"Unsupported data source format '{url}'")

    @staticmethod
    def parse_url(
        source: str, data_storage: "AbstractDataStorage", **kwargs
    ) -> Tuple["Client", str]:
        cls = Client.get_implementation(source)
        storage, path = cls.split_url(source, data_storage)
        client = cls.from_url(storage, **kwargs)
        return client, path

    @classmethod
    @abstractmethod
    def from_url(cls, url: str, **kwargs) -> "Client":
        ...

    @classmethod
    @abstractmethod
    def is_root_url(cls, url) -> bool:
        ...

    @classmethod
    @abstractmethod
    def split_url(
        cls, url: str, data_storage: "AbstractDataStorage"
    ) -> Tuple[str, str]:
        ...

    @classmethod
    @abstractmethod
    def ls_buckets(cls, **kwargs) -> Iterator[Bucket]:
        ...

    @property
    @abstractmethod
    def uri(self) -> str:
        ...
