import os
import re
from setuptools import setup

# Path to the schema file, relative to this setup.py file
# setup.py is in packages/duc-py/
# schema/duc.fbs is at the workspace root, so ../../schema/duc.fbs
SCHEMA_FILE_PATH = os.path.join(os.path.dirname(os.path.abspath(__file__)), '..', '..', 'schema', 'duc.fbs')
# Path where _version.py will be written
VERSION_PY_PATH = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'src', 'ducpy', '_version.py')

def get_schema_version_from_fbs(fbs_file_path):
    try:
        with open(fbs_file_path, 'r') as f:
            first_line = f.readline()
            match = re.search(r'//\s*SCHEMA_VERSION\s*=\s*(\d+)', first_line)
            if match:
                return int(match.group(1))
    except FileNotFoundError:
        # This is expected when building from sdist where schema file isn't present
        return None # Return None to indicate schema file was not found
    except Exception as e:
        print(f"Warning: Could not parse schema version from {fbs_file_path}: {e}. Defaulting to 0.")
    return 0 # Default version if parsing fails but file was found

def generate_version_py():
    """Generates the _version.py file with DUC_SCHEMA_VERSION."""
    schema_version_from_fbs = get_schema_version_from_fbs(SCHEMA_FILE_PATH)

    if schema_version_from_fbs is not None:
        # Schema file was found (development or first-pass build)
        # Or if it was found but parsing failed (defaulted to 0 by get_schema_version_from_fbs)
        # always regenerate _version.py
        print(f"Generating DUC schema version file at: {VERSION_PY_PATH} with version: {schema_version_from_fbs}")
        os.makedirs(os.path.dirname(VERSION_PY_PATH), exist_ok=True)
        with open(VERSION_PY_PATH, 'w') as f:
            f.write(f'# This file is auto-generated by setup.py when the package is built.\n')
            f.write(f'# Do not edit this file manually.\n')
            f.write(f'DUC_SCHEMA_VERSION = {schema_version_from_fbs}\n')
    elif os.path.exists(VERSION_PY_PATH):
        # Schema file not found (likely sdist build) AND _version.py already exists.
        # Trust the existing _version.py from the sdist.
        print(f"DUC schema file not found. Using existing _version.py: {VERSION_PY_PATH}")
    else:
        # Schema file not found AND _version.py does not exist (edge case, should not happen with sdist).
        # Fallback to generating with default 0 to ensure the file exists for import.
        print(f"DUC schema file not found and _version.py does not exist. Generating with default version 0 at: {VERSION_PY_PATH}")
        os.makedirs(os.path.dirname(VERSION_PY_PATH), exist_ok=True)
        with open(VERSION_PY_PATH, 'w') as f:
            f.write(f'# This file is auto-generated by setup.py (fallback).\n')
            f.write(f'DUC_SCHEMA_VERSION = 0\n')

# --- Main part of setup.py --- 

# Generate the version file when setup.py is processed by the build system.
generate_version_py()

# Minimal setup call.
# Most package metadata (name, version via setuptools-scm, etc.) and 
# package discovery are handled by pyproject.toml.
# This setup.py is present for compatibility and can be used for tasks 
# like C-extension building or build-time file generation if needed.
setup() 