import io
import matplotlib
# Force headless backend before importing pyplot
matplotlib.use('Agg')
import matplotlib.pyplot as plt
import numpy as np
import librosa
import librosa.display
from typing import Optional, Tuple

# Import from our local modules
from .config import SpectrogramConfig, WaveformConfig
from .item import AudioItem

class Visualizer:
    """
    Stateless engine for generating plotting buffers from AudioItems.
    """

    @staticmethod
    def _create_figure(dimensions: Tuple[int, int], dpi: int = 100):
        """
        Creates a figure with precise pixel dimensions and no margins.
        """
        w_px, h_px = dimensions
        # Convert pixels to inches
        fig = plt.figure(figsize=(w_px / dpi, h_px / dpi), dpi=dpi)
        # [left, bottom, width, height] in fractions of figure width/height
        # 0,0,1,1 fills the whole image, removing all white borders/axes labels space
        ax = fig.add_axes([0, 0, 1, 1])
        return fig, ax

    @classmethod
    def generate_spectrogram(
        cls, 
        item: AudioItem, 
        config: SpectrogramConfig
    ) -> Optional[io.BytesIO]:
        """
        Generates a spectrogram image buffer.
        Returns None if audio is empty/silent.
        """
        if item.is_empty or item.is_silent:
            return None

        fig = None
        try:
            fig, ax = cls._create_figure(config.dimensions)
            
            # 1. Compute STFT
            # Pad audio if shorter than n_fft to prevent librosa errors
            y = item.audio
            if len(y) < config.n_fft:
                y = np.pad(y, (0, int(config.n_fft - len(y))))

            D = librosa.stft(
                y, 
                n_fft=config.n_fft, 
                hop_length=config.hop_length,
                win_length=config.win_length,
                window=config.window
            )
            
            # 2. Convert to dB
            S_db = librosa.amplitude_to_db(np.abs(D), ref=np.max)
            
            # 3. Plot
            img = librosa.display.specshow(
                S_db,
                sr=item.sr,
                hop_length=config.hop_length,
                x_axis='time',
                y_axis=config.plot_freq_scale, # 'log' or 'linear'
                cmap=config.cmap,
                vmin=config.vmin_db,
                vmax=config.vmax_db,
                ax=ax
            )

            # 4. Axis visibility
            if not config.show_axis:
                ax.axis('off')
            else:
                # If axes are shown, we need to adjust the "add_axes" to make room,
                # but for this specific implementation of "filling the cell",
                # we typically want axis OFF. If user requests ON, we revert to standard layout.
                # Re-clearing the specific axes setup for a standard one:
                fig.clf()
                ax = fig.add_subplot(111)
                librosa.display.specshow(
                    S_db, sr=item.sr, hop_length=config.hop_length,
                    x_axis='time', y_axis=config.plot_freq_scale,
                    cmap=config.cmap, vmin=config.vmin_db, vmax=config.vmax_db, ax=ax
                )
                plt.tight_layout()

            # 5. Save to buffer
            buf = io.BytesIO()
            plt.savefig(buf, format='png', dpi=100)
            buf.seek(0)
            return buf

        except Exception as e:
            print(f"Error generating spectrogram: {e}")
            return None
        finally:
            if fig:
                plt.close(fig)

    @classmethod
    def generate_waveform(
        cls, 
        item: AudioItem, 
        config: WaveformConfig,
        gt_item: Optional[AudioItem] = None
    ) -> Optional[io.BytesIO]:
        """
        Generates a waveform image buffer.
        Supports overlaying a Ground Truth (GT) signal behind the generated signal.
        """
        if item.is_empty and (gt_item is None or gt_item.is_empty):
            return None

        fig = None
        try:
            fig, ax = cls._create_figure(config.dimensions)

            # Helper to plot a single array
            def plot_signal(audio_data, color, linewidth):
                # Downsample for visualization speed if array is massive (>100k samples)
                # simple stride slicing
                if len(audio_data) > 100000:
                    step = len(audio_data) // 5000  # target ~5000 points
                    plot_data = audio_data[::step]
                else:
                    plot_data = audio_data
                
                ax.plot(plot_data, color=color, linewidth=linewidth)

            # 1. Plot Ground Truth (Background)
            if config.overlay_gt and gt_item is not None and not gt_item.is_empty:
                plot_signal(gt_item.audio, config.color_gt, config.linewidth)

            # 2. Plot Generated (Foreground)
            if not item.is_empty and not item.is_silent:
                plot_signal(item.audio, config.color_gen, config.linewidth)

            # 3. Styling
            ax.set_ylim(config.ylim)
            ax.set_xlim(0, max(len(item.audio) if not item.is_empty else 0, 
                               len(gt_item.audio) if gt_item else 0))
            ax.axis('off') # Waveforms are strictly aesthetic in the table view
            
            # 4. Save
            buf = io.BytesIO()
            plt.savefig(buf, format='png', dpi=100, transparent=True)
            buf.seek(0)
            return buf

        except Exception as e:
            print(f"Error generating waveform: {e}")
            return None
        finally:
            if fig:
                plt.close(fig)