package org.funz.script;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.funz.conf.Configuration;
import org.funz.log.Log;

/**
 *
 * @author richet
 */
public abstract class MathExpression {

    public String getLastMessage() {
        return "?";
    }

    public static class MathException extends Exception {

        public MathException(String what) {
            super(what);
        }
    }
    private static MathExpression defaultInstance;
    protected String name;
    List<String> globalVariables = new LinkedList<String>();

    static {
        //Automated instanciation of class MathExpression using -DMathExpression.class=org.comp.math.myengine setting
        if (System.getProperty("MathExpression.class") != null) {
            try {                
                System.out.println("Class " + System.getProperty("MathExpression.class") + " will be used as default MathExpression.");
                SetDefaultInstance((Class) Class.forName(System.getProperty("MathExpression.class"), true, MathExpression.class.getClassLoader()));
            } catch (ClassNotFoundException c) {
                System.err.println("Class " + System.getProperty("MathExpression.class") + " not found. No MathExpression class set.");
            }
        }
    }

    static List<MathExpression> all = Collections.synchronizedList(new ArrayList());
    
    public MathExpression(String name) {
        this.name = name;
        all.add(this);
    }

    @Override
    public void finalize() throws Throwable {
        all.remove(this);
        super.finalize();
    }

    public static synchronized void End() {
        while (all.size()>0) {
            if (all.get(0)!=null)
                try {
                    all.get(0).finalize();
                } catch (Throwable e) {
                    if (Log.level>=10) e.printStackTrace();
                }
        }
    }

    public String getName() {
        return name;
    }

    public static void SetDefaultInstance(Class MathExpressionClass) {
        if (defaultInstance != null && MathExpressionClass.isInstance(defaultInstance)) {
            Log.out("Already used MathExpression.defaultInstance of type " + MathExpressionClass, 3);
            try {
                defaultInstance.reset();
                return;
            } catch (MathException ex) {
                Log.err("Could not reset MathExpression. Creating new MathExpression.defaultInstance", 3);
            }
        }
        SetDefaultInstance(NewInstance(MathExpressionClass, "Default_" + Configuration.timeDigest()));
    }

    public static void SetDefaultInstance(MathExpression instance) {
        defaultInstance = instance;
    }

    /**
     * @return the main instance
     */
    public static MathExpression GetDefaultInstance() {
        return defaultInstance;
    }

    /**
     * @return the instance
     */
    public static MathExpression NewInstance(Class MathExpressionClass, String name) {
        MathExpression instance = null;
        if (MathExpressionClass == null) {
            System.err.println("No MathExpression Class given.");
            return null;
        }
        try {
            instance = (MathExpression) MathExpressionClass.getConstructor(String.class).newInstance(name);
            //System.err.println("instance="+instance);
        } catch (Exception ie) {
            ie.printStackTrace(System.err);
        }
        return instance;
    }

    /**
     * Eval expression with given vars (generally numeric).
     */
    public abstract Object eval(String expression, Map<String, Object> vars) throws MathException;

    public static synchronized Object Eval(String expression, Map<String, Object> vars) throws Exception {
        if (GetDefaultInstance() == null) {
            throw new Exception("No default instance available.");
        }
        return GetDefaultInstance().eval(expression, vars);
    }

    /**
     * Set objects.
     */
    public boolean set(String... expression) throws MathException {
        boolean done = true;
        for (int i = 0; i < expression.length; i++) {
            String e = expression[i];
            if (e != null) {
                done = done & set(e);
            }
        }
        return done;
    }

    public abstract boolean set(String expression) throws MathException;

    public static synchronized boolean Set(String... expression) throws MathException {
        return GetDefaultInstance().set(expression);
    }

    public static boolean Set(String expression) throws MathException {
        return GetDefaultInstance().set(expression);
    }

    /**
     * unSet objects.
     */
    public abstract void reset() throws MathException;

    public static synchronized void Reset() throws MathException {
        GetDefaultInstance().reset();
    }

    /**
     * Used to get name of engine impl.
     */
    public abstract String getEngineName();

    public static String GetEngineName() {
        return GetDefaultInstance().getEngineName();
    }

    public abstract List<String> listVariables(boolean includeGlobalEnvironment, boolean includeShadowVariables);

    public final static String ALL_OPERANDS = ",;:.!?{}()[]<>+-*/=\\%&|^~$@#";

    public String getOperands() {
        return ALL_OPERANDS;
    }

    public static String GetOperands() {
        return GetDefaultInstance().getOperands();
    }

    public abstract String getMinusInfinityExpression();

    public static String GetMinusInfinityExpression() {
        return GetDefaultInstance().getMinusInfinityExpression();
    }

    public abstract String getPlusInfinityExpression();

    public static String GetPlusInfinityExpression() {
        return GetDefaultInstance().getPlusInfinityExpression();
    }
    final static String AW = "((\\A)|(\\W))(";
    final static String Az = ")((\\W)|(\\z))";

    public static String replaceVariable(final String expr, final String var, final String val) {
        String regexp = AW + var + Az;
        Matcher m = Pattern.compile(regexp).matcher(expr);
        if (m.find()) {
            return expr.replace(m.group(), m.group().replace(var, val));
        } else {
            return expr;
        }
    }
}
