"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.STORAGE_KEY_FORMAT_REGEX = exports.CATALOG_KEY = exports.NOT_SUPPORTED_SUFFIX = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_CSHARP = exports.DOCS_KEY_SUFFIX_JAVA = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_JAVA = docsKeySuffix(language_1.DocumentationLanguage.JAVA);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_CSHARP = docsKeySuffix(language_1.DocumentationLanguage.CSHARP);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
//# sourceMappingURL=data:application/json;base64,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