"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.throughVpcEndpoint = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
/**
 * Decorates an S3 Bucket so that grants are made including a VPC endpoint
 * policy.
 *
 * This currently only supports the `grantRead`, `grantWrite`, and `grantDelete`
 * APIs.
 *
 * @param bucket      the bucket to be wrapped.
 * @param vpcEndpoint the VPC Endpoint for S3 to be used.
 *
 * @returns a decorated S3 Bucket.
 */
function throughVpcEndpoint(bucket, vpcEndpoint) {
    return new Proxy(bucket, {
        get(target, property, _receiver) {
            switch (property) {
                case 'grantDelete':
                    return decoratedGrantDelete.bind(target);
                case 'grantRead':
                    return decoratedGrantRead.bind(target);
                case 'grantWrite':
                    return decoratedGrantWrite.bind(target);
                default:
                    if (typeof property === 'string' && /^grant([A-Z]|$)/.test(property)) {
                        throw new Error(`No VPC Endpoint policy grants will be added for ${property} on ${bucket.node.path}`);
                    }
                    return target[property];
            }
        },
        getOwnPropertyDescriptor(target, property) {
            const realDescriptor = Object.getOwnPropertyDescriptor(target, property);
            switch (property) {
                case 'grantDelete':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantDelete,
                        get: undefined,
                        set: undefined,
                    };
                case 'grantRead':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantRead,
                        get: undefined,
                        set: undefined,
                    };
                case 'grantWrite':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantWrite,
                        get: undefined,
                        set: undefined,
                    };
                default:
                    if (typeof property === 'string' && /^grant([A-Z]|$)/.test(property)) {
                        console.warn(`No VPC Endpoint policy grants will be added for ${property} on ${bucket.node.path}`);
                    }
                    return realDescriptor;
            }
        },
    });
    function decoratedGrantDelete(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantDelete(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:DeleteObject*'],
                resources: [this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
    function decoratedGrantRead(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantRead(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [this.bucketArn, this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
    function decoratedGrantWrite(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantWrite(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:Abort*', 's3:DeleteObject*', 's3:PutObject*'],
                resources: [this.bucketArn, this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
}
exports.throughVpcEndpoint = throughVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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