"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifact = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const storage_1 = require("../s3/storage");
const code_artifact_forwarder_1 = require("./codeartifact/code-artifact-forwarder");
const constants_lambda_shared_1 = require("./codeartifact/constants.lambda-shared");
/**
 * A package source that obtains package data from an npm CodeArtifact
 * repository.
 */
class CodeArtifact {
    constructor(props) {
        this.props = props;
    }
    bind(scope, { denyList, ingestion, licenseList, monitoring, queue }) {
        const idPrefix = this.props.repository.node.path;
        const repositoryId = `${this.props.repository.attrDomainOwner}:${this.props.repository.attrDomainName}/${this.props.repository.attrName}`;
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(scope);
        const bucket = this.props.bucket || storageFactory.newBucket(scope, `${idPrefix}/StagingBucket`, {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const dlq = new aws_sqs_1.Queue(scope, `${idPrefix}/DLQ`, {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const forwarder = new code_artifact_forwarder_1.CodeArtifactForwarder(scope, `${idPrefix}/Forwarder`, {
            deadLetterQueue: dlq,
            description: `[${scope.node.path}/CodeArtifact/${repositoryId}] Handle CodeArtifact EventBridge events`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.seconds(60),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(forwarder);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(forwarder);
        licenseList.grantRead(forwarder);
        queue.grantSendMessages(forwarder);
        forwarder.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['codeartifact:GetPackageVersionAsset'],
            resources: [
                core_1.Stack.of(scope).formatArn({
                    service: 'codeartifact',
                    resource: 'package',
                    arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                    resourceName: [
                        this.props.repository.attrDomainName,
                        this.props.repository.attrName,
                        'npm',
                        '*',
                    ].join('/'),
                }),
            ],
        }));
        const rule = new aws_events_1.Rule(scope, `${idPrefix}/EventBridge`, {
            description: `${scope.node.path}/CodeArtifact/${repositoryId}/EventBridge`,
            eventPattern: {
                source: ['aws.codeartifact'],
                detailType: ['CodeArtifact Package Version State Change'],
                detail: {
                    domainOwner: this.props.repository.attrDomainOwner,
                    domainName: this.props.repository.attrDomainName,
                    repositoryName: this.props.repository.attrName,
                    packageFormat: 'npm',
                },
            },
            targets: [new aws_events_targets_1.LambdaFunction(forwarder)],
        });
        const failureAlarm = forwarder.metricErrors().createAlarm(scope, `${idPrefix}/Forwarder/Failures`, {
            alarmName: `${scope.node.path}/CodeArtifact/${repositoryId}/Forwarder`,
            alarmDescription: [
                `The CodeArtifact fowarder for ${repositoryId} is failing`,
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(forwarder)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm(`CodeArtifact:${this.props.repository.attrName} Failures`, failureAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: dlq.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: dlq.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${idPrefix}/Forwarder/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/CodeArtifact/${repositoryId}/DLQNotEmpty`,
            alarmDescription: [
                `The CodeArtifact fowarder for ${repositoryId} is failing`,
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(forwarder)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(dlq)}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addLowSeverityAlarm(`CodeArtifact/${repositoryId} DLQ Not Empty`, dlqNotEmptyAlarm);
        rule.node.addDependency(failureAlarm, dlqNotEmptyAlarm);
        return {
            name: `CodeArtifact: ${repositoryId}`,
            links: [{
                    name: 'CodeArtifact',
                    url: deep_link_1.codeArtifactRepositoryUrl(this.props.repository),
                    primary: true,
                }, {
                    name: 'Forwarder Function',
                    url: deep_link_1.lambdaFunctionUrl(forwarder),
                }, {
                    name: 'Search Log group',
                    url: deep_link_1.lambdaSearchLogGroupUrl(forwarder),
                }, {
                    name: 'DLQ',
                    url: deep_link_1.sqsQueueUrl(dlq),
                }],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(forwarder.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(forwarder.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            forwarder.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            dlq.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            dlq.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            dlq.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Quality Metrics',
                        left: [
                            metric_utils_1.fillMetric(this.metricNotJsiiEnabledCount({ label: 'Not a jsii package' }), 0),
                            metric_utils_1.fillMetric(this.metricIneligibleLicense({ label: 'Ineligible License' }), 0),
                            metric_utils_1.fillMetric(this.metricDenyListedCount({ label: 'Deny Listed' }), 0),
                            metric_utils_1.fillMetric(this.metricDeletedCount({ label: 'Deletion Events' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
            ],
        };
    }
    /**
     * The count of package versions that were ignored due to being in the deny list.
     */
    metricDenyListedCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: this.props.repository.attrDomainOwner,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "DenyListedCount" /* DENY_LISTED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The number of package versions that were ignored due to using an ineloigible license.
     */
    metricIneligibleLicense(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "IneligibleLicenseCount" /* INELIGIBLE_LICENSE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The number of package versions that were deleted from CodeArtifact (those events are not
     * handled currently).
     */
    metricDeletedCount(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "DeletedCount" /* DELETED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The number of package versions that do not have a jsii assembly in the package.
     */
    metricNotJsiiEnabledCount(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.CodeArtifact = CodeArtifact;
_a = JSII_RTTI_SYMBOL_1;
CodeArtifact[_a] = { fqn: "construct-hub.sources.CodeArtifact", version: "0.3.255" };
//# sourceMappingURL=data:application/json;base64,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