
# test the renaming of files
# test that you can't save unless you give the correct number of paths

import unittest
from pathlib import Path
import tempfile

from ibllib.io.extractors.widefield import Widefield
from ibllib.pipes.widefield_tasks import WidefieldRegisterRaw


class TestWidefieldExtractor(unittest.TestCase):

    def setUp(self):
        # make temp directory and store the results
        self.td = tempfile.TemporaryDirectory()
        self.session_path = Path(self.td.name).joinpath('CSK-im-001', '2018-04-01', '002')
        self.widefield_path = self.session_path.joinpath('raw_widefield_data')
        self.widefield_path.mkdir(parents=True)
        self.wf = Widefield(session_path=self.session_path)

        for fname in self.wf.raw_names:
            self.widefield_path.joinpath(fname).touch()

    def test_save_files(self):
        """
        Test extractor save method that renames all widefield files and moves them to alf folder
        :return:
        """

        self.wf._save()

        for fname in self.wf.save_names:
            if fname:
                assert self.session_path.joinpath('alf/widefield', fname).exists()

    def test_remove_files(self):
        """
        Test the removal of motion corrected files
        :return:
        """
        motion_files = list(self.widefield_path.glob('motion*'))
        assert len(motion_files) == 3

        self.wf.remove_files(file_prefix='motion')

        for m_file in motion_files:
            assert not m_file.exists()

    def test_timestamp_saving(self):
        """
        Test that correct path must be provided to extract timestamps when specifying save paths
        :return:
        """
        with self.assertRaises(AssertionError):
            self.wf.sync_timestamps(save=True, save_paths=['lala.npy'])

    def tearDown(self) -> None:
        self.td.cleanup()


class TestWidefieldRegister(unittest.TestCase):

    def setUp(self):
        # make temp directory and store the results
        self.td = tempfile.TemporaryDirectory()
        self.session_path = Path(self.td.name).joinpath('CSK-im-001', '2018-04-01', '002')
        self.widefield_path = self.session_path.joinpath('raw_widefield_data')
        self.widefield_path.mkdir(parents=True)
        self.wf = WidefieldRegisterRaw(session_path=self.session_path)
        self.wf.get_signatures()

        for fname in self.wf.signature['input_files']:
            if 'camlog' in fname[0]:
                name = 'lala.camlog'
            else:
                name = fname[0]
            self.session_path.joinpath(fname[1], name).touch()

    def test_rename_files(self):
        """
        Test register raw rename method that renames raw widefield files and moves them to alf folder
        :return:
        """

        self.wf.rename_files(symlink_old=False)

        for fname in self.wf.signature['output_files']:
            assert self.session_path.joinpath(fname[1], fname[0]).exists()

    def tearDown(self) -> None:
        self.td.cleanup()
