"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/*
 * Creates an Appsync GraphQL API and with multiple tables.
 * Testing for importing, querying, and mutability.
 *
 * Stack verification steps:
 * Add to a table through appsync GraphQL API.
 * Read from a table through appsync API.
 *
 * -- aws appsync list-graphql-apis                 -- obtain apiId               --
 * -- aws appsync get-graphql-api --api-id [apiId]  -- obtain GraphQL endpoint    --
 * -- aws appsync list-api-keys --api-id [apiId]    -- obtain api key             --
 * -- bash verify.integ.graphql.sh [apiKey] [url]   -- shows query and mutation   --
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphqlApi(stack, 'Api', {
    name: 'demoapi',
    schema: lib_1.Schema.fromAsset(path_1.join(__dirname, 'integ.graphql.graphql')),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.API_KEY,
            },
        ],
    },
});
const noneDS = api.addNoneDataSource('none', { name: 'None' });
noneDS.createResolver({
    typeName: 'Query',
    fieldName: 'getServiceVersion',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: '2017-02-28',
    })),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: 'v1',
    })),
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
new aws_dynamodb_1.Table(stack, 'PaymentTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const paymentTable = aws_dynamodb_1.Table.fromTableName(stack, 'ImportedPaymentTable', 'PaymentTable');
const customerDS = api.addDynamoDbDataSource('customerDs', customerTable, { name: 'Customer' });
const orderDS = api.addDynamoDbDataSource('orderDs', orderTable, { name: 'Order' });
const paymentDS = api.addDynamoDbDataSource('paymentDs', paymentTable, { name: 'Payment' });
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: 'Eq', op: lib_1.KeyCondition.eq },
    { suffix: 'Lt', op: lib_1.KeyCondition.lt },
    { suffix: 'Le', op: lib_1.KeyCondition.le },
    { suffix: 'Gt', op: lib_1.KeyCondition.gt },
    { suffix: 'Ge', op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
paymentDS.createResolver({
    typeName: 'Query',
    fieldName: 'getPayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
paymentDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'savePayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('payment')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const httpDS = api.addHttpDataSource('ds', 'https://aws.amazon.com/', { name: 'http' });
httpDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'doPostOnAws',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(`{
    "version": "2018-05-29",
    "method": "POST",
    # if full path is https://api.xxxxxxxxx.com/posts then resourcePath would be /posts
    "resourcePath": "/path/123",
    "params":{
        "body": $util.toJson($ctx.args),
        "headers":{
            "Content-Type": "application/json",
            "Authorization": "$ctx.request.headers.Authorization"
        }
    }
  }`),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(`
    ## Raise a GraphQL field error in case of a datasource invocation error
    #if($ctx.error)
      $util.error($ctx.error.message, $ctx.error.type)
    #end
    ## if the response status code is not 200, then return an error. Else return the body **
    #if($ctx.result.statusCode == 200)
        ## If response is 200, return the body.
        $ctx.result.body
    #else
        ## If response is not 200, append the response to error block.
        $utils.appendError($ctx.result.body, "$ctx.result.statusCode")
    #end
  `),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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