"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
// Schema Definitions
const type = new appsync.ObjectType('test', {
    definition: {
        version: t.required_string,
    },
});
const query = new appsync.ObjectType('Query', {
    definition: {
        getTests: new appsync.ResolvableField({
            returnType: type.attribute({ isRequiredList: true, isList: true }),
        }),
    },
});
const mutation = new appsync.ObjectType('Mutation', {
    definition: {
        addTest: new appsync.ResolvableField({
            returnType: type.attribute(),
            args: { version: t.required_string },
        }),
    },
});
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
describe('basic testing schema definition mode `code`', () => {
    test('definition mode `code` produces empty schema definition', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: '',
        });
    });
    test('definition mode `code` generates correct schema with addToSchema', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
        });
        api.addType(type);
        api.addType(query);
        api.addType(mutation);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${type.toString()}\n${query.toString()}\n${mutation.toString()}\n`,
        });
    });
    test('definition mode `code` allows for api to addQuery', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
        });
        api.addQuery('test', new appsync.ResolvableField({
            returnType: t.string,
        }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: 'schema {\n  query: Query\n}\ntype Query {\n  test: String\n}\n',
        });
    });
    test('definition mode `code` allows for schema to addQuery', () => {
        // WHEN
        const schema = new appsync.Schema();
        new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema,
        });
        schema.addQuery('test', new appsync.ResolvableField({
            returnType: t.string,
        }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: 'schema {\n  query: Query\n}\ntype Query {\n  test: String\n}\n',
        });
    });
    test('definition mode `code` allows for api to addMutation', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
        });
        api.addMutation('test', new appsync.ResolvableField({
            returnType: t.string,
        }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: 'schema {\n  mutation: Mutation\n}\ntype Mutation {\n  test: String\n}\n',
        });
    });
    test('definition mode `code` allows for schema to addMutation', () => {
        // WHEN
        const schema = new appsync.Schema();
        new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema,
        });
        schema.addMutation('test', new appsync.ResolvableField({
            returnType: t.string,
        }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: 'schema {\n  mutation: Mutation\n}\ntype Mutation {\n  test: String\n}\n',
        });
    });
    test('definition mode `code` allows for api to addSubscription', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
        });
        api.addSubscription('test', new appsync.ResolvableField({
            returnType: t.string,
        }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: 'schema {\n  subscription: Subscription\n}\ntype Subscription {\n  test: String\n}\n',
        });
    });
    test('definition mode `code` allows for schema to addSubscription', () => {
        // WHEN
        const schema = new appsync.Schema();
        new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema,
        });
        schema.addSubscription('test', new appsync.ResolvableField({
            returnType: t.string,
        }));
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: 'schema {\n  subscription: Subscription\n}\ntype Subscription {\n  test: String\n}\n',
        });
    });
    test('definition mode `code` addSubscription w/ @aws_subscribe', () => {
        // WHE
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
        });
        api.addSubscription('test', new appsync.ResolvableField({
            returnType: t.string,
            directives: [appsync.Directive.subscribe('test1')],
        }));
        const out = 'schema {\n  subscription: Subscription\n}\ntype Subscription {\n  test: String\n  @aws_subscribe(mutations: ["test1"])\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: out,
        });
    });
});
describe('testing schema definition mode `file`', () => {
    test('definition mode `file` produces correct output', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${type.toString()}\n${query.toString()}\n${mutation.toString()}\n`,
        });
    });
    test('definition mode `file` errors when addType for object is called', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addType(new appsync.ObjectType('blah', {
                definition: { fail: t.id },
            }));
        }).toThrowError('API cannot add type because schema definition mode is not configured as CODE.');
    });
    test('definition mode `file` errors when addType for interface is called', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addType(new appsync.InterfaceType('blah', {
                definition: { fail: t.id },
            }));
        }).toThrowError('API cannot add type because schema definition mode is not configured as CODE.');
    });
    test('definition mode `file` errors when addToSchema is called', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addToSchema('blah');
        }).toThrowError('API cannot append to schema because schema definition mode is not configured as CODE.');
    });
    test('definition mode `file` errors when addQuery is called', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addQuery('blah', new appsync.ResolvableField({ returnType: t.string }));
        }).toThrowError('Unable to add query. Schema definition mode must be CODE. Received: FILE');
    });
    test('definition mode `file` errors when addMutation is called', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addMutation('blah', new appsync.ResolvableField({ returnType: t.string }));
        }).toThrowError('Unable to add mutation. Schema definition mode must be CODE. Received: FILE');
    });
    test('definition mode `file` errors when addSubscription is called', () => {
        // WHEN
        const api = new appsync.GraphqlApi(stack, 'API', {
            name: 'demo',
            schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(() => {
            api.addSubscription('blah', new appsync.ResolvableField({ returnType: t.string }));
        }).toThrowError('Unable to add subscription. Schema definition mode must be CODE. Received: FILE');
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwc3luYy1zY2hlbWEudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwcHN5bmMtc2NoZW1hLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSwrQkFBNEI7QUFDNUIsZ0NBQThCO0FBQzlCLHFDQUFxQztBQUNyQyxrQ0FBa0M7QUFDbEMsOENBQThDO0FBRTlDLHFCQUFxQjtBQUNyQixNQUFNLElBQUksR0FBRyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsTUFBTSxFQUFFO0lBQzFDLFVBQVUsRUFBRTtRQUNWLE9BQU8sRUFBRSxDQUFDLENBQUMsZUFBZTtLQUMzQjtDQUNGLENBQUMsQ0FBQztBQUNILE1BQU0sS0FBSyxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUU7SUFDNUMsVUFBVSxFQUFFO1FBQ1YsUUFBUSxFQUFFLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQztZQUNwQyxVQUFVLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLGNBQWMsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxDQUFDO1NBQ25FLENBQUM7S0FDSDtDQUNGLENBQUMsQ0FBQztBQUNILE1BQU0sUUFBUSxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxVQUFVLEVBQUU7SUFDbEQsVUFBVSxFQUFFO1FBQ1YsT0FBTyxFQUFFLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQztZQUNuQyxVQUFVLEVBQUUsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUM1QixJQUFJLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDLGVBQWUsRUFBRTtTQUNyQyxDQUFDO0tBQ0g7Q0FDRixDQUFDLENBQUM7QUFFSCxJQUFJLEtBQWdCLENBQUM7QUFDckIsVUFBVSxDQUFDLEdBQUcsRUFBRTtJQUNkLFFBQVE7SUFDUixLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7QUFDMUIsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMsNkNBQTZDLEVBQUUsR0FBRyxFQUFFO0lBRTNELElBQUksQ0FBQyx5REFBeUQsRUFBRSxHQUFHLEVBQUU7UUFDbkUsT0FBTztRQUNQLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxNQUFNO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUM5RCxVQUFVLEVBQUUsRUFBRTtTQUNmLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGtFQUFrRSxFQUFFLEdBQUcsRUFBRTtRQUM1RSxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDL0MsSUFBSSxFQUFFLE1BQU07U0FDYixDQUFDLENBQUM7UUFDSCxHQUFHLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xCLEdBQUcsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDbkIsR0FBRyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUV0QixPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDZCQUE2QixFQUFFO1lBQzlELFVBQVUsRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLEVBQUUsS0FBSyxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssUUFBUSxDQUFDLFFBQVEsRUFBRSxJQUFJO1NBQ2hGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG1EQUFtRCxFQUFFLEdBQUcsRUFBRTtRQUM3RCxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDL0MsSUFBSSxFQUFFLE1BQU07U0FDYixDQUFDLENBQUM7UUFDSCxHQUFHLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUM7WUFDL0MsVUFBVSxFQUFFLENBQUMsQ0FBQyxNQUFNO1NBQ3JCLENBQUMsQ0FBQyxDQUFDO1FBRUosT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUM5RCxVQUFVLEVBQUUsZ0VBQWdFO1NBQzdFLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtRQUNoRSxPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDcEMsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDbkMsSUFBSSxFQUFFLE1BQU07WUFDWixNQUFNO1NBQ1AsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsSUFBSSxPQUFPLENBQUMsZUFBZSxDQUFDO1lBQ2xELFVBQVUsRUFBRSxDQUFDLENBQUMsTUFBTTtTQUNyQixDQUFDLENBQUMsQ0FBQztRQUVKLE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsNkJBQTZCLEVBQUU7WUFDOUQsVUFBVSxFQUFFLGdFQUFnRTtTQUM3RSxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzREFBc0QsRUFBRSxHQUFHLEVBQUU7UUFDaEUsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQy9DLElBQUksRUFBRSxNQUFNO1NBQ2IsQ0FBQyxDQUFDO1FBQ0gsR0FBRyxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxPQUFPLENBQUMsZUFBZSxDQUFDO1lBQ2xELFVBQVUsRUFBRSxDQUFDLENBQUMsTUFBTTtTQUNyQixDQUFDLENBQUMsQ0FBQztRQUVKLE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsNkJBQTZCLEVBQUU7WUFDOUQsVUFBVSxFQUFFLHlFQUF5RTtTQUN0RixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx5REFBeUQsRUFBRSxHQUFHLEVBQUU7UUFDbkUsT0FBTztRQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksT0FBTyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ3BDLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxNQUFNO1lBQ1osTUFBTTtTQUNQLENBQUMsQ0FBQztRQUNILE1BQU0sQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQztZQUNyRCxVQUFVLEVBQUUsQ0FBQyxDQUFDLE1BQU07U0FDckIsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDZCQUE2QixFQUFFO1lBQzlELFVBQVUsRUFBRSx5RUFBeUU7U0FDdEYsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMERBQTBELEVBQUUsR0FBRyxFQUFFO1FBQ3BFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUMvQyxJQUFJLEVBQUUsTUFBTTtTQUNiLENBQUMsQ0FBQztRQUNILEdBQUcsQ0FBQyxlQUFlLENBQUMsTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQztZQUN0RCxVQUFVLEVBQUUsQ0FBQyxDQUFDLE1BQU07U0FDckIsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDZCQUE2QixFQUFFO1lBQzlELFVBQVUsRUFBRSxxRkFBcUY7U0FDbEcsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNkRBQTZELEVBQUUsR0FBRyxFQUFFO1FBQ3ZFLE9BQU87UUFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUNwQyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsTUFBTTtZQUNaLE1BQU07U0FDUCxDQUFDLENBQUM7UUFDSCxNQUFNLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRSxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUM7WUFDekQsVUFBVSxFQUFFLENBQUMsQ0FBQyxNQUFNO1NBQ3JCLENBQUMsQ0FBQyxDQUFDO1FBRUosT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUM5RCxVQUFVLEVBQUUscUZBQXFGO1NBQ2xHLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDBEQUEwRCxFQUFFLEdBQUcsRUFBRTtRQUNwRSxNQUFNO1FBQ04sTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDL0MsSUFBSSxFQUFFLE1BQU07U0FDYixDQUFDLENBQUM7UUFDSCxHQUFHLENBQUMsZUFBZSxDQUFDLE1BQU0sRUFBRSxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUM7WUFDdEQsVUFBVSxFQUFFLENBQUMsQ0FBQyxNQUFNO1lBQ3BCLFVBQVUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQ25ELENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxHQUFHLEdBQUcsNkhBQTZILENBQUM7UUFFMUksT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUM5RCxVQUFVLEVBQUUsR0FBRztTQUNoQixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtJQUVyRCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO1FBQzFELE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsTUFBTTtZQUNaLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxXQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7U0FDMUUsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw2QkFBNkIsRUFBRTtZQUM5RCxVQUFVLEVBQUUsR0FBRyxJQUFJLENBQUMsUUFBUSxFQUFFLEtBQUssS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLFFBQVEsQ0FBQyxRQUFRLEVBQUUsSUFBSTtTQUNoRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxpRUFBaUUsRUFBRSxHQUFHLEVBQUU7UUFDM0UsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQy9DLElBQUksRUFBRSxNQUFNO1lBQ1osTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFdBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRTtnQkFDekMsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUU7YUFDM0IsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsK0VBQStFLENBQUMsQ0FBQztJQUNuRyxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxvRUFBb0UsRUFBRSxHQUFHLEVBQUU7UUFDOUUsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQy9DLElBQUksRUFBRSxNQUFNO1lBQ1osTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFdBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLEdBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxPQUFPLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRTtnQkFDNUMsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUU7YUFDM0IsQ0FBQyxDQUFDLENBQUM7UUFDTixDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsK0VBQStFLENBQUMsQ0FBQztJQUNuRyxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7UUFDcEUsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQy9DLElBQUksRUFBRSxNQUFNO1lBQ1osTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFdBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLEdBQUcsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDMUIsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLHVGQUF1RixDQUFDLENBQUM7SUFDM0csQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO1FBQ2pFLE9BQU87UUFDUCxNQUFNLEdBQUcsR0FBRyxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUMvQyxJQUFJLEVBQUUsTUFBTTtZQUNaLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxXQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7U0FDMUUsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixHQUFHLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxJQUFJLE9BQU8sQ0FBQyxlQUFlLENBQUMsRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM5RSxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsMEVBQTBFLENBQUMsQ0FBQztJQUM5RixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywwREFBMEQsRUFBRSxHQUFHLEVBQUU7UUFDcEUsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQy9DLElBQUksRUFBRSxNQUFNO1lBQ1osTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFdBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztTQUMxRSxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLEdBQUcsQ0FBQyxXQUFXLENBQUMsTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLGVBQWUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ2pGLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyw2RUFBNkUsQ0FBQyxDQUFDO0lBQ2pHLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDhEQUE4RCxFQUFFLEdBQUcsRUFBRTtRQUN4RSxPQUFPO1FBQ1AsTUFBTSxHQUFHLEdBQUcsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDL0MsSUFBSSxFQUFFLE1BQU07WUFDWixNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsV0FBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1NBQzFFLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsR0FBRyxDQUFDLGVBQWUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxPQUFPLENBQUMsZUFBZSxDQUFDLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDckYsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLGlGQUFpRixDQUFDLENBQUM7SUFDckcsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGpvaW4gfSBmcm9tICdwYXRoJztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgYXBwc3luYyBmcm9tICcuLi9saWInO1xuaW1wb3J0ICogYXMgdCBmcm9tICcuL3NjYWxhci10eXBlLWRlZmludGlvbnMnO1xuXG4vLyBTY2hlbWEgRGVmaW5pdGlvbnNcbmNvbnN0IHR5cGUgPSBuZXcgYXBwc3luYy5PYmplY3RUeXBlKCd0ZXN0Jywge1xuICBkZWZpbml0aW9uOiB7XG4gICAgdmVyc2lvbjogdC5yZXF1aXJlZF9zdHJpbmcsXG4gIH0sXG59KTtcbmNvbnN0IHF1ZXJ5ID0gbmV3IGFwcHN5bmMuT2JqZWN0VHlwZSgnUXVlcnknLCB7XG4gIGRlZmluaXRpb246IHtcbiAgICBnZXRUZXN0czogbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHtcbiAgICAgIHJldHVyblR5cGU6IHR5cGUuYXR0cmlidXRlKHsgaXNSZXF1aXJlZExpc3Q6IHRydWUsIGlzTGlzdDogdHJ1ZSB9KSxcbiAgICB9KSxcbiAgfSxcbn0pO1xuY29uc3QgbXV0YXRpb24gPSBuZXcgYXBwc3luYy5PYmplY3RUeXBlKCdNdXRhdGlvbicsIHtcbiAgZGVmaW5pdGlvbjoge1xuICAgIGFkZFRlc3Q6IG5ldyBhcHBzeW5jLlJlc29sdmFibGVGaWVsZCh7XG4gICAgICByZXR1cm5UeXBlOiB0eXBlLmF0dHJpYnV0ZSgpLFxuICAgICAgYXJnczogeyB2ZXJzaW9uOiB0LnJlcXVpcmVkX3N0cmluZyB9LFxuICAgIH0pLFxuICB9LFxufSk7XG5cbmxldCBzdGFjazogY2RrLlN0YWNrO1xuYmVmb3JlRWFjaCgoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xufSk7XG5cbmRlc2NyaWJlKCdiYXNpYyB0ZXN0aW5nIHNjaGVtYSBkZWZpbml0aW9uIG1vZGUgYGNvZGVgJywgKCkgPT4ge1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgY29kZWAgcHJvZHVjZXMgZW1wdHkgc2NoZW1hIGRlZmluaXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdBUEknLCB7XG4gICAgICBuYW1lOiAnZGVtbycsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTFNjaGVtYScsIHtcbiAgICAgIERlZmluaXRpb246ICcnLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkZWZpbml0aW9uIG1vZGUgYGNvZGVgIGdlbmVyYXRlcyBjb3JyZWN0IHNjaGVtYSB3aXRoIGFkZFRvU2NoZW1hJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgIH0pO1xuICAgIGFwaS5hZGRUeXBlKHR5cGUpO1xuICAgIGFwaS5hZGRUeXBlKHF1ZXJ5KTtcbiAgICBhcGkuYWRkVHlwZShtdXRhdGlvbik7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTFNjaGVtYScsIHtcbiAgICAgIERlZmluaXRpb246IGAke3R5cGUudG9TdHJpbmcoKX1cXG4ke3F1ZXJ5LnRvU3RyaW5nKCl9XFxuJHttdXRhdGlvbi50b1N0cmluZygpfVxcbmAsXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgY29kZWAgYWxsb3dzIGZvciBhcGkgdG8gYWRkUXVlcnknLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwaSA9IG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdBUEknLCB7XG4gICAgICBuYW1lOiAnZGVtbycsXG4gICAgfSk7XG4gICAgYXBpLmFkZFF1ZXJ5KCd0ZXN0JywgbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHtcbiAgICAgIHJldHVyblR5cGU6IHQuc3RyaW5nLFxuICAgIH0pKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMU2NoZW1hJywge1xuICAgICAgRGVmaW5pdGlvbjogJ3NjaGVtYSB7XFxuICBxdWVyeTogUXVlcnlcXG59XFxudHlwZSBRdWVyeSB7XFxuICB0ZXN0OiBTdHJpbmdcXG59XFxuJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnZGVmaW5pdGlvbiBtb2RlIGBjb2RlYCBhbGxvd3MgZm9yIHNjaGVtYSB0byBhZGRRdWVyeScsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc2NoZW1hID0gbmV3IGFwcHN5bmMuU2NoZW1hKCk7XG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ0FQSScsIHtcbiAgICAgIG5hbWU6ICdkZW1vJyxcbiAgICAgIHNjaGVtYSxcbiAgICB9KTtcbiAgICBzY2hlbWEuYWRkUXVlcnkoJ3Rlc3QnLCBuZXcgYXBwc3luYy5SZXNvbHZhYmxlRmllbGQoe1xuICAgICAgcmV0dXJuVHlwZTogdC5zdHJpbmcsXG4gICAgfSkpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkdyYXBoUUxTY2hlbWEnLCB7XG4gICAgICBEZWZpbml0aW9uOiAnc2NoZW1hIHtcXG4gIHF1ZXJ5OiBRdWVyeVxcbn1cXG50eXBlIFF1ZXJ5IHtcXG4gIHRlc3Q6IFN0cmluZ1xcbn1cXG4nLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkZWZpbml0aW9uIG1vZGUgYGNvZGVgIGFsbG93cyBmb3IgYXBpIHRvIGFkZE11dGF0aW9uJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgIH0pO1xuICAgIGFwaS5hZGRNdXRhdGlvbigndGVzdCcsIG5ldyBhcHBzeW5jLlJlc29sdmFibGVGaWVsZCh7XG4gICAgICByZXR1cm5UeXBlOiB0LnN0cmluZyxcbiAgICB9KSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTFNjaGVtYScsIHtcbiAgICAgIERlZmluaXRpb246ICdzY2hlbWEge1xcbiAgbXV0YXRpb246IE11dGF0aW9uXFxufVxcbnR5cGUgTXV0YXRpb24ge1xcbiAgdGVzdDogU3RyaW5nXFxufVxcbicsXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgY29kZWAgYWxsb3dzIGZvciBzY2hlbWEgdG8gYWRkTXV0YXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHNjaGVtYSA9IG5ldyBhcHBzeW5jLlNjaGVtYSgpO1xuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdBUEknLCB7XG4gICAgICBuYW1lOiAnZGVtbycsXG4gICAgICBzY2hlbWEsXG4gICAgfSk7XG4gICAgc2NoZW1hLmFkZE11dGF0aW9uKCd0ZXN0JywgbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHtcbiAgICAgIHJldHVyblR5cGU6IHQuc3RyaW5nLFxuICAgIH0pKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMU2NoZW1hJywge1xuICAgICAgRGVmaW5pdGlvbjogJ3NjaGVtYSB7XFxuICBtdXRhdGlvbjogTXV0YXRpb25cXG59XFxudHlwZSBNdXRhdGlvbiB7XFxuICB0ZXN0OiBTdHJpbmdcXG59XFxuJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnZGVmaW5pdGlvbiBtb2RlIGBjb2RlYCBhbGxvd3MgZm9yIGFwaSB0byBhZGRTdWJzY3JpcHRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwaSA9IG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdBUEknLCB7XG4gICAgICBuYW1lOiAnZGVtbycsXG4gICAgfSk7XG4gICAgYXBpLmFkZFN1YnNjcmlwdGlvbigndGVzdCcsIG5ldyBhcHBzeW5jLlJlc29sdmFibGVGaWVsZCh7XG4gICAgICByZXR1cm5UeXBlOiB0LnN0cmluZyxcbiAgICB9KSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTFNjaGVtYScsIHtcbiAgICAgIERlZmluaXRpb246ICdzY2hlbWEge1xcbiAgc3Vic2NyaXB0aW9uOiBTdWJzY3JpcHRpb25cXG59XFxudHlwZSBTdWJzY3JpcHRpb24ge1xcbiAgdGVzdDogU3RyaW5nXFxufVxcbicsXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgY29kZWAgYWxsb3dzIGZvciBzY2hlbWEgdG8gYWRkU3Vic2NyaXB0aW9uJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBzY2hlbWEgPSBuZXcgYXBwc3luYy5TY2hlbWEoKTtcbiAgICBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgICAgc2NoZW1hLFxuICAgIH0pO1xuICAgIHNjaGVtYS5hZGRTdWJzY3JpcHRpb24oJ3Rlc3QnLCBuZXcgYXBwc3luYy5SZXNvbHZhYmxlRmllbGQoe1xuICAgICAgcmV0dXJuVHlwZTogdC5zdHJpbmcsXG4gICAgfSkpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkdyYXBoUUxTY2hlbWEnLCB7XG4gICAgICBEZWZpbml0aW9uOiAnc2NoZW1hIHtcXG4gIHN1YnNjcmlwdGlvbjogU3Vic2NyaXB0aW9uXFxufVxcbnR5cGUgU3Vic2NyaXB0aW9uIHtcXG4gIHRlc3Q6IFN0cmluZ1xcbn1cXG4nLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkZWZpbml0aW9uIG1vZGUgYGNvZGVgIGFkZFN1YnNjcmlwdGlvbiB3LyBAYXdzX3N1YnNjcmliZScsICgpID0+IHtcbiAgICAvLyBXSEVcbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgIH0pO1xuICAgIGFwaS5hZGRTdWJzY3JpcHRpb24oJ3Rlc3QnLCBuZXcgYXBwc3luYy5SZXNvbHZhYmxlRmllbGQoe1xuICAgICAgcmV0dXJuVHlwZTogdC5zdHJpbmcsXG4gICAgICBkaXJlY3RpdmVzOiBbYXBwc3luYy5EaXJlY3RpdmUuc3Vic2NyaWJlKCd0ZXN0MScpXSxcbiAgICB9KSk7XG5cbiAgICBjb25zdCBvdXQgPSAnc2NoZW1hIHtcXG4gIHN1YnNjcmlwdGlvbjogU3Vic2NyaXB0aW9uXFxufVxcbnR5cGUgU3Vic2NyaXB0aW9uIHtcXG4gIHRlc3Q6IFN0cmluZ1xcbiAgQGF3c19zdWJzY3JpYmUobXV0YXRpb25zOiBbXCJ0ZXN0MVwiXSlcXG59XFxuJztcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMU2NoZW1hJywge1xuICAgICAgRGVmaW5pdGlvbjogb3V0LFxuICAgIH0pO1xuICB9KTtcbn0pO1xuXG5kZXNjcmliZSgndGVzdGluZyBzY2hlbWEgZGVmaW5pdGlvbiBtb2RlIGBmaWxlYCcsICgpID0+IHtcblxuICB0ZXN0KCdkZWZpbml0aW9uIG1vZGUgYGZpbGVgIHByb2R1Y2VzIGNvcnJlY3Qgb3V0cHV0JywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMU2NoZW1hJywge1xuICAgICAgRGVmaW5pdGlvbjogYCR7dHlwZS50b1N0cmluZygpfVxcbiR7cXVlcnkudG9TdHJpbmcoKX1cXG4ke211dGF0aW9uLnRvU3RyaW5nKCl9XFxuYCxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnZGVmaW5pdGlvbiBtb2RlIGBmaWxlYCBlcnJvcnMgd2hlbiBhZGRUeXBlIGZvciBvYmplY3QgaXMgY2FsbGVkJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgYXBpLmFkZFR5cGUobmV3IGFwcHN5bmMuT2JqZWN0VHlwZSgnYmxhaCcsIHtcbiAgICAgICAgZGVmaW5pdGlvbjogeyBmYWlsOiB0LmlkIH0sXG4gICAgICB9KSk7XG4gICAgfSkudG9UaHJvd0Vycm9yKCdBUEkgY2Fubm90IGFkZCB0eXBlIGJlY2F1c2Ugc2NoZW1hIGRlZmluaXRpb24gbW9kZSBpcyBub3QgY29uZmlndXJlZCBhcyBDT0RFLicpO1xuICB9KTtcblxuICB0ZXN0KCdkZWZpbml0aW9uIG1vZGUgYGZpbGVgIGVycm9ycyB3aGVuIGFkZFR5cGUgZm9yIGludGVyZmFjZSBpcyBjYWxsZWQnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGFwaSA9IG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdBUEknLCB7XG4gICAgICBuYW1lOiAnZGVtbycsXG4gICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChqb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBhcGkuYWRkVHlwZShuZXcgYXBwc3luYy5JbnRlcmZhY2VUeXBlKCdibGFoJywge1xuICAgICAgICBkZWZpbml0aW9uOiB7IGZhaWw6IHQuaWQgfSxcbiAgICAgIH0pKTtcbiAgICB9KS50b1Rocm93RXJyb3IoJ0FQSSBjYW5ub3QgYWRkIHR5cGUgYmVjYXVzZSBzY2hlbWEgZGVmaW5pdGlvbiBtb2RlIGlzIG5vdCBjb25maWd1cmVkIGFzIENPREUuJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgZmlsZWAgZXJyb3JzIHdoZW4gYWRkVG9TY2hlbWEgaXMgY2FsbGVkJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgYXBpLmFkZFRvU2NoZW1hKCdibGFoJyk7XG4gICAgfSkudG9UaHJvd0Vycm9yKCdBUEkgY2Fubm90IGFwcGVuZCB0byBzY2hlbWEgYmVjYXVzZSBzY2hlbWEgZGVmaW5pdGlvbiBtb2RlIGlzIG5vdCBjb25maWd1cmVkIGFzIENPREUuJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgZmlsZWAgZXJyb3JzIHdoZW4gYWRkUXVlcnkgaXMgY2FsbGVkJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgYXBpLmFkZFF1ZXJ5KCdibGFoJywgbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHsgcmV0dXJuVHlwZTogdC5zdHJpbmcgfSkpO1xuICAgIH0pLnRvVGhyb3dFcnJvcignVW5hYmxlIHRvIGFkZCBxdWVyeS4gU2NoZW1hIGRlZmluaXRpb24gbW9kZSBtdXN0IGJlIENPREUuIFJlY2VpdmVkOiBGSUxFJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgZmlsZWAgZXJyb3JzIHdoZW4gYWRkTXV0YXRpb24gaXMgY2FsbGVkJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCBhcGkgPSBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgbmFtZTogJ2RlbW8nLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgYXBpLmFkZE11dGF0aW9uKCdibGFoJywgbmV3IGFwcHN5bmMuUmVzb2x2YWJsZUZpZWxkKHsgcmV0dXJuVHlwZTogdC5zdHJpbmcgfSkpO1xuICAgIH0pLnRvVGhyb3dFcnJvcignVW5hYmxlIHRvIGFkZCBtdXRhdGlvbi4gU2NoZW1hIGRlZmluaXRpb24gbW9kZSBtdXN0IGJlIENPREUuIFJlY2VpdmVkOiBGSUxFJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluaXRpb24gbW9kZSBgZmlsZWAgZXJyb3JzIHdoZW4gYWRkU3Vic2NyaXB0aW9uIGlzIGNhbGxlZCcsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgYXBpID0gbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ0FQSScsIHtcbiAgICAgIG5hbWU6ICdkZW1vJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KGpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy50ZXN0LmdyYXBocWwnKSksXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIGFwaS5hZGRTdWJzY3JpcHRpb24oJ2JsYWgnLCBuZXcgYXBwc3luYy5SZXNvbHZhYmxlRmllbGQoeyByZXR1cm5UeXBlOiB0LnN0cmluZyB9KSk7XG4gICAgfSkudG9UaHJvd0Vycm9yKCdVbmFibGUgdG8gYWRkIHN1YnNjcmlwdGlvbi4gU2NoZW1hIGRlZmluaXRpb24gbW9kZSBtdXN0IGJlIENPREUuIFJlY2VpdmVkOiBGSUxFJyk7XG4gIH0pO1xufSk7Il19