"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
// GLOBAL GIVEN
let stack;
let api;
beforeEach(() => {
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'baseApi', {
        name: 'api',
        schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
    });
});
describe('Lambda Data Source configuration', () => {
    // GIVEN
    let func;
    beforeEach(() => {
        func = new lambda.Function(stack, 'func', {
            code: lambda.Code.fromAsset('test/verify'),
            handler: 'iam-query.handler',
            runtime: lambda.Runtime.NODEJS_12_X,
        });
    });
    test('default configuration produces name `TableCDKDataSource`', () => {
        // WHEN
        api.addLambdaDataSource('ds', func);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AWS_LAMBDA',
            Name: 'ds',
        });
    });
    test('appsync configures name correctly', () => {
        // WHEN
        api.addLambdaDataSource('ds', func, {
            name: 'custom',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AWS_LAMBDA',
            Name: 'custom',
        });
    });
    test('appsync configures name and description correctly', () => {
        // WHEN
        api.addLambdaDataSource('ds', func, {
            name: 'custom',
            description: 'custom description',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AWS_LAMBDA',
            Name: 'custom',
            Description: 'custom description',
        });
    });
    test('appsync errors when creating multiple lambda data sources with no configuration', () => {
        // THEN
        expect(() => {
            api.addLambdaDataSource('ds', func);
            api.addLambdaDataSource('ds', func);
        }).toThrow("There is already a Construct with name 'ds' in GraphqlApi [baseApi]");
    });
});
describe('adding lambda data source from imported api', () => {
    let func;
    beforeEach(() => {
        func = new lambda.Function(stack, 'func', {
            code: lambda.Code.fromAsset('test/verify'),
            handler: 'iam-query.handler',
            runtime: lambda.Runtime.NODEJS_12_X,
        });
    });
    test('imported api can add LambdaDbDataSource from id', () => {
        // WHEN
        const importedApi = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
        });
        importedApi.addLambdaDataSource('ds', func);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AWS_LAMBDA',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
    test('imported api can add LambdaDataSource from attributes', () => {
        // WHEN
        const importedApi = appsync.GraphqlApi.fromGraphqlApiAttributes(stack, 'importedApi', {
            graphqlApiId: api.apiId,
            graphqlApiArn: api.arn,
        });
        importedApi.addLambdaDataSource('ds', func);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::DataSource', {
            Type: 'AWS_LAMBDA',
            ApiId: { 'Fn::GetAtt': ['baseApiCDA4D43A', 'ApiId'] },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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