"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const path = require("path");
const cognito = require("@aws-cdk/aws-cognito");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
// GIVEN
let stack;
beforeEach(() => {
    stack = new cdk.Stack();
});
describe('AppSync API Key Authorization', () => {
    test('AppSync creates default api key', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
        });
        // THEN
        expect(stack).toHaveResource('AWS::AppSync::ApiKey');
    });
    test('AppSync creates api key from additionalAuthorizationModes', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
                additionalAuthorizationModes: [
                    { authorizationType: appsync.AuthorizationType.API_KEY },
                ],
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::AppSync::ApiKey');
    });
    test('AppSync does not create unspecified api key from additionalAuthorizationModes', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
            },
        });
        // THEN
        expect(stack).not.toHaveResource('AWS::AppSync::ApiKey');
    });
    test('appsync does not create unspecified api key with empty additionalAuthorizationModes', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
                additionalAuthorizationModes: [],
            },
        });
        // THEN
        expect(stack).not.toHaveResource('AWS::AppSync::ApiKey');
    });
    test('appsync creates configured api key with additionalAuthorizationModes', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
                additionalAuthorizationModes: [{
                        authorizationType: appsync.AuthorizationType.API_KEY,
                        apiKeyConfig: { description: 'Custom Description' },
                    }],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::ApiKey', {
            Description: 'Custom Description',
        });
    });
    test('apiKeyConfig creates default with valid expiration date', () => {
        const expirationDate = cdk.Expiration.after(cdk.Duration.days(10)).toEpoch();
        // WHEN
        new appsync.GraphqlApi(stack, 'API', {
            name: 'apiKeyUnitTest',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.auth.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.API_KEY,
                    apiKeyConfig: {
                        expires: cdk.Expiration.after(cdk.Duration.days(10)),
                    },
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::ApiKey', {
            ApiId: { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
            Expires: expirationDate,
        });
    });
    test('apiKeyConfig fails if expire argument less than a day', () => {
        // WHEN
        const when = () => {
            new appsync.GraphqlApi(stack, 'API', {
                name: 'apiKeyUnitTest',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.auth.graphql')),
                authorizationConfig: {
                    defaultAuthorization: {
                        authorizationType: appsync.AuthorizationType.API_KEY,
                        apiKeyConfig: {
                            expires: cdk.Expiration.after(cdk.Duration.hours(1)),
                        },
                    },
                },
            });
        };
        // THEN
        expect(when).toThrowError('API key expiration must be between 1 and 365 days.');
    });
    test('apiKeyConfig fails if expire argument greater than 365 day', () => {
        // WHEN
        const when = () => {
            new appsync.GraphqlApi(stack, 'API', {
                name: 'apiKeyUnitTest',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.auth.graphql')),
                authorizationConfig: {
                    defaultAuthorization: {
                        authorizationType: appsync.AuthorizationType.API_KEY,
                        apiKeyConfig: {
                            expires: cdk.Expiration.after(cdk.Duration.days(366)),
                        },
                    },
                },
            });
        };
        // THEN
        expect(when).toThrowError('API key expiration must be between 1 and 365 days.');
    });
    test('appsync creates configured api key with additionalAuthorizationModes (not as first element)', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.USER_POOL,
                        userPoolConfig: { userPool: new cognito.UserPool(stack, 'myPool') },
                    },
                    {
                        authorizationType: appsync.AuthorizationType.API_KEY,
                        apiKeyConfig: { description: 'Custom Description' },
                    },
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::ApiKey', {
            Description: 'Custom Description',
        });
    });
    test('appsync fails when empty default and API_KEY in additional', () => {
        // THEN
        expect(() => {
            new appsync.GraphqlApi(stack, 'api', {
                name: 'api',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
                authorizationConfig: {
                    additionalAuthorizationModes: [{
                            authorizationType: appsync.AuthorizationType.API_KEY,
                        }],
                },
            });
        }).toThrowError('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
    });
    test('appsync fails when multiple API_KEY auth modes', () => {
        // THEN
        expect(() => {
            new appsync.GraphqlApi(stack, 'api', {
                name: 'api',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
                authorizationConfig: {
                    defaultAuthorization: { authorizationType: appsync.AuthorizationType.API_KEY },
                    additionalAuthorizationModes: [{
                            authorizationType: appsync.AuthorizationType.API_KEY,
                        }],
                },
            });
        }).toThrowError('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
    });
    test('appsync fails when multiple API_KEY auth modes in additionalXxx', () => {
        // THEN
        expect(() => {
            new appsync.GraphqlApi(stack, 'api', {
                name: 'api',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
                authorizationConfig: {
                    defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
                    additionalAuthorizationModes: [
                        { authorizationType: appsync.AuthorizationType.API_KEY },
                        { authorizationType: appsync.AuthorizationType.API_KEY },
                    ],
                },
            });
        }).toThrowError('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
    });
});
describe('AppSync IAM Authorization', () => {
    test('Iam authorization configurable in default authorization', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'AWS_IAM',
        });
    });
    test('Iam authorization configurable in additional authorization', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                additionalAuthorizationModes: [{ authorizationType: appsync.AuthorizationType.IAM }],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AdditionalAuthenticationProviders: [{ AuthenticationType: 'AWS_IAM' }],
        });
    });
    test('appsync fails when multiple iam auth modes', () => {
        // THEN
        expect(() => {
            new appsync.GraphqlApi(stack, 'api', {
                name: 'api',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
                authorizationConfig: {
                    defaultAuthorization: { authorizationType: appsync.AuthorizationType.IAM },
                    additionalAuthorizationModes: [{ authorizationType: appsync.AuthorizationType.IAM }],
                },
            });
        }).toThrowError('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
    });
    test('appsync fails when multiple IAM auth modes in additionalXxx', () => {
        // THEN
        expect(() => {
            new appsync.GraphqlApi(stack, 'api', {
                name: 'api',
                schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
                authorizationConfig: {
                    additionalAuthorizationModes: [
                        { authorizationType: appsync.AuthorizationType.IAM },
                        { authorizationType: appsync.AuthorizationType.IAM },
                    ],
                },
            });
        }).toThrowError('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
    });
});
describe('AppSync User Pool Authorization', () => {
    let userPool;
    beforeEach(() => {
        userPool = new cognito.UserPool(stack, 'pool');
    });
    test('User Pool authorization configurable in default authorization has default configuration', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool },
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
            UserPoolConfig: {
                AwsRegion: { Ref: 'AWS::Region' },
                DefaultAction: 'ALLOW',
                UserPoolId: { Ref: 'pool056F3F7E' },
            },
        });
    });
    test('User Pool authorization configurable in default authorization', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: {
                        userPool,
                        appIdClientRegex: 'test',
                        defaultAction: appsync.UserPoolDefaultAction.DENY,
                    },
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
            UserPoolConfig: {
                AwsRegion: { Ref: 'AWS::Region' },
                DefaultAction: 'DENY',
                AppIdClientRegex: 'test',
                UserPoolId: { Ref: 'pool056F3F7E' },
            },
        });
    });
    test('User Pool authorization configurable in additional authorization has default configuration', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                additionalAuthorizationModes: [{
                        authorizationType: appsync.AuthorizationType.USER_POOL,
                        userPoolConfig: { userPool },
                    }],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AdditionalAuthenticationProviders: [{
                    AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    UserPoolConfig: {
                        AwsRegion: { Ref: 'AWS::Region' },
                        UserPoolId: { Ref: 'pool056F3F7E' },
                    },
                }],
        });
    });
    test('User Pool property defaultAction does not configure when in additional auth', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                additionalAuthorizationModes: [{
                        authorizationType: appsync.AuthorizationType.USER_POOL,
                        userPoolConfig: {
                            userPool,
                            appIdClientRegex: 'test',
                            defaultAction: appsync.UserPoolDefaultAction.DENY,
                        },
                    }],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AdditionalAuthenticationProviders: [{
                    AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    UserPoolConfig: {
                        AwsRegion: { Ref: 'AWS::Region' },
                        AppIdClientRegex: 'test',
                        UserPoolId: { Ref: 'pool056F3F7E' },
                    },
                }],
        });
    });
    test('User Pool property defaultAction does not configure when in additional auth', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.USER_POOL,
                    userPoolConfig: { userPool },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.USER_POOL,
                        userPoolConfig: { userPool },
                    },
                    {
                        authorizationType: appsync.AuthorizationType.USER_POOL,
                        userPoolConfig: {
                            userPool,
                            appIdClientRegex: 'test',
                            defaultAction: appsync.UserPoolDefaultAction.DENY,
                        },
                    },
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
            UserPoolConfig: {
                AwsRegion: { Ref: 'AWS::Region' },
                DefaultAction: 'ALLOW',
                UserPoolId: { Ref: 'pool056F3F7E' },
            },
            AdditionalAuthenticationProviders: [
                {
                    AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    UserPoolConfig: {
                        AwsRegion: { Ref: 'AWS::Region' },
                        UserPoolId: { Ref: 'pool056F3F7E' },
                    },
                },
                {
                    AuthenticationType: 'AMAZON_COGNITO_USER_POOLS',
                    UserPoolConfig: {
                        AwsRegion: { Ref: 'AWS::Region' },
                        AppIdClientRegex: 'test',
                        UserPoolId: { Ref: 'pool056F3F7E' },
                    },
                },
            ],
        });
    });
});
describe('AppSync OIDC Authorization', () => {
    test('OIDC authorization configurable in default authorization has default configuration', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.OIDC,
                    openIdConnectConfig: { oidcProvider: 'test' },
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'OPENID_CONNECT',
            OpenIDConnectConfig: {
                Issuer: 'test',
            },
        });
    });
    test('User Pool authorization configurable in default authorization', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.OIDC,
                    openIdConnectConfig: {
                        oidcProvider: 'test',
                        clientId: 'id',
                        tokenExpiryFromAuth: 1,
                        tokenExpiryFromIssue: 1,
                    },
                },
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'OPENID_CONNECT',
            OpenIDConnectConfig: {
                AuthTTL: 1,
                ClientId: 'id',
                IatTTL: 1,
                Issuer: 'test',
            },
        });
    });
    test('OIDC authorization configurable in additional authorization has default configuration', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                additionalAuthorizationModes: [{
                        authorizationType: appsync.AuthorizationType.OIDC,
                        openIdConnectConfig: { oidcProvider: 'test' },
                    }],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AdditionalAuthenticationProviders: [{
                    AuthenticationType: 'OPENID_CONNECT',
                    OpenIDConnectConfig: {
                        Issuer: 'test',
                    },
                }],
        });
    });
    test('User Pool authorization configurable in additional authorization', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                additionalAuthorizationModes: [{
                        authorizationType: appsync.AuthorizationType.OIDC,
                        openIdConnectConfig: {
                            oidcProvider: 'test',
                            clientId: 'id',
                            tokenExpiryFromAuth: 1,
                            tokenExpiryFromIssue: 1,
                        },
                    }],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AdditionalAuthenticationProviders: [{
                    AuthenticationType: 'OPENID_CONNECT',
                    OpenIDConnectConfig: {
                        AuthTTL: 1,
                        ClientId: 'id',
                        IatTTL: 1,
                        Issuer: 'test',
                    },
                }],
        });
    });
    test('User Pool authorization configurable in with multiple authorization', () => {
        // WHEN
        new appsync.GraphqlApi(stack, 'api', {
            name: 'api',
            schema: appsync.Schema.fromAsset(path.join(__dirname, 'appsync.test.graphql')),
            authorizationConfig: {
                defaultAuthorization: {
                    authorizationType: appsync.AuthorizationType.OIDC,
                    openIdConnectConfig: { oidcProvider: 'test' },
                },
                additionalAuthorizationModes: [
                    {
                        authorizationType: appsync.AuthorizationType.OIDC,
                        openIdConnectConfig: {
                            oidcProvider: 'test1',
                            clientId: 'id',
                            tokenExpiryFromAuth: 1,
                            tokenExpiryFromIssue: 1,
                        },
                    },
                    {
                        authorizationType: appsync.AuthorizationType.OIDC,
                        openIdConnectConfig: {
                            oidcProvider: 'test2',
                            clientId: 'id',
                            tokenExpiryFromAuth: 1,
                            tokenExpiryFromIssue: 1,
                        },
                    },
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLApi', {
            AuthenticationType: 'OPENID_CONNECT',
            OpenIDConnectConfig: { Issuer: 'test' },
            AdditionalAuthenticationProviders: [
                {
                    AuthenticationType: 'OPENID_CONNECT',
                    OpenIDConnectConfig: {
                        AuthTTL: 1,
                        ClientId: 'id',
                        IatTTL: 1,
                        Issuer: 'test1',
                    },
                },
                {
                    AuthenticationType: 'OPENID_CONNECT',
                    OpenIDConnectConfig: {
                        AuthTTL: 1,
                        ClientId: 'id',
                        IatTTL: 1,
                        Issuer: 'test2',
                    },
                },
            ],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwc3luYy1hdXRoLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcHBzeW5jLWF1dGgudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdDQUE4QjtBQUM5Qiw2QkFBNkI7QUFDN0IsZ0RBQWdEO0FBQ2hELHFDQUFxQztBQUNyQyxrQ0FBa0M7QUFFbEMsUUFBUTtBQUNSLElBQUksS0FBZ0IsQ0FBQztBQUNyQixVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0FBQzFCLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtJQUM3QyxJQUFJLENBQUMsaUNBQWlDLEVBQUUsR0FBRyxFQUFFO1FBQzNDLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1NBQy9FLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDdkQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMkRBQTJELEVBQUUsR0FBRyxFQUFFO1FBQ3JFLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRSxFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUU7Z0JBQzFFLDRCQUE0QixFQUFFO29CQUM1QixFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLEVBQUU7aUJBQ3pEO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3ZELENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLCtFQUErRSxFQUFFLEdBQUcsRUFBRTtRQUN6RixPQUFPO1FBQ1AsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDbkMsSUFBSSxFQUFFLEtBQUs7WUFDWCxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztZQUM5RSxtQkFBbUIsRUFBRTtnQkFDbkIsb0JBQW9CLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsR0FBRyxFQUFFO2FBQzNFO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxHQUFHLENBQUMsY0FBYyxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDM0QsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMscUZBQXFGLEVBQUUsR0FBRyxFQUFFO1FBQy9GLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRSxFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUU7Z0JBQzFFLDRCQUE0QixFQUFFLEVBQUU7YUFDakM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUMzRCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxzRUFBc0UsRUFBRSxHQUFHLEVBQUU7UUFDaEYsT0FBTztRQUNQLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxLQUFLO1lBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDOUUsbUJBQW1CLEVBQUU7Z0JBQ25CLG9CQUFvQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRTtnQkFDMUUsNEJBQTRCLEVBQUUsQ0FBQzt3QkFDN0IsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU87d0JBQ3BELFlBQVksRUFBRSxFQUFFLFdBQVcsRUFBRSxvQkFBb0IsRUFBRTtxQkFDcEQsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxzQkFBc0IsRUFBRTtZQUN2RCxXQUFXLEVBQUUsb0JBQW9CO1NBQ2xDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHlEQUF5RCxFQUFFLEdBQUcsRUFBRTtRQUNuRSxNQUFNLGNBQWMsR0FBVyxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBRXJGLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsZ0JBQWdCO1lBQ3RCLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRTtvQkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU87b0JBQ3BELFlBQVksRUFBRTt3QkFDWixPQUFPLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7cUJBQ3JEO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQixFQUFFO1lBQ3ZELEtBQUssRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFDLGFBQWEsRUFBRSxPQUFPLENBQUMsRUFBRTtZQUNqRCxPQUFPLEVBQUUsY0FBYztTQUN4QixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx1REFBdUQsRUFBRSxHQUFHLEVBQUU7UUFDakUsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLEdBQUcsRUFBRTtZQUNoQixJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDbkMsSUFBSSxFQUFFLGdCQUFnQjtnQkFDdEIsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7Z0JBQzlFLG1CQUFtQixFQUFFO29CQUNuQixvQkFBb0IsRUFBRTt3QkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU87d0JBQ3BELFlBQVksRUFBRTs0QkFDWixPQUFPLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7eUJBQ3JEO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBRUYsT0FBTztRQUNQLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxZQUFZLENBQUMsb0RBQW9ELENBQUMsQ0FBQztJQUNsRixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw0REFBNEQsRUFBRSxHQUFHLEVBQUU7UUFDdEUsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLEdBQUcsRUFBRTtZQUNoQixJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDbkMsSUFBSSxFQUFFLGdCQUFnQjtnQkFDdEIsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7Z0JBQzlFLG1CQUFtQixFQUFFO29CQUNuQixvQkFBb0IsRUFBRTt3QkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU87d0JBQ3BELFlBQVksRUFBRTs0QkFDWixPQUFPLEVBQUUsR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7eUJBQ3REO3FCQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBRUYsT0FBTztRQUNQLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxZQUFZLENBQUMsb0RBQW9ELENBQUMsQ0FBQztJQUNsRixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw2RkFBNkYsRUFBRSxHQUFHLEVBQUU7UUFDdkcsT0FBTztRQUNQLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxLQUFLO1lBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDOUUsbUJBQW1CLEVBQUU7Z0JBQ25CLG9CQUFvQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRTtnQkFDMUUsNEJBQTRCLEVBQUU7b0JBQzVCO3dCQUNFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO3dCQUN0RCxjQUFjLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxPQUFPLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsRUFBRTtxQkFDcEU7b0JBQ0Q7d0JBQ0UsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU87d0JBQ3BELFlBQVksRUFBRSxFQUFFLFdBQVcsRUFBRSxvQkFBb0IsRUFBRTtxQkFDcEQ7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCLEVBQUU7WUFDdkQsV0FBVyxFQUFFLG9CQUFvQjtTQUNsQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw0REFBNEQsRUFBRSxHQUFHLEVBQUU7UUFDdEUsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDbkMsSUFBSSxFQUFFLEtBQUs7Z0JBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7Z0JBQzlFLG1CQUFtQixFQUFFO29CQUNuQiw0QkFBNEIsRUFBRSxDQUFDOzRCQUM3QixpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsT0FBTzt5QkFDckQsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO0lBQ3ZJLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGdEQUFnRCxFQUFFLEdBQUcsRUFBRTtRQUMxRCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO2dCQUNuQyxJQUFJLEVBQUUsS0FBSztnQkFDWCxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztnQkFDOUUsbUJBQW1CLEVBQUU7b0JBQ25CLG9CQUFvQixFQUFFLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sRUFBRTtvQkFDOUUsNEJBQTRCLEVBQUUsQ0FBQzs0QkFDN0IsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU87eUJBQ3JELENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsbUhBQW1ILENBQUMsQ0FBQztJQUN2SSxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxpRUFBaUUsRUFBRSxHQUFHLEVBQUU7UUFDM0UsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtnQkFDbkMsSUFBSSxFQUFFLEtBQUs7Z0JBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7Z0JBQzlFLG1CQUFtQixFQUFFO29CQUNuQixvQkFBb0IsRUFBRSxFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUU7b0JBQzFFLDRCQUE0QixFQUFFO3dCQUM1QixFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxPQUFPLEVBQUU7d0JBQ3hELEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sRUFBRTtxQkFDekQ7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsbUhBQW1ILENBQUMsQ0FBQztJQUN2SSxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLDJCQUEyQixFQUFFLEdBQUcsRUFBRTtJQUN6QyxJQUFJLENBQUMseURBQXlELEVBQUUsR0FBRyxFQUFFO1FBQ25FLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRSxFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUU7YUFDM0U7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDBCQUEwQixFQUFFO1lBQzNELGtCQUFrQixFQUFFLFNBQVM7U0FDOUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO1FBQ3RFLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQiw0QkFBNEIsRUFBRSxDQUFDLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxDQUFDO2FBQ3JGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywwQkFBMEIsRUFBRTtZQUMzRCxpQ0FBaUMsRUFBRSxDQUFDLEVBQUUsa0JBQWtCLEVBQUUsU0FBUyxFQUFFLENBQUM7U0FDdkUsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNENBQTRDLEVBQUUsR0FBRyxFQUFFO1FBQ3RELE9BQU87UUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7Z0JBQ25DLElBQUksRUFBRSxLQUFLO2dCQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO2dCQUM5RSxtQkFBbUIsRUFBRTtvQkFDbkIsb0JBQW9CLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsR0FBRyxFQUFFO29CQUMxRSw0QkFBNEIsRUFBRSxDQUFDLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxDQUFDO2lCQUNyRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQywrR0FBK0csQ0FBQyxDQUFDO0lBQ25JLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtRQUN2RSxPQUFPO1FBQ1AsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO2dCQUNuQyxJQUFJLEVBQUUsS0FBSztnQkFDWCxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztnQkFDOUUsbUJBQW1CLEVBQUU7b0JBQ25CLDRCQUE0QixFQUFFO3dCQUM1QixFQUFFLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLEVBQUU7d0JBQ3BELEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRTtxQkFDckQ7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsK0dBQStHLENBQUMsQ0FBQztJQUNuSSxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxJQUFJLFFBQTBCLENBQUM7SUFDL0IsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLFFBQVEsR0FBRyxJQUFJLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ2pELENBQUMsQ0FBQyxDQUFDO0lBQ0gsSUFBSSxDQUFDLHlGQUF5RixFQUFFLEdBQUcsRUFBRTtRQUNuRyxPQUFPO1FBQ1AsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDbkMsSUFBSSxFQUFFLEtBQUs7WUFDWCxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztZQUM5RSxtQkFBbUIsRUFBRTtnQkFDbkIsb0JBQW9CLEVBQUU7b0JBQ3BCLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO29CQUN0RCxjQUFjLEVBQUUsRUFBRSxRQUFRLEVBQUU7aUJBQzdCO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDBCQUEwQixFQUFFO1lBQzNELGtCQUFrQixFQUFFLDJCQUEyQjtZQUMvQyxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTtnQkFDakMsYUFBYSxFQUFFLE9BQU87Z0JBQ3RCLFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxjQUFjLEVBQUU7YUFDcEM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywrREFBK0QsRUFBRSxHQUFHLEVBQUU7UUFDekUsT0FBTztRQUNQLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxLQUFLO1lBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDOUUsbUJBQW1CLEVBQUU7Z0JBQ25CLG9CQUFvQixFQUFFO29CQUNwQixpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUztvQkFDdEQsY0FBYyxFQUFFO3dCQUNkLFFBQVE7d0JBQ1IsZ0JBQWdCLEVBQUUsTUFBTTt3QkFDeEIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJO3FCQUNsRDtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywwQkFBMEIsRUFBRTtZQUMzRCxrQkFBa0IsRUFBRSwyQkFBMkI7WUFDL0MsY0FBYyxFQUFFO2dCQUNkLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7Z0JBQ2pDLGFBQWEsRUFBRSxNQUFNO2dCQUNyQixnQkFBZ0IsRUFBRSxNQUFNO2dCQUN4QixVQUFVLEVBQUUsRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO2FBQ3BDO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNEZBQTRGLEVBQUUsR0FBRyxFQUFFO1FBQ3RHLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQiw0QkFBNEIsRUFBRSxDQUFDO3dCQUM3QixpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUzt3QkFDdEQsY0FBYyxFQUFFLEVBQUUsUUFBUSxFQUFFO3FCQUM3QixDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDBCQUEwQixFQUFFO1lBQzNELGlDQUFpQyxFQUFFLENBQUM7b0JBQ2xDLGtCQUFrQixFQUFFLDJCQUEyQjtvQkFDL0MsY0FBYyxFQUFFO3dCQUNkLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7d0JBQ2pDLFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxjQUFjLEVBQUU7cUJBQ3BDO2lCQUNGLENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw2RUFBNkUsRUFBRSxHQUFHLEVBQUU7UUFDdkYsT0FBTztRQUNQLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxLQUFLO1lBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDOUUsbUJBQW1CLEVBQUU7Z0JBQ25CLDRCQUE0QixFQUFFLENBQUM7d0JBQzdCLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO3dCQUN0RCxjQUFjLEVBQUU7NEJBQ2QsUUFBUTs0QkFDUixnQkFBZ0IsRUFBRSxNQUFNOzRCQUN4QixhQUFhLEVBQUUsT0FBTyxDQUFDLHFCQUFxQixDQUFDLElBQUk7eUJBQ2xEO3FCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7WUFDM0QsaUNBQWlDLEVBQUUsQ0FBQztvQkFDbEMsa0JBQWtCLEVBQUUsMkJBQTJCO29CQUMvQyxjQUFjLEVBQUU7d0JBQ2QsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTt3QkFDakMsZ0JBQWdCLEVBQUUsTUFBTTt3QkFDeEIsVUFBVSxFQUFFLEVBQUUsR0FBRyxFQUFFLGNBQWMsRUFBRTtxQkFDcEM7aUJBQ0YsQ0FBQztTQUNILENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDZFQUE2RSxFQUFFLEdBQUcsRUFBRTtRQUN2RixPQUFPO1FBQ1AsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDbkMsSUFBSSxFQUFFLEtBQUs7WUFDWCxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztZQUM5RSxtQkFBbUIsRUFBRTtnQkFDbkIsb0JBQW9CLEVBQUU7b0JBQ3BCLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO29CQUN0RCxjQUFjLEVBQUUsRUFBRSxRQUFRLEVBQUU7aUJBQzdCO2dCQUNELDRCQUE0QixFQUFFO29CQUM1Qjt3QkFDRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUzt3QkFDdEQsY0FBYyxFQUFFLEVBQUUsUUFBUSxFQUFFO3FCQUM3QjtvQkFDRDt3QkFDRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsU0FBUzt3QkFDdEQsY0FBYyxFQUFFOzRCQUNkLFFBQVE7NEJBQ1IsZ0JBQWdCLEVBQUUsTUFBTTs0QkFDeEIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJO3lCQUNsRDtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywwQkFBMEIsRUFBRTtZQUMzRCxrQkFBa0IsRUFBRSwyQkFBMkI7WUFDL0MsY0FBYyxFQUFFO2dCQUNkLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7Z0JBQ2pDLGFBQWEsRUFBRSxPQUFPO2dCQUN0QixVQUFVLEVBQUUsRUFBRSxHQUFHLEVBQUUsY0FBYyxFQUFFO2FBQ3BDO1lBQ0QsaUNBQWlDLEVBQUU7Z0JBQ2pDO29CQUNFLGtCQUFrQixFQUFFLDJCQUEyQjtvQkFDL0MsY0FBYyxFQUFFO3dCQUNkLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7d0JBQ2pDLFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxjQUFjLEVBQUU7cUJBQ3BDO2lCQUNGO2dCQUNEO29CQUNFLGtCQUFrQixFQUFFLDJCQUEyQjtvQkFDL0MsY0FBYyxFQUFFO3dCQUNkLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUU7d0JBQ2pDLGdCQUFnQixFQUFFLE1BQU07d0JBQ3hCLFVBQVUsRUFBRSxFQUFFLEdBQUcsRUFBRSxjQUFjLEVBQUU7cUJBQ3BDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsUUFBUSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtJQUMxQyxJQUFJLENBQUMsb0ZBQW9GLEVBQUUsR0FBRyxFQUFFO1FBQzlGLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRTtvQkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLElBQUk7b0JBQ2pELG1CQUFtQixFQUFFLEVBQUUsWUFBWSxFQUFFLE1BQU0sRUFBRTtpQkFDOUM7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7WUFDM0Qsa0JBQWtCLEVBQUUsZ0JBQWdCO1lBQ3BDLG1CQUFtQixFQUFFO2dCQUNuQixNQUFNLEVBQUUsTUFBTTthQUNmO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsK0RBQStELEVBQUUsR0FBRyxFQUFFO1FBQ3pFLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQixvQkFBb0IsRUFBRTtvQkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixDQUFDLElBQUk7b0JBQ2pELG1CQUFtQixFQUFFO3dCQUNuQixZQUFZLEVBQUUsTUFBTTt3QkFDcEIsUUFBUSxFQUFFLElBQUk7d0JBQ2QsbUJBQW1CLEVBQUUsQ0FBQzt3QkFDdEIsb0JBQW9CLEVBQUUsQ0FBQztxQkFDeEI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7WUFDM0Qsa0JBQWtCLEVBQUUsZ0JBQWdCO1lBQ3BDLG1CQUFtQixFQUFFO2dCQUNuQixPQUFPLEVBQUUsQ0FBQztnQkFDVixRQUFRLEVBQUUsSUFBSTtnQkFDZCxNQUFNLEVBQUUsQ0FBQztnQkFDVCxNQUFNLEVBQUUsTUFBTTthQUNmO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsdUZBQXVGLEVBQUUsR0FBRyxFQUFFO1FBQ2pHLE9BQU87UUFDUCxJQUFJLE9BQU8sQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRTtZQUNuQyxJQUFJLEVBQUUsS0FBSztZQUNYLE1BQU0sRUFBRSxPQUFPLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBQzlFLG1CQUFtQixFQUFFO2dCQUNuQiw0QkFBNEIsRUFBRSxDQUFDO3dCQUM3QixpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsSUFBSTt3QkFDakQsbUJBQW1CLEVBQUUsRUFBRSxZQUFZLEVBQUUsTUFBTSxFQUFFO3FCQUM5QyxDQUFDO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDBCQUEwQixFQUFFO1lBQzNELGlDQUFpQyxFQUFFLENBQUM7b0JBQ2xDLGtCQUFrQixFQUFFLGdCQUFnQjtvQkFDcEMsbUJBQW1CLEVBQUU7d0JBQ25CLE1BQU0sRUFBRSxNQUFNO3FCQUNmO2lCQUNGLENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxrRUFBa0UsRUFBRSxHQUFHLEVBQUU7UUFDNUUsT0FBTztRQUNQLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxFQUFFO1lBQ25DLElBQUksRUFBRSxLQUFLO1lBQ1gsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFDOUUsbUJBQW1CLEVBQUU7Z0JBQ25CLDRCQUE0QixFQUFFLENBQUM7d0JBQzdCLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJO3dCQUNqRCxtQkFBbUIsRUFBRTs0QkFDbkIsWUFBWSxFQUFFLE1BQU07NEJBQ3BCLFFBQVEsRUFBRSxJQUFJOzRCQUNkLG1CQUFtQixFQUFFLENBQUM7NEJBQ3RCLG9CQUFvQixFQUFFLENBQUM7eUJBQ3hCO3FCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsMEJBQTBCLEVBQUU7WUFDM0QsaUNBQWlDLEVBQUUsQ0FBQztvQkFDbEMsa0JBQWtCLEVBQUUsZ0JBQWdCO29CQUNwQyxtQkFBbUIsRUFBRTt3QkFDbkIsT0FBTyxFQUFFLENBQUM7d0JBQ1YsUUFBUSxFQUFFLElBQUk7d0JBQ2QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsTUFBTSxFQUFFLE1BQU07cUJBQ2Y7aUJBQ0YsQ0FBQztTQUNILENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHFFQUFxRSxFQUFFLEdBQUcsRUFBRTtRQUMvRSxPQUFPO1FBQ1AsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxLQUFLLEVBQUU7WUFDbkMsSUFBSSxFQUFFLEtBQUs7WUFDWCxNQUFNLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztZQUM5RSxtQkFBbUIsRUFBRTtnQkFDbkIsb0JBQW9CLEVBQUU7b0JBQ3BCLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJO29CQUNqRCxtQkFBbUIsRUFBRSxFQUFFLFlBQVksRUFBRSxNQUFNLEVBQUU7aUJBQzlDO2dCQUNELDRCQUE0QixFQUFFO29CQUM1Qjt3QkFDRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsSUFBSTt3QkFDakQsbUJBQW1CLEVBQUU7NEJBQ25CLFlBQVksRUFBRSxPQUFPOzRCQUNyQixRQUFRLEVBQUUsSUFBSTs0QkFDZCxtQkFBbUIsRUFBRSxDQUFDOzRCQUN0QixvQkFBb0IsRUFBRSxDQUFDO3lCQUN4QjtxQkFDRjtvQkFDRDt3QkFDRSxpQkFBaUIsRUFBRSxPQUFPLENBQUMsaUJBQWlCLENBQUMsSUFBSTt3QkFDakQsbUJBQW1CLEVBQUU7NEJBQ25CLFlBQVksRUFBRSxPQUFPOzRCQUNyQixRQUFRLEVBQUUsSUFBSTs0QkFDZCxtQkFBbUIsRUFBRSxDQUFDOzRCQUN0QixvQkFBb0IsRUFBRSxDQUFDO3lCQUN4QjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywwQkFBMEIsRUFBRTtZQUMzRCxrQkFBa0IsRUFBRSxnQkFBZ0I7WUFDcEMsbUJBQW1CLEVBQUUsRUFBRSxNQUFNLEVBQUUsTUFBTSxFQUFFO1lBQ3ZDLGlDQUFpQyxFQUFFO2dCQUNqQztvQkFDRSxrQkFBa0IsRUFBRSxnQkFBZ0I7b0JBQ3BDLG1CQUFtQixFQUFFO3dCQUNuQixPQUFPLEVBQUUsQ0FBQzt3QkFDVixRQUFRLEVBQUUsSUFBSTt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxNQUFNLEVBQUUsT0FBTztxQkFDaEI7aUJBQ0Y7Z0JBQ0Q7b0JBQ0Usa0JBQWtCLEVBQUUsZ0JBQWdCO29CQUNwQyxtQkFBbUIsRUFBRTt3QkFDbkIsT0FBTyxFQUFFLENBQUM7d0JBQ1YsUUFBUSxFQUFFLElBQUk7d0JBQ2QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsTUFBTSxFQUFFLE9BQU87cUJBQ2hCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgY29nbml0byBmcm9tICdAYXdzLWNkay9hd3MtY29nbml0byc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBhcHBzeW5jIGZyb20gJy4uL2xpYic7XG5cbi8vIEdJVkVOXG5sZXQgc3RhY2s6IGNkay5TdGFjaztcbmJlZm9yZUVhY2goKCkgPT4ge1xuICBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbn0pO1xuXG5kZXNjcmliZSgnQXBwU3luYyBBUEkgS2V5IEF1dGhvcml6YXRpb24nLCAoKSA9PiB7XG4gIHRlc3QoJ0FwcFN5bmMgY3JlYXRlcyBkZWZhdWx0IGFwaSBrZXknLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkFwcFN5bmM6OkFwaUtleScpO1xuICB9KTtcblxuICB0ZXN0KCdBcHBTeW5jIGNyZWF0ZXMgYXBpIGtleSBmcm9tIGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXMnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHsgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuSUFNIH0sXG4gICAgICAgIGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXM6IFtcbiAgICAgICAgICB7IGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpBcHBTeW5jOjpBcGlLZXknKTtcbiAgfSk7XG5cbiAgdGVzdCgnQXBwU3luYyBkb2VzIG5vdCBjcmVhdGUgdW5zcGVjaWZpZWQgYXBpIGtleSBmcm9tIGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXMnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHsgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuSUFNIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykubm90LnRvSGF2ZVJlc291cmNlKCdBV1M6OkFwcFN5bmM6OkFwaUtleScpO1xuICB9KTtcblxuICB0ZXN0KCdhcHBzeW5jIGRvZXMgbm90IGNyZWF0ZSB1bnNwZWNpZmllZCBhcGkga2V5IHdpdGggZW1wdHkgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcycsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjogeyBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0gfSxcbiAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW10sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykubm90LnRvSGF2ZVJlc291cmNlKCdBV1M6OkFwcFN5bmM6OkFwaUtleScpO1xuICB9KTtcblxuICB0ZXN0KCdhcHBzeW5jIGNyZWF0ZXMgY29uZmlndXJlZCBhcGkga2V5IHdpdGggYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlcycsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjogeyBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0gfSxcbiAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW3tcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5BUElfS0VZLFxuICAgICAgICAgIGFwaUtleUNvbmZpZzogeyBkZXNjcmlwdGlvbjogJ0N1c3RvbSBEZXNjcmlwdGlvbicgfSxcbiAgICAgICAgfV0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkFwaUtleScsIHtcbiAgICAgIERlc2NyaXB0aW9uOiAnQ3VzdG9tIERlc2NyaXB0aW9uJyxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnYXBpS2V5Q29uZmlnIGNyZWF0ZXMgZGVmYXVsdCB3aXRoIHZhbGlkIGV4cGlyYXRpb24gZGF0ZScsICgpID0+IHtcbiAgICBjb25zdCBleHBpcmF0aW9uRGF0ZTogbnVtYmVyID0gY2RrLkV4cGlyYXRpb24uYWZ0ZXIoY2RrLkR1cmF0aW9uLmRheXMoMTApKS50b0Vwb2NoKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ0FQSScsIHtcbiAgICAgIG5hbWU6ICdhcGlLZXlVbml0VGVzdCcsXG4gICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy5hdXRoLmdyYXBocWwnKSksXG4gICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiB7XG4gICAgICAgIGRlZmF1bHRBdXRob3JpemF0aW9uOiB7XG4gICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSxcbiAgICAgICAgICBhcGlLZXlDb25maWc6IHtcbiAgICAgICAgICAgIGV4cGlyZXM6IGNkay5FeHBpcmF0aW9uLmFmdGVyKGNkay5EdXJhdGlvbi5kYXlzKDEwKSksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkFwaUtleScsIHtcbiAgICAgIEFwaUlkOiB7ICdGbjo6R2V0QXR0JzogWydBUEk2MkVBMUNGRicsICdBcGlJZCddIH0sXG4gICAgICBFeHBpcmVzOiBleHBpcmF0aW9uRGF0ZSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnYXBpS2V5Q29uZmlnIGZhaWxzIGlmIGV4cGlyZSBhcmd1bWVudCBsZXNzIHRoYW4gYSBkYXknLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHdoZW4gPSAoKSA9PiB7XG4gICAgICBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnQVBJJywge1xuICAgICAgICBuYW1lOiAnYXBpS2V5VW5pdFRlc3QnLFxuICAgICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy5hdXRoLmdyYXBocWwnKSksXG4gICAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjoge1xuICAgICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSxcbiAgICAgICAgICAgIGFwaUtleUNvbmZpZzoge1xuICAgICAgICAgICAgICBleHBpcmVzOiBjZGsuRXhwaXJhdGlvbi5hZnRlcihjZGsuRHVyYXRpb24uaG91cnMoMSkpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qod2hlbikudG9UaHJvd0Vycm9yKCdBUEkga2V5IGV4cGlyYXRpb24gbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDM2NSBkYXlzLicpO1xuICB9KTtcblxuICB0ZXN0KCdhcGlLZXlDb25maWcgZmFpbHMgaWYgZXhwaXJlIGFyZ3VtZW50IGdyZWF0ZXIgdGhhbiAzNjUgZGF5JywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCB3aGVuID0gKCkgPT4ge1xuICAgICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ0FQSScsIHtcbiAgICAgICAgbmFtZTogJ2FwaUtleVVuaXRUZXN0JyxcbiAgICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMuYXV0aC5ncmFwaHFsJykpLFxuICAgICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiB7XG4gICAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHtcbiAgICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVksXG4gICAgICAgICAgICBhcGlLZXlDb25maWc6IHtcbiAgICAgICAgICAgICAgZXhwaXJlczogY2RrLkV4cGlyYXRpb24uYWZ0ZXIoY2RrLkR1cmF0aW9uLmRheXMoMzY2KSksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9O1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCh3aGVuKS50b1Rocm93RXJyb3IoJ0FQSSBrZXkgZXhwaXJhdGlvbiBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMzY1IGRheXMuJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FwcHN5bmMgY3JlYXRlcyBjb25maWd1cmVkIGFwaSBrZXkgd2l0aCBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzIChub3QgYXMgZmlyc3QgZWxlbWVudCknLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHsgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuSUFNIH0sXG4gICAgICAgIGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wsXG4gICAgICAgICAgICB1c2VyUG9vbENvbmZpZzogeyB1c2VyUG9vbDogbmV3IGNvZ25pdG8uVXNlclBvb2woc3RhY2ssICdteVBvb2wnKSB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSxcbiAgICAgICAgICAgIGFwaUtleUNvbmZpZzogeyBkZXNjcmlwdGlvbjogJ0N1c3RvbSBEZXNjcmlwdGlvbicgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpBcGlLZXknLCB7XG4gICAgICBEZXNjcmlwdGlvbjogJ0N1c3RvbSBEZXNjcmlwdGlvbicsXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FwcHN5bmMgZmFpbHMgd2hlbiBlbXB0eSBkZWZhdWx0IGFuZCBBUElfS0VZIGluIGFkZGl0aW9uYWwnLCAoKSA9PiB7XG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnYXBpJywge1xuICAgICAgICBuYW1lOiAnYXBpJyxcbiAgICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiB7XG4gICAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW3tcbiAgICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVksXG4gICAgICAgICAgfV0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93RXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBBUElfS0VZIGNvbmZpZ3VyYXRpb24uIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbCcpO1xuICB9KTtcblxuICB0ZXN0KCdhcHBzeW5jIGZhaWxzIHdoZW4gbXVsdGlwbGUgQVBJX0tFWSBhdXRoIG1vZGVzJywgKCkgPT4ge1xuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgICAgbmFtZTogJ2FwaScsXG4gICAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICAgIGRlZmF1bHRBdXRob3JpemF0aW9uOiB7IGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkgfSxcbiAgICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbe1xuICAgICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSxcbiAgICAgICAgICB9XSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3dFcnJvcignWW91IGNhblxcJ3QgZHVwbGljYXRlIEFQSV9LRVkgY29uZmlndXJhdGlvbi4gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9zZWN1cml0eS5odG1sJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FwcHN5bmMgZmFpbHMgd2hlbiBtdWx0aXBsZSBBUElfS0VZIGF1dGggbW9kZXMgaW4gYWRkaXRpb25hbFh4eCcsICgpID0+IHtcbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy50ZXN0LmdyYXBocWwnKSksXG4gICAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjogeyBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0gfSxcbiAgICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbXG4gICAgICAgICAgICB7IGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkgfSxcbiAgICAgICAgICAgIHsgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuQVBJX0tFWSB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93RXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBBUElfS0VZIGNvbmZpZ3VyYXRpb24uIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvc2VjdXJpdHkuaHRtbCcpO1xuICB9KTtcbn0pO1xuXG5kZXNjcmliZSgnQXBwU3luYyBJQU0gQXV0aG9yaXphdGlvbicsICgpID0+IHtcbiAgdGVzdCgnSWFtIGF1dGhvcml6YXRpb24gY29uZmlndXJhYmxlIGluIGRlZmF1bHQgYXV0aG9yaXphdGlvbicsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjogeyBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0gfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTEFwaScsIHtcbiAgICAgIEF1dGhlbnRpY2F0aW9uVHlwZTogJ0FXU19JQU0nLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdJYW0gYXV0aG9yaXphdGlvbiBjb25maWd1cmFibGUgaW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnYXBpJywge1xuICAgICAgbmFtZTogJ2FwaScsXG4gICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy50ZXN0LmdyYXBocWwnKSksXG4gICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiB7XG4gICAgICAgIGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXM6IFt7IGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLklBTSB9XSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTEFwaScsIHtcbiAgICAgIEFkZGl0aW9uYWxBdXRoZW50aWNhdGlvblByb3ZpZGVyczogW3sgQXV0aGVudGljYXRpb25UeXBlOiAnQVdTX0lBTScgfV0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FwcHN5bmMgZmFpbHMgd2hlbiBtdWx0aXBsZSBpYW0gYXV0aCBtb2RlcycsICgpID0+IHtcbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy50ZXN0LmdyYXBocWwnKSksXG4gICAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjogeyBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0gfSxcbiAgICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbeyBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5JQU0gfV0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93RXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBJQU0gY29uZmlndXJhdGlvbi4gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9zZWN1cml0eS5odG1sJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FwcHN5bmMgZmFpbHMgd2hlbiBtdWx0aXBsZSBJQU0gYXV0aCBtb2RlcyBpbiBhZGRpdGlvbmFsWHh4JywgKCkgPT4ge1xuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgICAgbmFtZTogJ2FwaScsXG4gICAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICAgIGFkZGl0aW9uYWxBdXRob3JpemF0aW9uTW9kZXM6IFtcbiAgICAgICAgICAgIHsgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuSUFNIH0sXG4gICAgICAgICAgICB7IGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLklBTSB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93RXJyb3IoJ1lvdSBjYW5cXCd0IGR1cGxpY2F0ZSBJQU0gY29uZmlndXJhdGlvbi4gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9zZWN1cml0eS5odG1sJyk7XG4gIH0pO1xufSk7XG5cbmRlc2NyaWJlKCdBcHBTeW5jIFVzZXIgUG9vbCBBdXRob3JpemF0aW9uJywgKCkgPT4ge1xuICBsZXQgdXNlclBvb2w6IGNvZ25pdG8uVXNlclBvb2w7XG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIHVzZXJQb29sID0gbmV3IGNvZ25pdG8uVXNlclBvb2woc3RhY2ssICdwb29sJyk7XG4gIH0pO1xuICB0ZXN0KCdVc2VyIFBvb2wgYXV0aG9yaXphdGlvbiBjb25maWd1cmFibGUgaW4gZGVmYXVsdCBhdXRob3JpemF0aW9uIGhhcyBkZWZhdWx0IGNvbmZpZ3VyYXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHtcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wsXG4gICAgICAgICAgdXNlclBvb2xDb25maWc6IHsgdXNlclBvb2wgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTEFwaScsIHtcbiAgICAgIEF1dGhlbnRpY2F0aW9uVHlwZTogJ0FNQVpPTl9DT0dOSVRPX1VTRVJfUE9PTFMnLFxuICAgICAgVXNlclBvb2xDb25maWc6IHtcbiAgICAgICAgQXdzUmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICBEZWZhdWx0QWN0aW9uOiAnQUxMT1cnLFxuICAgICAgICBVc2VyUG9vbElkOiB7IFJlZjogJ3Bvb2wwNTZGM0Y3RScgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ1VzZXIgUG9vbCBhdXRob3JpemF0aW9uIGNvbmZpZ3VyYWJsZSBpbiBkZWZhdWx0IGF1dGhvcml6YXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHtcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5VU0VSX1BPT0wsXG4gICAgICAgICAgdXNlclBvb2xDb25maWc6IHtcbiAgICAgICAgICAgIHVzZXJQb29sLFxuICAgICAgICAgICAgYXBwSWRDbGllbnRSZWdleDogJ3Rlc3QnLFxuICAgICAgICAgICAgZGVmYXVsdEFjdGlvbjogYXBwc3luYy5Vc2VyUG9vbERlZmF1bHRBY3Rpb24uREVOWSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMQXBpJywge1xuICAgICAgQXV0aGVudGljYXRpb25UeXBlOiAnQU1BWk9OX0NPR05JVE9fVVNFUl9QT09MUycsXG4gICAgICBVc2VyUG9vbENvbmZpZzoge1xuICAgICAgICBBd3NSZWdpb246IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgIERlZmF1bHRBY3Rpb246ICdERU5ZJyxcbiAgICAgICAgQXBwSWRDbGllbnRSZWdleDogJ3Rlc3QnLFxuICAgICAgICBVc2VyUG9vbElkOiB7IFJlZjogJ3Bvb2wwNTZGM0Y3RScgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ1VzZXIgUG9vbCBhdXRob3JpemF0aW9uIGNvbmZpZ3VyYWJsZSBpbiBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24gaGFzIGRlZmF1bHQgY29uZmlndXJhdGlvbicsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbe1xuICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCxcbiAgICAgICAgICB1c2VyUG9vbENvbmZpZzogeyB1c2VyUG9vbCB9LFxuICAgICAgICB9XSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6QXBwU3luYzo6R3JhcGhRTEFwaScsIHtcbiAgICAgIEFkZGl0aW9uYWxBdXRoZW50aWNhdGlvblByb3ZpZGVyczogW3tcbiAgICAgICAgQXV0aGVudGljYXRpb25UeXBlOiAnQU1BWk9OX0NPR05JVE9fVVNFUl9QT09MUycsXG4gICAgICAgIFVzZXJQb29sQ29uZmlnOiB7XG4gICAgICAgICAgQXdzUmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgIFVzZXJQb29sSWQ6IHsgUmVmOiAncG9vbDA1NkYzRjdFJyB9LFxuICAgICAgICB9LFxuICAgICAgfV0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ1VzZXIgUG9vbCBwcm9wZXJ0eSBkZWZhdWx0QWN0aW9uIGRvZXMgbm90IGNvbmZpZ3VyZSB3aGVuIGluIGFkZGl0aW9uYWwgYXV0aCcsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbe1xuICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCxcbiAgICAgICAgICB1c2VyUG9vbENvbmZpZzoge1xuICAgICAgICAgICAgdXNlclBvb2wsXG4gICAgICAgICAgICBhcHBJZENsaWVudFJlZ2V4OiAndGVzdCcsXG4gICAgICAgICAgICBkZWZhdWx0QWN0aW9uOiBhcHBzeW5jLlVzZXJQb29sRGVmYXVsdEFjdGlvbi5ERU5ZLFxuICAgICAgICAgIH0sXG4gICAgICAgIH1dLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMQXBpJywge1xuICAgICAgQWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiBbe1xuICAgICAgICBBdXRoZW50aWNhdGlvblR5cGU6ICdBTUFaT05fQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgICAgICAgVXNlclBvb2xDb25maWc6IHtcbiAgICAgICAgICBBd3NSZWdpb246IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgICAgQXBwSWRDbGllbnRSZWdleDogJ3Rlc3QnLFxuICAgICAgICAgIFVzZXJQb29sSWQ6IHsgUmVmOiAncG9vbDA1NkYzRjdFJyB9LFxuICAgICAgICB9LFxuICAgICAgfV0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ1VzZXIgUG9vbCBwcm9wZXJ0eSBkZWZhdWx0QWN0aW9uIGRvZXMgbm90IGNvbmZpZ3VyZSB3aGVuIGluIGFkZGl0aW9uYWwgYXV0aCcsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjoge1xuICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCxcbiAgICAgICAgICB1c2VyUG9vbENvbmZpZzogeyB1c2VyUG9vbCB9LFxuICAgICAgICB9LFxuICAgICAgICBhZGRpdGlvbmFsQXV0aG9yaXphdGlvbk1vZGVzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MLFxuICAgICAgICAgICAgdXNlclBvb2xDb25maWc6IHsgdXNlclBvb2wgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCxcbiAgICAgICAgICAgIHVzZXJQb29sQ29uZmlnOiB7XG4gICAgICAgICAgICAgIHVzZXJQb29sLFxuICAgICAgICAgICAgICBhcHBJZENsaWVudFJlZ2V4OiAndGVzdCcsXG4gICAgICAgICAgICAgIGRlZmF1bHRBY3Rpb246IGFwcHN5bmMuVXNlclBvb2xEZWZhdWx0QWN0aW9uLkRFTlksXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkdyYXBoUUxBcGknLCB7XG4gICAgICBBdXRoZW50aWNhdGlvblR5cGU6ICdBTUFaT05fQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgICAgIFVzZXJQb29sQ29uZmlnOiB7XG4gICAgICAgIEF3c1JlZ2lvbjogeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSxcbiAgICAgICAgRGVmYXVsdEFjdGlvbjogJ0FMTE9XJyxcbiAgICAgICAgVXNlclBvb2xJZDogeyBSZWY6ICdwb29sMDU2RjNGN0UnIH0sXG4gICAgICB9LFxuICAgICAgQWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBdXRoZW50aWNhdGlvblR5cGU6ICdBTUFaT05fQ09HTklUT19VU0VSX1BPT0xTJyxcbiAgICAgICAgICBVc2VyUG9vbENvbmZpZzoge1xuICAgICAgICAgICAgQXdzUmVnaW9uOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICAgICAgVXNlclBvb2xJZDogeyBSZWY6ICdwb29sMDU2RjNGN0UnIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIEF1dGhlbnRpY2F0aW9uVHlwZTogJ0FNQVpPTl9DT0dOSVRPX1VTRVJfUE9PTFMnLFxuICAgICAgICAgIFVzZXJQb29sQ29uZmlnOiB7XG4gICAgICAgICAgICBBd3NSZWdpb246IHsgUmVmOiAnQVdTOjpSZWdpb24nIH0sXG4gICAgICAgICAgICBBcHBJZENsaWVudFJlZ2V4OiAndGVzdCcsXG4gICAgICAgICAgICBVc2VyUG9vbElkOiB7IFJlZjogJ3Bvb2wwNTZGM0Y3RScgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG59KTtcblxuZGVzY3JpYmUoJ0FwcFN5bmMgT0lEQyBBdXRob3JpemF0aW9uJywgKCkgPT4ge1xuICB0ZXN0KCdPSURDIGF1dGhvcml6YXRpb24gY29uZmlndXJhYmxlIGluIGRlZmF1bHQgYXV0aG9yaXphdGlvbiBoYXMgZGVmYXVsdCBjb25maWd1cmF0aW9uJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBuZXcgYXBwc3luYy5HcmFwaHFsQXBpKHN0YWNrLCAnYXBpJywge1xuICAgICAgbmFtZTogJ2FwaScsXG4gICAgICBzY2hlbWE6IGFwcHN5bmMuU2NoZW1hLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnYXBwc3luYy50ZXN0LmdyYXBocWwnKSksXG4gICAgICBhdXRob3JpemF0aW9uQ29uZmlnOiB7XG4gICAgICAgIGRlZmF1bHRBdXRob3JpemF0aW9uOiB7XG4gICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwcHN5bmMuQXV0aG9yaXphdGlvblR5cGUuT0lEQyxcbiAgICAgICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB7IG9pZGNQcm92aWRlcjogJ3Rlc3QnIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkdyYXBoUUxBcGknLCB7XG4gICAgICBBdXRoZW50aWNhdGlvblR5cGU6ICdPUEVOSURfQ09OTkVDVCcsXG4gICAgICBPcGVuSURDb25uZWN0Q29uZmlnOiB7XG4gICAgICAgIElzc3VlcjogJ3Rlc3QnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnVXNlciBQb29sIGF1dGhvcml6YXRpb24gY29uZmlndXJhYmxlIGluIGRlZmF1bHQgYXV0aG9yaXphdGlvbicsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgbmV3IGFwcHN5bmMuR3JhcGhxbEFwaShzdGFjaywgJ2FwaScsIHtcbiAgICAgIG5hbWU6ICdhcGknLFxuICAgICAgc2NoZW1hOiBhcHBzeW5jLlNjaGVtYS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2FwcHN5bmMudGVzdC5ncmFwaHFsJykpLFxuICAgICAgYXV0aG9yaXphdGlvbkNvbmZpZzoge1xuICAgICAgICBkZWZhdWx0QXV0aG9yaXphdGlvbjoge1xuICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLk9JREMsXG4gICAgICAgICAgb3BlbklkQ29ubmVjdENvbmZpZzoge1xuICAgICAgICAgICAgb2lkY1Byb3ZpZGVyOiAndGVzdCcsXG4gICAgICAgICAgICBjbGllbnRJZDogJ2lkJyxcbiAgICAgICAgICAgIHRva2VuRXhwaXJ5RnJvbUF1dGg6IDEsXG4gICAgICAgICAgICB0b2tlbkV4cGlyeUZyb21Jc3N1ZTogMSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMQXBpJywge1xuICAgICAgQXV0aGVudGljYXRpb25UeXBlOiAnT1BFTklEX0NPTk5FQ1QnLFxuICAgICAgT3BlbklEQ29ubmVjdENvbmZpZzoge1xuICAgICAgICBBdXRoVFRMOiAxLFxuICAgICAgICBDbGllbnRJZDogJ2lkJyxcbiAgICAgICAgSWF0VFRMOiAxLFxuICAgICAgICBJc3N1ZXI6ICd0ZXN0JyxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ09JREMgYXV0aG9yaXphdGlvbiBjb25maWd1cmFibGUgaW4gYWRkaXRpb25hbCBhdXRob3JpemF0aW9uIGhhcyBkZWZhdWx0IGNvbmZpZ3VyYXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW3tcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5PSURDLFxuICAgICAgICAgIG9wZW5JZENvbm5lY3RDb25maWc6IHsgb2lkY1Byb3ZpZGVyOiAndGVzdCcgfSxcbiAgICAgICAgfV0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkdyYXBoUUxBcGknLCB7XG4gICAgICBBZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlcnM6IFt7XG4gICAgICAgIEF1dGhlbnRpY2F0aW9uVHlwZTogJ09QRU5JRF9DT05ORUNUJyxcbiAgICAgICAgT3BlbklEQ29ubmVjdENvbmZpZzoge1xuICAgICAgICAgIElzc3VlcjogJ3Rlc3QnLFxuICAgICAgICB9LFxuICAgICAgfV0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ1VzZXIgUG9vbCBhdXRob3JpemF0aW9uIGNvbmZpZ3VyYWJsZSBpbiBhZGRpdGlvbmFsIGF1dGhvcml6YXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW3tcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5PSURDLFxuICAgICAgICAgIG9wZW5JZENvbm5lY3RDb25maWc6IHtcbiAgICAgICAgICAgIG9pZGNQcm92aWRlcjogJ3Rlc3QnLFxuICAgICAgICAgICAgY2xpZW50SWQ6ICdpZCcsXG4gICAgICAgICAgICB0b2tlbkV4cGlyeUZyb21BdXRoOiAxLFxuICAgICAgICAgICAgdG9rZW5FeHBpcnlGcm9tSXNzdWU6IDEsXG4gICAgICAgICAgfSxcbiAgICAgICAgfV0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkFwcFN5bmM6OkdyYXBoUUxBcGknLCB7XG4gICAgICBBZGRpdGlvbmFsQXV0aGVudGljYXRpb25Qcm92aWRlcnM6IFt7XG4gICAgICAgIEF1dGhlbnRpY2F0aW9uVHlwZTogJ09QRU5JRF9DT05ORUNUJyxcbiAgICAgICAgT3BlbklEQ29ubmVjdENvbmZpZzoge1xuICAgICAgICAgIEF1dGhUVEw6IDEsXG4gICAgICAgICAgQ2xpZW50SWQ6ICdpZCcsXG4gICAgICAgICAgSWF0VFRMOiAxLFxuICAgICAgICAgIElzc3VlcjogJ3Rlc3QnLFxuICAgICAgICB9LFxuICAgICAgfV0sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ1VzZXIgUG9vbCBhdXRob3JpemF0aW9uIGNvbmZpZ3VyYWJsZSBpbiB3aXRoIG11bHRpcGxlIGF1dGhvcml6YXRpb24nLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIG5ldyBhcHBzeW5jLkdyYXBocWxBcGkoc3RhY2ssICdhcGknLCB7XG4gICAgICBuYW1lOiAnYXBpJyxcbiAgICAgIHNjaGVtYTogYXBwc3luYy5TY2hlbWEuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdhcHBzeW5jLnRlc3QuZ3JhcGhxbCcpKSxcbiAgICAgIGF1dGhvcml6YXRpb25Db25maWc6IHtcbiAgICAgICAgZGVmYXVsdEF1dGhvcml6YXRpb246IHtcbiAgICAgICAgICBhdXRob3JpemF0aW9uVHlwZTogYXBwc3luYy5BdXRob3JpemF0aW9uVHlwZS5PSURDLFxuICAgICAgICAgIG9wZW5JZENvbm5lY3RDb25maWc6IHsgb2lkY1Byb3ZpZGVyOiAndGVzdCcgfSxcbiAgICAgICAgfSxcbiAgICAgICAgYWRkaXRpb25hbEF1dGhvcml6YXRpb25Nb2RlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLk9JREMsXG4gICAgICAgICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB7XG4gICAgICAgICAgICAgIG9pZGNQcm92aWRlcjogJ3Rlc3QxJyxcbiAgICAgICAgICAgICAgY2xpZW50SWQ6ICdpZCcsXG4gICAgICAgICAgICAgIHRva2VuRXhwaXJ5RnJvbUF1dGg6IDEsXG4gICAgICAgICAgICAgIHRva2VuRXhwaXJ5RnJvbUlzc3VlOiAxLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGF1dGhvcml6YXRpb25UeXBlOiBhcHBzeW5jLkF1dGhvcml6YXRpb25UeXBlLk9JREMsXG4gICAgICAgICAgICBvcGVuSWRDb25uZWN0Q29uZmlnOiB7XG4gICAgICAgICAgICAgIG9pZGNQcm92aWRlcjogJ3Rlc3QyJyxcbiAgICAgICAgICAgICAgY2xpZW50SWQ6ICdpZCcsXG4gICAgICAgICAgICAgIHRva2VuRXhwaXJ5RnJvbUF1dGg6IDEsXG4gICAgICAgICAgICAgIHRva2VuRXhwaXJ5RnJvbUlzc3VlOiAxLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpBcHBTeW5jOjpHcmFwaFFMQXBpJywge1xuICAgICAgQXV0aGVudGljYXRpb25UeXBlOiAnT1BFTklEX0NPTk5FQ1QnLFxuICAgICAgT3BlbklEQ29ubmVjdENvbmZpZzogeyBJc3N1ZXI6ICd0ZXN0JyB9LFxuICAgICAgQWRkaXRpb25hbEF1dGhlbnRpY2F0aW9uUHJvdmlkZXJzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBBdXRoZW50aWNhdGlvblR5cGU6ICdPUEVOSURfQ09OTkVDVCcsXG4gICAgICAgICAgT3BlbklEQ29ubmVjdENvbmZpZzoge1xuICAgICAgICAgICAgQXV0aFRUTDogMSxcbiAgICAgICAgICAgIENsaWVudElkOiAnaWQnLFxuICAgICAgICAgICAgSWF0VFRMOiAxLFxuICAgICAgICAgICAgSXNzdWVyOiAndGVzdDEnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHtcbiAgICAgICAgICBBdXRoZW50aWNhdGlvblR5cGU6ICdPUEVOSURfQ09OTkVDVCcsXG4gICAgICAgICAgT3BlbklEQ29ubmVjdENvbmZpZzoge1xuICAgICAgICAgICAgQXV0aFRUTDogMSxcbiAgICAgICAgICAgIENsaWVudElkOiAnaWQnLFxuICAgICAgICAgICAgSWF0VFRMOiAxLFxuICAgICAgICAgICAgSXNzdWVyOiAndGVzdDInLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pO1xuICB9KTtcbn0pOyJdfQ==