"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnumType = exports.UnionType = exports.InputType = exports.ObjectType = exports.InterfaceType = void 0;
const private_1 = require("./private");
const schema_field_1 = require("./schema-field");
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
        this.directives = props.directives;
    }
    /**
     * Create a GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'interface',
            name: this.name,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Interface Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
            directives: props.directives,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Object Types must have both fieldName and field options.');
        }
        this.generateResolver(options.fieldName, options.field.fieldOptions);
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'type',
            name: this.name,
            interfaceTypes: this.interfaceTypes,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * Generate the resolvers linked to this Object Type
     */
    generateResolver(fieldName, options) {
        if (!(options === null || options === void 0 ? void 0 : options.dataSource))
            return;
        if (!this.resolvers) {
            this.resolvers = [];
        }
        this.resolvers.push(options.dataSource.createResolver({
            typeName: this.name,
            fieldName: fieldName,
            pipelineConfig: options.pipelineConfig,
            requestMappingTemplate: options.requestMappingTemplate,
            responseMappingTemplate: options.responseMappingTemplate,
        }));
    }
}
exports.ObjectType = ObjectType;
/**
 * Input Types are abstract types that define complex objects.
 * They are used in arguments to represent
 *
 * @experimental
 */
class InputType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create a GraphQL Type representing this Input Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this input type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'input',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
            modes: this.modes,
        });
    }
    /**
     * Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Input Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InputType = InputType;
/**
 * Union Types are abstract types that are similar to Interface Types,
 * but they cannot to specify any common fields between types.
 *
 * Note that fields of a union type need to be object types. In other words,
 * you can't create a union type out of interfaces, other unions, or inputs.
 *
 * @experimental
 */
class UnionType {
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((def) => this.addField({ field: def.attribute() }));
    }
    /**
     * Create a GraphQL Type representing this Union Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this Union type
     */
    toString() {
        // Return a string that appends all Object Types for this Union Type
        // i.e. 'union Example = example1 | example2'
        return Object.values(this.definition).reduce((acc, field) => `${acc} ${field.toString()} |`, `union ${this.name} =`).slice(0, -2);
    }
    /**
     * Add a field to this Union Type
     *
     * Input Types must have field options and the IField must be an Object Type.
     *
     * @param options the options to add a field
     */
    addField(options) {
        var _a;
        if (options.fieldName) {
            throw new Error('Union Types cannot be configured with the fieldName option. Use the field option instead.');
        }
        if (!options.field) {
            throw new Error('Union Types must be configured with the field option.');
        }
        if (options.field && !(options.field.intermediateType instanceof ObjectType)) {
            throw new Error('Fields for Union Types must be Object Types.');
        }
        this.definition[((_a = options.field) === null || _a === void 0 ? void 0 : _a.toString()) + 'id'] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.UnionType = UnionType;
/**
 * Enum Types are abstract types that includes a set of fields
 * that represent the strings this type can create.
 *
 * @experimental
 */
class EnumType {
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((fieldName) => this.addField({ fieldName }));
    }
    /**
     * Create an GraphQL Type representing this Enum Type
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this enum type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'enum',
            name: this.name,
            fields: Object.keys(this.definition),
            modes: this.modes,
        });
    }
    /**
     * Add a field to this Enum Type
     *
     * To add a field to this Enum Type, you must only configure
     * addField with the fieldName options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (options.field) {
            throw new Error('Enum Type fields consist of strings. Use the fieldName option instead of the field option.');
        }
        if (!options.fieldName) {
            throw new Error('When adding a field to an Enum Type, you must configure the fieldName option.');
        }
        if (options.fieldName.indexOf(' ') > -1) {
            throw new Error(`Enum Type values cannot have whitespace. Received: ${options.fieldName}`);
        }
        this.definition[options.fieldName] = schema_field_1.GraphqlType.string();
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.EnumType = EnumType;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZW1hLWludGVybWVkaWF0ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNjaGVtYS1pbnRlcm1lZGlhdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsdUNBQTBDO0FBRzFDLGlEQUFzRjtBQXdCdEY7Ozs7O0dBS0c7QUFDSCxNQUFhLGFBQWE7SUFvQnhCLFlBQW1CLElBQVksRUFBRSxLQUE4QjtRQUM3RCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksU0FBUyxDQUFDLE9BQXlCO1FBQ3hDLE9BQU8sMEJBQVcsQ0FBQyxZQUFZLENBQUM7WUFDOUIsTUFBTSxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxNQUFNO1lBQ3ZCLFVBQVUsRUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsVUFBVTtZQUMvQixjQUFjLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLGNBQWM7WUFDdkMsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRO1FBQ2IsT0FBTyx1QkFBYSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxXQUFXO1lBQ25CLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNmLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTtZQUMzQixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7Z0JBQy9DLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQ25DLE9BQU8sR0FBRyxHQUFHLEdBQUcsS0FBSyxDQUFDLFlBQVksRUFBRSxLQUFLLEtBQUssQ0FBQyxRQUFRLEVBQUUsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDckcsQ0FBQyxDQUFDO1lBQ0YsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLO1NBQ2xCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxRQUFRLENBQUMsT0FBd0I7UUFDdEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztTQUNoRjtRQUNELElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxpQkFBaUIsQ0FBQyxHQUFlO1FBQ3RDLElBQUksQ0FBQyxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQztRQUN2QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Q0FDRjtBQS9FRCxzQ0ErRUM7QUFxQkQ7Ozs7R0FJRztBQUNILE1BQWEsVUFBVyxTQUFRLGFBQWE7SUFZM0MsWUFBbUIsSUFBWSxFQUFFLEtBQXdCOztRQUN2RCxNQUFNLE9BQU8sR0FBRztZQUNkLFVBQVUsY0FBRSxLQUFLLENBQUMsY0FBYywwQ0FBRSxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsYUFBYSxFQUFFLEVBQUU7Z0JBQzlELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUMxRCxDQUFDLEVBQUUsS0FBSyxDQUFDLFVBQVUsb0NBQUssS0FBSyxDQUFDLFVBQVU7WUFDeEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1NBQzdCLENBQUM7UUFDRixLQUFLLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3JCLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztRQUMzQyxJQUFJLENBQUMsU0FBUyxHQUFHLEVBQUUsQ0FBQztRQUVwQixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRTtZQUNqRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3pDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3ZELENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFFBQVEsQ0FBQyxPQUF3QjtRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQywwREFBMEQsQ0FBQyxDQUFDO1NBQzdFO1FBQ0QsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNyRSxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDO0lBQ3JELENBQUM7SUFFRDs7T0FFRztJQUNJLFFBQVE7UUFDYixPQUFPLHVCQUFhLENBQUM7WUFDbkIsTUFBTSxFQUFFLE1BQU07WUFDZCxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7WUFDZixjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7WUFDbkMsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO1lBQzNCLE1BQU0sRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtnQkFDL0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDbkMsT0FBTyxHQUFHLEdBQUcsR0FBRyxLQUFLLENBQUMsWUFBWSxFQUFFLEtBQUssS0FBSyxDQUFDLFFBQVEsRUFBRSxHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUNyRyxDQUFDLENBQUM7WUFDRixLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7U0FDbEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ08sZ0JBQWdCLENBQUMsU0FBaUIsRUFBRSxPQUFnQztRQUM1RSxJQUFJLEVBQUMsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLFVBQVUsQ0FBQTtZQUFFLE9BQU87UUFDakMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFBRSxJQUFJLENBQUMsU0FBUyxHQUFHLEVBQUUsQ0FBQztTQUFFO1FBQzdDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUFDO1lBQ3BELFFBQVEsRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNuQixTQUFTLEVBQUUsU0FBUztZQUNwQixjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWM7WUFDdEMsc0JBQXNCLEVBQUUsT0FBTyxDQUFDLHNCQUFzQjtZQUN0RCx1QkFBdUIsRUFBRSxPQUFPLENBQUMsdUJBQXVCO1NBQ3pELENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQztDQUNGO0FBM0VELGdDQTJFQztBQUVEOzs7OztHQUtHO0FBQ0gsTUFBYSxTQUFTO0lBY3BCLFlBQW1CLElBQVksRUFBRSxLQUE4QjtRQUM3RCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxTQUFTLENBQUMsT0FBeUI7UUFDeEMsT0FBTywwQkFBVyxDQUFDLFlBQVksQ0FBQztZQUM5QixNQUFNLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLE1BQU07WUFDdkIsVUFBVSxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxVQUFVO1lBQy9CLGNBQWMsRUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsY0FBYztZQUN2QyxnQkFBZ0IsRUFBRSxJQUFJO1NBQ3ZCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLFFBQVE7UUFDYixPQUFPLHVCQUFhLENBQUM7WUFDbkIsTUFBTSxFQUFFLE9BQU87WUFDZixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7WUFDZixNQUFNLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FDL0MsR0FBRyxHQUFHLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLEVBQUUsS0FBSyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUM7WUFDckYsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLO1NBQ2xCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxRQUFRLENBQUMsT0FBd0I7UUFDdEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELENBQUMsQ0FBQztTQUM1RTtRQUNELElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxLQUFLLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxpQkFBaUIsQ0FBQyxHQUFlO1FBQ3RDLElBQUksQ0FBQyxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQztRQUN2QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Q0FDRjtBQXJFRCw4QkFxRUM7QUFjRDs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsU0FBUztJQWNwQixZQUFtQixJQUFZLEVBQUUsT0FBeUI7UUFDeEQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUM7UUFDakIsSUFBSSxDQUFDLFVBQVUsR0FBRyxFQUFFLENBQUM7UUFDckIsT0FBTyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxLQUFLLEVBQUUsR0FBRyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQzdFLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksU0FBUyxDQUFDLE9BQXlCO1FBQ3hDLE9BQU8sMEJBQVcsQ0FBQyxZQUFZLENBQUM7WUFDOUIsTUFBTSxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxNQUFNO1lBQ3ZCLFVBQVUsRUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsVUFBVTtZQUMvQixjQUFjLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLGNBQWM7WUFDdkMsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRO1FBQ2Isb0VBQW9FO1FBQ3BFLDZDQUE2QztRQUM3QyxPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsRUFBRSxDQUMxRCxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxTQUFTLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN6RSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksUUFBUSxDQUFDLE9BQXdCOztRQUN0QyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUU7WUFDckIsTUFBTSxJQUFJLEtBQUssQ0FBQywyRkFBMkYsQ0FBQyxDQUFDO1NBQzlHO1FBQ0QsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLEVBQUU7WUFDbEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO1NBQzFFO1FBQ0QsSUFBSSxPQUFPLENBQUMsS0FBSyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGdCQUFnQixZQUFZLFVBQVUsQ0FBQyxFQUFFO1lBQzVFLE1BQU0sSUFBSSxLQUFLLENBQUMsOENBQThDLENBQUMsQ0FBQztTQUNqRTtRQUNELElBQUksQ0FBQyxVQUFVLENBQUMsT0FBQSxPQUFPLENBQUMsS0FBSywwQ0FBRSxRQUFRLE1BQUssSUFBSSxDQUFDLEdBQUcsT0FBTyxDQUFDLEtBQUssQ0FBQztJQUNwRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGlCQUFpQixDQUFDLEdBQWU7UUFDdEMsSUFBSSxDQUFDLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztDQUNGO0FBekVELDhCQXlFQztBQWNEOzs7OztHQUtHO0FBQ0gsTUFBYSxRQUFRO0lBY25CLFlBQW1CLElBQVksRUFBRSxPQUF3QjtRQUN2RCxJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQztRQUNqQixJQUFJLENBQUMsVUFBVSxHQUFHLEVBQUUsQ0FBQztRQUNyQixPQUFPLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFNBQWlCLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDOUUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLE9BQXlCO1FBQ3hDLE9BQU8sMEJBQVcsQ0FBQyxZQUFZLENBQUM7WUFDOUIsTUFBTSxFQUFFLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxNQUFNO1lBQ3ZCLFVBQVUsRUFBRSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsVUFBVTtZQUMvQixjQUFjLEVBQUUsT0FBTyxhQUFQLE9BQU8sdUJBQVAsT0FBTyxDQUFFLGNBQWM7WUFDdkMsZ0JBQWdCLEVBQUUsSUFBSTtTQUN2QixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRO1FBQ2IsT0FBTyx1QkFBYSxDQUFDO1lBQ25CLE1BQU0sRUFBRSxNQUFNO1lBQ2QsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO1lBQ2YsTUFBTSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztZQUNwQyxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7U0FDbEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxRQUFRLENBQUMsT0FBd0I7UUFDdEMsSUFBSSxPQUFPLENBQUMsS0FBSyxFQUFFO1lBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMsNEZBQTRGLENBQUMsQ0FBQztTQUMvRztRQUNELElBQUksQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0VBQStFLENBQUMsQ0FBQztTQUNsRztRQUNELElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUU7WUFDdkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUM7U0FDNUY7UUFDRCxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsR0FBRywwQkFBVyxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBQzVELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksaUJBQWlCLENBQUMsR0FBZTtRQUN0QyxJQUFJLENBQUMsS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUM7UUFDdkIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0NBQ0Y7QUExRUQsNEJBMEVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXV0aG9yaXphdGlvblR5cGUsIEdyYXBocWxBcGkgfSBmcm9tICcuL2dyYXBocWxhcGknO1xuaW1wb3J0IHsgc2hhcGVBZGRpdGlvbiB9IGZyb20gJy4vcHJpdmF0ZSc7XG5pbXBvcnQgeyBSZXNvbHZlciB9IGZyb20gJy4vcmVzb2x2ZXInO1xuaW1wb3J0IHsgRGlyZWN0aXZlLCBJRmllbGQsIElJbnRlcm1lZGlhdGVUeXBlLCBBZGRGaWVsZE9wdGlvbnMgfSBmcm9tICcuL3NjaGVtYS1iYXNlJztcbmltcG9ydCB7IEJhc2VUeXBlT3B0aW9ucywgR3JhcGhxbFR5cGUsIFJlc29sdmFibGVGaWVsZE9wdGlvbnMgfSBmcm9tICcuL3NjaGVtYS1maWVsZCc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgY29uZmlndXJpbmcgYW4gSW50ZXJtZWRpYXRlIFR5cGVcbiAqXG4gKiBAcGFyYW0gZGVmaW5pdGlvbiAtIHRoZSB2YXJpYWJsZXMgYW5kIHR5cGVzIHRoYXQgZGVmaW5lIHRoaXMgdHlwZVxuICogaS5lLiB7IHN0cmluZzogR3JhcGhxbFR5cGUsIHN0cmluZzogR3JhcGhxbFR5cGUgfVxuICogQHBhcmFtIGRpcmVjdGl2ZXMgLSB0aGUgZGlyZWN0aXZlcyBmb3IgdGhpcyBvYmplY3QgdHlwZVxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbnRlcm1lZGlhdGVUeXBlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiB0aGUgYXR0cmlidXRlcyBvZiB0aGlzIHR5cGVcbiAgICovXG4gIHJlYWRvbmx5IGRlZmluaXRpb246IHsgW2tleTogc3RyaW5nXTogSUZpZWxkIH07XG4gIC8qKlxuICAgKiB0aGUgZGlyZWN0aXZlcyBmb3IgdGhpcyBvYmplY3QgdHlwZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRpcmVjdGl2ZXNcbiAgICovXG4gIHJlYWRvbmx5IGRpcmVjdGl2ZXM/OiBEaXJlY3RpdmVbXTtcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgVHlwZXMgYXJlIGFic3RyYWN0IHR5cGVzIHRoYXQgaW5jbHVkZXMgYSBjZXJ0YWluIHNldCBvZiBmaWVsZHNcbiAqIHRoYXQgb3RoZXIgdHlwZXMgbXVzdCBpbmNsdWRlIGlmIHRoZXkgaW1wbGVtZW50IHRoZSBpbnRlcmZhY2UuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgSW50ZXJmYWNlVHlwZSBpbXBsZW1lbnRzIElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgLyoqXG4gICAqIHRoZSBuYW1lIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIHRoZSBhdHRyaWJ1dGVzIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlZmluaXRpb246IHsgW2tleTogc3RyaW5nXTogSUZpZWxkIH07XG4gIC8qKlxuICAgKiB0aGUgZGlyZWN0aXZlcyBmb3IgdGhpcyBvYmplY3QgdHlwZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRpcmVjdGl2ZXNcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkaXJlY3RpdmVzPzogRGlyZWN0aXZlW107XG4gIC8qKlxuICAgKiB0aGUgYXV0aG9yaXphdGlvbiBtb2RlcyBmb3IgdGhpcyBpbnRlcm1lZGlhdGUgdHlwZVxuICAgKi9cbiAgcHJvdGVjdGVkIG1vZGVzPzogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICBwdWJsaWMgY29uc3RydWN0b3IobmFtZTogc3RyaW5nLCBwcm9wczogSW50ZXJtZWRpYXRlVHlwZU9wdGlvbnMpIHtcbiAgICB0aGlzLm5hbWUgPSBuYW1lO1xuICAgIHRoaXMuZGVmaW5pdGlvbiA9IHByb3BzLmRlZmluaXRpb247XG4gICAgdGhpcy5kaXJlY3RpdmVzID0gcHJvcHMuZGlyZWN0aXZlcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBHcmFwaFFMIFR5cGUgcmVwcmVzZW50aW5nIHRoaXMgSW50ZXJtZWRpYXRlIFR5cGVcbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgdGhlIG9wdGlvbnMgdG8gY29uZmlndXJlIHRoaXMgYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgYXR0cmlidXRlKG9wdGlvbnM/OiBCYXNlVHlwZU9wdGlvbnMpOiBHcmFwaHFsVHlwZSB7XG4gICAgcmV0dXJuIEdyYXBocWxUeXBlLmludGVybWVkaWF0ZSh7XG4gICAgICBpc0xpc3Q6IG9wdGlvbnM/LmlzTGlzdCxcbiAgICAgIGlzUmVxdWlyZWQ6IG9wdGlvbnM/LmlzUmVxdWlyZWQsXG4gICAgICBpc1JlcXVpcmVkTGlzdDogb3B0aW9ucz8uaXNSZXF1aXJlZExpc3QsXG4gICAgICBpbnRlcm1lZGlhdGVUeXBlOiB0aGlzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSBzdHJpbmcgb2YgdGhpcyBvYmplY3QgdHlwZVxuICAgKi9cbiAgcHVibGljIHRvU3RyaW5nKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHNoYXBlQWRkaXRpb24oe1xuICAgICAgcHJlZml4OiAnaW50ZXJmYWNlJyxcbiAgICAgIG5hbWU6IHRoaXMubmFtZSxcbiAgICAgIGRpcmVjdGl2ZXM6IHRoaXMuZGlyZWN0aXZlcyxcbiAgICAgIGZpZWxkczogT2JqZWN0LmtleXModGhpcy5kZWZpbml0aW9uKS5tYXAoKGtleSkgPT4ge1xuICAgICAgICBjb25zdCBmaWVsZCA9IHRoaXMuZGVmaW5pdGlvbltrZXldO1xuICAgICAgICByZXR1cm4gYCR7a2V5fSR7ZmllbGQuYXJnc1RvU3RyaW5nKCl9OiAke2ZpZWxkLnRvU3RyaW5nKCl9JHtmaWVsZC5kaXJlY3RpdmVzVG9TdHJpbmcodGhpcy5tb2Rlcyl9YDtcbiAgICAgIH0pLFxuICAgICAgbW9kZXM6IHRoaXMubW9kZXMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgZmllbGQgdG8gdGhpcyBJbnRlcmZhY2UgVHlwZS5cbiAgICpcbiAgICogSW50ZXJmYWNlIFR5cGVzIG11c3QgaGF2ZSBib3RoIGZpZWxkTmFtZSBhbmQgZmllbGQgb3B0aW9ucy5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgdGhlIG9wdGlvbnMgdG8gYWRkIGEgZmllbGRcbiAgICovXG4gIHB1YmxpYyBhZGRGaWVsZChvcHRpb25zOiBBZGRGaWVsZE9wdGlvbnMpOiB2b2lkIHtcbiAgICBpZiAoIW9wdGlvbnMuZmllbGROYW1lIHx8ICFvcHRpb25zLmZpZWxkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0ludGVyZmFjZSBUeXBlcyBtdXN0IGhhdmUgYm90aCBmaWVsZE5hbWUgYW5kIGZpZWxkIG9wdGlvbnMuJyk7XG4gICAgfVxuICAgIHRoaXMuZGVmaW5pdGlvbltvcHRpb25zLmZpZWxkTmFtZV0gPSBvcHRpb25zLmZpZWxkO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldGhvZCBjYWxsZWQgd2hlbiB0aGUgc3RyaW5naWZ5aW5nIEludGVybWVkaWF0ZSBUeXBlcyBmb3Igc2NoZW1hIGdlbmVyYXRpb25cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2JpbmRUb0dyYXBocWxBcGkoYXBpOiBHcmFwaHFsQXBpKTogSUludGVybWVkaWF0ZVR5cGUge1xuICAgIHRoaXMubW9kZXMgPSBhcGkubW9kZXM7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBjb25maWd1cmluZyBhbiBPYmplY3QgVHlwZVxuICpcbiAqIEBwYXJhbSBkZWZpbml0aW9uIC0gdGhlIHZhcmlhYmxlcyBhbmQgdHlwZXMgdGhhdCBkZWZpbmUgdGhpcyB0eXBlXG4gKiBpLmUuIHsgc3RyaW5nOiBHcmFwaHFsVHlwZSwgc3RyaW5nOiBHcmFwaHFsVHlwZSB9XG4gKiBAcGFyYW0gaW50ZXJmYWNlVHlwZXMgLSB0aGUgaW50ZXJmYWNlcyB0aGF0IHRoaXMgb2JqZWN0IHR5cGUgaW1wbGVtZW50c1xuICogQHBhcmFtIGRpcmVjdGl2ZXMgLSB0aGUgZGlyZWN0aXZlcyBmb3IgdGhpcyBvYmplY3QgdHlwZVxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBPYmplY3RUeXBlT3B0aW9ucyBleHRlbmRzIEludGVybWVkaWF0ZVR5cGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBJbnRlcmZhY2UgVHlwZXMgdGhpcyBPYmplY3QgVHlwZSBpbXBsZW1lbnRzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gaW50ZXJmYWNlIHR5cGVzXG4gICAqL1xuICByZWFkb25seSBpbnRlcmZhY2VUeXBlcz86IEludGVyZmFjZVR5cGVbXTtcbn1cblxuLyoqXG4gKiBPYmplY3QgVHlwZXMgYXJlIHR5cGVzIGRlY2xhcmVkIGJ5IHlvdS5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBjbGFzcyBPYmplY3RUeXBlIGV4dGVuZHMgSW50ZXJmYWNlVHlwZSBpbXBsZW1lbnRzIElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgLyoqXG4gICAqIFRoZSBJbnRlcmZhY2UgVHlwZXMgdGhpcyBPYmplY3QgVHlwZSBpbXBsZW1lbnRzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gaW50ZXJmYWNlIHR5cGVzXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJmYWNlVHlwZXM/OiBJbnRlcmZhY2VUeXBlW107XG4gIC8qKlxuICAgKiBUaGUgcmVzb2x2ZXJzIGxpbmtlZCB0byB0aGlzIGRhdGEgc291cmNlXG4gICAqL1xuICBwdWJsaWMgcmVzb2x2ZXJzPzogUmVzb2x2ZXJbXTtcblxuICBwdWJsaWMgY29uc3RydWN0b3IobmFtZTogc3RyaW5nLCBwcm9wczogT2JqZWN0VHlwZU9wdGlvbnMpIHtcbiAgICBjb25zdCBvcHRpb25zID0ge1xuICAgICAgZGVmaW5pdGlvbjogcHJvcHMuaW50ZXJmYWNlVHlwZXM/LnJlZHVjZSgoZGVmLCBpbnRlcmZhY2VUeXBlKSA9PiB7XG4gICAgICAgIHJldHVybiBPYmplY3QuYXNzaWduKHt9LCBkZWYsIGludGVyZmFjZVR5cGUuZGVmaW5pdGlvbik7XG4gICAgICB9LCBwcm9wcy5kZWZpbml0aW9uKSA/PyBwcm9wcy5kZWZpbml0aW9uLFxuICAgICAgZGlyZWN0aXZlczogcHJvcHMuZGlyZWN0aXZlcyxcbiAgICB9O1xuICAgIHN1cGVyKG5hbWUsIG9wdGlvbnMpO1xuICAgIHRoaXMuaW50ZXJmYWNlVHlwZXMgPSBwcm9wcy5pbnRlcmZhY2VUeXBlcztcbiAgICB0aGlzLnJlc29sdmVycyA9IFtdO1xuXG4gICAgT2JqZWN0LmtleXModGhpcy5kZWZpbml0aW9uKS5mb3JFYWNoKChmaWVsZE5hbWUpID0+IHtcbiAgICAgIGNvbnN0IGZpZWxkID0gdGhpcy5kZWZpbml0aW9uW2ZpZWxkTmFtZV07XG4gICAgICB0aGlzLmdlbmVyYXRlUmVzb2x2ZXIoZmllbGROYW1lLCBmaWVsZC5maWVsZE9wdGlvbnMpO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGZpZWxkIHRvIHRoaXMgT2JqZWN0IFR5cGUuXG4gICAqXG4gICAqIE9iamVjdCBUeXBlcyBtdXN0IGhhdmUgYm90aCBmaWVsZE5hbWUgYW5kIGZpZWxkIG9wdGlvbnMuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIHRoZSBvcHRpb25zIHRvIGFkZCBhIGZpZWxkXG4gICAqL1xuICBwdWJsaWMgYWRkRmllbGQob3B0aW9uczogQWRkRmllbGRPcHRpb25zKTogdm9pZCB7XG4gICAgaWYgKCFvcHRpb25zLmZpZWxkTmFtZSB8fCAhb3B0aW9ucy5maWVsZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdPYmplY3QgVHlwZXMgbXVzdCBoYXZlIGJvdGggZmllbGROYW1lIGFuZCBmaWVsZCBvcHRpb25zLicpO1xuICAgIH1cbiAgICB0aGlzLmdlbmVyYXRlUmVzb2x2ZXIob3B0aW9ucy5maWVsZE5hbWUsIG9wdGlvbnMuZmllbGQuZmllbGRPcHRpb25zKTtcbiAgICB0aGlzLmRlZmluaXRpb25bb3B0aW9ucy5maWVsZE5hbWVdID0gb3B0aW9ucy5maWVsZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgc3RyaW5nIG9mIHRoaXMgb2JqZWN0IHR5cGVcbiAgICovXG4gIHB1YmxpYyB0b1N0cmluZygpOiBzdHJpbmcge1xuICAgIHJldHVybiBzaGFwZUFkZGl0aW9uKHtcbiAgICAgIHByZWZpeDogJ3R5cGUnLFxuICAgICAgbmFtZTogdGhpcy5uYW1lLFxuICAgICAgaW50ZXJmYWNlVHlwZXM6IHRoaXMuaW50ZXJmYWNlVHlwZXMsXG4gICAgICBkaXJlY3RpdmVzOiB0aGlzLmRpcmVjdGl2ZXMsXG4gICAgICBmaWVsZHM6IE9iamVjdC5rZXlzKHRoaXMuZGVmaW5pdGlvbikubWFwKChrZXkpID0+IHtcbiAgICAgICAgY29uc3QgZmllbGQgPSB0aGlzLmRlZmluaXRpb25ba2V5XTtcbiAgICAgICAgcmV0dXJuIGAke2tleX0ke2ZpZWxkLmFyZ3NUb1N0cmluZygpfTogJHtmaWVsZC50b1N0cmluZygpfSR7ZmllbGQuZGlyZWN0aXZlc1RvU3RyaW5nKHRoaXMubW9kZXMpfWA7XG4gICAgICB9KSxcbiAgICAgIG1vZGVzOiB0aGlzLm1vZGVzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSByZXNvbHZlcnMgbGlua2VkIHRvIHRoaXMgT2JqZWN0IFR5cGVcbiAgICovXG4gIHByb3RlY3RlZCBnZW5lcmF0ZVJlc29sdmVyKGZpZWxkTmFtZTogc3RyaW5nLCBvcHRpb25zPzogUmVzb2x2YWJsZUZpZWxkT3B0aW9ucyk6IHZvaWQge1xuICAgIGlmICghb3B0aW9ucz8uZGF0YVNvdXJjZSkgcmV0dXJuO1xuICAgIGlmICghdGhpcy5yZXNvbHZlcnMpIHsgdGhpcy5yZXNvbHZlcnMgPSBbXTsgfVxuICAgIHRoaXMucmVzb2x2ZXJzLnB1c2gob3B0aW9ucy5kYXRhU291cmNlLmNyZWF0ZVJlc29sdmVyKHtcbiAgICAgIHR5cGVOYW1lOiB0aGlzLm5hbWUsXG4gICAgICBmaWVsZE5hbWU6IGZpZWxkTmFtZSxcbiAgICAgIHBpcGVsaW5lQ29uZmlnOiBvcHRpb25zLnBpcGVsaW5lQ29uZmlnLFxuICAgICAgcmVxdWVzdE1hcHBpbmdUZW1wbGF0ZTogb3B0aW9ucy5yZXF1ZXN0TWFwcGluZ1RlbXBsYXRlLFxuICAgICAgcmVzcG9uc2VNYXBwaW5nVGVtcGxhdGU6IG9wdGlvbnMucmVzcG9uc2VNYXBwaW5nVGVtcGxhdGUsXG4gICAgfSkpO1xuICB9XG59XG5cbi8qKlxuICogSW5wdXQgVHlwZXMgYXJlIGFic3RyYWN0IHR5cGVzIHRoYXQgZGVmaW5lIGNvbXBsZXggb2JqZWN0cy5cbiAqIFRoZXkgYXJlIHVzZWQgaW4gYXJndW1lbnRzIHRvIHJlcHJlc2VudFxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIElucHV0VHlwZSBpbXBsZW1lbnRzIElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgLyoqXG4gICAqIHRoZSBuYW1lIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIHRoZSBhdHRyaWJ1dGVzIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlZmluaXRpb246IHsgW2tleTogc3RyaW5nXTogSUZpZWxkIH07XG4gIC8qKlxuICAgKiB0aGUgYXV0aG9yaXphdGlvbiBtb2RlcyBmb3IgdGhpcyBpbnRlcm1lZGlhdGUgdHlwZVxuICAgKi9cbiAgcHJvdGVjdGVkIG1vZGVzPzogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICBwdWJsaWMgY29uc3RydWN0b3IobmFtZTogc3RyaW5nLCBwcm9wczogSW50ZXJtZWRpYXRlVHlwZU9wdGlvbnMpIHtcbiAgICB0aGlzLm5hbWUgPSBuYW1lO1xuICAgIHRoaXMuZGVmaW5pdGlvbiA9IHByb3BzLmRlZmluaXRpb247XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgR3JhcGhRTCBUeXBlIHJlcHJlc2VudGluZyB0aGlzIElucHV0IFR5cGVcbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgdGhlIG9wdGlvbnMgdG8gY29uZmlndXJlIHRoaXMgYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgYXR0cmlidXRlKG9wdGlvbnM/OiBCYXNlVHlwZU9wdGlvbnMpOiBHcmFwaHFsVHlwZSB7XG4gICAgcmV0dXJuIEdyYXBocWxUeXBlLmludGVybWVkaWF0ZSh7XG4gICAgICBpc0xpc3Q6IG9wdGlvbnM/LmlzTGlzdCxcbiAgICAgIGlzUmVxdWlyZWQ6IG9wdGlvbnM/LmlzUmVxdWlyZWQsXG4gICAgICBpc1JlcXVpcmVkTGlzdDogb3B0aW9ucz8uaXNSZXF1aXJlZExpc3QsXG4gICAgICBpbnRlcm1lZGlhdGVUeXBlOiB0aGlzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSBzdHJpbmcgb2YgdGhpcyBpbnB1dCB0eXBlXG4gICAqL1xuICBwdWJsaWMgdG9TdHJpbmcoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gc2hhcGVBZGRpdGlvbih7XG4gICAgICBwcmVmaXg6ICdpbnB1dCcsXG4gICAgICBuYW1lOiB0aGlzLm5hbWUsXG4gICAgICBmaWVsZHM6IE9iamVjdC5rZXlzKHRoaXMuZGVmaW5pdGlvbikubWFwKChrZXkpID0+XG4gICAgICAgIGAke2tleX0ke3RoaXMuZGVmaW5pdGlvbltrZXldLmFyZ3NUb1N0cmluZygpfTogJHt0aGlzLmRlZmluaXRpb25ba2V5XS50b1N0cmluZygpfWApLFxuICAgICAgbW9kZXM6IHRoaXMubW9kZXMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgZmllbGQgdG8gdGhpcyBJbnB1dCBUeXBlLlxuICAgKlxuICAgKiBJbnB1dCBUeXBlcyBtdXN0IGhhdmUgYm90aCBmaWVsZE5hbWUgYW5kIGZpZWxkIG9wdGlvbnMuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIHRoZSBvcHRpb25zIHRvIGFkZCBhIGZpZWxkXG4gICAqL1xuICBwdWJsaWMgYWRkRmllbGQob3B0aW9uczogQWRkRmllbGRPcHRpb25zKTogdm9pZCB7XG4gICAgaWYgKCFvcHRpb25zLmZpZWxkTmFtZSB8fCAhb3B0aW9ucy5maWVsZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJbnB1dCBUeXBlcyBtdXN0IGhhdmUgYm90aCBmaWVsZE5hbWUgYW5kIGZpZWxkIG9wdGlvbnMuJyk7XG4gICAgfVxuICAgIHRoaXMuZGVmaW5pdGlvbltvcHRpb25zLmZpZWxkTmFtZV0gPSBvcHRpb25zLmZpZWxkO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldGhvZCBjYWxsZWQgd2hlbiB0aGUgc3RyaW5naWZ5aW5nIEludGVybWVkaWF0ZSBUeXBlcyBmb3Igc2NoZW1hIGdlbmVyYXRpb25cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2JpbmRUb0dyYXBocWxBcGkoYXBpOiBHcmFwaHFsQXBpKTogSUludGVybWVkaWF0ZVR5cGUge1xuICAgIHRoaXMubW9kZXMgPSBhcGkubW9kZXM7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBjb25maWd1cmluZyBhbiBVbmlvbiBUeXBlXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVuaW9uVHlwZU9wdGlvbnMge1xuICAvKipcbiAgICogdGhlIG9iamVjdCB0eXBlcyBmb3IgdGhpcyB1bmlvbiB0eXBlXG4gICAqL1xuICByZWFkb25seSBkZWZpbml0aW9uOiBJSW50ZXJtZWRpYXRlVHlwZVtdO1xufVxuXG4vKipcbiAqIFVuaW9uIFR5cGVzIGFyZSBhYnN0cmFjdCB0eXBlcyB0aGF0IGFyZSBzaW1pbGFyIHRvIEludGVyZmFjZSBUeXBlcyxcbiAqIGJ1dCB0aGV5IGNhbm5vdCB0byBzcGVjaWZ5IGFueSBjb21tb24gZmllbGRzIGJldHdlZW4gdHlwZXMuXG4gKlxuICogTm90ZSB0aGF0IGZpZWxkcyBvZiBhIHVuaW9uIHR5cGUgbmVlZCB0byBiZSBvYmplY3QgdHlwZXMuIEluIG90aGVyIHdvcmRzLFxuICogeW91IGNhbid0IGNyZWF0ZSBhIHVuaW9uIHR5cGUgb3V0IG9mIGludGVyZmFjZXMsIG90aGVyIHVuaW9ucywgb3IgaW5wdXRzLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIFVuaW9uVHlwZSBpbXBsZW1lbnRzIElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgLyoqXG4gICAqIHRoZSBuYW1lIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIHRoZSBhdHRyaWJ1dGVzIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlZmluaXRpb246IHsgW2tleTogc3RyaW5nXTogSUZpZWxkIH07XG4gIC8qKlxuICAgKiB0aGUgYXV0aG9yaXphdGlvbiBtb2RlcyBzdXBwb3J0ZWQgYnkgdGhpcyBpbnRlcm1lZGlhdGUgdHlwZVxuICAgKi9cbiAgcHJvdGVjdGVkIG1vZGVzPzogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICBwdWJsaWMgY29uc3RydWN0b3IobmFtZTogc3RyaW5nLCBvcHRpb25zOiBVbmlvblR5cGVPcHRpb25zKSB7XG4gICAgdGhpcy5uYW1lID0gbmFtZTtcbiAgICB0aGlzLmRlZmluaXRpb24gPSB7fTtcbiAgICBvcHRpb25zLmRlZmluaXRpb24ubWFwKChkZWYpID0+IHRoaXMuYWRkRmllbGQoeyBmaWVsZDogZGVmLmF0dHJpYnV0ZSgpIH0pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBHcmFwaFFMIFR5cGUgcmVwcmVzZW50aW5nIHRoaXMgVW5pb24gVHlwZVxuICAgKlxuICAgKiBAcGFyYW0gb3B0aW9ucyB0aGUgb3B0aW9ucyB0byBjb25maWd1cmUgdGhpcyBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBhdHRyaWJ1dGUob3B0aW9ucz86IEJhc2VUeXBlT3B0aW9ucyk6IEdyYXBocWxUeXBlIHtcbiAgICByZXR1cm4gR3JhcGhxbFR5cGUuaW50ZXJtZWRpYXRlKHtcbiAgICAgIGlzTGlzdDogb3B0aW9ucz8uaXNMaXN0LFxuICAgICAgaXNSZXF1aXJlZDogb3B0aW9ucz8uaXNSZXF1aXJlZCxcbiAgICAgIGlzUmVxdWlyZWRMaXN0OiBvcHRpb25zPy5pc1JlcXVpcmVkTGlzdCxcbiAgICAgIGludGVybWVkaWF0ZVR5cGU6IHRoaXMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIHN0cmluZyBvZiB0aGlzIFVuaW9uIHR5cGVcbiAgICovXG4gIHB1YmxpYyB0b1N0cmluZygpOiBzdHJpbmcge1xuICAgIC8vIFJldHVybiBhIHN0cmluZyB0aGF0IGFwcGVuZHMgYWxsIE9iamVjdCBUeXBlcyBmb3IgdGhpcyBVbmlvbiBUeXBlXG4gICAgLy8gaS5lLiAndW5pb24gRXhhbXBsZSA9IGV4YW1wbGUxIHwgZXhhbXBsZTInXG4gICAgcmV0dXJuIE9iamVjdC52YWx1ZXModGhpcy5kZWZpbml0aW9uKS5yZWR1Y2UoKGFjYywgZmllbGQpID0+XG4gICAgICBgJHthY2N9ICR7ZmllbGQudG9TdHJpbmcoKX0gfGAsIGB1bmlvbiAke3RoaXMubmFtZX0gPWApLnNsaWNlKDAsIC0yKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBmaWVsZCB0byB0aGlzIFVuaW9uIFR5cGVcbiAgICpcbiAgICogSW5wdXQgVHlwZXMgbXVzdCBoYXZlIGZpZWxkIG9wdGlvbnMgYW5kIHRoZSBJRmllbGQgbXVzdCBiZSBhbiBPYmplY3QgVHlwZS5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgdGhlIG9wdGlvbnMgdG8gYWRkIGEgZmllbGRcbiAgICovXG4gIHB1YmxpYyBhZGRGaWVsZChvcHRpb25zOiBBZGRGaWVsZE9wdGlvbnMpOiB2b2lkIHtcbiAgICBpZiAob3B0aW9ucy5maWVsZE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVW5pb24gVHlwZXMgY2Fubm90IGJlIGNvbmZpZ3VyZWQgd2l0aCB0aGUgZmllbGROYW1lIG9wdGlvbi4gVXNlIHRoZSBmaWVsZCBvcHRpb24gaW5zdGVhZC4nKTtcbiAgICB9XG4gICAgaWYgKCFvcHRpb25zLmZpZWxkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1VuaW9uIFR5cGVzIG11c3QgYmUgY29uZmlndXJlZCB3aXRoIHRoZSBmaWVsZCBvcHRpb24uJyk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLmZpZWxkICYmICEob3B0aW9ucy5maWVsZC5pbnRlcm1lZGlhdGVUeXBlIGluc3RhbmNlb2YgT2JqZWN0VHlwZSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRmllbGRzIGZvciBVbmlvbiBUeXBlcyBtdXN0IGJlIE9iamVjdCBUeXBlcy4nKTtcbiAgICB9XG4gICAgdGhpcy5kZWZpbml0aW9uW29wdGlvbnMuZmllbGQ/LnRvU3RyaW5nKCkgKyAnaWQnXSA9IG9wdGlvbnMuZmllbGQ7XG4gIH1cblxuICAvKipcbiAgICogTWV0aG9kIGNhbGxlZCB3aGVuIHRoZSBzdHJpbmdpZnlpbmcgSW50ZXJtZWRpYXRlIFR5cGVzIGZvciBzY2hlbWEgZ2VuZXJhdGlvblxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfYmluZFRvR3JhcGhxbEFwaShhcGk6IEdyYXBocWxBcGkpOiBJSW50ZXJtZWRpYXRlVHlwZSB7XG4gICAgdGhpcy5tb2RlcyA9IGFwaS5tb2RlcztcbiAgICByZXR1cm4gdGhpcztcbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNvbmZpZ3VyaW5nIGFuIEVudW0gVHlwZVxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbnVtVHlwZU9wdGlvbnMge1xuICAvKipcbiAgICogdGhlIGF0dHJpYnV0ZXMgb2YgdGhpcyB0eXBlXG4gICAqL1xuICByZWFkb25seSBkZWZpbml0aW9uOiBzdHJpbmdbXTtcbn1cblxuLyoqXG4gKiBFbnVtIFR5cGVzIGFyZSBhYnN0cmFjdCB0eXBlcyB0aGF0IGluY2x1ZGVzIGEgc2V0IG9mIGZpZWxkc1xuICogdGhhdCByZXByZXNlbnQgdGhlIHN0cmluZ3MgdGhpcyB0eXBlIGNhbiBjcmVhdGUuXG4gKlxuICogQGV4cGVyaW1lbnRhbFxuICovXG5leHBvcnQgY2xhc3MgRW51bVR5cGUgaW1wbGVtZW50cyBJSW50ZXJtZWRpYXRlVHlwZSB7XG4gIC8qKlxuICAgKiB0aGUgbmFtZSBvZiB0aGlzIHR5cGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBuYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiB0aGUgYXR0cmlidXRlcyBvZiB0aGlzIHR5cGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWZpbml0aW9uOiB7IFtrZXk6IHN0cmluZ106IElGaWVsZCB9O1xuICAvKipcbiAgICogdGhlIGF1dGhvcml6YXRpb24gbW9kZXMgZm9yIHRoaXMgaW50ZXJtZWRpYXRlIHR5cGVcbiAgICovXG4gIHByb3RlY3RlZCBtb2Rlcz86IEF1dGhvcml6YXRpb25UeXBlW107XG5cbiAgcHVibGljIGNvbnN0cnVjdG9yKG5hbWU6IHN0cmluZywgb3B0aW9uczogRW51bVR5cGVPcHRpb25zKSB7XG4gICAgdGhpcy5uYW1lID0gbmFtZTtcbiAgICB0aGlzLmRlZmluaXRpb24gPSB7fTtcbiAgICBvcHRpb25zLmRlZmluaXRpb24ubWFwKChmaWVsZE5hbWU6IHN0cmluZykgPT4gdGhpcy5hZGRGaWVsZCh7IGZpZWxkTmFtZSB9KSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGFuIEdyYXBoUUwgVHlwZSByZXByZXNlbnRpbmcgdGhpcyBFbnVtIFR5cGVcbiAgICovXG4gIHB1YmxpYyBhdHRyaWJ1dGUob3B0aW9ucz86IEJhc2VUeXBlT3B0aW9ucyk6IEdyYXBocWxUeXBlIHtcbiAgICByZXR1cm4gR3JhcGhxbFR5cGUuaW50ZXJtZWRpYXRlKHtcbiAgICAgIGlzTGlzdDogb3B0aW9ucz8uaXNMaXN0LFxuICAgICAgaXNSZXF1aXJlZDogb3B0aW9ucz8uaXNSZXF1aXJlZCxcbiAgICAgIGlzUmVxdWlyZWRMaXN0OiBvcHRpb25zPy5pc1JlcXVpcmVkTGlzdCxcbiAgICAgIGludGVybWVkaWF0ZVR5cGU6IHRoaXMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIHN0cmluZyBvZiB0aGlzIGVudW0gdHlwZVxuICAgKi9cbiAgcHVibGljIHRvU3RyaW5nKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHNoYXBlQWRkaXRpb24oe1xuICAgICAgcHJlZml4OiAnZW51bScsXG4gICAgICBuYW1lOiB0aGlzLm5hbWUsXG4gICAgICBmaWVsZHM6IE9iamVjdC5rZXlzKHRoaXMuZGVmaW5pdGlvbiksXG4gICAgICBtb2RlczogdGhpcy5tb2RlcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBmaWVsZCB0byB0aGlzIEVudW0gVHlwZVxuICAgKlxuICAgKiBUbyBhZGQgYSBmaWVsZCB0byB0aGlzIEVudW0gVHlwZSwgeW91IG11c3Qgb25seSBjb25maWd1cmVcbiAgICogYWRkRmllbGQgd2l0aCB0aGUgZmllbGROYW1lIG9wdGlvbnMuXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIHRoZSBvcHRpb25zIHRvIGFkZCBhIGZpZWxkXG4gICAqL1xuICBwdWJsaWMgYWRkRmllbGQob3B0aW9uczogQWRkRmllbGRPcHRpb25zKTogdm9pZCB7XG4gICAgaWYgKG9wdGlvbnMuZmllbGQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRW51bSBUeXBlIGZpZWxkcyBjb25zaXN0IG9mIHN0cmluZ3MuIFVzZSB0aGUgZmllbGROYW1lIG9wdGlvbiBpbnN0ZWFkIG9mIHRoZSBmaWVsZCBvcHRpb24uJyk7XG4gICAgfVxuICAgIGlmICghb3B0aW9ucy5maWVsZE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignV2hlbiBhZGRpbmcgYSBmaWVsZCB0byBhbiBFbnVtIFR5cGUsIHlvdSBtdXN0IGNvbmZpZ3VyZSB0aGUgZmllbGROYW1lIG9wdGlvbi4nKTtcbiAgICB9XG4gICAgaWYgKG9wdGlvbnMuZmllbGROYW1lLmluZGV4T2YoJyAnKSA+IC0xKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEVudW0gVHlwZSB2YWx1ZXMgY2Fubm90IGhhdmUgd2hpdGVzcGFjZS4gUmVjZWl2ZWQ6ICR7b3B0aW9ucy5maWVsZE5hbWV9YCk7XG4gICAgfVxuICAgIHRoaXMuZGVmaW5pdGlvbltvcHRpb25zLmZpZWxkTmFtZV0gPSBHcmFwaHFsVHlwZS5zdHJpbmcoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZXRob2QgY2FsbGVkIHdoZW4gdGhlIHN0cmluZ2lmeWluZyBJbnRlcm1lZGlhdGUgVHlwZXMgZm9yIHNjaGVtYSBnZW5lcmF0aW9uXG4gICAqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF9iaW5kVG9HcmFwaHFsQXBpKGFwaTogR3JhcGhxbEFwaSk6IElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgICB0aGlzLm1vZGVzID0gYXBpLm1vZGVzO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG59XG4iXX0=