"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResolvableField = exports.Field = exports.GraphqlType = void 0;
const schema_base_1 = require("./schema-base");
/**
 * The GraphQL Types in AppSync's GraphQL. GraphQL Types are the
 * building blocks for object types, queries, mutations, etc. They are
 * types like String, Int, Id or even Object Types you create.
 *
 * i.e. `String`, `String!`, `[String]`, `[String!]`, `[String]!`
 *
 * GraphQL Types are used to define the entirety of schema.
 */
class GraphqlType {
    constructor(type, options) {
        var _a, _b, _c;
        this.type = type;
        this.isList = (_a = options === null || options === void 0 ? void 0 : options.isList) !== null && _a !== void 0 ? _a : false;
        this.isRequired = (_b = options === null || options === void 0 ? void 0 : options.isRequired) !== null && _b !== void 0 ? _b : false;
        this.isRequiredList = (_c = options === null || options === void 0 ? void 0 : options.isRequiredList) !== null && _c !== void 0 ? _c : false;
        this.intermediateType = options === null || options === void 0 ? void 0 : options.intermediateType;
    }
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static id(options) {
        return new GraphqlType(schema_base_1.Type.ID, options);
    }
    /**
     * `String` scalar type is a free-form human-readable text.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static string(options) {
        return new GraphqlType(schema_base_1.Type.STRING, options);
    }
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static int(options) {
        return new GraphqlType(schema_base_1.Type.INT, options);
    }
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static float(options) {
        return new GraphqlType(schema_base_1.Type.FLOAT, options);
    }
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static boolean(options) {
        return new GraphqlType(schema_base_1.Type.BOOLEAN, options);
    }
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDate(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE, options);
    }
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIME, options);
    }
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDateTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE_TIME, options);
    }
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTimestamp(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIMESTAMP, options);
    }
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsEmail(options) {
        return new GraphqlType(schema_base_1.Type.AWS_EMAIL, options);
    }
    /**
     * `AWSJson` scalar type represents a JSON string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsJson(options) {
        return new GraphqlType(schema_base_1.Type.AWS_JSON, options);
    }
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsUrl(options) {
        return new GraphqlType(schema_base_1.Type.AWS_URL, options);
    }
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsPhone(options) {
        return new GraphqlType(schema_base_1.Type.AWS_PHONE, options);
    }
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsIpAddress(options) {
        return new GraphqlType(schema_base_1.Type.AWS_IP_ADDRESS, options);
    }
    /**
     * an intermediate type to be added as an attribute
     * (i.e. an interface or an object type)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     * - intermediateType
     */
    static intermediate(options) {
        if (!(options === null || options === void 0 ? void 0 : options.intermediateType)) {
            throw new Error('GraphQL Type of interface must be configured with corresponding Intermediate Type');
        }
        return new GraphqlType(schema_base_1.Type.INTERMEDIATE, options);
    }
    /**
     * Generate the string for this attribute
     */
    toString() {
        var _a;
        // If an Object Type, we use the name of the Object Type
        let type = this.intermediateType ? (_a = this.intermediateType) === null || _a === void 0 ? void 0 : _a.name : this.type;
        // If configured as required, the GraphQL Type becomes required
        type = this.isRequired ? `${type}!` : type;
        // If configured with isXxxList, the GraphQL Type becomes a list
        type = this.isList || this.isRequiredList ? `[${type}]` : type;
        // If configured with isRequiredList, the list becomes required
        type = this.isRequiredList ? `${type}!` : type;
        return type;
    }
    /**
     * Generate the arguments for this field
     */
    argsToString() {
        return '';
    }
    /**
     * Generate the directives for this field
     */
    directivesToString(_modes) {
        return '';
    }
}
exports.GraphqlType = GraphqlType;
/**
 * Fields build upon Graphql Types and provide typing
 * and arguments.
 */
class Field extends GraphqlType {
    constructor(options) {
        const props = {
            isList: options.returnType.isList,
            isRequired: options.returnType.isRequired,
            isRequiredList: options.returnType.isRequiredList,
            intermediateType: options.returnType.intermediateType,
        };
        super(options.returnType.type, props);
        this.fieldOptions = options;
    }
    /**
     * Generate the args string of this resolvable field
     */
    argsToString() {
        if (!this.fieldOptions || !this.fieldOptions.args) {
            return '';
        }
        return Object.keys(this.fieldOptions.args).reduce((acc, key) => { var _a, _b; return `${acc}${key}: ${(_b = (_a = this.fieldOptions) === null || _a === void 0 ? void 0 : _a.args) === null || _b === void 0 ? void 0 : _b[key].toString()} `; }, '(').slice(0, -1) + ')';
    }
    /**
     * Generate the directives for this field
     */
    directivesToString(modes) {
        if (!this.fieldOptions || !this.fieldOptions.directives) {
            return '';
        }
        return this.fieldOptions.directives.reduce((acc, directive) => `${acc}${directive._bindToAuthModes(modes).toString()} `, '\n  ').slice(0, -1);
    }
}
exports.Field = Field;
/**
 * Resolvable Fields build upon Graphql Types and provide fields
 * that can resolve into operations on a data source.
 */
class ResolvableField extends Field {
    constructor(options) {
        const props = {
            returnType: options.returnType,
            args: options.args,
        };
        super(props);
        this.fieldOptions = options;
    }
}
exports.ResolvableField = ResolvableField;
//# sourceMappingURL=data:application/json;base64,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