"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Type = exports.Directive = void 0;
const graphqlapi_1 = require("./graphqlapi");
/**
 * Directives for types
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 * @experimental
 */
class Directive {
    constructor(statement, options) {
        this.statement = statement;
        this.mode = options === null || options === void 0 ? void 0 : options.mode;
        this.mutationFields = options === null || options === void 0 ? void 0 : options.mutationFields;
    }
    /**
     * Add the @aws_iam directive
     */
    static iam() {
        return new Directive('@aws_iam', { mode: graphqlapi_1.AuthorizationType.IAM });
    }
    /**
     * Add the @aws_oidc directive
     */
    static oidc() {
        return new Directive('@aws_oidc', { mode: graphqlapi_1.AuthorizationType.OIDC });
    }
    /**
     * Add the @aws_api_key directive
     */
    static apiKey() {
        return new Directive('@aws_api_key', { mode: graphqlapi_1.AuthorizationType.API_KEY });
    }
    /**
     * Add the @aws_auth or @aws_cognito_user_pools directive
     *
     * @param groups the groups to allow access to
     */
    static cognito(...groups) {
        if (groups.length === 0) {
            throw new Error(`Cognito authorization requires at least one Cognito group to be supplied. Received: ${groups.length}`);
        }
        // this function creates the cognito groups as a string (i.e. ["group1", "group2", "group3"])
        const stringify = (array) => {
            return array.reduce((acc, element) => `${acc}"${element}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_auth(cognito_groups: [${stringify(groups)}])`, { mode: graphqlapi_1.AuthorizationType.USER_POOL });
    }
    /**
     * Add the @aws_subscribe directive. Only use for top level Subscription type.
     *
     * @param mutations the mutation fields to link to
     */
    static subscribe(...mutations) {
        if (mutations.length === 0) {
            throw new Error(`Subscribe directive requires at least one mutation field to be supplied. Received: ${mutations.length}`);
        }
        // this function creates the subscribe directive as a string (i.e. ["mutation_field_1", "mutation_field_2"])
        const stringify = (array) => {
            return array.reduce((acc, mutation) => `${acc}"${mutation}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_subscribe(mutations: [${stringify(mutations)}])`, { mutationFields: mutations });
    }
    /**
     * Add a custom directive
     *
     * @param statement - the directive statement to append
     */
    static custom(statement) {
        return new Directive(statement);
    }
    /**
     * Method called when the stringifying Directive for schema generation
     *
     * @param modes the authorization modes
     *
     * @internal
     */
    _bindToAuthModes(modes) {
        this.modes = modes;
        return this;
    }
    /**
     * Generate the directive statement
     */
    toString() {
        if (this.modes && this.mode && !this.modes.some((mode) => mode === this.mode)) {
            throw new Error(`No Authorization Type ${this.mode} declared in GraphQL Api.`);
        }
        if (this.mode === graphqlapi_1.AuthorizationType.USER_POOL && this.modes && this.modes.length > 1) {
            this.statement = this.statement.replace('@aws_auth', '@aws_cognito_user_pools');
        }
        return this.statement;
    }
}
exports.Directive = Directive;
/**
 * Enum containing the Types that can be used to define ObjectTypes
 */
var Type;
(function (Type) {
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     */
    Type["ID"] = "ID";
    /**
     * `String` scalar type is a free-form human-readable text.
     */
    Type["STRING"] = "String";
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     */
    Type["INT"] = "Int";
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     */
    Type["FLOAT"] = "Float";
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     */
    Type["BOOLEAN"] = "Boolean";
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Calendar_dates
     */
    Type["AWS_DATE"] = "AWSDate";
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Times
     */
    Type["AWS_TIME"] = "AWSTime";
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Combined_date_and_time_representations
     */
    Type["AWS_DATE_TIME"] = "AWSDateTime";
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     */
    Type["AWS_TIMESTAMP"] = "AWSTimestamp";
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     */
    Type["AWS_EMAIL"] = "AWSEmail";
    /**
     * `AWSJson` scalar type represents a JSON string.
     */
    Type["AWS_JSON"] = "AWSJSON";
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     */
    Type["AWS_URL"] = "AWSURL";
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     */
    Type["AWS_PHONE"] = "AWSPhone";
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     */
    Type["AWS_IP_ADDRESS"] = "AWSIPAddress";
    /**
     * Type used for Intermediate Types
     * (i.e. an interface or an object type)
     */
    Type["INTERMEDIATE"] = "INTERMEDIATE";
})(Type = exports.Type || (exports.Type = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2NoZW1hLWJhc2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzY2hlbWEtYmFzZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2Q0FBNkQ7QUF1TDdEOzs7Ozs7R0FNRztBQUNILE1BQWEsU0FBUztJQXVGcEIsWUFBb0IsU0FBaUIsRUFBRSxPQUEwQjtRQUMvRCxJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztRQUMzQixJQUFJLENBQUMsSUFBSSxHQUFHLE9BQU8sYUFBUCxPQUFPLHVCQUFQLE9BQU8sQ0FBRSxJQUFJLENBQUM7UUFDMUIsSUFBSSxDQUFDLGNBQWMsR0FBRyxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsY0FBYyxDQUFDO0lBQ2hELENBQUM7SUExRkQ7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRztRQUNmLE9BQU8sSUFBSSxTQUFTLENBQUMsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLDhCQUFpQixDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7SUFDcEUsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLElBQUk7UUFDaEIsT0FBTyxJQUFJLFNBQVMsQ0FBQyxXQUFXLEVBQUUsRUFBRSxJQUFJLEVBQUUsOEJBQWlCLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsTUFBTTtRQUNsQixPQUFPLElBQUksU0FBUyxDQUFDLGNBQWMsRUFBRSxFQUFFLElBQUksRUFBRSw4QkFBaUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLE1BQWdCO1FBQ3ZDLElBQUksTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1RkFBdUYsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDekg7UUFDRCw2RkFBNkY7UUFDN0YsTUFBTSxTQUFTLEdBQUcsQ0FBQyxLQUFlLEVBQVUsRUFBRTtZQUM1QyxPQUFPLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQyxHQUFHLEdBQUcsSUFBSSxPQUFPLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakYsQ0FBQyxDQUFDO1FBQ0YsT0FBTyxJQUFJLFNBQVMsQ0FBQyw4QkFBOEIsU0FBUyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsOEJBQWlCLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUNuSCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFtQjtRQUM1QyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzFCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0ZBQXNGLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQzNIO1FBQ0QsNEdBQTRHO1FBQzVHLE1BQU0sU0FBUyxHQUFHLENBQUMsS0FBZSxFQUFVLEVBQUU7WUFDNUMsT0FBTyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLENBQUMsR0FBRyxHQUFHLElBQUksUUFBUSxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25GLENBQUMsQ0FBQztRQUNGLE9BQU8sSUFBSSxTQUFTLENBQUMsOEJBQThCLFNBQVMsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLEVBQUUsY0FBYyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUM7SUFDOUcsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLFNBQWlCO1FBQ3BDLE9BQU8sSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQWdDRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0IsQ0FBQyxLQUEyQjtRQUNqRCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztRQUNuQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7T0FFRztJQUNJLFFBQVE7UUFDYixJQUFJLElBQUksQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1lBQzdFLE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLElBQUksQ0FBQyxJQUFJLDJCQUEyQixDQUFDLENBQUM7U0FDaEY7UUFDRCxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssOEJBQWlCLENBQUMsU0FBUyxJQUFJLElBQUksQ0FBQyxLQUFLLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3BGLElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLHlCQUF5QixDQUFDLENBQUM7U0FDakY7UUFDRCxPQUFPLElBQUksQ0FBQyxTQUFTLENBQUM7SUFDeEIsQ0FBQztDQUNGO0FBckhELDhCQXFIQztBQUVEOztHQUVHO0FBQ0gsSUFBWSxJQW9GWDtBQXBGRCxXQUFZLElBQUk7SUFDZDs7OztPQUlHO0lBQ0gsaUJBQVMsQ0FBQTtJQUNUOztPQUVHO0lBQ0gseUJBQWlCLENBQUE7SUFDakI7O09BRUc7SUFDSCxtQkFBVyxDQUFBO0lBQ1g7O09BRUc7SUFDSCx1QkFBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCwyQkFBbUIsQ0FBQTtJQUVuQjs7Ozs7O09BTUc7SUFDSCw0QkFBb0IsQ0FBQTtJQUNwQjs7Ozs7O09BTUc7SUFDSCw0QkFBb0IsQ0FBQTtJQUNwQjs7Ozs7O09BTUc7SUFDSCxxQ0FBNkIsQ0FBQTtJQUM3Qjs7OztPQUlHO0lBQ0gsc0NBQThCLENBQUE7SUFDOUI7O09BRUc7SUFDSCw4QkFBc0IsQ0FBQTtJQUN0Qjs7T0FFRztJQUNILDRCQUFvQixDQUFBO0lBQ3BCOzs7O09BSUc7SUFDSCwwQkFBa0IsQ0FBQTtJQUNsQjs7OztPQUlHO0lBQ0gsOEJBQXNCLENBQUE7SUFDdEI7O09BRUc7SUFDSCx1Q0FBK0IsQ0FBQTtJQUUvQjs7O09BR0c7SUFDSCxxQ0FBNkIsQ0FBQTtBQUMvQixDQUFDLEVBcEZXLElBQUksR0FBSixZQUFJLEtBQUosWUFBSSxRQW9GZiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEF1dGhvcml6YXRpb25UeXBlLCBHcmFwaHFsQXBpIH0gZnJvbSAnLi9ncmFwaHFsYXBpJztcbmltcG9ydCB7IFJlc29sdmVyIH0gZnJvbSAnLi9yZXNvbHZlcic7XG5pbXBvcnQgeyBSZXNvbHZhYmxlRmllbGRPcHRpb25zLCBCYXNlVHlwZU9wdGlvbnMsIEdyYXBocWxUeXBlIH0gZnJvbSAnLi9zY2hlbWEtZmllbGQnO1xuaW1wb3J0IHsgSW50ZXJmYWNlVHlwZSB9IGZyb20gJy4vc2NoZW1hLWludGVybWVkaWF0ZSc7XG5cbi8qKlxuICogQSBHcmFwaHFsIEZpZWxkXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUZpZWxkIHtcbiAgLyoqXG4gICAqIHRoZSB0eXBlIG9mIGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZTogVHlwZTtcblxuICAvKipcbiAgICogcHJvcGVydHkgZGV0ZXJtaW5pbmcgaWYgdGhpcyBhdHRyaWJ1dGUgaXMgYSBsaXN0XG4gICAqIGkuZS4gaWYgdHJ1ZSwgYXR0cmlidXRlIHdvdWxkIGJlIGBbVHlwZV1gXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpc0xpc3Q6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIHByb3BlcnR5IGRldGVybWluaW5nIGlmIHRoaXMgYXR0cmlidXRlIGlzIG5vbi1udWxsYWJsZVxuICAgKiBpLmUuIGlmIHRydWUsIGF0dHJpYnV0ZSB3b3VsZCBiZSBgVHlwZSFgIGFuZCB0aGlzIGF0dHJpYnV0ZVxuICAgKiBtdXN0IGFsd2F5cyBoYXZlIGEgdmFsdWVcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGlzUmVxdWlyZWQ6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIHByb3BlcnR5IGRldGVybWluaW5nIGlmIHRoaXMgYXR0cmlidXRlIGlzIGEgbm9uLW51bGxhYmxlIGxpc3RcbiAgICogaS5lLiBpZiB0cnVlLCBhdHRyaWJ1dGUgd291bGQgYmUgYFsgVHlwZSBdIWAgYW5kIHRoaXMgYXR0cmlidXRlJ3NcbiAgICogbGlzdCBtdXN0IGFsd2F5cyBoYXZlIGEgdmFsdWVcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGlzUmVxdWlyZWRMaXN0OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgb3B0aW9ucyB0byBtYWtlIHRoaXMgZmllbGQgcmVzb2x2YWJsZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vdCBhIHJlc29sdmFibGUgZmllbGRcbiAgICovXG4gIHJlYWRvbmx5IGZpZWxkT3B0aW9ucz86IFJlc29sdmFibGVGaWVsZE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIHRoZSBpbnRlcm1lZGlhdGUgdHlwZSBsaW5rZWQgdG8gdGhpcyBhdHRyaWJ1dGVcbiAgICogKGkuZS4gYW4gaW50ZXJmYWNlIG9yIGFuIG9iamVjdClcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBpbnRlcm1lZGlhdGUgdHlwZVxuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJtZWRpYXRlVHlwZT86IElJbnRlcm1lZGlhdGVUeXBlO1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgc3RyaW5nIGZvciB0aGlzIGF0dHJpYnV0ZVxuICAgKi9cbiAgdG9TdHJpbmcoKTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgYXJndW1lbnRzIGZvciB0aGlzIGZpZWxkXG4gICAqL1xuICBhcmdzVG9TdHJpbmcoKTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSB0aGUgZGlyZWN0aXZlcyBmb3IgdGhpcyBmaWVsZFxuICAgKlxuICAgKiBAcGFyYW0gbW9kZXMgdGhlIGF1dGhvcml6YXRpb24gbW9kZXMgb2YgdGhlIGdyYXBocWwgYXBpXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gYXV0aG9yaXphdGlvbiBtb2Rlc1xuICAgKi9cbiAgZGlyZWN0aXZlc1RvU3RyaW5nKG1vZGVzPzogQXV0aG9yaXphdGlvblR5cGVbXSk6IHN0cmluZ1xufVxuXG4vKipcbiAqIFRoZSBvcHRpb25zIHRvIGFkZCBhIGZpZWxkIHRvIGFuIEludGVybWVkaWF0ZSBUeXBlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkRmllbGRPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBmaWVsZFxuICAgKlxuICAgKiBUaGlzIG9wdGlvbiBtdXN0IGJlIGNvbmZpZ3VyZWQgZm9yIE9iamVjdCwgSW50ZXJmYWNlLFxuICAgKiBJbnB1dCBhbmQgRW51bSBUeXBlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBmaWVsZE5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGZpZWxkTmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSByZXNvbHZhYmxlIGZpZWxkIHRvIGFkZFxuICAgKlxuICAgKiBUaGlzIG9wdGlvbiBtdXN0IGJlIGNvbmZpZ3VyZWQgZm9yIE9iamVjdCwgSW50ZXJmYWNlLFxuICAgKiBJbnB1dCBhbmQgVW5pb24gVHlwZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gSUZpZWxkXG4gICAqL1xuICByZWFkb25seSBmaWVsZD86IElGaWVsZDtcbn1cblxuLyoqXG4gKiBJbnRlcm1lZGlhdGUgVHlwZXMgYXJlIHR5cGVzIHRoYXQgaW5jbHVkZXMgYSBjZXJ0YWluIHNldCBvZiBmaWVsZHNcbiAqIHRoYXQgZGVmaW5lIHRoZSBlbnRpcmV0eSBvZiB5b3VyIHNjaGVtYVxuICovXG5leHBvcnQgaW50ZXJmYWNlIElJbnRlcm1lZGlhdGVUeXBlIHtcbiAgLyoqXG4gICAqIHRoZSBuYW1lIG9mIHRoaXMgdHlwZVxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiB0aGUgYXR0cmlidXRlcyBvZiB0aGlzIHR5cGVcbiAgICovXG4gIHJlYWRvbmx5IGRlZmluaXRpb246IHsgW2tleTogc3RyaW5nXTogSUZpZWxkIH07XG5cbiAgLyoqXG4gICAqIFRoZSBJbnRlcmZhY2UgVHlwZXMgdGhpcyBJbnRlcm1lZGlhdGUgVHlwZSBpbXBsZW1lbnRzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gaW50ZXJmYWNlIHR5cGVzXG4gICAqL1xuICByZWFkb25seSBpbnRlcmZhY2VUeXBlcz86IEludGVyZmFjZVR5cGVbXTtcblxuICAvKipcbiAgICogdGhlIGRpcmVjdGl2ZXMgZm9yIHRoaXMgb2JqZWN0IHR5cGVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBkaXJlY3RpdmVzXG4gICAqL1xuICByZWFkb25seSBkaXJlY3RpdmVzPzogRGlyZWN0aXZlW107XG5cbiAgLyoqXG4gICAqIFRoZSByZXNvbHZlcnMgbGlua2VkIHRvIHRoaXMgZGF0YSBzb3VyY2VcbiAgICovXG4gIHJlc29sdmVycz86IFJlc29sdmVyW107XG5cbiAgLyoqXG4gICAqIHRoZSBpbnRlcm1lZGlhdGUgdHlwZSBsaW5rZWQgdG8gdGhpcyBhdHRyaWJ1dGVcbiAgICogKGkuZS4gYW4gaW50ZXJmYWNlIG9yIGFuIG9iamVjdClcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBpbnRlcm1lZGlhdGUgdHlwZVxuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJtZWRpYXRlVHlwZT86IElJbnRlcm1lZGlhdGVUeXBlO1xuXG4gIC8qKlxuICAgKiBNZXRob2QgY2FsbGVkIHdoZW4gdGhlIHN0cmluZ2lmeWluZyBJbnRlcm1lZGlhdGUgVHlwZXMgZm9yIHNjaGVtYSBnZW5lcmF0aW9uXG4gICAqXG4gICAqIEBwYXJhbSBhcGkgVGhlIGJpbmRpbmcgR3JhcGhRTCBBcGkgW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV1cbiAgICpcbiAgICogQGludGVybmFsXG4gICAqL1xuICBfYmluZFRvR3JhcGhxbEFwaShhcGk6IEdyYXBocWxBcGkpOiBJSW50ZXJtZWRpYXRlVHlwZTtcblxuICAvKipcbiAgICogQ3JlYXRlIGFuIEdyYXBoUUwgVHlwZSByZXByZXNlbnRpbmcgdGhpcyBJbnRlcm1lZGlhdGUgVHlwZVxuICAgKlxuICAgKiBAcGFyYW0gb3B0aW9ucyB0aGUgb3B0aW9ucyB0byBjb25maWd1cmUgdGhpcyBhdHRyaWJ1dGVcbiAgICogLSBpc0xpc3RcbiAgICogLSBpc1JlcXVpcmVkXG4gICAqIC0gaXNSZXF1aXJlZExpc3RcbiAgICovXG4gIGF0dHJpYnV0ZShvcHRpb25zPzogQmFzZVR5cGVPcHRpb25zKTogR3JhcGhxbFR5cGU7XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIHRoZSBzdHJpbmcgb2YgdGhpcyBvYmplY3QgdHlwZVxuICAgKi9cbiAgdG9TdHJpbmcoKTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBZGQgYSBmaWVsZCB0byB0aGlzIEludGVybWVkaWF0ZSBUeXBlXG4gICAqL1xuICBhZGRGaWVsZChvcHRpb25zOiBBZGRGaWVsZE9wdGlvbnMpOiB2b2lkO1xufVxuXG5pbnRlcmZhY2UgRGlyZWN0aXZlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgYXV0aG9yaXphdGlvbiB0eXBlIG9mIHRoaXMgZGlyZWN0aXZlXG4gICAqL1xuICByZWFkb25seSBtb2RlPzogQXV0aG9yaXphdGlvblR5cGU7XG5cbiAgLyoqXG4gICAqIE11dGF0aW9uIGZpZWxkcyBmb3IgYSBzdWJzY3JpcHRpb24gZGlyZWN0aXZlXG4gICAqL1xuICByZWFkb25seSBtdXRhdGlvbkZpZWxkcz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIERpcmVjdGl2ZXMgZm9yIHR5cGVzXG4gKlxuICogaS5lLiBAYXdzX2lhbSBvciBAYXdzX3N1YnNjcmliZVxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIERpcmVjdGl2ZSB7XG4gIC8qKlxuICAgKiBBZGQgdGhlIEBhd3NfaWFtIGRpcmVjdGl2ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpYW0oKTogRGlyZWN0aXZlIHtcbiAgICByZXR1cm4gbmV3IERpcmVjdGl2ZSgnQGF3c19pYW0nLCB7IG1vZGU6IEF1dGhvcml6YXRpb25UeXBlLklBTSB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgdGhlIEBhd3Nfb2lkYyBkaXJlY3RpdmVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgb2lkYygpOiBEaXJlY3RpdmUge1xuICAgIHJldHVybiBuZXcgRGlyZWN0aXZlKCdAYXdzX29pZGMnLCB7IG1vZGU6IEF1dGhvcml6YXRpb25UeXBlLk9JREMgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIHRoZSBAYXdzX2FwaV9rZXkgZGlyZWN0aXZlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFwaUtleSgpOiBEaXJlY3RpdmUge1xuICAgIHJldHVybiBuZXcgRGlyZWN0aXZlKCdAYXdzX2FwaV9rZXknLCB7IG1vZGU6IEF1dGhvcml6YXRpb25UeXBlLkFQSV9LRVkgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIHRoZSBAYXdzX2F1dGggb3IgQGF3c19jb2duaXRvX3VzZXJfcG9vbHMgZGlyZWN0aXZlXG4gICAqXG4gICAqIEBwYXJhbSBncm91cHMgdGhlIGdyb3VwcyB0byBhbGxvdyBhY2Nlc3MgdG9cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY29nbml0byguLi5ncm91cHM6IHN0cmluZ1tdKTogRGlyZWN0aXZlIHtcbiAgICBpZiAoZ3JvdXBzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDb2duaXRvIGF1dGhvcml6YXRpb24gcmVxdWlyZXMgYXQgbGVhc3Qgb25lIENvZ25pdG8gZ3JvdXAgdG8gYmUgc3VwcGxpZWQuIFJlY2VpdmVkOiAke2dyb3Vwcy5sZW5ndGh9YCk7XG4gICAgfVxuICAgIC8vIHRoaXMgZnVuY3Rpb24gY3JlYXRlcyB0aGUgY29nbml0byBncm91cHMgYXMgYSBzdHJpbmcgKGkuZS4gW1wiZ3JvdXAxXCIsIFwiZ3JvdXAyXCIsIFwiZ3JvdXAzXCJdKVxuICAgIGNvbnN0IHN0cmluZ2lmeSA9IChhcnJheTogc3RyaW5nW10pOiBzdHJpbmcgPT4ge1xuICAgICAgcmV0dXJuIGFycmF5LnJlZHVjZSgoYWNjLCBlbGVtZW50KSA9PiBgJHthY2N9XCIke2VsZW1lbnR9XCIsIGAsICcnKS5zbGljZSgwLCAtMik7XG4gICAgfTtcbiAgICByZXR1cm4gbmV3IERpcmVjdGl2ZShgQGF3c19hdXRoKGNvZ25pdG9fZ3JvdXBzOiBbJHtzdHJpbmdpZnkoZ3JvdXBzKX1dKWAsIHsgbW9kZTogQXV0aG9yaXphdGlvblR5cGUuVVNFUl9QT09MIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCB0aGUgQGF3c19zdWJzY3JpYmUgZGlyZWN0aXZlLiBPbmx5IHVzZSBmb3IgdG9wIGxldmVsIFN1YnNjcmlwdGlvbiB0eXBlLlxuICAgKlxuICAgKiBAcGFyYW0gbXV0YXRpb25zIHRoZSBtdXRhdGlvbiBmaWVsZHMgdG8gbGluayB0b1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBzdWJzY3JpYmUoLi4ubXV0YXRpb25zOiBzdHJpbmdbXSk6IERpcmVjdGl2ZSB7XG4gICAgaWYgKG11dGF0aW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgU3Vic2NyaWJlIGRpcmVjdGl2ZSByZXF1aXJlcyBhdCBsZWFzdCBvbmUgbXV0YXRpb24gZmllbGQgdG8gYmUgc3VwcGxpZWQuIFJlY2VpdmVkOiAke211dGF0aW9ucy5sZW5ndGh9YCk7XG4gICAgfVxuICAgIC8vIHRoaXMgZnVuY3Rpb24gY3JlYXRlcyB0aGUgc3Vic2NyaWJlIGRpcmVjdGl2ZSBhcyBhIHN0cmluZyAoaS5lLiBbXCJtdXRhdGlvbl9maWVsZF8xXCIsIFwibXV0YXRpb25fZmllbGRfMlwiXSlcbiAgICBjb25zdCBzdHJpbmdpZnkgPSAoYXJyYXk6IHN0cmluZ1tdKTogc3RyaW5nID0+IHtcbiAgICAgIHJldHVybiBhcnJheS5yZWR1Y2UoKGFjYywgbXV0YXRpb24pID0+IGAke2FjY31cIiR7bXV0YXRpb259XCIsIGAsICcnKS5zbGljZSgwLCAtMik7XG4gICAgfTtcbiAgICByZXR1cm4gbmV3IERpcmVjdGl2ZShgQGF3c19zdWJzY3JpYmUobXV0YXRpb25zOiBbJHtzdHJpbmdpZnkobXV0YXRpb25zKX1dKWAsIHsgbXV0YXRpb25GaWVsZHM6IG11dGF0aW9ucyB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBjdXN0b20gZGlyZWN0aXZlXG4gICAqXG4gICAqIEBwYXJhbSBzdGF0ZW1lbnQgLSB0aGUgZGlyZWN0aXZlIHN0YXRlbWVudCB0byBhcHBlbmRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgY3VzdG9tKHN0YXRlbWVudDogc3RyaW5nKTogRGlyZWN0aXZlIHtcbiAgICByZXR1cm4gbmV3IERpcmVjdGl2ZShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBhdXRob3JpemF0aW9uIHR5cGUgb2YgdGhpcyBkaXJlY3RpdmVcbiAgICpcbiAgICogQGRlZmF1bHQgLSBub3QgYW4gYXV0aG9yaXphdGlvbiBkaXJlY3RpdmVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBtb2RlPzogQXV0aG9yaXphdGlvblR5cGU7XG5cbiAgLyoqXG4gICAqIE11dGF0aW9uIGZpZWxkcyBmb3IgYSBzdWJzY3JpcHRpb24gZGlyZWN0aXZlXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm90IGEgc3Vic2NyaXB0aW9uIGRpcmVjdGl2ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG11dGF0aW9uRmllbGRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIHRoZSBkaXJlY3RpdmUgc3RhdGVtZW50XG4gICAqL1xuICBwcml2YXRlIHN0YXRlbWVudDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiB0aGUgYXV0aG9yaXphdGlvbiBtb2RlcyBmb3IgdGhpcyBpbnRlcm1lZGlhdGUgdHlwZVxuICAgKi9cbiAgcHJvdGVjdGVkIG1vZGVzPzogQXV0aG9yaXphdGlvblR5cGVbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHN0YXRlbWVudDogc3RyaW5nLCBvcHRpb25zPzogRGlyZWN0aXZlT3B0aW9ucykge1xuICAgIHRoaXMuc3RhdGVtZW50ID0gc3RhdGVtZW50O1xuICAgIHRoaXMubW9kZSA9IG9wdGlvbnM/Lm1vZGU7XG4gICAgdGhpcy5tdXRhdGlvbkZpZWxkcyA9IG9wdGlvbnM/Lm11dGF0aW9uRmllbGRzO1xuICB9XG5cbiAgLyoqXG4gICAqIE1ldGhvZCBjYWxsZWQgd2hlbiB0aGUgc3RyaW5naWZ5aW5nIERpcmVjdGl2ZSBmb3Igc2NoZW1hIGdlbmVyYXRpb25cbiAgICpcbiAgICogQHBhcmFtIG1vZGVzIHRoZSBhdXRob3JpemF0aW9uIG1vZGVzXG4gICAqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF9iaW5kVG9BdXRoTW9kZXMobW9kZXM/OiBBdXRob3JpemF0aW9uVHlwZVtdKTogRGlyZWN0aXZlIHtcbiAgICB0aGlzLm1vZGVzID0gbW9kZXM7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR2VuZXJhdGUgdGhlIGRpcmVjdGl2ZSBzdGF0ZW1lbnRcbiAgICovXG4gIHB1YmxpYyB0b1N0cmluZygpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLm1vZGVzICYmIHRoaXMubW9kZSAmJiAhdGhpcy5tb2Rlcy5zb21lKChtb2RlKSA9PiBtb2RlID09PSB0aGlzLm1vZGUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYE5vIEF1dGhvcml6YXRpb24gVHlwZSAke3RoaXMubW9kZX0gZGVjbGFyZWQgaW4gR3JhcGhRTCBBcGkuYCk7XG4gICAgfVxuICAgIGlmICh0aGlzLm1vZGUgPT09IEF1dGhvcml6YXRpb25UeXBlLlVTRVJfUE9PTCAmJiB0aGlzLm1vZGVzICYmIHRoaXMubW9kZXMubGVuZ3RoID4gMSkge1xuICAgICAgdGhpcy5zdGF0ZW1lbnQgPSB0aGlzLnN0YXRlbWVudC5yZXBsYWNlKCdAYXdzX2F1dGgnLCAnQGF3c19jb2duaXRvX3VzZXJfcG9vbHMnKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuc3RhdGVtZW50O1xuICB9XG59XG5cbi8qKlxuICogRW51bSBjb250YWluaW5nIHRoZSBUeXBlcyB0aGF0IGNhbiBiZSB1c2VkIHRvIGRlZmluZSBPYmplY3RUeXBlc1xuICovXG5leHBvcnQgZW51bSBUeXBlIHtcbiAgLyoqXG4gICAqIGBJRGAgc2NhbGFyIHR5cGUgaXMgYSB1bmlxdWUgaWRlbnRpZmllci4gYElEYCB0eXBlIGlzIHNlcmlhbGl6ZWQgc2ltaWxhciB0byBgU3RyaW5nYC5cbiAgICpcbiAgICogT2Z0ZW4gdXNlZCBhcyBhIGtleSBmb3IgYSBjYWNoZSBhbmQgbm90IGludGVuZGVkIHRvIGJlIGh1bWFuLXJlYWRhYmxlLlxuICAgKi9cbiAgSUQgPSAnSUQnLFxuICAvKipcbiAgICogYFN0cmluZ2Agc2NhbGFyIHR5cGUgaXMgYSBmcmVlLWZvcm0gaHVtYW4tcmVhZGFibGUgdGV4dC5cbiAgICovXG4gIFNUUklORyA9ICdTdHJpbmcnLFxuICAvKipcbiAgICogYEludGAgc2NhbGFyIHR5cGUgaXMgYSBzaWduZWQgbm9uLWZyYWN0aW9uYWwgbnVtZXJpY2FsIHZhbHVlLlxuICAgKi9cbiAgSU5UID0gJ0ludCcsXG4gIC8qKlxuICAgKiBgRmxvYXRgIHNjYWxhciB0eXBlIGlzIGEgc2lnbmVkIGRvdWJsZS1wcmVjaXNpb24gZnJhY3Rpb25hbCB2YWx1ZS5cbiAgICovXG4gIEZMT0FUID0gJ0Zsb2F0JyxcbiAgLyoqXG4gICAqIGBCb29sZWFuYCBzY2FsYXIgdHlwZSBpcyBhIGJvb2xlYW4gdmFsdWU6IHRydWUgb3IgZmFsc2UuXG4gICAqL1xuICBCT09MRUFOID0gJ0Jvb2xlYW4nLFxuXG4gIC8qKlxuICAgKiBgQVdTRGF0ZWAgc2NhbGFyIHR5cGUgcmVwcmVzZW50cyBhIHZhbGlkIGV4dGVuZGVkIGBJU08gODYwMSBEYXRlYCBzdHJpbmcuXG4gICAqXG4gICAqIEluIG90aGVyIHdvcmRzLCBhY2NlcHRzIGRhdGUgc3RyaW5ncyBpbiB0aGUgZm9ybSBvZiBgWVlZWS1NTS1ERGAuIEl0IGFjY2VwdHMgdGltZSB6b25lIG9mZnNldHMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9lbi53aWtpcGVkaWEub3JnL3dpa2kvSVNPXzg2MDEjQ2FsZW5kYXJfZGF0ZXNcbiAgICovXG4gIEFXU19EQVRFID0gJ0FXU0RhdGUnLFxuICAvKipcbiAgICogYEFXU1RpbWVgIHNjYWxhciB0eXBlIHJlcHJlc2VudHMgYSB2YWxpZCBleHRlbmRlZCBgSVNPIDg2MDEgVGltZWAgc3RyaW5nLlxuICAgKlxuICAgKiBJbiBvdGhlciB3b3JkcywgYWNjZXB0cyBkYXRlIHN0cmluZ3MgaW4gdGhlIGZvcm0gb2YgYGhoOm1tOnNzLnNzc2AuIEl0IGFjY2VwdHMgdGltZSB6b25lIG9mZnNldHMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9lbi53aWtpcGVkaWEub3JnL3dpa2kvSVNPXzg2MDEjVGltZXNcbiAgICovXG4gIEFXU19USU1FID0gJ0FXU1RpbWUnLFxuICAvKipcbiAgICogYEFXU0RhdGVUaW1lYCBzY2FsYXIgdHlwZSByZXByZXNlbnRzIGEgdmFsaWQgZXh0ZW5kZWQgYElTTyA4NjAxIERhdGVUaW1lYCBzdHJpbmcuXG4gICAqXG4gICAqIEluIG90aGVyIHdvcmRzLCBhY2NlcHRzIGRhdGUgc3RyaW5ncyBpbiB0aGUgZm9ybSBvZiBgWVlZWS1NTS1ERFRoaDptbTpzcy5zc3NaYC4gSXQgYWNjZXB0cyB0aW1lIHpvbmUgb2Zmc2V0cy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2VuLndpa2lwZWRpYS5vcmcvd2lraS9JU09fODYwMSNDb21iaW5lZF9kYXRlX2FuZF90aW1lX3JlcHJlc2VudGF0aW9uc1xuICAgKi9cbiAgQVdTX0RBVEVfVElNRSA9ICdBV1NEYXRlVGltZScsXG4gIC8qKlxuICAgKiBgQVdTVGltZXN0YW1wYCBzY2FsYXIgdHlwZSByZXByZXNlbnRzIHRoZSBudW1iZXIgb2Ygc2Vjb25kcyBzaW5jZSBgMTk3MC0wMS0wMVQwMDowMFpgLlxuICAgKlxuICAgKiBUaW1lc3RhbXBzIGFyZSBzZXJpYWxpemVkIGFuZCBkZXNlcmlhbGl6ZWQgYXMgbnVtYmVycy5cbiAgICovXG4gIEFXU19USU1FU1RBTVAgPSAnQVdTVGltZXN0YW1wJyxcbiAgLyoqXG4gICAqIGBBV1NFbWFpbGAgc2NhbGFyIHR5cGUgcmVwcmVzZW50cyBhbiBlbWFpbCBhZGRyZXNzIHN0cmluZyAoaS5lLmB1c2VybmFtZUBleGFtcGxlLmNvbWApXG4gICAqL1xuICBBV1NfRU1BSUwgPSAnQVdTRW1haWwnLFxuICAvKipcbiAgICogYEFXU0pzb25gIHNjYWxhciB0eXBlIHJlcHJlc2VudHMgYSBKU09OIHN0cmluZy5cbiAgICovXG4gIEFXU19KU09OID0gJ0FXU0pTT04nLFxuICAvKipcbiAgICogYEFXU1VSTGAgc2NhbGFyIHR5cGUgcmVwcmVzZXRucyBhIHZhbGlkIFVSTCBzdHJpbmcuXG4gICAqXG4gICAqIFVSTHMgd2lodG91dCBzY2hlbWVzIG9yIGNvbnRhaW4gZG91YmxlIHNsYXNoZXMgYXJlIGNvbnNpZGVyZWQgaW52YWxpZC5cbiAgICovXG4gIEFXU19VUkwgPSAnQVdTVVJMJyxcbiAgLyoqXG4gICAqIGBBV1NQaG9uZWAgc2NhbGFyIHR5cGUgcmVwcmVzZW50cyBhIHZhbGlkIHBob25lIG51bWJlci4gUGhvbmUgbnVtYmVycyBtYXliZSBiZSB3aGl0ZXNwYWNlIGRlbGltaXRlZCBvciBoeXBoZW5hdGVkLlxuICAgKlxuICAgKiBUaGUgbnVtYmVyIGNhbiBzcGVjaWZ5IGEgY291bnRyeSBjb2RlIGF0IHRoZSBiZWdpbm5pbmcsIGJ1dCBpcyBub3QgcmVxdWlyZWQgZm9yIFVTIHBob25lIG51bWJlcnMuXG4gICAqL1xuICBBV1NfUEhPTkUgPSAnQVdTUGhvbmUnLFxuICAvKipcbiAgICogYEFXU0lQQWRkcmVzc2Agc2NhbGFyIHR5cGUgcmVzcHJlc2VudHMgYSB2YWxpZCBgSVB2NGAgb2YgYElQdjZgIGFkZHJlc3Mgc3RyaW5nLlxuICAgKi9cbiAgQVdTX0lQX0FERFJFU1MgPSAnQVdTSVBBZGRyZXNzJyxcblxuICAvKipcbiAgICogVHlwZSB1c2VkIGZvciBJbnRlcm1lZGlhdGUgVHlwZXNcbiAgICogKGkuZS4gYW4gaW50ZXJmYWNlIG9yIGFuIG9iamVjdCB0eXBlKVxuICAgKi9cbiAgSU5URVJNRURJQVRFID0gJ0lOVEVSTUVESUFURScsXG59Il19