"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Values = exports.AttributeValuesStep = exports.AttributeValues = exports.PartitionKey = exports.PrimaryKey = exports.SortKeyStep = exports.PartitionKeyStep = exports.Assign = exports.KeyCondition = void 0;
const private_1 = require("./private");
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new private_1.BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new private_1.BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new private_1.Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", ${this.arg}))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
      ${assignments.join(',')}
    }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the variables required for `renderTemplate`.
     */
    renderVariables() {
        return `#set($input = ${this.container})
      ${this.assignments.map(a => a.putInMap('input')).join('\n')}`;
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return '"attributeValues": $util.dynamodb.toMapValuesJson($input)';
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
//# sourceMappingURL=data:application/json;base64,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