r'''Galaxy spectrum module.


Models
======

.. autosummary::
   :nosignatures:
   :toctree: ../api/

   dirichlet_coefficients

'''

import numpy as np


def dirichlet_coefficients(redshift, alpha0, alpha1, z1=1.):
    r"""Dirichlet-distributed SED coefficients.

    Spectral coefficients to calculate the rest-frame spectral energy
        distribution of a galaxy following the Herbel et al. model in [1].

    Parameters
    ----------
    redshift : (nz,) array_like
        The redshift values of the galaxies for which the coefficients want to
        be sampled.
    alpha0, alpha1 : (nc,) array_like
        Factors parameterising the Dirichlet distribution according to Equation
        (3.9) in [1].
    z1 : float or scalar, optional
       Reference redshift at which alpha = alpha1. The default value is 1.0.

    Returns
    -------
    coefficients : (nz, nc) ndarray
        The spectral coefficients of the galaxies. The shape is (n, nc) with nz
        the number of redshifts and nc the number of coefficients.

    Notes
    -----
    The rest-frame spectral energy distribution of galaxies can be written as a
    linear combination of the five kcorrect ([2]) template spectra :math:`f_i`
    (see [1])

    .. math::

        f(\lambda) = \sum_{i=1}^5 c_i f_i(\lambda) \;,

    where the coefficients :math:'c_i' were shown to follow a Dirichlet
    distribution of order 5. The five parameters describing the Dirichlet
    distribution are given by

    .. math::

        \alpha_i(z) = (\alpha_{i,0})^{1-z/z_1} \cdot (\alpha_{i,1})^{z/z_1} ;,.

    Here, :math:'\alpha_{i,0}' describes the galaxy population at redshift
    :math:'z=0' and :math:'\alpha_{i,1}' the population at :math:'z=z_1 > 0'.
    These parameters depend on the galaxy type and we chose :math:'z_1=1'.

    This code works for a general number of templates.

    Examples
    --------
    >>> from skypy.galaxy.spectrum import dirichlet_coefficients
    >>> import numpy as np

    Sample the coefficients according to [1] for n blue galaxies with redshifts
    between 0 and 1.

    >>> n = 100000
    >>> alpha0 = np.array([2.079, 3.524, 1.917, 1.992, 2.536])
    >>> alpha1 = np.array([2.265, 3.862, 1.921, 1.685, 2.480])
    >>> redshift = np.linspace(0,2, n)
    >>> coefficients = dirichlet_coefficients(redshift, alpha0, alpha1)


    References
    ----------
    .. [1] Herbel J., Kacprzak T., Amara A. et al., 2017, Journal of Cosmology
           and Astroparticle Physics, Issue 08, article id. 035 (2017)
    .. [2] Blanton M. R., Roweis S., 2007, The Astronomical Journal,
           Volume 133, Page 734
    """
    if np.isscalar(alpha0) or np.isscalar(alpha1):
        raise ValueError("alpha0 and alpha1 must be array_like.")
    return_shape = (*np.shape(redshift), *np.shape(alpha0))
    redshift = np.atleast_1d(redshift)[:, np.newaxis]

    alpha = np.power(alpha0, 1. - redshift / z1) * \
        np.power(alpha1, redshift / z1)

    # To sample Dirichlet distributed variables of order k we first sample k
    # Gamma distributed variables y_i with the parameters alpha_i.
    # Normalising all y_i by the sum of all k y_i gives us the k Dirichlet
    # distributed variables.
    y = np.random.gamma(alpha)
    sum_y = y.sum(axis=1)
    coefficients = np.divide(y.T, sum_y.T).T
    return coefficients.reshape(return_shape)
