"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RdsDataSource = exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const appsync_function_1 = require("./appsync-function");
const appsync_generated_1 = require("./appsync.generated");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) Abstract AppSync datasource implementation.
 *
 * Do not use directly but use subclasses for concrete datasources
 *
 * @experimental
 */
class BaseDataSource extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props, extended) {
        var _a, _b;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        const name = (_a = props.name) !== null && _a !== void 0 ? _a : id;
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: name,
            description: props.description,
            serviceRoleArn: (_b = this.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            ...extended,
        });
        this.name = name;
        this.api = props.api;
    }
    /**
     * (experimental) creates a new resolver for this datasource and API using the given properties.
     *
     * @experimental
     */
    createResolver(props) {
        return this.api.createResolver({ dataSource: this, ...props });
    }
    /**
     * (experimental) creates a new appsync function for this datasource and API using the given properties.
     *
     * @experimental
     */
    createFunction(props) {
        return new appsync_function_1.AppsyncFunction(this, `${props.name}Function`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * (experimental) Abstract AppSync datasource implementation.
 *
 * Do not use directly but use subclasses for resource backed datasources
 *
 * @experimental
 */
class BackedDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * (experimental) An AppSync dummy datasource.
 *
 * @experimental
 */
class NoneDataSource extends BaseDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * (experimental) An AppSync datasource backed by a DynamoDB table.
 *
 * @experimental
 */
class DynamoDbDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * (experimental) An AppSync datasource backed by a http endpoint.
 *
 * @experimental
 */
class HttpDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        const authorizationConfig = props.authorizationConfig ? {
            authorizationType: 'AWS_IAM',
            awsIamConfig: props.authorizationConfig,
        } : undefined;
        super(scope, id, props, {
            type: 'HTTP',
            httpConfig: {
                endpoint: props.endpoint,
                authorizationConfig,
            },
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * (experimental) An AppSync datasource backed by a Lambda function.
 *
 * @experimental
 */
class LambdaDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
/**
 * (experimental) An AppSync datasource backed by RDS.
 *
 * @experimental
 */
class RdsDataSource extends BackedDataSource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'RELATIONAL_DATABASE',
            relationalDatabaseConfig: {
                rdsHttpEndpointConfig: {
                    awsRegion: props.databaseCluster.stack.region,
                    dbClusterIdentifier: core_1.Lazy.string({
                        produce: () => {
                            return core_1.Stack.of(this).formatArn({
                                service: 'rds',
                                resource: `cluster:${props.databaseCluster.clusterIdentifier}`,
                            });
                        },
                    }),
                    awsSecretStoreArn: props.secretStore.secretArn,
                },
                relationalDatabaseSourceType: 'RDS_HTTP_ENDPOINT',
            },
        });
        const clusterArn = core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: `cluster:${props.databaseCluster.clusterIdentifier}`,
        });
        props.secretStore.grantRead(this);
        // Change to grant with RDS grant becomes implemented
        aws_iam_1.Grant.addToPrincipal({
            grantee: this,
            actions: [
                'rds-data:DeleteItems',
                'rds-data:ExecuteSql',
                'rds-data:ExecuteStatement',
                'rds-data:GetItems',
                'rds-data:InsertItems',
                'rds-data:UpdateItems',
            ],
            resourceArns: [clusterArn, `${clusterArn}:*`],
            scope: this,
        });
    }
}
exports.RdsDataSource = RdsDataSource;
//# sourceMappingURL=data:application/json;base64,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