# Licensed under a 3-clause BSD style license - see LICENSE.rst
"""Calculate a calibration factor from a standard flux value."""

from astropy import log
import numpy as np

from sofia_redux.calibration.pipecal_error import PipeCalError

__all__ = ['pipecal_calfac']


def pipecal_calfac(flux, flux_err, config):
    """
    Calculate the calibration factor for a flux standard.

    Flux values are as measured from aperture photometry on
    an uncalibrated image of a known standard source.  It is
    assumed that the image has already been telluric-corrected
    to the reference ZA and altitude/PWV.  The calibration
    factor is intended to divide raw data that has also been
    telluric-corrected to the same reference atmosphere, in order
    to calibrate it to physical units (Jy/pixel).  The
    error on the calibration factory is calculated from the flux
    error and the model flux error, added in quadrature with
    appropriate scale factors.

    The model flux and its associated error must be present
    in the `config` data structure, as well as a mean and pivot
    wavelength, and a color correction factor.  That is, the keys
    'wref', 'lpivot', 'std_flux', 'std_eflux', and 'color_corr'
    must be present in the configuration dictionary.

    The calibration factor is calculated as::

        calfac = (flux * Lpivot^2) /
                 (color_cor * model_flux * Lmean^2)


    Parameters
    ----------
    flux : float
        Observed flux of standard, from aperture photometry.
    flux_err : float
        1-sigma uncertainty on flux.
    config : dictionary
        Describes the configuration of this run, as generated by
        `pipecal_config`.

    Returns
    -------
    ref_calfac : float
        Calibration factor
    ref_ecalfac : float
        1-sigma uncertainty on calibration factor

    Raises
    ------
    PipeCalError
        Any improper inputs

    """

    # Get the values from pipecal config
    try:
        lmean = config['wref']
        lpivot = config['lpivot']
        model_flux = config['std_flux']
        model_flux_err = config['std_eflux']
        color_corr = config['color_corr']
    except KeyError:
        message = 'Invalid config'
        log.error(message)
        raise PipeCalError(message)

    # Check all values are positive
    parameters = np.array([lmean, lpivot, model_flux, model_flux_err,
                           color_corr, flux, flux_err])
    if np.any(parameters <= 0):
        message = 'All inputs to pipecal_calfac must be positive'
        log.error(message)
        raise PipeCalError(message)

    # Calculate the calibration factor at the reference ZA, Alt/PWV
    # Equation:
    #  calfac = (flux * Lpivot**2) /
    #           (color_cor * model_flux * Lmean**2)
    calconst = lpivot**2 / (color_corr * model_flux * lmean**2)
    ref_calfac = flux * calconst
    ref_ecalfac = np.sqrt((flux_err * calconst)**2
                          + (ref_calfac * model_flux_err / 100.)**2)

    return ref_calfac, ref_ecalfac
