# -*- coding: utf-8 -*-
# pylint: disable=R0201

import os
import re

import ast
#==============================================================================
from pyccel.parser.extend_tree import extend_tree
from pyccel.parser.extend_tree import CommentLine, CommentMultiLine
#==============================================================================

from sympy.core.function import Function
from sympy import Symbol
from sympy import IndexedBase
from sympy import Tuple
from sympy import Lambda
from sympy import Dict

#==============================================================================

from pyccel.ast.basic import PyccelAstNode

from pyccel.ast.core import ParserResult
from pyccel.ast.core import String
from pyccel.ast.core import Nil
from pyccel.ast.core import DottedName, DottedVariable
from pyccel.ast.core import Assign
from pyccel.ast.core import AugAssign
from pyccel.ast.core import Return
from pyccel.ast.core import Pass
from pyccel.ast.core import FunctionDef
from pyccel.ast.core import PythonFunction, SympyFunction
from pyccel.ast.core import ClassDef
from pyccel.ast.core import For, FunctionalFor
from pyccel.ast.core import If, IfTernaryOperator
from pyccel.ast.core import While
from pyccel.ast.core import Del
from pyccel.ast.core import Assert
from pyccel.ast.core import PythonTuple
from pyccel.ast.core import Comment, EmptyLine, NewLine
from pyccel.ast.core import Break, Continue
from pyccel.ast.core import Slice
from pyccel.ast.core import Argument, ValuedArgument
from pyccel.ast.core import Is, IsNot
from pyccel.ast.core import Import
from pyccel.ast.core import AsName
from pyccel.ast.core import CommentBlock
from pyccel.ast.core import With
from pyccel.ast.core import List
from pyccel.ast.core import StarredArguments
from pyccel.ast.core import CodeBlock
from pyccel.ast.core import create_variable
from pyccel.ast.core import _atomic

from pyccel.ast.core import PyccelPow, PyccelAdd, PyccelMul, PyccelDiv, PyccelMod, PyccelFloorDiv
from pyccel.ast.core import PyccelEq,  PyccelNe,  PyccelLt,  PyccelLe,  PyccelGt,  PyccelGe
from pyccel.ast.core import PyccelAnd, PyccelOr,  PyccelNot, PyccelMinus, PyccelAssociativeParenthesis
from pyccel.ast.core import PyccelOperator, PyccelUnary

from pyccel.ast.builtins import Print
from pyccel.ast.headers  import Header, MetaVariable
from pyccel.ast.numbers  import Integer, Float, Complex, BooleanFalse, BooleanTrue
from pyccel.ast.functionalexpr import FunctionalSum, FunctionalMax, FunctionalMin

from pyccel.parser.base import BasicParser
from pyccel.parser.utilities import read_file
from pyccel.parser.utilities import get_default_path

from pyccel.parser.syntax.headers import parse as hdr_parse
from pyccel.parser.syntax.openmp  import parse as omp_parse
from pyccel.parser.syntax.openacc import parse as acc_parse

from pyccel.errors.errors import Errors, PyccelSyntaxError

# TODO - remove import * and only import what we need
#      - use OrderedDict whenever it is possible
from pyccel.errors.messages import *

#==============================================================================
errors = Errors()
#==============================================================================

strip_ansi_escape = re.compile(r'(\x9B|\x1B\[)[0-?]*[ -\/]*[@-~]|[\n\t\r]')

# use this to delete ansi_escape characters from a string
# Useful for very coarse version differentiation.

#==============================================================================

def change_priority( expr ):
    """
       Python ast does not parse parentheses.
       This function inserts parentheses if they are required

       Examples
       --------
       >>> change_priority(PyccelMinus(1,PyccelMinus(1,1)))
       PyccelMinus(PyccelMinus(1,1),1)

       >>> change_priority(PyccelMinus(1,PyccelMul(1,1)))
       PyccelMinus(1,PyccelMul(1,1))

       >>> change_priority(PyccelDiv(1,PyccelMul(1,1)))
       PyccelMul(PyccelDiv(1, 1), 1)

    """
    first  = expr.args[0]
    second = expr.args[1]

    if isinstance(first,  PyccelOperator) and first.p  <= expr.p:
        first = PyccelAssociativeParenthesis(first)

    if isinstance(second, PyccelOperator) and second.p <= expr.p:
        second = PyccelAssociativeParenthesis(second)

    expr = expr.func(first, second)

    return expr

class SyntaxParser(BasicParser):

    """ Class for a Syntax Parser.

        inputs: str
            filename or code to parse as a string
    """

    def __init__(self, inputs, **kwargs):
        BasicParser.__init__(self, **kwargs)

        # check if inputs is a file
        code = inputs
        if os.path.isfile(inputs):

            # we don't use is_valid_filename_py since it uses absolute path
            # file extension

            ext = inputs.split(""".""")[-1]
            if not ext in ['py', 'pyh']:
                errors = Errors()
                errors.report(INVALID_FILE_EXTENSION, symbol=ext,
                              severity='fatal')
                errors.check()

            code = read_file(inputs)
            self._filename = inputs

        self._code = code

        self._scope = []

        tree = extend_tree(code)

        self._fst = tree
        

        self.parse(verbose=True)

    def parse(self, verbose=False):
        """converts python ast to sympy ast."""

        if self.syntax_done:
            print ('> syntax analysis already done')
            return self.ast

        # TODO - add settings to Errors
        #      - filename

        errors = Errors()
        if self.filename:
            errors.set_target(self.filename, 'file')
        errors.set_parser_stage('syntax')

        PyccelAstNode.stage = 'syntactic'
        ast = self._visit(self.fst)

        self._ast = ast

        self._visit_done = True

        return ast

    def _treat_iterable(self, stmt):

        return [self._visit(i) for i in stmt]

    def _visit(self, stmt):
        """Creates AST from FST."""

        # TODO - add settings to Errors
        #      - line and column
        #      - blocking errors

        cls = type(stmt)
        syntax_method = '_visit_' + cls.__name__
        if hasattr(self, syntax_method):
            self._scope.append(stmt)
            result = getattr(self, syntax_method)(stmt)
            self._scope.pop()
            return result

        # Unknown object, we raise an error.
        if hasattr(stmt, 'lineno'):
            bounding_box = (stmt.lineno, stmt.col_offset)
        else:
            bounding_box = None
        errors.report(PYCCEL_RESTRICTION_UNSUPPORTED_SYNTAX, symbol=ast.dump(stmt),
                      bounding_box=bounding_box,
                      severity='fatal')

    def _visit_Module(self, stmt):
        """ Visits the ast and splits the result into elements relevant for the module or the program"""
        prog          = []
        mod           = []
        start         = []
        current_file  = start
        targets       = []
        n_empty_lines = 0
        is_prog       = False
        body          = [self._visit(v) for v in stmt.body]

        new_body      = []
        for i in body:
            if isinstance(i, CodeBlock):
                new_body += list(i.body)
            else:
                new_body.append(i)

        body = new_body
        for v in body:

            if n_empty_lines > 3:
                current_file = start
            if isinstance(v,(FunctionDef, ClassDef)):
                # Functions and classes are always defined in a module
                n_empty_lines = 0
                mod.append(v)
                targets.append(v.name)
                current_file = mod
            elif isinstance(v,(Header, Comment, CommentBlock)):
                # Headers and Comments are defined in the same block as the following object
                n_empty_lines = 0
                current_file = start
                current_file.append(v)
            elif isinstance(v, (NewLine, EmptyLine)):
                # EmptyLines are defined in the same block as the previous line
                current_file.append(v)
                n_empty_lines += 1
            elif isinstance(v, Import):
                # Imports are defined in both the module and the program
                n_empty_lines = 0
                mod.append(v)
                prog.append(v)
            else:
                # Everything else is defined in a module
                is_prog = True
                n_empty_lines = 0
                prog.append(v)
                current_file = prog

            # If the current file is now a program or a module. Add headers and comments before the line we just read
            if len(start)>0 and current_file is not start:
                current_file[-1:-1] = start
                start = []
        if len(start)>0:
            mod.extend(start)

        # Define the names of the module and program
        # The module name allows it to be correctly referenced from an import command
        current_mod_name = os.path.splitext(os.path.basename(self._filename))[0]
        prog_name = 'prog_' + current_mod_name
        mod_code = CodeBlock(mod) if len(targets)>0 else None
        if is_prog:
            if mod_code:
                expr = Import(source=current_mod_name, target = targets)
                prog.insert(0,expr)
            prog_code = CodeBlock(prog)
            prog_code.set_fst(stmt)
        else:
            prog_code = None
            # If the file only contains headers
            if mod_code is None:
                mod_code = CodeBlock(mod)
        assert( mod_code is not None or prog_code is not None)
        code = ParserResult(program   = prog_code,
                            module    = mod_code,
                            prog_name = prog_name,
                            mod_name  = current_mod_name)
        code.set_fst(stmt)
        code._fst.lineno=1
        code._fst.col_offset=1
        return code

    def _visit_Expr(self, stmt):
        return self._visit(stmt.value)

    def _visit_Tuple(self, stmt):
        return PythonTuple(*self._treat_iterable(stmt.elts))

    def _visit_List(self, stmt):
        return List(*self._treat_iterable(stmt.elts), sympify=False)

    def _visit_tuple(self, stmt):
        return Tuple(*self._treat_iterable(stmt), sympify=False)

    def _visit_list(self, stmt):
        return self._treat_iterable(stmt)

    def _visit_alias(self, stmt):
        if not isinstance(stmt.name, str):
            raise TypeError('Expecting a string')

        old = self._visit(stmt.name)

        if stmt.asname:
            new = self._visit(stmt.asname)
            return AsName(old, new)
        else:
            return old

    def _visit_Dict(self, stmt):

        d = {}
        for key, value in zip(stmt.keys, stmt.values):

            key = self._visit(key)
            value = self._visit(value)

            # sympy does not allow keys to be strings

            if isinstance(key, String):
                errors.report(SYMPY_RESTRICTION_DICT_KEYS,
                              severity='error')

            d[key] = value
        return Dict(d)

    def _visit_NoneType(self, stmt):
        return Nil()

    def _visit_str(self, stmt):

        return stmt

    def _visit_Str(self, stmt):
        val =  stmt.s
        if isinstance(self._scope[-2], ast.Expr):
            return CommentBlock(val)
        return String(val)

    def _visit_Num(self, stmt):
        val = stmt.n

        if isinstance(val, int):
            return Integer(val)
        elif isinstance(val, float):
            return Float(val)
        elif isinstance(val, complex):
            return Complex(Float(val.real), Float(val.imag))
        else:
            raise NotImplementedError('Num type {} not recognised'.format(type(val)))

    def _visit_Assign(self, stmt):

        lhs = self._visit(stmt.targets)
        if len(lhs)==1:
            lhs = lhs[0]
        else:
            lhs = PythonTuple(*lhs)

        rhs = self._visit(stmt.value)
        expr = Assign(lhs, rhs)

        # we set the fst to keep track of needed information for errors

        expr.set_fst(stmt)
        return expr

    def _visit_AugAssign(self, stmt):

        lhs = self._visit(stmt.target)
        rhs = self._visit(stmt.value)
        if isinstance(stmt.op, ast.Add):
            expr = AugAssign(lhs, '+', rhs)
        elif isinstance(stmt.op, ast.Sub):
            expr = AugAssign(lhs, '-', rhs)
        elif isinstance(stmt.op, ast.Mult):
            expr = AugAssign(lhs, '*', rhs)
        elif isinstance(stmt.op, ast.Div):
            expr = AugAssign(lhs, '/', rhs)
        elif isinstance(stmt.op, ast.Mod):
            expr = AugAssign(lhs, '%', rhs)
        else:
            errors.report(PYCCEL_RESTRICTION_TODO, symbol = stmt,
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='fatal')

        # we set the fst to keep track of needed information for errors

        expr.set_fst(stmt)
        return expr

    def _visit_arguments(self, stmt):
        arguments = []
        if stmt.vararg or stmt.kwarg:
            errors.report(VARARGS, symbol = stmt,
                    bounding_box=(stmt.lineno, stmt.col_offset),
                    severity='fatal')

        if stmt.args:
            n_expl = len(stmt.args)-len(stmt.defaults)
            arguments += [Argument(a.arg) for a in stmt.args[:n_expl]]
            arguments += [ValuedArgument(Argument(a.arg),self._visit(d)) for a,d in zip(stmt.args[n_expl:],stmt.defaults)]
        elif stmt.kwonlyargs:
            arguments += [ValuedArgument(Argument(a.arg),self._visit(d)) for a,d in zip(stmt.kwonlyargs,stmt.kw_defaults)]

        return arguments

    def _visit_Constant(self, stmt):
        # New in python3.8 this class contains NameConstant, Num, and String types
        if stmt.value is None:
            return Nil()

        elif stmt.value is True:
            return BooleanTrue()

        elif stmt.value is False:
            return BooleanFalse()

        elif isinstance(stmt.value, int):
            return Integer(stmt.value)

        elif isinstance(stmt.value, float):
            return Float(stmt.value)

        elif isinstance(stmt.value, complex):
            return Complex(Float(stmt.value.real), Float(stmt.value.imag))

        elif isinstance(stmt.value, str):
            return self._visit_Str(stmt)

        else:
            raise NotImplementedError('Constant type {} not recognised'.format(type(stmt.value)))

    def _visit_NameConstant(self, stmt):
        if stmt.value is None:
            return Nil()

        elif stmt.value is True:
            return BooleanTrue()

        elif stmt.value is False:
            return BooleanFalse()

        else:
            raise NotImplementedError("Unknown NameConstant : {}".format(stmt.value))


    def _visit_Name(self, stmt):
        return Symbol(stmt.id)

    def _treat_import_source(self, source, level):
        source = '.'*level + str(source)
        if source.count('.') == 0:
            source = Symbol(source)
        else:
            source = DottedName(*source.split('.'))

        return get_default_path(source)

    def _visit_Import(self, stmt):
        expr = []
        for name in stmt.names:
            imp = self._visit(name)
            if isinstance(imp, AsName):
                source = AsName(self._treat_import_source(imp.name, 0), imp.target)
            else:
                source = self._treat_import_source(imp, 0)
            import_line = Import(source)
            import_line.set_fst(stmt)
            self.insert_import(import_line)
            expr.append(import_line)

        if len(expr)==1:
            return expr[0]
        else:
            expr = CodeBlock(expr)
            expr.set_fst(stmt)
            return expr

    def _visit_ImportFrom(self, stmt):

        source = self._treat_import_source(stmt.module, stmt.level)

        targets = []
        for i in stmt.names:
            s = self._visit(i)
            if s == '*':
                errors.report(PYCCEL_RESTRICTION_IMPORT_STAR,
                              bounding_box=(stmt.lineno, stmt.col_offset),
                              severity='error')

            targets.append(s)

        expr = Import(source, targets)
        expr.set_fst(stmt)
        self.insert_import(expr)
        return expr

    def _visit_Delete(self, stmt):
        arg = self._visit(stmt.targets)
        return Del(arg)

    def _visit_UnaryOp(self, stmt):

        target = self._visit(stmt.operand)

        if isinstance(stmt.op, ast.Not):
            Func = PyccelNot

        elif isinstance(stmt.op, ast.UAdd):
            Func = PyccelUnary

        elif isinstance(stmt.op, ast.USub):
            Func = PyccelMinus

        elif isinstance(stmt.op, ast.Invert):

            errors.report(PYCCEL_RESTRICTION_UNARY_OPERATOR,
                          bounding_box=(stmt.lineno, stmt.col_offset),
                          severity='fatal')
        else:
            errors.report(PYCCEL_RESTRICTION_UNSUPPORTED_SYNTAX,
                          bounding_box=(stmt.lineno, stmt.col_offset),
                          severity='fatal')

        if isinstance(target,  PyccelOperator) and target.p  <= Func.p:
            target = PyccelAssociativeParenthesis(target)

        return PyccelUnary(Func(target))

    def _visit_BinOp(self, stmt):

        first  = self._visit(stmt.left)
        second = self._visit(stmt.right)

        if isinstance(stmt.op, ast.Add):
            expr = PyccelAdd(first, second)
            return change_priority(expr)

        elif isinstance(stmt.op, ast.Mult):
            expr = PyccelMul(first, second)
            return change_priority(expr)

        elif isinstance(stmt.op, ast.Sub):
            expr = PyccelMinus(first, second)
            return change_priority(expr)

        elif isinstance(stmt.op, ast.Div):
            expr = PyccelDiv(first, second)
            return change_priority(expr)

        elif isinstance(stmt.op, ast.Pow):
            expr = PyccelPow(first, second)
            return change_priority(expr)

        elif isinstance(stmt.op, ast.FloorDiv):
            expr = PyccelFloorDiv(first, second)
            return change_priority(expr)

        elif isinstance(stmt.op, ast.Mod):
            expr = PyccelMod(first, second)
            return change_priority(expr)
        else:
            errors.report(PYCCEL_RESTRICTION_UNSUPPORTED_SYNTAX,
                          bounding_box=(stmt.lineno, stmt.col_offset),
                          severity='fatal')

    def _visit_BoolOp(self, stmt):

        first = self._visit(stmt.values[0])
        second = self._visit(stmt.values[1])

        if isinstance(stmt.op, ast.And):
            if isinstance(second, PyccelOr):
                args  = second.args
                first = PyccelAnd(first, args[0]  )
                return  PyccelOr (first, *args[1:])
            else:
                return PyccelAnd(first, second)

        if isinstance(stmt.op, ast.Or):
            if isinstance(second, PyccelAnd):
                args  = second.args
                first = PyccelOr(first, args[0])
                return  PyccelAnd(first, *args[1:])
            else:
                return PyccelOr(first, second)

        errors.report(PYCCEL_RESTRICTION_UNSUPPORTED_SYNTAX,
                      symbol = ast.dump(stmt.op),
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='fatal')

    def _visit_Compare(self, stmt):
        if len(stmt.ops)>1:
            errors.report(PYCCEL_RESTRICTION_MULTIPLE_COMPARISONS,
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='fatal')

        first = self._visit(stmt.left)
        second = self._visit(stmt.comparators[0])
        op = stmt.ops[0]

        if isinstance(op, ast.Eq):
            return PyccelEq(first, second)
        if isinstance(op, ast.NotEq):
            return PyccelNe(first, second)
        if isinstance(op, ast.Lt):
            return PyccelLt(first, second)
        if isinstance(op, ast.Gt):
            return PyccelGt(first, second)
        if isinstance(op, ast.LtE):
            return PyccelLe(first, second)
        if isinstance(op, ast.GtE):
            return PyccelGe(first, second)
        if isinstance(op, ast.Is):
            return Is(first, second)
        if isinstance(op, ast.IsNot):
            return IsNot(first, second)

        errors.report(PYCCEL_RESTRICTION_UNSUPPORTED_SYNTAX,
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='fatal')

    def _visit_Return(self, stmt):
        results = self._visit(stmt.value)
        if not isinstance(results, (list, PythonTuple, List)):
            results = [results]
        expr = Return(results)
        expr.set_fst(stmt)
        return expr

    def _visit_Pass(self, stmt):
        return Pass()

    def _visit_FunctionDef(self, stmt):

        #  TODO check all inputs and which ones should be treated in stage 1 or 2

        name = self._visit(stmt.name)
        name = name.replace("'", '')

        arguments    = self._visit(stmt.args)

        local_vars   = []
        global_vars  = []
        header       = None
        hide         = False
        kind         = 'function'
        is_pure      = False
        is_elemental = False
        is_private   = False
        imports      = []

        # TODO improve later
        decorators = {str(d) if isinstance(d,Symbol) else str(type(d)): d \
                            for d in self._visit(stmt.decorator_list)}

        if 'bypass' in decorators:
            return EmptyLine()

        if 'stack_array' in decorators:
            args = list(decorators['stack_array'].args)
            for i in range(len(args)):
                args[i] = str(args[i]).replace("'", '')
            decorators['stack_array'] = tuple(args)
        # extract the types to construct a header
        if 'types' in decorators:
            types = []
            results = []
            container = types
            i = 0
            ls = decorators['types'].args
            while i<len(ls) :
                arg = ls[i]

                if isinstance(arg, Symbol):
                    arg = arg.name
                    container.append(arg)
                elif isinstance(arg, String):
                    arg = str(arg)
                    arg = arg.strip("'").strip('"')
                    container.append(arg)
                elif isinstance(arg, ValuedArgument):
                    arg_name = arg.name
                    arg  = arg.value
                    container = results
                    if not arg_name == 'results':
                        msg = 'Argument "{}" provided to the types decorator is not valid'.format(arg_name)
                        errors.report(msg,
                                      bounding_box=(stmt.lineno, stmt.col_offset),
                                      severity='error')
                    else:
                        ls = arg if isinstance(arg, PythonTuple) else [arg]
                        i = -1
                else:
                    msg = 'Invalid argument of type {} passed to types decorator'.format(type(arg))
                    errors.report(msg,
                                  bounding_box=(stmt.lineno, stmt.col_offset),
                                  severity='error')

                i = i+1

            txt  = '#$ header ' + name
            txt += '(' + ','.join(types) + ')'

            if results:
                txt += ' results(' + ','.join(results) + ')'

            header = hdr_parse(stmts=txt)
            if name in self.namespace.static_functions:
                header = header.to_static()

        body = stmt.body

        if 'sympy' in decorators.keys():
            # TODO maybe we should run pylint here
            stmt.decorators.pop()
            func = SympyFunction(name, arguments, [],
                    [stmt.__str__()])
            func.set_fst(stmt)
            self.insert_function(func)
            return EmptyLine()

        elif 'python' in decorators.keys():

            # TODO maybe we should run pylint here

            stmt.decorators.pop()
            func = PythonFunction(name, arguments, [],
                    [stmt.__str__()])
            func.set_fst(stmt)
            self.insert_function(func)
            return EmptyLine()

        else:
            body = self._visit(body)

        if 'pure' in decorators.keys():
            is_pure = True

        if 'elemental' in decorators.keys():
            is_elemental = True

        if 'private' in decorators.keys():
            is_private = True

        returns = [i.expr for i in _atomic(body, cls=Return)]
        assert all(len(i) == len(returns[0]) for i in returns)
        results = []
        for i in zip(*returns):
            if not all(i[0]==j for j in i) or not isinstance(i[0], Symbol):
                results.append(create_variable(i[0]))
            elif isinstance(i[0], Symbol) and any(i[0].name==x.name for x in arguments):
                results.append(create_variable(i[0]))
            else:
                results.append(i[0])

        func = FunctionDef(
               name,
               arguments,
               results,
               body,
               local_vars=local_vars,
               global_vars=global_vars,
               hide=hide,
               kind=kind,
               is_pure=is_pure,
               is_elemental=is_elemental,
               is_private=is_private,
               imports=imports,
               decorators=decorators,
               header=header)

        func.set_fst(stmt)
        return func

    def _visit_ClassDef(self, stmt):

        name = stmt.name
        methods = [self._visit(i) for i in stmt.body if isinstance(i, ast.FunctionDef)]
        for i in methods:
            i.set_cls_name(name)
        attributes = methods[0].arguments
        parent = [self._visit(i) for i in stmt.bases]
        expr = ClassDef(name=name, attributes=attributes,
                        methods=methods, parent=parent)

        # we set the fst to keep track of needed information for errors

        expr.set_fst(stmt)
        return expr

    def _visit_Subscript(self, stmt):

        ch = stmt
        args = []
        while isinstance(ch, ast.Subscript):
            val = self._visit(ch.slice)
            if isinstance(val, (Tuple, PythonTuple)):
                args += val
            else:
                args.insert(0, val)
            ch = ch.value
        args = tuple(args)
        var = self._visit(ch)
        var = IndexedBase(var)[args]
        return var

    def _visit_ExtSlice(self, stmt):
        return self._visit(tuple(stmt.dims))

    def _visit_Slice(self, stmt):

        upper = self._visit(stmt.upper)
        lower = self._visit(stmt.lower)

        if stmt.step is not None:
            raise NotImplementedError("Steps in slices are not implemented")

        if not isinstance(upper, Nil) and not isinstance(lower, Nil):

            return Slice(lower, upper)
        elif not isinstance(lower, Nil):

            return Slice(lower, None)
        elif not isinstance(upper, Nil):

            return Slice(None, upper)
        else:

            return Slice(None, None)

    def _visit_Index(self, stmt):
        return self._visit(stmt.value)

    def _visit_Attribute(self, stmt):
        val  = self._visit(stmt.value)
        attr = Symbol(stmt.attr)
        return DottedVariable(val, attr)


    def _visit_Call(self, stmt):

        args = []
        if stmt.args:
            args += self._visit(stmt.args)
        if stmt.keywords:
            args += self._visit(stmt.keywords)

        if len(args) == 0:
            args = ()

        func = self._visit(stmt.func)

        if isinstance(func, Symbol):
            f_name = func.name
            if str(f_name) == "print":
                func = Print(PythonTuple(*args))
            else:
                func = Function(f_name)(*args)
        elif isinstance(func, DottedVariable):
            f_name = func.rhs.name
            func_attr = Function(f_name)(*args)
            func = DottedVariable(func.lhs, func_attr)
        else:
            raise NotImplementedError(' Unknown function type {}'.format(str(type(func))))

        return func

    def _visit_keyword(self, stmt):

        target = stmt.arg
        val = self._visit(stmt.value)
        return ValuedArgument(target, val)

    def _visit_For(self, stmt):

        iterator = self._visit(stmt.target)
        iterable = self._visit(stmt.iter)
        body = self._visit(stmt.body)
        expr = For(iterator, iterable, body, strict=False)
        expr.set_fst(stmt)
        return expr

    def _visit_comprehension(self, stmt):

        iterator = self._visit(stmt.target)
        iterable = self._visit(stmt.iter)
        ifs = stmt.ifs
        expr = For(iterator, iterable, [], strict=False)
        expr.set_fst(stmt)
        return expr

    def _visit_ListComp(self, stmt):

        import numpy as np
        result = self._visit(stmt.elt)
        generators = list(self._visit(stmt.generators))

        if not isinstance(self._scope[-2],ast.Assign):
            errors.report(PYCCEL_RESTRICTION_LIST_COMPREHENSION_ASSIGN,
                          symbol = ast.dump(stmt),
                          bounding_box=(stmt.lineno, stmt.col_offset),
                          severity='error')
            lhs = create_variable(result)
        else:
            lhs = self._visit(self._scope[-2].targets)
            if len(lhs)==1:
                lhs = lhs[0]
            else:
                raise NotImplementedError("A list comprehension cannot be unpacked")

        index = create_variable(lhs)

        args = [index]
        target = IndexedBase(lhs)[args]
        target = Assign(target, result)
        assign1 = Assign(index, Integer(0))
        assign1.set_fst(stmt)
        target.set_fst(stmt)
        generators[-1].insert2body(target)
        assign2 = Assign(index, PyccelAdd(index, Integer(1)))
        assign2.set_fst(stmt)
        generators[-1].insert2body(assign2)

        indices = [generators[-1].target]
        while len(generators) > 1:
            F = generators.pop()
            generators[-1].insert2body(F)
            indices.append(generators[-1].target)
        indices = indices[::-1]
        return FunctionalFor([assign1, generators[-1]],target.rhs, target.lhs,
                             indices, index)

    def _visit_GeneratorExp(self, stmt):


        result = self._visit(stmt.elt)

        generators = self._visit(stmt.generators)
        parent = self._scope[-3]
        if not isinstance(parent, ast.Call):
            raise NotImplementedError("GeneratorExp is not the argument of a function call")

        name = str(self._visit(parent.func))

        grandparent = self._scope[-4]
        if isinstance(grandparent, ast.Assign):
            if len(grandparent.targets) != 1:
                raise NotImplementedError("Cannot unpack function with generator expression argument")
            lhs = self._visit(grandparent.targets[0])
            cond = False
        else:
            lhs = create_variable(result)
            cond = True

        body = result
        if name == 'sum':
            body = AugAssign(lhs, '+', body)
        else:
            body = Function(name)(lhs, body)
            body = Assign(lhs, body)

        body.set_fst(parent)
        indices = []
        generators = list(generators)
        while len(generators) > 0:
            indices.append(generators[-1].target)
            generators[-1].insert2body(body)
            body = generators.pop()
        indices = indices[::-1]
        body = [body]
        if name == 'sum':
            expr = FunctionalSum(body, result, lhs, indices, None)
        elif name == 'min':
            expr = FunctionalMin(body, result, lhs, indices, None)
        elif name == 'max':
            expr = FunctionalMax(body, result, lhs, indices, None)
        else:
            errors.report(PYCCEL_RESTRICTION_TODO,
                          symbol = name,
                          bounding_box=(stmt.lineno, stmt.col_offset),
                          severity='fatal')

        expr.set_fst(stmt)
        return expr

    def _visit_If(self, stmt):

        test = self._visit(stmt.test)
        body = self._visit(stmt.body)
        orelse = self._visit(stmt.orelse)

        if len(orelse)==1 and isinstance(orelse[0],If):
            orelse = orelse[0]._args
            return If(Tuple(test, body, sympify=False), *orelse)
        else:
            orelse = Tuple(BooleanTrue(), orelse, sympify=False)
            return If(Tuple(test, body, sympify=False), orelse)

    def _visit_IfExp(self, stmt):

        test1 = self._visit(stmt.test)
        first = self._visit(stmt.body)
        second = self._visit(stmt.orelse)
        args = [Tuple(test1, [first], sympify=False),
                Tuple(BooleanTrue(), [second], sympify=False)]
        expr = IfTernaryOperator(*args)
        expr.set_fst(stmt)
        return expr

    def _visit_While(self, stmt):

        test = self._visit(stmt.test)
        body = self._visit(stmt.body)
        return While(test, body)

    def _visit_Assert(self, stmt):
        expr = self._visit(stmt.test)
        return Assert(expr)

    def _visit_CommentMultiLine(self, stmt):

        exprs = []
        # if annotated comment
        for com in stmt.s.split('\n'):
            if com.startswith('#$'):
                env = com[2:].lstrip()
                if env.startswith('omp'):
                    exprs.append(omp_parse(stmts=com))
                elif env.startswith('acc'):
                    exprs.append(acc_parse(stmts=com))
                elif env.startswith('header'):
                    expr = hdr_parse(stmts=com)
                    if isinstance(expr, MetaVariable):

                        # a metavar will not appear in the semantic stage.
                        # but can be used to modify the ast

                        self._metavars[str(expr.name)] = str(expr.value)
                        expr = EmptyLine()
                    else:
                        expr.set_fst(stmt)

                    exprs.append(expr)
                else:
                    errors.report(PYCCEL_INVALID_HEADER,
                                 bounding_box=(stmt.lineno, stmt.col_offset),
                                  severity='error')
            else:

                txt = com[1:].lstrip()
                exprs.append(Comment(txt))

        if len(exprs) == 1:
            return exprs[0]
        else:
            return CodeBlock(exprs)

    def _visit_CommentLine(self, stmt):

        # if annotated comment

        if stmt.s.startswith('#$'):
            env = stmt.s[2:].lstrip()
            if env.startswith('omp'):
                return omp_parse(stmts=stmt.s)
            elif env.startswith('acc'):
                return acc_parse(stmts=stmt.s)
            elif env.startswith('header'):
                expr = hdr_parse(stmts=stmt.s)
                if isinstance(expr, MetaVariable):

                    # a metavar will not appear in the semantic stage.
                    # but can be used to modify the ast

                    self._metavars[str(expr.name)] = str(expr.value)
                    expr = EmptyLine()
                else:
                    expr.set_fst(stmt)

                return expr
            else:

                errors.report(PYCCEL_INVALID_HEADER,
                              bounding_box=(stmt.lineno, stmt.col_offset),
                              severity='error')

        else:
            txt = stmt.s[1:].lstrip()
            return Comment(txt)

    def _visit_Break(self, stmt):
        return Break()

    def _visit_Continue(self, stmt):
        return Continue()

    def _visit_Lambda(self, stmt):

        expr = self._visit(stmt.body)
        args = self._visit(stmt.args)

        return Lambda(tuple(args), expr)

    def _visit_withitem(self, stmt):
        # stmt.optional_vars
        context = self._visit(stmt.context_expr)
        if stmt.optional_vars:
            return AsName(context, stmt.optional_vars)
        else:
            return context

    def _visit_With(self, stmt):
        domain = self._visit(stmt.items)
        if len(domain) == 1:
            domain = domain[0]
        body = self._visit(stmt.body)
        settings = None
        return With(domain, body, settings)

    def _visit_Try(self, stmt):
        # this is a blocking error, since we don't want to convert the try body
        errors.report(PYCCEL_RESTRICTION_TRY_EXCEPT_FINALLY,
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='error')

    def _visit_Raise(self, stmt):
        errors.report(PYCCEL_RESTRICTION_RAISE,
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='error')

    def _visit_Yield(self, stmt):
        errors.report(PYCCEL_RESTRICTION_YIELD,
                      bounding_box=(stmt.lineno, stmt.col_offset),
                      severity='error')

    def _visit_Starred(self, stmt):
        return StarredArguments(self._visit(stmt.value))

#==============================================================================


if __name__ == '__main__':
    import sys

    try:
        filename = sys.argv[1]
    except IndexError:
        raise ValueError('Expecting an argument for filename')

    parser = SyntaxParser(filename)
    print(parser.ast)

