import { IAppsyncFunction } from './appsync-function';
import { BaseDataSource } from './data-source';
import { AuthorizationType } from './graphqlapi';
import { MappingTemplate } from './mapping-template';
import { Type, IField, IIntermediateType, Directive } from './schema-base';
/**
 * Base options for GraphQL Types
 *
 * @option isList - is this attribute a list
 * @option isRequired - is this attribute non-nullable
 * @option isRequiredList - is this attribute a non-nullable list
 *
 */
export interface BaseTypeOptions {
    /**
     * property determining if this attribute is a list
     * i.e. if true, attribute would be [Type]
     *
     * @default - false
     */
    readonly isList?: boolean;
    /**
     * property determining if this attribute is non-nullable
     * i.e. if true, attribute would be Type!
     *
     * @default - false
     */
    readonly isRequired?: boolean;
    /**
     * property determining if this attribute is a non-nullable list
     * i.e. if true, attribute would be [ Type ]!
     * or if isRequired true, attribe would be [ Type! ]!
     *
     * @default - false
     */
    readonly isRequiredList?: boolean;
}
/**
 * Options for GraphQL Types
 *
 * @option isList - is this attribute a list
 * @option isRequired - is this attribute non-nullable
 * @option isRequiredList - is this attribute a non-nullable list
 * @option objectType - the object type linked to this attribute
 *
 */
export interface GraphqlTypeOptions extends BaseTypeOptions {
    /**
     * the intermediate type linked to this attribute
     * @default - no intermediate type
     */
    readonly intermediateType?: IIntermediateType;
}
/**
 * The GraphQL Types in AppSync's GraphQL. GraphQL Types are the
 * building blocks for object types, queries, mutations, etc. They are
 * types like String, Int, Id or even Object Types you create.
 *
 * i.e. `String`, `String!`, `[String]`, `[String!]`, `[String]!`
 *
 * GraphQL Types are used to define the entirety of schema.
 */
export declare class GraphqlType implements IField {
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static id(options?: BaseTypeOptions): GraphqlType;
    /**
     * `String` scalar type is a free-form human-readable text.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static string(options?: BaseTypeOptions): GraphqlType;
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static int(options?: BaseTypeOptions): GraphqlType;
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static float(options?: BaseTypeOptions): GraphqlType;
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static boolean(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDate(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTime(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDateTime(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTimestamp(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsEmail(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSJson` scalar type represents a JSON string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsJson(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsUrl(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsPhone(options?: BaseTypeOptions): GraphqlType;
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsIpAddress(options?: BaseTypeOptions): GraphqlType;
    /**
     * an intermediate type to be added as an attribute
     * (i.e. an interface or an object type)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     * - intermediateType
     */
    static intermediate(options?: GraphqlTypeOptions): GraphqlType;
    /**
     * the type of attribute
     */
    readonly type: Type;
    /**
     * property determining if this attribute is a list
     * i.e. if true, attribute would be `[Type]`
     *
     * @default - false
     */
    readonly isList: boolean;
    /**
     * property determining if this attribute is non-nullable
     * i.e. if true, attribute would be `Type!` and this attribute
     * must always have a value
     *
     * @default - false
     */
    readonly isRequired: boolean;
    /**
     * property determining if this attribute is a non-nullable list
     * i.e. if true, attribute would be `[ Type ]!` and this attribute's
     * list must always have a value
     *
     * @default - false
     */
    readonly isRequiredList: boolean;
    /**
     * the intermediate type linked to this attribute
     * (i.e. an interface or an object)
     *
     * @default - no intermediate type
     */
    readonly intermediateType?: IIntermediateType;
    protected constructor(type: Type, options?: GraphqlTypeOptions);
    /**
     * Generate the string for this attribute
     */
    toString(): string;
    /**
     * Generate the arguments for this field
     */
    argsToString(): string;
    /**
     * Generate the directives for this field
     */
    directivesToString(_modes?: AuthorizationType[]): string;
}
/**
 * Properties for configuring a field
 *
 * @options args - the variables and types that define the arguments
 *
 * i.e. { string: GraphqlType, string: GraphqlType }
 */
export interface FieldOptions {
    /**
     * The return type for this field
     */
    readonly returnType: GraphqlType;
    /**
     * The arguments for this field.
     *
     * i.e. type Example (first: String second: String) {}
     * - where 'first' and 'second' are key values for args
     * and 'String' is the GraphqlType
     *
     * @default - no arguments
     */
    readonly args?: {
        [key: string]: GraphqlType;
    };
    /**
     * the directives for this field
     *
     * @default - no directives
     */
    readonly directives?: Directive[];
}
/**
 * Fields build upon Graphql Types and provide typing
 * and arguments.
 */
export declare class Field extends GraphqlType implements IField {
    /**
     * The options for this field
     *
     * @default - no arguments
     */
    readonly fieldOptions?: ResolvableFieldOptions;
    constructor(options: FieldOptions);
    /**
     * Generate the args string of this resolvable field
     */
    argsToString(): string;
    /**
     * Generate the directives for this field
     */
    directivesToString(modes?: AuthorizationType[]): string;
}
/**
 * Properties for configuring a resolvable field
 *
 * @options dataSource - the data source linked to this resolvable field
 * @options requestMappingTemplate - the mapping template for requests to this resolver
 * @options responseMappingTemplate - the mapping template for responses from this resolver
 */
export interface ResolvableFieldOptions extends FieldOptions {
    /**
     * The data source creating linked to this resolvable field
     *
     * @default - no data source
     */
    readonly dataSource?: BaseDataSource;
    /**
     * configuration of the pipeline resolver
     *
     * @default - no pipeline resolver configuration
     * An empty array or undefined prop will set resolver to be of type unit
     */
    readonly pipelineConfig?: IAppsyncFunction[];
    /**
     * The request mapping template for this resolver
     *
     * @default - No mapping template
     */
    readonly requestMappingTemplate?: MappingTemplate;
    /**
     * The response mapping template for this resolver
     *
     * @default - No mapping template
     */
    readonly responseMappingTemplate?: MappingTemplate;
}
/**
 * Resolvable Fields build upon Graphql Types and provide fields
 * that can resolve into operations on a data source.
 */
export declare class ResolvableField extends Field implements IField {
    /**
     * The options to make this field resolvable
     *
     * @default - not a resolvable field
     */
    readonly fieldOptions?: ResolvableFieldOptions;
    constructor(options: ResolvableFieldOptions);
}
