import argparse
import os
import sys

__author__ = "Moritz Steigerwald"
__version__ = "0.1.0"
__license__ = "MIT"

from src.pipe.DemoDownloader import DemoDownloader
from src.pipe.EventGatherer import EventGatherer
from src.pipe.EventType import EventType


class GoScrape(object):
    """
    Main class for the CLI to handle all incoming commands and parse given arguments
    """

    def __init__(self) -> None:
        """
        Top level argument parser to handle the main commands [events, fetch] and --version
        """
        parser = argparse.ArgumentParser(
            description='A universal scraping tool to acquire CS:GO demofiles from professional esports events '
                        'provided by hltv.org',
            usage='''goscrape <command> [<args>]

The most commonly used goscrape commands are:
   events     Identify all professional csgo events in a given timeframe and create a lookup
   fetch      Download all demofiles for either a given eventid or by providing a lookup file generated by the events command
''')
        # Specify output of "--version"
        parser.add_argument(
            "--version",
            action="version",
            version="%(prog)s (version {version})".format(version=__version__))

        parser.add_argument('command', help='Subcommand to run', choices=[
            'events', 'fetch'])
        args = parser.parse_args(sys.argv[1:2])
        if not hasattr(self, args.command):
            print('Unrecognized command')
            parser.print_help()
            exit(1)

        getattr(self, args.command)()

    @staticmethod
    def dir_path(path: str) -> str:
        """
        verify if the given path string is a valid directory
        :param path: a path to be checked
        :return: the given path as string if invalid throws error
        """
        if os.path.isdir(path):
            return path
        else:
            raise NotADirectoryError(path)

    def events(self) -> None:
        """
        The main events command to identify all professional cs:go events in a given timeframe and create a lookup
        to further download demofiles or gather other game statistics
        :return: None
        """
        parser = argparse.ArgumentParser(
            description='Identify all professional cs:go events in a given timeframe and create a lookup to further '
                        'download demofiles or gather game statistics')

        parser.add_argument("-s", "--start", action="store", dest="start",
                            help='the start date from when events should be pulled from hltv.org', required=True,
                            metavar="STARTDATE")

        parser.add_argument("-e", "--end", action="store", dest="end",
                            help='the end date to which events should be pulled from hltv.org', required=True,
                            metavar="ENDDATE")

        parser.add_argument("-f", "--file", action="store", dest="filepath",
                            help='the output filepath to which the json lookup should be saved', default=None,
                            type=self.dir_path, metavar="STORAGEPATH")

        parser.add_argument("-m", "--matches", action='store_true',
                            help='whether match information and demo urls should be included as well')

        parser.add_argument('-t', '--type', type=EventType, choices=list(EventType),
                            default=EventType.ONLINE, help='the event type', metavar="EVENT TYPE")

        args = parser.parse_args(sys.argv[2:])
        gatherer = EventGatherer()

        start = args.start
        end = args.end
        filepath = args.filepath
        get_matches = args.matches
        event_type = args.type
        gatherer.get_events_lookup(
            start=start, end=end, storage_path=filepath, type=event_type, get_matches=get_matches)

    def fetch(self) -> None:
        """
        the main fetch command to download all demofiles for either a given eventid or by providing a lookup file
        generated by the events command from hltv.org
        :return: None
        """
        parser = argparse.ArgumentParser(
            description='Download all demofiles for either a given eventid or by providing a lookup file generated by '
                        'the events command from hltv.org')

        event_group = parser.add_mutually_exclusive_group(required=True)

        event_group.add_argument("-e", "--event", action="store", dest="eventid",
                                 help='the event id if just one event should be scraped and not an entire file',
                                 type=int, metavar="EVENT ID")

        event_group.add_argument("-f", "--file", action="store", dest="source_file",
                                 help='the lookup file provided by the events command from which the demos should be '
                                      'downloaded',
                                 default=None, metavar="LOOKUP FILE")

        parser.add_argument("-o", "--output", action="store", dest="output_file",
                            help='the output filepath to which the demofiles should be downloaded', default=None,
                            type=self.dir_path, metavar="STORAGEPATH")

        parser.add_argument("-m", "--multiprocessing", action='store_true',
                            help='whether multiprocessing should be utilized or not')

        args = parser.parse_args(sys.argv[2:])
        downloader = DemoDownloader()
        event_id = args.eventid
        source_file_path = args.source_file
        output_file_path = args.output_file
        multiprocessing = args.multiprocessing
        downloader.download_demos(event_id=event_id, event_file=source_file_path,
                                  output_file=output_file_path, multiprocessing=multiprocessing)
