"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
    get keyArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'kms',
            // aliasName already contains the '/'
            resource: this.aliasName,
        });
    }
    get keyId() {
        return this.aliasName;
    }
    addAlias(alias) {
        return this.aliasTargetKey.addAlias(alias);
    }
    addToResourcePolicy(statement, allowNoOp) {
        this.aliasTargetKey.addToResourcePolicy(statement, allowNoOp);
    }
    grant(grantee, ...actions) {
        return this.aliasTargetKey.grant(grantee, ...actions);
    }
    grantDecrypt(grantee) {
        return this.aliasTargetKey.grantDecrypt(grantee);
    }
    grantEncrypt(grantee) {
        return this.aliasTargetKey.grantEncrypt(grantee);
    }
    grantEncryptDecrypt(grantee) {
        return this.aliasTargetKey.grantEncryptDecrypt(grantee);
    }
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    constructor(scope, id, props) {
        let aliasName = props.aliasName;
        if (!core_1.Token.isUnresolved(aliasName)) {
            if (!aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                aliasName = REQUIRED_ALIAS_PREFIX + aliasName;
            }
            if (aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${aliasName}`);
            }
            if (aliasName.toLocaleLowerCase().startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${aliasName}`);
            }
            if (!aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error(`Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-`);
            }
        }
        super(scope, id, {
            physicalName: aliasName,
        });
        this.aliasTargetKey = props.targetKey;
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: this.physicalName,
            targetKeyId: this.aliasTargetKey.keyArn
        });
        this.aliasName = this.getResourceNameAttribute(resource.aliasName);
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
    }
    static fromAliasAttributes(scope, id, attrs) {
        // tslint:disable-next-line: class-name
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    generatePhysicalName() {
        return REQUIRED_ALIAS_PREFIX + super.generatePhysicalName();
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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