"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateCloudfrontSite = exports.CreateBasicSite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const route53 = require("aws-cdk-lib/aws-route53");
const targets = require("aws-cdk-lib/aws-route53-targets");
const s3 = require("aws-cdk-lib/aws-s3");
const s3deploy = require("aws-cdk-lib/aws-s3-deployment");
const core_1 = require("aws-cdk-lib/core");
const constructs_1 = require("constructs");
/**
 * @stability stable
 */
class CreateBasicSite extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const hostedZoneLookup = route53.HostedZone.fromLookup(this, 'WebsiteHostedZone', {
            domainName: props.hostedZone,
        });
        const websiteRedirectBucket = new s3.Bucket(scope, 'WebsiteRedirectBucket', {
            bucketName: `www.${props.hostedZone}`,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            websiteRedirect: {
                hostName: props.hostedZone,
                protocol: s3.RedirectProtocol.HTTP,
            },
        });
        const websiteBucket = new s3.Bucket(scope, 'WebsiteBucket', {
            bucketName: props.hostedZone,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            websiteIndexDocument: props.indexDoc,
            websiteErrorDocument: props.errorDoc,
            publicReadAccess: true,
            encryption: s3.BucketEncryption.S3_MANAGED,
        });
        new s3deploy.BucketDeployment(scope, 'WebsiteDeploy', {
            sources: [s3deploy.Source.asset(props.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        new route53.ARecord(scope, 'WebisteAlias', {
            zone: hostedZoneLookup,
            recordName: props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(websiteBucket)),
        });
        new route53.ARecord(scope, 'WebisteRedirectAlias', {
            zone: hostedZoneLookup,
            recordName: `www.${props.hostedZone}`,
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(websiteRedirectBucket)),
        });
    }
}
exports.CreateBasicSite = CreateBasicSite;
_a = JSII_RTTI_SYMBOL_1;
CreateBasicSite[_a] = { fqn: "cdk-simplewebsite-deploy.CreateBasicSite", version: "2.0.1" };
/**
 * @stability stable
 */
class CreateCloudfrontSite extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.domain && props.subDomain) {
            throw new Error('Domain and sub domain parameters cannot both be defined');
        }
        const hostedZoneLookup = route53.HostedZone.fromLookup(this, 'WebsiteHostedZone', {
            domainName: props.hostedZone,
        });
        const errorResponses = [];
        if (props.errorDoc) {
            errorResponses.push({
                httpStatus: 404,
                responsePagePath: `/${props.errorDoc}`,
            });
            errorResponses.push({
                httpStatus: 403,
                responsePagePath: `/${props.errorDoc}`,
            });
        }
        else {
            errorResponses.push({
                httpStatus: 404,
                responseHttpStatus: 200,
                responsePagePath: `/${props.indexDoc}`,
            });
            errorResponses.push({
                httpStatus: 403,
                responseHttpStatus: 200,
                responsePagePath: `/${props.indexDoc}`,
            });
        }
        const subjectAlternativeNames = [];
        if (props.domain)
            subjectAlternativeNames.push(props.domain);
        if (props.subDomain)
            subjectAlternativeNames.push(props.subDomain);
        const websiteCert = new acm.DnsValidatedCertificate(this, 'WebsiteCert', {
            domainName: props.hostedZone,
            subjectAlternativeNames,
            hostedZone: hostedZoneLookup,
            region: 'us-east-1',
        });
        const websiteBucket = new s3.Bucket(scope, 'WebsiteBucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
        });
        const domainNames = [];
        if (props.domain) {
            domainNames.push(props.domain);
        }
        else {
            domainNames.push(props.hostedZone);
        }
        if (props.subDomain)
            domainNames.push(props.subDomain);
        const websiteDist = new cloudfront.Distribution(scope, 'WebsiteDist', {
            defaultBehavior: {
                origin: new origins.S3Origin(websiteBucket),
                allowedMethods: cloudfront.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
                viewerProtocolPolicy: cloudfront.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2019,
            priceClass: props.priceClass
                ? props.priceClass
                : cloudfront.PriceClass.PRICE_CLASS_100,
            errorResponses,
            defaultRootObject: props.indexDoc,
            domainNames,
            certificate: websiteCert,
        });
        new s3deploy.BucketDeployment(scope, 'WebsiteDeploy', {
            sources: [s3deploy.Source.asset(props.websiteFolder)],
            destinationBucket: websiteBucket,
            distribution: websiteDist,
        });
        new route53.ARecord(scope, 'WebisteDomainAlias', {
            zone: hostedZoneLookup,
            recordName: props.domain ? props.domain : props.hostedZone,
            target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(websiteDist)),
        });
        if (props.subDomain) {
            new route53.ARecord(scope, 'WebisteSubDomainAlias', {
                zone: hostedZoneLookup,
                recordName: props.subDomain,
                target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(websiteDist)),
            });
        }
    }
}
exports.CreateCloudfrontSite = CreateCloudfrontSite;
_b = JSII_RTTI_SYMBOL_1;
CreateCloudfrontSite[_b] = { fqn: "cdk-simplewebsite-deploy.CreateCloudfrontSite", version: "2.0.1" };
//# sourceMappingURL=data:application/json;base64,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