"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectNotificationEvents = exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_canned_metrics_generated_1 = require("./codebuild-canned-metrics.generated");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
const VPC_POLICY_SYM = Symbol.for('@aws-cdk/aws-codebuild.roleVpcPolicy');
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    enableBatchBuilds() {
        return undefined;
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', { target });
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensionsMap: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.buildsSum, props);
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.durationAverage, props);
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.succeededBuildsSum, props);
    }
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.failedBuildsSum, props);
    }
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    notifyOnBuildSucceeded(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [ProjectNotificationEvents.BUILD_SUCCEEDED],
        });
    }
    notifyOnBuildFailed(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [ProjectNotificationEvents.BUILD_FAILED],
        });
    }
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.projectArn,
        };
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ProjectName: this.projectName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_ProjectProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props, environmentVariables),
            fileSystemLocations: core_1.Lazy.any({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.string({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            queuedTimeoutInMinutes: props.queuedTimeout && props.queuedTimeout.toMinutes(),
            concurrentBuildLimit: props.concurrentBuildLimit,
            secondarySources: core_1.Lazy.any({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.any({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.any({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
            logsConfig: this.renderLoggingConfiguration(props.logging),
            buildBatchConfig: core_1.Lazy.any({
                produce: () => {
                    const config = this._batchServiceRole ? {
                        serviceRole: this._batchServiceRole.roleArn,
                    } : undefined;
                    return config;
                },
            }),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                    'codebuild:BatchPutCodeCoverages',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        if (isBindableBuildImage(this.buildImage)) {
            this.buildImage.bind(this, this, {});
        }
    }
    static fromProjectArn(scope, id, projectArn) {
        const parsedArn = core_1.Stack.of(scope).splitArn(projectArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i, {
                    account: parsedArn.account,
                    region: parsedArn.region,
                });
                this.projectArn = projectArn;
                this.projectName = parsedArn.resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable},
     * which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty},
     * which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables
     * @param validateNoPlainTextSecrets whether to throw an exception
     *   if any of the plain text environment variables contain secrets, defaults to 'false'
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     */
    static serializeEnvVariables(environmentVariables, validateNoPlainTextSecrets = false, principal) {
        var _d;
        const ret = new Array();
        const ssmIamResources = new Array();
        const secretsManagerIamResources = new Set();
        const kmsIamResources = new Set();
        for (const [name, envVariable] of Object.entries(environmentVariables)) {
            const envVariableValue = (_d = envVariable.value) === null || _d === void 0 ? void 0 : _d.toString();
            const cfnEnvVariable = {
                name,
                type: envVariable.type || BuildEnvironmentVariableType.PLAINTEXT,
                value: envVariableValue,
            };
            ret.push(cfnEnvVariable);
            // validate that the plain-text environment variables don't contain any secrets in them
            if (validateNoPlainTextSecrets && cfnEnvVariable.type === BuildEnvironmentVariableType.PLAINTEXT) {
                const fragments = core_1.Tokenization.reverseString(cfnEnvVariable.value);
                for (const token of fragments.tokens) {
                    if (token instanceof core_1.SecretValue) {
                        throw new Error(`Plaintext environment variable '${name}' contains a secret value! ` +
                            'This means the value of this variable will be visible in plain text in the AWS Console. ' +
                            "Please consider using CodeBuild's SecretsManager environment variables feature instead. " +
                            "If you'd like to continue with having this secret in the plaintext environment variables, " +
                            'please set the checkSecretsInPlainTextEnvVariables property to false');
                    }
                }
            }
            if (principal) {
                const stack = core_1.Stack.of(principal);
                // save the SSM env variables
                if (envVariable.type === BuildEnvironmentVariableType.PARAMETER_STORE) {
                    ssmIamResources.push(stack.formatArn({
                        service: 'ssm',
                        resource: 'parameter',
                        // If the parameter name starts with / the resource name is not separated with a double '/'
                        // arn:aws:ssm:region:1111111111:parameter/PARAM_NAME
                        resourceName: envVariableValue.startsWith('/')
                            ? envVariableValue.slice(1)
                            : envVariableValue,
                    }));
                }
                // save SecretsManager env variables
                if (envVariable.type === BuildEnvironmentVariableType.SECRETS_MANAGER) {
                    // We have 3 basic cases here of what envVariableValue can be:
                    // 1. A string that starts with 'arn:' (and might contain Token fragments).
                    // 2. A Token.
                    // 3. A simple value, like 'secret-id'.
                    if (envVariableValue.startsWith('arn:')) {
                        const parsedArn = stack.splitArn(envVariableValue, core_1.ArnFormat.COLON_RESOURCE_NAME);
                        if (!parsedArn.resourceName) {
                            throw new Error('SecretManager ARN is missing the name of the secret: ' + envVariableValue);
                        }
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = parsedArn.resourceName.split(':')[0];
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            // since we don't know whether the ARN was full, or partial
                            // (CodeBuild supports both),
                            // stick a "*" at the end, which makes it work for both
                            resourceName: `${secretName}*`,
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                            partition: parsedArn.partition,
                            account: parsedArn.account,
                            region: parsedArn.region,
                        }));
                        // if secret comes from another account, SecretsManager will need to access
                        // KMS on the other account as well to be able to get the secret
                        if (parsedArn.account && core_1.Token.compareStrings(parsedArn.account, stack.account) === core_1.TokenComparison.DIFFERENT) {
                            kmsIamResources.add(stack.formatArn({
                                service: 'kms',
                                resource: 'key',
                                // We do not know the ID of the key, but since this is a cross-account access,
                                // the key policies have to allow this access, so a wildcard is safe here
                                resourceName: '*',
                                arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                                partition: parsedArn.partition,
                                account: parsedArn.account,
                                region: parsedArn.region,
                            }));
                        }
                    }
                    else if (core_1.Token.isUnresolved(envVariableValue)) {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        let secretArn = envVariableValue.split(':')[0];
                        // parse the Token, and see if it represents a single resource
                        // (we will assume it's a Secret from SecretsManager)
                        const fragments = core_1.Tokenization.reverseString(envVariableValue);
                        if (fragments.tokens.length === 1) {
                            const resolvable = fragments.tokens[0];
                            if (core_1.Reference.isReference(resolvable)) {
                                // check the Stack the resource owning the reference belongs to
                                const resourceStack = core_1.Stack.of(resolvable.target);
                                if (core_1.Token.compareStrings(stack.account, resourceStack.account) === core_1.TokenComparison.DIFFERENT) {
                                    // since this is a cross-account access,
                                    // add the appropriate KMS permissions
                                    kmsIamResources.add(stack.formatArn({
                                        service: 'kms',
                                        resource: 'key',
                                        // We do not know the ID of the key, but since this is a cross-account access,
                                        // the key policies have to allow this access, so a wildcard is safe here
                                        resourceName: '*',
                                        arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                                        partition: resourceStack.partition,
                                        account: resourceStack.account,
                                        region: resourceStack.region,
                                    }));
                                    // Work around a bug in SecretsManager -
                                    // when the access is cross-environment,
                                    // Secret.secretArn returns a partial ARN!
                                    // So add a "*" at the end, so that the permissions work
                                    secretArn = `${secretArn}-??????`;
                                }
                            }
                        }
                        // if we are passed a Token, we should assume it's the ARN of the Secret
                        // (as the name would not work anyway, because it would be the full name, which CodeBuild does not support)
                        secretsManagerIamResources.add(secretArn);
                    }
                    else {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = envVariableValue.split(':')[0];
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            resourceName: `${secretName}-??????`,
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                        }));
                    }
                }
            }
        }
        if (ssmIamResources.length !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['ssm:GetParameters'],
                resources: ssmIamResources,
            }));
        }
        if (secretsManagerIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['secretsmanager:GetSecretValue'],
                resources: Array.from(secretsManagerIamResources),
            }));
        }
        if (kmsIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['kms:Decrypt'],
                resources: Array.from(kmsIamResources),
            }));
        }
        return ret;
    }
    enableBatchBuilds() {
        if (!this._batchServiceRole) {
            this._batchServiceRole = new iam.Role(this, 'BatchServiceRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
            });
            this._batchServiceRole.addToPrincipalPolicy(new iam.PolicyStatement({
                resources: [core_1.Lazy.string({
                        produce: () => this.projectArn,
                    })],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                    'codebuild:RetryBuild',
                ],
            }));
        }
        return {
            role: this._batchServiceRole,
        };
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_ISource(secondarySource);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecondarySource);
            }
            throw error;
        }
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add
     */
    addFileSystemLocation(fileSystemLocation) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_IFileSystemLocation(fileSystemLocation);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFileSystemLocation);
            }
            throw error;
        }
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_IArtifacts(secondaryArtifact);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSecondaryArtifact);
            }
            throw error;
        }
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in
     * @param options additional options for the binding
     */
    bindToCodePipeline(_scope, options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_BindToCodePipelineOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bindToCodePipeline);
            }
            throw error;
        }
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * @override
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(props, projectVars = {}) {
        var _d, _e, _f, _g, _h;
        const env = (_d = props.environment) !== null && _d !== void 0 ? _d : {};
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? ImagePullPrincipalType.CODEBUILD
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_e = this.buildImage.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.grantRead(this);
        }
        const secret = this.buildImage.secretsManagerCredentials;
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: secret
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    // Secrets must be referenced by either the full ARN (with SecretsManager suffix), or by name.
                    // "Partial" ARNs (without the suffix) will fail a validation regex at deploy-time.
                    credential: (_f = secret.secretFullArn) !== null && _f !== void 0 ? _f : secret.secretName,
                }
                : undefined,
            certificate: (_g = env.certificate) === null || _g === void 0 ? void 0 : _g.bucket.arnForObjects(env.certificate.objectKey),
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars
                ? Project.serializeEnvVariables(vars, (_h = props.checkSecretsInPlainTextEnvVariables) !== null && _h !== void 0 ? _h : true, this)
                : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    renderLoggingConfiguration(props) {
        var _d, _e, _f, _g, _h;
        if (props === undefined) {
            return undefined;
        }
        let s3Config = undefined;
        let cloudwatchConfig = undefined;
        if (props.s3) {
            const s3Logs = props.s3;
            s3Config = {
                status: ((_d = s3Logs.enabled) !== null && _d !== void 0 ? _d : true) ? 'ENABLED' : 'DISABLED',
                location: `${s3Logs.bucket.bucketName}` + (s3Logs.prefix ? `/${s3Logs.prefix}` : ''),
                encryptionDisabled: s3Logs.encrypted,
            };
            (_e = s3Logs.bucket) === null || _e === void 0 ? void 0 : _e.grantWrite(this);
        }
        if (props.cloudWatch) {
            const cloudWatchLogs = props.cloudWatch;
            const status = ((_f = cloudWatchLogs.enabled) !== null && _f !== void 0 ? _f : true) ? 'ENABLED' : 'DISABLED';
            if (status === 'ENABLED' && !(cloudWatchLogs.logGroup)) {
                throw new Error('Specifying a LogGroup is required if CloudWatch logging for CodeBuild is enabled');
            }
            (_g = cloudWatchLogs.logGroup) === null || _g === void 0 ? void 0 : _g.grantWrite(this);
            cloudwatchConfig = {
                status,
                groupName: (_h = cloudWatchLogs.logGroup) === null || _h === void 0 ? void 0 : _h.logGroupName,
                streamName: cloudWatchLogs.prefix,
            };
        }
        return {
            s3Logs: s3Config,
            cloudWatchLogs: cloudwatchConfig,
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        // If the same Role is used for multiple Projects, always creating a new `iam.Policy`
        // will attach the same policy multiple times, probably exceeding the maximum size of the
        // Role policy. Make sure we only do it once for the same role.
        //
        // This deduplication could be a feature of the Role itself, but that feels risky and
        // is hard to implement (what with Tokens and all). Safer to fix it locally for now.
        let policy = this.role[VPC_POLICY_SYM];
        if (!policy) {
            policy = new iam.Policy(this, 'PolicyDocument', {
                statements: [
                    new iam.PolicyStatement({
                        resources: ['*'],
                        actions: [
                            'ec2:CreateNetworkInterface',
                            'ec2:DescribeNetworkInterfaces',
                            'ec2:DeleteNetworkInterface',
                            'ec2:DescribeSubnets',
                            'ec2:DescribeSecurityGroups',
                            'ec2:DescribeDhcpOptions',
                            'ec2:DescribeVpcs',
                        ],
                    }),
                ],
            });
            this.role.attachInlinePolicy(policy);
            this.role[VPC_POLICY_SYM] = policy;
        }
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "@aws-cdk/aws-codebuild.Project", version: "1.153.1" };
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    /**
     * CODEBUILD specifies that CodeBuild uses its own identity when pulling the image.
     * This means the resource policy of the ECR repository that hosts the image will be modified to trust
     * CodeBuild's service principal.
     * This is the required principal type when using CodeBuild's pre-defined images.
     */
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    /**
     * SERVICE_ROLE specifies that AWS CodeBuild uses the project's role when pulling the image.
     * The role will be granted pull permissions on the ECR repository hosting the image.
     */
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
// Keep around to resolve a circular dependency until removing deprecated ARM image constants from LinuxBuildImage
// eslint-disable-next-line no-duplicate-imports, import/order
const linux_arm_build_image_1 = require("./linux-arm-build-image");
/**
 * A CodeBuild image running x86-64 Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(props) {
        this.type = 'LINUX_CONTAINER';
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a x86-64 Linux build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_DockerImageOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDockerRegistry);
            }
            throw error;
        }
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @returns A x86-64 Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tagOrDigest Image tag or digest (default "latest", digests must start with `sha256:`)
     */
    static fromEcrRepository(repository, tagOrDigest = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTagOrDigest(tagOrDigest),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a x86-64 Linux build image.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @returns A Docker image provided by CodeBuild.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     *
     * @param id The image identifier
     * @example 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    validate(_) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_BuildEnvironment(_);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.validate);
            }
            throw error;
        }
        return [];
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
_b = JSII_RTTI_SYMBOL_1;
LinuxBuildImage[_b] = { fqn: "@aws-cdk/aws-codebuild.LinuxBuildImage", version: "1.153.1" };
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/** The `aws/codebuild/standard:4.0` build image. */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
/** The `aws/codebuild/standard:5.0` build image. */
LinuxBuildImage.STANDARD_5_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:5.0');
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/** The Amazon Linux 2 x86_64 standard image, version `3.0`. */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
/** @deprecated Use LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_1_0 instead. */
LinuxBuildImage.AMAZON_LINUX_2_ARM = linux_arm_build_image_1.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_1_0;
/**
 * Image "aws/codebuild/amazonlinux2-aarch64-standard:2.0".
 * @deprecated Use LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0 instead.
 * */
LinuxBuildImage.AMAZON_LINUX_2_ARM_2 = linux_arm_build_image_1.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * Environment type for Windows Docker images
 */
var WindowsImageType;
(function (WindowsImageType) {
    /**
     * The standard environment type, WINDOWS_CONTAINER
     */
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    /**
     * The WINDOWS_SERVER_2019_CONTAINER environment type
     */
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(props) {
        var _d;
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_d = props.imageType) !== null && _d !== void 0 ? _d : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_DockerImageOptions(options);
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_WindowsImageType(imageType);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromDockerRegistry);
            }
            throw error;
        }
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @returns A Windows build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tagOrDigest Image tag or digest (default "latest", digests must start with `sha256:`)
     */
    static fromEcrRepository(repository, tagOrDigest = 'latest', imageType = WindowsImageType.STANDARD) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_WindowsImageType(imageType);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromEcrRepository);
            }
            throw error;
        }
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTagOrDigest(tagOrDigest),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_WindowsImageType(imageType);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAsset);
            }
            throw error;
        }
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    validate(buildEnvironment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codebuild_BuildEnvironment(buildEnvironment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.validate);
            }
            throw error;
        }
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsBuildImage[_c] = { fqn: "@aws-cdk/aws-codebuild.WindowsBuildImage", version: "1.153.1" };
/**
 * Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is
 * based off Windows Server Core 2016.
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is
 * based off Windows Server Core 2019.
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    /**
     * An environment variable in plaintext format.
     */
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    /**
     * An environment variable stored in AWS Secrets Manager.
     */
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
/**
 * The list of event types for AWS Codebuild
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-buildproject
 */
var ProjectNotificationEvents;
(function (ProjectNotificationEvents) {
    /**
     * Trigger notification when project build state failed
     */
    ProjectNotificationEvents["BUILD_FAILED"] = "codebuild-project-build-state-failed";
    /**
     * Trigger notification when project build state succeeded
     */
    ProjectNotificationEvents["BUILD_SUCCEEDED"] = "codebuild-project-build-state-succeeded";
    /**
     * Trigger notification when project build state in progress
     */
    ProjectNotificationEvents["BUILD_IN_PROGRESS"] = "codebuild-project-build-state-in-progress";
    /**
     * Trigger notification when project build state stopped
     */
    ProjectNotificationEvents["BUILD_STOPPED"] = "codebuild-project-build-state-stopped";
    /**
     * Trigger notification when project build phase failure
     */
    ProjectNotificationEvents["BUILD_PHASE_FAILED"] = "codebuild-project-build-phase-failure";
    /**
     * Trigger notification when project build phase success
     */
    ProjectNotificationEvents["BUILD_PHASE_SUCCEEDED"] = "codebuild-project-build-phase-success";
})(ProjectNotificationEvents = exports.ProjectNotificationEvents || (exports.ProjectNotificationEvents = {}));
function isBindableBuildImage(x) {
    return typeof x === 'object' && !!x && !!x.bind;
}
//# sourceMappingURL=data:application/json;base64,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