#!/usr/bin/env python3
from biscot import Alignment
from biscot import Blat
from biscot import Key
from biscot import Map
from biscot import Misc

import argparse
import logging
import os
import sys


def run():
    Misc.print("")

    parser = argparse.ArgumentParser(
        prog="py",
        description="Reorganizes scaffolds generated by the Bionano scaffolding tool to remove small gaps that are unnecessary",
        formatter_class=argparse.RawTextHelpFormatter,
        add_help=False,
    )

    mandatory_args = parser.add_argument_group("Mandatory arguments")
    mandatory_args.add_argument(
        "--cmap-ref",
        action="store",
        dest="cmap_ref_path",
        help="Path to the reference CMAP file (example : '...NGS_contigs_HYBRID_Export_r.cmap')",
        default="",
        required=True,
    )
    mandatory_args.add_argument(
        "--cmap-1",
        action="store",
        dest="cmap_1_path",
        help="Path to a query CMAP file (example : 'E_CTTAAG_Q_NGScontigs_A_HYBRID_q.cmap')",
        default="",
        required=True,
    )
    mandatory_args.add_argument(
        "--xmap-1",
        action="store",
        dest="xmap_1_path",
        help="Path to a XMAP file",
        default="",
        required=True,
    )
    mandatory_args.add_argument(
        "--key",
        action="store",
        dest="key_path",
        help="Path to the Bionano 'key' file giving the correspondance between maps and contigs",
        default="",
        required=True,
    )
    mandatory_args.add_argument(
        "--contigs",
        action="store",
        dest="contigs_path",
        help="Path to the contigs file that was scaffolded by the Bionano Access",
        default="",
        required=True,
    )

    secondary_map_args = parser.add_argument_group("Secondary map argument")
    secondary_map_args.add_argument(
        "--cmap-2",
        action="store",
        dest="cmap_2_path",
        help="Path to a query cmap (example : 'E_GCTCTTC_Q_NGScontigs_A_HYBRID_q.cmap')",
        default=None,
        required=False,
    )
    secondary_map_args.add_argument(
        "--xmap-2",
        action="store",
        dest="xmap_2_path",
        help="Path to a query xmap (example : 'E_GCTCTTC_Q_NGScontigs_A_HYBRID.xmap')",
        default=None,
        required=False,
    )

    two_enzymes_xmap_args = parser.add_argument_group("Two enzymes xmap args")
    two_enzymes_xmap_args.add_argument(
        "--xmap-2enz",
        action="store",
        dest="xmap_two_enzymes",
        help="Path to a xmap file containing mappings for both enzymes",
        default=None,
        required=False,
    )

    optional_args = parser.add_argument_group("Optional arguments")
    optional_args.add_argument(
        "--aggressive",
        action="store_true",
        dest="aggressive_mode",
        help="Enables Blat phase",
        default=False,
    )
    optional_args.add_argument(
        "--only-confirmed-pos",
        action="store_true",
        dest="only_confirmed_positions",
        help="To be retained, an alignment position in --xmap-1 or --xmap-2 has to be present in --xmap-2enz",
        default=False,
    )
    optional_args.add_argument(
        "--output",
        "-o",
        action="store",
        dest="output_directory",
        help="Name of the output directory (default : %(default)s)",
        default="biscot",
        required=False,
    )
    optional_args.add_argument(
        "--debug",
        action="store_true",
        dest="debug",
        help="Activates debug level logs (default : %(default)s)",
        default=False,
        required=False,
    )
    optional_args.add_argument(
        "--help", "-h", action="help", help="show this help message and exit"
    )

    args = parser.parse_args()

    # User input sanity check
    Misc.check_path(
        [
            args.cmap_ref_path,
            args.cmap_1_path,
            args.xmap_1_path,
            args.key_path,
            args.contigs_path,
        ]
    )
    if args.cmap_2_path:
        Misc.check_path([args.cmap_2_path, args.xmap_2_path])
        args.cmap_2_path = os.path.abspath(args.cmap_2_path)
        args.xmap_2_path = os.path.abspath(args.xmap_2_path)
    if args.xmap_two_enzymes:
        Misc.check_path([args.xmap_two_enzymes])
        args.xmap_two_enzymes = os.path.abspath(args.xmap_two_enzymes)

    args.cmap_ref_path = os.path.abspath(args.cmap_ref_path)
    args.cmap_1_path = os.path.abspath(args.cmap_1_path)
    args.xmap_1_path = os.path.abspath(args.xmap_1_path)
    args.key_path = os.path.abspath(args.key_path)
    args.contigs_path = os.path.abspath(args.contigs_path)

    if args.only_confirmed_positions and not args.xmap_two_enzymes:
        print(
            f"Error: to use the --only-confirmed-pos, you have to indicate --xmap-2enz"
        )
        exit(-1)

    try:
        os.mkdir(args.output_directory)
    except:
        if args.debug:
            print(f"Directory {args.output_directory} already created")
    os.chdir(args.output_directory)

    with open("log", "w"):
        pass
    Misc.setup_logging(args.debug)

    logging.info(f"Command used : {' '.join(sys.argv)}")

    logging.info("Parsing key file")
    key_dict = Key.parse_key(args.key_path)

    logging.info("Parsing reference cmap file")
    reference_maps_dict = Map.parse_reference_cmap(args.cmap_ref_path)

    logging.info("Parsing contigs cmap file(s)")
    contigs_map_dict = Map.parse_contig_cmap(args.cmap_1_path, args.cmap_2_path)

    logging.info("Parsing xmap(s)")
    deleted_xmap_records = {}
    Alignment.parse_xmap(
        reference_maps_dict,
        args.xmap_1_path,
        args.xmap_2_path,
        deleted_xmap_records,
        args.xmap_two_enzymes,
        args.only_confirmed_positions,
    )

    extended_key_dict = Key.extend_key_dict(key_dict, reference_maps_dict)

    logging.info("Trying to integrate contained maps")
    Alignment.solve_alignment_containment(
        reference_maps_dict, contigs_map_dict, extended_key_dict
    )

    try:
        os.mkdir("Phase_1")
    except:
        logging.debug("Directory Phase_1 already created")

    scaffolded_maps = Alignment.print_agp(
        reference_maps_dict, extended_key_dict, deleted_xmap_records, contigs_map_dict,
    )

    contigs_sequence_dict = {}
    Misc.load_contigs(contigs_sequence_dict, args.contigs_path)
    Alignment.write_unplaced_contigs(key_dict, contigs_sequence_dict, scaffolded_maps)

    Misc.agp_to_fasta(
        contigs_sequence_dict, "Phase_1/phase_1.agp", "Phase_1/phase_1.fasta"
    )

    if args.aggressive_mode:
        logging.info("Starting Blat phase")
        Blat.blat_phase(contigs_sequence_dict)
        Misc.agp_to_fasta(contigs_sequence_dict, "scaffolds.agp", "scaffolds.fasta")
    else:
        os.system("cp Phase_1/phase_1.agp scaffolds.agp")
        os.system("cp Phase_1/phase_1.fasta scaffolds.fasta")

    logging.info(
        f"Final AGP and scaffolds file are in {args.output_directory}/scaffolds.agp and {args.output_directory}/scaffolds.fasta"
    )
