'''
# Tasks for AWS Step Functions

[AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/welcome.html) is a web service that enables you to coordinate the
components of distributed applications and microservices using visual workflows.
You build applications from individual components that each perform a discrete
function, or task, allowing you to scale and change applications quickly.

A [Task](https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-task-state.html) state represents a single unit of work performed by a state machine.
All work in your state machine is performed by tasks.

This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.

## Table Of Contents

* [Tasks for AWS Step Functions](#tasks-for-aws-step-functions)

  * [Table Of Contents](#table-of-contents)
  * [Task](#task)
  * [Paths](#paths)

    * [InputPath](#inputpath)
    * [OutputPath](#outputpath)
    * [ResultPath](#resultpath)
  * [Task parameters from the state JSON](#task-parameters-from-the-state-json)
  * [Evaluate Expression](#evaluate-expression)
  * [API Gateway](#api-gateway)

    * [Call REST API Endpoint](#call-rest-api-endpoint)
    * [Call HTTP API Endpoint](#call-http-api-endpoint)
  * [AWS SDK](#aws-sdk)
  * [Athena](#athena)

    * [StartQueryExecution](#startqueryexecution)
    * [GetQueryExecution](#getqueryexecution)
    * [GetQueryResults](#getqueryresults)
    * [StopQueryExecution](#stopqueryexecution)
  * [Batch](#batch)

    * [SubmitJob](#submitjob)
  * [CodeBuild](#codebuild)

    * [StartBuild](#startbuild)
  * [DynamoDB](#dynamodb)

    * [GetItem](#getitem)
    * [PutItem](#putitem)
    * [DeleteItem](#deleteitem)
    * [UpdateItem](#updateitem)
  * [ECS](#ecs)

    * [RunTask](#runtask)

      * [EC2](#ec2)
      * [Fargate](#fargate)
  * [EMR](#emr)

    * [Create Cluster](#create-cluster)
    * [Termination Protection](#termination-protection)
    * [Terminate Cluster](#terminate-cluster)
    * [Add Step](#add-step)
    * [Cancel Step](#cancel-step)
    * [Modify Instance Fleet](#modify-instance-fleet)
    * [Modify Instance Group](#modify-instance-group)
  * [EMR on EKS](#emr-on-eks)

    * [Create Virtual Cluster](#create-virtual-cluster)
    * [Delete Virtual Cluster](#delete-virtual-cluster)
    * [Start Job Run](#start-job-run)
  * [EKS](#eks)

    * [Call](#call)
  * [EventBridge](#eventbridge)

    * [Put Events](#put-events)
  * [Glue](#glue)
  * [Glue DataBrew](#glue-databrew)
  * [Lambda](#lambda)
  * [SageMaker](#sagemaker)

    * [Create Training Job](#create-training-job)
    * [Create Transform Job](#create-transform-job)
    * [Create Endpoint](#create-endpoint)
    * [Create Endpoint Config](#create-endpoint-config)
    * [Create Model](#create-model)
    * [Update Endpoint](#update-endpoint)
  * [SNS](#sns)
  * [Step Functions](#step-functions)

    * [Start Execution](#start-execution)
    * [Invoke Activity](#invoke-activity)
  * [SQS](#sqs)

## Task

A Task state represents a single unit of work performed by a state machine. In the
CDK, the exact work to be done is determined by a class that implements `IStepFunctionsTask`.

AWS Step Functions [integrates](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-service-integrations.html) with some AWS services so that you can call API
actions, and coordinate executions directly from the Amazon States Language in
Step Functions. You can directly call and pass parameters to the APIs of those
services.

## Paths

In the Amazon States Language, a [path](https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-paths.html) is a string beginning with `$` that you
can use to identify components within JSON text.

Learn more about input and output processing in Step Functions [here](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-input-output-filtering.html)

### InputPath

Both `InputPath` and `Parameters` fields provide a way to manipulate JSON as it
moves through your workflow. AWS Step Functions applies the `InputPath` field first,
and then the `Parameters` field. You can first filter your raw input to a selection
you want using InputPath, and then apply Parameters to manipulate that input
further, or add new values. If you don't specify an `InputPath`, a default value
of `$` will be used.

The following example provides the field named `input` as the input to the `Task`
state that runs a Lambda function.

```python
# fn: lambda.Function

submit_job = tasks.LambdaInvoke(self, "Invoke Handler",
    lambda_function=fn,
    input_path="$.input"
)
```

### OutputPath

Tasks also allow you to select a portion of the state output to pass to the next
state. This enables you to filter out unwanted information, and pass only the
portion of the JSON that you care about. If you don't specify an `OutputPath`,
a default value of `$` will be used. This passes the entire JSON node to the next
state.

The [response](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseSyntax) from a Lambda function includes the response from the function
as well as other metadata.

The following example assigns the output from the Task to a field named `result`

```python
# fn: lambda.Function

submit_job = tasks.LambdaInvoke(self, "Invoke Handler",
    lambda_function=fn,
    output_path="$.Payload.result"
)
```

### ResultSelector

You can use [`ResultSelector`](https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector)
to manipulate the raw result of a Task, Map or Parallel state before it is
passed to [`ResultPath`](###ResultPath). For service integrations, the raw
result contains metadata in addition to the response payload. You can use
ResultSelector to construct a JSON payload that becomes the effective result
using static values or references to the raw result or context object.

The following example extracts the output payload of a Lambda function Task and combines
it with some static values and the state name from the context object.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke Handler",
    lambda_function=fn,
    result_selector={
        "lambda_output": sfn.JsonPath.string_at("$.Payload"),
        "invoke_request_id": sfn.JsonPath.string_at("$.SdkResponseMetadata.RequestId"),
        "static_value": {
            "foo": "bar"
        },
        "state_name": sfn.JsonPath.string_at("$.State.Name")
    }
)
```

### ResultPath

The output of a state can be a copy of its input, the result it produces (for
example, output from a Task state’s Lambda function), or a combination of its
input and result. Use [`ResultPath`](https://docs.aws.amazon.com/step-functions/latest/dg/input-output-resultpath.html) to control which combination of these is
passed to the state output. If you don't specify an `ResultPath`, a default
value of `$` will be used.

The following example adds the item from calling DynamoDB's `getItem` API to the state
input and passes it to the next state.

```python
# my_table: dynamodb.Table

tasks.DynamoPutItem(self, "PutItem",
    item={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-id")
    },
    table=my_table,
    result_path="$.Item"
)
```

⚠️ The `OutputPath` is computed after applying `ResultPath`. All service integrations
return metadata as part of their response. When using `ResultPath`, it's not possible to
merge a subset of the task output to the input.

## Task parameters from the state JSON

Most tasks take parameters. Parameter values can either be static, supplied directly
in the workflow definition (by specifying their values), or a value available at runtime
in the state machine's execution (either as its input or an output of a prior state).
Parameter values available at runtime can be specified via the `JsonPath` class,
using methods such as `JsonPath.stringAt()`.

The following example provides the field named `input` as the input to the Lambda function
and invokes it asynchronously.

```python
# fn: lambda.Function


submit_job = tasks.LambdaInvoke(self, "Invoke Handler",
    lambda_function=fn,
    payload=sfn.TaskInput.from_json_path_at("$.input"),
    invocation_type=tasks.LambdaInvocationType.EVENT
)
```

You can also use [intrinsic functions](https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-intrinsic-functions.html) available on `JsonPath`, for example `JsonPath.format()`.
Here is an example of starting an Athena query that is dynamically created using the task input:

```python
start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Athena Start Query",
    query_string=sfn.JsonPath.format("select contacts where year={};", sfn.JsonPath.string_at("$.year")),
    query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
        database_name="interactions"
    ),
    result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
        encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
            encryption_option=tasks.EncryptionOption.S3_MANAGED
        ),
        output_location=dynamodb.aws_s3.Location(
            bucket_name="mybucket",
            object_key="myprefix"
        )
    ),
    integration_pattern=sfn.IntegrationPattern.RUN_JOB
)
```

Each service integration has its own set of parameters that can be supplied.

## Evaluate Expression

Use the `EvaluateExpression` to perform simple operations referencing state paths. The
`expression` referenced in the task will be evaluated in a Lambda function
(`eval()`). This allows you to not have to write Lambda code for simple operations.

Example: convert a wait time from milliseconds to seconds, concat this in a message and wait:

```python
convert_to_seconds = tasks.EvaluateExpression(self, "Convert to seconds",
    expression="$.waitMilliseconds / 1000",
    result_path="$.waitSeconds"
)

create_message = tasks.EvaluateExpression(self, "Create message",
    # Note: this is a string inside a string.
    expression="`Now waiting ${$.waitSeconds} seconds...`",
    runtime=lambda_.Runtime.NODEJS_14_X,
    result_path="$.message"
)

publish_message = tasks.SnsPublish(self, "Publish message",
    topic=sns.Topic(self, "cool-topic"),
    message=sfn.TaskInput.from_json_path_at("$.message"),
    result_path="$.sns"
)

wait = sfn.Wait(self, "Wait",
    time=sfn.WaitTime.seconds_path("$.waitSeconds")
)

sfn.StateMachine(self, "StateMachine",
    definition=convert_to_seconds.next(create_message).next(publish_message).next(wait)
)
```

The `EvaluateExpression` supports a `runtime` prop to specify the Lambda
runtime to use to evaluate the expression. Currently, only runtimes
of the Node.js family are supported.

## API Gateway

Step Functions supports [API Gateway](https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html) through the service integration pattern.

HTTP APIs are designed for low-latency, cost-effective integrations with AWS services, including AWS Lambda, and HTTP endpoints.
HTTP APIs support OIDC and OAuth 2.0 authorization, and come with built-in support for CORS and automatic deployments.
Previous-generation REST APIs currently offer more features. More details can be found [here](https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-vs-rest.html).

### Call REST API Endpoint

The `CallApiGatewayRestApiEndpoint` calls the REST API endpoint.

```python
import monocdk as apigateway

rest_api = apigateway.RestApi(self, "MyRestApi")

invoke_task = tasks.CallApiGatewayRestApiEndpoint(self, "Call REST API",
    api=rest_api,
    stage_name="prod",
    method=tasks.HttpMethod.GET
)
```

Be aware that the header values must be arrays. When passing the Task Token
in the headers field `WAIT_FOR_TASK_TOKEN` integration, use
`JsonPath.array()` to wrap the token in an array:

```python
import monocdk as apigateway
# api: apigateway.RestApi


tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
    api=api,
    stage_name="Stage",
    method=tasks.HttpMethod.PUT,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    headers=sfn.TaskInput.from_object({
        "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
    })
)
```

### Call HTTP API Endpoint

The `CallApiGatewayHttpApiEndpoint` calls the HTTP API endpoint.

```python
import monocdk as apigatewayv2

http_api = apigatewayv2.HttpApi(self, "MyHttpApi")

invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
    api_id=http_api.api_id,
    api_stack=Stack.of(http_api),
    method=tasks.HttpMethod.GET
)
```

### AWS SDK

Step Functions supports calling [AWS service's API actions](https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html)
through the service integration pattern.

You can use Step Functions' AWS SDK integrations to call any of the over two hundred AWS services
directly from your state machine, giving you access to over nine thousand API actions.

```python
# my_bucket: s3.Bucket

get_object = tasks.CallAwsService(self, "GetObject",
    service="s3",
    action="getObject",
    parameters={
        "Bucket": my_bucket.bucket_name,
        "Key": sfn.JsonPath.string_at("$.key")
    },
    iam_resources=[my_bucket.arn_for_objects("*")]
)
```

Use camelCase for actions and PascalCase for parameter names.

The task automatically adds an IAM statement to the state machine role's policy based on the
service and action called. The resources for this statement must be specified in `iamResources`.

Use the `iamAction` prop to manually specify the IAM action name in the case where the IAM
action name does not match with the API service/action name:

```python
list_buckets = tasks.CallAwsService(self, "ListBuckets",
    service="s3",
    action="listBuckets",
    iam_resources=["*"],
    iam_action="s3:ListAllMyBuckets"
)
```

## Athena

Step Functions supports [Athena](https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html) through the service integration pattern.

### StartQueryExecution

The [StartQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_StartQueryExecution.html) API runs the SQL query statement.

```python
start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Start Athena Query",
    query_string=sfn.JsonPath.string_at("$.queryString"),
    query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
        database_name="mydatabase"
    ),
    result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
        encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
            encryption_option=tasks.EncryptionOption.S3_MANAGED
        ),
        output_location=dynamodb.aws_s3.Location(
            bucket_name="query-results-bucket",
            object_key="folder"
        )
    )
)
```

### GetQueryExecution

The [GetQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryExecution.html) API gets information about a single execution of a query.

```python
get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

### GetQueryResults

The [GetQueryResults](https://docs.aws.amazon.com/athena/latest/APIReference/API_GetQueryResults.html) API that streams the results of a single query execution specified by QueryExecutionId from S3.

```python
get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

### StopQueryExecution

The [StopQueryExecution](https://docs.aws.amazon.com/athena/latest/APIReference/API_StopQueryExecution.html) API that stops a query execution.

```python
stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
    query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
)
```

## Batch

Step Functions supports [Batch](https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html) through the service integration pattern.

### SubmitJob

The [SubmitJob](https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html) API submits an AWS Batch job from a job definition.

```python
import monocdk as batch
# batch_job_definition: batch.JobDefinition
# batch_queue: batch.JobQueue


task = tasks.BatchSubmitJob(self, "Submit Job",
    job_definition_arn=batch_job_definition.job_definition_arn,
    job_name="MyJob",
    job_queue_arn=batch_queue.job_queue_arn
)
```

## CodeBuild

Step Functions supports [CodeBuild](https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html) through the service integration pattern.

### StartBuild

[StartBuild](https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html) starts a CodeBuild Project by Project Name.

```python
import monocdk as codebuild


codebuild_project = codebuild.Project(self, "Project",
    project_name="MyTestProject",
    build_spec=codebuild.BuildSpec.from_object({
        "version": "0.2",
        "phases": {
            "build": {
                "commands": ["echo \"Hello, CodeBuild!\""
                ]
            }
        }
    })
)

task = tasks.CodeBuildStartBuild(self, "Task",
    project=codebuild_project,
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    environment_variables_override={
        "ZONE": codebuild.aws_codebuild.BuildEnvironmentVariable(
            type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
            value=sfn.JsonPath.string_at("$.envVariables.zone")
        )
    }
)
```

## DynamoDB

You can call DynamoDB APIs from a `Task` state.
Read more about calling DynamoDB APIs [here](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ddb.html)

### GetItem

The [GetItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html) operation returns a set of attributes for the item with the given primary key.

```python
# my_table: dynamodb.Table

tasks.DynamoGetItem(self, "Get Item",
    key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
    table=my_table
)
```

### PutItem

The [PutItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html) operation creates a new item, or replaces an old item with a new item.

```python
# my_table: dynamodb.Table

tasks.DynamoPutItem(self, "PutItem",
    item={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-007"),
        "Text": tasks.DynamoAttributeValue.from_string(sfn.JsonPath.string_at("$.bar")),
        "TotalCount": tasks.DynamoAttributeValue.from_number(10)
    },
    table=my_table
)
```

### DeleteItem

The [DeleteItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html) operation deletes a single item in a table by primary key.

```python
# my_table: dynamodb.Table

tasks.DynamoDeleteItem(self, "DeleteItem",
    key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
    table=my_table,
    result_path=sfn.JsonPath.DISCARD
)
```

### UpdateItem

The [UpdateItem](https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html) operation edits an existing item's attributes, or adds a new item
to the table if it does not already exist.

```python
# my_table: dynamodb.Table

tasks.DynamoUpdateItem(self, "UpdateItem",
    key={
        "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
    },
    table=my_table,
    expression_attribute_values={
        ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
        ":rand": tasks.DynamoAttributeValue.from_number(20)
    },
    update_expression="SET TotalCount = :val + :rand"
)
```

## ECS

Step Functions supports [ECS/Fargate](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html) through the service integration pattern.

### RunTask

[RunTask](https://docs.aws.amazon.com/step-functions/latest/dg/connect-ecs.html) starts a new task using the specified task definition.

#### EC2

The EC2 launch type allows you to run your containerized applications on a cluster
of Amazon EC2 instances that you manage.

When a task that uses the EC2 launch type is launched, Amazon ECS must determine where
to place the task based on the requirements specified in the task definition, such as
CPU and memory. Similarly, when you scale down the task count, Amazon ECS must determine
which tasks to terminate. You can apply task placement strategies and constraints to
customize how Amazon ECS places and terminates tasks. Learn more about [task placement](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement.html)

The latest ACTIVE revision of the passed task definition is used for running the task.

The following example runs a job from a task definition on EC2

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)

cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
cluster.add_capacity("DefaultAutoScalingGroup",
    instance_type=ec2.InstanceType("t2.micro"),
    vpc_subnets=dynamodb.aws_ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)

task_definition = ecs.TaskDefinition(self, "TD",
    compatibility=ecs.Compatibility.EC2
)

task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "Run",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    launch_target=tasks.EcsEc2LaunchTarget(
        placement_strategies=[
            ecs.PlacementStrategy.spread_across_instances(),
            ecs.PlacementStrategy.packed_by_cpu(),
            ecs.PlacementStrategy.randomly()
        ],
        placement_constraints=[
            ecs.PlacementConstraint.member_of("blieptuut")
        ]
    )
)
```

#### Fargate

AWS Fargate is a serverless compute engine for containers that works with Amazon
Elastic Container Service (ECS). Fargate makes it easy for you to focus on building
your applications. Fargate removes the need to provision and manage servers, lets you
specify and pay for resources per application, and improves security through application
isolation by design. Learn more about [Fargate](https://aws.amazon.com/fargate/)

The Fargate launch type allows you to run your containerized applications without the need
to provision and manage the backend infrastructure. Just register your task definition and
Fargate launches the container for you. The latest ACTIVE revision of the passed
task definition is used for running the task. Learn more about
[Fargate Versioning](https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTaskDefinition.html)

The following example runs a job from a task definition on Fargate

```python
vpc = ec2.Vpc.from_lookup(self, "Vpc",
    is_default=True
)

cluster = ecs.Cluster(self, "FargateCluster", vpc=vpc)

task_definition = ecs.TaskDefinition(self, "TD",
    memory_mi_b="512",
    cpu="256",
    compatibility=ecs.Compatibility.FARGATE
)

container_definition = task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("foo/bar"),
    memory_limit_mi_b=256
)

run_task = tasks.EcsRunTask(self, "RunFargate",
    integration_pattern=sfn.IntegrationPattern.RUN_JOB,
    cluster=cluster,
    task_definition=task_definition,
    assign_public_ip=True,
    container_overrides=[dynamodb.aws_stepfunctions_tasks.ContainerOverride(
        container_definition=container_definition,
        environment=[dynamodb.aws_stepfunctions_tasks.TaskEnvironmentVariable(name="SOME_KEY", value=sfn.JsonPath.string_at("$.SomeKey"))]
    )],
    launch_target=tasks.EcsFargateLaunchTarget()
)
```

## EMR

Step Functions supports Amazon EMR through the service integration pattern.
The service integration APIs correspond to Amazon EMR APIs but differ in the
parameters that are used.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html) about the differences when using these service integrations.

### Create Cluster

Creates and starts running a cluster (job flow).
Corresponds to the [`runJobFlow`](https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html) API in EMR.

```python
cluster_role = iam.Role(self, "ClusterRole",
    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
)

service_role = iam.Role(self, "ServiceRole",
    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
)

auto_scaling_role = iam.Role(self, "AutoScalingRole",
    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
)

auto_scaling_role.assume_role_policy.add_statements(
    iam.PolicyStatement(
        effect=iam.Effect.ALLOW,
        principals=[
            iam.ServicePrincipal("application-autoscaling.amazonaws.com")
        ],
        actions=["sts:AssumeRole"
        ]
    ))

tasks.EmrCreateCluster(self, "Create Cluster",
    instances=dynamodb.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty(),
    cluster_role=cluster_role,
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    service_role=service_role,
    auto_scaling_role=auto_scaling_role
)
```

If you want to run multiple steps in [parallel](https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-concurrent-steps.html),
you can specify the `stepConcurrencyLevel` property. The concurrency range is between 1
and 256 inclusive, where the default concurrency of 1 means no step concurrency is allowed.
`stepConcurrencyLevel` requires the EMR release label to be 5.28.0 or above.

```python
tasks.EmrCreateCluster(self, "Create Cluster",
    instances=dynamodb.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty(),
    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
    step_concurrency_level=10
)
```

### Termination Protection

Locks a cluster (job flow) so the EC2 instances in the cluster cannot be
terminated by user intervention, an API call, or a job-flow error.

Corresponds to the [`setTerminationProtection`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr.html) API in EMR.

```python
tasks.EmrSetClusterTerminationProtection(self, "Task",
    cluster_id="ClusterId",
    termination_protected=False
)
```

### Terminate Cluster

Shuts down a cluster (job flow).
Corresponds to the [`terminateJobFlows`](https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html) API in EMR.

```python
tasks.EmrTerminateCluster(self, "Task",
    cluster_id="ClusterId"
)
```

### Add Step

Adds a new step to a running cluster.
Corresponds to the [`addJobFlowSteps`](https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html) API in EMR.

```python
tasks.EmrAddStep(self, "Task",
    cluster_id="ClusterId",
    name="StepName",
    jar="Jar",
    action_on_failure=tasks.ActionOnFailure.CONTINUE
)
```

### Cancel Step

Cancels a pending step in a running cluster.
Corresponds to the [`cancelSteps`](https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html) API in EMR.

```python
tasks.EmrCancelStep(self, "Task",
    cluster_id="ClusterId",
    step_id="StepId"
)
```

### Modify Instance Fleet

Modifies the target On-Demand and target Spot capacities for the instance
fleet with the specified InstanceFleetName.

Corresponds to the [`modifyInstanceFleet`](https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html) API in EMR.

```python
tasks.EmrModifyInstanceFleetByName(self, "Task",
    cluster_id="ClusterId",
    instance_fleet_name="InstanceFleetName",
    target_on_demand_capacity=2,
    target_spot_capacity=0
)
```

### Modify Instance Group

Modifies the number of nodes and configuration settings of an instance group.

Corresponds to the [`modifyInstanceGroups`](https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html) API in EMR.

```python
tasks.EmrModifyInstanceGroupByName(self, "Task",
    cluster_id="ClusterId",
    instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
    instance_group=dynamodb.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
        instance_count=1
    )
)
```

## EMR on EKS

Step Functions supports Amazon EMR on EKS through the service integration pattern.
The service integration APIs correspond to Amazon EMR on EKS APIs, but differ in the parameters that are used.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html) about the differences when using these service integrations.

[Setting up](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html) the EKS cluster is required.

### Create Virtual Cluster

The [CreateVirtualCluster](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_CreateVirtualCluster.html) API creates a single virtual cluster that's mapped to a single Kubernetes namespace.

The EKS cluster containing the Kubernetes namespace where the virtual cluster will be mapped can be passed in from the task input.

```python
tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId"))
)
```

The EKS cluster can also be passed in directly.

```python
import monocdk as eks

# eks_cluster: eks.Cluster


tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_cluster(eks_cluster)
)
```

By default, the Kubernetes namespace that a virtual cluster maps to is "default", but a specific namespace within an EKS cluster can be selected.

```python
tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
    eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
    eks_namespace="specified-namespace"
)
```

### Delete Virtual Cluster

The [DeleteVirtualCluster](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_DeleteVirtualCluster.html) API deletes a virtual cluster.

```python
tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
    virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
)
```

### Start Job Run

The [StartJobRun](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_StartJobRun.html) API starts a job run. A job is a unit of work that you submit to Amazon EMR on EKS for execution. The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query. A job run is an execution of the job on the virtual cluster.

Required setup:

* If not done already, follow the [steps](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up.html) to setup EMR on EKS and [create an EKS Cluster](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-eks-readme.html#quick-start).
* Enable [Cluster access](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-cluster-access.html)
* Enable [IAM Role access](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-enable-IAM.html)

The following actions must be performed if the virtual cluster ID is supplied from the task input. Otherwise, if it is supplied statically in the state machine definition, these actions will be done automatically.

* Create an [IAM role](https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-iam.Role.html)
* Update the [Role Trust Policy](https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-trust-policy.html) of the Job Execution Role.

The job can be configured with spark submit parameters:

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
        spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    )
)
```

Configuring the job can also be done via application configuration:

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_name="EMR-Containers-Job",
    job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
        spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
        )
    ),
    application_config=[dynamodb.aws_stepfunctions_tasks.ApplicationConfiguration(
        classification=tasks.Classification.SPARK_DEFAULTS,
        properties={
            "spark.executor.instances": "1",
            "spark.executor.memory": "512M"
        }
    )]
)
```

Job monitoring can be enabled if `monitoring.logging` is set true. This automatically generates an S3 bucket and CloudWatch logs.

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
        spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    ),
    monitoring=dynamodb.aws_stepfunctions_tasks.Monitoring(
        logging=True
    )
)
```

Otherwise, providing monitoring for jobs with existing log groups and log buckets is also available.

```python
import monocdk as logs


log_group = logs.LogGroup(self, "Log Group")
log_bucket = s3.Bucket(self, "S3 Bucket")

tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_driver=logs.aws_stepfunctions_tasks.JobDriver(
        spark_submit_job_driver=logs.aws_stepfunctions_tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    ),
    monitoring=logs.aws_stepfunctions_tasks.Monitoring(
        log_group=log_group,
        log_bucket=log_bucket
    )
)
```

Users can provide their own existing Job Execution Role.

```python
tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
    virtual_cluster=tasks.VirtualClusterInput.from_task_input(sfn.TaskInput.from_json_path_at("$.VirtualClusterId")),
    release_label=tasks.ReleaseLabel.EMR_6_2_0,
    job_name="EMR-Containers-Job",
    execution_role=iam.Role.from_role_arn(self, "Job-Execution-Role", "arn:aws:iam::xxxxxxxxxxxx:role/JobExecutionRole"),
    job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
        spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
            entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
            spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
        )
    )
)
```

## EKS

Step Functions supports Amazon EKS through the service integration pattern.
The service integration APIs correspond to Amazon EKS APIs.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html) about the differences when using these service integrations.

### Call

Read and write Kubernetes resource objects via a Kubernetes API endpoint.
Corresponds to the [`call`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html) API in Step Functions Connector.

The following code snippet includes a Task state that uses eks:call to list the pods.

```python
import monocdk as eks


my_eks_cluster = eks.Cluster(self, "my sample cluster",
    version=eks.KubernetesVersion.V1_18,
    cluster_name="myEksCluster"
)

tasks.EksCall(self, "Call a EKS Endpoint",
    cluster=my_eks_cluster,
    http_method=tasks.HttpMethods.GET,
    http_path="/api/v1/namespaces/default/pods"
)
```

## EventBridge

Step Functions supports Amazon EventBridge through the service integration pattern.
The service integration APIs correspond to Amazon EventBridge APIs.

[Read more](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html) about the differences when using these service integrations.

### Put Events

Send events to an EventBridge bus.
Corresponds to the [`put-events`](https://docs.aws.amazon.com/step-functions/latest/dg/connect-eventbridge.html) API in Step Functions Connector.

The following code snippet includes a Task state that uses events:putevents to send an event to the default bus.

```python
import monocdk as events


my_event_bus = events.EventBus(self, "EventBus",
    event_bus_name="MyEventBus1"
)

tasks.EventBridgePutEvents(self, "Send an event to EventBridge",
    entries=[events.aws_stepfunctions_tasks.EventBridgePutEventsEntry(
        detail=sfn.TaskInput.from_object({
            "Message": "Hello from Step Functions!"
        }),
        event_bus=my_event_bus,
        detail_type="MessageFromStepFunctions",
        source="step.functions"
    )]
)
```

## Glue

Step Functions supports [AWS Glue](https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html) through the service integration pattern.

You can call the [`StartJobRun`](https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun) API from a `Task` state.

```python
tasks.GlueStartJobRun(self, "Task",
    glue_job_name="my-glue-job",
    arguments=sfn.TaskInput.from_object({
        "key": "value"
    }),
    timeout=Duration.minutes(30),
    notify_delay_after=Duration.minutes(5)
)
```

## Glue DataBrew

Step Functions supports [AWS Glue DataBrew](https://docs.aws.amazon.com/step-functions/latest/dg/connect-databrew.html) through the service integration pattern.

You can call the [`StartJobRun`](https://docs.aws.amazon.com/databrew/latest/dg/API_StartJobRun.html) API from a `Task` state.

```python
tasks.GlueDataBrewStartJobRun(self, "Task",
    name="databrew-job"
)
```

## Lambda

[Invoke](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html) a Lambda function.

You can specify the input to your Lambda function through the `payload` attribute.
By default, Step Functions invokes Lambda function with the state input (JSON path '$')
as the input.

The following snippet invokes a Lambda Function with the state input as the payload
by referencing the `$` path.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with state input",
    lambda_function=fn
)
```

When a function is invoked, the Lambda service sends  [these response
elements](https://docs.aws.amazon.com/lambda/latest/dg/API_Invoke.html#API_Invoke_ResponseElements)
back.

⚠️ The response from the Lambda function is in an attribute called `Payload`

The following snippet invokes a Lambda Function by referencing the `$.Payload` path
to reference the output of a Lambda executed before it.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with empty object as payload",
    lambda_function=fn,
    payload=sfn.TaskInput.from_object({})
)

# use the output of fn as input
tasks.LambdaInvoke(self, "Invoke with payload field in the state input",
    lambda_function=fn,
    payload=sfn.TaskInput.from_json_path_at("$.Payload")
)
```

The following snippet invokes a Lambda and sets the task output to only include
the Lambda function response.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke and set function response as task output",
    lambda_function=fn,
    output_path="$.Payload"
)
```

If you want to combine the input and the Lambda function response you can use
the `payloadResponseOnly` property and specify the `resultPath`. This will put the
Lambda function ARN directly in the "Resource" string, but it conflicts with the
integrationPattern, invocationType, clientContext, and qualifier properties.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke and combine function response with task input",
    lambda_function=fn,
    payload_response_only=True,
    result_path="$.fn"
)
```

You can have Step Functions pause a task, and wait for an external process to
return a task token. Read more about the [callback pattern](https://docs.aws.amazon.com/step-functions/latest/dg/callback-task-sample-sqs.html#call-back-lambda-example)

To use the callback pattern, set the `token` property on the task. Call the Step
Functions `SendTaskSuccess` or `SendTaskFailure` APIs with the token to
indicate that the task has completed and the state machine should resume execution.

The following snippet invokes a Lambda with the task token as part of the input
to the Lambda.

```python
# fn: lambda.Function

tasks.LambdaInvoke(self, "Invoke with callback",
    lambda_function=fn,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    payload=sfn.TaskInput.from_object({
        "token": sfn.JsonPath.task_token,
        "input": sfn.JsonPath.string_at("$.someField")
    })
)
```

⚠️ The task will pause until it receives that task token back with a `SendTaskSuccess` or `SendTaskFailure`
call. Learn more about [Callback with the Task
Token](https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token).

AWS Lambda can occasionally experience transient service errors. In this case, invoking Lambda
results in a 500 error, such as `ServiceException`, `AWSLambdaException`, or `SdkClientException`.
As a best practice, the `LambdaInvoke` task will retry on those errors with an interval of 2 seconds,
a back-off rate of 2 and 6 maximum attempts. Set the `retryOnServiceExceptions` prop to `false` to
disable this behavior.

## SageMaker

Step Functions supports [AWS SageMaker](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html) through the service integration pattern.

If your training job or model uses resources from AWS Marketplace,
[network isolation is required](https://docs.aws.amazon.com/sagemaker/latest/dg/mkt-algo-model-internet-free.html).
To do so, set the `enableNetworkIsolation` property to `true` for `SageMakerCreateModel` or `SageMakerCreateTrainingJob`.

To set environment variables for the Docker container use the `environment` property.

### Create Training Job

You can call the [`CreateTrainingJob`](https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTrainingJob.html) API from a `Task` state.

```python
tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
    training_job_name=sfn.JsonPath.string_at("$.JobName"),
    algorithm_specification=dynamodb.aws_stepfunctions_tasks.AlgorithmSpecification(
        algorithm_name="BlazingText",
        training_input_mode=tasks.InputMode.FILE
    ),
    input_data_config=[dynamodb.aws_stepfunctions_tasks.Channel(
        channel_name="train",
        data_source=dynamodb.aws_stepfunctions_tasks.DataSource(
            s3_data_source=dynamodb.aws_stepfunctions_tasks.S3DataSource(
                s3_data_type=tasks.S3DataType.S3_PREFIX,
                s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
            )
        )
    )],
    output_data_config=dynamodb.aws_stepfunctions_tasks.OutputDataConfig(
        s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
    ),
    resource_config=dynamodb.aws_stepfunctions_tasks.ResourceConfig(
        instance_count=1,
        instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
        volume_size=Size.gibibytes(50)
    ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
    stopping_condition=dynamodb.aws_stepfunctions_tasks.StoppingCondition(
        max_runtime=Duration.hours(2)
    )
)
```

### Create Transform Job

You can call the [`CreateTransformJob`](https://docs.aws.amazon.com/sagemaker/latest/dg/API_CreateTransformJob.html) API from a `Task` state.

```python
tasks.SageMakerCreateTransformJob(self, "Batch Inference",
    transform_job_name="MyTransformJob",
    model_name="MyModelName",
    model_client_options=dynamodb.aws_stepfunctions_tasks.ModelClientOptions(
        invocations_max_retries=3,  # default is 0
        invocations_timeout=Duration.minutes(5)
    ),
    transform_input=dynamodb.aws_stepfunctions_tasks.TransformInput(
        transform_data_source=dynamodb.aws_stepfunctions_tasks.TransformDataSource(
            s3_data_source=dynamodb.aws_stepfunctions_tasks.TransformS3DataSource(
                s3_uri="s3://inputbucket/train",
                s3_data_type=tasks.S3DataType.S3_PREFIX
            )
        )
    ),
    transform_output=dynamodb.aws_stepfunctions_tasks.TransformOutput(
        s3_output_path="s3://outputbucket/TransformJobOutputPath"
    ),
    transform_resources=dynamodb.aws_stepfunctions_tasks.TransformResources(
        instance_count=1,
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE)
    )
)
```

### Create Endpoint

You can call the [`CreateEndpoint`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpoint.html) API from a `Task` state.

```python
tasks.SageMakerCreateEndpoint(self, "SagemakerEndpoint",
    endpoint_name=sfn.JsonPath.string_at("$.EndpointName"),
    endpoint_config_name=sfn.JsonPath.string_at("$.EndpointConfigName")
)
```

### Create Endpoint Config

You can call the [`CreateEndpointConfig`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateEndpointConfig.html) API from a `Task` state.

```python
tasks.SageMakerCreateEndpointConfig(self, "SagemakerEndpointConfig",
    endpoint_config_name="MyEndpointConfig",
    production_variants=[dynamodb.aws_stepfunctions_tasks.ProductionVariant(
        initial_instance_count=2,
        instance_type=ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
        model_name="MyModel",
        variant_name="awesome-variant"
    )]
)
```

### Create Model

You can call the [`CreateModel`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_CreateModel.html) API from a `Task` state.

```python
tasks.SageMakerCreateModel(self, "Sagemaker",
    model_name="MyModel",
    primary_container=tasks.ContainerDefinition(
        image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
        mode=tasks.Mode.SINGLE_MODEL,
        model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
    )
)
```

### Update Endpoint

You can call the [`UpdateEndpoint`](https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_UpdateEndpoint.html) API from a `Task` state.

```python
tasks.SageMakerUpdateEndpoint(self, "SagemakerEndpoint",
    endpoint_name=sfn.JsonPath.string_at("$.Endpoint.Name"),
    endpoint_config_name=sfn.JsonPath.string_at("$.Endpoint.EndpointConfig")
)
```

## SNS

Step Functions supports [Amazon SNS](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sns.html) through the service integration pattern.

You can call the [`Publish`](https://docs.aws.amazon.com/sns/latest/api/API_Publish.html) API from a `Task` state to publish to an SNS topic.

```python
topic = sns.Topic(self, "Topic")

# Use a field from the execution data as message.
task1 = tasks.SnsPublish(self, "Publish1",
    topic=topic,
    integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
    message=sfn.TaskInput.from_data_at("$.state.message"),
    message_attributes={
        "place": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
            value=sfn.JsonPath.string_at("$.place")
        ),
        "pic": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
            # BINARY must be explicitly set
            data_type=tasks.MessageAttributeDataType.BINARY,
            value=sfn.JsonPath.string_at("$.pic")
        ),
        "people": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
            value=4
        ),
        "handles": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
            value=["@kslater", "@jjf", null, "@mfanning"]
        )
    }
)

# Combine a field from the execution data with
# a literal object.
task2 = tasks.SnsPublish(self, "Publish2",
    topic=topic,
    message=sfn.TaskInput.from_object({
        "field1": "somedata",
        "field2": sfn.JsonPath.string_at("$.field2")
    })
)
```

## Step Functions

### Start Execution

You can manage [AWS Step Functions](https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html) executions.

AWS Step Functions supports it's own [`StartExecution`](https://docs.aws.amazon.com/step-functions/latest/apireference/API_StartExecution.html) API as a service integration.

```python
# Define a state machine with one Pass state
child = sfn.StateMachine(self, "ChildStateMachine",
    definition=sfn.Chain.start(sfn.Pass(self, "PassState"))
)

# Include the state machine in a Task state with callback pattern
task = tasks.StepFunctionsStartExecution(self, "ChildTask",
    state_machine=child,
    integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
    input=sfn.TaskInput.from_object({
        "token": sfn.JsonPath.task_token,
        "foo": "bar"
    }),
    name="MyExecutionName"
)

# Define a second state machine with the Task state above
sfn.StateMachine(self, "ParentStateMachine",
    definition=task
)
```

You can utilize [Associate Workflow Executions](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-nested-workflows.html#nested-execution-startid)
via the `associateWithParent` property. This allows the Step Functions UI to link child
executions from parent executions, making it easier to trace execution flow across state machines.

```python
# child: sfn.StateMachine

task = tasks.StepFunctionsStartExecution(self, "ChildTask",
    state_machine=child,
    associate_with_parent=True
)
```

This will add the payload `AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID.$: $$.Execution.Id` to the
`input`property for you, which will pass the execution ID from the context object to the
execution input. It requires `input` to be an object or not be set at all.

### Invoke Activity

You can invoke a [Step Functions Activity](https://docs.aws.amazon.com/step-functions/latest/dg/concepts-activities.html) which enables you to have
a task in your state machine where the work is performed by a *worker* that can
be hosted on Amazon EC2, Amazon ECS, AWS Lambda, basically anywhere. Activities
are a way to associate code running somewhere (known as an activity worker) with
a specific task in a state machine.

When Step Functions reaches an activity task state, the workflow waits for an
activity worker to poll for a task. An activity worker polls Step Functions by
using GetActivityTask, and sending the ARN for the related activity.

After the activity worker completes its work, it can provide a report of its
success or failure by using `SendTaskSuccess` or `SendTaskFailure`. These two
calls use the taskToken provided by GetActivityTask to associate the result
with that task.

The following example creates an activity and creates a task that invokes the activity.

```python
submit_job_activity = sfn.Activity(self, "SubmitJob")

tasks.StepFunctionsInvokeActivity(self, "Submit Job",
    activity=submit_job_activity
)
```

## SQS

Step Functions supports [Amazon SQS](https://docs.aws.amazon.com/step-functions/latest/dg/connect-sqs.html)

You can call the [`SendMessage`](https://docs.aws.amazon.com/AWSSimpleQueueService/latest/APIReference/API_SendMessage.html) API from a `Task` state
to send a message to an SQS queue.

```python
queue = sqs.Queue(self, "Queue")

# Use a field from the execution data as message.
task1 = tasks.SqsSendMessage(self, "Send1",
    queue=queue,
    message_body=sfn.TaskInput.from_json_path_at("$.message")
)

# Combine a field from the execution data with
# a literal object.
task2 = tasks.SqsSendMessage(self, "Send2",
    queue=queue,
    message_body=sfn.TaskInput.from_object({
        "field1": "somedata",
        "field2": sfn.JsonPath.string_at("$.field2")
    })
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from .._jsii import *

import constructs
from .. import (
    Duration as _Duration_070aa057,
    IgnoreMode as _IgnoreMode_31d8bf46,
    Size as _Size_7fbd4337,
    Stack as _Stack_9f43e4a3,
    SymlinkFollowMode as _SymlinkFollowMode_abf4527a,
)
from ..assets import FollowMode as _FollowMode_98b05cc5
from ..aws_apigateway import IRestApi as _IRestApi_a27d4788
from ..aws_codebuild import (
    BuildEnvironmentVariable as _BuildEnvironmentVariable_00095c97,
    IProject as _IProject_6da8803e,
)
from ..aws_dynamodb import ITable as _ITable_24826f7e
from ..aws_ec2 import (
    Connections as _Connections_57ccbda9,
    IConnectable as _IConnectable_c1c0e72c,
    ISecurityGroup as _ISecurityGroup_cdbba9d3,
    IVpc as _IVpc_6d1f76c4,
    InstanceSize as _InstanceSize_decfb0b1,
    InstanceType as _InstanceType_072ad323,
    SubnetSelection as _SubnetSelection_1284e62c,
)
from ..aws_ecr import IRepository as _IRepository_8b4d2894
from ..aws_ecr_assets import (
    DockerImageAssetInvalidationOptions as _DockerImageAssetInvalidationOptions_e5375707,
    DockerImageAssetProps as _DockerImageAssetProps_beaa8517,
    NetworkMode as _NetworkMode_2d19607e,
)
from ..aws_ecs import (
    ContainerDefinition as _ContainerDefinition_7934d1e1,
    FargatePlatformVersion as _FargatePlatformVersion_8169c79a,
    ICluster as _ICluster_42c4ec1a,
    ITaskDefinition as _ITaskDefinition_ee0d1862,
    PlacementConstraint as _PlacementConstraint_e22ac48c,
    PlacementStrategy as _PlacementStrategy_ea27367e,
    TaskDefinition as _TaskDefinition_c0dacfb4,
)
from ..aws_eks import ICluster as _ICluster_e7e676cf
from ..aws_events import IEventBus as _IEventBus_2ca38c95
from ..aws_iam import (
    IGrantable as _IGrantable_4c5a91d1,
    IPrincipal as _IPrincipal_93b48231,
    IRole as _IRole_59af6f50,
    PolicyStatement as _PolicyStatement_296fe8a3,
)
from ..aws_kms import IKey as _IKey_36930160
from ..aws_lambda import IFunction as _IFunction_6e14f09e, Runtime as _Runtime_932d369a
from ..aws_logs import ILogGroup as _ILogGroup_846e17a0
from ..aws_s3 import IBucket as _IBucket_73486e29, Location as _Location_cce991ca
from ..aws_sns import ITopic as _ITopic_465e36b9
from ..aws_sqs import IQueue as _IQueue_45a01ab4
from ..aws_stepfunctions import (
    IActivity as _IActivity_4524952f,
    IStateMachine as _IStateMachine_269a89c4,
    IStepFunctionsTask as _IStepFunctionsTask_82eb09ab,
    IntegrationPattern as _IntegrationPattern_fbb35786,
    ServiceIntegrationPattern as _ServiceIntegrationPattern_5581ee88,
    StepFunctionsTaskConfig as _StepFunctionsTaskConfig_fa131821,
    Task as _Task_747ca4f2,
    TaskInput as _TaskInput_71ab46f6,
    TaskMetricsConfig as _TaskMetricsConfig_5aaea59e,
    TaskStateBase as _TaskStateBase_c3080646,
    TaskStateBaseProps as _TaskStateBaseProps_b13f6332,
)


class AcceleratorClass(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.AcceleratorClass",
):
    '''(experimental) The generation of Elastic Inference (EI) instance.

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        accelerator_class = stepfunctions_tasks.AcceleratorClass.of("version")
    '''

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, version: builtins.str) -> "AcceleratorClass":
        '''(experimental) Custom AcceleratorType.

        :param version: - Elastic Inference accelerator generation.

        :stability: experimental
        '''
        return typing.cast("AcceleratorClass", jsii.sinvoke(cls, "of", [version]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="EIA1")
    def EIA1(cls) -> "AcceleratorClass":
        '''(experimental) Elastic Inference accelerator 1st generation.

        :stability: experimental
        '''
        return typing.cast("AcceleratorClass", jsii.sget(cls, "EIA1"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="EIA2")
    def EIA2(cls) -> "AcceleratorClass":
        '''(experimental) Elastic Inference accelerator 2nd generation.

        :stability: experimental
        '''
        return typing.cast("AcceleratorClass", jsii.sget(cls, "EIA2"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="version")
    def version(self) -> builtins.str:
        '''(experimental) - Elastic Inference accelerator generation.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "version"))


class AcceleratorType(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.AcceleratorType",
):
    '''(experimental) The size of the Elastic Inference (EI) instance to use for the production variant.

    EI instances provide on-demand GPU computing for inference

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/ei.html
    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        accelerator_type = stepfunctions_tasks.AcceleratorType("instanceTypeIdentifier")
    '''

    def __init__(self, instance_type_identifier: builtins.str) -> None:
        '''
        :param instance_type_identifier: -

        :stability: experimental
        '''
        jsii.create(self.__class__, self, [instance_type_identifier])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(
        cls,
        accelerator_class: AcceleratorClass,
        instance_size: _InstanceSize_decfb0b1,
    ) -> "AcceleratorType":
        '''(experimental) AcceleratorType.

        This class takes a combination of a class and size.

        :param accelerator_class: -
        :param instance_size: -

        :stability: experimental
        '''
        return typing.cast("AcceleratorType", jsii.sinvoke(cls, "of", [accelerator_class, instance_size]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''(experimental) Return the accelerator type as a dotted string.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.ActionOnFailure")
class ActionOnFailure(enum.Enum):
    '''(experimental) The action to take when the cluster step fails.

    :default: CONTINUE

    :see:

    https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html

    Here, they are named as TERMINATE_JOB_FLOW, TERMINATE_CLUSTER, CANCEL_AND_WAIT, and CONTINUE respectively.
    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrAddStep(self, "Task",
            cluster_id="ClusterId",
            name="StepName",
            jar="Jar",
            action_on_failure=tasks.ActionOnFailure.CONTINUE
        )
    '''

    TERMINATE_CLUSTER = "TERMINATE_CLUSTER"
    '''(experimental) Terminate the Cluster on Step Failure.

    :stability: experimental
    '''
    CANCEL_AND_WAIT = "CANCEL_AND_WAIT"
    '''(experimental) Cancel Step execution and enter WAITING state.

    :stability: experimental
    '''
    CONTINUE = "CONTINUE"
    '''(experimental) Continue to the next Step.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.AlgorithmSpecification",
    jsii_struct_bases=[],
    name_mapping={
        "algorithm_name": "algorithmName",
        "metric_definitions": "metricDefinitions",
        "training_image": "trainingImage",
        "training_input_mode": "trainingInputMode",
    },
)
class AlgorithmSpecification:
    def __init__(
        self,
        *,
        algorithm_name: typing.Optional[builtins.str] = None,
        metric_definitions: typing.Optional[typing.Sequence["MetricDefinition"]] = None,
        training_image: typing.Optional["DockerImage"] = None,
        training_input_mode: typing.Optional["InputMode"] = None,
    ) -> None:
        '''(experimental) Specify the training algorithm and algorithm-specific metadata.

        :param algorithm_name: (experimental) Name of the algorithm resource to use for the training job. This must be an algorithm resource that you created or subscribe to on AWS Marketplace. If you specify a value for this parameter, you can't specify a value for TrainingImage. Default: - No algorithm is specified
        :param metric_definitions: (experimental) List of metric definition objects. Each object specifies the metric name and regular expressions used to parse algorithm logs. Default: - No metrics
        :param training_image: (experimental) Registry path of the Docker image that contains the training algorithm. Default: - No Docker image is specified
        :param training_input_mode: (experimental) Input mode that the algorithm supports. Default: 'File' mode

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
                training_job_name=sfn.JsonPath.string_at("$.JobName"),
                algorithm_specification=dynamodb.aws_stepfunctions_tasks.AlgorithmSpecification(
                    algorithm_name="BlazingText",
                    training_input_mode=tasks.InputMode.FILE
                ),
                input_data_config=[dynamodb.aws_stepfunctions_tasks.Channel(
                    channel_name="train",
                    data_source=dynamodb.aws_stepfunctions_tasks.DataSource(
                        s3_data_source=dynamodb.aws_stepfunctions_tasks.S3DataSource(
                            s3_data_type=tasks.S3DataType.S3_PREFIX,
                            s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                        )
                    )
                )],
                output_data_config=dynamodb.aws_stepfunctions_tasks.OutputDataConfig(
                    s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
                ),
                resource_config=dynamodb.aws_stepfunctions_tasks.ResourceConfig(
                    instance_count=1,
                    instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                    volume_size=Size.gibibytes(50)
                ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
                stopping_condition=dynamodb.aws_stepfunctions_tasks.StoppingCondition(
                    max_runtime=Duration.hours(2)
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if algorithm_name is not None:
            self._values["algorithm_name"] = algorithm_name
        if metric_definitions is not None:
            self._values["metric_definitions"] = metric_definitions
        if training_image is not None:
            self._values["training_image"] = training_image
        if training_input_mode is not None:
            self._values["training_input_mode"] = training_input_mode

    @builtins.property
    def algorithm_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of the algorithm resource to use for the training job.

        This must be an algorithm resource that you created or subscribe to on AWS Marketplace.
        If you specify a value for this parameter, you can't specify a value for TrainingImage.

        :default: - No algorithm is specified

        :stability: experimental
        '''
        result = self._values.get("algorithm_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def metric_definitions(self) -> typing.Optional[typing.List["MetricDefinition"]]:
        '''(experimental) List of metric definition objects.

        Each object specifies the metric name and regular expressions used to parse algorithm logs.

        :default: - No metrics

        :stability: experimental
        '''
        result = self._values.get("metric_definitions")
        return typing.cast(typing.Optional[typing.List["MetricDefinition"]], result)

    @builtins.property
    def training_image(self) -> typing.Optional["DockerImage"]:
        '''(experimental) Registry path of the Docker image that contains the training algorithm.

        :default: - No Docker image is specified

        :stability: experimental
        '''
        result = self._values.get("training_image")
        return typing.cast(typing.Optional["DockerImage"], result)

    @builtins.property
    def training_input_mode(self) -> typing.Optional["InputMode"]:
        '''(experimental) Input mode that the algorithm supports.

        :default: 'File' mode

        :stability: experimental
        '''
        result = self._values.get("training_input_mode")
        return typing.cast(typing.Optional["InputMode"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AlgorithmSpecification(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ApplicationConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "classification": "classification",
        "nested_config": "nestedConfig",
        "properties": "properties",
    },
)
class ApplicationConfiguration:
    def __init__(
        self,
        *,
        classification: "Classification",
        nested_config: typing.Optional[typing.Sequence["ApplicationConfiguration"]] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''(experimental) A configuration specification to be used when provisioning virtual clusters, which can include configurations for applications and software bundled with Amazon EMR on EKS.

        A configuration consists of a classification, properties, and optional nested configurations.
        A classification refers to an application-specific configuration file.
        Properties are the settings you want to change in that file.

        :param classification: (experimental) The classification within a configuration. Length Constraints: Minimum length of 1. Maximum length of 1024.
        :param nested_config: (experimental) A list of additional configurations to apply within a configuration object. Array Members: Maximum number of 100 items. Default: - No other configurations
        :param properties: (experimental) A set of properties specified within a configuration classification. Map Entries: Maximum number of 100 items. Default: - No properties

        :see: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-configure-apps.html
        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # application_configuration_: stepfunctions_tasks.ApplicationConfiguration
            # classification: stepfunctions_tasks.Classification
            
            application_configuration = stepfunctions_tasks.ApplicationConfiguration(
                classification=classification,
            
                # the properties below are optional
                nested_config=[stepfunctions_tasks.ApplicationConfiguration(
                    classification=classification,
            
                    # the properties below are optional
                    nested_config=[application_configuration_],
                    properties={
                        "properties_key": "properties"
                    }
                )],
                properties={
                    "properties_key": "properties"
                }
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "classification": classification,
        }
        if nested_config is not None:
            self._values["nested_config"] = nested_config
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def classification(self) -> "Classification":
        '''(experimental) The classification within a configuration.

        Length Constraints: Minimum length of 1. Maximum length of 1024.

        :stability: experimental
        '''
        result = self._values.get("classification")
        assert result is not None, "Required property 'classification' is missing"
        return typing.cast("Classification", result)

    @builtins.property
    def nested_config(self) -> typing.Optional[typing.List["ApplicationConfiguration"]]:
        '''(experimental) A list of additional configurations to apply within a configuration object.

        Array Members: Maximum number of 100 items.

        :default: - No other configurations

        :stability: experimental
        '''
        result = self._values.get("nested_config")
        return typing.cast(typing.Optional[typing.List["ApplicationConfiguration"]], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A set of properties specified within a configuration classification.

        Map Entries: Maximum number of 100 items.

        :default: - No properties

        :stability: experimental
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.AssembleWith")
class AssembleWith(enum.Enum):
    '''(experimental) How to assemble the results of the transform job as a single S3 object.

    :stability: experimental
    '''

    NONE = "NONE"
    '''(experimental) Concatenate the results in binary format.

    :stability: experimental
    '''
    LINE = "LINE"
    '''(experimental) Add a newline character at the end of every transformed record.

    :stability: experimental
    '''


class AthenaGetQueryExecution(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaGetQueryExecution",
):
    '''(experimental) Get an Athena Query Execution as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param query_execution_id: (experimental) Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = AthenaGetQueryExecutionProps(
            query_execution_id=query_execution_id,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaGetQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaGetQueryExecutionProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''(experimental) Properties for getting a Query Execution.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param query_execution_id: (experimental) Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``

        :stability: experimental
        :exampleMetadata: infused

        Example::

            get_query_execution_job = tasks.AthenaGetQueryExecution(self, "Get Query Execution",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''(experimental) Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``

        :stability: experimental
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaGetQueryResults(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaGetQueryResults",
):
    '''(experimental) Get an Athena Query Results as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param query_execution_id: (experimental) Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: (experimental) Max number of results. Default: 1000
        :param next_token: (experimental) Pagination token. Default: - No next token
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = AthenaGetQueryResultsProps(
            query_execution_id=query_execution_id,
            max_results=max_results,
            next_token=next_token,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaGetQueryResultsProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "query_execution_id": "queryExecutionId",
        "max_results": "maxResults",
        "next_token": "nextToken",
    },
)
class AthenaGetQueryResultsProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        query_execution_id: builtins.str,
        max_results: typing.Optional[jsii.Number] = None,
        next_token: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for getting a Query Results.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param query_execution_id: (experimental) Query that will be retrieved. Example value: ``adfsaf-23trf23-f23rt23``
        :param max_results: (experimental) Max number of results. Default: 1000
        :param next_token: (experimental) Pagination token. Default: - No next token

        :stability: experimental
        :exampleMetadata: infused

        Example::

            get_query_results_job = tasks.AthenaGetQueryResults(self, "Get Query Results",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if max_results is not None:
            self._values["max_results"] = max_results
        if next_token is not None:
            self._values["next_token"] = next_token

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''(experimental) Query that will be retrieved.

        Example value: ``adfsaf-23trf23-f23rt23``

        :stability: experimental
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def max_results(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Max number of results.

        :default: 1000

        :stability: experimental
        '''
        result = self._values.get("max_results")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def next_token(self) -> typing.Optional[builtins.str]:
        '''(experimental) Pagination token.

        :default: - No next token

        :stability: experimental
        '''
        result = self._values.get("next_token")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaGetQueryResultsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaStartQueryExecution(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaStartQueryExecution",
):
    '''(experimental) Start an Athena Query as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Athena Start Query",
            query_string=sfn.JsonPath.format("select contacts where year={};", sfn.JsonPath.string_at("$.year")),
            query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
                database_name="interactions"
            ),
            result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
                encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
                    encryption_option=tasks.EncryptionOption.S3_MANAGED
                ),
                output_location=dynamodb.aws_s3.Location(
                    bucket_name="mybucket",
                    object_key="myprefix"
                )
            ),
            integration_pattern=sfn.IntegrationPattern.RUN_JOB
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        query_execution_context: typing.Optional["QueryExecutionContext"] = None,
        result_configuration: typing.Optional["ResultConfiguration"] = None,
        work_group: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param query_string: (experimental) Query that will be started.
        :param client_request_token: (experimental) Unique string string to ensure idempotence. Default: - No client request token
        :param query_execution_context: (experimental) Database within which query executes. Default: - No query execution context
        :param result_configuration: (experimental) Configuration on how and where to save query. Default: - No result configuration
        :param work_group: (experimental) Configuration on how and where to save query. Default: - No work group
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = AthenaStartQueryExecutionProps(
            query_string=query_string,
            client_request_token=client_request_token,
            query_execution_context=query_execution_context,
            result_configuration=result_configuration,
            work_group=work_group,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaStartQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "query_string": "queryString",
        "client_request_token": "clientRequestToken",
        "query_execution_context": "queryExecutionContext",
        "result_configuration": "resultConfiguration",
        "work_group": "workGroup",
    },
)
class AthenaStartQueryExecutionProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        query_string: builtins.str,
        client_request_token: typing.Optional[builtins.str] = None,
        query_execution_context: typing.Optional["QueryExecutionContext"] = None,
        result_configuration: typing.Optional["ResultConfiguration"] = None,
        work_group: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for starting a Query Execution.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param query_string: (experimental) Query that will be started.
        :param client_request_token: (experimental) Unique string string to ensure idempotence. Default: - No client request token
        :param query_execution_context: (experimental) Database within which query executes. Default: - No query execution context
        :param result_configuration: (experimental) Configuration on how and where to save query. Default: - No result configuration
        :param work_group: (experimental) Configuration on how and where to save query. Default: - No work group

        :stability: experimental
        :exampleMetadata: infused

        Example::

            start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Athena Start Query",
                query_string=sfn.JsonPath.format("select contacts where year={};", sfn.JsonPath.string_at("$.year")),
                query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
                    database_name="interactions"
                ),
                result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
                    encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
                        encryption_option=tasks.EncryptionOption.S3_MANAGED
                    ),
                    output_location=dynamodb.aws_s3.Location(
                        bucket_name="mybucket",
                        object_key="myprefix"
                    )
                ),
                integration_pattern=sfn.IntegrationPattern.RUN_JOB
            )
        '''
        if isinstance(query_execution_context, dict):
            query_execution_context = QueryExecutionContext(**query_execution_context)
        if isinstance(result_configuration, dict):
            result_configuration = ResultConfiguration(**result_configuration)
        self._values: typing.Dict[str, typing.Any] = {
            "query_string": query_string,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if query_execution_context is not None:
            self._values["query_execution_context"] = query_execution_context
        if result_configuration is not None:
            self._values["result_configuration"] = result_configuration
        if work_group is not None:
            self._values["work_group"] = work_group

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def query_string(self) -> builtins.str:
        '''(experimental) Query that will be started.

        :stability: experimental
        '''
        result = self._values.get("query_string")
        assert result is not None, "Required property 'query_string' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[builtins.str]:
        '''(experimental) Unique string string to ensure idempotence.

        :default: - No client request token

        :stability: experimental
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def query_execution_context(self) -> typing.Optional["QueryExecutionContext"]:
        '''(experimental) Database within which query executes.

        :default: - No query execution context

        :stability: experimental
        '''
        result = self._values.get("query_execution_context")
        return typing.cast(typing.Optional["QueryExecutionContext"], result)

    @builtins.property
    def result_configuration(self) -> typing.Optional["ResultConfiguration"]:
        '''(experimental) Configuration on how and where to save query.

        :default: - No result configuration

        :stability: experimental
        '''
        result = self._values.get("result_configuration")
        return typing.cast(typing.Optional["ResultConfiguration"], result)

    @builtins.property
    def work_group(self) -> typing.Optional[builtins.str]:
        '''(experimental) Configuration on how and where to save query.

        :default: - No work group

        :stability: experimental
        '''
        result = self._values.get("work_group")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStartQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AthenaStopQueryExecution(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaStopQueryExecution",
):
    '''(experimental) Stop an Athena Query Execution as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-athena.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
            query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        query_execution_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param query_execution_id: (experimental) Query that will be stopped.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = AthenaStopQueryExecutionProps(
            query_execution_id=query_execution_id,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.AthenaStopQueryExecutionProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "query_execution_id": "queryExecutionId",
    },
)
class AthenaStopQueryExecutionProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        query_execution_id: builtins.str,
    ) -> None:
        '''(experimental) Properties for stoping a Query Execution.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param query_execution_id: (experimental) Query that will be stopped.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            stop_query_execution_job = tasks.AthenaStopQueryExecution(self, "Stop Query Execution",
                query_execution_id=sfn.JsonPath.string_at("$.QueryExecutionId")
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "query_execution_id": query_execution_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def query_execution_id(self) -> builtins.str:
        '''(experimental) Query that will be stopped.

        :stability: experimental
        '''
        result = self._values.get("query_execution_id")
        assert result is not None, "Required property 'query_execution_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AthenaStopQueryExecutionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.AuthType")
class AuthType(enum.Enum):
    '''(experimental) The authentication method used to call the endpoint.

    :stability: experimental
    '''

    NO_AUTH = "NO_AUTH"
    '''(experimental) Call the API direclty with no authorization method.

    :stability: experimental
    '''
    IAM_ROLE = "IAM_ROLE"
    '''(experimental) Use the IAM role associated with the current state machine for authorization.

    :stability: experimental
    '''
    RESOURCE_POLICY = "RESOURCE_POLICY"
    '''(experimental) Use the resource policy of the API for authorization.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.BatchContainerOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "environment": "environment",
        "gpu_count": "gpuCount",
        "instance_type": "instanceType",
        "memory": "memory",
        "vcpus": "vcpus",
    },
)
class BatchContainerOverrides:
    def __init__(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        instance_type: typing.Optional[_InstanceType_072ad323] = None,
        memory: typing.Optional[_Size_7fbd4337] = None,
        vcpus: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The overrides that should be sent to a container.

        :param command: (experimental) The command to send to the container that overrides the default command from the Docker image or the job definition. Default: - No command overrides
        :param environment: (experimental) The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the job definition. Default: - No environment overrides
        :param gpu_count: (experimental) The number of physical GPUs to reserve for the container. The number of GPUs reserved for all containers in a job should not exceed the number of available GPUs on the compute resource that the job is launched on. Default: - No GPU reservation
        :param instance_type: (experimental) The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container jobs. Default: - No instance type overrides
        :param memory: (experimental) Memory reserved for the job. Default: - No memory overrides. The memory supplied in the job definition will be used.
        :param vcpus: (experimental) The number of vCPUs to reserve for the container. This value overrides the value set in the job definition. Default: - No vCPUs overrides

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_ec2 as ec2
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # instance_type: ec2.InstanceType
            # size: monocdk.Size
            
            batch_container_overrides = stepfunctions_tasks.BatchContainerOverrides(
                command=["command"],
                environment={
                    "environment_key": "environment"
                },
                gpu_count=123,
                instance_type=instance_type,
                memory=size,
                vcpus=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if memory is not None:
            self._values["memory"] = memory
        if vcpus is not None:
            self._values["vcpus"] = vcpus

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command to send to the container that overrides the default command from the Docker image or the job definition.

        :default: - No command overrides

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to send to the container.

        You can add new environment variables, which are added to the container
        at launch, or you can override the existing environment variables from
        the Docker image or the job definition.

        :default: - No environment overrides

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of physical GPUs to reserve for the container.

        The number of GPUs reserved for all containers in a job
        should not exceed the number of available GPUs on the compute
        resource that the job is launched on.

        :default: - No GPU reservation

        :stability: experimental
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[_InstanceType_072ad323]:
        '''(experimental) The instance type to use for a multi-node parallel job.

        This parameter is not valid for single-node container jobs.

        :default: - No instance type overrides

        :stability: experimental
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[_InstanceType_072ad323], result)

    @builtins.property
    def memory(self) -> typing.Optional[_Size_7fbd4337]:
        '''(experimental) Memory reserved for the job.

        :default: - No memory overrides. The memory supplied in the job definition will be used.

        :stability: experimental
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[_Size_7fbd4337], result)

    @builtins.property
    def vcpus(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of vCPUs to reserve for the container.

        This value overrides the value set in the job definition.

        :default: - No vCPUs overrides

        :stability: experimental
        '''
        result = self._values.get("vcpus")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchContainerOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.BatchJobDependency",
    jsii_struct_bases=[],
    name_mapping={"job_id": "jobId", "type": "type"},
)
class BatchJobDependency:
    def __init__(
        self,
        *,
        job_id: typing.Optional[builtins.str] = None,
        type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) An object representing an AWS Batch job dependency.

        :param job_id: (experimental) The job ID of the AWS Batch job associated with this dependency. Default: - No jobId
        :param type: (experimental) The type of the job dependency. Default: - No type

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            batch_job_dependency = stepfunctions_tasks.BatchJobDependency(
                job_id="jobId",
                type="type"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if job_id is not None:
            self._values["job_id"] = job_id
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def job_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) The job ID of the AWS Batch job associated with this dependency.

        :default: - No jobId

        :stability: experimental
        '''
        result = self._values.get("job_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''(experimental) The type of the job dependency.

        :default: - No type

        :stability: experimental
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchJobDependency(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.BatchStrategy")
class BatchStrategy(enum.Enum):
    '''(experimental) Specifies the number of records to include in a mini-batch for an HTTP inference request.

    :stability: experimental
    '''

    MULTI_RECORD = "MULTI_RECORD"
    '''(experimental) Fits multiple records in a mini-batch.

    :stability: experimental
    '''
    SINGLE_RECORD = "SINGLE_RECORD"
    '''(experimental) Use a single record when making an invocation request.

    :stability: experimental
    '''


class BatchSubmitJob(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.BatchSubmitJob",
):
    '''(experimental) Task to submits an AWS Batch job from a job definition.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-batch.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as batch
        # batch_job_definition: batch.JobDefinition
        # batch_queue: batch.JobQueue
        
        
        task = tasks.BatchSubmitJob(self, "Submit Job",
            job_definition_arn=batch_job_definition.job_definition_arn,
            job_name="MyJob",
            job_queue_arn=batch_queue.job_queue_arn
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[BatchContainerOverrides] = None,
        depends_on: typing.Optional[typing.Sequence[BatchJobDependency]] = None,
        payload: typing.Optional[_TaskInput_71ab46f6] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param job_definition_arn: (experimental) The arn of the job definition used by this job.
        :param job_name: (experimental) The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: (experimental) The arn of the job queue into which the job is submitted.
        :param array_size: (experimental) The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: (experimental) The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: (experimental) A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: (experimental) A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: (experimental) The payload to be passed as parameters to the batch job. Default: - No parameters are passed
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = BatchSubmitJobProps(
            job_definition_arn=job_definition_arn,
            job_name=job_name,
            job_queue_arn=job_queue_arn,
            array_size=array_size,
            attempts=attempts,
            container_overrides=container_overrides,
            depends_on=depends_on,
            payload=payload,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.BatchSubmitJobProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "job_definition_arn": "jobDefinitionArn",
        "job_name": "jobName",
        "job_queue_arn": "jobQueueArn",
        "array_size": "arraySize",
        "attempts": "attempts",
        "container_overrides": "containerOverrides",
        "depends_on": "dependsOn",
        "payload": "payload",
    },
)
class BatchSubmitJobProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        job_definition_arn: builtins.str,
        job_name: builtins.str,
        job_queue_arn: builtins.str,
        array_size: typing.Optional[jsii.Number] = None,
        attempts: typing.Optional[jsii.Number] = None,
        container_overrides: typing.Optional[BatchContainerOverrides] = None,
        depends_on: typing.Optional[typing.Sequence[BatchJobDependency]] = None,
        payload: typing.Optional[_TaskInput_71ab46f6] = None,
    ) -> None:
        '''(experimental) Properties for RunBatchJob.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param job_definition_arn: (experimental) The arn of the job definition used by this job.
        :param job_name: (experimental) The name of the job. The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.
        :param job_queue_arn: (experimental) The arn of the job queue into which the job is submitted.
        :param array_size: (experimental) The array size can be between 2 and 10,000. If you specify array properties for a job, it becomes an array job. For more information, see Array Jobs in the AWS Batch User Guide. Default: - No array size
        :param attempts: (experimental) The number of times to move a job to the RUNNABLE status. You may specify between 1 and 10 attempts. If the value of attempts is greater than one, the job is retried on failure the same number of attempts as the value. Default: 1
        :param container_overrides: (experimental) A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive. Default: - No container overrides
        :param depends_on: (experimental) A list of dependencies for the job. A job can depend upon a maximum of 20 jobs. Default: - No dependencies
        :param payload: (experimental) The payload to be passed as parameters to the batch job. Default: - No parameters are passed

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as batch
            # batch_job_definition: batch.JobDefinition
            # batch_queue: batch.JobQueue
            
            
            task = tasks.BatchSubmitJob(self, "Submit Job",
                job_definition_arn=batch_job_definition.job_definition_arn,
                job_name="MyJob",
                job_queue_arn=batch_queue.job_queue_arn
            )
        '''
        if isinstance(container_overrides, dict):
            container_overrides = BatchContainerOverrides(**container_overrides)
        self._values: typing.Dict[str, typing.Any] = {
            "job_definition_arn": job_definition_arn,
            "job_name": job_name,
            "job_queue_arn": job_queue_arn,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if array_size is not None:
            self._values["array_size"] = array_size
        if attempts is not None:
            self._values["attempts"] = attempts
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if depends_on is not None:
            self._values["depends_on"] = depends_on
        if payload is not None:
            self._values["payload"] = payload

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def job_definition_arn(self) -> builtins.str:
        '''(experimental) The arn of the job definition used by this job.

        :stability: experimental
        '''
        result = self._values.get("job_definition_arn")
        assert result is not None, "Required property 'job_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_name(self) -> builtins.str:
        '''(experimental) The name of the job.

        The first character must be alphanumeric, and up to 128 letters (uppercase and lowercase),
        numbers, hyphens, and underscores are allowed.

        :stability: experimental
        '''
        result = self._values.get("job_name")
        assert result is not None, "Required property 'job_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def job_queue_arn(self) -> builtins.str:
        '''(experimental) The arn of the job queue into which the job is submitted.

        :stability: experimental
        '''
        result = self._values.get("job_queue_arn")
        assert result is not None, "Required property 'job_queue_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def array_size(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The array size can be between 2 and 10,000.

        If you specify array properties for a job, it becomes an array job.
        For more information, see Array Jobs in the AWS Batch User Guide.

        :default: - No array size

        :stability: experimental
        '''
        result = self._values.get("array_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def attempts(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of times to move a job to the RUNNABLE status.

        You may specify between 1 and 10 attempts.
        If the value of attempts is greater than one,
        the job is retried on failure the same number of attempts as the value.

        :default: 1

        :stability: experimental
        '''
        result = self._values.get("attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[BatchContainerOverrides]:
        '''(experimental) A list of container overrides in JSON format that specify the name of a container in the specified job definition and the overrides it should receive.

        :default: - No container overrides

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-containerOverrides
        :stability: experimental
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[BatchContainerOverrides], result)

    @builtins.property
    def depends_on(self) -> typing.Optional[typing.List[BatchJobDependency]]:
        '''(experimental) A list of dependencies for the job.

        A job can depend upon a maximum of 20 jobs.

        :default: - No dependencies

        :see: https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html#Batch-SubmitJob-request-dependsOn
        :stability: experimental
        '''
        result = self._values.get("depends_on")
        return typing.cast(typing.Optional[typing.List[BatchJobDependency]], result)

    @builtins.property
    def payload(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) The payload to be passed as parameters to the batch job.

        :default: - No parameters are passed

        :stability: experimental
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BatchSubmitJobProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.CallApiGatewayEndpointBaseProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class CallApiGatewayEndpointBaseProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_71ab46f6] = None,
        query_parameters: typing.Optional[_TaskInput_71ab46f6] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
    ) -> None:
        '''(experimental) Base CallApiGatewayEdnpoint Task Props.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param method: (experimental) Http method for the API.
        :param api_path: (experimental) Path parameters appended after API endpoint. Default: - No path
        :param auth_type: (experimental) Authentication methods. Default: AuthType.NO_AUTH
        :param headers: (experimental) HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: (experimental) Query strings attatched to end of request. Default: - No query parameters
        :param request_body: (experimental) HTTP Request body. Default: - No request body

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_stepfunctions as stepfunctions
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # duration: monocdk.Duration
            # result_selector: Any
            # task_input: stepfunctions.TaskInput
            
            call_api_gateway_endpoint_base_props = stepfunctions_tasks.CallApiGatewayEndpointBaseProps(
                method=stepfunctions_tasks.HttpMethod.GET,
            
                # the properties below are optional
                api_path="apiPath",
                auth_type=stepfunctions_tasks.AuthType.NO_AUTH,
                comment="comment",
                headers=task_input,
                heartbeat=duration,
                input_path="inputPath",
                integration_pattern=stepfunctions.IntegrationPattern.REQUEST_RESPONSE,
                output_path="outputPath",
                query_parameters=task_input,
                request_body=task_input,
                result_path="resultPath",
                result_selector={
                    "result_selector_key": result_selector
                },
                timeout=duration
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "method": method,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''(experimental) Http method for the API.

        :stability: experimental
        '''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) Path parameters appended after API endpoint.

        :default: - No path

        :stability: experimental
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''(experimental) Authentication methods.

        :default: AuthType.NO_AUTH

        :stability: experimental
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) HTTP request information that does not relate to contents of the request.

        :default: - No headers

        :stability: experimental
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) Query strings attatched to end of request.

        :default: - No query parameters

        :stability: experimental
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) HTTP Request body.

        :default: - No request body

        :stability: experimental
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayEndpointBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallApiGatewayHttpApiEndpoint(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpoint",
):
    '''(experimental) Call HTTP API endpoint as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as apigatewayv2
        
        http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
        
        invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
            api_id=http_api.api_id,
            api_stack=Stack.of(http_api),
            method=tasks.HttpMethod.GET
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        api_id: builtins.str,
        api_stack: _Stack_9f43e4a3,
        stage_name: typing.Optional[builtins.str] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_71ab46f6] = None,
        query_parameters: typing.Optional[_TaskInput_71ab46f6] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param api_id: (experimental) The Id of the API to call.
        :param api_stack: (experimental) The Stack in which the API is defined.
        :param stage_name: (experimental) Name of the stage where the API is deployed to in API Gateway. Default: '$default'
        :param method: (experimental) Http method for the API.
        :param api_path: (experimental) Path parameters appended after API endpoint. Default: - No path
        :param auth_type: (experimental) Authentication methods. Default: AuthType.NO_AUTH
        :param headers: (experimental) HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: (experimental) Query strings attatched to end of request. Default: - No query parameters
        :param request_body: (experimental) HTTP Request body. Default: - No request body
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = CallApiGatewayHttpApiEndpointProps(
            api_id=api_id,
            api_stack=api_stack,
            stage_name=stage_name,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="createPolicyStatements")
    def _create_policy_statements(self) -> typing.List[_PolicyStatement_296fe8a3]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.List[_PolicyStatement_296fe8a3], jsii.invoke(self, "createPolicyStatements", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiEndpoint")
    def _api_endpoint(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiEndpoint"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="arnForExecuteApi")
    def _arn_for_execute_api(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "arnForExecuteApi"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stageName")
    def _stage_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.CallApiGatewayHttpApiEndpointProps",
    jsii_struct_bases=[CallApiGatewayEndpointBaseProps],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
        "api_id": "apiId",
        "api_stack": "apiStack",
        "stage_name": "stageName",
    },
)
class CallApiGatewayHttpApiEndpointProps(CallApiGatewayEndpointBaseProps):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_71ab46f6] = None,
        query_parameters: typing.Optional[_TaskInput_71ab46f6] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
        api_id: builtins.str,
        api_stack: _Stack_9f43e4a3,
        stage_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for calling an HTTP API Endpoint.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param method: (experimental) Http method for the API.
        :param api_path: (experimental) Path parameters appended after API endpoint. Default: - No path
        :param auth_type: (experimental) Authentication methods. Default: AuthType.NO_AUTH
        :param headers: (experimental) HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: (experimental) Query strings attatched to end of request. Default: - No query parameters
        :param request_body: (experimental) HTTP Request body. Default: - No request body
        :param api_id: (experimental) The Id of the API to call.
        :param api_stack: (experimental) The Stack in which the API is defined.
        :param stage_name: (experimental) Name of the stage where the API is deployed to in API Gateway. Default: '$default'

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as apigatewayv2
            
            http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
            
            invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
                api_id=http_api.api_id,
                api_stack=Stack.of(http_api),
                method=tasks.HttpMethod.GET
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "method": method,
            "api_id": api_id,
            "api_stack": api_stack,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body
        if stage_name is not None:
            self._values["stage_name"] = stage_name

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''(experimental) Http method for the API.

        :stability: experimental
        '''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) Path parameters appended after API endpoint.

        :default: - No path

        :stability: experimental
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''(experimental) Authentication methods.

        :default: AuthType.NO_AUTH

        :stability: experimental
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) HTTP request information that does not relate to contents of the request.

        :default: - No headers

        :stability: experimental
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) Query strings attatched to end of request.

        :default: - No query parameters

        :stability: experimental
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) HTTP Request body.

        :default: - No request body

        :stability: experimental
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def api_id(self) -> builtins.str:
        '''(experimental) The Id of the API to call.

        :stability: experimental
        '''
        result = self._values.get("api_id")
        assert result is not None, "Required property 'api_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def api_stack(self) -> _Stack_9f43e4a3:
        '''(experimental) The Stack in which the API is defined.

        :stability: experimental
        '''
        result = self._values.get("api_stack")
        assert result is not None, "Required property 'api_stack' is missing"
        return typing.cast(_Stack_9f43e4a3, result)

    @builtins.property
    def stage_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of the stage where the API is deployed to in API Gateway.

        :default: '$default'

        :stability: experimental
        '''
        result = self._values.get("stage_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayHttpApiEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallApiGatewayRestApiEndpoint(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpoint",
):
    '''(experimental) Call REST API endpoint as a Task.

    Be aware that the header values must be arrays. When passing the Task Token
    in the headers field ``WAIT_FOR_TASK_TOKEN`` integration, use
    ``JsonPath.array()`` to wrap the token in an array::

       # Example automatically generated from non-compiling source. May contain errors.
       import aws_cdk.aws_apigateway as apigateway
       # api: apigateway.RestApi


       tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
           api=api,
           stage_name="Stage",
           method=tasks.HttpMethod.PUT,
           integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
           headers=sfn.TaskInput.from_object({
               "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
           })
       )

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-api-gateway.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as apigateway
        # api: apigateway.RestApi
        
        
        tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
            api=api,
            stage_name="Stage",
            method=tasks.HttpMethod.PUT,
            integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            headers=sfn.TaskInput.from_object({
                "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
            })
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        api: _IRestApi_a27d4788,
        stage_name: builtins.str,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_71ab46f6] = None,
        query_parameters: typing.Optional[_TaskInput_71ab46f6] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param api: (experimental) API to call.
        :param stage_name: (experimental) Name of the stage where the API is deployed to in API Gateway.
        :param method: (experimental) Http method for the API.
        :param api_path: (experimental) Path parameters appended after API endpoint. Default: - No path
        :param auth_type: (experimental) Authentication methods. Default: AuthType.NO_AUTH
        :param headers: (experimental) HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: (experimental) Query strings attatched to end of request. Default: - No query parameters
        :param request_body: (experimental) HTTP Request body. Default: - No request body
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = CallApiGatewayRestApiEndpointProps(
            api=api,
            stage_name=stage_name,
            method=method,
            api_path=api_path,
            auth_type=auth_type,
            headers=headers,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="createPolicyStatements")
    def _create_policy_statements(self) -> typing.List[_PolicyStatement_296fe8a3]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.List[_PolicyStatement_296fe8a3], jsii.invoke(self, "createPolicyStatements", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiEndpoint")
    def _api_endpoint(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "apiEndpoint"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="arnForExecuteApi")
    def _arn_for_execute_api(self) -> builtins.str:
        '''
        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "arnForExecuteApi"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stageName")
    def _stage_name(self) -> typing.Optional[builtins.str]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "stageName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.CallApiGatewayRestApiEndpointProps",
    jsii_struct_bases=[CallApiGatewayEndpointBaseProps],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "method": "method",
        "api_path": "apiPath",
        "auth_type": "authType",
        "headers": "headers",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
        "api": "api",
        "stage_name": "stageName",
    },
)
class CallApiGatewayRestApiEndpointProps(CallApiGatewayEndpointBaseProps):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        method: "HttpMethod",
        api_path: typing.Optional[builtins.str] = None,
        auth_type: typing.Optional[AuthType] = None,
        headers: typing.Optional[_TaskInput_71ab46f6] = None,
        query_parameters: typing.Optional[_TaskInput_71ab46f6] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
        api: _IRestApi_a27d4788,
        stage_name: builtins.str,
    ) -> None:
        '''(experimental) Properties for calling an REST API Endpoint.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param method: (experimental) Http method for the API.
        :param api_path: (experimental) Path parameters appended after API endpoint. Default: - No path
        :param auth_type: (experimental) Authentication methods. Default: AuthType.NO_AUTH
        :param headers: (experimental) HTTP request information that does not relate to contents of the request. Default: - No headers
        :param query_parameters: (experimental) Query strings attatched to end of request. Default: - No query parameters
        :param request_body: (experimental) HTTP Request body. Default: - No request body
        :param api: (experimental) API to call.
        :param stage_name: (experimental) Name of the stage where the API is deployed to in API Gateway.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as apigateway
            # api: apigateway.RestApi
            
            
            tasks.CallApiGatewayRestApiEndpoint(self, "Endpoint",
                api=api,
                stage_name="Stage",
                method=tasks.HttpMethod.PUT,
                integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                headers=sfn.TaskInput.from_object({
                    "TaskToken": sfn.JsonPath.array(sfn.JsonPath.task_token)
                })
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "method": method,
            "api": api,
            "stage_name": stage_name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if api_path is not None:
            self._values["api_path"] = api_path
        if auth_type is not None:
            self._values["auth_type"] = auth_type
        if headers is not None:
            self._values["headers"] = headers
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def method(self) -> "HttpMethod":
        '''(experimental) Http method for the API.

        :stability: experimental
        '''
        result = self._values.get("method")
        assert result is not None, "Required property 'method' is missing"
        return typing.cast("HttpMethod", result)

    @builtins.property
    def api_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) Path parameters appended after API endpoint.

        :default: - No path

        :stability: experimental
        '''
        result = self._values.get("api_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def auth_type(self) -> typing.Optional[AuthType]:
        '''(experimental) Authentication methods.

        :default: AuthType.NO_AUTH

        :stability: experimental
        '''
        result = self._values.get("auth_type")
        return typing.cast(typing.Optional[AuthType], result)

    @builtins.property
    def headers(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) HTTP request information that does not relate to contents of the request.

        :default: - No headers

        :stability: experimental
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) Query strings attatched to end of request.

        :default: - No query parameters

        :stability: experimental
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) HTTP Request body.

        :default: - No request body

        :stability: experimental
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def api(self) -> _IRestApi_a27d4788:
        '''(experimental) API to call.

        :stability: experimental
        '''
        result = self._values.get("api")
        assert result is not None, "Required property 'api' is missing"
        return typing.cast(_IRestApi_a27d4788, result)

    @builtins.property
    def stage_name(self) -> builtins.str:
        '''(experimental) Name of the stage where the API is deployed to in API Gateway.

        :stability: experimental
        '''
        result = self._values.get("stage_name")
        assert result is not None, "Required property 'stage_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallApiGatewayRestApiEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CallAwsService(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.CallAwsService",
):
    '''(experimental) A StepFunctions task to call an AWS service API.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # my_bucket: s3.Bucket
        
        get_object = tasks.CallAwsService(self, "GetObject",
            service="s3",
            action="getObject",
            parameters={
                "Bucket": my_bucket.bucket_name,
                "Key": sfn.JsonPath.string_at("$.key")
            },
            iam_resources=[my_bucket.arn_for_objects("*")]
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param action: (experimental) The API action to call. Use camelCase.
        :param iam_resources: (experimental) The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: (experimental) The AWS service to call.
        :param iam_action: (experimental) The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: (experimental) Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = CallAwsServiceProps(
            action=action,
            iam_resources=iam_resources,
            service=service,
            iam_action=iam_action,
            parameters=parameters,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.CallAwsServiceProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "action": "action",
        "iam_resources": "iamResources",
        "service": "service",
        "iam_action": "iamAction",
        "parameters": "parameters",
    },
)
class CallAwsServiceProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        action: builtins.str,
        iam_resources: typing.Sequence[builtins.str],
        service: builtins.str,
        iam_action: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''(experimental) Properties for calling an AWS service's API action from your state machine.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param action: (experimental) The API action to call. Use camelCase.
        :param iam_resources: (experimental) The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. By default the action for this IAM statement will be ``service:action``.
        :param service: (experimental) The AWS service to call.
        :param iam_action: (experimental) The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call. Use in the case where the IAM action name does not match with the API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``. Default: - service:action
        :param parameters: (experimental) Parameters for the API action call. Use PascalCase for the parameter names. Default: - no parameters

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        :stability: experimental
        :exampleMetadata: infused

        Example::

            # my_bucket: s3.Bucket
            
            get_object = tasks.CallAwsService(self, "GetObject",
                service="s3",
                action="getObject",
                parameters={
                    "Bucket": my_bucket.bucket_name,
                    "Key": sfn.JsonPath.string_at("$.key")
                },
                iam_resources=[my_bucket.arn_for_objects("*")]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "action": action,
            "iam_resources": iam_resources,
            "service": service,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if iam_action is not None:
            self._values["iam_action"] = iam_action
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def action(self) -> builtins.str:
        '''(experimental) The API action to call.

        Use camelCase.

        :stability: experimental
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_resources(self) -> typing.List[builtins.str]:
        '''(experimental) The resources for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        By default the action for this IAM statement will be ``service:action``.

        :stability: experimental
        '''
        result = self._values.get("iam_resources")
        assert result is not None, "Required property 'iam_resources' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def service(self) -> builtins.str:
        '''(experimental) The AWS service to call.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/supported-services-awssdk.html
        :stability: experimental
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def iam_action(self) -> typing.Optional[builtins.str]:
        '''(experimental) The action for the IAM statement that will be added to the state machine role's policy to allow the state machine to make the API call.

        Use in the case where the IAM action name does not match with the
        API service/action name, e.g. ``s3:ListBuckets`` requires ``s3:ListAllMyBuckets``.

        :default: - service:action

        :stability: experimental
        '''
        result = self._values.get("iam_action")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) Parameters for the API action call.

        Use PascalCase for the parameter names.

        :default: - no parameters

        :stability: experimental
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CallAwsServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.Channel",
    jsii_struct_bases=[],
    name_mapping={
        "channel_name": "channelName",
        "data_source": "dataSource",
        "compression_type": "compressionType",
        "content_type": "contentType",
        "input_mode": "inputMode",
        "record_wrapper_type": "recordWrapperType",
        "shuffle_config": "shuffleConfig",
    },
)
class Channel:
    def __init__(
        self,
        *,
        channel_name: builtins.str,
        data_source: "DataSource",
        compression_type: typing.Optional["CompressionType"] = None,
        content_type: typing.Optional[builtins.str] = None,
        input_mode: typing.Optional["InputMode"] = None,
        record_wrapper_type: typing.Optional["RecordWrapperType"] = None,
        shuffle_config: typing.Optional["ShuffleConfig"] = None,
    ) -> None:
        '''(experimental) Describes the training, validation or test dataset and the Amazon S3 location where it is stored.

        :param channel_name: (experimental) Name of the channel.
        :param data_source: (experimental) Location of the channel data.
        :param compression_type: (experimental) Compression type if training data is compressed. Default: - None
        :param content_type: (experimental) The MIME type of the data. Default: - None
        :param input_mode: (experimental) Input mode to use for the data channel in a training job. Default: - None
        :param record_wrapper_type: (experimental) Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format. In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record. If the input data is already in RecordIO format, you don't need to set this attribute. Default: - None
        :param shuffle_config: (experimental) Shuffle config option for input data in a channel. Default: - None

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # s3_location: stepfunctions_tasks.S3Location
            
            channel = stepfunctions_tasks.Channel(
                channel_name="channelName",
                data_source=stepfunctions_tasks.DataSource(
                    s3_data_source=stepfunctions_tasks.S3DataSource(
                        s3_location=s3_location,
            
                        # the properties below are optional
                        attribute_names=["attributeNames"],
                        s3_data_distribution_type=stepfunctions_tasks.S3DataDistributionType.FULLY_REPLICATED,
                        s3_data_type=stepfunctions_tasks.S3DataType.MANIFEST_FILE
                    )
                ),
            
                # the properties below are optional
                compression_type=stepfunctions_tasks.CompressionType.NONE,
                content_type="contentType",
                input_mode=stepfunctions_tasks.InputMode.PIPE,
                record_wrapper_type=stepfunctions_tasks.RecordWrapperType.NONE,
                shuffle_config=stepfunctions_tasks.ShuffleConfig(
                    seed=123
                )
            )
        '''
        if isinstance(data_source, dict):
            data_source = DataSource(**data_source)
        if isinstance(shuffle_config, dict):
            shuffle_config = ShuffleConfig(**shuffle_config)
        self._values: typing.Dict[str, typing.Any] = {
            "channel_name": channel_name,
            "data_source": data_source,
        }
        if compression_type is not None:
            self._values["compression_type"] = compression_type
        if content_type is not None:
            self._values["content_type"] = content_type
        if input_mode is not None:
            self._values["input_mode"] = input_mode
        if record_wrapper_type is not None:
            self._values["record_wrapper_type"] = record_wrapper_type
        if shuffle_config is not None:
            self._values["shuffle_config"] = shuffle_config

    @builtins.property
    def channel_name(self) -> builtins.str:
        '''(experimental) Name of the channel.

        :stability: experimental
        '''
        result = self._values.get("channel_name")
        assert result is not None, "Required property 'channel_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def data_source(self) -> "DataSource":
        '''(experimental) Location of the channel data.

        :stability: experimental
        '''
        result = self._values.get("data_source")
        assert result is not None, "Required property 'data_source' is missing"
        return typing.cast("DataSource", result)

    @builtins.property
    def compression_type(self) -> typing.Optional["CompressionType"]:
        '''(experimental) Compression type if training data is compressed.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("compression_type")
        return typing.cast(typing.Optional["CompressionType"], result)

    @builtins.property
    def content_type(self) -> typing.Optional[builtins.str]:
        '''(experimental) The MIME type of the data.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("content_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_mode(self) -> typing.Optional["InputMode"]:
        '''(experimental) Input mode to use for the data channel in a training job.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("input_mode")
        return typing.cast(typing.Optional["InputMode"], result)

    @builtins.property
    def record_wrapper_type(self) -> typing.Optional["RecordWrapperType"]:
        '''(experimental) Specify RecordIO as the value when input data is in raw format but the training algorithm requires the RecordIO format.

        In this case, Amazon SageMaker wraps each individual S3 object in a RecordIO record.
        If the input data is already in RecordIO format, you don't need to set this attribute.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("record_wrapper_type")
        return typing.cast(typing.Optional["RecordWrapperType"], result)

    @builtins.property
    def shuffle_config(self) -> typing.Optional["ShuffleConfig"]:
        '''(experimental) Shuffle config option for input data in a channel.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("shuffle_config")
        return typing.cast(typing.Optional["ShuffleConfig"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Channel(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Classification(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.Classification",
):
    '''(experimental) The classification within a EMR Containers application configuration.

    Class can be extended to add other classifications.
    For example, new Classification('xxx-yyy');

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
            virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
            release_label=tasks.ReleaseLabel.EMR_6_2_0,
            job_name="EMR-Containers-Job",
            job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
                spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
                    entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                )
            ),
            application_config=[dynamodb.aws_stepfunctions_tasks.ApplicationConfiguration(
                classification=tasks.Classification.SPARK_DEFAULTS,
                properties={
                    "spark.executor.instances": "1",
                    "spark.executor.memory": "512M"
                }
            )]
        )
    '''

    def __init__(self, classification_statement: builtins.str) -> None:
        '''(experimental) Creates a new Classification.

        :param classification_statement: A literal string in case a new EMR classification is released, if not already defined.

        :stability: experimental
        '''
        jsii.create(self.__class__, self, [classification_statement])

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SPARK")
    def SPARK(cls) -> "Classification":
        '''(experimental) Sets the maximizeResourceAllocation property to true or false.

        When true, Amazon EMR automatically configures spark-defaults properties based on cluster hardware configuration.

        For more info:

        :see: https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-spark-configure.html#emr-spark-maximizeresourceallocation
        :stability: experimental
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SPARK_DEFAULTS")
    def SPARK_DEFAULTS(cls) -> "Classification":
        '''(experimental) Sets values in the spark-defaults.conf file.

        For more info:

        :see: https://spark.apache.org/docs/latest/configuration.html
        :stability: experimental
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_DEFAULTS"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SPARK_ENV")
    def SPARK_ENV(cls) -> "Classification":
        '''(experimental) Sets values in the spark-env.sh file.

        For more info:

        :see: https://spark.apache.org/docs/latest/configuration.html#environment-variables
        :stability: experimental
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_ENV"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SPARK_HIVE_SITE")
    def SPARK_HIVE_SITE(cls) -> "Classification":
        '''(experimental) Sets values in the hive-site.xml for Spark.

        :stability: experimental
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_HIVE_SITE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SPARK_LOG4J")
    def SPARK_LOG4_J(cls) -> "Classification":
        '''(experimental) Sets values in the log4j.properties file.

        For more settings and info:

        :see: https://github.com/apache/spark/blob/master/conf/log4j.properties.template
        :stability: experimental
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_LOG4J"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SPARK_METRICS")
    def SPARK_METRICS(cls) -> "Classification":
        '''(experimental) Sets values in the metrics.properties file.

        For more settings and info:

        :see: https://github.com/apache/spark/blob/master/conf/metrics.properties.template
        :stability: experimental
        '''
        return typing.cast("Classification", jsii.sget(cls, "SPARK_METRICS"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="classificationStatement")
    def classification_statement(self) -> builtins.str:
        '''(experimental) A literal string in case a new EMR classification is released, if not already defined.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "classificationStatement"))


class CodeBuildStartBuild(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.CodeBuildStartBuild",
):
    '''(experimental) Start a CodeBuild Build as a task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-codebuild.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as codebuild
        
        
        codebuild_project = codebuild.Project(self, "Project",
            project_name="MyTestProject",
            build_spec=codebuild.BuildSpec.from_object({
                "version": "0.2",
                "phases": {
                    "build": {
                        "commands": ["echo \"Hello, CodeBuild!\""
                        ]
                    }
                }
            })
        )
        
        task = tasks.CodeBuildStartBuild(self, "Task",
            project=codebuild_project,
            integration_pattern=sfn.IntegrationPattern.RUN_JOB,
            environment_variables_override={
                "ZONE": codebuild.aws_codebuild.BuildEnvironmentVariable(
                    type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    value=sfn.JsonPath.string_at("$.envVariables.zone")
                )
            }
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        project: _IProject_6da8803e,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, _BuildEnvironmentVariable_00095c97]] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param project: (experimental) CodeBuild project to start.
        :param environment_variables_override: (experimental) A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = CodeBuildStartBuildProps(
            project=project,
            environment_variables_override=environment_variables_override,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.CodeBuildStartBuildProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "project": "project",
        "environment_variables_override": "environmentVariablesOverride",
    },
)
class CodeBuildStartBuildProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        project: _IProject_6da8803e,
        environment_variables_override: typing.Optional[typing.Mapping[builtins.str, _BuildEnvironmentVariable_00095c97]] = None,
    ) -> None:
        '''(experimental) Properties for CodeBuildStartBuild.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param project: (experimental) CodeBuild project to start.
        :param environment_variables_override: (experimental) A set of environment variables to be used for this build only. Default: - the latest environment variables already defined in the build project.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as codebuild
            
            
            codebuild_project = codebuild.Project(self, "Project",
                project_name="MyTestProject",
                build_spec=codebuild.BuildSpec.from_object({
                    "version": "0.2",
                    "phases": {
                        "build": {
                            "commands": ["echo \"Hello, CodeBuild!\""
                            ]
                        }
                    }
                })
            )
            
            task = tasks.CodeBuildStartBuild(self, "Task",
                project=codebuild_project,
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                environment_variables_override={
                    "ZONE": codebuild.aws_codebuild.BuildEnvironmentVariable(
                        type=codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                        value=sfn.JsonPath.string_at("$.envVariables.zone")
                    )
                }
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "project": project,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if environment_variables_override is not None:
            self._values["environment_variables_override"] = environment_variables_override

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def project(self) -> _IProject_6da8803e:
        '''(experimental) CodeBuild project to start.

        :stability: experimental
        '''
        result = self._values.get("project")
        assert result is not None, "Required property 'project' is missing"
        return typing.cast(_IProject_6da8803e, result)

    @builtins.property
    def environment_variables_override(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, _BuildEnvironmentVariable_00095c97]]:
        '''(experimental) A set of environment variables to be used for this build only.

        :default: - the latest environment variables already defined in the build project.

        :stability: experimental
        '''
        result = self._values.get("environment_variables_override")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, _BuildEnvironmentVariable_00095c97]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CodeBuildStartBuildProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.CommonEcsRunTaskProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "task_definition": "taskDefinition",
        "container_overrides": "containerOverrides",
        "integration_pattern": "integrationPattern",
    },
)
class CommonEcsRunTaskProps:
    def __init__(
        self,
        *,
        cluster: _ICluster_42c4ec1a,
        task_definition: _TaskDefinition_c0dacfb4,
        container_overrides: typing.Optional[typing.Sequence["ContainerOverride"]] = None,
        integration_pattern: typing.Optional[_ServiceIntegrationPattern_5581ee88] = None,
    ) -> None:
        '''(experimental) Basic properties for ECS Tasks.

        :param cluster: (experimental) The topic to run the task on.
        :param task_definition: (experimental) Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions
        :param container_overrides: (experimental) Container setting overrides. Key is the name of the container to override, value is the values you want to override. Default: - No overrides
        :param integration_pattern: (experimental) The service integration pattern indicates different ways to call RunTask in ECS. The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN. Default: FIRE_AND_FORGET

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_stepfunctions as stepfunctions
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # task_definition: ecs.TaskDefinition
            
            common_ecs_run_task_props = stepfunctions_tasks.CommonEcsRunTaskProps(
                cluster=cluster,
                task_definition=task_definition,
            
                # the properties below are optional
                container_overrides=[stepfunctions_tasks.ContainerOverride(
                    container_definition=container_definition,
            
                    # the properties below are optional
                    command=["command"],
                    cpu=123,
                    environment=[stepfunctions_tasks.TaskEnvironmentVariable(
                        name="name",
                        value="value"
                    )],
                    memory_limit=123,
                    memory_reservation=123
                )],
                integration_pattern=stepfunctions.ServiceIntegrationPattern.FIRE_AND_FORGET
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "task_definition": task_definition,
        }
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern

    @builtins.property
    def cluster(self) -> _ICluster_42c4ec1a:
        '''(experimental) The topic to run the task on.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_42c4ec1a, result)

    @builtins.property
    def task_definition(self) -> _TaskDefinition_c0dacfb4:
        '''(experimental) Task Definition used for running tasks in the service.

        Note: this must be TaskDefinition, and not ITaskDefinition,
        as it requires properties that are not known for imported task definitions

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(_TaskDefinition_c0dacfb4, result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[typing.List["ContainerOverride"]]:
        '''(experimental) Container setting overrides.

        Key is the name of the container to override, value is the
        values you want to override.

        :default: - No overrides

        :stability: experimental
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[typing.List["ContainerOverride"]], result)

    @builtins.property
    def integration_pattern(
        self,
    ) -> typing.Optional[_ServiceIntegrationPattern_5581ee88]:
        '''(experimental) The service integration pattern indicates different ways to call RunTask in ECS.

        The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.

        :default: FIRE_AND_FORGET

        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_ServiceIntegrationPattern_5581ee88], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonEcsRunTaskProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.CompressionType")
class CompressionType(enum.Enum):
    '''(experimental) Compression type of the data.

    :stability: experimental
    '''

    NONE = "NONE"
    '''(experimental) None compression type.

    :stability: experimental
    '''
    GZIP = "GZIP"
    '''(experimental) Gzip compression type.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ContainerDefinitionConfig",
    jsii_struct_bases=[],
    name_mapping={"parameters": "parameters"},
)
class ContainerDefinitionConfig:
    def __init__(
        self,
        *,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''(experimental) Configuration options for the ContainerDefinition.

        :param parameters: (experimental) Additional parameters to pass to the base task. Default: - No additional parameters passed

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # parameters: Any
            
            container_definition_config = stepfunctions_tasks.ContainerDefinitionConfig(
                parameters={
                    "parameters_key": parameters
                }
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) Additional parameters to pass to the base task.

        :default: - No additional parameters passed

        :stability: experimental
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDefinitionConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ContainerDefinitionOptions",
    jsii_struct_bases=[],
    name_mapping={
        "container_host_name": "containerHostName",
        "environment_variables": "environmentVariables",
        "image": "image",
        "mode": "mode",
        "model_package_name": "modelPackageName",
        "model_s3_location": "modelS3Location",
    },
)
class ContainerDefinitionOptions:
    def __init__(
        self,
        *,
        container_host_name: typing.Optional[builtins.str] = None,
        environment_variables: typing.Optional[_TaskInput_71ab46f6] = None,
        image: typing.Optional["DockerImage"] = None,
        mode: typing.Optional["Mode"] = None,
        model_package_name: typing.Optional[builtins.str] = None,
        model_s3_location: typing.Optional["S3Location"] = None,
    ) -> None:
        '''(experimental) Properties to define a ContainerDefinition.

        :param container_host_name: (experimental) This parameter is ignored for models that contain only a PrimaryContainer. When a ContainerDefinition is part of an inference pipeline, the value of the parameter uniquely identifies the container for the purposes of logging and metrics. Default: - None
        :param environment_variables: (experimental) The environment variables to set in the Docker container. Default: - No variables
        :param image: (experimental) The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored. Default: - None
        :param mode: (experimental) Defines how many models the container hosts. Default: - Mode.SINGLE_MODEL
        :param model_package_name: (experimental) The name or Amazon Resource Name (ARN) of the model package to use to create the model. Default: - None
        :param model_s3_location: (experimental) The S3 path where the model artifacts, which result from model training, are stored. This path must point to a single gzip compressed tar archive (.tar.gz suffix). The S3 path is required for Amazon SageMaker built-in algorithms, but not if you use your own algorithms. Default: - None

        :see: https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateModel(self, "Sagemaker",
                model_name="MyModel",
                primary_container=tasks.ContainerDefinition(
                    image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
                    mode=tasks.Mode.SINGLE_MODEL,
                    model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if container_host_name is not None:
            self._values["container_host_name"] = container_host_name
        if environment_variables is not None:
            self._values["environment_variables"] = environment_variables
        if image is not None:
            self._values["image"] = image
        if mode is not None:
            self._values["mode"] = mode
        if model_package_name is not None:
            self._values["model_package_name"] = model_package_name
        if model_s3_location is not None:
            self._values["model_s3_location"] = model_s3_location

    @builtins.property
    def container_host_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) This parameter is ignored for models that contain only a PrimaryContainer.

        When a ContainerDefinition is part of an inference pipeline,
        the value of the parameter uniquely identifies the container for the purposes of logging and metrics.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("container_host_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def environment_variables(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) The environment variables to set in the Docker container.

        :default: - No variables

        :stability: experimental
        '''
        result = self._values.get("environment_variables")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def image(self) -> typing.Optional["DockerImage"]:
        '''(experimental) The Amazon EC2 Container Registry (Amazon ECR) path where inference code is stored.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("image")
        return typing.cast(typing.Optional["DockerImage"], result)

    @builtins.property
    def mode(self) -> typing.Optional["Mode"]:
        '''(experimental) Defines how many models the container hosts.

        :default: - Mode.SINGLE_MODEL

        :stability: experimental
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional["Mode"], result)

    @builtins.property
    def model_package_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name or Amazon Resource Name (ARN) of the model package to use to create the model.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("model_package_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def model_s3_location(self) -> typing.Optional["S3Location"]:
        '''(experimental) The S3 path where the model artifacts, which result from model training, are stored.

        This path must point to a single gzip compressed tar archive (.tar.gz suffix).
        The S3 path is required for Amazon SageMaker built-in algorithms, but not if you use your own algorithms.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("model_s3_location")
        return typing.cast(typing.Optional["S3Location"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerDefinitionOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ContainerOverride",
    jsii_struct_bases=[],
    name_mapping={
        "container_definition": "containerDefinition",
        "command": "command",
        "cpu": "cpu",
        "environment": "environment",
        "memory_limit": "memoryLimit",
        "memory_reservation": "memoryReservation",
    },
)
class ContainerOverride:
    def __init__(
        self,
        *,
        container_definition: _ContainerDefinition_7934d1e1,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        cpu: typing.Optional[jsii.Number] = None,
        environment: typing.Optional[typing.Sequence["TaskEnvironmentVariable"]] = None,
        memory_limit: typing.Optional[jsii.Number] = None,
        memory_reservation: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) A list of container overrides that specify the name of a container and the overrides it should receive.

        :param container_definition: (experimental) Name of the container inside the task definition.
        :param command: (experimental) Command to run inside the container. Default: - Default command from the Docker image or the task definition
        :param cpu: (experimental) The number of cpu units reserved for the container. Default: - The default value from the task definition.
        :param environment: (experimental) The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the task definition. Default: - The existing environment variables from the Docker image or the task definition
        :param memory_limit: (experimental) The hard limit (in MiB) of memory to present to the container. Default: - The default value from the task definition.
        :param memory_reservation: (experimental) The soft limit (in MiB) of memory to reserve for the container. Default: - The default value from the task definition.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # container_definition: ecs.ContainerDefinition
            
            container_override = stepfunctions_tasks.ContainerOverride(
                container_definition=container_definition,
            
                # the properties below are optional
                command=["command"],
                cpu=123,
                environment=[stepfunctions_tasks.TaskEnvironmentVariable(
                    name="name",
                    value="value"
                )],
                memory_limit=123,
                memory_reservation=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "container_definition": container_definition,
        }
        if command is not None:
            self._values["command"] = command
        if cpu is not None:
            self._values["cpu"] = cpu
        if environment is not None:
            self._values["environment"] = environment
        if memory_limit is not None:
            self._values["memory_limit"] = memory_limit
        if memory_reservation is not None:
            self._values["memory_reservation"] = memory_reservation

    @builtins.property
    def container_definition(self) -> _ContainerDefinition_7934d1e1:
        '''(experimental) Name of the container inside the task definition.

        :stability: experimental
        '''
        result = self._values.get("container_definition")
        assert result is not None, "Required property 'container_definition' is missing"
        return typing.cast(_ContainerDefinition_7934d1e1, result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Command to run inside the container.

        :default: - Default command from the Docker image or the task definition

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of cpu units reserved for the container.

        :default: - The default value from the task definition.

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def environment(self) -> typing.Optional[typing.List["TaskEnvironmentVariable"]]:
        '''(experimental) The environment variables to send to the container.

        You can add new environment variables, which are added to the container at launch,
        or you can override the existing environment variables from the Docker image or the task definition.

        :default: - The existing environment variables from the Docker image or the task definition

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.List["TaskEnvironmentVariable"]], result)

    @builtins.property
    def memory_limit(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The hard limit (in MiB) of memory to present to the container.

        :default: - The default value from the task definition.

        :stability: experimental
        '''
        result = self._values.get("memory_limit")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def memory_reservation(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The soft limit (in MiB) of memory to reserve for the container.

        :default: - The default value from the task definition.

        :stability: experimental
        '''
        result = self._values.get("memory_reservation")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerOverride(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ContainerOverrides",
    jsii_struct_bases=[],
    name_mapping={
        "command": "command",
        "environment": "environment",
        "gpu_count": "gpuCount",
        "instance_type": "instanceType",
        "memory": "memory",
        "vcpus": "vcpus",
    },
)
class ContainerOverrides:
    def __init__(
        self,
        *,
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        instance_type: typing.Optional[_InstanceType_072ad323] = None,
        memory: typing.Optional[jsii.Number] = None,
        vcpus: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The overrides that should be sent to a container.

        :param command: (experimental) The command to send to the container that overrides the default command from the Docker image or the job definition. Default: - No command overrides
        :param environment: (experimental) The environment variables to send to the container. You can add new environment variables, which are added to the container at launch, or you can override the existing environment variables from the Docker image or the job definition. Default: - No environment overrides
        :param gpu_count: (experimental) The number of physical GPUs to reserve for the container. The number of GPUs reserved for all containers in a job should not exceed the number of available GPUs on the compute resource that the job is launched on. Default: - No GPU reservation
        :param instance_type: (experimental) The instance type to use for a multi-node parallel job. This parameter is not valid for single-node container jobs. Default: - No instance type overrides
        :param memory: (experimental) The number of MiB of memory reserved for the job. This value overrides the value set in the job definition. Default: - No memory overrides
        :param vcpus: (experimental) The number of vCPUs to reserve for the container. This value overrides the value set in the job definition. Default: - No vCPUs overrides

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ec2 as ec2
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # instance_type: ec2.InstanceType
            
            container_overrides = stepfunctions_tasks.ContainerOverrides(
                command=["command"],
                environment={
                    "environment_key": "environment"
                },
                gpu_count=123,
                instance_type=instance_type,
                memory=123,
                vcpus=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if command is not None:
            self._values["command"] = command
        if environment is not None:
            self._values["environment"] = environment
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if memory is not None:
            self._values["memory"] = memory
        if vcpus is not None:
            self._values["vcpus"] = vcpus

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command to send to the container that overrides the default command from the Docker image or the job definition.

        :default: - No command overrides

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to send to the container.

        You can add new environment variables, which are added to the container
        at launch, or you can override the existing environment variables from
        the Docker image or the job definition.

        :default: - No environment overrides

        :stability: experimental
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def gpu_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of physical GPUs to reserve for the container.

        The number of GPUs reserved for all containers in a job
        should not exceed the number of available GPUs on the compute
        resource that the job is launched on.

        :default: - No GPU reservation

        :stability: experimental
        '''
        result = self._values.get("gpu_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[_InstanceType_072ad323]:
        '''(experimental) The instance type to use for a multi-node parallel job.

        This parameter is not valid for single-node container jobs.

        :default: - No instance type overrides

        :stability: experimental
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[_InstanceType_072ad323], result)

    @builtins.property
    def memory(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of MiB of memory reserved for the job.

        This value overrides the value set in the job definition.

        :default: - No memory overrides

        :stability: experimental
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def vcpus(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of vCPUs to reserve for the container.

        This value overrides the value set in the job definition.

        :default: - No vCPUs overrides

        :stability: experimental
        '''
        result = self._values.get("vcpus")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ContainerOverrides(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.DataSource",
    jsii_struct_bases=[],
    name_mapping={"s3_data_source": "s3DataSource"},
)
class DataSource:
    def __init__(self, *, s3_data_source: "S3DataSource") -> None:
        '''(experimental) Location of the channel data.

        :param s3_data_source: (experimental) S3 location of the data source that is associated with a channel.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
                training_job_name=sfn.JsonPath.string_at("$.JobName"),
                algorithm_specification=dynamodb.aws_stepfunctions_tasks.AlgorithmSpecification(
                    algorithm_name="BlazingText",
                    training_input_mode=tasks.InputMode.FILE
                ),
                input_data_config=[dynamodb.aws_stepfunctions_tasks.Channel(
                    channel_name="train",
                    data_source=dynamodb.aws_stepfunctions_tasks.DataSource(
                        s3_data_source=dynamodb.aws_stepfunctions_tasks.S3DataSource(
                            s3_data_type=tasks.S3DataType.S3_PREFIX,
                            s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                        )
                    )
                )],
                output_data_config=dynamodb.aws_stepfunctions_tasks.OutputDataConfig(
                    s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
                ),
                resource_config=dynamodb.aws_stepfunctions_tasks.ResourceConfig(
                    instance_count=1,
                    instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                    volume_size=Size.gibibytes(50)
                ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
                stopping_condition=dynamodb.aws_stepfunctions_tasks.StoppingCondition(
                    max_runtime=Duration.hours(2)
                )
            )
        '''
        if isinstance(s3_data_source, dict):
            s3_data_source = S3DataSource(**s3_data_source)
        self._values: typing.Dict[str, typing.Any] = {
            "s3_data_source": s3_data_source,
        }

    @builtins.property
    def s3_data_source(self) -> "S3DataSource":
        '''(experimental) S3 location of the data source that is associated with a channel.

        :stability: experimental
        '''
        result = self._values.get("s3_data_source")
        assert result is not None, "Required property 's3_data_source' is missing"
        return typing.cast("S3DataSource", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DataSource(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DockerImage(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_stepfunctions_tasks.DockerImage",
):
    '''(experimental) Creates ``IDockerImage`` instances.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.SageMakerCreateModel(self, "Sagemaker",
            model_name="MyModel",
            primary_container=tasks.ContainerDefinition(
                image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
                mode=tasks.Mode.SINGLE_MODEL,
                model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromAsset") # type: ignore[misc]
    @builtins.classmethod
    def from_asset(
        cls,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        directory: builtins.str,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[_DockerImageAssetInvalidationOptions_e5375707] = None,
        network_mode: typing.Optional[_NetworkMode_2d19607e] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
    ) -> "DockerImage":
        '''(experimental) Reference a Docker image that is provided as an Asset in the current app.

        :param scope: the scope in which to create the Asset.
        :param id: the ID for the asset in the construct tree.
        :param directory: (experimental) The directory where the Dockerfile is stored. Any directory inside with a name that matches the CDK output folder (cdk.out by default) will be excluded from the asset
        :param build_args: (experimental) Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: (experimental) Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: (experimental) Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: (experimental) Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: (experimental) Docker target to build to. Default: - no target
        :param extra_hash: (deprecated) Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param exclude: (deprecated) Glob patterns to exclude from the copy. Default: nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (deprecated) The ignore behavior to use for exclude patterns. Default: - GLOB for file assets, DOCKER or GLOB for docker assets depending on whether the '
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER

        :stability: experimental
        '''
        props = _DockerImageAssetProps_beaa8517(
            directory=directory,
            build_args=build_args,
            file=file,
            invalidation=invalidation,
            network_mode=network_mode,
            repository_name=repository_name,
            target=target,
            extra_hash=extra_hash,
            exclude=exclude,
            follow=follow,
            ignore_mode=ignore_mode,
            follow_symlinks=follow_symlinks,
        )

        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromAsset", [scope, id, props]))

    @jsii.member(jsii_name="fromEcrRepository") # type: ignore[misc]
    @builtins.classmethod
    def from_ecr_repository(
        cls,
        repository: _IRepository_8b4d2894,
        tag: typing.Optional[builtins.str] = None,
    ) -> "DockerImage":
        '''(experimental) Reference a Docker image stored in an ECR repository.

        :param repository: the ECR repository where the image is hosted.
        :param tag: an optional ``tag``.

        :stability: experimental
        '''
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromEcrRepository", [repository, tag]))

    @jsii.member(jsii_name="fromJsonExpression") # type: ignore[misc]
    @builtins.classmethod
    def from_json_expression(
        cls,
        expression: builtins.str,
        allow_any_ecr_image_pull: typing.Optional[builtins.bool] = None,
    ) -> "DockerImage":
        '''(experimental) Reference a Docker image which URI is obtained from the task's input.

        :param expression: the JSON path expression with the task input.
        :param allow_any_ecr_image_pull: whether ECR access should be permitted (set to ``false`` if the image will never be in ECR).

        :stability: experimental
        '''
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromJsonExpression", [expression, allow_any_ecr_image_pull]))

    @jsii.member(jsii_name="fromRegistry") # type: ignore[misc]
    @builtins.classmethod
    def from_registry(cls, image_uri: builtins.str) -> "DockerImage":
        '''(experimental) Reference a Docker image by it's URI.

        When referencing ECR images, prefer using ``inEcr``.

        :param image_uri: the URI to the docker image.

        :stability: experimental
        '''
        return typing.cast("DockerImage", jsii.sinvoke(cls, "fromRegistry", [image_uri]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, task: "ISageMakerTask") -> "DockerImageConfig":
        '''(experimental) Called when the image is used by a SageMaker task.

        :param task: -

        :stability: experimental
        '''
        ...


class _DockerImageProxy(DockerImage):
    @jsii.member(jsii_name="bind")
    def bind(self, task: "ISageMakerTask") -> "DockerImageConfig":
        '''(experimental) Called when the image is used by a SageMaker task.

        :param task: -

        :stability: experimental
        '''
        return typing.cast("DockerImageConfig", jsii.invoke(self, "bind", [task]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, DockerImage).__jsii_proxy_class__ = lambda : _DockerImageProxy


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.DockerImageConfig",
    jsii_struct_bases=[],
    name_mapping={"image_uri": "imageUri"},
)
class DockerImageConfig:
    def __init__(self, *, image_uri: builtins.str) -> None:
        '''(experimental) Configuration for a using Docker image.

        :param image_uri: (experimental) The fully qualified URI of the Docker image.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            docker_image_config = stepfunctions_tasks.DockerImageConfig(
                image_uri="imageUri"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "image_uri": image_uri,
        }

    @builtins.property
    def image_uri(self) -> builtins.str:
        '''(experimental) The fully qualified URI of the Docker image.

        :stability: experimental
        '''
        result = self._values.get("image_uri")
        assert result is not None, "Required property 'image_uri' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DockerImageConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DynamoAttributeValue(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoAttributeValue",
):
    '''(experimental) Represents the data for an attribute.

    Each attribute value is described as a name-value pair.
    The name is the data type, and the value is the data itself.

    :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoDeleteItem(self, "DeleteItem",
            key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
            table=my_table,
            result_path=sfn.JsonPath.DISCARD
        )
    '''

    @jsii.member(jsii_name="booleanFromJsonPath") # type: ignore[misc]
    @builtins.classmethod
    def boolean_from_json_path(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Boolean from state input through Json path.

        For example:  "BOOL": true

        :param value: Json path that specifies state input to be used.

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "booleanFromJsonPath", [value]))

    @jsii.member(jsii_name="fromBinary") # type: ignore[misc]
    @builtins.classmethod
    def from_binary(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Binary.

        For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"

        :param value: base-64 encoded string.

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromBinary", [value]))

    @jsii.member(jsii_name="fromBinarySet") # type: ignore[misc]
    @builtins.classmethod
    def from_binary_set(
        cls,
        value: typing.Sequence[builtins.str],
    ) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Binary Set.

        For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromBinarySet", [value]))

    @jsii.member(jsii_name="fromBoolean") # type: ignore[misc]
    @builtins.classmethod
    def from_boolean(cls, value: builtins.bool) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Boolean.

        For example:  "BOOL": true

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromBoolean", [value]))

    @jsii.member(jsii_name="fromList") # type: ignore[misc]
    @builtins.classmethod
    def from_list(
        cls,
        value: typing.Sequence["DynamoAttributeValue"],
    ) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type List.

        For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromList", [value]))

    @jsii.member(jsii_name="fromMap") # type: ignore[misc]
    @builtins.classmethod
    def from_map(
        cls,
        value: typing.Mapping[builtins.str, "DynamoAttributeValue"],
    ) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Map.

        For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromMap", [value]))

    @jsii.member(jsii_name="fromNull") # type: ignore[misc]
    @builtins.classmethod
    def from_null(cls, value: builtins.bool) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Null.

        For example:  "NULL": true

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromNull", [value]))

    @jsii.member(jsii_name="fromNumber") # type: ignore[misc]
    @builtins.classmethod
    def from_number(cls, value: jsii.Number) -> "DynamoAttributeValue":
        '''(experimental) Sets a literal number.

        For example: 1234
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromNumber", [value]))

    @jsii.member(jsii_name="fromNumberSet") # type: ignore[misc]
    @builtins.classmethod
    def from_number_set(
        cls,
        value: typing.Sequence[jsii.Number],
    ) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Number Set.

        For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromNumberSet", [value]))

    @jsii.member(jsii_name="fromString") # type: ignore[misc]
    @builtins.classmethod
    def from_string(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type String.

        For example:  "S": "Hello"
        Strings may be literal values or as JsonPath. Example values:

        - ``DynamoAttributeValue.fromString('someValue')``
        - ``DynamoAttributeValue.fromString(JsonPath.stringAt('$.bar'))``

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromString", [value]))

    @jsii.member(jsii_name="fromStringSet") # type: ignore[misc]
    @builtins.classmethod
    def from_string_set(
        cls,
        value: typing.Sequence[builtins.str],
    ) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type String Set.

        For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "fromStringSet", [value]))

    @jsii.member(jsii_name="listFromJsonPath") # type: ignore[misc]
    @builtins.classmethod
    def list_from_json_path(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type List.

        For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"S", "Veggies"}]

        :param value: Json path that specifies state input to be used.

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "listFromJsonPath", [value]))

    @jsii.member(jsii_name="mapFromJsonPath") # type: ignore[misc]
    @builtins.classmethod
    def map_from_json_path(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Map.

        For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}

        :param value: Json path that specifies state input to be used.

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "mapFromJsonPath", [value]))

    @jsii.member(jsii_name="numberFromString") # type: ignore[misc]
    @builtins.classmethod
    def number_from_string(cls, value: builtins.str) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Number.

        For example:  "N": "123.45"
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        Numbers may be expressed as literal strings or as JsonPath

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "numberFromString", [value]))

    @jsii.member(jsii_name="numberSetFromStrings") # type: ignore[misc]
    @builtins.classmethod
    def number_set_from_strings(
        cls,
        value: typing.Sequence[builtins.str],
    ) -> "DynamoAttributeValue":
        '''(experimental) Sets an attribute of type Number Set.

        For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
        Numbers are sent across the network to DynamoDB as strings,
        to maximize compatibility across languages and libraries.
        However, DynamoDB treats them as number type attributes for mathematical operations.

        Numbers may be expressed as literal strings or as JsonPath

        :param value: -

        :stability: experimental
        '''
        return typing.cast("DynamoAttributeValue", jsii.sinvoke(cls, "numberSetFromStrings", [value]))

    @jsii.member(jsii_name="toObject")
    def to_object(self) -> typing.Any:
        '''(experimental) Returns the DynamoDB attribute value.

        :stability: experimental
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "toObject", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attributeValue")
    def attribute_value(self) -> typing.Any:
        '''(experimental) Represents the data for the attribute.

        Data can be
        i.e. "S": "Hello"

        :stability: experimental
        '''
        return typing.cast(typing.Any, jsii.get(self, "attributeValue"))


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.DynamoConsumedCapacity")
class DynamoConsumedCapacity(enum.Enum):
    '''(experimental) Determines the level of detail about provisioned throughput consumption that is returned.

    :stability: experimental
    '''

    INDEXES = "INDEXES"
    '''(experimental) The response includes the aggregate ConsumedCapacity for the operation, together with ConsumedCapacity for each table and secondary index that was accessed.

    :stability: experimental
    '''
    TOTAL = "TOTAL"
    '''(experimental) The response includes only the aggregate ConsumedCapacity for the operation.

    :stability: experimental
    '''
    NONE = "NONE"
    '''(experimental) No ConsumedCapacity details are included in the response.

    :stability: experimental
    '''


class DynamoDeleteItem(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoDeleteItem",
):
    '''(experimental) A StepFunctions task to call DynamoDeleteItem.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoDeleteItem(self, "DeleteItem",
            key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
            table=my_table,
            result_path=sfn.JsonPath.DISCARD
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional["DynamoItemCollectionMetrics"] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param key: (experimental) Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: (experimental) The name of the table containing the requested item.
        :param condition_expression: (experimental) A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: (experimental) One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: (experimental) Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: (experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = DynamoDeleteItemProps(
            key=key,
            table=table,
            condition_expression=condition_expression,
            expression_attribute_names=expression_attribute_names,
            expression_attribute_values=expression_attribute_values,
            return_consumed_capacity=return_consumed_capacity,
            return_item_collection_metrics=return_item_collection_metrics,
            return_values=return_values,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoDeleteItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "key": "key",
        "table": "table",
        "condition_expression": "conditionExpression",
        "expression_attribute_names": "expressionAttributeNames",
        "expression_attribute_values": "expressionAttributeValues",
        "return_consumed_capacity": "returnConsumedCapacity",
        "return_item_collection_metrics": "returnItemCollectionMetrics",
        "return_values": "returnValues",
    },
)
class DynamoDeleteItemProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional["DynamoItemCollectionMetrics"] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
    ) -> None:
        '''(experimental) Properties for DynamoDeleteItem Task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param key: (experimental) Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: (experimental) The name of the table containing the requested item.
        :param condition_expression: (experimental) A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: (experimental) One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: (experimental) Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: (experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoDeleteItem(self, "DeleteItem",
                key={"MessageId": tasks.DynamoAttributeValue.from_string("message-007")},
                table=my_table,
                result_path=sfn.JsonPath.DISCARD
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "key": key,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if condition_expression is not None:
            self._values["condition_expression"] = condition_expression
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if expression_attribute_values is not None:
            self._values["expression_attribute_values"] = expression_attribute_values
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity
        if return_item_collection_metrics is not None:
            self._values["return_item_collection_metrics"] = return_item_collection_metrics
        if return_values is not None:
            self._values["return_values"] = return_values

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def key(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''(experimental) Primary key of the item to retrieve.

        For the primary key, you must provide all of the attributes.
        For example, with a simple primary key, you only need to provide a value for the partition key.
        For a composite primary key, you must provide values for both the partition key and the sort key.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
        :stability: experimental
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_24826f7e:
        '''(experimental) The name of the table containing the requested item.

        :stability: experimental
        '''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_24826f7e, result)

    @builtins.property
    def condition_expression(self) -> typing.Optional[builtins.str]:
        '''(experimental) A condition that must be satisfied in order for a conditional DeleteItem to succeed.

        :default: - No condition expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ConditionExpression
        :stability: experimental
        '''
        result = self._values.get("condition_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) One or more substitution tokens for attribute names in an expression.

        :default: - No expression attribute names

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ExpressionAttributeNames
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def expression_attribute_values(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]]:
        '''(experimental) One or more values that can be substituted in an expression.

        :default: - No expression attribute values

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ExpressionAttributeValues
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''(experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ReturnConsumedCapacity
        :stability: experimental
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    @builtins.property
    def return_item_collection_metrics(
        self,
    ) -> typing.Optional["DynamoItemCollectionMetrics"]:
        '''(experimental) Determines whether item collection metrics are returned.

        If set to SIZE, the response includes statistics about item collections, if any,
        that were modified during the operation are returned in the response.
        If set to NONE (the default), no statistics are returned.

        :default: DynamoItemCollectionMetrics.NONE

        :stability: experimental
        '''
        result = self._values.get("return_item_collection_metrics")
        return typing.cast(typing.Optional["DynamoItemCollectionMetrics"], result)

    @builtins.property
    def return_values(self) -> typing.Optional["DynamoReturnValues"]:
        '''(experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were deleted.

        :default: DynamoReturnValues.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html#DDB-DeleteItem-request-ReturnValues
        :stability: experimental
        '''
        result = self._values.get("return_values")
        return typing.cast(typing.Optional["DynamoReturnValues"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoDeleteItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DynamoGetItem(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoGetItem",
):
    '''(experimental) A StepFunctions task to call DynamoGetItem.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoGetItem(self, "Get Item",
            key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
            table=my_table
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        consistent_read: typing.Optional[builtins.bool] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        projection_expression: typing.Optional[typing.Sequence["DynamoProjectionExpression"]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param key: (experimental) Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: (experimental) The name of the table containing the requested item.
        :param consistent_read: (experimental) Determines the read consistency model: If set to true, then the operation uses strongly consistent reads; otherwise, the operation uses eventually consistent reads. Default: false
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attributes
        :param projection_expression: (experimental) An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table. These attributes can include scalars, sets, or elements of a JSON document. Default: - No projection expression
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = DynamoGetItemProps(
            key=key,
            table=table,
            consistent_read=consistent_read,
            expression_attribute_names=expression_attribute_names,
            projection_expression=projection_expression,
            return_consumed_capacity=return_consumed_capacity,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoGetItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "key": "key",
        "table": "table",
        "consistent_read": "consistentRead",
        "expression_attribute_names": "expressionAttributeNames",
        "projection_expression": "projectionExpression",
        "return_consumed_capacity": "returnConsumedCapacity",
    },
)
class DynamoGetItemProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        consistent_read: typing.Optional[builtins.bool] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        projection_expression: typing.Optional[typing.Sequence["DynamoProjectionExpression"]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
    ) -> None:
        '''(experimental) Properties for DynamoGetItem Task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param key: (experimental) Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: (experimental) The name of the table containing the requested item.
        :param consistent_read: (experimental) Determines the read consistency model: If set to true, then the operation uses strongly consistent reads; otherwise, the operation uses eventually consistent reads. Default: false
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attributes
        :param projection_expression: (experimental) An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table. These attributes can include scalars, sets, or elements of a JSON document. Default: - No projection expression
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoGetItem(self, "Get Item",
                key={"message_id": tasks.DynamoAttributeValue.from_string("message-007")},
                table=my_table
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "key": key,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if consistent_read is not None:
            self._values["consistent_read"] = consistent_read
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if projection_expression is not None:
            self._values["projection_expression"] = projection_expression
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def key(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''(experimental) Primary key of the item to retrieve.

        For the primary key, you must provide all of the attributes.
        For example, with a simple primary key, you only need to provide a value for the partition key.
        For a composite primary key, you must provide values for both the partition key and the sort key.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
        :stability: experimental
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_24826f7e:
        '''(experimental) The name of the table containing the requested item.

        :stability: experimental
        '''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_24826f7e, result)

    @builtins.property
    def consistent_read(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Determines the read consistency model: If set to true, then the operation uses strongly consistent reads;

        otherwise, the operation uses eventually consistent reads.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("consistent_read")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) One or more substitution tokens for attribute names in an expression.

        :default: - No expression attributes

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ExpressionAttributeNames
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def projection_expression(
        self,
    ) -> typing.Optional[typing.List["DynamoProjectionExpression"]]:
        '''(experimental) An array of DynamoProjectionExpression that identifies one or more attributes to retrieve from the table.

        These attributes can include scalars, sets, or elements of a JSON document.

        :default: - No projection expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ProjectionExpression
        :stability: experimental
        '''
        result = self._values.get("projection_expression")
        return typing.cast(typing.Optional[typing.List["DynamoProjectionExpression"]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''(experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-ReturnConsumedCapacity
        :stability: experimental
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoGetItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.DynamoItemCollectionMetrics")
class DynamoItemCollectionMetrics(enum.Enum):
    '''(experimental) Determines whether item collection metrics are returned.

    :stability: experimental
    '''

    SIZE = "SIZE"
    '''(experimental) If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation.

    :stability: experimental
    '''
    NONE = "NONE"
    '''(experimental) If set to NONE, no statistics are returned.

    :stability: experimental
    '''


class DynamoProjectionExpression(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoProjectionExpression",
):
    '''(experimental) Class to generate projection expression.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        dynamo_projection_expression = stepfunctions_tasks.DynamoProjectionExpression()
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="atIndex")
    def at_index(self, index: jsii.Number) -> "DynamoProjectionExpression":
        '''(experimental) Adds the array literal access for passed index.

        :param index: array index.

        :stability: experimental
        '''
        return typing.cast("DynamoProjectionExpression", jsii.invoke(self, "atIndex", [index]))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''(experimental) converts and return the string expression.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @jsii.member(jsii_name="withAttribute")
    def with_attribute(self, attr: builtins.str) -> "DynamoProjectionExpression":
        '''(experimental) Adds the passed attribute to the chain.

        :param attr: Attribute name.

        :stability: experimental
        '''
        return typing.cast("DynamoProjectionExpression", jsii.invoke(self, "withAttribute", [attr]))


class DynamoPutItem(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoPutItem",
):
    '''(experimental) A StepFunctions task to call DynamoPutItem.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoPutItem(self, "PutItem",
            item={
                "MessageId": tasks.DynamoAttributeValue.from_string("message-id")
            },
            table=my_table,
            result_path="$.Item"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        item: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param item: (experimental) A map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
        :param table: (experimental) The name of the table where the item should be written .
        :param condition_expression: (experimental) A condition that must be satisfied in order for a conditional PutItem operation to succeed. Default: - No condition expression
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: (experimental) One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: (experimental) The item collection metrics to returned in the response. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: (experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were updated with the PutItem request. Default: DynamoReturnValues.NONE
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = DynamoPutItemProps(
            item=item,
            table=table,
            condition_expression=condition_expression,
            expression_attribute_names=expression_attribute_names,
            expression_attribute_values=expression_attribute_values,
            return_consumed_capacity=return_consumed_capacity,
            return_item_collection_metrics=return_item_collection_metrics,
            return_values=return_values,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoPutItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "item": "item",
        "table": "table",
        "condition_expression": "conditionExpression",
        "expression_attribute_names": "expressionAttributeNames",
        "expression_attribute_values": "expressionAttributeValues",
        "return_consumed_capacity": "returnConsumedCapacity",
        "return_item_collection_metrics": "returnItemCollectionMetrics",
        "return_values": "returnValues",
    },
)
class DynamoPutItemProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        item: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional["DynamoReturnValues"] = None,
    ) -> None:
        '''(experimental) Properties for DynamoPutItem Task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param item: (experimental) A map of attribute name/value pairs, one for each attribute. Only the primary key attributes are required; you can optionally provide other attribute name-value pairs for the item.
        :param table: (experimental) The name of the table where the item should be written .
        :param condition_expression: (experimental) A condition that must be satisfied in order for a conditional PutItem operation to succeed. Default: - No condition expression
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: (experimental) One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: (experimental) The item collection metrics to returned in the response. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: (experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were updated with the PutItem request. Default: DynamoReturnValues.NONE

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoPutItem(self, "PutItem",
                item={
                    "MessageId": tasks.DynamoAttributeValue.from_string("message-id")
                },
                table=my_table,
                result_path="$.Item"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "item": item,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if condition_expression is not None:
            self._values["condition_expression"] = condition_expression
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if expression_attribute_values is not None:
            self._values["expression_attribute_values"] = expression_attribute_values
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity
        if return_item_collection_metrics is not None:
            self._values["return_item_collection_metrics"] = return_item_collection_metrics
        if return_values is not None:
            self._values["return_values"] = return_values

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def item(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''(experimental) A map of attribute name/value pairs, one for each attribute.

        Only the primary key attributes are required;
        you can optionally provide other attribute name-value pairs for the item.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-Item
        :stability: experimental
        '''
        result = self._values.get("item")
        assert result is not None, "Required property 'item' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_24826f7e:
        '''(experimental) The name of the table where the item should be written .

        :stability: experimental
        '''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_24826f7e, result)

    @builtins.property
    def condition_expression(self) -> typing.Optional[builtins.str]:
        '''(experimental) A condition that must be satisfied in order for a conditional PutItem operation to succeed.

        :default: - No condition expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ConditionExpression
        :stability: experimental
        '''
        result = self._values.get("condition_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) One or more substitution tokens for attribute names in an expression.

        :default: - No expression attribute names

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ExpressionAttributeNames
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def expression_attribute_values(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]]:
        '''(experimental) One or more values that can be substituted in an expression.

        :default: - No expression attribute values

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ExpressionAttributeValues
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''(experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ReturnConsumedCapacity
        :stability: experimental
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    @builtins.property
    def return_item_collection_metrics(
        self,
    ) -> typing.Optional[DynamoItemCollectionMetrics]:
        '''(experimental) The item collection metrics to returned in the response.

        :default: DynamoItemCollectionMetrics.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LSI.html#LSI.ItemCollections
        :stability: experimental
        '''
        result = self._values.get("return_item_collection_metrics")
        return typing.cast(typing.Optional[DynamoItemCollectionMetrics], result)

    @builtins.property
    def return_values(self) -> typing.Optional["DynamoReturnValues"]:
        '''(experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were updated with the PutItem request.

        :default: DynamoReturnValues.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html#DDB-PutItem-request-ReturnValues
        :stability: experimental
        '''
        result = self._values.get("return_values")
        return typing.cast(typing.Optional["DynamoReturnValues"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoPutItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.DynamoReturnValues")
class DynamoReturnValues(enum.Enum):
    '''(experimental) Use ReturnValues if you want to get the item attributes as they appear before or after they are changed.

    :stability: experimental
    '''

    NONE = "NONE"
    '''(experimental) Nothing is returned.

    :stability: experimental
    '''
    ALL_OLD = "ALL_OLD"
    '''(experimental) Returns all of the attributes of the item.

    :stability: experimental
    '''
    UPDATED_OLD = "UPDATED_OLD"
    '''(experimental) Returns only the updated attributes.

    :stability: experimental
    '''
    ALL_NEW = "ALL_NEW"
    '''(experimental) Returns all of the attributes of the item.

    :stability: experimental
    '''
    UPDATED_NEW = "UPDATED_NEW"
    '''(experimental) Returns only the updated attributes.

    :stability: experimental
    '''


class DynamoUpdateItem(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoUpdateItem",
):
    '''(experimental) A StepFunctions task to call DynamoUpdateItem.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # my_table: dynamodb.Table
        
        tasks.DynamoUpdateItem(self, "UpdateItem",
            key={
                "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
            },
            table=my_table,
            expression_attribute_values={
                ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
                ":rand": tasks.DynamoAttributeValue.from_number(20)
            },
            update_expression="SET TotalCount = :val + :rand"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional[DynamoReturnValues] = None,
        update_expression: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param key: (experimental) Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: (experimental) The name of the table containing the requested item.
        :param condition_expression: (experimental) A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: (experimental) One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: (experimental) Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: (experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE
        :param update_expression: (experimental) An expression that defines one or more attributes to be updated, the action to be performed on them, and new values for them. Default: - No update expression
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = DynamoUpdateItemProps(
            key=key,
            table=table,
            condition_expression=condition_expression,
            expression_attribute_names=expression_attribute_names,
            expression_attribute_values=expression_attribute_values,
            return_consumed_capacity=return_consumed_capacity,
            return_item_collection_metrics=return_item_collection_metrics,
            return_values=return_values,
            update_expression=update_expression,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.DynamoUpdateItemProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "key": "key",
        "table": "table",
        "condition_expression": "conditionExpression",
        "expression_attribute_names": "expressionAttributeNames",
        "expression_attribute_values": "expressionAttributeValues",
        "return_consumed_capacity": "returnConsumedCapacity",
        "return_item_collection_metrics": "returnItemCollectionMetrics",
        "return_values": "returnValues",
        "update_expression": "updateExpression",
    },
)
class DynamoUpdateItemProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        key: typing.Mapping[builtins.str, DynamoAttributeValue],
        table: _ITable_24826f7e,
        condition_expression: typing.Optional[builtins.str] = None,
        expression_attribute_names: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        expression_attribute_values: typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]] = None,
        return_consumed_capacity: typing.Optional[DynamoConsumedCapacity] = None,
        return_item_collection_metrics: typing.Optional[DynamoItemCollectionMetrics] = None,
        return_values: typing.Optional[DynamoReturnValues] = None,
        update_expression: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for DynamoUpdateItem Task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param key: (experimental) Primary key of the item to retrieve. For the primary key, you must provide all of the attributes. For example, with a simple primary key, you only need to provide a value for the partition key. For a composite primary key, you must provide values for both the partition key and the sort key.
        :param table: (experimental) The name of the table containing the requested item.
        :param condition_expression: (experimental) A condition that must be satisfied in order for a conditional DeleteItem to succeed. Default: - No condition expression
        :param expression_attribute_names: (experimental) One or more substitution tokens for attribute names in an expression. Default: - No expression attribute names
        :param expression_attribute_values: (experimental) One or more values that can be substituted in an expression. Default: - No expression attribute values
        :param return_consumed_capacity: (experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response. Default: DynamoConsumedCapacity.NONE
        :param return_item_collection_metrics: (experimental) Determines whether item collection metrics are returned. If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation are returned in the response. If set to NONE (the default), no statistics are returned. Default: DynamoItemCollectionMetrics.NONE
        :param return_values: (experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were deleted. Default: DynamoReturnValues.NONE
        :param update_expression: (experimental) An expression that defines one or more attributes to be updated, the action to be performed on them, and new values for them. Default: - No update expression

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # my_table: dynamodb.Table
            
            tasks.DynamoUpdateItem(self, "UpdateItem",
                key={
                    "MessageId": tasks.DynamoAttributeValue.from_string("message-007")
                },
                table=my_table,
                expression_attribute_values={
                    ":val": tasks.DynamoAttributeValue.number_from_string(sfn.JsonPath.string_at("$.Item.TotalCount.N")),
                    ":rand": tasks.DynamoAttributeValue.from_number(20)
                },
                update_expression="SET TotalCount = :val + :rand"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "key": key,
            "table": table,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if condition_expression is not None:
            self._values["condition_expression"] = condition_expression
        if expression_attribute_names is not None:
            self._values["expression_attribute_names"] = expression_attribute_names
        if expression_attribute_values is not None:
            self._values["expression_attribute_values"] = expression_attribute_values
        if return_consumed_capacity is not None:
            self._values["return_consumed_capacity"] = return_consumed_capacity
        if return_item_collection_metrics is not None:
            self._values["return_item_collection_metrics"] = return_item_collection_metrics
        if return_values is not None:
            self._values["return_values"] = return_values
        if update_expression is not None:
            self._values["update_expression"] = update_expression

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def key(self) -> typing.Mapping[builtins.str, DynamoAttributeValue]:
        '''(experimental) Primary key of the item to retrieve.

        For the primary key, you must provide all of the attributes.
        For example, with a simple primary key, you only need to provide a value for the partition key.
        For a composite primary key, you must provide values for both the partition key and the sort key.

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html#DDB-GetItem-request-Key
        :stability: experimental
        '''
        result = self._values.get("key")
        assert result is not None, "Required property 'key' is missing"
        return typing.cast(typing.Mapping[builtins.str, DynamoAttributeValue], result)

    @builtins.property
    def table(self) -> _ITable_24826f7e:
        '''(experimental) The name of the table containing the requested item.

        :stability: experimental
        '''
        result = self._values.get("table")
        assert result is not None, "Required property 'table' is missing"
        return typing.cast(_ITable_24826f7e, result)

    @builtins.property
    def condition_expression(self) -> typing.Optional[builtins.str]:
        '''(experimental) A condition that must be satisfied in order for a conditional DeleteItem to succeed.

        :default: - No condition expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ConditionExpression
        :stability: experimental
        '''
        result = self._values.get("condition_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def expression_attribute_names(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) One or more substitution tokens for attribute names in an expression.

        :default: - No expression attribute names

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ExpressionAttributeNames
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_names")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def expression_attribute_values(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]]:
        '''(experimental) One or more values that can be substituted in an expression.

        :default: - No expression attribute values

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ExpressionAttributeValues
        :stability: experimental
        '''
        result = self._values.get("expression_attribute_values")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, DynamoAttributeValue]], result)

    @builtins.property
    def return_consumed_capacity(self) -> typing.Optional[DynamoConsumedCapacity]:
        '''(experimental) Determines the level of detail about provisioned throughput consumption that is returned in the response.

        :default: DynamoConsumedCapacity.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ReturnConsumedCapacity
        :stability: experimental
        '''
        result = self._values.get("return_consumed_capacity")
        return typing.cast(typing.Optional[DynamoConsumedCapacity], result)

    @builtins.property
    def return_item_collection_metrics(
        self,
    ) -> typing.Optional[DynamoItemCollectionMetrics]:
        '''(experimental) Determines whether item collection metrics are returned.

        If set to SIZE, the response includes statistics about item collections, if any,
        that were modified during the operation are returned in the response.
        If set to NONE (the default), no statistics are returned.

        :default: DynamoItemCollectionMetrics.NONE

        :stability: experimental
        '''
        result = self._values.get("return_item_collection_metrics")
        return typing.cast(typing.Optional[DynamoItemCollectionMetrics], result)

    @builtins.property
    def return_values(self) -> typing.Optional[DynamoReturnValues]:
        '''(experimental) Use ReturnValues if you want to get the item attributes as they appeared before they were deleted.

        :default: DynamoReturnValues.NONE

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-ReturnValues
        :stability: experimental
        '''
        result = self._values.get("return_values")
        return typing.cast(typing.Optional[DynamoReturnValues], result)

    @builtins.property
    def update_expression(self) -> typing.Optional[builtins.str]:
        '''(experimental) An expression that defines one or more attributes to be updated, the action to be performed on them, and new values for them.

        :default: - No update expression

        :see: https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html#DDB-UpdateItem-request-UpdateExpression
        :stability: experimental
        '''
        result = self._values.get("update_expression")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DynamoUpdateItemProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsEc2LaunchTargetOptions",
    jsii_struct_bases=[],
    name_mapping={
        "placement_constraints": "placementConstraints",
        "placement_strategies": "placementStrategies",
    },
)
class EcsEc2LaunchTargetOptions:
    def __init__(
        self,
        *,
        placement_constraints: typing.Optional[typing.Sequence[_PlacementConstraint_e22ac48c]] = None,
        placement_strategies: typing.Optional[typing.Sequence[_PlacementStrategy_ea27367e]] = None,
    ) -> None:
        '''(experimental) Options to run an ECS task on EC2 in StepFunctions and ECS.

        :param placement_constraints: (experimental) Placement constraints. Default: - None
        :param placement_strategies: (experimental) Placement strategies. Default: - None

        :stability: experimental
        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc.from_lookup(self, "Vpc",
                is_default=True
            )
            
            cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
            cluster.add_capacity("DefaultAutoScalingGroup",
                instance_type=ec2.InstanceType("t2.micro"),
                vpc_subnets=dynamodb.aws_ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
            )
            
            task_definition = ecs.TaskDefinition(self, "TD",
                compatibility=ecs.Compatibility.EC2
            )
            
            task_definition.add_container("TheContainer",
                image=ecs.ContainerImage.from_registry("foo/bar"),
                memory_limit_mi_b=256
            )
            
            run_task = tasks.EcsRunTask(self, "Run",
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                cluster=cluster,
                task_definition=task_definition,
                launch_target=tasks.EcsEc2LaunchTarget(
                    placement_strategies=[
                        ecs.PlacementStrategy.spread_across_instances(),
                        ecs.PlacementStrategy.packed_by_cpu(),
                        ecs.PlacementStrategy.randomly()
                    ],
                    placement_constraints=[
                        ecs.PlacementConstraint.member_of("blieptuut")
                    ]
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if placement_strategies is not None:
            self._values["placement_strategies"] = placement_strategies

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.List[_PlacementConstraint_e22ac48c]]:
        '''(experimental) Placement constraints.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.List[_PlacementConstraint_e22ac48c]], result)

    @builtins.property
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.List[_PlacementStrategy_ea27367e]]:
        '''(experimental) Placement strategies.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("placement_strategies")
        return typing.cast(typing.Optional[typing.List[_PlacementStrategy_ea27367e]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsEc2LaunchTargetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsFargateLaunchTargetOptions",
    jsii_struct_bases=[],
    name_mapping={"platform_version": "platformVersion"},
)
class EcsFargateLaunchTargetOptions:
    def __init__(self, *, platform_version: _FargatePlatformVersion_8169c79a) -> None:
        '''(experimental) Properties to define an ECS service.

        :param platform_version: (experimental) Refers to a specific runtime environment for Fargate task infrastructure. Fargate platform version is a combination of the kernel and container runtime versions.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            ecs_fargate_launch_target_options = stepfunctions_tasks.EcsFargateLaunchTargetOptions(
                platform_version=ecs.FargatePlatformVersion.LATEST
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "platform_version": platform_version,
        }

    @builtins.property
    def platform_version(self) -> _FargatePlatformVersion_8169c79a:
        '''(experimental) Refers to a specific runtime environment for Fargate task infrastructure.

        Fargate platform version is a combination of the kernel and container runtime versions.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html
        :stability: experimental
        '''
        result = self._values.get("platform_version")
        assert result is not None, "Required property 'platform_version' is missing"
        return typing.cast(_FargatePlatformVersion_8169c79a, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsFargateLaunchTargetOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsLaunchTargetConfig",
    jsii_struct_bases=[],
    name_mapping={"parameters": "parameters"},
)
class EcsLaunchTargetConfig:
    def __init__(
        self,
        *,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''(experimental) Configuration options for the ECS launch type.

        :param parameters: (experimental) Additional parameters to pass to the base task. Default: - No additional parameters passed

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # parameters: Any
            
            ecs_launch_target_config = stepfunctions_tasks.EcsLaunchTargetConfig(
                parameters={
                    "parameters_key": parameters
                }
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) Additional parameters to pass to the base task.

        :default: - No additional parameters passed

        :stability: experimental
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsLaunchTargetConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IConnectable_c1c0e72c)
class EcsRunTask(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsRunTask",
):
    '''(experimental) Run a Task on ECS or Fargate.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        vpc = ec2.Vpc.from_lookup(self, "Vpc",
            is_default=True
        )
        
        cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
        cluster.add_capacity("DefaultAutoScalingGroup",
            instance_type=ec2.InstanceType("t2.micro"),
            vpc_subnets=dynamodb.aws_ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
        )
        
        task_definition = ecs.TaskDefinition(self, "TD",
            compatibility=ecs.Compatibility.EC2
        )
        
        task_definition.add_container("TheContainer",
            image=ecs.ContainerImage.from_registry("foo/bar"),
            memory_limit_mi_b=256
        )
        
        run_task = tasks.EcsRunTask(self, "Run",
            integration_pattern=sfn.IntegrationPattern.RUN_JOB,
            cluster=cluster,
            task_definition=task_definition,
            launch_target=tasks.EcsEc2LaunchTarget(
                placement_strategies=[
                    ecs.PlacementStrategy.spread_across_instances(),
                    ecs.PlacementStrategy.packed_by_cpu(),
                    ecs.PlacementStrategy.randomly()
                ],
                placement_constraints=[
                    ecs.PlacementConstraint.member_of("blieptuut")
                ]
            )
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster: _ICluster_42c4ec1a,
        launch_target: "IEcsLaunchTarget",
        task_definition: _TaskDefinition_c0dacfb4,
        assign_public_ip: typing.Optional[builtins.bool] = None,
        container_overrides: typing.Optional[typing.Sequence[ContainerOverride]] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_cdbba9d3]] = None,
        subnets: typing.Optional[_SubnetSelection_1284e62c] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: (experimental) The ECS cluster to run the task on.
        :param launch_target: (experimental) An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.
        :param task_definition: (experimental) [disable-awslint:ref-via-interface] Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions
        :param assign_public_ip: (experimental) Assign public IP addresses to each task. Default: false
        :param container_overrides: (experimental) Container setting overrides. Specify the container to use and the overrides to apply. Default: - No overrides
        :param security_groups: (experimental) Existing security groups to use for the tasks. Default: - A new security group is created
        :param subnets: (experimental) Subnets to place the task's ENIs. Default: - Public subnets if assignPublicIp is set. Private subnets otherwise.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EcsRunTaskProps(
            cluster=cluster,
            launch_target=launch_target,
            task_definition=task_definition,
            assign_public_ip=assign_public_ip,
            container_overrides=container_overrides,
            security_groups=security_groups,
            subnets=subnets,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connections")
    def connections(self) -> _Connections_57ccbda9:
        '''(experimental) Manage allowed network traffic for this service.

        :stability: experimental
        '''
        return typing.cast(_Connections_57ccbda9, jsii.get(self, "connections"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.implements(_IConnectable_c1c0e72c, _IStepFunctionsTask_82eb09ab)
class EcsRunTaskBase(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsRunTaskBase",
):
    '''(deprecated) A StepFunctions Task to run a Task on ECS or Fargate.

    :deprecated: No replacement

    :stability: deprecated
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        from monocdk import aws_stepfunctions as stepfunctions
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        # cluster: ecs.Cluster
        # container_definition: ecs.ContainerDefinition
        # parameters: Any
        # task_definition: ecs.TaskDefinition
        
        ecs_run_task_base = stepfunctions_tasks.EcsRunTaskBase(
            cluster=cluster,
            task_definition=task_definition,
        
            # the properties below are optional
            container_overrides=[stepfunctions_tasks.ContainerOverride(
                container_definition=container_definition,
        
                # the properties below are optional
                command=["command"],
                cpu=123,
                environment=[stepfunctions_tasks.TaskEnvironmentVariable(
                    name="name",
                    value="value"
                )],
                memory_limit=123,
                memory_reservation=123
            )],
            integration_pattern=stepfunctions.ServiceIntegrationPattern.FIRE_AND_FORGET,
            parameters={
                "parameters_key": parameters
            }
        )
    '''

    def __init__(
        self,
        *,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        cluster: _ICluster_42c4ec1a,
        task_definition: _TaskDefinition_c0dacfb4,
        container_overrides: typing.Optional[typing.Sequence[ContainerOverride]] = None,
        integration_pattern: typing.Optional[_ServiceIntegrationPattern_5581ee88] = None,
    ) -> None:
        '''
        :param parameters: (deprecated) Additional parameters to pass to the base task. Default: - No additional parameters passed
        :param cluster: (experimental) The topic to run the task on.
        :param task_definition: (experimental) Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions
        :param container_overrides: (experimental) Container setting overrides. Key is the name of the container to override, value is the values you want to override. Default: - No overrides
        :param integration_pattern: (experimental) The service integration pattern indicates different ways to call RunTask in ECS. The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN. Default: FIRE_AND_FORGET

        :stability: deprecated
        '''
        props = EcsRunTaskBaseProps(
            parameters=parameters,
            cluster=cluster,
            task_definition=task_definition,
            container_overrides=container_overrides,
            integration_pattern=integration_pattern,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="bind")
    def bind(self, task: _Task_747ca4f2) -> _StepFunctionsTaskConfig_fa131821:
        '''(deprecated) Called when the task object is used in a workflow.

        :param task: -

        :stability: deprecated
        '''
        return typing.cast(_StepFunctionsTaskConfig_fa131821, jsii.invoke(self, "bind", [task]))

    @jsii.member(jsii_name="configureAwsVpcNetworking")
    def _configure_aws_vpc_networking(
        self,
        vpc: _IVpc_6d1f76c4,
        assign_public_ip: typing.Optional[builtins.bool] = None,
        subnet_selection: typing.Optional[_SubnetSelection_1284e62c] = None,
        security_group: typing.Optional[_ISecurityGroup_cdbba9d3] = None,
    ) -> None:
        '''
        :param vpc: -
        :param assign_public_ip: -
        :param subnet_selection: -
        :param security_group: -

        :stability: deprecated
        '''
        return typing.cast(None, jsii.invoke(self, "configureAwsVpcNetworking", [vpc, assign_public_ip, subnet_selection, security_group]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connections")
    def connections(self) -> _Connections_57ccbda9:
        '''(deprecated) Manage allowed network traffic for this service.

        :stability: deprecated
        '''
        return typing.cast(_Connections_57ccbda9, jsii.get(self, "connections"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsRunTaskBaseProps",
    jsii_struct_bases=[CommonEcsRunTaskProps],
    name_mapping={
        "cluster": "cluster",
        "task_definition": "taskDefinition",
        "container_overrides": "containerOverrides",
        "integration_pattern": "integrationPattern",
        "parameters": "parameters",
    },
)
class EcsRunTaskBaseProps(CommonEcsRunTaskProps):
    def __init__(
        self,
        *,
        cluster: _ICluster_42c4ec1a,
        task_definition: _TaskDefinition_c0dacfb4,
        container_overrides: typing.Optional[typing.Sequence[ContainerOverride]] = None,
        integration_pattern: typing.Optional[_ServiceIntegrationPattern_5581ee88] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''(deprecated) Construction properties for the BaseRunTaskProps.

        :param cluster: (experimental) The topic to run the task on.
        :param task_definition: (experimental) Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions
        :param container_overrides: (experimental) Container setting overrides. Key is the name of the container to override, value is the values you want to override. Default: - No overrides
        :param integration_pattern: (experimental) The service integration pattern indicates different ways to call RunTask in ECS. The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN. Default: FIRE_AND_FORGET
        :param parameters: (deprecated) Additional parameters to pass to the base task. Default: - No additional parameters passed

        :deprecated: No replacement

        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_stepfunctions as stepfunctions
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # parameters: Any
            # task_definition: ecs.TaskDefinition
            
            ecs_run_task_base_props = stepfunctions_tasks.EcsRunTaskBaseProps(
                cluster=cluster,
                task_definition=task_definition,
            
                # the properties below are optional
                container_overrides=[stepfunctions_tasks.ContainerOverride(
                    container_definition=container_definition,
            
                    # the properties below are optional
                    command=["command"],
                    cpu=123,
                    environment=[stepfunctions_tasks.TaskEnvironmentVariable(
                        name="name",
                        value="value"
                    )],
                    memory_limit=123,
                    memory_reservation=123
                )],
                integration_pattern=stepfunctions.ServiceIntegrationPattern.FIRE_AND_FORGET,
                parameters={
                    "parameters_key": parameters
                }
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "task_definition": task_definition,
        }
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if parameters is not None:
            self._values["parameters"] = parameters

    @builtins.property
    def cluster(self) -> _ICluster_42c4ec1a:
        '''(experimental) The topic to run the task on.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_42c4ec1a, result)

    @builtins.property
    def task_definition(self) -> _TaskDefinition_c0dacfb4:
        '''(experimental) Task Definition used for running tasks in the service.

        Note: this must be TaskDefinition, and not ITaskDefinition,
        as it requires properties that are not known for imported task definitions

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(_TaskDefinition_c0dacfb4, result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[typing.List[ContainerOverride]]:
        '''(experimental) Container setting overrides.

        Key is the name of the container to override, value is the
        values you want to override.

        :default: - No overrides

        :stability: experimental
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[typing.List[ContainerOverride]], result)

    @builtins.property
    def integration_pattern(
        self,
    ) -> typing.Optional[_ServiceIntegrationPattern_5581ee88]:
        '''(experimental) The service integration pattern indicates different ways to call RunTask in ECS.

        The valid value for Lambda is FIRE_AND_FORGET, SYNC and WAIT_FOR_TASK_TOKEN.

        :default: FIRE_AND_FORGET

        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_ServiceIntegrationPattern_5581ee88], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(deprecated) Additional parameters to pass to the base task.

        :default: - No additional parameters passed

        :stability: deprecated
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsRunTaskBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EcsRunTaskProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster": "cluster",
        "launch_target": "launchTarget",
        "task_definition": "taskDefinition",
        "assign_public_ip": "assignPublicIp",
        "container_overrides": "containerOverrides",
        "security_groups": "securityGroups",
        "subnets": "subnets",
    },
)
class EcsRunTaskProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster: _ICluster_42c4ec1a,
        launch_target: "IEcsLaunchTarget",
        task_definition: _TaskDefinition_c0dacfb4,
        assign_public_ip: typing.Optional[builtins.bool] = None,
        container_overrides: typing.Optional[typing.Sequence[ContainerOverride]] = None,
        security_groups: typing.Optional[typing.Sequence[_ISecurityGroup_cdbba9d3]] = None,
        subnets: typing.Optional[_SubnetSelection_1284e62c] = None,
    ) -> None:
        '''(experimental) Properties for ECS Tasks.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster: (experimental) The ECS cluster to run the task on.
        :param launch_target: (experimental) An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.
        :param task_definition: (experimental) [disable-awslint:ref-via-interface] Task Definition used for running tasks in the service. Note: this must be TaskDefinition, and not ITaskDefinition, as it requires properties that are not known for imported task definitions
        :param assign_public_ip: (experimental) Assign public IP addresses to each task. Default: false
        :param container_overrides: (experimental) Container setting overrides. Specify the container to use and the overrides to apply. Default: - No overrides
        :param security_groups: (experimental) Existing security groups to use for the tasks. Default: - A new security group is created
        :param subnets: (experimental) Subnets to place the task's ENIs. Default: - Public subnets if assignPublicIp is set. Private subnets otherwise.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc.from_lookup(self, "Vpc",
                is_default=True
            )
            
            cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
            cluster.add_capacity("DefaultAutoScalingGroup",
                instance_type=ec2.InstanceType("t2.micro"),
                vpc_subnets=dynamodb.aws_ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
            )
            
            task_definition = ecs.TaskDefinition(self, "TD",
                compatibility=ecs.Compatibility.EC2
            )
            
            task_definition.add_container("TheContainer",
                image=ecs.ContainerImage.from_registry("foo/bar"),
                memory_limit_mi_b=256
            )
            
            run_task = tasks.EcsRunTask(self, "Run",
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                cluster=cluster,
                task_definition=task_definition,
                launch_target=tasks.EcsEc2LaunchTarget(
                    placement_strategies=[
                        ecs.PlacementStrategy.spread_across_instances(),
                        ecs.PlacementStrategy.packed_by_cpu(),
                        ecs.PlacementStrategy.randomly()
                    ],
                    placement_constraints=[
                        ecs.PlacementConstraint.member_of("blieptuut")
                    ]
                )
            )
        '''
        if isinstance(subnets, dict):
            subnets = _SubnetSelection_1284e62c(**subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "launch_target": launch_target,
            "task_definition": task_definition,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if assign_public_ip is not None:
            self._values["assign_public_ip"] = assign_public_ip
        if container_overrides is not None:
            self._values["container_overrides"] = container_overrides
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if subnets is not None:
            self._values["subnets"] = subnets

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster(self) -> _ICluster_42c4ec1a:
        '''(experimental) The ECS cluster to run the task on.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_42c4ec1a, result)

    @builtins.property
    def launch_target(self) -> "IEcsLaunchTarget":
        '''(experimental) An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.

        :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html
        :stability: experimental
        '''
        result = self._values.get("launch_target")
        assert result is not None, "Required property 'launch_target' is missing"
        return typing.cast("IEcsLaunchTarget", result)

    @builtins.property
    def task_definition(self) -> _TaskDefinition_c0dacfb4:
        '''(experimental) [disable-awslint:ref-via-interface] Task Definition used for running tasks in the service.

        Note: this must be TaskDefinition, and not ITaskDefinition,
        as it requires properties that are not known for imported task definitions

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(_TaskDefinition_c0dacfb4, result)

    @builtins.property
    def assign_public_ip(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Assign public IP addresses to each task.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("assign_public_ip")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def container_overrides(self) -> typing.Optional[typing.List[ContainerOverride]]:
        '''(experimental) Container setting overrides.

        Specify the container to use and the overrides to apply.

        :default: - No overrides

        :stability: experimental
        '''
        result = self._values.get("container_overrides")
        return typing.cast(typing.Optional[typing.List[ContainerOverride]], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[_ISecurityGroup_cdbba9d3]]:
        '''(experimental) Existing security groups to use for the tasks.

        :default: - A new security group is created

        :stability: experimental
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[_ISecurityGroup_cdbba9d3]], result)

    @builtins.property
    def subnets(self) -> typing.Optional[_SubnetSelection_1284e62c]:
        '''(experimental) Subnets to place the task's ENIs.

        :default: - Public subnets if assignPublicIp is set. Private subnets otherwise.

        :stability: experimental
        '''
        result = self._values.get("subnets")
        return typing.cast(typing.Optional[_SubnetSelection_1284e62c], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EcsRunTaskProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EksCall(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EksCall",
):
    '''(experimental) Call a EKS endpoint as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as eks
        
        
        my_eks_cluster = eks.Cluster(self, "my sample cluster",
            version=eks.KubernetesVersion.V1_18,
            cluster_name="myEksCluster"
        )
        
        tasks.EksCall(self, "Call a EKS Endpoint",
            cluster=my_eks_cluster,
            http_method=tasks.HttpMethods.GET,
            http_path="/api/v1/namespaces/default/pods"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster: _ICluster_e7e676cf,
        http_method: "HttpMethods",
        http_path: builtins.str,
        query_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Sequence[builtins.str]]] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster: (experimental) The EKS cluster.
        :param http_method: (experimental) HTTP method ("GET", "POST", "PUT", ...) part of HTTP request.
        :param http_path: (experimental) HTTP path of the Kubernetes REST API operation For example: /api/v1/namespaces/default/pods.
        :param query_parameters: (experimental) Query Parameters part of HTTP request. Default: - no query parameters
        :param request_body: (experimental) Request body part of HTTP request. Default: - No request body
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EksCallProps(
            cluster=cluster,
            http_method=http_method,
            http_path=http_path,
            query_parameters=query_parameters,
            request_body=request_body,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''(experimental) No policies are required due to eks:call is an Http service integration and does not call and EKS API directly.

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-eks.html#connect-eks-permissions
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EksCallProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster": "cluster",
        "http_method": "httpMethod",
        "http_path": "httpPath",
        "query_parameters": "queryParameters",
        "request_body": "requestBody",
    },
)
class EksCallProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster: _ICluster_e7e676cf,
        http_method: "HttpMethods",
        http_path: builtins.str,
        query_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Sequence[builtins.str]]] = None,
        request_body: typing.Optional[_TaskInput_71ab46f6] = None,
    ) -> None:
        '''(experimental) Properties for calling a EKS endpoint with EksCall.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster: (experimental) The EKS cluster.
        :param http_method: (experimental) HTTP method ("GET", "POST", "PUT", ...) part of HTTP request.
        :param http_path: (experimental) HTTP path of the Kubernetes REST API operation For example: /api/v1/namespaces/default/pods.
        :param query_parameters: (experimental) Query Parameters part of HTTP request. Default: - no query parameters
        :param request_body: (experimental) Request body part of HTTP request. Default: - No request body

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as eks
            
            
            my_eks_cluster = eks.Cluster(self, "my sample cluster",
                version=eks.KubernetesVersion.V1_18,
                cluster_name="myEksCluster"
            )
            
            tasks.EksCall(self, "Call a EKS Endpoint",
                cluster=my_eks_cluster,
                http_method=tasks.HttpMethods.GET,
                http_path="/api/v1/namespaces/default/pods"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "http_method": http_method,
            "http_path": http_path,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if request_body is not None:
            self._values["request_body"] = request_body

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster(self) -> _ICluster_e7e676cf:
        '''(experimental) The EKS cluster.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(_ICluster_e7e676cf, result)

    @builtins.property
    def http_method(self) -> "HttpMethods":
        '''(experimental) HTTP method ("GET", "POST", "PUT", ...) part of HTTP request.

        :stability: experimental
        '''
        result = self._values.get("http_method")
        assert result is not None, "Required property 'http_method' is missing"
        return typing.cast("HttpMethods", result)

    @builtins.property
    def http_path(self) -> builtins.str:
        '''(experimental) HTTP path of the Kubernetes REST API operation For example: /api/v1/namespaces/default/pods.

        :stability: experimental
        '''
        result = self._values.get("http_path")
        assert result is not None, "Required property 'http_path' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def query_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.List[builtins.str]]]:
        '''(experimental) Query Parameters part of HTTP request.

        :default: - no query parameters

        :stability: experimental
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.List[builtins.str]]], result)

    @builtins.property
    def request_body(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) Request body part of HTTP request.

        :default: - No request body

        :stability: experimental
        '''
        result = self._values.get("request_body")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EksCallProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EksClusterInput(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EksClusterInput",
):
    '''(experimental) Class that supports methods which return the EKS cluster name depending on input type.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
            eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
            eks_namespace="specified-namespace"
        )
    '''

    @jsii.member(jsii_name="fromCluster") # type: ignore[misc]
    @builtins.classmethod
    def from_cluster(cls, cluster: _ICluster_e7e676cf) -> "EksClusterInput":
        '''(experimental) Specify an existing EKS Cluster as the name for this Cluster.

        :param cluster: -

        :stability: experimental
        '''
        return typing.cast("EksClusterInput", jsii.sinvoke(cls, "fromCluster", [cluster]))

    @jsii.member(jsii_name="fromTaskInput") # type: ignore[misc]
    @builtins.classmethod
    def from_task_input(cls, task_input: _TaskInput_71ab46f6) -> "EksClusterInput":
        '''(experimental) Specify a Task Input as the name for this Cluster.

        :param task_input: -

        :stability: experimental
        '''
        return typing.cast("EksClusterInput", jsii.sinvoke(cls, "fromTaskInput", [task_input]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> builtins.str:
        '''(experimental) The name of the EKS Cluster.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "clusterName"))


class EmrAddStep(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrAddStep",
):
    '''(experimental) A Step Functions Task to add a Step to an EMR Cluster.

    The StepConfiguration is defined as Parameters in the state machine definition.

    OUTPUT: the StepId

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrAddStep(self, "Task",
            cluster_id="ClusterId",
            name="StepName",
            jar="Jar",
            action_on_failure=tasks.ActionOnFailure.CONTINUE
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        jar: builtins.str,
        name: builtins.str,
        action_on_failure: typing.Optional[ActionOnFailure] = None,
        args: typing.Optional[typing.Sequence[builtins.str]] = None,
        main_class: typing.Optional[builtins.str] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster_id: (experimental) The ClusterId to add the Step to.
        :param jar: (experimental) A path to a JAR file run during the step.
        :param name: (experimental) The name of the Step.
        :param action_on_failure: (experimental) The action to take when the cluster step fails. Default: ActionOnFailure.CONTINUE
        :param args: (experimental) A list of command line arguments passed to the JAR file's main function when executed. Default: - No args
        :param main_class: (experimental) The name of the main class in the specified Java file. If not specified, the JAR file should specify a Main-Class in its manifest file. Default: - No mainClass
        :param properties: (experimental) A list of Java properties that are set when the step runs. You can use these properties to pass key value pairs to your main function. Default: - No properties
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrAddStepProps(
            cluster_id=cluster_id,
            jar=jar,
            name=name,
            action_on_failure=action_on_failure,
            args=args,
            main_class=main_class,
            properties=properties,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrAddStepProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "jar": "jar",
        "name": "name",
        "action_on_failure": "actionOnFailure",
        "args": "args",
        "main_class": "mainClass",
        "properties": "properties",
    },
)
class EmrAddStepProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster_id: builtins.str,
        jar: builtins.str,
        name: builtins.str,
        action_on_failure: typing.Optional[ActionOnFailure] = None,
        args: typing.Optional[typing.Sequence[builtins.str]] = None,
        main_class: typing.Optional[builtins.str] = None,
        properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''(experimental) Properties for EmrAddStep.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster_id: (experimental) The ClusterId to add the Step to.
        :param jar: (experimental) A path to a JAR file run during the step.
        :param name: (experimental) The name of the Step.
        :param action_on_failure: (experimental) The action to take when the cluster step fails. Default: ActionOnFailure.CONTINUE
        :param args: (experimental) A list of command line arguments passed to the JAR file's main function when executed. Default: - No args
        :param main_class: (experimental) The name of the main class in the specified Java file. If not specified, the JAR file should specify a Main-Class in its manifest file. Default: - No mainClass
        :param properties: (experimental) A list of Java properties that are set when the step runs. You can use these properties to pass key value pairs to your main function. Default: - No properties

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrAddStep(self, "Task",
                cluster_id="ClusterId",
                name="StepName",
                jar="Jar",
                action_on_failure=tasks.ActionOnFailure.CONTINUE
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_id": cluster_id,
            "jar": jar,
            "name": name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if action_on_failure is not None:
            self._values["action_on_failure"] = action_on_failure
        if args is not None:
            self._values["args"] = args
        if main_class is not None:
            self._values["main_class"] = main_class
        if properties is not None:
            self._values["properties"] = properties

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''(experimental) The ClusterId to add the Step to.

        :stability: experimental
        '''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def jar(self) -> builtins.str:
        '''(experimental) A path to a JAR file run during the step.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        :stability: experimental
        '''
        result = self._values.get("jar")
        assert result is not None, "Required property 'jar' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) The name of the Step.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def action_on_failure(self) -> typing.Optional[ActionOnFailure]:
        '''(experimental) The action to take when the cluster step fails.

        :default: ActionOnFailure.CONTINUE

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_StepConfig.html
        :stability: experimental
        '''
        result = self._values.get("action_on_failure")
        return typing.cast(typing.Optional[ActionOnFailure], result)

    @builtins.property
    def args(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of command line arguments passed to the JAR file's main function when executed.

        :default: - No args

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        :stability: experimental
        '''
        result = self._values.get("args")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def main_class(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the main class in the specified Java file.

        If not specified, the JAR file should specify a Main-Class in its manifest file.

        :default: - No mainClass

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        :stability: experimental
        '''
        result = self._values.get("main_class")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def properties(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A list of Java properties that are set when the step runs.

        You can use these properties to pass key value pairs to your main function.

        :default: - No properties

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_HadoopJarStepConfig.html
        :stability: experimental
        '''
        result = self._values.get("properties")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrAddStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrCancelStep(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrCancelStep",
):
    '''(experimental) A Step Functions Task to to cancel a Step on an EMR Cluster.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrCancelStep(self, "Task",
            cluster_id="ClusterId",
            step_id="StepId"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        step_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster_id: (experimental) The ClusterId to update.
        :param step_id: (experimental) The StepId to cancel.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrCancelStepProps(
            cluster_id=cluster_id,
            step_id=step_id,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrCancelStepProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "step_id": "stepId",
    },
)
class EmrCancelStepProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster_id: builtins.str,
        step_id: builtins.str,
    ) -> None:
        '''(experimental) Properties for EmrCancelStep.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster_id: (experimental) The ClusterId to update.
        :param step_id: (experimental) The StepId to cancel.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrCancelStep(self, "Task",
                cluster_id="ClusterId",
                step_id="StepId"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_id": cluster_id,
            "step_id": step_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''(experimental) The ClusterId to update.

        :stability: experimental
        '''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def step_id(self) -> builtins.str:
        '''(experimental) The StepId to cancel.

        :stability: experimental
        '''
        result = self._values.get("step_id")
        assert result is not None, "Required property 'step_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrCancelStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrContainersCreateVirtualCluster(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrContainersCreateVirtualCluster",
):
    '''(experimental) Task that creates an EMR Containers virtual cluster from an EKS cluster.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
            eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
            eks_namespace="specified-namespace"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        eks_cluster: EksClusterInput,
        eks_namespace: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        virtual_cluster_name: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param eks_cluster: (experimental) EKS Cluster or task input that contains the name of the cluster.
        :param eks_namespace: (experimental) The namespace of an EKS cluster. Default: - 'default'
        :param tags: (experimental) The tags assigned to the virtual cluster. Default: {}
        :param virtual_cluster_name: (experimental) Name of the virtual cluster that will be created. Default: - the name of the state machine execution that runs this task and state name
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrContainersCreateVirtualClusterProps(
            eks_cluster=eks_cluster,
            eks_namespace=eks_namespace,
            tags=tags,
            virtual_cluster_name=virtual_cluster_name,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrContainersCreateVirtualClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "eks_cluster": "eksCluster",
        "eks_namespace": "eksNamespace",
        "tags": "tags",
        "virtual_cluster_name": "virtualClusterName",
    },
)
class EmrContainersCreateVirtualClusterProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        eks_cluster: EksClusterInput,
        eks_namespace: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        virtual_cluster_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties to define a EMR Containers CreateVirtualCluster Task on an EKS cluster.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param eks_cluster: (experimental) EKS Cluster or task input that contains the name of the cluster.
        :param eks_namespace: (experimental) The namespace of an EKS cluster. Default: - 'default'
        :param tags: (experimental) The tags assigned to the virtual cluster. Default: {}
        :param virtual_cluster_name: (experimental) Name of the virtual cluster that will be created. Default: - the name of the state machine execution that runs this task and state name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrContainersCreateVirtualCluster(self, "Create a Virtual Cluster",
                eks_cluster=tasks.EksClusterInput.from_task_input(sfn.TaskInput.from_text("clusterId")),
                eks_namespace="specified-namespace"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "eks_cluster": eks_cluster,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if eks_namespace is not None:
            self._values["eks_namespace"] = eks_namespace
        if tags is not None:
            self._values["tags"] = tags
        if virtual_cluster_name is not None:
            self._values["virtual_cluster_name"] = virtual_cluster_name

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def eks_cluster(self) -> EksClusterInput:
        '''(experimental) EKS Cluster or task input that contains the name of the cluster.

        :stability: experimental
        '''
        result = self._values.get("eks_cluster")
        assert result is not None, "Required property 'eks_cluster' is missing"
        return typing.cast(EksClusterInput, result)

    @builtins.property
    def eks_namespace(self) -> typing.Optional[builtins.str]:
        '''(experimental) The namespace of an EKS cluster.

        :default: - 'default'

        :stability: experimental
        '''
        result = self._values.get("eks_namespace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The tags assigned to the virtual cluster.

        :default: {}

        :stability: experimental
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def virtual_cluster_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of the virtual cluster that will be created.

        :default: - the name of the state machine execution that runs this task and state name

        :stability: experimental
        '''
        result = self._values.get("virtual_cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrContainersCreateVirtualClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrContainersDeleteVirtualCluster(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrContainersDeleteVirtualCluster",
):
    '''(experimental) Deletes an EMR Containers virtual cluster as a Task.

    :see: https://docs.amazonaws.cn/en_us/step-functions/latest/dg/connect-emr-eks.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
            virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        virtual_cluster_id: _TaskInput_71ab46f6,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param virtual_cluster_id: (experimental) The ID of the virtual cluster that will be deleted.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrContainersDeleteVirtualClusterProps(
            virtual_cluster_id=virtual_cluster_id,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrContainersDeleteVirtualClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "virtual_cluster_id": "virtualClusterId",
    },
)
class EmrContainersDeleteVirtualClusterProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        virtual_cluster_id: _TaskInput_71ab46f6,
    ) -> None:
        '''(experimental) Properties to define a EMR Containers DeleteVirtualCluster Task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param virtual_cluster_id: (experimental) The ID of the virtual cluster that will be deleted.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrContainersDeleteVirtualCluster(self, "Delete a Virtual Cluster",
                virtual_cluster_id=sfn.TaskInput.from_json_path_at("$.virtualCluster")
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "virtual_cluster_id": virtual_cluster_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def virtual_cluster_id(self) -> _TaskInput_71ab46f6:
        '''(experimental) The ID of the virtual cluster that will be deleted.

        :stability: experimental
        '''
        result = self._values.get("virtual_cluster_id")
        assert result is not None, "Required property 'virtual_cluster_id' is missing"
        return typing.cast(_TaskInput_71ab46f6, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrContainersDeleteVirtualClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IGrantable_4c5a91d1)
class EmrContainersStartJobRun(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrContainersStartJobRun",
):
    '''(experimental) Starts a job run.

    A job is a unit of work that you submit to Amazon EMR on EKS for execution.
    The work performed by the job can be defined by a Spark jar, PySpark script, or SparkSQL query.
    A job run is an execution of the job on the virtual cluster.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-emr-eks.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
            virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
            release_label=tasks.ReleaseLabel.EMR_6_2_0,
            job_name="EMR-Containers-Job",
            job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
                spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
                    entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                )
            ),
            application_config=[dynamodb.aws_stepfunctions_tasks.ApplicationConfiguration(
                classification=tasks.Classification.SPARK_DEFAULTS,
                properties={
                    "spark.executor.instances": "1",
                    "spark.executor.memory": "512M"
                }
            )]
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        job_driver: "JobDriver",
        release_label: "ReleaseLabel",
        virtual_cluster: "VirtualClusterInput",
        application_config: typing.Optional[typing.Sequence[ApplicationConfiguration]] = None,
        execution_role: typing.Optional[_IRole_59af6f50] = None,
        job_name: typing.Optional[builtins.str] = None,
        monitoring: typing.Optional["Monitoring"] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param job_driver: (experimental) The job driver for the job run.
        :param release_label: (experimental) The Amazon EMR release version to use for the job run.
        :param virtual_cluster: (experimental) The ID of the virtual cluster where the job will be run.
        :param application_config: (experimental) The configurations for the application running in the job run. Maximum of 100 items Default: - No application config
        :param execution_role: (experimental) The execution role for the job run. If ``virtualClusterId`` is from a JSON input path, an execution role must be provided. If an execution role is provided, follow the documentation to update the role trust policy. Default: - Automatically generated only when the provided ``virtualClusterId`` is not an encoded JSON path
        :param job_name: (experimental) The name of the job run. Default: - No job run name
        :param monitoring: (experimental) Configuration for monitoring the job run. Default: - logging enabled and resources automatically generated if ``monitoring.logging`` is set to ``true``
        :param tags: (experimental) The tags assigned to job runs. Default: - None
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrContainersStartJobRunProps(
            job_driver=job_driver,
            release_label=release_label,
            virtual_cluster=virtual_cluster,
            application_config=application_config,
            execution_role=execution_role,
            job_name=job_name,
            monitoring=monitoring,
            tags=tags,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="grantPrincipal")
    def grant_principal(self) -> _IPrincipal_93b48231:
        '''(experimental) The principal to grant permissions to.

        :stability: experimental
        '''
        return typing.cast(_IPrincipal_93b48231, jsii.get(self, "grantPrincipal"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrContainersStartJobRunProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "job_driver": "jobDriver",
        "release_label": "releaseLabel",
        "virtual_cluster": "virtualCluster",
        "application_config": "applicationConfig",
        "execution_role": "executionRole",
        "job_name": "jobName",
        "monitoring": "monitoring",
        "tags": "tags",
    },
)
class EmrContainersStartJobRunProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        job_driver: "JobDriver",
        release_label: "ReleaseLabel",
        virtual_cluster: "VirtualClusterInput",
        application_config: typing.Optional[typing.Sequence[ApplicationConfiguration]] = None,
        execution_role: typing.Optional[_IRole_59af6f50] = None,
        job_name: typing.Optional[builtins.str] = None,
        monitoring: typing.Optional["Monitoring"] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
    ) -> None:
        '''(experimental) The props for a EMR Containers StartJobRun Task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param job_driver: (experimental) The job driver for the job run.
        :param release_label: (experimental) The Amazon EMR release version to use for the job run.
        :param virtual_cluster: (experimental) The ID of the virtual cluster where the job will be run.
        :param application_config: (experimental) The configurations for the application running in the job run. Maximum of 100 items Default: - No application config
        :param execution_role: (experimental) The execution role for the job run. If ``virtualClusterId`` is from a JSON input path, an execution role must be provided. If an execution role is provided, follow the documentation to update the role trust policy. Default: - Automatically generated only when the provided ``virtualClusterId`` is not an encoded JSON path
        :param job_name: (experimental) The name of the job run. Default: - No job run name
        :param monitoring: (experimental) Configuration for monitoring the job run. Default: - logging enabled and resources automatically generated if ``monitoring.logging`` is set to ``true``
        :param tags: (experimental) The tags assigned to job runs. Default: - None

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
                virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
                release_label=tasks.ReleaseLabel.EMR_6_2_0,
                job_name="EMR-Containers-Job",
                job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
                    spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
                        entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                    )
                ),
                application_config=[dynamodb.aws_stepfunctions_tasks.ApplicationConfiguration(
                    classification=tasks.Classification.SPARK_DEFAULTS,
                    properties={
                        "spark.executor.instances": "1",
                        "spark.executor.memory": "512M"
                    }
                )]
            )
        '''
        if isinstance(job_driver, dict):
            job_driver = JobDriver(**job_driver)
        if isinstance(monitoring, dict):
            monitoring = Monitoring(**monitoring)
        self._values: typing.Dict[str, typing.Any] = {
            "job_driver": job_driver,
            "release_label": release_label,
            "virtual_cluster": virtual_cluster,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if application_config is not None:
            self._values["application_config"] = application_config
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if job_name is not None:
            self._values["job_name"] = job_name
        if monitoring is not None:
            self._values["monitoring"] = monitoring
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def job_driver(self) -> "JobDriver":
        '''(experimental) The job driver for the job run.

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_JobDriver.html
        :stability: experimental
        '''
        result = self._values.get("job_driver")
        assert result is not None, "Required property 'job_driver' is missing"
        return typing.cast("JobDriver", result)

    @builtins.property
    def release_label(self) -> "ReleaseLabel":
        '''(experimental) The Amazon EMR release version to use for the job run.

        :stability: experimental
        '''
        result = self._values.get("release_label")
        assert result is not None, "Required property 'release_label' is missing"
        return typing.cast("ReleaseLabel", result)

    @builtins.property
    def virtual_cluster(self) -> "VirtualClusterInput":
        '''(experimental) The ID of the virtual cluster where the job will be run.

        :stability: experimental
        '''
        result = self._values.get("virtual_cluster")
        assert result is not None, "Required property 'virtual_cluster' is missing"
        return typing.cast("VirtualClusterInput", result)

    @builtins.property
    def application_config(
        self,
    ) -> typing.Optional[typing.List[ApplicationConfiguration]]:
        '''(experimental) The configurations for the application running in the job run.

        Maximum of 100 items

        :default: - No application config

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_Configuration.html
        :stability: experimental
        '''
        result = self._values.get("application_config")
        return typing.cast(typing.Optional[typing.List[ApplicationConfiguration]], result)

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The execution role for the job run.

        If ``virtualClusterId`` is from a JSON input path, an execution role must be provided.
        If an execution role is provided, follow the documentation to update the role trust policy.

        :default: - Automatically generated only when the provided ``virtualClusterId`` is not an encoded JSON path

        :see: https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/setting-up-trust-policy.html
        :stability: experimental
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def job_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the job run.

        :default: - No job run name

        :stability: experimental
        '''
        result = self._values.get("job_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def monitoring(self) -> typing.Optional["Monitoring"]:
        '''(experimental) Configuration for monitoring the job run.

        :default: - logging enabled and resources automatically generated if ``monitoring.logging`` is set to ``true``

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_MonitoringConfiguration.html
        :stability: experimental
        '''
        result = self._values.get("monitoring")
        return typing.cast(typing.Optional["Monitoring"], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The tags assigned to job runs.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrContainersStartJobRunProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrCreateCluster(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster",
):
    '''(experimental) A Step Functions Task to create an EMR Cluster.

    The ClusterConfiguration is defined as Parameters in the state machine definition.

    OUTPUT: the ClusterId.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        cluster_role = iam.Role(self, "ClusterRole",
            assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
        )
        
        service_role = iam.Role(self, "ServiceRole",
            assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
        )
        
        auto_scaling_role = iam.Role(self, "AutoScalingRole",
            assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
        )
        
        auto_scaling_role.assume_role_policy.add_statements(
            iam.PolicyStatement(
                effect=iam.Effect.ALLOW,
                principals=[
                    iam.ServicePrincipal("application-autoscaling.amazonaws.com")
                ],
                actions=["sts:AssumeRole"
                ]
            ))
        
        tasks.EmrCreateCluster(self, "Create Cluster",
            instances=dynamodb.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty(),
            cluster_role=cluster_role,
            name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
            service_role=service_role,
            auto_scaling_role=auto_scaling_role
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        instances: "EmrCreateCluster.InstancesConfigProperty",
        name: builtins.str,
        additional_info: typing.Optional[builtins.str] = None,
        applications: typing.Optional[typing.Sequence["EmrCreateCluster.ApplicationConfigProperty"]] = None,
        auto_scaling_role: typing.Optional[_IRole_59af6f50] = None,
        bootstrap_actions: typing.Optional[typing.Sequence["EmrCreateCluster.BootstrapActionConfigProperty"]] = None,
        cluster_role: typing.Optional[_IRole_59af6f50] = None,
        configurations: typing.Optional[typing.Sequence["EmrCreateCluster.ConfigurationProperty"]] = None,
        custom_ami_id: typing.Optional[builtins.str] = None,
        ebs_root_volume_size: typing.Optional[_Size_7fbd4337] = None,
        kerberos_attributes: typing.Optional["EmrCreateCluster.KerberosAttributesProperty"] = None,
        log_uri: typing.Optional[builtins.str] = None,
        release_label: typing.Optional[builtins.str] = None,
        scale_down_behavior: typing.Optional["EmrCreateCluster.EmrClusterScaleDownBehavior"] = None,
        security_configuration: typing.Optional[builtins.str] = None,
        service_role: typing.Optional[_IRole_59af6f50] = None,
        step_concurrency_level: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        visible_to_all_users: typing.Optional[builtins.bool] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param instances: (experimental) A specification of the number and type of Amazon EC2 instances.
        :param name: (experimental) The Name of the Cluster.
        :param additional_info: (experimental) A JSON string for selecting additional features. Default: - None
        :param applications: (experimental) A case-insensitive list of applications for Amazon EMR to install and configure when launching the cluster. Default: - EMR selected default
        :param auto_scaling_role: (experimental) An IAM role for automatic scaling policies. Default: - A role will be created.
        :param bootstrap_actions: (experimental) A list of bootstrap actions to run before Hadoop starts on the cluster nodes. Default: - None
        :param cluster_role: (experimental) Also called instance profile and EC2 role. An IAM role for an EMR cluster. The EC2 instances of the cluster assume this role. This attribute has been renamed from jobFlowRole to clusterRole to align with other ERM/StepFunction integration parameters. Default: - - A Role will be created
        :param configurations: (experimental) The list of configurations supplied for the EMR cluster you are creating. Default: - None
        :param custom_ami_id: (experimental) The ID of a custom Amazon EBS-backed Linux AMI. Default: - None
        :param ebs_root_volume_size: (experimental) The size of the EBS root device volume of the Linux AMI that is used for each EC2 instance. Default: - EMR selected default
        :param kerberos_attributes: (experimental) Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration. Default: - None
        :param log_uri: (experimental) The location in Amazon S3 to write the log files of the job flow. Default: - None
        :param release_label: (experimental) The Amazon EMR release label, which determines the version of open-source application packages installed on the cluster. Default: - EMR selected default
        :param scale_down_behavior: (experimental) Specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized. Default: - EMR selected default
        :param security_configuration: (experimental) The name of a security configuration to apply to the cluster. Default: - None
        :param service_role: (experimental) The IAM role that will be assumed by the Amazon EMR service to access AWS resources on your behalf. Default: - A role will be created that Amazon EMR service can assume.
        :param step_concurrency_level: (experimental) Specifies the step concurrency level to allow multiple steps to run in parallel. Requires EMR release label 5.28.0 or above. Must be in range [1, 256]. Default: 1 - no step concurrency allowed
        :param tags: (experimental) A list of tags to associate with a cluster and propagate to Amazon EC2 instances. Default: - None
        :param visible_to_all_users: (experimental) A value of true indicates that all IAM users in the AWS account can perform cluster actions if they have the proper IAM policy permissions. Default: true
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrCreateClusterProps(
            instances=instances,
            name=name,
            additional_info=additional_info,
            applications=applications,
            auto_scaling_role=auto_scaling_role,
            bootstrap_actions=bootstrap_actions,
            cluster_role=cluster_role,
            configurations=configurations,
            custom_ami_id=custom_ami_id,
            ebs_root_volume_size=ebs_root_volume_size,
            kerberos_attributes=kerberos_attributes,
            log_uri=log_uri,
            release_label=release_label,
            scale_down_behavior=scale_down_behavior,
            security_configuration=security_configuration,
            service_role=service_role,
            step_concurrency_level=step_concurrency_level,
            tags=tags,
            visible_to_all_users=visible_to_all_users,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoScalingRole")
    def auto_scaling_role(self) -> _IRole_59af6f50:
        '''(experimental) The autoscaling role for the EMR Cluster.

        Only available after task has been added to a state machine.

        :stability: experimental
        '''
        return typing.cast(_IRole_59af6f50, jsii.get(self, "autoScalingRole"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterRole")
    def cluster_role(self) -> _IRole_59af6f50:
        '''(experimental) The instance role for the EMR Cluster.

        Only available after task has been added to a state machine.

        :stability: experimental
        '''
        return typing.cast(_IRole_59af6f50, jsii.get(self, "clusterRole"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceRole")
    def service_role(self) -> _IRole_59af6f50:
        '''(experimental) The service role for the EMR Cluster.

        Only available after task has been added to a state machine.

        :stability: experimental
        '''
        return typing.cast(_IRole_59af6f50, jsii.get(self, "serviceRole"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ApplicationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "additional_info": "additionalInfo",
            "args": "args",
            "version": "version",
        },
    )
    class ApplicationConfigProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            additional_info: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
            args: typing.Optional[typing.Sequence[builtins.str]] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''(experimental) Properties for the EMR Cluster Applications.

            Applies to Amazon EMR releases 4.0 and later. A case-insensitive list of applications for Amazon EMR to install and configure when launching
            the cluster.

            See the RunJobFlow API for complete documentation on input parameters

            :param name: (experimental) The name of the application.
            :param additional_info: (experimental) This option is for advanced users only. This is meta information about third-party applications that third-party vendors use for testing purposes. Default: No additionalInfo
            :param args: (experimental) Arguments for Amazon EMR to pass to the application. Default: No args
            :param version: (experimental) The version of the application. Default: No version

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_Application.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                application_config_property = stepfunctions_tasks.EmrCreateCluster.ApplicationConfigProperty(
                    name="name",
                
                    # the properties below are optional
                    additional_info={
                        "additional_info_key": "additionalInfo"
                    },
                    args=["args"],
                    version="version"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if additional_info is not None:
                self._values["additional_info"] = additional_info
            if args is not None:
                self._values["args"] = args
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def name(self) -> builtins.str:
            '''(experimental) The name of the application.

            :stability: experimental
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def additional_info(
            self,
        ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
            '''(experimental) This option is for advanced users only.

            This is meta information about third-party applications that third-party vendors use
            for testing purposes.

            :default: No additionalInfo

            :stability: experimental
            '''
            result = self._values.get("additional_info")
            return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

        @builtins.property
        def args(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) Arguments for Amazon EMR to pass to the application.

            :default: No args

            :stability: experimental
            '''
            result = self._values.get("args")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''(experimental) The version of the application.

            :default: No version

            :stability: experimental
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ApplicationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.AutoScalingPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"constraints": "constraints", "rules": "rules"},
    )
    class AutoScalingPolicyProperty:
        def __init__(
            self,
            *,
            constraints: "EmrCreateCluster.ScalingConstraintsProperty",
            rules: typing.Sequence["EmrCreateCluster.ScalingRuleProperty"],
        ) -> None:
            '''(experimental) An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.

            :param constraints: (experimental) The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity will not cause an instance group to grow above or below these limits.
            :param rules: (experimental) The scale-in and scale-out rules that comprise the automatic scaling policy.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_AutoScalingPolicy.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # duration: monocdk.Duration
                
                auto_scaling_policy_property = stepfunctions_tasks.EmrCreateCluster.AutoScalingPolicyProperty(
                    constraints=stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty(
                        max_capacity=123,
                        min_capacity=123
                    ),
                    rules=[stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty(
                        action=stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                            simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                                scaling_adjustment=123,
                
                                # the properties below are optional
                                adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                cool_down=123
                            ),
                
                            # the properties below are optional
                            market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                        ),
                        name="name",
                        trigger=stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                            cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                                comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                                metric_name="metricName",
                                period=duration,
                
                                # the properties below are optional
                                dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                                    key="key",
                                    value="value"
                                )],
                                evaluation_periods=123,
                                namespace="namespace",
                                statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                                threshold=123,
                                unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                            )
                        ),
                
                        # the properties below are optional
                        description="description"
                    )]
                )
            '''
            if isinstance(constraints, dict):
                constraints = ScalingConstraintsProperty(**constraints)
            self._values: typing.Dict[str, typing.Any] = {
                "constraints": constraints,
                "rules": rules,
            }

        @builtins.property
        def constraints(self) -> "EmrCreateCluster.ScalingConstraintsProperty":
            '''(experimental) The upper and lower EC2 instance limits for an automatic scaling policy.

            Automatic scaling activity will not cause an instance
            group to grow above or below these limits.

            :stability: experimental
            '''
            result = self._values.get("constraints")
            assert result is not None, "Required property 'constraints' is missing"
            return typing.cast("EmrCreateCluster.ScalingConstraintsProperty", result)

        @builtins.property
        def rules(self) -> typing.List["EmrCreateCluster.ScalingRuleProperty"]:
            '''(experimental) The scale-in and scale-out rules that comprise the automatic scaling policy.

            :stability: experimental
            '''
            result = self._values.get("rules")
            assert result is not None, "Required property 'rules' is missing"
            return typing.cast(typing.List["EmrCreateCluster.ScalingRuleProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AutoScalingPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.BootstrapActionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "name": "name",
            "script_bootstrap_action": "scriptBootstrapAction",
        },
    )
    class BootstrapActionConfigProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            script_bootstrap_action: "EmrCreateCluster.ScriptBootstrapActionConfigProperty",
        ) -> None:
            '''(experimental) Configuration of a bootstrap action.

            See the RunJobFlow API for complete documentation on input parameters

            :param name: (experimental) The name of the bootstrap action.
            :param script_bootstrap_action: (experimental) The script run by the bootstrap action.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_BootstrapActionConfig.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                bootstrap_action_config_property = stepfunctions_tasks.EmrCreateCluster.BootstrapActionConfigProperty(
                    name="name",
                    script_bootstrap_action=stepfunctions_tasks.EmrCreateCluster.ScriptBootstrapActionConfigProperty(
                        path="path",
                
                        # the properties below are optional
                        args=["args"]
                    )
                )
            '''
            if isinstance(script_bootstrap_action, dict):
                script_bootstrap_action = ScriptBootstrapActionConfigProperty(**script_bootstrap_action)
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
                "script_bootstrap_action": script_bootstrap_action,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''(experimental) The name of the bootstrap action.

            :stability: experimental
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def script_bootstrap_action(
            self,
        ) -> "EmrCreateCluster.ScriptBootstrapActionConfigProperty":
            '''(experimental) The script run by the bootstrap action.

            :stability: experimental
            '''
            result = self._values.get("script_bootstrap_action")
            assert result is not None, "Required property 'script_bootstrap_action' is missing"
            return typing.cast("EmrCreateCluster.ScriptBootstrapActionConfigProperty", result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BootstrapActionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator"
    )
    class CloudWatchAlarmComparisonOperator(enum.Enum):
        '''(experimental) CloudWatch Alarm Comparison Operators.

        :stability: experimental
        '''

        GREATER_THAN_OR_EQUAL = "GREATER_THAN_OR_EQUAL"
        '''(experimental) GREATER_THAN_OR_EQUAL.

        :stability: experimental
        '''
        GREATER_THAN = "GREATER_THAN"
        '''(experimental) GREATER_THAN.

        :stability: experimental
        '''
        LESS_THAN = "LESS_THAN"
        '''(experimental) LESS_THAN.

        :stability: experimental
        '''
        LESS_THAN_OR_EQUAL = "LESS_THAN_OR_EQUAL"
        '''(experimental) LESS_THAN_OR_EQUAL.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "comparison_operator": "comparisonOperator",
            "metric_name": "metricName",
            "period": "period",
            "dimensions": "dimensions",
            "evaluation_periods": "evaluationPeriods",
            "namespace": "namespace",
            "statistic": "statistic",
            "threshold": "threshold",
            "unit": "unit",
        },
    )
    class CloudWatchAlarmDefinitionProperty:
        def __init__(
            self,
            *,
            comparison_operator: "EmrCreateCluster.CloudWatchAlarmComparisonOperator",
            metric_name: builtins.str,
            period: _Duration_070aa057,
            dimensions: typing.Optional[typing.Sequence["EmrCreateCluster.MetricDimensionProperty"]] = None,
            evaluation_periods: typing.Optional[jsii.Number] = None,
            namespace: typing.Optional[builtins.str] = None,
            statistic: typing.Optional["EmrCreateCluster.CloudWatchAlarmStatistic"] = None,
            threshold: typing.Optional[jsii.Number] = None,
            unit: typing.Optional["EmrCreateCluster.CloudWatchAlarmUnit"] = None,
        ) -> None:
            '''(experimental) The definition of a CloudWatch metric alarm, which determines when an automatic scaling activity is triggered.

            When the defined alarm conditions
            are satisfied, scaling activity begins.

            :param comparison_operator: (experimental) Determines how the metric specified by MetricName is compared to the value specified by Threshold.
            :param metric_name: (experimental) The name of the CloudWatch metric that is watched to determine an alarm condition.
            :param period: (experimental) The period, in seconds, over which the statistic is applied. EMR CloudWatch metrics are emitted every five minutes (300 seconds), so if an EMR CloudWatch metric is specified, specify 300.
            :param dimensions: (experimental) A CloudWatch metric dimension. Default: - No dimensions
            :param evaluation_periods: (experimental) The number of periods, in five-minute increments, during which the alarm condition must exist before the alarm triggers automatic scaling activity. Default: 1
            :param namespace: (experimental) The namespace for the CloudWatch metric. Default: 'AWS/ElasticMapReduce'
            :param statistic: (experimental) The statistic to apply to the metric associated with the alarm. Default: CloudWatchAlarmStatistic.AVERAGE
            :param threshold: (experimental) The value against which the specified statistic is compared. Default: - None
            :param unit: (experimental) The unit of measure associated with the CloudWatch metric being watched. The value specified for Unit must correspond to the units specified in the CloudWatch metric. Default: CloudWatchAlarmUnit.NONE

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_CloudWatchAlarmDefinition.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # duration: monocdk.Duration
                
                cloud_watch_alarm_definition_property = stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                    comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                    metric_name="metricName",
                    period=duration,
                
                    # the properties below are optional
                    dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                        key="key",
                        value="value"
                    )],
                    evaluation_periods=123,
                    namespace="namespace",
                    statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                    threshold=123,
                    unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "comparison_operator": comparison_operator,
                "metric_name": metric_name,
                "period": period,
            }
            if dimensions is not None:
                self._values["dimensions"] = dimensions
            if evaluation_periods is not None:
                self._values["evaluation_periods"] = evaluation_periods
            if namespace is not None:
                self._values["namespace"] = namespace
            if statistic is not None:
                self._values["statistic"] = statistic
            if threshold is not None:
                self._values["threshold"] = threshold
            if unit is not None:
                self._values["unit"] = unit

        @builtins.property
        def comparison_operator(
            self,
        ) -> "EmrCreateCluster.CloudWatchAlarmComparisonOperator":
            '''(experimental) Determines how the metric specified by MetricName is compared to the value specified by Threshold.

            :stability: experimental
            '''
            result = self._values.get("comparison_operator")
            assert result is not None, "Required property 'comparison_operator' is missing"
            return typing.cast("EmrCreateCluster.CloudWatchAlarmComparisonOperator", result)

        @builtins.property
        def metric_name(self) -> builtins.str:
            '''(experimental) The name of the CloudWatch metric that is watched to determine an alarm condition.

            :stability: experimental
            '''
            result = self._values.get("metric_name")
            assert result is not None, "Required property 'metric_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def period(self) -> _Duration_070aa057:
            '''(experimental) The period, in seconds, over which the statistic is applied.

            EMR CloudWatch metrics are emitted every five minutes (300 seconds), so if
            an EMR CloudWatch metric is specified, specify 300.

            :stability: experimental
            '''
            result = self._values.get("period")
            assert result is not None, "Required property 'period' is missing"
            return typing.cast(_Duration_070aa057, result)

        @builtins.property
        def dimensions(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.MetricDimensionProperty"]]:
            '''(experimental) A CloudWatch metric dimension.

            :default: - No dimensions

            :stability: experimental
            '''
            result = self._values.get("dimensions")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.MetricDimensionProperty"]], result)

        @builtins.property
        def evaluation_periods(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The number of periods, in five-minute increments, during which the alarm condition must exist before the alarm triggers automatic scaling activity.

            :default: 1

            :stability: experimental
            '''
            result = self._values.get("evaluation_periods")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''(experimental) The namespace for the CloudWatch metric.

            :default: 'AWS/ElasticMapReduce'

            :stability: experimental
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def statistic(
            self,
        ) -> typing.Optional["EmrCreateCluster.CloudWatchAlarmStatistic"]:
            '''(experimental) The statistic to apply to the metric associated with the alarm.

            :default: CloudWatchAlarmStatistic.AVERAGE

            :stability: experimental
            '''
            result = self._values.get("statistic")
            return typing.cast(typing.Optional["EmrCreateCluster.CloudWatchAlarmStatistic"], result)

        @builtins.property
        def threshold(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The value against which the specified statistic is compared.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("threshold")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def unit(self) -> typing.Optional["EmrCreateCluster.CloudWatchAlarmUnit"]:
            '''(experimental) The unit of measure associated with the CloudWatch metric being watched.

            The value specified for Unit must correspond to the units
            specified in the CloudWatch metric.

            :default: CloudWatchAlarmUnit.NONE

            :stability: experimental
            '''
            result = self._values.get("unit")
            return typing.cast(typing.Optional["EmrCreateCluster.CloudWatchAlarmUnit"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CloudWatchAlarmDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic"
    )
    class CloudWatchAlarmStatistic(enum.Enum):
        '''(experimental) CloudWatch Alarm Statistics.

        :stability: experimental
        '''

        SAMPLE_COUNT = "SAMPLE_COUNT"
        '''(experimental) SAMPLE_COUNT.

        :stability: experimental
        '''
        AVERAGE = "AVERAGE"
        '''(experimental) AVERAGE.

        :stability: experimental
        '''
        SUM = "SUM"
        '''(experimental) SUM.

        :stability: experimental
        '''
        MINIMUM = "MINIMUM"
        '''(experimental) MINIMUM.

        :stability: experimental
        '''
        MAXIMUM = "MAXIMUM"
        '''(experimental) MAXIMUM.

        :stability: experimental
        '''

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit"
    )
    class CloudWatchAlarmUnit(enum.Enum):
        '''(experimental) CloudWatch Alarm Units.

        :stability: experimental
        '''

        NONE = "NONE"
        '''(experimental) NONE.

        :stability: experimental
        '''
        SECONDS = "SECONDS"
        '''(experimental) SECONDS.

        :stability: experimental
        '''
        MICRO_SECONDS = "MICRO_SECONDS"
        '''(experimental) MICRO_SECONDS.

        :stability: experimental
        '''
        MILLI_SECONDS = "MILLI_SECONDS"
        '''(experimental) MILLI_SECONDS.

        :stability: experimental
        '''
        BYTES = "BYTES"
        '''(experimental) BYTES.

        :stability: experimental
        '''
        KILO_BYTES = "KILO_BYTES"
        '''(experimental) KILO_BYTES.

        :stability: experimental
        '''
        MEGA_BYTES = "MEGA_BYTES"
        '''(experimental) MEGA_BYTES.

        :stability: experimental
        '''
        GIGA_BYTES = "GIGA_BYTES"
        '''(experimental) GIGA_BYTES.

        :stability: experimental
        '''
        TERA_BYTES = "TERA_BYTES"
        '''(experimental) TERA_BYTES.

        :stability: experimental
        '''
        BITS = "BITS"
        '''(experimental) BITS.

        :stability: experimental
        '''
        KILO_BITS = "KILO_BITS"
        '''(experimental) KILO_BITS.

        :stability: experimental
        '''
        MEGA_BITS = "MEGA_BITS"
        '''(experimental) MEGA_BITS.

        :stability: experimental
        '''
        GIGA_BITS = "GIGA_BITS"
        '''(experimental) GIGA_BITS.

        :stability: experimental
        '''
        TERA_BITS = "TERA_BITS"
        '''(experimental) TERA_BITS.

        :stability: experimental
        '''
        PERCENT = "PERCENT"
        '''(experimental) PERCENT.

        :stability: experimental
        '''
        COUNT = "COUNT"
        '''(experimental) COUNT.

        :stability: experimental
        '''
        BYTES_PER_SECOND = "BYTES_PER_SECOND"
        '''(experimental) BYTES_PER_SECOND.

        :stability: experimental
        '''
        KILO_BYTES_PER_SECOND = "KILO_BYTES_PER_SECOND"
        '''(experimental) KILO_BYTES_PER_SECOND.

        :stability: experimental
        '''
        MEGA_BYTES_PER_SECOND = "MEGA_BYTES_PER_SECOND"
        '''(experimental) MEGA_BYTES_PER_SECOND.

        :stability: experimental
        '''
        GIGA_BYTES_PER_SECOND = "GIGA_BYTES_PER_SECOND"
        '''(experimental) GIGA_BYTES_PER_SECOND.

        :stability: experimental
        '''
        TERA_BYTES_PER_SECOND = "TERA_BYTES_PER_SECOND"
        '''(experimental) TERA_BYTES_PER_SECOND.

        :stability: experimental
        '''
        BITS_PER_SECOND = "BITS_PER_SECOND"
        '''(experimental) BITS_PER_SECOND.

        :stability: experimental
        '''
        KILO_BITS_PER_SECOND = "KILO_BITS_PER_SECOND"
        '''(experimental) KILO_BITS_PER_SECOND.

        :stability: experimental
        '''
        MEGA_BITS_PER_SECOND = "MEGA_BITS_PER_SECOND"
        '''(experimental) MEGA_BITS_PER_SECOND.

        :stability: experimental
        '''
        GIGA_BITS_PER_SECOND = "GIGA_BITS_PER_SECOND"
        '''(experimental) GIGA_BITS_PER_SECOND.

        :stability: experimental
        '''
        TERA_BITS_PER_SECOND = "TERA_BITS_PER_SECOND"
        '''(experimental) TERA_BITS_PER_SECOND.

        :stability: experimental
        '''
        COUNT_PER_SECOND = "COUNT_PER_SECOND"
        '''(experimental) COUNT_PER_SECOND.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "classification": "classification",
            "configurations": "configurations",
            "properties": "properties",
        },
    )
    class ConfigurationProperty:
        def __init__(
            self,
            *,
            classification: typing.Optional[builtins.str] = None,
            configurations: typing.Optional[typing.Sequence["EmrCreateCluster.ConfigurationProperty"]] = None,
            properties: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        ) -> None:
            '''(experimental) An optional configuration specification to be used when provisioning cluster instances, which can include configurations for applications and software bundled with Amazon EMR.

            See the RunJobFlow API for complete documentation on input parameters

            :param classification: (experimental) The classification within a configuration. Default: No classification
            :param configurations: (experimental) A list of additional configurations to apply within a configuration object. Default: No configurations
            :param properties: (experimental) A set of properties specified within a configuration classification. Default: No properties

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_Configuration.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                
                configuration_property = stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                    classification="classification",
                    configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                        classification="classification",
                        configurations=[configuration_property_],
                        properties={
                            "properties_key": "properties"
                        }
                    )],
                    properties={
                        "properties_key": "properties"
                    }
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if classification is not None:
                self._values["classification"] = classification
            if configurations is not None:
                self._values["configurations"] = configurations
            if properties is not None:
                self._values["properties"] = properties

        @builtins.property
        def classification(self) -> typing.Optional[builtins.str]:
            '''(experimental) The classification within a configuration.

            :default: No classification

            :stability: experimental
            '''
            result = self._values.get("classification")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]]:
            '''(experimental) A list of additional configurations to apply within a configuration object.

            :default: No configurations

            :stability: experimental
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]], result)

        @builtins.property
        def properties(
            self,
        ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
            '''(experimental) A set of properties specified within a configuration classification.

            :default: No properties

            :stability: experimental
            '''
            result = self._values.get("properties")
            return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "volume_specification": "volumeSpecification",
            "volumes_per_instance": "volumesPerInstance",
        },
    )
    class EbsBlockDeviceConfigProperty:
        def __init__(
            self,
            *,
            volume_specification: "EmrCreateCluster.VolumeSpecificationProperty",
            volumes_per_instance: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(experimental) Configuration of requested EBS block device associated with the instance group with count of volumes that will be associated to every instance.

            :param volume_specification: (experimental) EBS volume specifications such as volume type, IOPS, and size (GiB) that will be requested for the EBS volume attached to an EC2 instance in the cluster.
            :param volumes_per_instance: (experimental) Number of EBS volumes with a specific volume configuration that will be associated with every instance in the instance group. Default: EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_EbsBlockDeviceConfig.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # size: monocdk.Size
                
                ebs_block_device_config_property = stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                    volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                        volume_size=size,
                        volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                        # the properties below are optional
                        iops=123
                    ),
                
                    # the properties below are optional
                    volumes_per_instance=123
                )
            '''
            if isinstance(volume_specification, dict):
                volume_specification = VolumeSpecificationProperty(**volume_specification)
            self._values: typing.Dict[str, typing.Any] = {
                "volume_specification": volume_specification,
            }
            if volumes_per_instance is not None:
                self._values["volumes_per_instance"] = volumes_per_instance

        @builtins.property
        def volume_specification(
            self,
        ) -> "EmrCreateCluster.VolumeSpecificationProperty":
            '''(experimental) EBS volume specifications such as volume type, IOPS, and size (GiB) that will be requested for the EBS volume attached to an EC2 instance in the cluster.

            :stability: experimental
            '''
            result = self._values.get("volume_specification")
            assert result is not None, "Required property 'volume_specification' is missing"
            return typing.cast("EmrCreateCluster.VolumeSpecificationProperty", result)

        @builtins.property
        def volumes_per_instance(self) -> typing.Optional[jsii.Number]:
            '''(experimental) Number of EBS volumes with a specific volume configuration that will be associated with every instance in the instance group.

            :default: EMR selected default

            :stability: experimental
            '''
            result = self._values.get("volumes_per_instance")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsBlockDeviceConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType"
    )
    class EbsBlockDeviceVolumeType(enum.Enum):
        '''(experimental) EBS Volume Types.

        :stability: experimental
        '''

        GP2 = "GP2"
        '''(experimental) gp2 Volume Type.

        :stability: experimental
        '''
        IO1 = "IO1"
        '''(experimental) io1 Volume Type.

        :stability: experimental
        '''
        STANDARD = "STANDARD"
        '''(experimental) Standard Volume Type.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ebs_block_device_configs": "ebsBlockDeviceConfigs",
            "ebs_optimized": "ebsOptimized",
        },
    )
    class EbsConfigurationProperty:
        def __init__(
            self,
            *,
            ebs_block_device_configs: typing.Optional[typing.Sequence["EmrCreateCluster.EbsBlockDeviceConfigProperty"]] = None,
            ebs_optimized: typing.Optional[builtins.bool] = None,
        ) -> None:
            '''(experimental) The Amazon EBS configuration of a cluster instance.

            :param ebs_block_device_configs: (experimental) An array of Amazon EBS volume specifications attached to a cluster instance. Default: - None
            :param ebs_optimized: (experimental) Indicates whether an Amazon EBS volume is EBS-optimized. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_EbsConfiguration.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # size: monocdk.Size
                
                ebs_configuration_property = stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                    ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                        volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                            volume_size=size,
                            volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                            # the properties below are optional
                            iops=123
                        ),
                
                        # the properties below are optional
                        volumes_per_instance=123
                    )],
                    ebs_optimized=False
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if ebs_block_device_configs is not None:
                self._values["ebs_block_device_configs"] = ebs_block_device_configs
            if ebs_optimized is not None:
                self._values["ebs_optimized"] = ebs_optimized

        @builtins.property
        def ebs_block_device_configs(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.EbsBlockDeviceConfigProperty"]]:
            '''(experimental) An array of Amazon EBS volume specifications attached to a cluster instance.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("ebs_block_device_configs")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.EbsBlockDeviceConfigProperty"]], result)

        @builtins.property
        def ebs_optimized(self) -> typing.Optional[builtins.bool]:
            '''(experimental) Indicates whether an Amazon EBS volume is EBS-optimized.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("ebs_optimized")
            return typing.cast(typing.Optional[builtins.bool], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.EmrClusterScaleDownBehavior"
    )
    class EmrClusterScaleDownBehavior(enum.Enum):
        '''(experimental) The Cluster ScaleDownBehavior specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html#EMR-RunJobFlow-request-ScaleDownBehavior
        :stability: experimental
        '''

        TERMINATE_AT_INSTANCE_HOUR = "TERMINATE_AT_INSTANCE_HOUR"
        '''(experimental) Indicates that Amazon EMR terminates nodes at the instance-hour boundary, regardless of when the request to terminate the instance was submitted.

        This option is only available with Amazon EMR 5.1.0 and later and is the default for clusters created using that version

        :stability: experimental
        '''
        TERMINATE_AT_TASK_COMPLETION = "TERMINATE_AT_TASK_COMPLETION"
        '''(experimental) Indicates that Amazon EMR adds nodes to a deny list and drains tasks from nodes before terminating the Amazon EC2 instances, regardless of the instance-hour boundary.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstanceFleetConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_fleet_type": "instanceFleetType",
            "instance_type_configs": "instanceTypeConfigs",
            "launch_specifications": "launchSpecifications",
            "name": "name",
            "target_on_demand_capacity": "targetOnDemandCapacity",
            "target_spot_capacity": "targetSpotCapacity",
        },
    )
    class InstanceFleetConfigProperty:
        def __init__(
            self,
            *,
            instance_fleet_type: "EmrCreateCluster.InstanceRoleType",
            instance_type_configs: typing.Optional[typing.Sequence["EmrCreateCluster.InstanceTypeConfigProperty"]] = None,
            launch_specifications: typing.Optional["EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty"] = None,
            name: typing.Optional[builtins.str] = None,
            target_on_demand_capacity: typing.Optional[jsii.Number] = None,
            target_spot_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(experimental) The configuration that defines an instance fleet.

            :param instance_fleet_type: (experimental) The node type that the instance fleet hosts. Valid values are MASTER,CORE,and TASK.
            :param instance_type_configs: (experimental) The instance type configurations that define the EC2 instances in the instance fleet. Default: No instanceTpeConfigs
            :param launch_specifications: (experimental) The launch specification for the instance fleet. Default: No launchSpecifications
            :param name: (experimental) The friendly name of the instance fleet. Default: No name
            :param target_on_demand_capacity: (experimental) The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand instances to provision. Default: No targetOnDemandCapacity
            :param target_spot_capacity: (experimental) The target capacity of Spot units for the instance fleet, which determines how many Spot instances to provision. Default: No targetSpotCapacity

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetConfig.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                # size: monocdk.Size
                
                instance_fleet_config_property = stepfunctions_tasks.EmrCreateCluster.InstanceFleetConfigProperty(
                    instance_fleet_type=stepfunctions_tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                
                    # the properties below are optional
                    instance_type_configs=[stepfunctions_tasks.EmrCreateCluster.InstanceTypeConfigProperty(
                        instance_type="instanceType",
                
                        # the properties below are optional
                        bid_price="bidPrice",
                        bid_price_as_percentage_of_on_demand_price=123,
                        configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                            classification="classification",
                            configurations=[configuration_property_],
                            properties={
                                "properties_key": "properties"
                            }
                        )],
                        ebs_configuration=stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                            ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                                volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                                    volume_size=size,
                                    volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                                    # the properties below are optional
                                    iops=123
                                ),
                
                                # the properties below are optional
                                volumes_per_instance=123
                            )],
                            ebs_optimized=False
                        ),
                        weighted_capacity=123
                    )],
                    launch_specifications=stepfunctions_tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(
                        spot_specification=stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                            timeout_action=stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction.SWITCH_TO_ON_DEMAND,
                            timeout_duration_minutes=123,
                
                            # the properties below are optional
                            allocation_strategy=stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                            block_duration_minutes=123
                        )
                    ),
                    name="name",
                    target_on_demand_capacity=123,
                    target_spot_capacity=123
                )
            '''
            if isinstance(launch_specifications, dict):
                launch_specifications = InstanceFleetProvisioningSpecificationsProperty(**launch_specifications)
            self._values: typing.Dict[str, typing.Any] = {
                "instance_fleet_type": instance_fleet_type,
            }
            if instance_type_configs is not None:
                self._values["instance_type_configs"] = instance_type_configs
            if launch_specifications is not None:
                self._values["launch_specifications"] = launch_specifications
            if name is not None:
                self._values["name"] = name
            if target_on_demand_capacity is not None:
                self._values["target_on_demand_capacity"] = target_on_demand_capacity
            if target_spot_capacity is not None:
                self._values["target_spot_capacity"] = target_spot_capacity

        @builtins.property
        def instance_fleet_type(self) -> "EmrCreateCluster.InstanceRoleType":
            '''(experimental) The node type that the instance fleet hosts.

            Valid values are MASTER,CORE,and TASK.

            :stability: experimental
            '''
            result = self._values.get("instance_fleet_type")
            assert result is not None, "Required property 'instance_fleet_type' is missing"
            return typing.cast("EmrCreateCluster.InstanceRoleType", result)

        @builtins.property
        def instance_type_configs(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.InstanceTypeConfigProperty"]]:
            '''(experimental) The instance type configurations that define the EC2 instances in the instance fleet.

            :default: No instanceTpeConfigs

            :stability: experimental
            '''
            result = self._values.get("instance_type_configs")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.InstanceTypeConfigProperty"]], result)

        @builtins.property
        def launch_specifications(
            self,
        ) -> typing.Optional["EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty"]:
            '''(experimental) The launch specification for the instance fleet.

            :default: No launchSpecifications

            :stability: experimental
            '''
            result = self._values.get("launch_specifications")
            return typing.cast(typing.Optional["EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty"], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''(experimental) The friendly name of the instance fleet.

            :default: No name

            :stability: experimental
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_on_demand_capacity(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The target capacity of On-Demand units for the instance fleet, which determines how many On-Demand instances to provision.

            :default: No targetOnDemandCapacity

            :stability: experimental
            '''
            result = self._values.get("target_on_demand_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def target_spot_capacity(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The target capacity of Spot units for the instance fleet, which determines how many Spot instances to provision.

            :default: No targetSpotCapacity

            :stability: experimental
            '''
            result = self._values.get("target_spot_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceFleetConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty",
        jsii_struct_bases=[],
        name_mapping={"spot_specification": "spotSpecification"},
    )
    class InstanceFleetProvisioningSpecificationsProperty:
        def __init__(
            self,
            *,
            spot_specification: "EmrCreateCluster.SpotProvisioningSpecificationProperty",
        ) -> None:
            '''(experimental) The launch specification for Spot instances in the fleet, which determines the defined duration and provisioning timeout behavior.

            :param spot_specification: (experimental) The launch specification for Spot instances in the fleet, which determines the defined duration and provisioning timeout behavior.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetProvisioningSpecifications.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                instance_fleet_provisioning_specifications_property = stepfunctions_tasks.EmrCreateCluster.InstanceFleetProvisioningSpecificationsProperty(
                    spot_specification=stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                        timeout_action=stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction.SWITCH_TO_ON_DEMAND,
                        timeout_duration_minutes=123,
                
                        # the properties below are optional
                        allocation_strategy=stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                        block_duration_minutes=123
                    )
                )
            '''
            if isinstance(spot_specification, dict):
                spot_specification = SpotProvisioningSpecificationProperty(**spot_specification)
            self._values: typing.Dict[str, typing.Any] = {
                "spot_specification": spot_specification,
            }

        @builtins.property
        def spot_specification(
            self,
        ) -> "EmrCreateCluster.SpotProvisioningSpecificationProperty":
            '''(experimental) The launch specification for Spot instances in the fleet, which determines the defined duration and provisioning timeout behavior.

            :stability: experimental
            '''
            result = self._values.get("spot_specification")
            assert result is not None, "Required property 'spot_specification' is missing"
            return typing.cast("EmrCreateCluster.SpotProvisioningSpecificationProperty", result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceFleetProvisioningSpecificationsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstanceGroupConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_count": "instanceCount",
            "instance_role": "instanceRole",
            "instance_type": "instanceType",
            "auto_scaling_policy": "autoScalingPolicy",
            "bid_price": "bidPrice",
            "configurations": "configurations",
            "ebs_configuration": "ebsConfiguration",
            "market": "market",
            "name": "name",
        },
    )
    class InstanceGroupConfigProperty:
        def __init__(
            self,
            *,
            instance_count: jsii.Number,
            instance_role: "EmrCreateCluster.InstanceRoleType",
            instance_type: builtins.str,
            auto_scaling_policy: typing.Optional["EmrCreateCluster.AutoScalingPolicyProperty"] = None,
            bid_price: typing.Optional[builtins.str] = None,
            configurations: typing.Optional[typing.Sequence["EmrCreateCluster.ConfigurationProperty"]] = None,
            ebs_configuration: typing.Optional["EmrCreateCluster.EbsConfigurationProperty"] = None,
            market: typing.Optional["EmrCreateCluster.InstanceMarket"] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''(experimental) Configuration defining a new instance group.

            :param instance_count: (experimental) Target number of instances for the instance group.
            :param instance_role: (experimental) The role of the instance group in the cluster.
            :param instance_type: (experimental) The EC2 instance type for all instances in the instance group.
            :param auto_scaling_policy: (experimental) An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. Default: - None
            :param bid_price: (experimental) The bid price for each EC2 Spot instance type as defined by InstanceType. Expressed in USD. Default: - None
            :param configurations: (experimental) The list of configurations supplied for an EMR cluster instance group. Default: - None
            :param ebs_configuration: (experimental) EBS configurations that will be attached to each EC2 instance in the instance group. Default: - None
            :param market: (experimental) Market type of the EC2 instances used to create a cluster node. Default: - EMR selected default
            :param name: (experimental) Friendly name given to the instance group. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceGroupConfig.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                # duration: monocdk.Duration
                # size: monocdk.Size
                
                instance_group_config_property = stepfunctions_tasks.EmrCreateCluster.InstanceGroupConfigProperty(
                    instance_count=123,
                    instance_role=stepfunctions_tasks.EmrCreateCluster.InstanceRoleType.MASTER,
                    instance_type="instanceType",
                
                    # the properties below are optional
                    auto_scaling_policy=stepfunctions_tasks.EmrCreateCluster.AutoScalingPolicyProperty(
                        constraints=stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty(
                            max_capacity=123,
                            min_capacity=123
                        ),
                        rules=[stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty(
                            action=stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                                simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                                    scaling_adjustment=123,
                
                                    # the properties below are optional
                                    adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                                    cool_down=123
                                ),
                
                                # the properties below are optional
                                market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                            ),
                            name="name",
                            trigger=stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                                cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                                    comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                                    metric_name="metricName",
                                    period=duration,
                
                                    # the properties below are optional
                                    dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                                        key="key",
                                        value="value"
                                    )],
                                    evaluation_periods=123,
                                    namespace="namespace",
                                    statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                                    threshold=123,
                                    unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                                )
                            ),
                
                            # the properties below are optional
                            description="description"
                        )]
                    ),
                    bid_price="bidPrice",
                    configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                        classification="classification",
                        configurations=[configuration_property_],
                        properties={
                            "properties_key": "properties"
                        }
                    )],
                    ebs_configuration=stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                        ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                            volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                                volume_size=size,
                                volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                                # the properties below are optional
                                iops=123
                            ),
                
                            # the properties below are optional
                            volumes_per_instance=123
                        )],
                        ebs_optimized=False
                    ),
                    market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND,
                    name="name"
                )
            '''
            if isinstance(auto_scaling_policy, dict):
                auto_scaling_policy = AutoScalingPolicyProperty(**auto_scaling_policy)
            if isinstance(ebs_configuration, dict):
                ebs_configuration = EbsConfigurationProperty(**ebs_configuration)
            self._values: typing.Dict[str, typing.Any] = {
                "instance_count": instance_count,
                "instance_role": instance_role,
                "instance_type": instance_type,
            }
            if auto_scaling_policy is not None:
                self._values["auto_scaling_policy"] = auto_scaling_policy
            if bid_price is not None:
                self._values["bid_price"] = bid_price
            if configurations is not None:
                self._values["configurations"] = configurations
            if ebs_configuration is not None:
                self._values["ebs_configuration"] = ebs_configuration
            if market is not None:
                self._values["market"] = market
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def instance_count(self) -> jsii.Number:
            '''(experimental) Target number of instances for the instance group.

            :stability: experimental
            '''
            result = self._values.get("instance_count")
            assert result is not None, "Required property 'instance_count' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def instance_role(self) -> "EmrCreateCluster.InstanceRoleType":
            '''(experimental) The role of the instance group in the cluster.

            :stability: experimental
            '''
            result = self._values.get("instance_role")
            assert result is not None, "Required property 'instance_role' is missing"
            return typing.cast("EmrCreateCluster.InstanceRoleType", result)

        @builtins.property
        def instance_type(self) -> builtins.str:
            '''(experimental) The EC2 instance type for all instances in the instance group.

            :stability: experimental
            '''
            result = self._values.get("instance_type")
            assert result is not None, "Required property 'instance_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def auto_scaling_policy(
            self,
        ) -> typing.Optional["EmrCreateCluster.AutoScalingPolicyProperty"]:
            '''(experimental) An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("auto_scaling_policy")
            return typing.cast(typing.Optional["EmrCreateCluster.AutoScalingPolicyProperty"], result)

        @builtins.property
        def bid_price(self) -> typing.Optional[builtins.str]:
            '''(experimental) The bid price for each EC2 Spot instance type as defined by InstanceType.

            Expressed in USD.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("bid_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]]:
            '''(experimental) The list of configurations supplied for an EMR cluster instance group.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]], result)

        @builtins.property
        def ebs_configuration(
            self,
        ) -> typing.Optional["EmrCreateCluster.EbsConfigurationProperty"]:
            '''(experimental) EBS configurations that will be attached to each EC2 instance in the instance group.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("ebs_configuration")
            return typing.cast(typing.Optional["EmrCreateCluster.EbsConfigurationProperty"], result)

        @builtins.property
        def market(self) -> typing.Optional["EmrCreateCluster.InstanceMarket"]:
            '''(experimental) Market type of the EC2 instances used to create a cluster node.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("market")
            return typing.cast(typing.Optional["EmrCreateCluster.InstanceMarket"], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''(experimental) Friendly name given to the instance group.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceGroupConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstanceMarket"
    )
    class InstanceMarket(enum.Enum):
        '''(experimental) EC2 Instance Market.

        :stability: experimental
        '''

        ON_DEMAND = "ON_DEMAND"
        '''(experimental) On Demand Instance.

        :stability: experimental
        '''
        SPOT = "SPOT"
        '''(experimental) Spot Instance.

        :stability: experimental
        '''

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstanceRoleType"
    )
    class InstanceRoleType(enum.Enum):
        '''(experimental) Instance Role Types.

        :stability: experimental
        '''

        MASTER = "MASTER"
        '''(experimental) Master Node.

        :stability: experimental
        '''
        CORE = "CORE"
        '''(experimental) Core Node.

        :stability: experimental
        '''
        TASK = "TASK"
        '''(experimental) Task Node.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstanceTypeConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_type": "instanceType",
            "bid_price": "bidPrice",
            "bid_price_as_percentage_of_on_demand_price": "bidPriceAsPercentageOfOnDemandPrice",
            "configurations": "configurations",
            "ebs_configuration": "ebsConfiguration",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class InstanceTypeConfigProperty:
        def __init__(
            self,
            *,
            instance_type: builtins.str,
            bid_price: typing.Optional[builtins.str] = None,
            bid_price_as_percentage_of_on_demand_price: typing.Optional[jsii.Number] = None,
            configurations: typing.Optional[typing.Sequence["EmrCreateCluster.ConfigurationProperty"]] = None,
            ebs_configuration: typing.Optional["EmrCreateCluster.EbsConfigurationProperty"] = None,
            weighted_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(experimental) An instance type configuration for each instance type in an instance fleet, which determines the EC2 instances Amazon EMR attempts to provision to fulfill On-Demand and Spot target capacities.

            :param instance_type: (experimental) An EC2 instance type.
            :param bid_price: (experimental) The bid price for each EC2 Spot instance type as defined by InstanceType. Expressed in USD. Default: - None
            :param bid_price_as_percentage_of_on_demand_price: (experimental) The bid price, as a percentage of On-Demand price. Default: - None
            :param configurations: (experimental) A configuration classification that applies when provisioning cluster instances, which can include configurations for applications and software that run on the cluster. Default: - None
            :param ebs_configuration: (experimental) The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by InstanceType. Default: - None
            :param weighted_capacity: (experimental) The number of units that a provisioned instance of this type provides toward fulfilling the target capacities defined in the InstanceFleetConfig. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceTypeConfig.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # configuration_property_: stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty
                # size: monocdk.Size
                
                instance_type_config_property = stepfunctions_tasks.EmrCreateCluster.InstanceTypeConfigProperty(
                    instance_type="instanceType",
                
                    # the properties below are optional
                    bid_price="bidPrice",
                    bid_price_as_percentage_of_on_demand_price=123,
                    configurations=[stepfunctions_tasks.EmrCreateCluster.ConfigurationProperty(
                        classification="classification",
                        configurations=[configuration_property_],
                        properties={
                            "properties_key": "properties"
                        }
                    )],
                    ebs_configuration=stepfunctions_tasks.EmrCreateCluster.EbsConfigurationProperty(
                        ebs_block_device_configs=[stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceConfigProperty(
                            volume_specification=stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                                volume_size=size,
                                volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                                # the properties below are optional
                                iops=123
                            ),
                
                            # the properties below are optional
                            volumes_per_instance=123
                        )],
                        ebs_optimized=False
                    ),
                    weighted_capacity=123
                )
            '''
            if isinstance(ebs_configuration, dict):
                ebs_configuration = EbsConfigurationProperty(**ebs_configuration)
            self._values: typing.Dict[str, typing.Any] = {
                "instance_type": instance_type,
            }
            if bid_price is not None:
                self._values["bid_price"] = bid_price
            if bid_price_as_percentage_of_on_demand_price is not None:
                self._values["bid_price_as_percentage_of_on_demand_price"] = bid_price_as_percentage_of_on_demand_price
            if configurations is not None:
                self._values["configurations"] = configurations
            if ebs_configuration is not None:
                self._values["ebs_configuration"] = ebs_configuration
            if weighted_capacity is not None:
                self._values["weighted_capacity"] = weighted_capacity

        @builtins.property
        def instance_type(self) -> builtins.str:
            '''(experimental) An EC2 instance type.

            :stability: experimental
            '''
            result = self._values.get("instance_type")
            assert result is not None, "Required property 'instance_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def bid_price(self) -> typing.Optional[builtins.str]:
            '''(experimental) The bid price for each EC2 Spot instance type as defined by InstanceType.

            Expressed in USD.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("bid_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def bid_price_as_percentage_of_on_demand_price(
            self,
        ) -> typing.Optional[jsii.Number]:
            '''(experimental) The bid price, as a percentage of On-Demand price.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("bid_price_as_percentage_of_on_demand_price")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]]:
            '''(experimental) A configuration classification that applies when provisioning cluster instances, which can include configurations for applications and software that run on the cluster.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.ConfigurationProperty"]], result)

        @builtins.property
        def ebs_configuration(
            self,
        ) -> typing.Optional["EmrCreateCluster.EbsConfigurationProperty"]:
            '''(experimental) The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by InstanceType.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("ebs_configuration")
            return typing.cast(typing.Optional["EmrCreateCluster.EbsConfigurationProperty"], result)

        @builtins.property
        def weighted_capacity(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The number of units that a provisioned instance of this type provides toward fulfilling the target capacities defined in the InstanceFleetConfig.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("weighted_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceTypeConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "additional_master_security_groups": "additionalMasterSecurityGroups",
            "additional_slave_security_groups": "additionalSlaveSecurityGroups",
            "ec2_key_name": "ec2KeyName",
            "ec2_subnet_id": "ec2SubnetId",
            "ec2_subnet_ids": "ec2SubnetIds",
            "emr_managed_master_security_group": "emrManagedMasterSecurityGroup",
            "emr_managed_slave_security_group": "emrManagedSlaveSecurityGroup",
            "hadoop_version": "hadoopVersion",
            "instance_count": "instanceCount",
            "instance_fleets": "instanceFleets",
            "instance_groups": "instanceGroups",
            "master_instance_type": "masterInstanceType",
            "placement": "placement",
            "service_access_security_group": "serviceAccessSecurityGroup",
            "slave_instance_type": "slaveInstanceType",
            "termination_protected": "terminationProtected",
        },
    )
    class InstancesConfigProperty:
        def __init__(
            self,
            *,
            additional_master_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            additional_slave_security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            ec2_key_name: typing.Optional[builtins.str] = None,
            ec2_subnet_id: typing.Optional[builtins.str] = None,
            ec2_subnet_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
            emr_managed_master_security_group: typing.Optional[builtins.str] = None,
            emr_managed_slave_security_group: typing.Optional[builtins.str] = None,
            hadoop_version: typing.Optional[builtins.str] = None,
            instance_count: typing.Optional[jsii.Number] = None,
            instance_fleets: typing.Optional[typing.Sequence["EmrCreateCluster.InstanceFleetConfigProperty"]] = None,
            instance_groups: typing.Optional[typing.Sequence["EmrCreateCluster.InstanceGroupConfigProperty"]] = None,
            master_instance_type: typing.Optional[builtins.str] = None,
            placement: typing.Optional["EmrCreateCluster.PlacementTypeProperty"] = None,
            service_access_security_group: typing.Optional[builtins.str] = None,
            slave_instance_type: typing.Optional[builtins.str] = None,
            termination_protected: typing.Optional[builtins.bool] = None,
        ) -> None:
            '''(experimental) A specification of the number and type of Amazon EC2 instances.

            See the RunJobFlow API for complete documentation on input parameters

            :param additional_master_security_groups: (experimental) A list of additional Amazon EC2 security group IDs for the master node. Default: - None
            :param additional_slave_security_groups: (experimental) A list of additional Amazon EC2 security group IDs for the core and task nodes. Default: - None
            :param ec2_key_name: (experimental) The name of the EC2 key pair that can be used to ssh to the master node as the user called "hadoop.". Default: - None
            :param ec2_subnet_id: (experimental) Applies to clusters that use the uniform instance group configuration. To launch the cluster in Amazon Virtual Private Cloud (Amazon VPC), set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch. Default: EMR selected default
            :param ec2_subnet_ids: (experimental) Applies to clusters that use the instance fleet configuration. When multiple EC2 subnet IDs are specified, Amazon EMR evaluates them and launches instances in the optimal subnet. Default: EMR selected default
            :param emr_managed_master_security_group: (experimental) The identifier of the Amazon EC2 security group for the master node. Default: - None
            :param emr_managed_slave_security_group: (experimental) The identifier of the Amazon EC2 security group for the core and task nodes. Default: - None
            :param hadoop_version: (experimental) Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster. Default: - 0.18 if the AmiVersion parameter is not set. If AmiVersion is set, the version of Hadoop for that AMI version is used.
            :param instance_count: (experimental) The number of EC2 instances in the cluster. Default: 0
            :param instance_fleets: (experimental) Describes the EC2 instances and instance configurations for clusters that use the instance fleet configuration. The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions. Default: - None
            :param instance_groups: (experimental) Configuration for the instance groups in a cluster. Default: - None
            :param master_instance_type: (experimental) The EC2 instance type of the master node. Default: - None
            :param placement: (experimental) The Availability Zone in which the cluster runs. Default: - EMR selected default
            :param service_access_security_group: (experimental) The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private subnets. Default: - None
            :param slave_instance_type: (experimental) The EC2 instance type of the core and task nodes. Default: - None
            :param termination_protected: (experimental) Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API call, user intervention, or in the event of a job-flow error. Default: false

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_JobFlowInstancesConfig.html
            :stability: experimental
            :exampleMetadata: infused

            Example::

                cluster_role = iam.Role(self, "ClusterRole",
                    assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
                )
                
                service_role = iam.Role(self, "ServiceRole",
                    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
                )
                
                auto_scaling_role = iam.Role(self, "AutoScalingRole",
                    assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
                )
                
                auto_scaling_role.assume_role_policy.add_statements(
                    iam.PolicyStatement(
                        effect=iam.Effect.ALLOW,
                        principals=[
                            iam.ServicePrincipal("application-autoscaling.amazonaws.com")
                        ],
                        actions=["sts:AssumeRole"
                        ]
                    ))
                
                tasks.EmrCreateCluster(self, "Create Cluster",
                    instances=dynamodb.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty(),
                    cluster_role=cluster_role,
                    name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
                    service_role=service_role,
                    auto_scaling_role=auto_scaling_role
                )
            '''
            if isinstance(placement, dict):
                placement = PlacementTypeProperty(**placement)
            self._values: typing.Dict[str, typing.Any] = {}
            if additional_master_security_groups is not None:
                self._values["additional_master_security_groups"] = additional_master_security_groups
            if additional_slave_security_groups is not None:
                self._values["additional_slave_security_groups"] = additional_slave_security_groups
            if ec2_key_name is not None:
                self._values["ec2_key_name"] = ec2_key_name
            if ec2_subnet_id is not None:
                self._values["ec2_subnet_id"] = ec2_subnet_id
            if ec2_subnet_ids is not None:
                self._values["ec2_subnet_ids"] = ec2_subnet_ids
            if emr_managed_master_security_group is not None:
                self._values["emr_managed_master_security_group"] = emr_managed_master_security_group
            if emr_managed_slave_security_group is not None:
                self._values["emr_managed_slave_security_group"] = emr_managed_slave_security_group
            if hadoop_version is not None:
                self._values["hadoop_version"] = hadoop_version
            if instance_count is not None:
                self._values["instance_count"] = instance_count
            if instance_fleets is not None:
                self._values["instance_fleets"] = instance_fleets
            if instance_groups is not None:
                self._values["instance_groups"] = instance_groups
            if master_instance_type is not None:
                self._values["master_instance_type"] = master_instance_type
            if placement is not None:
                self._values["placement"] = placement
            if service_access_security_group is not None:
                self._values["service_access_security_group"] = service_access_security_group
            if slave_instance_type is not None:
                self._values["slave_instance_type"] = slave_instance_type
            if termination_protected is not None:
                self._values["termination_protected"] = termination_protected

        @builtins.property
        def additional_master_security_groups(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) A list of additional Amazon EC2 security group IDs for the master node.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("additional_master_security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def additional_slave_security_groups(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) A list of additional Amazon EC2 security group IDs for the core and task nodes.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("additional_slave_security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def ec2_key_name(self) -> typing.Optional[builtins.str]:
            '''(experimental) The name of the EC2 key pair that can be used to ssh to the master node as the user called "hadoop.".

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("ec2_key_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_subnet_id(self) -> typing.Optional[builtins.str]:
            '''(experimental) Applies to clusters that use the uniform instance group configuration.

            To launch the cluster in Amazon Virtual Private Cloud (Amazon VPC),
            set this parameter to the identifier of the Amazon VPC subnet where you want the cluster to launch.

            :default: EMR selected default

            :stability: experimental
            '''
            result = self._values.get("ec2_subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ec2_subnet_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) Applies to clusters that use the instance fleet configuration.

            When multiple EC2 subnet IDs are specified, Amazon EMR evaluates them and
            launches instances in the optimal subnet.

            :default: EMR selected default

            :stability: experimental
            '''
            result = self._values.get("ec2_subnet_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def emr_managed_master_security_group(self) -> typing.Optional[builtins.str]:
            '''(experimental) The identifier of the Amazon EC2 security group for the master node.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("emr_managed_master_security_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def emr_managed_slave_security_group(self) -> typing.Optional[builtins.str]:
            '''(experimental) The identifier of the Amazon EC2 security group for the core and task nodes.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("emr_managed_slave_security_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def hadoop_version(self) -> typing.Optional[builtins.str]:
            '''(experimental) Applies only to Amazon EMR release versions earlier than 4.0. The Hadoop version for the cluster.

            :default: - 0.18 if the AmiVersion parameter is not set. If AmiVersion is set, the version of Hadoop for that AMI version is used.

            :stability: experimental
            '''
            result = self._values.get("hadoop_version")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_count(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The number of EC2 instances in the cluster.

            :default: 0

            :stability: experimental
            '''
            result = self._values.get("instance_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def instance_fleets(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.InstanceFleetConfigProperty"]]:
            '''(experimental) Describes the EC2 instances and instance configurations for clusters that use the instance fleet configuration.

            The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("instance_fleets")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.InstanceFleetConfigProperty"]], result)

        @builtins.property
        def instance_groups(
            self,
        ) -> typing.Optional[typing.List["EmrCreateCluster.InstanceGroupConfigProperty"]]:
            '''(experimental) Configuration for the instance groups in a cluster.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("instance_groups")
            return typing.cast(typing.Optional[typing.List["EmrCreateCluster.InstanceGroupConfigProperty"]], result)

        @builtins.property
        def master_instance_type(self) -> typing.Optional[builtins.str]:
            '''(experimental) The EC2 instance type of the master node.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("master_instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def placement(
            self,
        ) -> typing.Optional["EmrCreateCluster.PlacementTypeProperty"]:
            '''(experimental) The Availability Zone in which the cluster runs.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("placement")
            return typing.cast(typing.Optional["EmrCreateCluster.PlacementTypeProperty"], result)

        @builtins.property
        def service_access_security_group(self) -> typing.Optional[builtins.str]:
            '''(experimental) The identifier of the Amazon EC2 security group for the Amazon EMR service to access clusters in VPC private subnets.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("service_access_security_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def slave_instance_type(self) -> typing.Optional[builtins.str]:
            '''(experimental) The EC2 instance type of the core and task nodes.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("slave_instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def termination_protected(self) -> typing.Optional[builtins.bool]:
            '''(experimental) Specifies whether to lock the cluster to prevent the Amazon EC2 instances from being terminated by API call, user intervention, or in the event of a job-flow error.

            :default: false

            :stability: experimental
            '''
            result = self._values.get("termination_protected")
            return typing.cast(typing.Optional[builtins.bool], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstancesConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.KerberosAttributesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "realm": "realm",
            "ad_domain_join_password": "adDomainJoinPassword",
            "ad_domain_join_user": "adDomainJoinUser",
            "cross_realm_trust_principal_password": "crossRealmTrustPrincipalPassword",
            "kdc_admin_password": "kdcAdminPassword",
        },
    )
    class KerberosAttributesProperty:
        def __init__(
            self,
            *,
            realm: builtins.str,
            ad_domain_join_password: typing.Optional[builtins.str] = None,
            ad_domain_join_user: typing.Optional[builtins.str] = None,
            cross_realm_trust_principal_password: typing.Optional[builtins.str] = None,
            kdc_admin_password: typing.Optional[builtins.str] = None,
        ) -> None:
            '''(experimental) Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration.

            See the RunJobFlow API for complete documentation on input parameters

            :param realm: (experimental) The name of the Kerberos realm to which all nodes in a cluster belong. For example, EC2.INTERNAL.
            :param ad_domain_join_password: (experimental) The Active Directory password for ADDomainJoinUser. Default: No adDomainJoinPassword
            :param ad_domain_join_user: (experimental) Required only when establishing a cross-realm trust with an Active Directory domain. A user with sufficient privileges to join resources to the domain. Default: No adDomainJoinUser
            :param cross_realm_trust_principal_password: (experimental) Required only when establishing a cross-realm trust with a KDC in a different realm. The cross-realm principal password, which must be identical across realms. Default: No crossRealmTrustPrincipalPassword
            :param kdc_admin_password: (experimental) The password used within the cluster for the kadmin service on the cluster-dedicated KDC, which maintains Kerberos principals, password policies, and keytabs for the cluster. Default: No kdcAdminPassword

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_KerberosAttributes.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                kerberos_attributes_property = stepfunctions_tasks.EmrCreateCluster.KerberosAttributesProperty(
                    realm="realm",
                
                    # the properties below are optional
                    ad_domain_join_password="adDomainJoinPassword",
                    ad_domain_join_user="adDomainJoinUser",
                    cross_realm_trust_principal_password="crossRealmTrustPrincipalPassword",
                    kdc_admin_password="kdcAdminPassword"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "realm": realm,
            }
            if ad_domain_join_password is not None:
                self._values["ad_domain_join_password"] = ad_domain_join_password
            if ad_domain_join_user is not None:
                self._values["ad_domain_join_user"] = ad_domain_join_user
            if cross_realm_trust_principal_password is not None:
                self._values["cross_realm_trust_principal_password"] = cross_realm_trust_principal_password
            if kdc_admin_password is not None:
                self._values["kdc_admin_password"] = kdc_admin_password

        @builtins.property
        def realm(self) -> builtins.str:
            '''(experimental) The name of the Kerberos realm to which all nodes in a cluster belong.

            For example, EC2.INTERNAL.

            :stability: experimental
            '''
            result = self._values.get("realm")
            assert result is not None, "Required property 'realm' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ad_domain_join_password(self) -> typing.Optional[builtins.str]:
            '''(experimental) The Active Directory password for ADDomainJoinUser.

            :default: No adDomainJoinPassword

            :stability: experimental
            '''
            result = self._values.get("ad_domain_join_password")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ad_domain_join_user(self) -> typing.Optional[builtins.str]:
            '''(experimental) Required only when establishing a cross-realm trust with an Active Directory domain.

            A user with sufficient privileges to join
            resources to the domain.

            :default: No adDomainJoinUser

            :stability: experimental
            '''
            result = self._values.get("ad_domain_join_user")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cross_realm_trust_principal_password(self) -> typing.Optional[builtins.str]:
            '''(experimental) Required only when establishing a cross-realm trust with a KDC in a different realm.

            The cross-realm principal password, which
            must be identical across realms.

            :default: No crossRealmTrustPrincipalPassword

            :stability: experimental
            '''
            result = self._values.get("cross_realm_trust_principal_password")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def kdc_admin_password(self) -> typing.Optional[builtins.str]:
            '''(experimental) The password used within the cluster for the kadmin service on the cluster-dedicated KDC, which maintains Kerberos principals, password policies, and keytabs for the cluster.

            :default: No kdcAdminPassword

            :stability: experimental
            '''
            result = self._values.get("kdc_admin_password")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KerberosAttributesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class MetricDimensionProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''(experimental) A CloudWatch dimension, which is specified using a Key (known as a Name in CloudWatch), Value pair.

            By default, Amazon EMR uses
            one dimension whose Key is JobFlowID and Value is a variable representing the cluster ID, which is ${emr.clusterId}. This enables
            the rule to bootstrap when the cluster ID becomes available

            :param key: (experimental) The dimension name.
            :param value: (experimental) The dimension value.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_MetricDimension.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                metric_dimension_property = stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                    key="key",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''(experimental) The dimension name.

            :stability: experimental
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''(experimental) The dimension value.

            :stability: experimental
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetricDimensionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.PlacementTypeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "availability_zones": "availabilityZones",
        },
    )
    class PlacementTypeProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''(experimental) The Amazon EC2 Availability Zone configuration of the cluster (job flow).

            :param availability_zone: (experimental) The Amazon EC2 Availability Zone for the cluster. AvailabilityZone is used for uniform instance groups, while AvailabilityZones (plural) is used for instance fleets. Default: - EMR selected default
            :param availability_zones: (experimental) When multiple Availability Zones are specified, Amazon EMR evaluates them and launches instances in the optimal Availability Zone. AvailabilityZones is used for instance fleets, while AvailabilityZone (singular) is used for uniform instance groups. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_PlacementType.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                placement_type_property = stepfunctions_tasks.EmrCreateCluster.PlacementTypeProperty(
                    availability_zone="availabilityZone",
                    availability_zones=["availabilityZones"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if availability_zones is not None:
                self._values["availability_zones"] = availability_zones

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''(experimental) The Amazon EC2 Availability Zone for the cluster.

            AvailabilityZone is used for uniform instance groups, while AvailabilityZones
            (plural) is used for instance fleets.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) When multiple Availability Zones are specified, Amazon EMR evaluates them and launches instances in the optimal Availability Zone.

            AvailabilityZones is used for instance fleets, while AvailabilityZone (singular) is used for uniform instance groups.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("availability_zones")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementTypeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "simple_scaling_policy_configuration": "simpleScalingPolicyConfiguration",
            "market": "market",
        },
    )
    class ScalingActionProperty:
        def __init__(
            self,
            *,
            simple_scaling_policy_configuration: "EmrCreateCluster.SimpleScalingPolicyConfigurationProperty",
            market: typing.Optional["EmrCreateCluster.InstanceMarket"] = None,
        ) -> None:
            '''(experimental) The type of adjustment the automatic scaling activity makes when triggered, and the periodicity of the adjustment.

            And an automatic scaling configuration, which describes how the policy adds or removes instances, the cooldown period,
            and the number of EC2 instances that will be added each time the CloudWatch metric alarm condition is satisfied.

            :param simple_scaling_policy_configuration: (experimental) The type of adjustment the automatic scaling activity makes when triggered, and the periodicity of the adjustment.
            :param market: (experimental) Not available for instance groups. Instance groups use the market type specified for the group. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingAction.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                scaling_action_property = stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                    simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                        scaling_adjustment=123,
                
                        # the properties below are optional
                        adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                        cool_down=123
                    ),
                
                    # the properties below are optional
                    market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                )
            '''
            if isinstance(simple_scaling_policy_configuration, dict):
                simple_scaling_policy_configuration = SimpleScalingPolicyConfigurationProperty(**simple_scaling_policy_configuration)
            self._values: typing.Dict[str, typing.Any] = {
                "simple_scaling_policy_configuration": simple_scaling_policy_configuration,
            }
            if market is not None:
                self._values["market"] = market

        @builtins.property
        def simple_scaling_policy_configuration(
            self,
        ) -> "EmrCreateCluster.SimpleScalingPolicyConfigurationProperty":
            '''(experimental) The type of adjustment the automatic scaling activity makes when triggered, and the periodicity of the adjustment.

            :stability: experimental
            '''
            result = self._values.get("simple_scaling_policy_configuration")
            assert result is not None, "Required property 'simple_scaling_policy_configuration' is missing"
            return typing.cast("EmrCreateCluster.SimpleScalingPolicyConfigurationProperty", result)

        @builtins.property
        def market(self) -> typing.Optional["EmrCreateCluster.InstanceMarket"]:
            '''(experimental) Not available for instance groups.

            Instance groups use the market type specified for the group.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("market")
            return typing.cast(typing.Optional["EmrCreateCluster.InstanceMarket"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType"
    )
    class ScalingAdjustmentType(enum.Enum):
        '''(experimental) AutoScaling Adjustment Type.

        :stability: experimental
        '''

        CHANGE_IN_CAPACITY = "CHANGE_IN_CAPACITY"
        '''(experimental) CHANGE_IN_CAPACITY.

        :stability: experimental
        '''
        PERCENT_CHANGE_IN_CAPACITY = "PERCENT_CHANGE_IN_CAPACITY"
        '''(experimental) PERCENT_CHANGE_IN_CAPACITY.

        :stability: experimental
        '''
        EXACT_CAPACITY = "EXACT_CAPACITY"
        '''(experimental) EXACT_CAPACITY.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty",
        jsii_struct_bases=[],
        name_mapping={"max_capacity": "maxCapacity", "min_capacity": "minCapacity"},
    )
    class ScalingConstraintsProperty:
        def __init__(
            self,
            *,
            max_capacity: jsii.Number,
            min_capacity: jsii.Number,
        ) -> None:
            '''(experimental) The upper and lower EC2 instance limits for an automatic scaling policy.

            Automatic scaling activities triggered by automatic scaling
            rules will not cause an instance group to grow above or below these limits.

            :param max_capacity: (experimental) The upper boundary of EC2 instances in an instance group beyond which scaling activities are not allowed to grow. Scale-out activities will not add instances beyond this boundary.
            :param min_capacity: (experimental) The lower boundary of EC2 instances in an instance group below which scaling activities are not allowed to shrink. Scale-in activities will not terminate instances below this boundary.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingConstraints.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                scaling_constraints_property = stepfunctions_tasks.EmrCreateCluster.ScalingConstraintsProperty(
                    max_capacity=123,
                    min_capacity=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_capacity": max_capacity,
                "min_capacity": min_capacity,
            }

        @builtins.property
        def max_capacity(self) -> jsii.Number:
            '''(experimental) The upper boundary of EC2 instances in an instance group beyond which scaling activities are not allowed to grow.

            Scale-out
            activities will not add instances beyond this boundary.

            :stability: experimental
            '''
            result = self._values.get("max_capacity")
            assert result is not None, "Required property 'max_capacity' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def min_capacity(self) -> jsii.Number:
            '''(experimental) The lower boundary of EC2 instances in an instance group below which scaling activities are not allowed to shrink.

            Scale-in
            activities will not terminate instances below this boundary.

            :stability: experimental
            '''
            result = self._values.get("min_capacity")
            assert result is not None, "Required property 'min_capacity' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingConstraintsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "name": "name",
            "trigger": "trigger",
            "description": "description",
        },
    )
    class ScalingRuleProperty:
        def __init__(
            self,
            *,
            action: "EmrCreateCluster.ScalingActionProperty",
            name: builtins.str,
            trigger: "EmrCreateCluster.ScalingTriggerProperty",
            description: typing.Optional[builtins.str] = None,
        ) -> None:
            '''(experimental) A scale-in or scale-out rule that defines scaling activity, including the CloudWatch metric alarm that triggers activity, how EC2 instances are added or removed, and the periodicity of adjustments.

            :param action: (experimental) The conditions that trigger an automatic scaling activity.
            :param name: (experimental) The name used to identify an automatic scaling rule. Rule names must be unique within a scaling policy.
            :param trigger: (experimental) The CloudWatch alarm definition that determines when automatic scaling activity is triggered.
            :param description: (experimental) A friendly, more verbose description of the automatic scaling rule. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingRule.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # duration: monocdk.Duration
                
                scaling_rule_property = stepfunctions_tasks.EmrCreateCluster.ScalingRuleProperty(
                    action=stepfunctions_tasks.EmrCreateCluster.ScalingActionProperty(
                        simple_scaling_policy_configuration=stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                            scaling_adjustment=123,
                
                            # the properties below are optional
                            adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                            cool_down=123
                        ),
                
                        # the properties below are optional
                        market=stepfunctions_tasks.EmrCreateCluster.InstanceMarket.ON_DEMAND
                    ),
                    name="name",
                    trigger=stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                        cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                            comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                            metric_name="metricName",
                            period=duration,
                
                            # the properties below are optional
                            dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                                key="key",
                                value="value"
                            )],
                            evaluation_periods=123,
                            namespace="namespace",
                            statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                            threshold=123,
                            unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                        )
                    ),
                
                    # the properties below are optional
                    description="description"
                )
            '''
            if isinstance(action, dict):
                action = ScalingActionProperty(**action)
            if isinstance(trigger, dict):
                trigger = ScalingTriggerProperty(**trigger)
            self._values: typing.Dict[str, typing.Any] = {
                "action": action,
                "name": name,
                "trigger": trigger,
            }
            if description is not None:
                self._values["description"] = description

        @builtins.property
        def action(self) -> "EmrCreateCluster.ScalingActionProperty":
            '''(experimental) The conditions that trigger an automatic scaling activity.

            :stability: experimental
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast("EmrCreateCluster.ScalingActionProperty", result)

        @builtins.property
        def name(self) -> builtins.str:
            '''(experimental) The name used to identify an automatic scaling rule.

            Rule names must be unique within a scaling policy.

            :stability: experimental
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def trigger(self) -> "EmrCreateCluster.ScalingTriggerProperty":
            '''(experimental) The CloudWatch alarm definition that determines when automatic scaling activity is triggered.

            :stability: experimental
            '''
            result = self._values.get("trigger")
            assert result is not None, "Required property 'trigger' is missing"
            return typing.cast("EmrCreateCluster.ScalingTriggerProperty", result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''(experimental) A friendly, more verbose description of the automatic scaling rule.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingRuleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty",
        jsii_struct_bases=[],
        name_mapping={"cloud_watch_alarm_definition": "cloudWatchAlarmDefinition"},
    )
    class ScalingTriggerProperty:
        def __init__(
            self,
            *,
            cloud_watch_alarm_definition: "EmrCreateCluster.CloudWatchAlarmDefinitionProperty",
        ) -> None:
            '''(experimental) The conditions that trigger an automatic scaling activity and the definition of a CloudWatch metric alarm.

            When the defined alarm conditions are met along with other trigger parameters, scaling activity begins.

            :param cloud_watch_alarm_definition: (experimental) The definition of a CloudWatch metric alarm. When the defined alarm conditions are met along with other trigger parameters, scaling activity begins.

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScalingTrigger.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # duration: monocdk.Duration
                
                scaling_trigger_property = stepfunctions_tasks.EmrCreateCluster.ScalingTriggerProperty(
                    cloud_watch_alarm_definition=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmDefinitionProperty(
                        comparison_operator=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmComparisonOperator.GREATER_THAN_OR_EQUAL,
                        metric_name="metricName",
                        period=duration,
                
                        # the properties below are optional
                        dimensions=[stepfunctions_tasks.EmrCreateCluster.MetricDimensionProperty(
                            key="key",
                            value="value"
                        )],
                        evaluation_periods=123,
                        namespace="namespace",
                        statistic=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmStatistic.SAMPLE_COUNT,
                        threshold=123,
                        unit=stepfunctions_tasks.EmrCreateCluster.CloudWatchAlarmUnit.NONE
                    )
                )
            '''
            if isinstance(cloud_watch_alarm_definition, dict):
                cloud_watch_alarm_definition = CloudWatchAlarmDefinitionProperty(**cloud_watch_alarm_definition)
            self._values: typing.Dict[str, typing.Any] = {
                "cloud_watch_alarm_definition": cloud_watch_alarm_definition,
            }

        @builtins.property
        def cloud_watch_alarm_definition(
            self,
        ) -> "EmrCreateCluster.CloudWatchAlarmDefinitionProperty":
            '''(experimental) The definition of a CloudWatch metric alarm.

            When the defined alarm conditions are met along with other trigger parameters,
            scaling activity begins.

            :stability: experimental
            '''
            result = self._values.get("cloud_watch_alarm_definition")
            assert result is not None, "Required property 'cloud_watch_alarm_definition' is missing"
            return typing.cast("EmrCreateCluster.CloudWatchAlarmDefinitionProperty", result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScalingTriggerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.ScriptBootstrapActionConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path", "args": "args"},
    )
    class ScriptBootstrapActionConfigProperty:
        def __init__(
            self,
            *,
            path: builtins.str,
            args: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''(experimental) Configuration of the script to run during a bootstrap action.

            :param path: (experimental) Location of the script to run during a bootstrap action. Can be either a location in Amazon S3 or on a local file system.
            :param args: (experimental) A list of command line arguments to pass to the bootstrap action script. Default: No args

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ScriptBootstrapActionConfig.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                script_bootstrap_action_config_property = stepfunctions_tasks.EmrCreateCluster.ScriptBootstrapActionConfigProperty(
                    path="path",
                
                    # the properties below are optional
                    args=["args"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "path": path,
            }
            if args is not None:
                self._values["args"] = args

        @builtins.property
        def path(self) -> builtins.str:
            '''(experimental) Location of the script to run during a bootstrap action.

            Can be either a location in Amazon S3 or on a local file system.

            :stability: experimental
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def args(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) A list of command line arguments to pass to the bootstrap action script.

            :default: No args

            :stability: experimental
            '''
            result = self._values.get("args")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScriptBootstrapActionConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "scaling_adjustment": "scalingAdjustment",
            "adjustment_type": "adjustmentType",
            "cool_down": "coolDown",
        },
    )
    class SimpleScalingPolicyConfigurationProperty:
        def __init__(
            self,
            *,
            scaling_adjustment: jsii.Number,
            adjustment_type: typing.Optional["EmrCreateCluster.ScalingAdjustmentType"] = None,
            cool_down: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(experimental) An automatic scaling configuration, which describes how the policy adds or removes instances, the cooldown period, and the number of EC2 instances that will be added each time the CloudWatch metric alarm condition is satisfied.

            :param scaling_adjustment: (experimental) The amount by which to scale in or scale out, based on the specified AdjustmentType. A positive value adds to the instance group's EC2 instance count while a negative number removes instances. If AdjustmentType is set to EXACT_CAPACITY, the number should only be a positive integer.
            :param adjustment_type: (experimental) The way in which EC2 instances are added (if ScalingAdjustment is a positive number) or terminated (if ScalingAdjustment is a negative number) each time the scaling activity is triggered. Default: - None
            :param cool_down: (experimental) The amount of time, in seconds, after a scaling activity completes before any further trigger-related scaling activities can start. Default: 0

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_SimpleScalingPolicyConfiguration.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                simple_scaling_policy_configuration_property = stepfunctions_tasks.EmrCreateCluster.SimpleScalingPolicyConfigurationProperty(
                    scaling_adjustment=123,
                
                    # the properties below are optional
                    adjustment_type=stepfunctions_tasks.EmrCreateCluster.ScalingAdjustmentType.CHANGE_IN_CAPACITY,
                    cool_down=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "scaling_adjustment": scaling_adjustment,
            }
            if adjustment_type is not None:
                self._values["adjustment_type"] = adjustment_type
            if cool_down is not None:
                self._values["cool_down"] = cool_down

        @builtins.property
        def scaling_adjustment(self) -> jsii.Number:
            '''(experimental) The amount by which to scale in or scale out, based on the specified AdjustmentType.

            A positive value adds to the instance group's
            EC2 instance count while a negative number removes instances. If AdjustmentType is set to EXACT_CAPACITY, the number should only be
            a positive integer.

            :stability: experimental
            '''
            result = self._values.get("scaling_adjustment")
            assert result is not None, "Required property 'scaling_adjustment' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def adjustment_type(
            self,
        ) -> typing.Optional["EmrCreateCluster.ScalingAdjustmentType"]:
            '''(experimental) The way in which EC2 instances are added (if ScalingAdjustment is a positive number) or terminated (if ScalingAdjustment is a negative number) each time the scaling activity is triggered.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("adjustment_type")
            return typing.cast(typing.Optional["EmrCreateCluster.ScalingAdjustmentType"], result)

        @builtins.property
        def cool_down(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The amount of time, in seconds, after a scaling activity completes before any further trigger-related scaling activities can start.

            :default: 0

            :stability: experimental
            '''
            result = self._values.get("cool_down")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SimpleScalingPolicyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy"
    )
    class SpotAllocationStrategy(enum.Enum):
        '''(experimental) Spot Allocation Strategies.

        Specifies the strategy to use in launching Spot Instance fleets. For example, "capacity-optimized" launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
        :stability: experimental
        '''

        CAPACITY_OPTIMIZED = "CAPACITY_OPTIMIZED"
        '''(experimental) Capacity-optimized, which launches instances from Spot Instance pools with optimal capacity for the number of instances that are launching.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "timeout_action": "timeoutAction",
            "timeout_duration_minutes": "timeoutDurationMinutes",
            "allocation_strategy": "allocationStrategy",
            "block_duration_minutes": "blockDurationMinutes",
        },
    )
    class SpotProvisioningSpecificationProperty:
        def __init__(
            self,
            *,
            timeout_action: "EmrCreateCluster.SpotTimeoutAction",
            timeout_duration_minutes: jsii.Number,
            allocation_strategy: typing.Optional["EmrCreateCluster.SpotAllocationStrategy"] = None,
            block_duration_minutes: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(experimental) The launch specification for Spot instances in the instance fleet, which determines the defined duration and provisioning timeout behavior.

            :param timeout_action: (experimental) The action to take when TargetSpotCapacity has not been fulfilled when the TimeoutDurationMinutes has expired.
            :param timeout_duration_minutes: (experimental) The spot provisioning timeout period in minutes.
            :param allocation_strategy: (experimental) Specifies the strategy to use in launching Spot Instance fleets. Default: - No allocation strategy, i.e. spot instance type will be chosen based on current price only
            :param block_duration_minutes: (experimental) The defined duration for Spot instances (also known as Spot blocks) in minutes. Default: - No blockDurationMinutes

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_SpotProvisioningSpecification.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                spot_provisioning_specification_property = stepfunctions_tasks.EmrCreateCluster.SpotProvisioningSpecificationProperty(
                    timeout_action=stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction.SWITCH_TO_ON_DEMAND,
                    timeout_duration_minutes=123,
                
                    # the properties below are optional
                    allocation_strategy=stepfunctions_tasks.EmrCreateCluster.SpotAllocationStrategy.CAPACITY_OPTIMIZED,
                    block_duration_minutes=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "timeout_action": timeout_action,
                "timeout_duration_minutes": timeout_duration_minutes,
            }
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if block_duration_minutes is not None:
                self._values["block_duration_minutes"] = block_duration_minutes

        @builtins.property
        def timeout_action(self) -> "EmrCreateCluster.SpotTimeoutAction":
            '''(experimental) The action to take when TargetSpotCapacity has not been fulfilled when the TimeoutDurationMinutes has expired.

            :stability: experimental
            '''
            result = self._values.get("timeout_action")
            assert result is not None, "Required property 'timeout_action' is missing"
            return typing.cast("EmrCreateCluster.SpotTimeoutAction", result)

        @builtins.property
        def timeout_duration_minutes(self) -> jsii.Number:
            '''(experimental) The spot provisioning timeout period in minutes.

            :stability: experimental
            '''
            result = self._values.get("timeout_duration_minutes")
            assert result is not None, "Required property 'timeout_duration_minutes' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def allocation_strategy(
            self,
        ) -> typing.Optional["EmrCreateCluster.SpotAllocationStrategy"]:
            '''(experimental) Specifies the strategy to use in launching Spot Instance fleets.

            :default: - No allocation strategy, i.e. spot instance type will be chosen based on current price only

            :stability: experimental
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional["EmrCreateCluster.SpotAllocationStrategy"], result)

        @builtins.property
        def block_duration_minutes(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The defined duration for Spot instances (also known as Spot blocks) in minutes.

            :default: - No blockDurationMinutes

            :stability: experimental
            '''
            result = self._values.get("block_duration_minutes")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SpotProvisioningSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.enum(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.SpotTimeoutAction"
    )
    class SpotTimeoutAction(enum.Enum):
        '''(experimental) Spot Timeout Actions.

        :stability: experimental
        '''

        SWITCH_TO_ON_DEMAND = "SWITCH_TO_ON_DEMAND"
        '''(experimental) SWITCH_TO_ON_DEMAND.

        :stability: experimental
        '''
        TERMINATE_CLUSTER = "TERMINATE_CLUSTER"
        '''(experimental) TERMINATE_CLUSTER.

        :stability: experimental
        '''

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
            "iops": "iops",
        },
    )
    class VolumeSpecificationProperty:
        def __init__(
            self,
            *,
            volume_size: _Size_7fbd4337,
            volume_type: "EmrCreateCluster.EbsBlockDeviceVolumeType",
            iops: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''(experimental) EBS volume specifications such as volume type, IOPS, and size (GiB) that will be requested for the EBS volume attached to an EC2 instance in the cluster.

            :param volume_size: (experimental) The volume size. If the volume type is EBS-optimized, the minimum value is 10GiB. Maximum size is 1TiB
            :param volume_type: (experimental) The volume type. Volume types supported are gp2, io1, standard.
            :param iops: (experimental) The number of I/O operations per second (IOPS) that the volume supports. Default: - EMR selected default

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_VolumeSpecification.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # size: monocdk.Size
                
                volume_specification_property = stepfunctions_tasks.EmrCreateCluster.VolumeSpecificationProperty(
                    volume_size=size,
                    volume_type=stepfunctions_tasks.EmrCreateCluster.EbsBlockDeviceVolumeType.GP2,
                
                    # the properties below are optional
                    iops=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "volume_size": volume_size,
                "volume_type": volume_type,
            }
            if iops is not None:
                self._values["iops"] = iops

        @builtins.property
        def volume_size(self) -> _Size_7fbd4337:
            '''(experimental) The volume size.

            If the volume type is EBS-optimized, the minimum value is 10GiB.
            Maximum size is 1TiB

            :stability: experimental
            '''
            result = self._values.get("volume_size")
            assert result is not None, "Required property 'volume_size' is missing"
            return typing.cast(_Size_7fbd4337, result)

        @builtins.property
        def volume_type(self) -> "EmrCreateCluster.EbsBlockDeviceVolumeType":
            '''(experimental) The volume type.

            Volume types supported are gp2, io1, standard.

            :stability: experimental
            '''
            result = self._values.get("volume_type")
            assert result is not None, "Required property 'volume_type' is missing"
            return typing.cast("EmrCreateCluster.EbsBlockDeviceVolumeType", result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''(experimental) The number of I/O operations per second (IOPS) that the volume supports.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrCreateClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "instances": "instances",
        "name": "name",
        "additional_info": "additionalInfo",
        "applications": "applications",
        "auto_scaling_role": "autoScalingRole",
        "bootstrap_actions": "bootstrapActions",
        "cluster_role": "clusterRole",
        "configurations": "configurations",
        "custom_ami_id": "customAmiId",
        "ebs_root_volume_size": "ebsRootVolumeSize",
        "kerberos_attributes": "kerberosAttributes",
        "log_uri": "logUri",
        "release_label": "releaseLabel",
        "scale_down_behavior": "scaleDownBehavior",
        "security_configuration": "securityConfiguration",
        "service_role": "serviceRole",
        "step_concurrency_level": "stepConcurrencyLevel",
        "tags": "tags",
        "visible_to_all_users": "visibleToAllUsers",
    },
)
class EmrCreateClusterProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        instances: EmrCreateCluster.InstancesConfigProperty,
        name: builtins.str,
        additional_info: typing.Optional[builtins.str] = None,
        applications: typing.Optional[typing.Sequence[EmrCreateCluster.ApplicationConfigProperty]] = None,
        auto_scaling_role: typing.Optional[_IRole_59af6f50] = None,
        bootstrap_actions: typing.Optional[typing.Sequence[EmrCreateCluster.BootstrapActionConfigProperty]] = None,
        cluster_role: typing.Optional[_IRole_59af6f50] = None,
        configurations: typing.Optional[typing.Sequence[EmrCreateCluster.ConfigurationProperty]] = None,
        custom_ami_id: typing.Optional[builtins.str] = None,
        ebs_root_volume_size: typing.Optional[_Size_7fbd4337] = None,
        kerberos_attributes: typing.Optional[EmrCreateCluster.KerberosAttributesProperty] = None,
        log_uri: typing.Optional[builtins.str] = None,
        release_label: typing.Optional[builtins.str] = None,
        scale_down_behavior: typing.Optional[EmrCreateCluster.EmrClusterScaleDownBehavior] = None,
        security_configuration: typing.Optional[builtins.str] = None,
        service_role: typing.Optional[_IRole_59af6f50] = None,
        step_concurrency_level: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        visible_to_all_users: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Properties for EmrCreateCluster.

        See the RunJobFlow API for complete documentation on input parameters

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param instances: (experimental) A specification of the number and type of Amazon EC2 instances.
        :param name: (experimental) The Name of the Cluster.
        :param additional_info: (experimental) A JSON string for selecting additional features. Default: - None
        :param applications: (experimental) A case-insensitive list of applications for Amazon EMR to install and configure when launching the cluster. Default: - EMR selected default
        :param auto_scaling_role: (experimental) An IAM role for automatic scaling policies. Default: - A role will be created.
        :param bootstrap_actions: (experimental) A list of bootstrap actions to run before Hadoop starts on the cluster nodes. Default: - None
        :param cluster_role: (experimental) Also called instance profile and EC2 role. An IAM role for an EMR cluster. The EC2 instances of the cluster assume this role. This attribute has been renamed from jobFlowRole to clusterRole to align with other ERM/StepFunction integration parameters. Default: - - A Role will be created
        :param configurations: (experimental) The list of configurations supplied for the EMR cluster you are creating. Default: - None
        :param custom_ami_id: (experimental) The ID of a custom Amazon EBS-backed Linux AMI. Default: - None
        :param ebs_root_volume_size: (experimental) The size of the EBS root device volume of the Linux AMI that is used for each EC2 instance. Default: - EMR selected default
        :param kerberos_attributes: (experimental) Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration. Default: - None
        :param log_uri: (experimental) The location in Amazon S3 to write the log files of the job flow. Default: - None
        :param release_label: (experimental) The Amazon EMR release label, which determines the version of open-source application packages installed on the cluster. Default: - EMR selected default
        :param scale_down_behavior: (experimental) Specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized. Default: - EMR selected default
        :param security_configuration: (experimental) The name of a security configuration to apply to the cluster. Default: - None
        :param service_role: (experimental) The IAM role that will be assumed by the Amazon EMR service to access AWS resources on your behalf. Default: - A role will be created that Amazon EMR service can assume.
        :param step_concurrency_level: (experimental) Specifies the step concurrency level to allow multiple steps to run in parallel. Requires EMR release label 5.28.0 or above. Must be in range [1, 256]. Default: 1 - no step concurrency allowed
        :param tags: (experimental) A list of tags to associate with a cluster and propagate to Amazon EC2 instances. Default: - None
        :param visible_to_all_users: (experimental) A value of true indicates that all IAM users in the AWS account can perform cluster actions if they have the proper IAM policy permissions. Default: true

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html
        :stability: experimental
        :exampleMetadata: infused

        Example::

            cluster_role = iam.Role(self, "ClusterRole",
                assumed_by=iam.ServicePrincipal("ec2.amazonaws.com")
            )
            
            service_role = iam.Role(self, "ServiceRole",
                assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
            )
            
            auto_scaling_role = iam.Role(self, "AutoScalingRole",
                assumed_by=iam.ServicePrincipal("elasticmapreduce.amazonaws.com")
            )
            
            auto_scaling_role.assume_role_policy.add_statements(
                iam.PolicyStatement(
                    effect=iam.Effect.ALLOW,
                    principals=[
                        iam.ServicePrincipal("application-autoscaling.amazonaws.com")
                    ],
                    actions=["sts:AssumeRole"
                    ]
                ))
            
            tasks.EmrCreateCluster(self, "Create Cluster",
                instances=dynamodb.aws_stepfunctions_tasks.EmrCreateCluster.InstancesConfigProperty(),
                cluster_role=cluster_role,
                name=sfn.TaskInput.from_json_path_at("$.ClusterName").value,
                service_role=service_role,
                auto_scaling_role=auto_scaling_role
            )
        '''
        if isinstance(instances, dict):
            instances = EmrCreateCluster.InstancesConfigProperty(**instances)
        if isinstance(kerberos_attributes, dict):
            kerberos_attributes = EmrCreateCluster.KerberosAttributesProperty(**kerberos_attributes)
        self._values: typing.Dict[str, typing.Any] = {
            "instances": instances,
            "name": name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if applications is not None:
            self._values["applications"] = applications
        if auto_scaling_role is not None:
            self._values["auto_scaling_role"] = auto_scaling_role
        if bootstrap_actions is not None:
            self._values["bootstrap_actions"] = bootstrap_actions
        if cluster_role is not None:
            self._values["cluster_role"] = cluster_role
        if configurations is not None:
            self._values["configurations"] = configurations
        if custom_ami_id is not None:
            self._values["custom_ami_id"] = custom_ami_id
        if ebs_root_volume_size is not None:
            self._values["ebs_root_volume_size"] = ebs_root_volume_size
        if kerberos_attributes is not None:
            self._values["kerberos_attributes"] = kerberos_attributes
        if log_uri is not None:
            self._values["log_uri"] = log_uri
        if release_label is not None:
            self._values["release_label"] = release_label
        if scale_down_behavior is not None:
            self._values["scale_down_behavior"] = scale_down_behavior
        if security_configuration is not None:
            self._values["security_configuration"] = security_configuration
        if service_role is not None:
            self._values["service_role"] = service_role
        if step_concurrency_level is not None:
            self._values["step_concurrency_level"] = step_concurrency_level
        if tags is not None:
            self._values["tags"] = tags
        if visible_to_all_users is not None:
            self._values["visible_to_all_users"] = visible_to_all_users

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def instances(self) -> EmrCreateCluster.InstancesConfigProperty:
        '''(experimental) A specification of the number and type of Amazon EC2 instances.

        :stability: experimental
        '''
        result = self._values.get("instances")
        assert result is not None, "Required property 'instances' is missing"
        return typing.cast(EmrCreateCluster.InstancesConfigProperty, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) The Name of the Cluster.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def additional_info(self) -> typing.Optional[builtins.str]:
        '''(experimental) A JSON string for selecting additional features.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def applications(
        self,
    ) -> typing.Optional[typing.List[EmrCreateCluster.ApplicationConfigProperty]]:
        '''(experimental) A case-insensitive list of applications for Amazon EMR to install and configure when launching the cluster.

        :default: - EMR selected default

        :stability: experimental
        '''
        result = self._values.get("applications")
        return typing.cast(typing.Optional[typing.List[EmrCreateCluster.ApplicationConfigProperty]], result)

    @builtins.property
    def auto_scaling_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) An IAM role for automatic scaling policies.

        :default: - A role will be created.

        :stability: experimental
        '''
        result = self._values.get("auto_scaling_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def bootstrap_actions(
        self,
    ) -> typing.Optional[typing.List[EmrCreateCluster.BootstrapActionConfigProperty]]:
        '''(experimental) A list of bootstrap actions to run before Hadoop starts on the cluster nodes.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("bootstrap_actions")
        return typing.cast(typing.Optional[typing.List[EmrCreateCluster.BootstrapActionConfigProperty]], result)

    @builtins.property
    def cluster_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) Also called instance profile and EC2 role.

        An IAM role for an EMR cluster. The EC2 instances of the cluster assume this role.

        This attribute has been renamed from jobFlowRole to clusterRole to align with other ERM/StepFunction integration parameters.

        :default:

        -
        - A Role will be created

        :stability: experimental
        '''
        result = self._values.get("cluster_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def configurations(
        self,
    ) -> typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]]:
        '''(experimental) The list of configurations supplied for the EMR cluster you are creating.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("configurations")
        return typing.cast(typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]], result)

    @builtins.property
    def custom_ami_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) The ID of a custom Amazon EBS-backed Linux AMI.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("custom_ami_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ebs_root_volume_size(self) -> typing.Optional[_Size_7fbd4337]:
        '''(experimental) The size of the EBS root device volume of the Linux AMI that is used for each EC2 instance.

        :default: - EMR selected default

        :stability: experimental
        '''
        result = self._values.get("ebs_root_volume_size")
        return typing.cast(typing.Optional[_Size_7fbd4337], result)

    @builtins.property
    def kerberos_attributes(
        self,
    ) -> typing.Optional[EmrCreateCluster.KerberosAttributesProperty]:
        '''(experimental) Attributes for Kerberos configuration when Kerberos authentication is enabled using a security configuration.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("kerberos_attributes")
        return typing.cast(typing.Optional[EmrCreateCluster.KerberosAttributesProperty], result)

    @builtins.property
    def log_uri(self) -> typing.Optional[builtins.str]:
        '''(experimental) The location in Amazon S3 to write the log files of the job flow.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("log_uri")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def release_label(self) -> typing.Optional[builtins.str]:
        '''(experimental) The Amazon EMR release label, which determines the version of open-source application packages installed on the cluster.

        :default: - EMR selected default

        :stability: experimental
        '''
        result = self._values.get("release_label")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale_down_behavior(
        self,
    ) -> typing.Optional[EmrCreateCluster.EmrClusterScaleDownBehavior]:
        '''(experimental) Specifies the way that individual Amazon EC2 instances terminate when an automatic scale-in activity occurs or an instance group is resized.

        :default: - EMR selected default

        :stability: experimental
        '''
        result = self._values.get("scale_down_behavior")
        return typing.cast(typing.Optional[EmrCreateCluster.EmrClusterScaleDownBehavior], result)

    @builtins.property
    def security_configuration(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of a security configuration to apply to the cluster.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("security_configuration")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The IAM role that will be assumed by the Amazon EMR service to access AWS resources on your behalf.

        :default: - A role will be created that Amazon EMR service can assume.

        :stability: experimental
        '''
        result = self._values.get("service_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def step_concurrency_level(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Specifies the step concurrency level to allow multiple steps to run in parallel.

        Requires EMR release label 5.28.0 or above.
        Must be in range [1, 256].

        :default: 1 - no step concurrency allowed

        :stability: experimental
        '''
        result = self._values.get("step_concurrency_level")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A list of tags to associate with a cluster and propagate to Amazon EC2 instances.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def visible_to_all_users(self) -> typing.Optional[builtins.bool]:
        '''(experimental) A value of true indicates that all IAM users in the AWS account can perform cluster actions if they have the proper IAM policy permissions.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("visible_to_all_users")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrCreateClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrModifyInstanceFleetByName(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceFleetByName",
):
    '''(experimental) A Step Functions Task to to modify an InstanceFleet on an EMR Cluster.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrModifyInstanceFleetByName(self, "Task",
            cluster_id="ClusterId",
            instance_fleet_name="InstanceFleetName",
            target_on_demand_capacity=2,
            target_spot_capacity=0
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        instance_fleet_name: builtins.str,
        target_on_demand_capacity: jsii.Number,
        target_spot_capacity: jsii.Number,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster_id: (experimental) The ClusterId to update.
        :param instance_fleet_name: (experimental) The InstanceFleetName to update.
        :param target_on_demand_capacity: (experimental) The target capacity of On-Demand units for the instance fleet. Default: - None
        :param target_spot_capacity: (experimental) The target capacity of Spot units for the instance fleet. Default: - None
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrModifyInstanceFleetByNameProps(
            cluster_id=cluster_id,
            instance_fleet_name=instance_fleet_name,
            target_on_demand_capacity=target_on_demand_capacity,
            target_spot_capacity=target_spot_capacity,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceFleetByNameProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "instance_fleet_name": "instanceFleetName",
        "target_on_demand_capacity": "targetOnDemandCapacity",
        "target_spot_capacity": "targetSpotCapacity",
    },
)
class EmrModifyInstanceFleetByNameProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster_id: builtins.str,
        instance_fleet_name: builtins.str,
        target_on_demand_capacity: jsii.Number,
        target_spot_capacity: jsii.Number,
    ) -> None:
        '''(experimental) Properties for EmrModifyInstanceFleetByName.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster_id: (experimental) The ClusterId to update.
        :param instance_fleet_name: (experimental) The InstanceFleetName to update.
        :param target_on_demand_capacity: (experimental) The target capacity of On-Demand units for the instance fleet. Default: - None
        :param target_spot_capacity: (experimental) The target capacity of Spot units for the instance fleet. Default: - None

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrModifyInstanceFleetByName(self, "Task",
                cluster_id="ClusterId",
                instance_fleet_name="InstanceFleetName",
                target_on_demand_capacity=2,
                target_spot_capacity=0
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_id": cluster_id,
            "instance_fleet_name": instance_fleet_name,
            "target_on_demand_capacity": target_on_demand_capacity,
            "target_spot_capacity": target_spot_capacity,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''(experimental) The ClusterId to update.

        :stability: experimental
        '''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_fleet_name(self) -> builtins.str:
        '''(experimental) The InstanceFleetName to update.

        :stability: experimental
        '''
        result = self._values.get("instance_fleet_name")
        assert result is not None, "Required property 'instance_fleet_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_on_demand_capacity(self) -> jsii.Number:
        '''(experimental) The target capacity of On-Demand units for the instance fleet.

        :default: - None

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetModifyConfig.html
        :stability: experimental
        '''
        result = self._values.get("target_on_demand_capacity")
        assert result is not None, "Required property 'target_on_demand_capacity' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def target_spot_capacity(self) -> jsii.Number:
        '''(experimental) The target capacity of Spot units for the instance fleet.

        :default: - None

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceFleetModifyConfig.html
        :stability: experimental
        '''
        result = self._values.get("target_spot_capacity")
        assert result is not None, "Required property 'target_spot_capacity' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrModifyInstanceFleetByNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrModifyInstanceGroupByName(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName",
):
    '''(experimental) A Step Functions Task to to modify an InstanceGroup on an EMR Cluster.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrModifyInstanceGroupByName(self, "Task",
            cluster_id="ClusterId",
            instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
            instance_group=dynamodb.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
                instance_count=1
            )
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        instance_group: "EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty",
        instance_group_name: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster_id: (experimental) The ClusterId to update.
        :param instance_group: (experimental) The JSON that you want to provide to your ModifyInstanceGroup call as input. This uses the same syntax as the ModifyInstanceGroups API.
        :param instance_group_name: (experimental) The InstanceGroupName to update.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrModifyInstanceGroupByNameProps(
            cluster_id=cluster_id,
            instance_group=instance_group,
            instance_group_name=instance_group_name,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "configurations": "configurations",
            "e_c2_instance_ids_to_terminate": "eC2InstanceIdsToTerminate",
            "instance_count": "instanceCount",
            "shrink_policy": "shrinkPolicy",
        },
    )
    class InstanceGroupModifyConfigProperty:
        def __init__(
            self,
            *,
            configurations: typing.Optional[typing.Sequence[EmrCreateCluster.ConfigurationProperty]] = None,
            e_c2_instance_ids_to_terminate: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_count: typing.Optional[jsii.Number] = None,
            shrink_policy: typing.Optional["EmrModifyInstanceGroupByName.ShrinkPolicyProperty"] = None,
        ) -> None:
            '''(experimental) Modify the size or configurations of an instance group.

            :param configurations: (experimental) A list of new or modified configurations to apply for an instance group. Default: - None
            :param e_c2_instance_ids_to_terminate: (experimental) The EC2 InstanceIds to terminate. After you terminate the instances, the instance group will not return to its original requested size. Default: - None
            :param instance_count: (experimental) Target size for the instance group. Default: - None
            :param shrink_policy: (experimental) Policy for customizing shrink operations. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceGroupModifyConfig.html
            :stability: experimental
            :exampleMetadata: infused

            Example::

                tasks.EmrModifyInstanceGroupByName(self, "Task",
                    cluster_id="ClusterId",
                    instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
                    instance_group=dynamodb.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
                        instance_count=1
                    )
                )
            '''
            if isinstance(shrink_policy, dict):
                shrink_policy = ShrinkPolicyProperty(**shrink_policy)
            self._values: typing.Dict[str, typing.Any] = {}
            if configurations is not None:
                self._values["configurations"] = configurations
            if e_c2_instance_ids_to_terminate is not None:
                self._values["e_c2_instance_ids_to_terminate"] = e_c2_instance_ids_to_terminate
            if instance_count is not None:
                self._values["instance_count"] = instance_count
            if shrink_policy is not None:
                self._values["shrink_policy"] = shrink_policy

        @builtins.property
        def configurations(
            self,
        ) -> typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]]:
            '''(experimental) A list of new or modified configurations to apply for an instance group.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("configurations")
            return typing.cast(typing.Optional[typing.List[EmrCreateCluster.ConfigurationProperty]], result)

        @builtins.property
        def e_c2_instance_ids_to_terminate(
            self,
        ) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) The EC2 InstanceIds to terminate.

            After you terminate the instances, the instance group will not return to its original requested size.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("e_c2_instance_ids_to_terminate")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_count(self) -> typing.Optional[jsii.Number]:
            '''(experimental) Target size for the instance group.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("instance_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def shrink_policy(
            self,
        ) -> typing.Optional["EmrModifyInstanceGroupByName.ShrinkPolicyProperty"]:
            '''(experimental) Policy for customizing shrink operations.

            :default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ShrinkPolicy.html
            :stability: experimental
            '''
            result = self._values.get("shrink_policy")
            return typing.cast(typing.Optional["EmrModifyInstanceGroupByName.ShrinkPolicyProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceGroupModifyConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceResizePolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instances_to_protect": "instancesToProtect",
            "instances_to_terminate": "instancesToTerminate",
            "instance_termination_timeout": "instanceTerminationTimeout",
        },
    )
    class InstanceResizePolicyProperty:
        def __init__(
            self,
            *,
            instances_to_protect: typing.Optional[typing.Sequence[builtins.str]] = None,
            instances_to_terminate: typing.Optional[typing.Sequence[builtins.str]] = None,
            instance_termination_timeout: typing.Optional[_Duration_070aa057] = None,
        ) -> None:
            '''(experimental) Custom policy for requesting termination protection or termination of specific instances when shrinking an instance group.

            :param instances_to_protect: (experimental) Specific list of instances to be protected when shrinking an instance group. Default: - No instances will be protected when shrinking an instance group
            :param instances_to_terminate: (experimental) Specific list of instances to be terminated when shrinking an instance group. Default: - No instances will be terminated when shrinking an instance group.
            :param instance_termination_timeout: (experimental) Decommissioning timeout override for the specific list of instances to be terminated. Default: cdk.Duration.seconds

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_InstanceResizePolicy.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # duration: monocdk.Duration
                
                instance_resize_policy_property = stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceResizePolicyProperty(
                    instances_to_protect=["instancesToProtect"],
                    instances_to_terminate=["instancesToTerminate"],
                    instance_termination_timeout=duration
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if instances_to_protect is not None:
                self._values["instances_to_protect"] = instances_to_protect
            if instances_to_terminate is not None:
                self._values["instances_to_terminate"] = instances_to_terminate
            if instance_termination_timeout is not None:
                self._values["instance_termination_timeout"] = instance_termination_timeout

        @builtins.property
        def instances_to_protect(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) Specific list of instances to be protected when shrinking an instance group.

            :default: - No instances will be protected when shrinking an instance group

            :stability: experimental
            '''
            result = self._values.get("instances_to_protect")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instances_to_terminate(self) -> typing.Optional[typing.List[builtins.str]]:
            '''(experimental) Specific list of instances to be terminated when shrinking an instance group.

            :default: - No instances will be terminated when shrinking an instance group.

            :stability: experimental
            '''
            result = self._values.get("instances_to_terminate")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def instance_termination_timeout(self) -> typing.Optional[_Duration_070aa057]:
            '''(experimental) Decommissioning timeout override for the specific list of instances to be terminated.

            :default: cdk.Duration.seconds

            :stability: experimental
            '''
            result = self._values.get("instance_termination_timeout")
            return typing.cast(typing.Optional[_Duration_070aa057], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceResizePolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.ShrinkPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "decommission_timeout": "decommissionTimeout",
            "instance_resize_policy": "instanceResizePolicy",
        },
    )
    class ShrinkPolicyProperty:
        def __init__(
            self,
            *,
            decommission_timeout: typing.Optional[_Duration_070aa057] = None,
            instance_resize_policy: typing.Optional["EmrModifyInstanceGroupByName.InstanceResizePolicyProperty"] = None,
        ) -> None:
            '''(experimental) Policy for customizing shrink operations.

            Allows configuration of decommissioning timeout and targeted instance shrinking.

            :param decommission_timeout: (experimental) The desired timeout for decommissioning an instance. Overrides the default YARN decommissioning timeout. Default: - EMR selected default
            :param instance_resize_policy: (experimental) Custom policy for requesting termination protection or termination of specific instances when shrinking an instance group. Default: - None

            :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ShrinkPolicy.html
            :stability: experimental
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                import monocdk as monocdk
                from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
                
                # duration: monocdk.Duration
                
                shrink_policy_property = stepfunctions_tasks.EmrModifyInstanceGroupByName.ShrinkPolicyProperty(
                    decommission_timeout=duration,
                    instance_resize_policy=stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceResizePolicyProperty(
                        instances_to_protect=["instancesToProtect"],
                        instances_to_terminate=["instancesToTerminate"],
                        instance_termination_timeout=duration
                    )
                )
            '''
            if isinstance(instance_resize_policy, dict):
                instance_resize_policy = InstanceResizePolicyProperty(**instance_resize_policy)
            self._values: typing.Dict[str, typing.Any] = {}
            if decommission_timeout is not None:
                self._values["decommission_timeout"] = decommission_timeout
            if instance_resize_policy is not None:
                self._values["instance_resize_policy"] = instance_resize_policy

        @builtins.property
        def decommission_timeout(self) -> typing.Optional[_Duration_070aa057]:
            '''(experimental) The desired timeout for decommissioning an instance.

            Overrides the default YARN decommissioning timeout.

            :default: - EMR selected default

            :stability: experimental
            '''
            result = self._values.get("decommission_timeout")
            return typing.cast(typing.Optional[_Duration_070aa057], result)

        @builtins.property
        def instance_resize_policy(
            self,
        ) -> typing.Optional["EmrModifyInstanceGroupByName.InstanceResizePolicyProperty"]:
            '''(experimental) Custom policy for requesting termination protection or termination of specific instances when shrinking an instance group.

            :default: - None

            :stability: experimental
            '''
            result = self._values.get("instance_resize_policy")
            return typing.cast(typing.Optional["EmrModifyInstanceGroupByName.InstanceResizePolicyProperty"], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ShrinkPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrModifyInstanceGroupByNameProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "instance_group": "instanceGroup",
        "instance_group_name": "instanceGroupName",
    },
)
class EmrModifyInstanceGroupByNameProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster_id: builtins.str,
        instance_group: EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty,
        instance_group_name: builtins.str,
    ) -> None:
        '''(experimental) Properties for EmrModifyInstanceGroupByName.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster_id: (experimental) The ClusterId to update.
        :param instance_group: (experimental) The JSON that you want to provide to your ModifyInstanceGroup call as input. This uses the same syntax as the ModifyInstanceGroups API.
        :param instance_group_name: (experimental) The InstanceGroupName to update.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrModifyInstanceGroupByName(self, "Task",
                cluster_id="ClusterId",
                instance_group_name=sfn.JsonPath.string_at("$.InstanceGroupName"),
                instance_group=dynamodb.aws_stepfunctions_tasks.EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(
                    instance_count=1
                )
            )
        '''
        if isinstance(instance_group, dict):
            instance_group = EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty(**instance_group)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_id": cluster_id,
            "instance_group": instance_group,
            "instance_group_name": instance_group_name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''(experimental) The ClusterId to update.

        :stability: experimental
        '''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_group(
        self,
    ) -> EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty:
        '''(experimental) The JSON that you want to provide to your ModifyInstanceGroup call as input.

        This uses the same syntax as the ModifyInstanceGroups API.

        :see: https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html
        :stability: experimental
        '''
        result = self._values.get("instance_group")
        assert result is not None, "Required property 'instance_group' is missing"
        return typing.cast(EmrModifyInstanceGroupByName.InstanceGroupModifyConfigProperty, result)

    @builtins.property
    def instance_group_name(self) -> builtins.str:
        '''(experimental) The InstanceGroupName to update.

        :stability: experimental
        '''
        result = self._values.get("instance_group_name")
        assert result is not None, "Required property 'instance_group_name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrModifyInstanceGroupByNameProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrSetClusterTerminationProtection(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrSetClusterTerminationProtection",
):
    '''(experimental) A Step Functions Task to to set Termination Protection on an EMR Cluster.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrSetClusterTerminationProtection(self, "Task",
            cluster_id="ClusterId",
            termination_protected=False
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        termination_protected: builtins.bool,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster_id: (experimental) The ClusterId to update.
        :param termination_protected: (experimental) Termination protection indicator.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrSetClusterTerminationProtectionProps(
            cluster_id=cluster_id,
            termination_protected=termination_protected,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrSetClusterTerminationProtectionProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster_id": "clusterId",
        "termination_protected": "terminationProtected",
    },
)
class EmrSetClusterTerminationProtectionProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster_id: builtins.str,
        termination_protected: builtins.bool,
    ) -> None:
        '''(experimental) Properties for EmrSetClusterTerminationProtection.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster_id: (experimental) The ClusterId to update.
        :param termination_protected: (experimental) Termination protection indicator.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrSetClusterTerminationProtection(self, "Task",
                cluster_id="ClusterId",
                termination_protected=False
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_id": cluster_id,
            "termination_protected": termination_protected,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''(experimental) The ClusterId to update.

        :stability: experimental
        '''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def termination_protected(self) -> builtins.bool:
        '''(experimental) Termination protection indicator.

        :stability: experimental
        '''
        result = self._values.get("termination_protected")
        assert result is not None, "Required property 'termination_protected' is missing"
        return typing.cast(builtins.bool, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrSetClusterTerminationProtectionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EmrTerminateCluster(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrTerminateCluster",
):
    '''(experimental) A Step Functions Task to terminate an EMR Cluster.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrTerminateCluster(self, "Task",
            cluster_id="ClusterId"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        cluster_id: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param cluster_id: (experimental) The ClusterId to terminate.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EmrTerminateClusterProps(
            cluster_id=cluster_id,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EmrTerminateClusterProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "cluster_id": "clusterId",
    },
)
class EmrTerminateClusterProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        cluster_id: builtins.str,
    ) -> None:
        '''(experimental) Properties for EmrTerminateCluster.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param cluster_id: (experimental) The ClusterId to terminate.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrTerminateCluster(self, "Task",
                cluster_id="ClusterId"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_id": cluster_id,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def cluster_id(self) -> builtins.str:
        '''(experimental) The ClusterId to terminate.

        :stability: experimental
        '''
        result = self._values.get("cluster_id")
        assert result is not None, "Required property 'cluster_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EmrTerminateClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EncryptionConfiguration",
    jsii_struct_bases=[],
    name_mapping={
        "encryption_option": "encryptionOption",
        "encryption_key": "encryptionKey",
    },
)
class EncryptionConfiguration:
    def __init__(
        self,
        *,
        encryption_option: "EncryptionOption",
        encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''(experimental) Encryption Configuration of the S3 bucket.

        :param encryption_option: (experimental) Type of S3 server-side encryption enabled. Default: EncryptionOption.S3_MANAGED
        :param encryption_key: (experimental) KMS key ARN or ID. Default: - No KMS key for Encryption Option SSE_S3 and default master key for Encryption Option SSE_KMS and CSE_KMS

        :see: https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html
        :stability: experimental
        :exampleMetadata: infused

        Example::

            start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Athena Start Query",
                query_string=sfn.JsonPath.format("select contacts where year={};", sfn.JsonPath.string_at("$.year")),
                query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
                    database_name="interactions"
                ),
                result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
                    encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
                        encryption_option=tasks.EncryptionOption.S3_MANAGED
                    ),
                    output_location=dynamodb.aws_s3.Location(
                        bucket_name="mybucket",
                        object_key="myprefix"
                    )
                ),
                integration_pattern=sfn.IntegrationPattern.RUN_JOB
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "encryption_option": encryption_option,
        }
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key

    @builtins.property
    def encryption_option(self) -> "EncryptionOption":
        '''(experimental) Type of S3 server-side encryption enabled.

        :default: EncryptionOption.S3_MANAGED

        :stability: experimental
        '''
        result = self._values.get("encryption_option")
        assert result is not None, "Required property 'encryption_option' is missing"
        return typing.cast("EncryptionOption", result)

    @builtins.property
    def encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) KMS key ARN or ID.

        :default: - No KMS key for Encryption Option SSE_S3 and default master key for Encryption Option SSE_KMS and CSE_KMS

        :stability: experimental
        '''
        result = self._values.get("encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EncryptionConfiguration(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.EncryptionOption")
class EncryptionOption(enum.Enum):
    '''(experimental) Encryption Options of the S3 bucket.

    :see: https://docs.aws.amazon.com/athena/latest/APIReference/API_EncryptionConfiguration.html#athena-Type-EncryptionConfiguration-EncryptionOption
    :stability: experimental
    :exampleMetadata: infused

    Example::

        start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Athena Start Query",
            query_string=sfn.JsonPath.format("select contacts where year={};", sfn.JsonPath.string_at("$.year")),
            query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
                database_name="interactions"
            ),
            result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
                encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
                    encryption_option=tasks.EncryptionOption.S3_MANAGED
                ),
                output_location=dynamodb.aws_s3.Location(
                    bucket_name="mybucket",
                    object_key="myprefix"
                )
            ),
            integration_pattern=sfn.IntegrationPattern.RUN_JOB
        )
    '''

    S3_MANAGED = "S3_MANAGED"
    '''(experimental) Server side encryption (SSE) with an Amazon S3-managed key.

    :see: https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
    :stability: experimental
    '''
    KMS = "KMS"
    '''(experimental) Server-side encryption (SSE) with an AWS KMS key managed by the account owner.

    :see: https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
    :stability: experimental
    '''
    CLIENT_SIDE_KMS = "CLIENT_SIDE_KMS"
    '''(experimental) Client-side encryption (CSE) with an AWS KMS key managed by the account owner.

    :see: https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
    :stability: experimental
    '''


class EvaluateExpression(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EvaluateExpression",
):
    '''(experimental) A Step Functions Task to evaluate an expression.

    OUTPUT: the output of this task is the evaluated expression.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        convert_to_seconds = tasks.EvaluateExpression(self, "Convert to seconds",
            expression="$.waitMilliseconds / 1000",
            result_path="$.waitSeconds"
        )
        
        create_message = tasks.EvaluateExpression(self, "Create message",
            # Note: this is a string inside a string.
            expression="`Now waiting ${$.waitSeconds} seconds...`",
            runtime=lambda_.Runtime.NODEJS_14_X,
            result_path="$.message"
        )
        
        publish_message = tasks.SnsPublish(self, "Publish message",
            topic=sns.Topic(self, "cool-topic"),
            message=sfn.TaskInput.from_json_path_at("$.message"),
            result_path="$.sns"
        )
        
        wait = sfn.Wait(self, "Wait",
            time=sfn.WaitTime.seconds_path("$.waitSeconds")
        )
        
        sfn.StateMachine(self, "StateMachine",
            definition=convert_to_seconds.next(create_message).next(publish_message).next(wait)
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        expression: builtins.str,
        runtime: typing.Optional[_Runtime_932d369a] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param expression: (experimental) The expression to evaluate. The expression may contain state paths. Example value: ``'$.a + $.b'``
        :param runtime: (experimental) The runtime language to use to evaluate the expression. Default: lambda.Runtime.NODEJS_14_X
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EvaluateExpressionProps(
            expression=expression,
            runtime=runtime,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EvaluateExpressionProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "expression": "expression",
        "runtime": "runtime",
    },
)
class EvaluateExpressionProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        expression: builtins.str,
        runtime: typing.Optional[_Runtime_932d369a] = None,
    ) -> None:
        '''(experimental) Properties for EvaluateExpression.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param expression: (experimental) The expression to evaluate. The expression may contain state paths. Example value: ``'$.a + $.b'``
        :param runtime: (experimental) The runtime language to use to evaluate the expression. Default: lambda.Runtime.NODEJS_14_X

        :stability: experimental
        :exampleMetadata: infused

        Example::

            convert_to_seconds = tasks.EvaluateExpression(self, "Convert to seconds",
                expression="$.waitMilliseconds / 1000",
                result_path="$.waitSeconds"
            )
            
            create_message = tasks.EvaluateExpression(self, "Create message",
                # Note: this is a string inside a string.
                expression="`Now waiting ${$.waitSeconds} seconds...`",
                runtime=lambda_.Runtime.NODEJS_14_X,
                result_path="$.message"
            )
            
            publish_message = tasks.SnsPublish(self, "Publish message",
                topic=sns.Topic(self, "cool-topic"),
                message=sfn.TaskInput.from_json_path_at("$.message"),
                result_path="$.sns"
            )
            
            wait = sfn.Wait(self, "Wait",
                time=sfn.WaitTime.seconds_path("$.waitSeconds")
            )
            
            sfn.StateMachine(self, "StateMachine",
                definition=convert_to_seconds.next(create_message).next(publish_message).next(wait)
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "expression": expression,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if runtime is not None:
            self._values["runtime"] = runtime

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def expression(self) -> builtins.str:
        '''(experimental) The expression to evaluate. The expression may contain state paths.

        Example value: ``'$.a + $.b'``

        :stability: experimental
        '''
        result = self._values.get("expression")
        assert result is not None, "Required property 'expression' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def runtime(self) -> typing.Optional[_Runtime_932d369a]:
        '''(experimental) The runtime language to use to evaluate the expression.

        :default: lambda.Runtime.NODEJS_14_X

        :stability: experimental
        '''
        result = self._values.get("runtime")
        return typing.cast(typing.Optional[_Runtime_932d369a], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EvaluateExpressionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class EventBridgePutEvents(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.EventBridgePutEvents",
):
    '''(experimental) A StepFunctions Task to send events to an EventBridge event bus.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as events
        
        
        my_event_bus = events.EventBus(self, "EventBus",
            event_bus_name="MyEventBus1"
        )
        
        tasks.EventBridgePutEvents(self, "Send an event to EventBridge",
            entries=[events.aws_stepfunctions_tasks.EventBridgePutEventsEntry(
                detail=sfn.TaskInput.from_object({
                    "Message": "Hello from Step Functions!"
                }),
                event_bus=my_event_bus,
                detail_type="MessageFromStepFunctions",
                source="step.functions"
            )]
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        entries: typing.Sequence["EventBridgePutEventsEntry"],
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param entries: (experimental) The entries that will be sent (must be at least 1).
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = EventBridgePutEventsProps(
            entries=entries,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EventBridgePutEventsEntry",
    jsii_struct_bases=[],
    name_mapping={
        "detail": "detail",
        "detail_type": "detailType",
        "source": "source",
        "event_bus": "eventBus",
    },
)
class EventBridgePutEventsEntry:
    def __init__(
        self,
        *,
        detail: _TaskInput_71ab46f6,
        detail_type: builtins.str,
        source: builtins.str,
        event_bus: typing.Optional[_IEventBus_2ca38c95] = None,
    ) -> None:
        '''(experimental) An entry to be sent to EventBridge.

        :param detail: (experimental) The event body. Can either be provided as an object or as a JSON-serialized string
        :param detail_type: (experimental) Used along with the source field to help identify the fields and values expected in the detail field. For example, events by CloudTrail have detail type "AWS API Call via CloudTrail"
        :param source: (experimental) The service or application that caused this event to be generated. Example value: ``com.example.service``
        :param event_bus: (experimental) The event bus the entry will be sent to. Default: - event is sent to account's default event bus

        :see: https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEventsRequestEntry.html
        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_events as events
            from monocdk import aws_stepfunctions as stepfunctions
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # event_bus: events.EventBus
            # task_input: stepfunctions.TaskInput
            
            event_bridge_put_events_entry = stepfunctions_tasks.EventBridgePutEventsEntry(
                detail=task_input,
                detail_type="detailType",
                source="source",
            
                # the properties below are optional
                event_bus=event_bus
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "detail": detail,
            "detail_type": detail_type,
            "source": source,
        }
        if event_bus is not None:
            self._values["event_bus"] = event_bus

    @builtins.property
    def detail(self) -> _TaskInput_71ab46f6:
        '''(experimental) The event body.

        Can either be provided as an object or as a JSON-serialized string

        :stability: experimental

        Example::

            sfn.TaskInput.from_text("{\"instance-id\": \"i-1234567890abcdef0\", \"state\": \"terminated\"}")
            sfn.TaskInput.from_object({"Message": "Hello from Step Functions"})
            sfn.TaskInput.from_json_path_at("$.EventDetail")
        '''
        result = self._values.get("detail")
        assert result is not None, "Required property 'detail' is missing"
        return typing.cast(_TaskInput_71ab46f6, result)

    @builtins.property
    def detail_type(self) -> builtins.str:
        '''(experimental) Used along with the source field to help identify the fields and values expected in the detail field.

        For example, events by CloudTrail have detail type "AWS API Call via CloudTrail"

        :see: https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-events.html
        :stability: experimental
        '''
        result = self._values.get("detail_type")
        assert result is not None, "Required property 'detail_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def source(self) -> builtins.str:
        '''(experimental) The service or application that caused this event to be generated.

        Example value: ``com.example.service``

        :see: https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-events.html
        :stability: experimental
        '''
        result = self._values.get("source")
        assert result is not None, "Required property 'source' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def event_bus(self) -> typing.Optional[_IEventBus_2ca38c95]:
        '''(experimental) The event bus the entry will be sent to.

        :default: - event is sent to account's default event bus

        :stability: experimental
        '''
        result = self._values.get("event_bus")
        return typing.cast(typing.Optional[_IEventBus_2ca38c95], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventBridgePutEventsEntry(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.EventBridgePutEventsProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "entries": "entries",
    },
)
class EventBridgePutEventsProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        entries: typing.Sequence[EventBridgePutEventsEntry],
    ) -> None:
        '''(experimental) Properties for sending events with PutEvents.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param entries: (experimental) The entries that will be sent (must be at least 1).

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as events
            
            
            my_event_bus = events.EventBus(self, "EventBus",
                event_bus_name="MyEventBus1"
            )
            
            tasks.EventBridgePutEvents(self, "Send an event to EventBridge",
                entries=[events.aws_stepfunctions_tasks.EventBridgePutEventsEntry(
                    detail=sfn.TaskInput.from_object({
                        "Message": "Hello from Step Functions!"
                    }),
                    event_bus=my_event_bus,
                    detail_type="MessageFromStepFunctions",
                    source="step.functions"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "entries": entries,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def entries(self) -> typing.List[EventBridgePutEventsEntry]:
        '''(experimental) The entries that will be sent (must be at least 1).

        :stability: experimental
        '''
        result = self._values.get("entries")
        assert result is not None, "Required property 'entries' is missing"
        return typing.cast(typing.List[EventBridgePutEventsEntry], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EventBridgePutEventsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GlueDataBrewStartJobRun(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.GlueDataBrewStartJobRun",
):
    '''(experimental) Start a Job run as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-databrew.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.GlueDataBrewStartJobRun(self, "Task",
            name="databrew-job"
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        name: builtins.str,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param name: (experimental) Glue DataBrew Job to run.
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = GlueDataBrewStartJobRunProps(
            name=name,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.GlueDataBrewStartJobRunProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "name": "name",
    },
)
class GlueDataBrewStartJobRunProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        name: builtins.str,
    ) -> None:
        '''(experimental) Properties for starting a job run with StartJobRun.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param name: (experimental) Glue DataBrew Job to run.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.GlueDataBrewStartJobRun(self, "Task",
                name="databrew-job"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) Glue DataBrew Job to run.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GlueDataBrewStartJobRunProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GlueStartJobRun(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.GlueStartJobRun",
):
    '''(experimental) Starts an AWS Glue job in a Task state.

    OUTPUT: the output of this task is a JobRun structure, for details consult
    https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.GlueStartJobRun(self, "Task",
            glue_job_name="my-glue-job",
            arguments=sfn.TaskInput.from_object({
                "key": "value"
            }),
            timeout=Duration.minutes(30),
            notify_delay_after=Duration.minutes(5)
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        glue_job_name: builtins.str,
        arguments: typing.Optional[_TaskInput_71ab46f6] = None,
        notify_delay_after: typing.Optional[_Duration_070aa057] = None,
        security_configuration: typing.Optional[builtins.str] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param glue_job_name: (experimental) Glue job name.
        :param arguments: (experimental) The job arguments specifically for this run. For this job run, they replace the default arguments set in the job definition itself. Default: - Default arguments set in the job definition
        :param notify_delay_after: (experimental) After a job run starts, the number of minutes to wait before sending a job run delay notification. Must be at least 1 minute. Default: - Default delay set in the job definition
        :param security_configuration: (experimental) The name of the SecurityConfiguration structure to be used with this job run. This must match the Glue API Default: - Default configuration set in the job definition
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = GlueStartJobRunProps(
            glue_job_name=glue_job_name,
            arguments=arguments,
            notify_delay_after=notify_delay_after,
            security_configuration=security_configuration,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.GlueStartJobRunProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "glue_job_name": "glueJobName",
        "arguments": "arguments",
        "notify_delay_after": "notifyDelayAfter",
        "security_configuration": "securityConfiguration",
    },
)
class GlueStartJobRunProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        glue_job_name: builtins.str,
        arguments: typing.Optional[_TaskInput_71ab46f6] = None,
        notify_delay_after: typing.Optional[_Duration_070aa057] = None,
        security_configuration: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Properties for starting an AWS Glue job as a task.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param glue_job_name: (experimental) Glue job name.
        :param arguments: (experimental) The job arguments specifically for this run. For this job run, they replace the default arguments set in the job definition itself. Default: - Default arguments set in the job definition
        :param notify_delay_after: (experimental) After a job run starts, the number of minutes to wait before sending a job run delay notification. Must be at least 1 minute. Default: - Default delay set in the job definition
        :param security_configuration: (experimental) The name of the SecurityConfiguration structure to be used with this job run. This must match the Glue API Default: - Default configuration set in the job definition

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.GlueStartJobRun(self, "Task",
                glue_job_name="my-glue-job",
                arguments=sfn.TaskInput.from_object({
                    "key": "value"
                }),
                timeout=Duration.minutes(30),
                notify_delay_after=Duration.minutes(5)
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "glue_job_name": glue_job_name,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if arguments is not None:
            self._values["arguments"] = arguments
        if notify_delay_after is not None:
            self._values["notify_delay_after"] = notify_delay_after
        if security_configuration is not None:
            self._values["security_configuration"] = security_configuration

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def glue_job_name(self) -> builtins.str:
        '''(experimental) Glue job name.

        :stability: experimental
        '''
        result = self._values.get("glue_job_name")
        assert result is not None, "Required property 'glue_job_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def arguments(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) The job arguments specifically for this run.

        For this job run, they replace the default arguments set in the job
        definition itself.

        :default: - Default arguments set in the job definition

        :stability: experimental
        '''
        result = self._values.get("arguments")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def notify_delay_after(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) After a job run starts, the number of minutes to wait before sending a job run delay notification.

        Must be at least 1 minute.

        :default: - Default delay set in the job definition

        :stability: experimental
        '''
        result = self._values.get("notify_delay_after")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def security_configuration(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the SecurityConfiguration structure to be used with this job run.

        This must match the Glue API

        :default: - Default configuration set in the job definition

        :see: https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-common.html#aws-glue-api-regex-oneLine
        :stability: experimental
        '''
        result = self._values.get("security_configuration")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GlueStartJobRunProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.HttpMethod")
class HttpMethod(enum.Enum):
    '''(experimental) Http Methods that API Gateway supports.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as apigatewayv2
        
        http_api = apigatewayv2.HttpApi(self, "MyHttpApi")
        
        invoke_task = tasks.CallApiGatewayHttpApiEndpoint(self, "Call HTTP API",
            api_id=http_api.api_id,
            api_stack=Stack.of(http_api),
            method=tasks.HttpMethod.GET
        )
    '''

    GET = "GET"
    '''(experimental) Retreive data from a server at the specified resource.

    :stability: experimental
    '''
    POST = "POST"
    '''(experimental) Send data to the API endpoint to create or udpate a resource.

    :stability: experimental
    '''
    PUT = "PUT"
    '''(experimental) Send data to the API endpoint to update or create a resource.

    :stability: experimental
    '''
    DELETE = "DELETE"
    '''(experimental) Delete the resource at the specified endpoint.

    :stability: experimental
    '''
    PATCH = "PATCH"
    '''(experimental) Apply partial modifications to the resource.

    :stability: experimental
    '''
    HEAD = "HEAD"
    '''(experimental) Retreive data from a server at the specified resource without the response body.

    :stability: experimental
    '''
    OPTIONS = "OPTIONS"
    '''(experimental) Return data describing what other methods and operations the server supports.

    :stability: experimental
    '''


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.HttpMethods")
class HttpMethods(enum.Enum):
    '''(experimental) Method type of a EKS call.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as eks
        
        
        my_eks_cluster = eks.Cluster(self, "my sample cluster",
            version=eks.KubernetesVersion.V1_18,
            cluster_name="myEksCluster"
        )
        
        tasks.EksCall(self, "Call a EKS Endpoint",
            cluster=my_eks_cluster,
            http_method=tasks.HttpMethods.GET,
            http_path="/api/v1/namespaces/default/pods"
        )
    '''

    GET = "GET"
    '''(experimental) Retrieve data from a server at the specified resource.

    :stability: experimental
    '''
    POST = "POST"
    '''(experimental) Send data to the API endpoint to create or update a resource.

    :stability: experimental
    '''
    PUT = "PUT"
    '''(experimental) Send data to the API endpoint to update or create a resource.

    :stability: experimental
    '''
    DELETE = "DELETE"
    '''(experimental) Delete the resource at the specified endpoint.

    :stability: experimental
    '''
    PATCH = "PATCH"
    '''(experimental) Apply partial modifications to the resource.

    :stability: experimental
    '''
    HEAD = "HEAD"
    '''(experimental) Retrieve data from a server at the specified resource without the response body.

    :stability: experimental
    '''


@jsii.interface(jsii_type="monocdk.aws_stepfunctions_tasks.IContainerDefinition")
class IContainerDefinition(typing_extensions.Protocol):
    '''(experimental) Configuration of the container used to host the model.

    :see: https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
    :stability: experimental
    '''

    @jsii.member(jsii_name="bind")
    def bind(self, task: "ISageMakerTask") -> ContainerDefinitionConfig:
        '''(experimental) Called when the ContainerDefinition is used by a SageMaker task.

        :param task: -

        :stability: experimental
        '''
        ...


class _IContainerDefinitionProxy:
    '''(experimental) Configuration of the container used to host the model.

    :see: https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ContainerDefinition.html
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_stepfunctions_tasks.IContainerDefinition"

    @jsii.member(jsii_name="bind")
    def bind(self, task: "ISageMakerTask") -> ContainerDefinitionConfig:
        '''(experimental) Called when the ContainerDefinition is used by a SageMaker task.

        :param task: -

        :stability: experimental
        '''
        return typing.cast(ContainerDefinitionConfig, jsii.invoke(self, "bind", [task]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IContainerDefinition).__jsii_proxy_class__ = lambda : _IContainerDefinitionProxy


@jsii.interface(jsii_type="monocdk.aws_stepfunctions_tasks.IEcsLaunchTarget")
class IEcsLaunchTarget(typing_extensions.Protocol):
    '''(experimental) An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.

    :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html
    :stability: experimental
    '''

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        task: EcsRunTask,
        *,
        task_definition: _ITaskDefinition_ee0d1862,
        cluster: typing.Optional[_ICluster_42c4ec1a] = None,
    ) -> EcsLaunchTargetConfig:
        '''(experimental) called when the ECS launch target is configured on RunTask.

        :param task: -
        :param task_definition: (experimental) Task definition to run Docker containers in Amazon ECS.
        :param cluster: (experimental) A regional grouping of one or more container instances on which you can run tasks and services. Default: - No cluster

        :stability: experimental
        '''
        ...


class _IEcsLaunchTargetProxy:
    '''(experimental) An Amazon ECS launch type determines the type of infrastructure on which your tasks and services are hosted.

    :see: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html
    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_stepfunctions_tasks.IEcsLaunchTarget"

    @jsii.member(jsii_name="bind")
    def bind(
        self,
        task: EcsRunTask,
        *,
        task_definition: _ITaskDefinition_ee0d1862,
        cluster: typing.Optional[_ICluster_42c4ec1a] = None,
    ) -> EcsLaunchTargetConfig:
        '''(experimental) called when the ECS launch target is configured on RunTask.

        :param task: -
        :param task_definition: (experimental) Task definition to run Docker containers in Amazon ECS.
        :param cluster: (experimental) A regional grouping of one or more container instances on which you can run tasks and services. Default: - No cluster

        :stability: experimental
        '''
        launch_target_options = LaunchTargetBindOptions(
            task_definition=task_definition, cluster=cluster
        )

        return typing.cast(EcsLaunchTargetConfig, jsii.invoke(self, "bind", [task, launch_target_options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEcsLaunchTarget).__jsii_proxy_class__ = lambda : _IEcsLaunchTargetProxy


@jsii.interface(jsii_type="monocdk.aws_stepfunctions_tasks.ISageMakerTask")
class ISageMakerTask(_IGrantable_4c5a91d1, typing_extensions.Protocol):
    '''(experimental) Task to train a machine learning model using Amazon SageMaker.

    :stability: experimental
    '''

    pass


class _ISageMakerTaskProxy(
    jsii.proxy_for(_IGrantable_4c5a91d1) # type: ignore[misc]
):
    '''(experimental) Task to train a machine learning model using Amazon SageMaker.

    :stability: experimental
    '''

    __jsii_type__: typing.ClassVar[str] = "monocdk.aws_stepfunctions_tasks.ISageMakerTask"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ISageMakerTask).__jsii_proxy_class__ = lambda : _ISageMakerTaskProxy


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.InputMode")
class InputMode(enum.Enum):
    '''(experimental) Input mode that the algorithm supports.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
            training_job_name=sfn.JsonPath.string_at("$.JobName"),
            algorithm_specification=dynamodb.aws_stepfunctions_tasks.AlgorithmSpecification(
                algorithm_name="BlazingText",
                training_input_mode=tasks.InputMode.FILE
            ),
            input_data_config=[dynamodb.aws_stepfunctions_tasks.Channel(
                channel_name="train",
                data_source=dynamodb.aws_stepfunctions_tasks.DataSource(
                    s3_data_source=dynamodb.aws_stepfunctions_tasks.S3DataSource(
                        s3_data_type=tasks.S3DataType.S3_PREFIX,
                        s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                    )
                )
            )],
            output_data_config=dynamodb.aws_stepfunctions_tasks.OutputDataConfig(
                s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
            ),
            resource_config=dynamodb.aws_stepfunctions_tasks.ResourceConfig(
                instance_count=1,
                instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                volume_size=Size.gibibytes(50)
            ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
            stopping_condition=dynamodb.aws_stepfunctions_tasks.StoppingCondition(
                max_runtime=Duration.hours(2)
            )
        )
    '''

    PIPE = "PIPE"
    '''(experimental) Pipe mode.

    :stability: experimental
    '''
    FILE = "FILE"
    '''(experimental) File mode.

    :stability: experimental
    '''


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.InvocationType")
class InvocationType(enum.Enum):
    '''(deprecated) Invocation type of a Lambda.

    :deprecated: use ``LambdaInvocationType``

    :stability: deprecated
    '''

    REQUEST_RESPONSE = "REQUEST_RESPONSE"
    '''(deprecated) Invoke synchronously.

    The API response includes the function response and additional data.

    :stability: deprecated
    '''
    EVENT = "EVENT"
    '''(deprecated) Invoke asynchronously.

    Send events that fail multiple times to the function's dead-letter queue (if it's configured).
    The API response only includes a status code.

    :stability: deprecated
    '''
    DRY_RUN = "DRY_RUN"
    '''(deprecated) TValidate parameter values and verify that the user or role has permission to invoke the function.

    :stability: deprecated
    '''


@jsii.implements(_IStepFunctionsTask_82eb09ab)
class InvokeActivity(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.InvokeActivity",
):
    '''(deprecated) A Step Functions Task to invoke an Activity worker.

    An Activity can be used directly as a Resource.

    :deprecated: use ``StepFunctionsInvokeActivity``

    :stability: deprecated
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        import monocdk as monocdk
        from monocdk import aws_stepfunctions as stepfunctions
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        # activity: stepfunctions.Activity
        # duration: monocdk.Duration
        
        invoke_activity = stepfunctions_tasks.InvokeActivity(activity,
            heartbeat=duration
        )
    '''

    def __init__(
        self,
        activity: _IActivity_4524952f,
        *,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param activity: -
        :param heartbeat: (deprecated) Maximum time between heart beats. If the time between heart beats takes longer than this, a 'Timeout' error is raised. Default: No heart beat timeout

        :stability: deprecated
        '''
        props = InvokeActivityProps(heartbeat=heartbeat)

        jsii.create(self.__class__, self, [activity, props])

    @jsii.member(jsii_name="bind")
    def bind(self, _task: _Task_747ca4f2) -> _StepFunctionsTaskConfig_fa131821:
        '''(deprecated) Called when the task object is used in a workflow.

        :param _task: -

        :stability: deprecated
        '''
        return typing.cast(_StepFunctionsTaskConfig_fa131821, jsii.invoke(self, "bind", [_task]))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.InvokeActivityProps",
    jsii_struct_bases=[],
    name_mapping={"heartbeat": "heartbeat"},
)
class InvokeActivityProps:
    def __init__(
        self,
        *,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(deprecated) Properties for FunctionTask.

        :param heartbeat: (deprecated) Maximum time between heart beats. If the time between heart beats takes longer than this, a 'Timeout' error is raised. Default: No heart beat timeout

        :deprecated: use ``StepFunctionsInvokeActivity`` and ``StepFunctionsInvokeActivityProps``.

        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # duration: monocdk.Duration
            
            invoke_activity_props = stepfunctions_tasks.InvokeActivityProps(
                heartbeat=duration
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) Maximum time between heart beats.

        If the time between heart beats takes longer than this, a 'Timeout' error is raised.

        :default: No heart beat timeout

        :stability: deprecated
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeActivityProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IStepFunctionsTask_82eb09ab)
class InvokeFunction(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.InvokeFunction",
):
    '''(deprecated) A Step Functions Task to invoke a Lambda function.

    The Lambda function Arn is defined as Resource in the state machine definition.

    OUTPUT: the output of this task is the return value of the Lambda Function.

    :deprecated: Use ``LambdaInvoke``

    :stability: deprecated
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_lambda as lambda_
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        # function_: lambda.Function
        # payload: Any
        
        invoke_function = stepfunctions_tasks.InvokeFunction(function_,
            payload={
                "payload_key": payload
            }
        )
    '''

    def __init__(
        self,
        lambda_function: _IFunction_6e14f09e,
        *,
        payload: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''
        :param lambda_function: -
        :param payload: (deprecated) The JSON that you want to provide to your Lambda function as input. This parameter is named as payload to keep consistent with RunLambdaTask class. Default: - The JSON data indicated by the task's InputPath is used as payload

        :stability: deprecated
        '''
        props = InvokeFunctionProps(payload=payload)

        jsii.create(self.__class__, self, [lambda_function, props])

    @jsii.member(jsii_name="bind")
    def bind(self, _task: _Task_747ca4f2) -> _StepFunctionsTaskConfig_fa131821:
        '''(deprecated) Called when the task object is used in a workflow.

        :param _task: -

        :stability: deprecated
        '''
        return typing.cast(_StepFunctionsTaskConfig_fa131821, jsii.invoke(self, "bind", [_task]))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.InvokeFunctionProps",
    jsii_struct_bases=[],
    name_mapping={"payload": "payload"},
)
class InvokeFunctionProps:
    def __init__(
        self,
        *,
        payload: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
    ) -> None:
        '''(deprecated) Properties for InvokeFunction.

        :param payload: (deprecated) The JSON that you want to provide to your Lambda function as input. This parameter is named as payload to keep consistent with RunLambdaTask class. Default: - The JSON data indicated by the task's InputPath is used as payload

        :deprecated: use ``LambdaInvoke``

        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # payload: Any
            
            invoke_function_props = stepfunctions_tasks.InvokeFunctionProps(
                payload={
                    "payload_key": payload
                }
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if payload is not None:
            self._values["payload"] = payload

    @builtins.property
    def payload(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(deprecated) The JSON that you want to provide to your Lambda function as input.

        This parameter is named as payload to keep consistent with RunLambdaTask class.

        :default: - The JSON data indicated by the task's InputPath is used as payload

        :stability: deprecated
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeFunctionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.JobDependency",
    jsii_struct_bases=[],
    name_mapping={"job_id": "jobId", "type": "type"},
)
class JobDependency:
    def __init__(
        self,
        *,
        job_id: typing.Optional[builtins.str] = None,
        type: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) An object representing an AWS Batch job dependency.

        :param job_id: (experimental) The job ID of the AWS Batch job associated with this dependency. Default: - No jobId
        :param type: (experimental) The type of the job dependency. Default: - No type

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            job_dependency = stepfunctions_tasks.JobDependency(
                job_id="jobId",
                type="type"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if job_id is not None:
            self._values["job_id"] = job_id
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def job_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) The job ID of the AWS Batch job associated with this dependency.

        :default: - No jobId

        :stability: experimental
        '''
        result = self._values.get("job_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''(experimental) The type of the job dependency.

        :default: - No type

        :stability: experimental
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JobDependency(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.JobDriver",
    jsii_struct_bases=[],
    name_mapping={"spark_submit_job_driver": "sparkSubmitJobDriver"},
)
class JobDriver:
    def __init__(self, *, spark_submit_job_driver: "SparkSubmitJobDriver") -> None:
        '''(experimental) Specify the driver that the EMR Containers job runs on.

        The job driver is used to provide an input for the job that will be run.

        :param spark_submit_job_driver: (experimental) The job driver parameters specified for spark submit.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
                virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
                release_label=tasks.ReleaseLabel.EMR_6_2_0,
                job_name="EMR-Containers-Job",
                job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
                    spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
                        entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                    )
                ),
                application_config=[dynamodb.aws_stepfunctions_tasks.ApplicationConfiguration(
                    classification=tasks.Classification.SPARK_DEFAULTS,
                    properties={
                        "spark.executor.instances": "1",
                        "spark.executor.memory": "512M"
                    }
                )]
            )
        '''
        if isinstance(spark_submit_job_driver, dict):
            spark_submit_job_driver = SparkSubmitJobDriver(**spark_submit_job_driver)
        self._values: typing.Dict[str, typing.Any] = {
            "spark_submit_job_driver": spark_submit_job_driver,
        }

    @builtins.property
    def spark_submit_job_driver(self) -> "SparkSubmitJobDriver":
        '''(experimental) The job driver parameters specified for spark submit.

        :see: https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_SparkSubmitJobDriver.html
        :stability: experimental
        '''
        result = self._values.get("spark_submit_job_driver")
        assert result is not None, "Required property 'spark_submit_job_driver' is missing"
        return typing.cast("SparkSubmitJobDriver", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "JobDriver(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.LambdaInvocationType")
class LambdaInvocationType(enum.Enum):
    '''(experimental) Invocation type of a Lambda.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # fn: lambda.Function
        
        
        submit_job = tasks.LambdaInvoke(self, "Invoke Handler",
            lambda_function=fn,
            payload=sfn.TaskInput.from_json_path_at("$.input"),
            invocation_type=tasks.LambdaInvocationType.EVENT
        )
    '''

    REQUEST_RESPONSE = "REQUEST_RESPONSE"
    '''(experimental) Invoke the function synchronously.

    Keep the connection open until the function returns a response or times out.
    The API response includes the function response and additional data.

    :stability: experimental
    '''
    EVENT = "EVENT"
    '''(experimental) Invoke the function asynchronously.

    Send events that fail multiple times to the function's dead-letter queue (if it's configured).
    The API response only includes a status code.

    :stability: experimental
    '''
    DRY_RUN = "DRY_RUN"
    '''(experimental) Validate parameter values and verify that the user or role has permission to invoke the function.

    :stability: experimental
    '''


class LambdaInvoke(
    _TaskStateBase_c3080646,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.LambdaInvoke",
):
    '''(experimental) Invoke a Lambda function as a Task.

    :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
    :stability: experimental
    :exampleMetadata: infused

    Example::

        import monocdk as lambda_
        
        # order_fn: lambda.Function
        
        
        submit_job = tasks.LambdaInvoke(self, "InvokeOrderProcessor",
            lambda_function=order_fn,
            payload=sfn.TaskInput.from_object({
                "OrderId": sfn.JsonPath.string_at("$.OrderId")
            })
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        lambda_function: _IFunction_6e14f09e,
        client_context: typing.Optional[builtins.str] = None,
        invocation_type: typing.Optional[LambdaInvocationType] = None,
        payload: typing.Optional[_TaskInput_71ab46f6] = None,
        payload_response_only: typing.Optional[builtins.bool] = None,
        qualifier: typing.Optional[builtins.str] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param lambda_function: (experimental) Lambda function to invoke.
        :param client_context: (experimental) Up to 3583 bytes of base64-encoded data about the invoking client to pass to the function. Default: - No context
        :param invocation_type: (experimental) Invocation type of the Lambda function. Default: InvocationType.REQUEST_RESPONSE
        :param payload: (experimental) The JSON that will be supplied as input to the Lambda function. Default: - The state input (JSON path '$')
        :param payload_response_only: (experimental) Invoke the Lambda in a way that only returns the payload response without additional metadata. The ``payloadResponseOnly`` property cannot be used if ``integrationPattern``, ``invocationType``, ``clientContext``, or ``qualifier`` are specified. It always uses the REQUEST_RESPONSE behavior. Default: false
        :param qualifier: (experimental) Version or alias to invoke a published version of the function. You only need to supply this if you want the version of the Lambda Function to depend on data in the state machine state. If not, you can pass the appropriate Alias or Version object directly as the ``lambdaFunction`` argument. Default: - Version or alias inherent to the ``lambdaFunction`` object.
        :param retry_on_service_exceptions: (experimental) Whether to retry on Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException`` and ``Lambda.SdkClientException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true
        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None

        :stability: experimental
        '''
        props = LambdaInvokeProps(
            lambda_function=lambda_function,
            client_context=client_context,
            invocation_type=invocation_type,
            payload=payload,
            payload_response_only=payload_response_only,
            qualifier=qualifier,
            retry_on_service_exceptions=retry_on_service_exceptions,
            comment=comment,
            heartbeat=heartbeat,
            input_path=input_path,
            integration_pattern=integration_pattern,
            output_path=output_path,
            result_path=result_path,
            result_selector=result_selector,
            timeout=timeout,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskMetrics")
    def _task_metrics(self) -> typing.Optional[_TaskMetricsConfig_5aaea59e]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[_TaskMetricsConfig_5aaea59e], jsii.get(self, "taskMetrics"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskPolicies")
    def _task_policies(self) -> typing.Optional[typing.List[_PolicyStatement_296fe8a3]]:
        '''
        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List[_PolicyStatement_296fe8a3]], jsii.get(self, "taskPolicies"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.LambdaInvokeProps",
    jsii_struct_bases=[_TaskStateBaseProps_b13f6332],
    name_mapping={
        "comment": "comment",
        "heartbeat": "heartbeat",
        "input_path": "inputPath",
        "integration_pattern": "integrationPattern",
        "output_path": "outputPath",
        "result_path": "resultPath",
        "result_selector": "resultSelector",
        "timeout": "timeout",
        "lambda_function": "lambdaFunction",
        "client_context": "clientContext",
        "invocation_type": "invocationType",
        "payload": "payload",
        "payload_response_only": "payloadResponseOnly",
        "qualifier": "qualifier",
        "retry_on_service_exceptions": "retryOnServiceExceptions",
    },
)
class LambdaInvokeProps(_TaskStateBaseProps_b13f6332):
    def __init__(
        self,
        *,
        comment: typing.Optional[builtins.str] = None,
        heartbeat: typing.Optional[_Duration_070aa057] = None,
        input_path: typing.Optional[builtins.str] = None,
        integration_pattern: typing.Optional[_IntegrationPattern_fbb35786] = None,
        output_path: typing.Optional[builtins.str] = None,
        result_path: typing.Optional[builtins.str] = None,
        result_selector: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        lambda_function: _IFunction_6e14f09e,
        client_context: typing.Optional[builtins.str] = None,
        invocation_type: typing.Optional[LambdaInvocationType] = None,
        payload: typing.Optional[_TaskInput_71ab46f6] = None,
        payload_response_only: typing.Optional[builtins.bool] = None,
        qualifier: typing.Optional[builtins.str] = None,
        retry_on_service_exceptions: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Properties for invoking a Lambda function with LambdaInvoke.

        :param comment: (experimental) An optional description for this state. Default: - No comment
        :param heartbeat: (experimental) Timeout for the heartbeat. Default: - None
        :param input_path: (experimental) JSONPath expression to select part of the state to be the input to this state. May also be the special value JsonPath.DISCARD, which will cause the effective input to be the empty object {}. Default: - The entire task input (JSON path '$')
        :param integration_pattern: (experimental) AWS Step Functions integrates with services directly in the Amazon States Language. You can control these AWS services using service integration patterns Default: IntegrationPattern.REQUEST_RESPONSE
        :param output_path: (experimental) JSONPath expression to select select a portion of the state output to pass to the next state. May also be the special value JsonPath.DISCARD, which will cause the effective output to be the empty object {}. Default: - The entire JSON node determined by the state input, the task result, and resultPath is passed to the next state (JSON path '$')
        :param result_path: (experimental) JSONPath expression to indicate where to inject the state's output. May also be the special value JsonPath.DISCARD, which will cause the state's input to become its output. Default: - Replaces the entire input with the result (JSON path '$')
        :param result_selector: (experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied. You can use ResultSelector to create a payload with values that are static or selected from the state's raw result. Default: - None
        :param timeout: (experimental) Timeout for the state machine. Default: - None
        :param lambda_function: (experimental) Lambda function to invoke.
        :param client_context: (experimental) Up to 3583 bytes of base64-encoded data about the invoking client to pass to the function. Default: - No context
        :param invocation_type: (experimental) Invocation type of the Lambda function. Default: InvocationType.REQUEST_RESPONSE
        :param payload: (experimental) The JSON that will be supplied as input to the Lambda function. Default: - The state input (JSON path '$')
        :param payload_response_only: (experimental) Invoke the Lambda in a way that only returns the payload response without additional metadata. The ``payloadResponseOnly`` property cannot be used if ``integrationPattern``, ``invocationType``, ``clientContext``, or ``qualifier`` are specified. It always uses the REQUEST_RESPONSE behavior. Default: false
        :param qualifier: (experimental) Version or alias to invoke a published version of the function. You only need to supply this if you want the version of the Lambda Function to depend on data in the state machine state. If not, you can pass the appropriate Alias or Version object directly as the ``lambdaFunction`` argument. Default: - Version or alias inherent to the ``lambdaFunction`` object.
        :param retry_on_service_exceptions: (experimental) Whether to retry on Lambda service exceptions. This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException`` and ``Lambda.SdkClientException`` with an interval of 2 seconds, a back-off rate of 2 and 6 maximum attempts. Default: true

        :stability: experimental
        :exampleMetadata: infused

        Example::

            import monocdk as lambda_
            
            # order_fn: lambda.Function
            
            
            submit_job = tasks.LambdaInvoke(self, "InvokeOrderProcessor",
                lambda_function=order_fn,
                payload=sfn.TaskInput.from_object({
                    "OrderId": sfn.JsonPath.string_at("$.OrderId")
                })
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "lambda_function": lambda_function,
        }
        if comment is not None:
            self._values["comment"] = comment
        if heartbeat is not None:
            self._values["heartbeat"] = heartbeat
        if input_path is not None:
            self._values["input_path"] = input_path
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if output_path is not None:
            self._values["output_path"] = output_path
        if result_path is not None:
            self._values["result_path"] = result_path
        if result_selector is not None:
            self._values["result_selector"] = result_selector
        if timeout is not None:
            self._values["timeout"] = timeout
        if client_context is not None:
            self._values["client_context"] = client_context
        if invocation_type is not None:
            self._values["invocation_type"] = invocation_type
        if payload is not None:
            self._values["payload"] = payload
        if payload_response_only is not None:
            self._values["payload_response_only"] = payload_response_only
        if qualifier is not None:
            self._values["qualifier"] = qualifier
        if retry_on_service_exceptions is not None:
            self._values["retry_on_service_exceptions"] = retry_on_service_exceptions

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(experimental) An optional description for this state.

        :default: - No comment

        :stability: experimental
        '''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def heartbeat(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the heartbeat.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("heartbeat")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def input_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select part of the state to be the input to this state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        input to be the empty object {}.

        :default: - The entire task input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("input_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def integration_pattern(self) -> typing.Optional[_IntegrationPattern_fbb35786]:
        '''(experimental) AWS Step Functions integrates with services directly in the Amazon States Language.

        You can control these AWS services using service integration patterns

        :default: IntegrationPattern.REQUEST_RESPONSE

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#connect-wait-token
        :stability: experimental
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_IntegrationPattern_fbb35786], result)

    @builtins.property
    def output_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to select select a portion of the state output to pass to the next state.

        May also be the special value JsonPath.DISCARD, which will cause the effective
        output to be the empty object {}.

        :default:

        - The entire JSON node determined by the state input, the task result,
        and resultPath is passed to the next state (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("output_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_path(self) -> typing.Optional[builtins.str]:
        '''(experimental) JSONPath expression to indicate where to inject the state's output.

        May also be the special value JsonPath.DISCARD, which will cause the state's
        input to become its output.

        :default: - Replaces the entire input with the result (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("result_path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def result_selector(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(experimental) The JSON that will replace the state's raw result and become the effective result before ResultPath is applied.

        You can use ResultSelector to create a payload with values that are static
        or selected from the state's raw result.

        :default: - None

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/input-output-inputpath-params.html#input-output-resultselector
        :stability: experimental
        '''
        result = self._values.get("result_selector")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Timeout for the state machine.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def lambda_function(self) -> _IFunction_6e14f09e:
        '''(experimental) Lambda function to invoke.

        :stability: experimental
        '''
        result = self._values.get("lambda_function")
        assert result is not None, "Required property 'lambda_function' is missing"
        return typing.cast(_IFunction_6e14f09e, result)

    @builtins.property
    def client_context(self) -> typing.Optional[builtins.str]:
        '''(experimental) Up to 3583 bytes of base64-encoded data about the invoking client to pass to the function.

        :default: - No context

        :stability: experimental
        '''
        result = self._values.get("client_context")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def invocation_type(self) -> typing.Optional[LambdaInvocationType]:
        '''(experimental) Invocation type of the Lambda function.

        :default: InvocationType.REQUEST_RESPONSE

        :stability: experimental
        '''
        result = self._values.get("invocation_type")
        return typing.cast(typing.Optional[LambdaInvocationType], result)

    @builtins.property
    def payload(self) -> typing.Optional[_TaskInput_71ab46f6]:
        '''(experimental) The JSON that will be supplied as input to the Lambda function.

        :default: - The state input (JSON path '$')

        :stability: experimental
        '''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[_TaskInput_71ab46f6], result)

    @builtins.property
    def payload_response_only(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Invoke the Lambda in a way that only returns the payload response without additional metadata.

        The ``payloadResponseOnly`` property cannot be used if ``integrationPattern``, ``invocationType``,
        ``clientContext``, or ``qualifier`` are specified.
        It always uses the REQUEST_RESPONSE behavior.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("payload_response_only")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def qualifier(self) -> typing.Optional[builtins.str]:
        '''(experimental) Version or alias to invoke a published version of the function.

        You only need to supply this if you want the version of the Lambda Function to depend
        on data in the state machine state. If not, you can pass the appropriate Alias or Version object
        directly as the ``lambdaFunction`` argument.

        :default: - Version or alias inherent to the ``lambdaFunction`` object.

        :stability: experimental
        '''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def retry_on_service_exceptions(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to retry on Lambda service exceptions.

        This handles ``Lambda.ServiceException``, ``Lambda.AWSLambdaException`` and
        ``Lambda.SdkClientException`` with an interval of 2 seconds, a back-off rate
        of 2 and 6 maximum attempts.

        :default: true

        :see: https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
        :stability: experimental
        '''
        result = self._values.get("retry_on_service_exceptions")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LambdaInvokeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.LaunchTargetBindOptions",
    jsii_struct_bases=[],
    name_mapping={"task_definition": "taskDefinition", "cluster": "cluster"},
)
class LaunchTargetBindOptions:
    def __init__(
        self,
        *,
        task_definition: _ITaskDefinition_ee0d1862,
        cluster: typing.Optional[_ICluster_42c4ec1a] = None,
    ) -> None:
        '''(experimental) Options for binding a launch target to an ECS run job task.

        :param task_definition: (experimental) Task definition to run Docker containers in Amazon ECS.
        :param cluster: (experimental) A regional grouping of one or more container instances on which you can run tasks and services. Default: - No cluster

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # cluster: ecs.Cluster
            # task_definition: ecs.TaskDefinition
            
            launch_target_bind_options = stepfunctions_tasks.LaunchTargetBindOptions(
                task_definition=task_definition,
            
                # the properties below are optional
                cluster=cluster
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "task_definition": task_definition,
        }
        if cluster is not None:
            self._values["cluster"] = cluster

    @builtins.property
    def task_definition(self) -> _ITaskDefinition_ee0d1862:
        '''(experimental) Task definition to run Docker containers in Amazon ECS.

        :stability: experimental
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(_ITaskDefinition_ee0d1862, result)

    @builtins.property
    def cluster(self) -> typing.Optional[_ICluster_42c4ec1a]:
        '''(experimental) A regional grouping of one or more container instances on which you can run tasks and services.

        :default: - No cluster

        :stability: experimental
        '''
        result = self._values.get("cluster")
        return typing.cast(typing.Optional[_ICluster_42c4ec1a], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "LaunchTargetBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.MessageAttribute",
    jsii_struct_bases=[],
    name_mapping={"value": "value", "data_type": "dataType"},
)
class MessageAttribute:
    def __init__(
        self,
        *,
        value: typing.Any,
        data_type: typing.Optional["MessageAttributeDataType"] = None,
    ) -> None:
        '''(experimental) A message attribute to add to the SNS message.

        :param value: (experimental) The value of the attribute.
        :param data_type: (experimental) The data type for the attribute. Default: determined by type inspection if possible, fallback is String

        :see: https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html
        :stability: experimental
        :exampleMetadata: infused

        Example::

            topic = sns.Topic(self, "Topic")
            
            # Use a field from the execution data as message.
            task1 = tasks.SnsPublish(self, "Publish1",
                topic=topic,
                integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
                message=sfn.TaskInput.from_data_at("$.state.message"),
                message_attributes={
                    "place": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                        value=sfn.JsonPath.string_at("$.place")
                    ),
                    "pic": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                        # BINARY must be explicitly set
                        data_type=tasks.MessageAttributeDataType.BINARY,
                        value=sfn.JsonPath.string_at("$.pic")
                    ),
                    "people": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                        value=4
                    ),
                    "handles": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                        value=["@kslater", "@jjf", null, "@mfanning"]
                    )
                }
            )
            
            # Combine a field from the execution data with
            # a literal object.
            task2 = tasks.SnsPublish(self, "Publish2",
                topic=topic,
                message=sfn.TaskInput.from_object({
                    "field1": "somedata",
                    "field2": sfn.JsonPath.string_at("$.field2")
                })
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "value": value,
        }
        if data_type is not None:
            self._values["data_type"] = data_type

    @builtins.property
    def value(self) -> typing.Any:
        '''(experimental) The value of the attribute.

        :stability: experimental
        '''
        result = self._values.get("value")
        assert result is not None, "Required property 'value' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def data_type(self) -> typing.Optional["MessageAttributeDataType"]:
        '''(experimental) The data type for the attribute.

        :default: determined by type inspection if possible, fallback is String

        :see: https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
        :stability: experimental
        '''
        result = self._values.get("data_type")
        return typing.cast(typing.Optional["MessageAttributeDataType"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MessageAttribute(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.MessageAttributeDataType")
class MessageAttributeDataType(enum.Enum):
    '''(experimental) The data type set for the SNS message attributes.

    :see: https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
    :stability: experimental
    :exampleMetadata: infused

    Example::

        topic = sns.Topic(self, "Topic")
        
        # Use a field from the execution data as message.
        task1 = tasks.SnsPublish(self, "Publish1",
            topic=topic,
            integration_pattern=sfn.IntegrationPattern.REQUEST_RESPONSE,
            message=sfn.TaskInput.from_data_at("$.state.message"),
            message_attributes={
                "place": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                    value=sfn.JsonPath.string_at("$.place")
                ),
                "pic": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                    # BINARY must be explicitly set
                    data_type=tasks.MessageAttributeDataType.BINARY,
                    value=sfn.JsonPath.string_at("$.pic")
                ),
                "people": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                    value=4
                ),
                "handles": dynamodb.aws_stepfunctions_tasks.MessageAttribute(
                    value=["@kslater", "@jjf", null, "@mfanning"]
                )
            }
        )
        
        # Combine a field from the execution data with
        # a literal object.
        task2 = tasks.SnsPublish(self, "Publish2",
            topic=topic,
            message=sfn.TaskInput.from_object({
                "field1": "somedata",
                "field2": sfn.JsonPath.string_at("$.field2")
            })
        )
    '''

    STRING = "STRING"
    '''(experimental) Strings are Unicode with UTF-8 binary encoding.

    :stability: experimental
    '''
    STRING_ARRAY = "STRING_ARRAY"
    '''(experimental) An array, formatted as a string.

    :see: https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
    :stability: experimental
    '''
    NUMBER = "NUMBER"
    '''(experimental) Numbers are positive or negative integers or floating-point numbers.

    :see: https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
    :stability: experimental
    '''
    BINARY = "BINARY"
    '''(experimental) Binary type attributes can store any binary data.

    :see: https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.MetricDefinition",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "regex": "regex"},
)
class MetricDefinition:
    def __init__(self, *, name: builtins.str, regex: builtins.str) -> None:
        '''(experimental) Specifies the metric name and regular expressions used to parse algorithm logs.

        :param name: (experimental) Name of the metric.
        :param regex: (experimental) Regular expression that searches the output of a training job and gets the value of the metric.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            metric_definition = stepfunctions_tasks.MetricDefinition(
                name="name",
                regex="regex"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
            "regex": regex,
        }

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) Name of the metric.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def regex(self) -> builtins.str:
        '''(experimental) Regular expression that searches the output of a training job and gets the value of the metric.

        :stability: experimental
        '''
        result = self._values.get("regex")
        assert result is not None, "Required property 'regex' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "MetricDefinition(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.Mode")
class Mode(enum.Enum):
    '''(experimental) Specifies how many models the container hosts.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.SageMakerCreateModel(self, "Sagemaker",
            model_name="MyModel",
            primary_container=tasks.ContainerDefinition(
                image=tasks.DockerImage.from_json_expression(sfn.JsonPath.string_at("$.Model.imageName")),
                mode=tasks.Mode.SINGLE_MODEL,
                model_s3_location=tasks.S3Location.from_json_expression("$.TrainingJob.ModelArtifacts.S3ModelArtifacts")
            )
        )
    '''

    SINGLE_MODEL = "SINGLE_MODEL"
    '''(experimental) Container hosts a single model.

    :stability: experimental
    '''
    MULTI_MODEL = "MULTI_MODEL"
    '''(experimental) Container hosts multiple models.

    :see: https://docs.aws.amazon.com/sagemaker/latest/dg/multi-model-endpoints.html
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ModelClientOptions",
    jsii_struct_bases=[],
    name_mapping={
        "invocations_max_retries": "invocationsMaxRetries",
        "invocations_timeout": "invocationsTimeout",
    },
)
class ModelClientOptions:
    def __init__(
        self,
        *,
        invocations_max_retries: typing.Optional[jsii.Number] = None,
        invocations_timeout: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Configures the timeout and maximum number of retries for processing a transform job invocation.

        :param invocations_max_retries: (experimental) The maximum number of retries when invocation requests are failing. Default: 0
        :param invocations_timeout: (experimental) The timeout duration for an invocation request. Default: Duration.minutes(1)

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTransformJob(self, "Batch Inference",
                transform_job_name="MyTransformJob",
                model_name="MyModelName",
                model_client_options=dynamodb.aws_stepfunctions_tasks.ModelClientOptions(
                    invocations_max_retries=3,  # default is 0
                    invocations_timeout=Duration.minutes(5)
                ),
                transform_input=dynamodb.aws_stepfunctions_tasks.TransformInput(
                    transform_data_source=dynamodb.aws_stepfunctions_tasks.TransformDataSource(
                        s3_data_source=dynamodb.aws_stepfunctions_tasks.TransformS3DataSource(
                            s3_uri="s3://inputbucket/train",
                            s3_data_type=tasks.S3DataType.S3_PREFIX
                        )
                    )
                ),
                transform_output=dynamodb.aws_stepfunctions_tasks.TransformOutput(
                    s3_output_path="s3://outputbucket/TransformJobOutputPath"
                ),
                transform_resources=dynamodb.aws_stepfunctions_tasks.TransformResources(
                    instance_count=1,
                    instance_type=ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE)
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if invocations_max_retries is not None:
            self._values["invocations_max_retries"] = invocations_max_retries
        if invocations_timeout is not None:
            self._values["invocations_timeout"] = invocations_timeout

    @builtins.property
    def invocations_max_retries(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of retries when invocation requests are failing.

        :default: 0

        :stability: experimental
        '''
        result = self._values.get("invocations_max_retries")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def invocations_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The timeout duration for an invocation request.

        :default: Duration.minutes(1)

        :stability: experimental
        '''
        result = self._values.get("invocations_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ModelClientOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.Monitoring",
    jsii_struct_bases=[],
    name_mapping={
        "log_bucket": "logBucket",
        "logging": "logging",
        "log_group": "logGroup",
        "log_stream_name_prefix": "logStreamNamePrefix",
        "persistent_app_ui": "persistentAppUI",
    },
)
class Monitoring:
    def __init__(
        self,
        *,
        log_bucket: typing.Optional[_IBucket_73486e29] = None,
        logging: typing.Optional[builtins.bool] = None,
        log_group: typing.Optional[_ILogGroup_846e17a0] = None,
        log_stream_name_prefix: typing.Optional[builtins.str] = None,
        persistent_app_ui: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) Configuration setting for monitoring.

        :param log_bucket: (experimental) Amazon S3 Bucket for monitoring log publishing. You can configure your jobs to send log information to Amazon S3. Default: - if ``logging`` is manually set to ``true`` and a ``logBucket`` is not provided, a ``logBucket`` will be automatically generated`.
        :param logging: (experimental) Enable logging for this job. If set to true, will automatically create a Cloudwatch Log Group and S3 bucket. This will be set to ``true`` implicitly if values are provided for ``logGroup`` or ``logBucket``. Default: true - true if values are provided for ``logGroup`` or ``logBucket``, false otherwise
        :param log_group: (experimental) A log group for CloudWatch monitoring. You can configure your jobs to send log information to CloudWatch Logs. Default: - if ``logging`` is manually set to ``true`` and a ``logGroup`` is not provided, a ``logGroup`` will be automatically generated`.
        :param log_stream_name_prefix: (experimental) A log stream name prefix for Cloudwatch monitoring. Default: - Log streams created in this log group have no default prefix
        :param persistent_app_ui: (experimental) Monitoring configurations for the persistent application UI. Default: true

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
                virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
                release_label=tasks.ReleaseLabel.EMR_6_2_0,
                job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
                    spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
                        entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py"),
                        spark_submit_parameters="--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.executor.cores=2 --conf spark.driver.cores=1"
                    )
                ),
                monitoring=dynamodb.aws_stepfunctions_tasks.Monitoring(
                    logging=True
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if log_bucket is not None:
            self._values["log_bucket"] = log_bucket
        if logging is not None:
            self._values["logging"] = logging
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_stream_name_prefix is not None:
            self._values["log_stream_name_prefix"] = log_stream_name_prefix
        if persistent_app_ui is not None:
            self._values["persistent_app_ui"] = persistent_app_ui

    @builtins.property
    def log_bucket(self) -> typing.Optional[_IBucket_73486e29]:
        '''(experimental) Amazon S3 Bucket for monitoring log publishing.

        You can configure your jobs to send log information to Amazon S3.

        :default: - if ``logging`` is manually set to ``true`` and a ``logBucket`` is not provided, a ``logBucket`` will be automatically generated`.

        :stability: experimental
        '''
        result = self._values.get("log_bucket")
        return typing.cast(typing.Optional[_IBucket_73486e29], result)

    @builtins.property
    def logging(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Enable logging for this job.

        If set to true, will automatically create a Cloudwatch Log Group and S3 bucket.
        This will be set to ``true`` implicitly if values are provided for ``logGroup`` or ``logBucket``.

        :default: true - true if values are provided for ``logGroup`` or ``logBucket``, false otherwise

        :stability: experimental
        '''
        result = self._values.get("logging")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_ILogGroup_846e17a0]:
        '''(experimental) A log group for CloudWatch monitoring.

        You can configure your jobs to send log information to CloudWatch Logs.

        :default: - if ``logging`` is manually set to ``true`` and a ``logGroup`` is not provided, a ``logGroup`` will be automatically generated`.

        :stability: experimental
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_ILogGroup_846e17a0], result)

    @builtins.property
    def log_stream_name_prefix(self) -> typing.Optional[builtins.str]:
        '''(experimental) A log stream name prefix for Cloudwatch monitoring.

        :default: - Log streams created in this log group have no default prefix

        :stability: experimental
        '''
        result = self._values.get("log_stream_name_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def persistent_app_ui(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Monitoring configurations for the persistent application UI.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("persistent_app_ui")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Monitoring(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.OutputDataConfig",
    jsii_struct_bases=[],
    name_mapping={
        "s3_output_location": "s3OutputLocation",
        "encryption_key": "encryptionKey",
    },
)
class OutputDataConfig:
    def __init__(
        self,
        *,
        s3_output_location: "S3Location",
        encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''(experimental) Configures the S3 bucket where SageMaker will save the result of model training.

        :param s3_output_location: (experimental) Identifies the S3 path where you want Amazon SageMaker to store the model artifacts.
        :param encryption_key: (experimental) Optional KMS encryption key that Amazon SageMaker uses to encrypt the model artifacts at rest using Amazon S3 server-side encryption. Default: - Amazon SageMaker uses the default KMS key for Amazon S3 for your role's account

        :stability: experimental
        :exampleMetadata: infused

        Example::

            tasks.SageMakerCreateTrainingJob(self, "TrainSagemaker",
                training_job_name=sfn.JsonPath.string_at("$.JobName"),
                algorithm_specification=dynamodb.aws_stepfunctions_tasks.AlgorithmSpecification(
                    algorithm_name="BlazingText",
                    training_input_mode=tasks.InputMode.FILE
                ),
                input_data_config=[dynamodb.aws_stepfunctions_tasks.Channel(
                    channel_name="train",
                    data_source=dynamodb.aws_stepfunctions_tasks.DataSource(
                        s3_data_source=dynamodb.aws_stepfunctions_tasks.S3DataSource(
                            s3_data_type=tasks.S3DataType.S3_PREFIX,
                            s3_location=tasks.S3Location.from_json_expression("$.S3Bucket")
                        )
                    )
                )],
                output_data_config=dynamodb.aws_stepfunctions_tasks.OutputDataConfig(
                    s3_output_location=tasks.S3Location.from_bucket(s3.Bucket.from_bucket_name(self, "Bucket", "mybucket"), "myoutputpath")
                ),
                resource_config=dynamodb.aws_stepfunctions_tasks.ResourceConfig(
                    instance_count=1,
                    instance_type=ec2.InstanceType(sfn.JsonPath.string_at("$.InstanceType")),
                    volume_size=Size.gibibytes(50)
                ),  # optional: default is 1 instance of EC2 `M4.XLarge` with `10GB` volume
                stopping_condition=dynamodb.aws_stepfunctions_tasks.StoppingCondition(
                    max_runtime=Duration.hours(2)
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "s3_output_location": s3_output_location,
        }
        if encryption_key is not None:
            self._values["encryption_key"] = encryption_key

    @builtins.property
    def s3_output_location(self) -> "S3Location":
        '''(experimental) Identifies the S3 path where you want Amazon SageMaker to store the model artifacts.

        :stability: experimental
        '''
        result = self._values.get("s3_output_location")
        assert result is not None, "Required property 's3_output_location' is missing"
        return typing.cast("S3Location", result)

    @builtins.property
    def encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) Optional KMS encryption key that Amazon SageMaker uses to encrypt the model artifacts at rest using Amazon S3 server-side encryption.

        :default: - Amazon SageMaker uses the default KMS key for Amazon S3 for your role's account

        :stability: experimental
        '''
        result = self._values.get("encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "OutputDataConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ProductionVariant",
    jsii_struct_bases=[],
    name_mapping={
        "instance_type": "instanceType",
        "model_name": "modelName",
        "variant_name": "variantName",
        "accelerator_type": "acceleratorType",
        "initial_instance_count": "initialInstanceCount",
        "initial_variant_weight": "initialVariantWeight",
    },
)
class ProductionVariant:
    def __init__(
        self,
        *,
        instance_type: _InstanceType_072ad323,
        model_name: builtins.str,
        variant_name: builtins.str,
        accelerator_type: typing.Optional[AcceleratorType] = None,
        initial_instance_count: typing.Optional[jsii.Number] = None,
        initial_variant_weight: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Identifies a model that you want to host and the resources to deploy for hosting it.

        :param instance_type: (experimental) The ML compute instance type.
        :param model_name: (experimental) The name of the model that you want to host. This is the name that you specified when creating the model.
        :param variant_name: (experimental) The name of the production variant.
        :param accelerator_type: (experimental) The size of the Elastic Inference (EI) instance to use for the production variant. Default: - None
        :param initial_instance_count: (experimental) Number of instances to launch initially. Default: - 1
        :param initial_variant_weight: (experimental) Determines initial traffic distribution among all of the models that you specify in the endpoint configuration. Default: - 1.0

        :see: https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_ProductionVariant.html
        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ec2 as ec2
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # accelerator_type: stepfunctions_tasks.AcceleratorType
            # instance_type: ec2.InstanceType
            
            production_variant = stepfunctions_tasks.ProductionVariant(
                instance_type=instance_type,
                model_name="modelName",
                variant_name="variantName",
            
                # the properties below are optional
                accelerator_type=accelerator_type,
                initial_instance_count=123,
                initial_variant_weight=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "instance_type": instance_type,
            "model_name": model_name,
            "variant_name": variant_name,
        }
        if accelerator_type is not None:
            self._values["accelerator_type"] = accelerator_type
        if initial_instance_count is not None:
            self._values["initial_instance_count"] = initial_instance_count
        if initial_variant_weight is not None:
            self._values["initial_variant_weight"] = initial_variant_weight

    @builtins.property
    def instance_type(self) -> _InstanceType_072ad323:
        '''(experimental) The ML compute instance type.

        :stability: experimental
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(_InstanceType_072ad323, result)

    @builtins.property
    def model_name(self) -> builtins.str:
        '''(experimental) The name of the model that you want to host.

        This is the name that you specified when creating the model.

        :stability: experimental
        '''
        result = self._values.get("model_name")
        assert result is not None, "Required property 'model_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def variant_name(self) -> builtins.str:
        '''(experimental) The name of the production variant.

        :stability: experimental
        '''
        result = self._values.get("variant_name")
        assert result is not None, "Required property 'variant_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def accelerator_type(self) -> typing.Optional[AcceleratorType]:
        '''(experimental) The size of the Elastic Inference (EI) instance to use for the production variant.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("accelerator_type")
        return typing.cast(typing.Optional[AcceleratorType], result)

    @builtins.property
    def initial_instance_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Number of instances to launch initially.

        :default: - 1

        :stability: experimental
        '''
        result = self._values.get("initial_instance_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def initial_variant_weight(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Determines initial traffic distribution among all of the models that you specify in the endpoint configuration.

        :default: - 1.0

        :stability: experimental
        '''
        result = self._values.get("initial_variant_weight")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ProductionVariant(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IStepFunctionsTask_82eb09ab)
class PublishToTopic(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.PublishToTopic",
):
    '''(deprecated) A Step Functions Task to publish messages to SNS topic.

    A Function can be used directly as a Resource, but this class mirrors
    integration with other AWS services via a specific class instance.

    :deprecated: Use ``SnsPublish``

    :stability: deprecated
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_sns as sns
        from monocdk import aws_stepfunctions as stepfunctions
        from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
        
        # task_input: stepfunctions.TaskInput
        # topic: sns.Topic
        
        publish_to_topic = stepfunctions_tasks.PublishToTopic(topic,
            message=task_input,
        
            # the properties below are optional
            integration_pattern=stepfunctions.ServiceIntegrationPattern.FIRE_AND_FORGET,
            message_per_subscription_type=False,
            subject="subject"
        )
    '''

    def __init__(
        self,
        topic: _ITopic_465e36b9,
        *,
        message: _TaskInput_71ab46f6,
        integration_pattern: typing.Optional[_ServiceIntegrationPattern_5581ee88] = None,
        message_per_subscription_type: typing.Optional[builtins.bool] = None,
        subject: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param topic: -
        :param message: (deprecated) The text message to send to the topic.
        :param integration_pattern: (deprecated) The service integration pattern indicates different ways to call Publish to SNS. The valid value is either FIRE_AND_FORGET or WAIT_FOR_TASK_TOKEN. Default: FIRE_AND_FORGET
        :param message_per_subscription_type: (deprecated) If true, send a different message to every subscription type. If this is set to true, message must be a JSON object with a "default" key and a key for every subscription type (such as "sqs", "email", etc.) The values are strings representing the messages being sent to every subscription type. Default: false
        :param subject: (deprecated) Used as the "Subject" line when the message is delivered to email endpoints. Also included, if present, in the standard JSON messages delivered to other endpoints. Default: - No subject

        :stability: deprecated
        '''
        props = PublishToTopicProps(
            message=message,
            integration_pattern=integration_pattern,
            message_per_subscription_type=message_per_subscription_type,
            subject=subject,
        )

        jsii.create(self.__class__, self, [topic, props])

    @jsii.member(jsii_name="bind")
    def bind(self, _task: _Task_747ca4f2) -> _StepFunctionsTaskConfig_fa131821:
        '''(deprecated) Called when the task object is used in a workflow.

        :param _task: -

        :stability: deprecated
        '''
        return typing.cast(_StepFunctionsTaskConfig_fa131821, jsii.invoke(self, "bind", [_task]))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.PublishToTopicProps",
    jsii_struct_bases=[],
    name_mapping={
        "message": "message",
        "integration_pattern": "integrationPattern",
        "message_per_subscription_type": "messagePerSubscriptionType",
        "subject": "subject",
    },
)
class PublishToTopicProps:
    def __init__(
        self,
        *,
        message: _TaskInput_71ab46f6,
        integration_pattern: typing.Optional[_ServiceIntegrationPattern_5581ee88] = None,
        message_per_subscription_type: typing.Optional[builtins.bool] = None,
        subject: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(deprecated) Properties for PublishTask.

        :param message: (deprecated) The text message to send to the topic.
        :param integration_pattern: (deprecated) The service integration pattern indicates different ways to call Publish to SNS. The valid value is either FIRE_AND_FORGET or WAIT_FOR_TASK_TOKEN. Default: FIRE_AND_FORGET
        :param message_per_subscription_type: (deprecated) If true, send a different message to every subscription type. If this is set to true, message must be a JSON object with a "default" key and a key for every subscription type (such as "sqs", "email", etc.) The values are strings representing the messages being sent to every subscription type. Default: false
        :param subject: (deprecated) Used as the "Subject" line when the message is delivered to email endpoints. Also included, if present, in the standard JSON messages delivered to other endpoints. Default: - No subject

        :deprecated: Use ``SnsPublish``

        :stability: deprecated
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_stepfunctions as stepfunctions
            from monocdk import aws_stepfunctions_tasks as stepfunctions_tasks
            
            # task_input: stepfunctions.TaskInput
            
            publish_to_topic_props = stepfunctions_tasks.PublishToTopicProps(
                message=task_input,
            
                # the properties below are optional
                integration_pattern=stepfunctions.ServiceIntegrationPattern.FIRE_AND_FORGET,
                message_per_subscription_type=False,
                subject="subject"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "message": message,
        }
        if integration_pattern is not None:
            self._values["integration_pattern"] = integration_pattern
        if message_per_subscription_type is not None:
            self._values["message_per_subscription_type"] = message_per_subscription_type
        if subject is not None:
            self._values["subject"] = subject

    @builtins.property
    def message(self) -> _TaskInput_71ab46f6:
        '''(deprecated) The text message to send to the topic.

        :stability: deprecated
        '''
        result = self._values.get("message")
        assert result is not None, "Required property 'message' is missing"
        return typing.cast(_TaskInput_71ab46f6, result)

    @builtins.property
    def integration_pattern(
        self,
    ) -> typing.Optional[_ServiceIntegrationPattern_5581ee88]:
        '''(deprecated) The service integration pattern indicates different ways to call Publish to SNS.

        The valid value is either FIRE_AND_FORGET or WAIT_FOR_TASK_TOKEN.

        :default: FIRE_AND_FORGET

        :stability: deprecated
        '''
        result = self._values.get("integration_pattern")
        return typing.cast(typing.Optional[_ServiceIntegrationPattern_5581ee88], result)

    @builtins.property
    def message_per_subscription_type(self) -> typing.Optional[builtins.bool]:
        '''(deprecated) If true, send a different message to every subscription type.

        If this is set to true, message must be a JSON object with a
        "default" key and a key for every subscription type (such as "sqs",
        "email", etc.) The values are strings representing the messages
        being sent to every subscription type.

        :default: false

        :see: https://docs.aws.amazon.com/sns/latest/api/API_Publish.html#API_Publish_RequestParameters
        :stability: deprecated
        '''
        result = self._values.get("message_per_subscription_type")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def subject(self) -> typing.Optional[builtins.str]:
        '''(deprecated) Used as the "Subject" line when the message is delivered to email endpoints.

        Also included, if present, in the standard JSON messages delivered to other endpoints.

        :default: - No subject

        :stability: deprecated
        '''
        result = self._values.get("subject")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PublishToTopicProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.QueryExecutionContext",
    jsii_struct_bases=[],
    name_mapping={"catalog_name": "catalogName", "database_name": "databaseName"},
)
class QueryExecutionContext:
    def __init__(
        self,
        *,
        catalog_name: typing.Optional[builtins.str] = None,
        database_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Database and data catalog context in which the query execution occurs.

        :param catalog_name: (experimental) Name of catalog used in query execution. Default: - No catalog
        :param database_name: (experimental) Name of database used in query execution. Default: - No database

        :see: https://docs.aws.amazon.com/athena/latest/APIReference/API_QueryExecutionContext.html
        :stability: experimental
        :exampleMetadata: infused

        Example::

            start_query_execution_job = tasks.AthenaStartQueryExecution(self, "Athena Start Query",
                query_string=sfn.JsonPath.format("select contacts where year={};", sfn.JsonPath.string_at("$.year")),
                query_execution_context=dynamodb.aws_stepfunctions_tasks.QueryExecutionContext(
                    database_name="interactions"
                ),
                result_configuration=dynamodb.aws_stepfunctions_tasks.ResultConfiguration(
                    encryption_configuration=dynamodb.aws_stepfunctions_tasks.EncryptionConfiguration(
                        encryption_option=tasks.EncryptionOption.S3_MANAGED
                    ),
                    output_location=dynamodb.aws_s3.Location(
                        bucket_name="mybucket",
                        object_key="myprefix"
                    )
                ),
                integration_pattern=sfn.IntegrationPattern.RUN_JOB
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if catalog_name is not None:
            self._values["catalog_name"] = catalog_name
        if database_name is not None:
            self._values["database_name"] = database_name

    @builtins.property
    def catalog_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of catalog used in query execution.

        :default: - No catalog

        :stability: experimental
        '''
        result = self._values.get("catalog_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def database_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of database used in query execution.

        :default: - No database

        :stability: experimental
        '''
        result = self._values.get("database_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "QueryExecutionContext(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_stepfunctions_tasks.RecordWrapperType")
class RecordWrapperType(enum.Enum):
    '''(experimental) Define the format of the input data.

    :stability: experimental
    '''

    NONE = "NONE"
    '''(experimental) None record wrapper type.

    :stability: experimental
    '''
    RECORD_IO = "RECORD_IO"
    '''(experimental) RecordIO record wrapper type.

    :stability: experimental
    '''


class ReleaseLabel(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_stepfunctions_tasks.ReleaseLabel",
):
    '''(experimental) The Amazon EMR release version to use for the job run.

    Can be extended to include new EMR releases

    For example, ``new ReleaseLabel('emr-x.xx.x-latest');``

    :stability: experimental
    :exampleMetadata: infused

    Example::

        tasks.EmrContainersStartJobRun(self, "EMR Containers Start Job Run",
            virtual_cluster=tasks.VirtualClusterInput.from_virtual_cluster_id("de92jdei2910fwedz"),
            release_label=tasks.ReleaseLabel.EMR_6_2_0,
            job_name="EMR-Containers-Job",
            job_driver=dynamodb.aws_stepfunctions_tasks.JobDriver(
                spark_submit_job_driver=dynamodb.aws_stepfunctions_tasks.SparkSubmitJobDriver(
                    entry_point=sfn.TaskInput.from_text("local:///usr/lib/spark/examples/src/main/python/pi.py")
                )
            ),
            application_config=[dynamodb.aws_stepfunctions_tasks.ApplicationConfiguration(
                classification=tasks.Classification.SPARK_DEFAULTS,
                properties={
                    "spark.executor.instances": "1",
                    "spark.executor.memory": "512M"
                }
            )]
        )
    '''

    def __init__(self, label: builtins.str) -> None:
        '''(experimental) Initializes the label string.

        :param label: A literal string that contains the release-version ex. 'emr-x.x.x-latest'

        :stability: experimental
        '''
        jsii.create(self.__class__, self, [label])

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="EMR_5_32_0")
    def EMR_5_32_0(cls) -> "ReleaseLabel":
        '''(experimental) EMR Release version 5.32.0.

        :stability: experimental
        '''
        return typing.cast("ReleaseLabel", jsii.sget(cls, "EMR_5_32_0"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="EMR_5_33_0")
    def EMR_5_33_0(cls) -> "ReleaseLabel":
        '''(experimental) EMR Release version 5.33.0.

        :stability: experimental
        '''
        return typing.cast("ReleaseLabel", jsii.sget(cls, "EMR_5_33_0"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="EMR_6_2_0")
    def EMR_6_2_0(cls) -> "ReleaseLabel":
        '''(experimental) EMR Release version 6.2.0.

        :stability: experimental
        '''
        return typing.cast("ReleaseLabel", jsii.sget(cls, "EMR_6_2_0"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="EMR_6_3_0")
    def EMR_6_3_0(cls) -> "ReleaseLabel":
        '''(experimental) EMR Release version 6.3.0.

        :stability: experimental
        '''
        return typing.cast("ReleaseLabel", jsii.sget(cls, "EMR_6_3_0"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="label")
    def label(self) -> builtins.str:
        '''(experimental) A literal string that contains the release-version ex.

        'emr-x.x.x-latest'

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "label"))


@jsii.data_type(
    jsii_type="monocdk.aws_stepfunctions_tasks.ResourceConfig",
    jsii_struct_bases=[],
    name_mapping={
        "instance_count": "instanceCount",
        "instance_type": "instanceType",
        "volume_size": "volumeSize",
        "volume_encryption_key": "volumeEncryptionKey",
    },
)
class ResourceConfig:
    def __init__(
        self,
        *,
        instance_count: jsii.Number,
        instance_type: _InstanceType_072ad323,
        volume_size: _Size_7fbd4337,
        volume_encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''(experimental) Specifies the resources, ML compute instances, and ML storage volumes to deploy for model training.

        :param instance_count: (experimental) The number of ML compute instances to use. Default: 1 instance.
        :param instance_type: (experimental) ML compute instance type. To provide an instance type from the task input, supply an instance type in the following way where the value in the task input is an EC2 instance type prepended with "ml.":: new ec2.InstanceType(sfn.JsonPath.stringAt('$.path.to.instanceType')); Default: ec2.InstanceType(ec2.InstanceClass.M4, ec2.InstanceType.XLARGE)
        :param volume_size: (experimental) Size of the ML storage volume that you want to provision. Default: 