'''
# Amazon ECS Construct Library

This package contains constructs for working with **Amazon Elastic Container
Service** (Amazon ECS).

Amazon Elastic Container Service (Amazon ECS) is a fully managed container orchestration service.

For further information on Amazon ECS,
see the [Amazon ECS documentation](https://docs.aws.amazon.com/ecs)

The following example creates an Amazon ECS cluster, adds capacity to it, and
runs a service on it:

```python
# vpc: ec2.Vpc


# Create an ECS cluster
cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

# Add capacity to it
cluster.add_capacity("DefaultAutoScalingGroupCapacity",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3
)

task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")

task_definition.add_container("DefaultContainer",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=512
)

# Instantiate an Amazon ECS Service
ecs_service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition
)
```

For a set of constructs defining common ECS architectural patterns, see the `@aws-cdk/aws-ecs-patterns` package.

## Launch Types: AWS Fargate vs Amazon EC2

There are two sets of constructs in this library; one to run tasks on Amazon EC2 and
one to run tasks on AWS Fargate.

* Use the `Ec2TaskDefinition` and `Ec2Service` constructs to run tasks on Amazon EC2 instances running in your account.
* Use the `FargateTaskDefinition` and `FargateService` constructs to run tasks on
  instances that are managed for you by AWS.
* Use the `ExternalTaskDefinition` and `ExternalService` constructs to run AWS ECS Anywhere tasks on self-managed infrastructure.

Here are the main differences:

* **Amazon EC2**: instances are under your control. Complete control of task to host
  allocation. Required to specify at least a memory reservation or limit for
  every container. Can use Host, Bridge and AwsVpc networking modes. Can attach
  Classic Load Balancer. Can share volumes between container and host.
* **AWS Fargate**: tasks run on AWS-managed instances, AWS manages task to host
  allocation for you. Requires specification of memory and cpu sizes at the
  taskdefinition level. Only supports AwsVpc networking modes and
  Application/Network Load Balancers. Only the AWS log driver is supported.
  Many host features are not supported such as adding kernel capabilities
  and mounting host devices/volumes inside the container.
* **AWS ECSAnywhere**: tasks are run and managed by AWS ECS Anywhere on infrastructure owned by the customer. Only Bridge networking mode is supported. Does not support autoscaling, load balancing, cloudmap or attachment of volumes.

For more information on Amazon EC2 vs AWS Fargate, networking and ECS Anywhere see the AWS Documentation:
[AWS Fargate](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/AWS_Fargate.html),
[Task Networking](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html),
[ECS Anywhere](https://aws.amazon.com/ecs/anywhere/)

## Clusters

A `Cluster` defines the infrastructure to run your
tasks on. You can run many tasks on a single cluster.

The following code creates a cluster that can run AWS Fargate tasks:

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)
```

The following code imports an existing cluster using the ARN which can be used to
import an Amazon ECS service either EC2 or Fargate.

```python
cluster_arn = "arn:aws:ecs:us-east-1:012345678910:cluster/clusterName"

cluster = ecs.Cluster.from_cluster_arn(self, "Cluster", cluster_arn)
```

To use tasks with Amazon EC2 launch-type, you have to add capacity to
the cluster in order for tasks to be scheduled on your instances.  Typically,
you add an AutoScalingGroup with instances running the latest
Amazon ECS-optimized AMI to the cluster. There is a method to build and add such an
AutoScalingGroup automatically, or you can supply a customized AutoScalingGroup
that you construct yourself. It's possible to add multiple AutoScalingGroups
with various instance types.

The following example creates an Amazon ECS cluster and adds capacity to it:

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

# Either add default capacity
cluster.add_capacity("DefaultAutoScalingGroupCapacity",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3
)

# Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.xlarge"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux(),
    # Or use Amazon ECS-Optimized Amazon Linux 2 AMI
    # machineImage: EcsOptimizedImage.amazonLinux2(),
    desired_capacity=3
)

cluster.add_auto_scaling_group(auto_scaling_group)
```

If you omit the property `vpc`, the construct will create a new VPC with two AZs.

By default, all machine images will auto-update to the latest version
on each deployment, causing a replacement of the instances in your AutoScalingGroup
if the AMI has been updated since the last deployment.

If task draining is enabled, ECS will transparently reschedule tasks on to the new
instances before terminating your old instances. If you have disabled task draining,
the tasks will be terminated along with the instance. To prevent that, you
can pick a non-updating AMI by passing `cacheInContext: true`, but be sure
to periodically update to the latest AMI manually by using the [CDK CLI
context management commands](https://docs.aws.amazon.com/cdk/latest/guide/context.html):

```python
# vpc: ec2.Vpc

auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    machine_image=ecs.EcsOptimizedImage.amazon_linux(cached_in_context=True),
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.micro")
)
```

### Bottlerocket

[Bottlerocket](https://aws.amazon.com/bottlerocket/) is a Linux-based open source operating system that is
purpose-built by AWS for running containers. You can launch Amazon ECS container instances with the Bottlerocket AMI.

The following example will create a capacity with self-managed Amazon EC2 capacity of 2 `c5.large` Linux instances running with `Bottlerocket` AMI.

The following example adds Bottlerocket capacity to the cluster:

```python
# cluster: ecs.Cluster


cluster.add_capacity("bottlerocket-asg",
    min_capacity=2,
    instance_type=ec2.InstanceType("c5.large"),
    machine_image=ecs.BottleRocketImage()
)
```

### ARM64 (Graviton) Instances

To launch instances with ARM64 hardware, you can use the Amazon ECS-optimized
Amazon Linux 2 (arm64) AMI. Based on Amazon Linux 2, this AMI is recommended
for use when launching your EC2 instances that are powered by Arm-based AWS
Graviton Processors.

```python
# cluster: ecs.Cluster


cluster.add_capacity("graviton-cluster",
    min_capacity=2,
    instance_type=ec2.InstanceType("c6g.large"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.ARM)
)
```

Bottlerocket is also supported:

```python
# cluster: ecs.Cluster


cluster.add_capacity("graviton-cluster",
    min_capacity=2,
    instance_type=ec2.InstanceType("c6g.large"),
    machine_image_type=ecs.MachineImageType.BOTTLEROCKET
)
```

### Spot Instances

To add spot instances into the cluster, you must specify the `spotPrice` in the `ecs.AddCapacityOptions` and optionally enable the `spotInstanceDraining` property.

```python
# cluster: ecs.Cluster


# Add an AutoScalingGroup with spot instances to the existing cluster
cluster.add_capacity("AsgSpot",
    max_capacity=2,
    min_capacity=2,
    desired_capacity=2,
    instance_type=ec2.InstanceType("c5.xlarge"),
    spot_price="0.0735",
    # Enable the Automated Spot Draining support for Amazon ECS
    spot_instance_draining=True
)
```

### SNS Topic Encryption

When the `ecs.AddCapacityOptions` that you provide has a non-zero `taskDrainTime` (the default) then an SNS topic and Lambda are created to ensure that the
cluster's instances have been properly drained of tasks before terminating. The SNS Topic is sent the instance-terminating lifecycle event from the AutoScalingGroup,
and the Lambda acts on that event. If you wish to engage [server-side encryption](https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html) for this SNS Topic
then you may do so by providing a KMS key for the `topicEncryptionKey` property of `ecs.AddCapacityOptions`.

```python
# Given
# cluster: ecs.Cluster
# key: kms.Key

# Then, use that key to encrypt the lifecycle-event SNS Topic.
cluster.add_capacity("ASGEncryptedSNS",
    instance_type=ec2.InstanceType("t2.xlarge"),
    desired_capacity=3,
    topic_encryption_key=key
)
```

## Task definitions

A task definition describes what a single copy of a **task** should look like.
A task definition has one or more containers; typically, it has one
main container (the *default container* is the first one that's added
to the task definition, and it is marked *essential*) and optionally
some supporting containers which are used to support the main container,
doings things like upload logs or metrics to monitoring services.

To run a task or service with Amazon EC2 launch type, use the `Ec2TaskDefinition`. For AWS Fargate tasks/services, use the
`FargateTaskDefinition`. For AWS ECS Anywhere use the `ExternalTaskDefinition`. These classes
provide simplified APIs that only contain properties relevant for each specific launch type.

For a `FargateTaskDefinition`, specify the task size (`memoryLimitMiB` and `cpu`):

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
```

On Fargate Platform Version 1.4.0 or later, you may specify up to 200GiB of
[ephemeral storage](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-task-storage.html#fargate-task-storage-pv14):

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256,
    ephemeral_storage_gi_b=100
)
```

To add containers to a task definition, call `addContainer()`:

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
container = fargate_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample")
)
```

For a `Ec2TaskDefinition`:

```python
ec2_task_definition = ecs.Ec2TaskDefinition(self, "TaskDef",
    network_mode=ecs.NetworkMode.BRIDGE
)

container = ec2_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024
)
```

For an `ExternalTaskDefinition`:

```python
external_task_definition = ecs.ExternalTaskDefinition(self, "TaskDef")

container = external_task_definition.add_container("WebContainer",
    # Use an image from DockerHub
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024
)
```

You can specify container properties when you add them to the task definition, or with various methods, e.g.:

To add a port mapping when adding a container to the task definition, specify the `portMappings` option:

```python
# task_definition: ecs.TaskDefinition


task_definition.add_container("WebContainer",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    port_mappings=[autoscaling.aws_ecs.PortMapping(container_port=3000)]
)
```

To add port mappings directly to a container definition, call `addPortMappings()`:

```python
# container: ecs.ContainerDefinition


container.add_port_mappings(
    container_port=3000
)
```

To add data volumes to a task definition, call `addVolume()`:

```python
fargate_task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    memory_limit_mi_b=512,
    cpu=256
)
volume = {
    # Use an Elastic FileSystem
    "name": "mydatavolume",
    "efs_volume_configuration": {
        "file_system_id": "EFS"
    }
}

container = fargate_task_definition.add_volume(volume)
```

> Note: ECS Anywhere doesn't support volume attachments in the task definition.

To use a TaskDefinition that can be used with either Amazon EC2 or
AWS Fargate launch types, use the `TaskDefinition` construct.

When creating a task definition you have to specify what kind of
tasks you intend to run: Amazon EC2, AWS Fargate, or both.
The following example uses both:

```python
task_definition = ecs.TaskDefinition(self, "TaskDef",
    memory_mi_b="512",
    cpu="256",
    network_mode=ecs.NetworkMode.AWS_VPC,
    compatibility=ecs.Compatibility.EC2_AND_FARGATE
)
```

### Images

Images supply the software that runs inside the container. Images can be
obtained from either DockerHub or from ECR repositories, built directly from a local Dockerfile, or use an existing tarball.

* `ecs.ContainerImage.fromRegistry(imageName)`: use a public image.
* `ecs.ContainerImage.fromRegistry(imageName, { credentials: mySecret })`: use a private image that requires credentials.
* `ecs.ContainerImage.fromEcrRepository(repo, tag)`: use the given ECR repository as the image
  to start. If no tag is provided, "latest" is assumed.
* `ecs.ContainerImage.fromAsset('./image')`: build and upload an
  image directly from a `Dockerfile` in your source directory.
* `ecs.ContainerImage.fromDockerImageAsset(asset)`: uses an existing
  `@aws-cdk/aws-ecr-assets.DockerImageAsset` as a container image.
* `ecs.ContainerImage.fromTarball(file)`: use an existing tarball.
* `new ecs.TagParameterContainerImage(repository)`: use the given ECR repository as the image
  but a CloudFormation parameter as the tag.

### Environment variables

To pass environment variables to the container, you can use the `environment`, `environmentFiles`, and `secrets` props.

```python
# secret: secretsmanager.Secret
# db_secret: secretsmanager.Secret
# parameter: ssm.StringParameter
# task_definition: ecs.TaskDefinition
# s3_bucket: s3.Bucket


new_container = task_definition.add_container("container",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    environment={ # clear text, not for sensitive data
        "STAGE": "prod"},
    environment_files=[ # list of environment files hosted either on local disk or S3
        ecs.EnvironmentFile.from_asset("./demo-env-file.env"),
        ecs.EnvironmentFile.from_bucket(s3_bucket, "assets/demo-env-file.env")],
    secrets={ # Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store at container start-up.
        "SECRET": ecs.Secret.from_secrets_manager(secret),
        "DB_PASSWORD": ecs.Secret.from_secrets_manager(db_secret, "password"),  # Reference a specific JSON field, (requires platform version 1.4.0 or later for Fargate tasks)
        "API_KEY": ecs.Secret.from_secrets_manager_version(secret, autoscaling.aws_ecs.SecretVersionInfo(version_id="12345"), "apiKey"),  # Reference a specific version of the secret by its version id or version stage (requires platform version 1.4.0 or later for Fargate tasks)
        "PARAMETER": ecs.Secret.from_ssm_parameter(parameter)}
)
new_container.add_environment("QUEUE_NAME", "MyQueue")
```

The task execution role is automatically granted read permissions on the secrets/parameters. Support for environment
files is restricted to the EC2 launch type for files hosted on S3. Further details provided in the AWS documentation
about [specifying environment variables](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html).

### System controls

To set system controls (kernel parameters) on the container, use the `systemControls` prop:

```python
# task_definition: ecs.TaskDefinition


task_definition.add_container("container",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_limit_mi_b=1024,
    system_controls=[autoscaling.aws_ecs.SystemControl(
        namespace="net",
        value="ipv4.tcp_tw_recycle"
    )
    ]
)
```

### Using Windows containers on Fargate

AWS Fargate supports Amazon ECS Windows containers. For more details, please see this [blog post](https://aws.amazon.com/tw/blogs/containers/running-windows-containers-with-amazon-ecs-on-aws-fargate/)

```python
# Create a Task Definition for the Windows container to start
task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    runtime_platform=autoscaling.aws_ecs.RuntimePlatform(
        operating_system_family=ecs.OperatingSystemFamily.WINDOWS_SERVER_2019_CORE,
        cpu_architecture=ecs.CpuArchitecture.X86_64
    ),
    cpu=1024,
    memory_limit_mi_b=2048
)

task_definition.add_container("windowsservercore",
    logging=ecs.LogDriver.aws_logs(stream_prefix="win-iis-on-fargate"),
    port_mappings=[autoscaling.aws_ecs.PortMapping(container_port=80)],
    image=ecs.ContainerImage.from_registry("mcr.microsoft.com/windows/servercore/iis:windowsservercore-ltsc2019")
)
```

### Using Graviton2 with Fargate

AWS Graviton2 supports AWS Fargate. For more details, please see this [blog post](https://aws.amazon.com/blogs/aws/announcing-aws-graviton2-support-for-aws-fargate-get-up-to-40-better-price-performance-for-your-serverless-containers/)

```python
# Create a Task Definition for running container on Graviton Runtime.
task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
    runtime_platform=autoscaling.aws_ecs.RuntimePlatform(
        operating_system_family=ecs.OperatingSystemFamily.LINUX,
        cpu_architecture=ecs.CpuArchitecture.ARM64
    ),
    cpu=1024,
    memory_limit_mi_b=2048
)

task_definition.add_container("webarm64",
    logging=ecs.LogDriver.aws_logs(stream_prefix="graviton2-on-fargate"),
    port_mappings=[autoscaling.aws_ecs.PortMapping(container_port=80)],
    image=ecs.ContainerImage.from_registry("public.ecr.aws/nginx/nginx:latest-arm64v8")
)
```

## Service

A `Service` instantiates a `TaskDefinition` on a `Cluster` a given number of
times, optionally associating them with a load balancer.
If a task fails,
Amazon ECS automatically restarts the task.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    desired_count=5
)
```

ECS Anywhere service definition looks like:

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.ExternalService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    desired_count=5
)
```

`Services` by default will create a security group if not provided.
If you'd like to specify which security groups to use you can override the `securityGroups` property.

### Deployment circuit breaker and rollback

Amazon ECS [deployment circuit breaker](https://aws.amazon.com/tw/blogs/containers/announcing-amazon-ecs-deployment-circuit-breaker/)
automatically rolls back unhealthy service deployments without the need for manual intervention. Use `circuitBreaker` to enable
deployment circuit breaker and optionally enable `rollback` for automatic rollback. See [Using the deployment circuit breaker](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html)
for more details.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition

service = ecs.FargateService(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    circuit_breaker=autoscaling.aws_ecs.DeploymentCircuitBreaker(rollback=True)
)
```

> Note: ECS Anywhere doesn't support deployment circuit breakers and rollback.

### Include an application/network load balancer

`Services` are load balancing targets and can be added to a target group, which will be attached to an application/network load balancers:

```python
# vpc: ec2.Vpc
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition

service = ecs.FargateService(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elbv2.ApplicationLoadBalancer(self, "LB", vpc=vpc, internet_facing=True)
listener = lb.add_listener("Listener", port=80)
target_group1 = listener.add_targets("ECS1",
    port=80,
    targets=[service]
)
target_group2 = listener.add_targets("ECS2",
    port=80,
    targets=[service.load_balancer_target(
        container_name="MyContainer",
        container_port=8080
    )]
)
```

> Note: ECS Anywhere doesn't support application/network load balancers.

Note that in the example above, the default `service` only allows you to register the first essential container or the first mapped port on the container as a target and add it to a new target group. To have more control over which container and port to register as targets, you can use `service.loadBalancerTarget()` to return a load balancing target for a specific container and port.

Alternatively, you can also create all load balancer targets to be registered in this service, add them to target groups, and attach target groups to listeners accordingly.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# vpc: ec2.Vpc

service = ecs.FargateService(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elbv2.ApplicationLoadBalancer(self, "LB", vpc=vpc, internet_facing=True)
listener = lb.add_listener("Listener", port=80)
service.register_load_balancer_targets(
    container_name="web",
    container_port=80,
    new_target_group_id="ECS",
    listener=ecs.ListenerConfig.application_listener(listener,
        protocol=elbv2.ApplicationProtocol.HTTPS
    )
)
```

### Using a Load Balancer from a different Stack

If you want to put your Load Balancer and the Service it is load balancing to in
different stacks, you may not be able to use the convenience methods
`loadBalancer.addListener()` and `listener.addTargets()`.

The reason is that these methods will create resources in the same Stack as the
object they're called on, which may lead to cyclic references between stacks.
Instead, you will have to create an `ApplicationListener` in the service stack,
or an empty `TargetGroup` in the load balancer stack that you attach your
service to.

See the [ecs/cross-stack-load-balancer example](https://github.com/aws-samples/aws-cdk-examples/tree/master/typescript/ecs/cross-stack-load-balancer/)
for the alternatives.

### Include a classic load balancer

`Services` can also be directly attached to a classic load balancer as targets:

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# vpc: ec2.Vpc

service = ecs.Ec2Service(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elb.LoadBalancer(self, "LB", vpc=vpc)
lb.add_listener(external_port=80)
lb.add_target(service)
```

Similarly, if you want to have more control over load balancer targeting:

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition
# vpc: ec2.Vpc

service = ecs.Ec2Service(self, "Service", cluster=cluster, task_definition=task_definition)

lb = elb.LoadBalancer(self, "LB", vpc=vpc)
lb.add_listener(external_port=80)
lb.add_target(service.load_balancer_target(
    container_name="MyContainer",
    container_port=80
))
```

There are two higher-level constructs available which include a load balancer for you that can be found in the aws-ecs-patterns module:

* `LoadBalancedFargateService`
* `LoadBalancedEc2Service`

## Task Auto-Scaling

You can configure the task count of a service to match demand. Task auto-scaling is
configured by calling `autoScaleTaskCount()`:

```python
# target: elbv2.ApplicationTargetGroup
# service: ecs.BaseService

scaling = service.auto_scale_task_count(max_capacity=10)
scaling.scale_on_cpu_utilization("CpuScaling",
    target_utilization_percent=50
)

scaling.scale_on_request_count("RequestScaling",
    requests_per_target=10000,
    target_group=target
)
```

Task auto-scaling is powered by *Application Auto-Scaling*.
See that section for details.

## Integration with CloudWatch Events

To start an Amazon ECS task on an Amazon EC2-backed Cluster, instantiate an
`@aws-cdk/aws-events-targets.EcsTask` instead of an `Ec2Service`:

```python
# cluster: ecs.Cluster

# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_asset(path.resolve(__dirname, "..", "eventhandler-image")),
    memory_limit_mi_b=256,
    logging=ecs.AwsLogDriver(stream_prefix="EventDemo", mode=ecs.AwsLogDriverMode.NON_BLOCKING)
)

# An Rule that describes the event trigger (in this case a scheduled run)
rule = events.Rule(self, "Rule",
    schedule=events.Schedule.expression("rate(1 min)")
)

# Pass an environment variable to the container 'TheContainer' in the task
rule.add_target(targets.EcsTask(
    cluster=cluster,
    task_definition=task_definition,
    task_count=1,
    container_overrides=[autoscaling.aws_events_targets.ContainerOverride(
        container_name="TheContainer",
        environment=[autoscaling.aws_events_targets.TaskEnvironmentVariable(
            name="I_WAS_TRIGGERED",
            value="From CloudWatch Events"
        )]
    )]
))
```

## Log Drivers

Currently Supported Log Drivers:

* awslogs
* fluentd
* gelf
* journald
* json-file
* splunk
* syslog
* awsfirelens
* Generic

### awslogs Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.aws_logs(stream_prefix="EventDemo")
)
```

### fluentd Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.fluentd()
)
```

### gelf Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.gelf(address="my-gelf-address")
)
```

### journald Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.journald()
)
```

### json-file Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.json_file()
)
```

### splunk Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.splunk(
        token=SecretValue.secrets_manager("my-splunk-token"),
        url="my-splunk-url"
    )
)
```

### syslog Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.syslog()
)
```

### firelens Log Driver

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.firelens(
        options={
            "Name": "firehose",
            "region": "us-west-2",
            "delivery_stream": "my-stream"
        }
    )
)
```

To pass secrets to the log configuration, use the `secretOptions` property of the log configuration. The task execution role is automatically granted read permissions on the secrets/parameters.

```python
# secret: secretsmanager.Secret
# parameter: ssm.StringParameter


task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.LogDrivers.firelens(
        options={},
        secret_options={ # Retrieved from AWS Secrets Manager or AWS Systems Manager Parameter Store
            "apikey": ecs.Secret.from_secrets_manager(secret),
            "host": ecs.Secret.from_ssm_parameter(parameter)}
    )
)
```

### Generic Log Driver

A generic log driver object exists to provide a lower level abstraction of the log driver configuration.

```python
# Create a Task Definition for the container to start
task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
task_definition.add_container("TheContainer",
    image=ecs.ContainerImage.from_registry("example-image"),
    memory_limit_mi_b=256,
    logging=ecs.GenericLogDriver(
        log_driver="fluentd",
        options={
            "tag": "example-tag"
        }
    )
)
```

## CloudMap Service Discovery

To register your ECS service with a CloudMap Service Registry, you may add the
`cloudMapOptions` property to your service:

```python
# task_definition: ecs.TaskDefinition
# cluster: ecs.Cluster


service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    cloud_map_options=autoscaling.aws_ecs.CloudMapOptions(
        # Create A records - useful for AWSVPC network mode.
        dns_record_type=cloudmap.DnsRecordType.A
    )
)
```

With `bridge` or `host` network modes, only `SRV` DNS record types are supported.
By default, `SRV` DNS record types will target the default container and default
port. However, you may target a different container and port on the same ECS task:

```python
# task_definition: ecs.TaskDefinition
# cluster: ecs.Cluster


# Add a container to the task definition
specific_container = task_definition.add_container("Container",
    image=ecs.ContainerImage.from_registry("/aws/aws-example-app"),
    memory_limit_mi_b=2048
)

# Add a port mapping
specific_container.add_port_mappings(
    container_port=7600,
    protocol=ecs.Protocol.TCP
)

ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    cloud_map_options=autoscaling.aws_ecs.CloudMapOptions(
        # Create SRV records - useful for bridge networking
        dns_record_type=cloudmap.DnsRecordType.SRV,
        # Targets port TCP port 7600 `specificContainer`
        container=specific_container,
        container_port=7600
    )
)
```

### Associate With a Specific CloudMap Service

You may associate an ECS service with a specific CloudMap service. To do
this, use the service's `associateCloudMapService` method:

```python
# cloud_map_service: cloudmap.Service
# ecs_service: ecs.FargateService


ecs_service.associate_cloud_map_service(
    service=cloud_map_service
)
```

## Capacity Providers

There are two major families of Capacity Providers: [AWS
Fargate](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/fargate-capacity-providers.html)
(including Fargate Spot) and EC2 [Auto Scaling
Group](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html)
Capacity Providers. Both are supported.

### Fargate Capacity Providers

To enable Fargate capacity providers, you can either set
`enableFargateCapacityProviders` to `true` when creating your cluster, or by
invoking the `enableFargateCapacityProviders()` method after creating your
cluster. This will add both `FARGATE` and `FARGATE_SPOT` as available capacity
providers on your cluster.

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "FargateCPCluster",
    vpc=vpc,
    enable_fargate_capacity_providers=True
)

task_definition = ecs.FargateTaskDefinition(self, "TaskDef")

task_definition.add_container("web",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample")
)

ecs.FargateService(self, "FargateService",
    cluster=cluster,
    task_definition=task_definition,
    capacity_provider_strategies=[autoscaling.aws_ecs.CapacityProviderStrategy(
        capacity_provider="FARGATE_SPOT",
        weight=2
    ), autoscaling.aws_ecs.CapacityProviderStrategy(
        capacity_provider="FARGATE",
        weight=1
    )
    ]
)
```

### Auto Scaling Group Capacity Providers

To add an Auto Scaling Group Capacity Provider, first create an EC2 Auto Scaling
Group. Then, create an `AsgCapacityProvider` and pass the Auto Scaling Group to
it in the constructor. Then add the Capacity Provider to the cluster. Finally,
you can refer to the Provider by its name in your service's or task's Capacity
Provider strategy.

By default, an Auto Scaling Group Capacity Provider will manage the Auto Scaling
Group's size for you. It will also enable managed termination protection, in
order to prevent EC2 Auto Scaling from terminating EC2 instances that have tasks
running on them. If you want to disable this behavior, set both
`enableManagedScaling` to and `enableManagedTerminationProtection` to `false`.

```python
# vpc: ec2.Vpc


cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc
)

auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
    vpc=vpc,
    instance_type=ec2.InstanceType("t2.micro"),
    machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
    min_capacity=0,
    max_capacity=100
)

capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
    auto_scaling_group=auto_scaling_group
)
cluster.add_asg_capacity_provider(capacity_provider)

task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")

task_definition.add_container("web",
    image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
    memory_reservation_mi_b=256
)

ecs.Ec2Service(self, "EC2Service",
    cluster=cluster,
    task_definition=task_definition,
    capacity_provider_strategies=[autoscaling.aws_ecs.CapacityProviderStrategy(
        capacity_provider=capacity_provider.capacity_provider_name,
        weight=1
    )
    ]
)
```

## Elastic Inference Accelerators

Currently, this feature is only supported for services with EC2 launch types.

To add elastic inference accelerators to your EC2 instance, first add
`inferenceAccelerators` field to the Ec2TaskDefinition and set the `deviceName`
and `deviceType` properties.

```python
inference_accelerators = [{
    "device_name": "device1",
    "device_type": "eia2.medium"
}]

task_definition = ecs.Ec2TaskDefinition(self, "Ec2TaskDef",
    inference_accelerators=inference_accelerators
)
```

To enable using the inference accelerators in the containers, add `inferenceAcceleratorResources`
field and set it to a list of device names used for the inference accelerators. Each value in the
list should match a `DeviceName` for an `InferenceAccelerator` specified in the task definition.

```python
# task_definition: ecs.TaskDefinition

inference_accelerator_resources = ["device1"]

task_definition.add_container("cont",
    image=ecs.ContainerImage.from_registry("test"),
    memory_limit_mi_b=1024,
    inference_accelerator_resources=inference_accelerator_resources
)
```

## ECS Exec command

Please note, ECS Exec leverages AWS Systems Manager (SSM). So as a prerequisite for the exec command
to work, you need to have the SSM plugin for the AWS CLI installed locally. For more information, see
[Install Session Manager plugin for AWS CLI](https://docs.aws.amazon.com/systems-manager/latest/userguide/session-manager-working-with-install-plugin.html).

To enable the ECS Exec feature for your containers, set the boolean flag `enableExecuteCommand` to `true` in
your `Ec2Service` or `FargateService`.

```python
# cluster: ecs.Cluster
# task_definition: ecs.TaskDefinition


service = ecs.Ec2Service(self, "Service",
    cluster=cluster,
    task_definition=task_definition,
    enable_execute_command=True
)
```

### Enabling logging

You can enable sending logs of your execute session commands to a CloudWatch log group or S3 bucket by configuring
the `executeCommandConfiguration` property for your cluster. The default configuration will send the
logs to the CloudWatch Logs using the `awslogs` log driver that is configured in your task definition. Please note,
when using your own `logConfiguration` the log group or S3 Bucket specified must already be created.

To encrypt data using your own KMS Customer Key (CMK), you must create a CMK and provide the key in the `kmsKey` field
of the `executeCommandConfiguration`. To use this key for encrypting CloudWatch log data or S3 bucket, make sure to associate the key
to these resources on creation.

```python
# vpc: ec2.Vpc

kms_key = kms.Key(self, "KmsKey")

# Pass the KMS key in the `encryptionKey` field to associate the key to the log group
log_group = logs.LogGroup(self, "LogGroup",
    encryption_key=kms_key
)

# Pass the KMS key in the `encryptionKey` field to associate the key to the S3 bucket
exec_bucket = s3.Bucket(self, "EcsExecBucket",
    encryption_key=kms_key
)

cluster = ecs.Cluster(self, "Cluster",
    vpc=vpc,
    execute_command_configuration=autoscaling.aws_ecs.ExecuteCommandConfiguration(
        kms_key=kms_key,
        log_configuration=autoscaling.aws_ecs.ExecuteCommandLogConfiguration(
            cloud_watch_log_group=log_group,
            cloud_watch_encryption_enabled=True,
            s3_bucket=exec_bucket,
            s3_encryption_enabled=True,
            s3_key_prefix="exec-command-output"
        ),
        logging=ecs.ExecuteCommandLogging.OVERRIDE
    )
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from .._jsii import *

import constructs
from .. import (
    AssetHashType as _AssetHashType_49193809,
    BundlingOptions as _BundlingOptions_ab115a99,
    CfnResource as _CfnResource_e0a482dc,
    CfnTag as _CfnTag_95fbdc29,
    Construct as _Construct_e78e779f,
    Duration as _Duration_070aa057,
    IInspectable as _IInspectable_82c04a63,
    IResolvable as _IResolvable_a771d0ef,
    IResource as _IResource_8c1dbbbd,
    IgnoreMode as _IgnoreMode_31d8bf46,
    Resource as _Resource_abff4495,
    SecretValue as _SecretValue_c18506ef,
    SymlinkFollowMode as _SymlinkFollowMode_abf4527a,
    TagManager as _TagManager_0b7ab120,
    TreeInspector as _TreeInspector_1cd1894e,
)
from ..assets import FollowMode as _FollowMode_98b05cc5
from ..aws_applicationautoscaling import (
    AdjustmentType as _AdjustmentType_c97ebf31,
    BaseScalableAttribute as _BaseScalableAttribute_8cad7a2c,
    BaseScalableAttributeProps as _BaseScalableAttributeProps_1600c742,
    BaseTargetTrackingProps as _BaseTargetTrackingProps_e4402570,
    BasicStepScalingPolicyProps as _BasicStepScalingPolicyProps_2c327255,
    EnableScalingProps as _EnableScalingProps_d2090945,
    MetricAggregationType as _MetricAggregationType_c42ebfd0,
    ScalingInterval as _ScalingInterval_ed8cf944,
    ScalingSchedule as _ScalingSchedule_6c3fc38f,
    Schedule as _Schedule_c2a5a45d,
    ServiceNamespace as _ServiceNamespace_42590eaf,
)
from ..aws_autoscaling import (
    AutoScalingGroup as _AutoScalingGroup_604a934f,
    BlockDevice as _BlockDevice_da8302ba,
    CommonAutoScalingGroupProps as _CommonAutoScalingGroupProps_0c339448,
    GroupMetrics as _GroupMetrics_c42c90fb,
    HealthCheck as _HealthCheck_f6164c37,
    IAutoScalingGroup as _IAutoScalingGroup_352723c6,
    Monitoring as _Monitoring_ab8b25ef,
    NotificationConfiguration as _NotificationConfiguration_14f038b9,
    RollingUpdateConfiguration as _RollingUpdateConfiguration_7b14e30c,
    Signals as _Signals_896b8d51,
    TerminationPolicy as _TerminationPolicy_633f7bc2,
    UpdatePolicy as _UpdatePolicy_ffeec124,
    UpdateType as _UpdateType_1d8acce6,
)
from ..aws_cloudwatch import (
    IMetric as _IMetric_5db43d61,
    Metric as _Metric_5b2b8e58,
    MetricOptions as _MetricOptions_1c185ae8,
    Unit as _Unit_113c79f9,
)
from ..aws_ec2 import (
    AmazonLinuxGeneration as _AmazonLinuxGeneration_d9650a6d,
    Connections as _Connections_57ccbda9,
    IMachineImage as _IMachineImage_45d3238d,
    ISecurityGroup as _ISecurityGroup_cdbba9d3,
    IVpc as _IVpc_6d1f76c4,
    InstanceArchitecture as _InstanceArchitecture_8b47f118,
    InstanceType as _InstanceType_072ad323,
    MachineImageConfig as _MachineImageConfig_963798fe,
    SubnetSelection as _SubnetSelection_1284e62c,
)
from ..aws_ecr import IRepository as _IRepository_8b4d2894
from ..aws_ecr_assets import (
    DockerImageAsset as _DockerImageAsset_3ac823f8,
    DockerImageAssetInvalidationOptions as _DockerImageAssetInvalidationOptions_e5375707,
    DockerImageAssetOptions as _DockerImageAssetOptions_6764a0de,
    NetworkMode as _NetworkMode_2d19607e,
)
from ..aws_elasticloadbalancing import (
    ILoadBalancerTarget as _ILoadBalancerTarget_ad59785d,
    LoadBalancer as _LoadBalancer_a7de240f,
)
from ..aws_elasticloadbalancingv2 import (
    AddApplicationTargetsProps as _AddApplicationTargetsProps_a85e0c25,
    AddNetworkTargetsProps as _AddNetworkTargetsProps_63ca961a,
    ApplicationListener as _ApplicationListener_835df0e5,
    ApplicationProtocol as _ApplicationProtocol_43133732,
    ApplicationProtocolVersion as _ApplicationProtocolVersion_01dcd421,
    ApplicationTargetGroup as _ApplicationTargetGroup_5b334416,
    HealthCheck as _HealthCheck_46284083,
    IApplicationLoadBalancerTarget as _IApplicationLoadBalancerTarget_db0fdc70,
    IApplicationTargetGroup as _IApplicationTargetGroup_5a474b2b,
    INetworkLoadBalancerTarget as _INetworkLoadBalancerTarget_6691f661,
    INetworkTargetGroup as _INetworkTargetGroup_6938b578,
    ListenerCondition as _ListenerCondition_c2f847ce,
    LoadBalancerTargetProps as _LoadBalancerTargetProps_8b99f223,
    NetworkListener as _NetworkListener_0917bd7b,
    Protocol as _Protocol_5d30b733,
    TargetGroupLoadBalancingAlgorithmType as _TargetGroupLoadBalancingAlgorithmType_9bd255dc,
)
from ..aws_iam import (
    Grant as _Grant_bcb5eae7,
    IGrantable as _IGrantable_4c5a91d1,
    IRole as _IRole_59af6f50,
    PolicyStatement as _PolicyStatement_296fe8a3,
)
from ..aws_kms import IKey as _IKey_36930160
from ..aws_logs import (
    ILogGroup as _ILogGroup_846e17a0, RetentionDays as _RetentionDays_6c560d31
)
from ..aws_s3 import IBucket as _IBucket_73486e29, Location as _Location_cce991ca
from ..aws_s3_assets import AssetOptions as _AssetOptions_bd2996da
from ..aws_secretsmanager import ISecret as _ISecret_22fb8757
from ..aws_servicediscovery import (
    DnsRecordType as _DnsRecordType_a28443d0,
    INamespace as _INamespace_eb21e42e,
    IService as _IService_66c1fbd2,
    NamespaceType as _NamespaceType_db55a104,
    Service as _Service_38744a4d,
)
from ..aws_sns import ITopic as _ITopic_465e36b9
from ..aws_ssm import IParameter as _IParameter_ce5fb757


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AddAutoScalingGroupCapacityOptions",
    jsii_struct_bases=[],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "task_drain_time": "taskDrainTime",
        "topic_encryption_key": "topicEncryptionKey",
    },
)
class AddAutoScalingGroupCapacityOptions:
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''(experimental) The properties for adding an AutoScalingGroup.

        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_ecs as ecs
            from monocdk import aws_kms as kms
            
            # duration: monocdk.Duration
            # key: kms.Key
            
            add_auto_scaling_group_capacity_options = ecs.AddAutoScalingGroupCapacityOptions(
                can_containers_access_instance_role=False,
                machine_image_type=ecs.MachineImageType.AMAZON_LINUX_2,
                spot_instance_draining=False,
                task_drain_time=duration,
                topic_encryption_key=key
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if task_drain_time is not None:
            self._values["task_drain_time"] = task_drain_time
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the containers can access the container instance role.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''(experimental) What type of machine image this is.

        Depending on the setting, different UserData will automatically be added
        to the ``AutoScalingGroup`` to configure it properly for use with ECS.

        If you create an ``AutoScalingGroup`` yourself and are adding it via
        ``addAutoScalingGroup()``, you must specify this value. If you are adding an
        ``autoScalingGroup`` via ``addCapacity``, this value will be determined
        from the ``machineImage`` you pass.

        :default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.

        :stability: experimental
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def task_drain_time(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The time period to wait before force terminating an instance that is draining.

        This creates a Lambda function that is used by a lifecycle hook for the
        AutoScalingGroup that will delay instance termination until all ECS tasks
        have drained from the instance. Set to 0 to disable task draining.

        Set to 0 to disable task draining.

        :default: Duration.minutes(5)

        :deprecated: The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.

        :stability: deprecated
        '''
        result = self._values.get("task_drain_time")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddAutoScalingGroupCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AddCapacityOptions",
    jsii_struct_bases=[
        AddAutoScalingGroupCapacityOptions, _CommonAutoScalingGroupProps_0c339448
    ],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "task_drain_time": "taskDrainTime",
        "topic_encryption_key": "topicEncryptionKey",
        "allow_all_outbound": "allowAllOutbound",
        "associate_public_ip_address": "associatePublicIpAddress",
        "auto_scaling_group_name": "autoScalingGroupName",
        "block_devices": "blockDevices",
        "cooldown": "cooldown",
        "desired_capacity": "desiredCapacity",
        "group_metrics": "groupMetrics",
        "health_check": "healthCheck",
        "ignore_unmodified_size_properties": "ignoreUnmodifiedSizeProperties",
        "instance_monitoring": "instanceMonitoring",
        "key_name": "keyName",
        "max_capacity": "maxCapacity",
        "max_instance_lifetime": "maxInstanceLifetime",
        "min_capacity": "minCapacity",
        "new_instances_protected_from_scale_in": "newInstancesProtectedFromScaleIn",
        "notifications": "notifications",
        "notifications_topic": "notificationsTopic",
        "replacing_update_min_successful_instances_percent": "replacingUpdateMinSuccessfulInstancesPercent",
        "resource_signal_count": "resourceSignalCount",
        "resource_signal_timeout": "resourceSignalTimeout",
        "rolling_update_configuration": "rollingUpdateConfiguration",
        "signals": "signals",
        "spot_price": "spotPrice",
        "termination_policies": "terminationPolicies",
        "update_policy": "updatePolicy",
        "update_type": "updateType",
        "vpc_subnets": "vpcSubnets",
        "instance_type": "instanceType",
        "machine_image": "machineImage",
    },
)
class AddCapacityOptions(
    AddAutoScalingGroupCapacityOptions,
    _CommonAutoScalingGroupProps_0c339448,
):
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
        allow_all_outbound: typing.Optional[builtins.bool] = None,
        associate_public_ip_address: typing.Optional[builtins.bool] = None,
        auto_scaling_group_name: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence[_BlockDevice_da8302ba]] = None,
        cooldown: typing.Optional[_Duration_070aa057] = None,
        desired_capacity: typing.Optional[jsii.Number] = None,
        group_metrics: typing.Optional[typing.Sequence[_GroupMetrics_c42c90fb]] = None,
        health_check: typing.Optional[_HealthCheck_f6164c37] = None,
        ignore_unmodified_size_properties: typing.Optional[builtins.bool] = None,
        instance_monitoring: typing.Optional[_Monitoring_ab8b25ef] = None,
        key_name: typing.Optional[builtins.str] = None,
        max_capacity: typing.Optional[jsii.Number] = None,
        max_instance_lifetime: typing.Optional[_Duration_070aa057] = None,
        min_capacity: typing.Optional[jsii.Number] = None,
        new_instances_protected_from_scale_in: typing.Optional[builtins.bool] = None,
        notifications: typing.Optional[typing.Sequence[_NotificationConfiguration_14f038b9]] = None,
        notifications_topic: typing.Optional[_ITopic_465e36b9] = None,
        replacing_update_min_successful_instances_percent: typing.Optional[jsii.Number] = None,
        resource_signal_count: typing.Optional[jsii.Number] = None,
        resource_signal_timeout: typing.Optional[_Duration_070aa057] = None,
        rolling_update_configuration: typing.Optional[_RollingUpdateConfiguration_7b14e30c] = None,
        signals: typing.Optional[_Signals_896b8d51] = None,
        spot_price: typing.Optional[builtins.str] = None,
        termination_policies: typing.Optional[typing.Sequence[_TerminationPolicy_633f7bc2]] = None,
        update_policy: typing.Optional[_UpdatePolicy_ffeec124] = None,
        update_type: typing.Optional[_UpdateType_1d8acce6] = None,
        vpc_subnets: typing.Optional[_SubnetSelection_1284e62c] = None,
        instance_type: _InstanceType_072ad323,
        machine_image: typing.Optional[_IMachineImage_45d3238d] = None,
    ) -> None:
        '''(experimental) The properties for adding instance capacity to an AutoScalingGroup.

        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        :param allow_all_outbound: (experimental) Whether the instances can initiate connections to anywhere by default. Default: true
        :param associate_public_ip_address: (experimental) Whether instances in the Auto Scaling Group should have public IP addresses associated with them. Default: - Use subnet setting.
        :param auto_scaling_group_name: (experimental) The name of the Auto Scaling group. This name must be unique per Region per account. Default: - Auto generated by CloudFormation
        :param block_devices: (experimental) Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. Default: - Uses the block device mapping of the AMI
        :param cooldown: (experimental) Default scaling cooldown for this AutoScalingGroup. Default: Duration.minutes(5)
        :param desired_capacity: (experimental) Initial amount of instances in the fleet. If this is set to a number, every deployment will reset the amount of instances to this number. It is recommended to leave this value blank. Default: minCapacity, and leave unchanged during deployment
        :param group_metrics: (experimental) Enable monitoring for group metrics, these metrics describe the group rather than any of its instances. To report all group metrics use ``GroupMetrics.all()`` Group metrics are reported in a granularity of 1 minute at no additional charge. Default: - no group metrics will be reported
        :param health_check: (experimental) Configuration for health checks. Default: - HealthCheck.ec2 with no grace period
        :param ignore_unmodified_size_properties: (experimental) If the ASG has scheduled actions, don't reset unchanged group sizes. Only used if the ASG has scheduled actions (which may scale your ASG up or down regardless of cdk deployments). If true, the size of the group will only be reset if it has been changed in the CDK app. If false, the sizes will always be changed back to what they were in the CDK app on deployment. Default: true
        :param instance_monitoring: (experimental) Controls whether instances in this group are launched with detailed or basic monitoring. When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes. Default: - Monitoring.DETAILED
        :param key_name: (experimental) Name of SSH keypair to grant access to instances. Default: - No SSH access will be possible.
        :param max_capacity: (experimental) Maximum number of instances in the fleet. Default: desiredCapacity
        :param max_instance_lifetime: (experimental) The maximum amount of time that an instance can be in service. The maximum duration applies to all current and future instances in the group. As an instance approaches its maximum duration, it is terminated and replaced, and cannot be used again. You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value, leave this property undefined. Default: none
        :param min_capacity: (experimental) Minimum number of instances in the fleet. Default: 1
        :param new_instances_protected_from_scale_in: (experimental) Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in. By default, Auto Scaling can terminate an instance at any time after launch when scaling in an Auto Scaling Group, subject to the group's termination policy. However, you may wish to protect newly-launched instances from being scaled in if they are going to run critical applications that should not be prematurely terminated. This flag must be enabled if the Auto Scaling Group will be associated with an ECS Capacity Provider with managed termination protection. Default: false
        :param notifications: (experimental) Configure autoscaling group to send notifications about fleet changes to an SNS topic(s). Default: - No fleet change notifications will be sent.
        :param notifications_topic: (deprecated) SNS topic to send notifications about fleet changes. Default: - No fleet change notifications will be sent.
        :param replacing_update_min_successful_instances_percent: (deprecated) Configuration for replacing updates. Only used if updateType == UpdateType.ReplacingUpdate. Specifies how many instances must signal success for the update to succeed. Default: minSuccessfulInstancesPercent
        :param resource_signal_count: (deprecated) How many ResourceSignal calls CloudFormation expects before the resource is considered created. Default: 1 if resourceSignalTimeout is set, 0 otherwise
        :param resource_signal_timeout: (deprecated) The length of time to wait for the resourceSignalCount. The maximum value is 43200 (12 hours). Default: Duration.minutes(5) if resourceSignalCount is set, N/A otherwise
        :param rolling_update_configuration: (deprecated) Configuration for rolling updates. Only used if updateType == UpdateType.RollingUpdate. Default: - RollingUpdateConfiguration with defaults.
        :param signals: (experimental) Configure waiting for signals during deployment. Use this to pause the CloudFormation deployment to wait for the instances in the AutoScalingGroup to report successful startup during creation and updates. The UserData script needs to invoke ``cfn-signal`` with a success or failure code after it is done setting up the instance. Without waiting for signals, the CloudFormation deployment will proceed as soon as the AutoScalingGroup has been created or updated but before the instances in the group have been started. For example, to have instances wait for an Elastic Load Balancing health check before they signal success, add a health-check verification by using the cfn-init helper script. For an example, see the verify_instance_health command in the Auto Scaling rolling updates sample template: https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml Default: - Do not wait for signals
        :param spot_price: (experimental) The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request. Spot Instances are launched when the price you specify exceeds the current Spot market price. Default: none
        :param termination_policies: (experimental) A policy or a list of policies that are used to select the instances to terminate. The policies are executed in the order that you list them. Default: - ``TerminationPolicy.DEFAULT``
        :param update_policy: (experimental) What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise
        :param update_type: (deprecated) What to do when an AutoScalingGroup's instance configuration is changed. This is applied when any of the settings on the ASG are changed that affect how the instances should be created (VPC, instance type, startup scripts, etc.). It indicates how the existing instances should be replaced with new instances matching the new config. By default, nothing is done and only new instances are launched with the new config. Default: UpdateType.None
        :param vpc_subnets: (experimental) Where to place instances within the VPC. Default: - All Private subnets.
        :param instance_type: (experimental) The EC2 instance type to use when launching instances into the AutoScalingGroup.
        :param machine_image: (experimental) The ECS-optimized AMI variant to use. The default is to use an ECS-optimized AMI of Amazon Linux 2 which is automatically updated to the latest version on every deployment. This will replace the instances in the AutoScalingGroup. Make sure you have not disabled task draining, to avoid downtime when the AMI updates. To use an image that does not update on every deployment, pass:: const machineImage = ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, { cachedInContext: true, }); For more information, see `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_. You must define either ``machineImage`` or ``machineImageType``, not both. Default: - Automatically updated, ECS-optimized Amazon Linux 2

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            
            cluster = ecs.Cluster(self, "Cluster",
                vpc=vpc
            )
            
            # Either add default capacity
            cluster.add_capacity("DefaultAutoScalingGroupCapacity",
                instance_type=ec2.InstanceType("t2.xlarge"),
                desired_capacity=3
            )
            
            # Or add customized capacity. Be sure to start the Amazon ECS-optimized AMI.
            auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
                vpc=vpc,
                instance_type=ec2.InstanceType("t2.xlarge"),
                machine_image=ecs.EcsOptimizedImage.amazon_linux(),
                # Or use Amazon ECS-Optimized Amazon Linux 2 AMI
                # machineImage: EcsOptimizedImage.amazonLinux2(),
                desired_capacity=3
            )
            
            cluster.add_auto_scaling_group(auto_scaling_group)
        '''
        if isinstance(rolling_update_configuration, dict):
            rolling_update_configuration = _RollingUpdateConfiguration_7b14e30c(**rolling_update_configuration)
        if isinstance(vpc_subnets, dict):
            vpc_subnets = _SubnetSelection_1284e62c(**vpc_subnets)
        self._values: typing.Dict[str, typing.Any] = {
            "instance_type": instance_type,
        }
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if task_drain_time is not None:
            self._values["task_drain_time"] = task_drain_time
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key
        if allow_all_outbound is not None:
            self._values["allow_all_outbound"] = allow_all_outbound
        if associate_public_ip_address is not None:
            self._values["associate_public_ip_address"] = associate_public_ip_address
        if auto_scaling_group_name is not None:
            self._values["auto_scaling_group_name"] = auto_scaling_group_name
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if cooldown is not None:
            self._values["cooldown"] = cooldown
        if desired_capacity is not None:
            self._values["desired_capacity"] = desired_capacity
        if group_metrics is not None:
            self._values["group_metrics"] = group_metrics
        if health_check is not None:
            self._values["health_check"] = health_check
        if ignore_unmodified_size_properties is not None:
            self._values["ignore_unmodified_size_properties"] = ignore_unmodified_size_properties
        if instance_monitoring is not None:
            self._values["instance_monitoring"] = instance_monitoring
        if key_name is not None:
            self._values["key_name"] = key_name
        if max_capacity is not None:
            self._values["max_capacity"] = max_capacity
        if max_instance_lifetime is not None:
            self._values["max_instance_lifetime"] = max_instance_lifetime
        if min_capacity is not None:
            self._values["min_capacity"] = min_capacity
        if new_instances_protected_from_scale_in is not None:
            self._values["new_instances_protected_from_scale_in"] = new_instances_protected_from_scale_in
        if notifications is not None:
            self._values["notifications"] = notifications
        if notifications_topic is not None:
            self._values["notifications_topic"] = notifications_topic
        if replacing_update_min_successful_instances_percent is not None:
            self._values["replacing_update_min_successful_instances_percent"] = replacing_update_min_successful_instances_percent
        if resource_signal_count is not None:
            self._values["resource_signal_count"] = resource_signal_count
        if resource_signal_timeout is not None:
            self._values["resource_signal_timeout"] = resource_signal_timeout
        if rolling_update_configuration is not None:
            self._values["rolling_update_configuration"] = rolling_update_configuration
        if signals is not None:
            self._values["signals"] = signals
        if spot_price is not None:
            self._values["spot_price"] = spot_price
        if termination_policies is not None:
            self._values["termination_policies"] = termination_policies
        if update_policy is not None:
            self._values["update_policy"] = update_policy
        if update_type is not None:
            self._values["update_type"] = update_type
        if vpc_subnets is not None:
            self._values["vpc_subnets"] = vpc_subnets
        if machine_image is not None:
            self._values["machine_image"] = machine_image

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the containers can access the container instance role.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''(experimental) What type of machine image this is.

        Depending on the setting, different UserData will automatically be added
        to the ``AutoScalingGroup`` to configure it properly for use with ECS.

        If you create an ``AutoScalingGroup`` yourself and are adding it via
        ``addAutoScalingGroup()``, you must specify this value. If you are adding an
        ``autoScalingGroup`` via ``addCapacity``, this value will be determined
        from the ``machineImage`` you pass.

        :default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.

        :stability: experimental
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def task_drain_time(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The time period to wait before force terminating an instance that is draining.

        This creates a Lambda function that is used by a lifecycle hook for the
        AutoScalingGroup that will delay instance termination until all ECS tasks
        have drained from the instance. Set to 0 to disable task draining.

        Set to 0 to disable task draining.

        :default: Duration.minutes(5)

        :deprecated: The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.

        :stability: deprecated
        '''
        result = self._values.get("task_drain_time")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    @builtins.property
    def allow_all_outbound(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether the instances can initiate connections to anywhere by default.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("allow_all_outbound")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def associate_public_ip_address(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether instances in the Auto Scaling Group should have public IP addresses associated with them.

        :default: - Use subnet setting.

        :stability: experimental
        '''
        result = self._values.get("associate_public_ip_address")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def auto_scaling_group_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the Auto Scaling group.

        This name must be unique per Region per account.

        :default: - Auto generated by CloudFormation

        :stability: experimental
        '''
        result = self._values.get("auto_scaling_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List[_BlockDevice_da8302ba]]:
        '''(experimental) Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        :stability: experimental
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List[_BlockDevice_da8302ba]], result)

    @builtins.property
    def cooldown(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Default scaling cooldown for this AutoScalingGroup.

        :default: Duration.minutes(5)

        :stability: experimental
        '''
        result = self._values.get("cooldown")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def desired_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Initial amount of instances in the fleet.

        If this is set to a number, every deployment will reset the amount of
        instances to this number. It is recommended to leave this value blank.

        :default: minCapacity, and leave unchanged during deployment

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-desiredcapacity
        :stability: experimental
        '''
        result = self._values.get("desired_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def group_metrics(self) -> typing.Optional[typing.List[_GroupMetrics_c42c90fb]]:
        '''(experimental) Enable monitoring for group metrics, these metrics describe the group rather than any of its instances.

        To report all group metrics use ``GroupMetrics.all()``
        Group metrics are reported in a granularity of 1 minute at no additional charge.

        :default: - no group metrics will be reported

        :stability: experimental
        '''
        result = self._values.get("group_metrics")
        return typing.cast(typing.Optional[typing.List[_GroupMetrics_c42c90fb]], result)

    @builtins.property
    def health_check(self) -> typing.Optional[_HealthCheck_f6164c37]:
        '''(experimental) Configuration for health checks.

        :default: - HealthCheck.ec2 with no grace period

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[_HealthCheck_f6164c37], result)

    @builtins.property
    def ignore_unmodified_size_properties(self) -> typing.Optional[builtins.bool]:
        '''(experimental) If the ASG has scheduled actions, don't reset unchanged group sizes.

        Only used if the ASG has scheduled actions (which may scale your ASG up
        or down regardless of cdk deployments). If true, the size of the group
        will only be reset if it has been changed in the CDK app. If false, the
        sizes will always be changed back to what they were in the CDK app
        on deployment.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("ignore_unmodified_size_properties")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def instance_monitoring(self) -> typing.Optional[_Monitoring_ab8b25ef]:
        '''(experimental) Controls whether instances in this group are launched with detailed or basic monitoring.

        When detailed monitoring is enabled, Amazon CloudWatch generates metrics every minute and your account
        is charged a fee. When you disable detailed monitoring, CloudWatch generates metrics every 5 minutes.

        :default: - Monitoring.DETAILED

        :see: https://docs.aws.amazon.com/autoscaling/latest/userguide/as-instance-monitoring.html#enable-as-instance-metrics
        :stability: experimental
        '''
        result = self._values.get("instance_monitoring")
        return typing.cast(typing.Optional[_Monitoring_ab8b25ef], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Name of SSH keypair to grant access to instances.

        :default: - No SSH access will be possible.

        :stability: experimental
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum number of instances in the fleet.

        :default: desiredCapacity

        :stability: experimental
        '''
        result = self._values.get("max_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_instance_lifetime(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The maximum amount of time that an instance can be in service.

        The maximum duration applies
        to all current and future instances in the group. As an instance approaches its maximum duration,
        it is terminated and replaced, and cannot be used again.

        You must specify a value of at least 604,800 seconds (7 days). To clear a previously set value,
        leave this property undefined.

        :default: none

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/asg-max-instance-lifetime.html
        :stability: experimental
        '''
        result = self._values.get("max_instance_lifetime")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def min_capacity(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Minimum number of instances in the fleet.

        :default: 1

        :stability: experimental
        '''
        result = self._values.get("min_capacity")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def new_instances_protected_from_scale_in(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether newly-launched instances are protected from termination by Amazon EC2 Auto Scaling when scaling in.

        By default, Auto Scaling can terminate an instance at any time after launch
        when scaling in an Auto Scaling Group, subject to the group's termination
        policy. However, you may wish to protect newly-launched instances from
        being scaled in if they are going to run critical applications that should
        not be prematurely terminated.

        This flag must be enabled if the Auto Scaling Group will be associated with
        an ECS Capacity Provider with managed termination protection.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("new_instances_protected_from_scale_in")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def notifications(
        self,
    ) -> typing.Optional[typing.List[_NotificationConfiguration_14f038b9]]:
        '''(experimental) Configure autoscaling group to send notifications about fleet changes to an SNS topic(s).

        :default: - No fleet change notifications will be sent.

        :see: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-as-group.html#cfn-as-group-notificationconfigurations
        :stability: experimental
        '''
        result = self._values.get("notifications")
        return typing.cast(typing.Optional[typing.List[_NotificationConfiguration_14f038b9]], result)

    @builtins.property
    def notifications_topic(self) -> typing.Optional[_ITopic_465e36b9]:
        '''(deprecated) SNS topic to send notifications about fleet changes.

        :default: - No fleet change notifications will be sent.

        :deprecated: use ``notifications``

        :stability: deprecated
        '''
        result = self._values.get("notifications_topic")
        return typing.cast(typing.Optional[_ITopic_465e36b9], result)

    @builtins.property
    def replacing_update_min_successful_instances_percent(
        self,
    ) -> typing.Optional[jsii.Number]:
        '''(deprecated) Configuration for replacing updates.

        Only used if updateType == UpdateType.ReplacingUpdate. Specifies how
        many instances must signal success for the update to succeed.

        :default: minSuccessfulInstancesPercent

        :deprecated: Use ``signals`` instead

        :stability: deprecated
        '''
        result = self._values.get("replacing_update_min_successful_instances_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def resource_signal_count(self) -> typing.Optional[jsii.Number]:
        '''(deprecated) How many ResourceSignal calls CloudFormation expects before the resource is considered created.

        :default: 1 if resourceSignalTimeout is set, 0 otherwise

        :deprecated: Use ``signals`` instead.

        :stability: deprecated
        '''
        result = self._values.get("resource_signal_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def resource_signal_timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The length of time to wait for the resourceSignalCount.

        The maximum value is 43200 (12 hours).

        :default: Duration.minutes(5) if resourceSignalCount is set, N/A otherwise

        :deprecated: Use ``signals`` instead.

        :stability: deprecated
        '''
        result = self._values.get("resource_signal_timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def rolling_update_configuration(
        self,
    ) -> typing.Optional[_RollingUpdateConfiguration_7b14e30c]:
        '''(deprecated) Configuration for rolling updates.

        Only used if updateType == UpdateType.RollingUpdate.

        :default: - RollingUpdateConfiguration with defaults.

        :deprecated: Use ``updatePolicy`` instead

        :stability: deprecated
        '''
        result = self._values.get("rolling_update_configuration")
        return typing.cast(typing.Optional[_RollingUpdateConfiguration_7b14e30c], result)

    @builtins.property
    def signals(self) -> typing.Optional[_Signals_896b8d51]:
        '''(experimental) Configure waiting for signals during deployment.

        Use this to pause the CloudFormation deployment to wait for the instances
        in the AutoScalingGroup to report successful startup during
        creation and updates. The UserData script needs to invoke ``cfn-signal``
        with a success or failure code after it is done setting up the instance.

        Without waiting for signals, the CloudFormation deployment will proceed as
        soon as the AutoScalingGroup has been created or updated but before the
        instances in the group have been started.

        For example, to have instances wait for an Elastic Load Balancing health check before
        they signal success, add a health-check verification by using the
        cfn-init helper script. For an example, see the verify_instance_health
        command in the Auto Scaling rolling updates sample template:

        https://github.com/awslabs/aws-cloudformation-templates/blob/master/aws/services/AutoScaling/AutoScalingRollingUpdates.yaml

        :default: - Do not wait for signals

        :stability: experimental
        '''
        result = self._values.get("signals")
        return typing.cast(typing.Optional[_Signals_896b8d51], result)

    @builtins.property
    def spot_price(self) -> typing.Optional[builtins.str]:
        '''(experimental) The maximum hourly price (in USD) to be paid for any Spot Instance launched to fulfill the request.

        Spot Instances are
        launched when the price you specify exceeds the current Spot market price.

        :default: none

        :stability: experimental
        '''
        result = self._values.get("spot_price")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def termination_policies(
        self,
    ) -> typing.Optional[typing.List[_TerminationPolicy_633f7bc2]]:
        '''(experimental) A policy or a list of policies that are used to select the instances to terminate.

        The policies are executed in the order that you list them.

        :default: - ``TerminationPolicy.DEFAULT``

        :see: https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html
        :stability: experimental
        '''
        result = self._values.get("termination_policies")
        return typing.cast(typing.Optional[typing.List[_TerminationPolicy_633f7bc2]], result)

    @builtins.property
    def update_policy(self) -> typing.Optional[_UpdatePolicy_ffeec124]:
        '''(experimental) What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: - ``UpdatePolicy.rollingUpdate()`` if using ``init``, ``UpdatePolicy.none()`` otherwise

        :stability: experimental
        '''
        result = self._values.get("update_policy")
        return typing.cast(typing.Optional[_UpdatePolicy_ffeec124], result)

    @builtins.property
    def update_type(self) -> typing.Optional[_UpdateType_1d8acce6]:
        '''(deprecated) What to do when an AutoScalingGroup's instance configuration is changed.

        This is applied when any of the settings on the ASG are changed that
        affect how the instances should be created (VPC, instance type, startup
        scripts, etc.). It indicates how the existing instances should be
        replaced with new instances matching the new config. By default, nothing
        is done and only new instances are launched with the new config.

        :default: UpdateType.None

        :deprecated: Use ``updatePolicy`` instead

        :stability: deprecated
        '''
        result = self._values.get("update_type")
        return typing.cast(typing.Optional[_UpdateType_1d8acce6], result)

    @builtins.property
    def vpc_subnets(self) -> typing.Optional[_SubnetSelection_1284e62c]:
        '''(experimental) Where to place instances within the VPC.

        :default: - All Private subnets.

        :stability: experimental
        '''
        result = self._values.get("vpc_subnets")
        return typing.cast(typing.Optional[_SubnetSelection_1284e62c], result)

    @builtins.property
    def instance_type(self) -> _InstanceType_072ad323:
        '''(experimental) The EC2 instance type to use when launching instances into the AutoScalingGroup.

        :stability: experimental
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(_InstanceType_072ad323, result)

    @builtins.property
    def machine_image(self) -> typing.Optional[_IMachineImage_45d3238d]:
        '''(experimental) The ECS-optimized AMI variant to use.

        The default is to use an ECS-optimized AMI of Amazon Linux 2 which is
        automatically updated to the latest version on every deployment. This will
        replace the instances in the AutoScalingGroup. Make sure you have not disabled
        task draining, to avoid downtime when the AMI updates.

        To use an image that does not update on every deployment, pass::

           machine_image = ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.STANDARD,
               cached_in_context=True
           )

        For more information, see `Amazon ECS-optimized
        AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_.

        You must define either ``machineImage`` or ``machineImageType``, not both.

        :default: - Automatically updated, ECS-optimized Amazon Linux 2

        :stability: experimental
        '''
        result = self._values.get("machine_image")
        return typing.cast(typing.Optional[_IMachineImage_45d3238d], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddCapacityOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.AmiHardwareType")
class AmiHardwareType(enum.Enum):
    '''(experimental) The ECS-optimized AMI variant to use.

    For more information, see
    `Amazon ECS-optimized AMIs <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # cluster: ecs.Cluster
        
        
        cluster.add_capacity("graviton-cluster",
            min_capacity=2,
            instance_type=ec2.InstanceType("c6g.large"),
            machine_image=ecs.EcsOptimizedImage.amazon_linux2(ecs.AmiHardwareType.ARM)
        )
    '''

    STANDARD = "STANDARD"
    '''(experimental) Use the standard Amazon ECS-optimized AMI.

    :stability: experimental
    '''
    GPU = "GPU"
    '''(experimental) Use the Amazon ECS GPU-optimized AMI.

    :stability: experimental
    '''
    ARM = "ARM"
    '''(experimental) Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AppMeshProxyConfigurationConfigProps",
    jsii_struct_bases=[],
    name_mapping={"container_name": "containerName", "properties": "properties"},
)
class AppMeshProxyConfigurationConfigProps:
    def __init__(
        self,
        *,
        container_name: builtins.str,
        properties: "AppMeshProxyConfigurationProps",
    ) -> None:
        '''(experimental) The configuration to use when setting an App Mesh proxy configuration.

        :param container_name: (experimental) The name of the container that will serve as the App Mesh proxy.
        :param properties: (experimental) The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            app_mesh_proxy_configuration_config_props = ecs.AppMeshProxyConfigurationConfigProps(
                container_name="containerName",
                properties=ecs.AppMeshProxyConfigurationProps(
                    app_ports=[123],
                    proxy_egress_port=123,
                    proxy_ingress_port=123,
            
                    # the properties below are optional
                    egress_ignored_iPs=["egressIgnoredIPs"],
                    egress_ignored_ports=[123],
                    ignored_gID=123,
                    ignored_uID=123
                )
            )
        '''
        if isinstance(properties, dict):
            properties = AppMeshProxyConfigurationProps(**properties)
        self._values: typing.Dict[str, typing.Any] = {
            "container_name": container_name,
            "properties": properties,
        }

    @builtins.property
    def container_name(self) -> builtins.str:
        '''(experimental) The name of the container that will serve as the App Mesh proxy.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        assert result is not None, "Required property 'container_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def properties(self) -> "AppMeshProxyConfigurationProps":
        '''(experimental) The set of network configuration parameters to provide the Container Network Interface (CNI) plugin.

        :stability: experimental
        '''
        result = self._values.get("properties")
        assert result is not None, "Required property 'properties' is missing"
        return typing.cast("AppMeshProxyConfigurationProps", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppMeshProxyConfigurationConfigProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AppMeshProxyConfigurationProps",
    jsii_struct_bases=[],
    name_mapping={
        "app_ports": "appPorts",
        "proxy_egress_port": "proxyEgressPort",
        "proxy_ingress_port": "proxyIngressPort",
        "egress_ignored_i_ps": "egressIgnoredIPs",
        "egress_ignored_ports": "egressIgnoredPorts",
        "ignored_gid": "ignoredGID",
        "ignored_uid": "ignoredUID",
    },
)
class AppMeshProxyConfigurationProps:
    def __init__(
        self,
        *,
        app_ports: typing.Sequence[jsii.Number],
        proxy_egress_port: jsii.Number,
        proxy_ingress_port: jsii.Number,
        egress_ignored_i_ps: typing.Optional[typing.Sequence[builtins.str]] = None,
        egress_ignored_ports: typing.Optional[typing.Sequence[jsii.Number]] = None,
        ignored_gid: typing.Optional[jsii.Number] = None,
        ignored_uid: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Interface for setting the properties of proxy configuration.

        :param app_ports: (experimental) The list of ports that the application uses. Network traffic to these ports is forwarded to the ProxyIngressPort and ProxyEgressPort.
        :param proxy_egress_port: (experimental) Specifies the port that outgoing traffic from the AppPorts is directed to.
        :param proxy_ingress_port: (experimental) Specifies the port that incoming traffic to the AppPorts is directed to.
        :param egress_ignored_i_ps: (experimental) The egress traffic going to these specified IP addresses is ignored and not redirected to the ProxyEgressPort. It can be an empty list.
        :param egress_ignored_ports: (experimental) The egress traffic going to these specified ports is ignored and not redirected to the ProxyEgressPort. It can be an empty list.
        :param ignored_gid: (experimental) The group ID (GID) of the proxy container as defined by the user parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If IgnoredUID is specified, this field can be empty.
        :param ignored_uid: (experimental) The user ID (UID) of the proxy container as defined by the user parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If IgnoredGID is specified, this field can be empty.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            app_mesh_proxy_configuration_props = ecs.AppMeshProxyConfigurationProps(
                app_ports=[123],
                proxy_egress_port=123,
                proxy_ingress_port=123,
            
                # the properties below are optional
                egress_ignored_iPs=["egressIgnoredIPs"],
                egress_ignored_ports=[123],
                ignored_gID=123,
                ignored_uID=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "app_ports": app_ports,
            "proxy_egress_port": proxy_egress_port,
            "proxy_ingress_port": proxy_ingress_port,
        }
        if egress_ignored_i_ps is not None:
            self._values["egress_ignored_i_ps"] = egress_ignored_i_ps
        if egress_ignored_ports is not None:
            self._values["egress_ignored_ports"] = egress_ignored_ports
        if ignored_gid is not None:
            self._values["ignored_gid"] = ignored_gid
        if ignored_uid is not None:
            self._values["ignored_uid"] = ignored_uid

    @builtins.property
    def app_ports(self) -> typing.List[jsii.Number]:
        '''(experimental) The list of ports that the application uses.

        Network traffic to these ports is forwarded to the ProxyIngressPort and ProxyEgressPort.

        :stability: experimental
        '''
        result = self._values.get("app_ports")
        assert result is not None, "Required property 'app_ports' is missing"
        return typing.cast(typing.List[jsii.Number], result)

    @builtins.property
    def proxy_egress_port(self) -> jsii.Number:
        '''(experimental) Specifies the port that outgoing traffic from the AppPorts is directed to.

        :stability: experimental
        '''
        result = self._values.get("proxy_egress_port")
        assert result is not None, "Required property 'proxy_egress_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def proxy_ingress_port(self) -> jsii.Number:
        '''(experimental) Specifies the port that incoming traffic to the AppPorts is directed to.

        :stability: experimental
        '''
        result = self._values.get("proxy_ingress_port")
        assert result is not None, "Required property 'proxy_ingress_port' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def egress_ignored_i_ps(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The egress traffic going to these specified IP addresses is ignored and not redirected to the ProxyEgressPort.

        It can be an empty list.

        :stability: experimental
        '''
        result = self._values.get("egress_ignored_i_ps")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def egress_ignored_ports(self) -> typing.Optional[typing.List[jsii.Number]]:
        '''(experimental) The egress traffic going to these specified ports is ignored and not redirected to the ProxyEgressPort.

        It can be an empty list.

        :stability: experimental
        '''
        result = self._values.get("egress_ignored_ports")
        return typing.cast(typing.Optional[typing.List[jsii.Number]], result)

    @builtins.property
    def ignored_gid(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The group ID (GID) of the proxy container as defined by the user parameter in a container definition.

        This is used to ensure the proxy ignores its own traffic. If IgnoredUID is specified, this field can be empty.

        :stability: experimental
        '''
        result = self._values.get("ignored_gid")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ignored_uid(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The user ID (UID) of the proxy container as defined by the user parameter in a container definition.

        This is used to ensure the proxy ignores its own traffic. If IgnoredGID is specified, this field can be empty.

        :stability: experimental
        '''
        result = self._values.get("ignored_uid")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AppMeshProxyConfigurationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AsgCapacityProvider(
    _Construct_e78e779f,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.AsgCapacityProvider",
):
    '''(experimental) An Auto Scaling Group Capacity Provider.

    This allows an ECS cluster to target
    a specific EC2 Auto Scaling Group for the placement of tasks. Optionally (and
    recommended), ECS can manage the number of instances in the ASG to fit the
    tasks, and can ensure that instances are not prematurely terminated while
    there are still tasks running on them.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # vpc: ec2.Vpc
        
        
        cluster = ecs.Cluster(self, "Cluster",
            vpc=vpc
        )
        
        auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
            vpc=vpc,
            instance_type=ec2.InstanceType("t2.micro"),
            machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
            min_capacity=0,
            max_capacity=100
        )
        
        capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
            auto_scaling_group=auto_scaling_group
        )
        cluster.add_asg_capacity_provider(capacity_provider)
        
        task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
        
        task_definition.add_container("web",
            image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
            memory_reservation_mi_b=256
        )
        
        ecs.Ec2Service(self, "EC2Service",
            cluster=cluster,
            task_definition=task_definition,
            capacity_provider_strategies=[autoscaling.aws_ecs.CapacityProviderStrategy(
                capacity_provider=capacity_provider.capacity_provider_name,
                weight=1
            )
            ]
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        auto_scaling_group: _IAutoScalingGroup_352723c6,
        capacity_provider_name: typing.Optional[builtins.str] = None,
        enable_managed_scaling: typing.Optional[builtins.bool] = None,
        enable_managed_termination_protection: typing.Optional[builtins.bool] = None,
        maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
        minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
        target_capacity_percent: typing.Optional[jsii.Number] = None,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param auto_scaling_group: (experimental) The autoscaling group to add as a Capacity Provider.
        :param capacity_provider_name: (experimental) The name of the capacity provider. If a name is specified, it cannot start with ``aws``, ``ecs``, or ``fargate``. If no name is specified, a default name in the CFNStackName-CFNResourceName-RandomString format is used. Default: CloudFormation-generated name
        :param enable_managed_scaling: (experimental) Whether to enable managed scaling. Default: true
        :param enable_managed_termination_protection: (experimental) Whether to enable managed termination protection. Default: true
        :param maximum_scaling_step_size: (experimental) Maximum scaling step size. In most cases this should be left alone. Default: 1000
        :param minimum_scaling_step_size: (experimental) Minimum scaling step size. In most cases this should be left alone. Default: 1
        :param target_capacity_percent: (experimental) Target capacity percent. In most cases this should be left alone. Default: 100
        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        props = AsgCapacityProviderProps(
            auto_scaling_group=auto_scaling_group,
            capacity_provider_name=capacity_provider_name,
            enable_managed_scaling=enable_managed_scaling,
            enable_managed_termination_protection=enable_managed_termination_protection,
            maximum_scaling_step_size=maximum_scaling_step_size,
            minimum_scaling_step_size=minimum_scaling_step_size,
            target_capacity_percent=target_capacity_percent,
            can_containers_access_instance_role=can_containers_access_instance_role,
            machine_image_type=machine_image_type,
            spot_instance_draining=spot_instance_draining,
            task_drain_time=task_drain_time,
            topic_encryption_key=topic_encryption_key,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoScalingGroup")
    def auto_scaling_group(self) -> _AutoScalingGroup_604a934f:
        '''(experimental) Auto Scaling Group.

        :stability: experimental
        '''
        return typing.cast(_AutoScalingGroup_604a934f, jsii.get(self, "autoScalingGroup"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviderName")
    def capacity_provider_name(self) -> builtins.str:
        '''(experimental) Capacity provider name.

        :default: Chosen by CloudFormation

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "capacityProviderName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="machineImageType")
    def machine_image_type(self) -> "MachineImageType":
        '''(experimental) Auto Scaling Group machineImageType.

        :stability: experimental
        '''
        return typing.cast("MachineImageType", jsii.get(self, "machineImageType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enableManagedTerminationProtection")
    def enable_managed_termination_protection(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether managed termination protection is enabled.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "enableManagedTerminationProtection"))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AsgCapacityProviderProps",
    jsii_struct_bases=[AddAutoScalingGroupCapacityOptions],
    name_mapping={
        "can_containers_access_instance_role": "canContainersAccessInstanceRole",
        "machine_image_type": "machineImageType",
        "spot_instance_draining": "spotInstanceDraining",
        "task_drain_time": "taskDrainTime",
        "topic_encryption_key": "topicEncryptionKey",
        "auto_scaling_group": "autoScalingGroup",
        "capacity_provider_name": "capacityProviderName",
        "enable_managed_scaling": "enableManagedScaling",
        "enable_managed_termination_protection": "enableManagedTerminationProtection",
        "maximum_scaling_step_size": "maximumScalingStepSize",
        "minimum_scaling_step_size": "minimumScalingStepSize",
        "target_capacity_percent": "targetCapacityPercent",
    },
)
class AsgCapacityProviderProps(AddAutoScalingGroupCapacityOptions):
    def __init__(
        self,
        *,
        can_containers_access_instance_role: typing.Optional[builtins.bool] = None,
        machine_image_type: typing.Optional["MachineImageType"] = None,
        spot_instance_draining: typing.Optional[builtins.bool] = None,
        task_drain_time: typing.Optional[_Duration_070aa057] = None,
        topic_encryption_key: typing.Optional[_IKey_36930160] = None,
        auto_scaling_group: _IAutoScalingGroup_352723c6,
        capacity_provider_name: typing.Optional[builtins.str] = None,
        enable_managed_scaling: typing.Optional[builtins.bool] = None,
        enable_managed_termination_protection: typing.Optional[builtins.bool] = None,
        maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
        minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
        target_capacity_percent: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The options for creating an Auto Scaling Group Capacity Provider.

        :param can_containers_access_instance_role: (experimental) Specifies whether the containers can access the container instance role. Default: false
        :param machine_image_type: (experimental) What type of machine image this is. Depending on the setting, different UserData will automatically be added to the ``AutoScalingGroup`` to configure it properly for use with ECS. If you create an ``AutoScalingGroup`` yourself and are adding it via ``addAutoScalingGroup()``, you must specify this value. If you are adding an ``autoScalingGroup`` via ``addCapacity``, this value will be determined from the ``machineImage`` you pass. Default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.
        :param spot_instance_draining: (experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services. For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_. Default: false
        :param task_drain_time: (deprecated) The time period to wait before force terminating an instance that is draining. This creates a Lambda function that is used by a lifecycle hook for the AutoScalingGroup that will delay instance termination until all ECS tasks have drained from the instance. Set to 0 to disable task draining. Set to 0 to disable task draining. Default: Duration.minutes(5)
        :param topic_encryption_key: (experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information. Default: The SNS Topic will not be encrypted.
        :param auto_scaling_group: (experimental) The autoscaling group to add as a Capacity Provider.
        :param capacity_provider_name: (experimental) The name of the capacity provider. If a name is specified, it cannot start with ``aws``, ``ecs``, or ``fargate``. If no name is specified, a default name in the CFNStackName-CFNResourceName-RandomString format is used. Default: CloudFormation-generated name
        :param enable_managed_scaling: (experimental) Whether to enable managed scaling. Default: true
        :param enable_managed_termination_protection: (experimental) Whether to enable managed termination protection. Default: true
        :param maximum_scaling_step_size: (experimental) Maximum scaling step size. In most cases this should be left alone. Default: 1000
        :param minimum_scaling_step_size: (experimental) Minimum scaling step size. In most cases this should be left alone. Default: 1
        :param target_capacity_percent: (experimental) Target capacity percent. In most cases this should be left alone. Default: 100

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # vpc: ec2.Vpc
            
            
            cluster = ecs.Cluster(self, "Cluster",
                vpc=vpc
            )
            
            auto_scaling_group = autoscaling.AutoScalingGroup(self, "ASG",
                vpc=vpc,
                instance_type=ec2.InstanceType("t2.micro"),
                machine_image=ecs.EcsOptimizedImage.amazon_linux2(),
                min_capacity=0,
                max_capacity=100
            )
            
            capacity_provider = ecs.AsgCapacityProvider(self, "AsgCapacityProvider",
                auto_scaling_group=auto_scaling_group
            )
            cluster.add_asg_capacity_provider(capacity_provider)
            
            task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
            
            task_definition.add_container("web",
                image=ecs.ContainerImage.from_registry("amazon/amazon-ecs-sample"),
                memory_reservation_mi_b=256
            )
            
            ecs.Ec2Service(self, "EC2Service",
                cluster=cluster,
                task_definition=task_definition,
                capacity_provider_strategies=[autoscaling.aws_ecs.CapacityProviderStrategy(
                    capacity_provider=capacity_provider.capacity_provider_name,
                    weight=1
                )
                ]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "auto_scaling_group": auto_scaling_group,
        }
        if can_containers_access_instance_role is not None:
            self._values["can_containers_access_instance_role"] = can_containers_access_instance_role
        if machine_image_type is not None:
            self._values["machine_image_type"] = machine_image_type
        if spot_instance_draining is not None:
            self._values["spot_instance_draining"] = spot_instance_draining
        if task_drain_time is not None:
            self._values["task_drain_time"] = task_drain_time
        if topic_encryption_key is not None:
            self._values["topic_encryption_key"] = topic_encryption_key
        if capacity_provider_name is not None:
            self._values["capacity_provider_name"] = capacity_provider_name
        if enable_managed_scaling is not None:
            self._values["enable_managed_scaling"] = enable_managed_scaling
        if enable_managed_termination_protection is not None:
            self._values["enable_managed_termination_protection"] = enable_managed_termination_protection
        if maximum_scaling_step_size is not None:
            self._values["maximum_scaling_step_size"] = maximum_scaling_step_size
        if minimum_scaling_step_size is not None:
            self._values["minimum_scaling_step_size"] = minimum_scaling_step_size
        if target_capacity_percent is not None:
            self._values["target_capacity_percent"] = target_capacity_percent

    @builtins.property
    def can_containers_access_instance_role(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the containers can access the container instance role.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("can_containers_access_instance_role")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def machine_image_type(self) -> typing.Optional["MachineImageType"]:
        '''(experimental) What type of machine image this is.

        Depending on the setting, different UserData will automatically be added
        to the ``AutoScalingGroup`` to configure it properly for use with ECS.

        If you create an ``AutoScalingGroup`` yourself and are adding it via
        ``addAutoScalingGroup()``, you must specify this value. If you are adding an
        ``autoScalingGroup`` via ``addCapacity``, this value will be determined
        from the ``machineImage`` you pass.

        :default: - Automatically determined from ``machineImage``, if available, otherwise ``MachineImageType.AMAZON_LINUX_2``.

        :stability: experimental
        '''
        result = self._values.get("machine_image_type")
        return typing.cast(typing.Optional["MachineImageType"], result)

    @builtins.property
    def spot_instance_draining(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specify whether to enable Automated Draining for Spot Instances running Amazon ECS Services.

        For more information, see `Using Spot Instances <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/container-instance-spot.html>`_.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("spot_instance_draining")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def task_drain_time(self) -> typing.Optional[_Duration_070aa057]:
        '''(deprecated) The time period to wait before force terminating an instance that is draining.

        This creates a Lambda function that is used by a lifecycle hook for the
        AutoScalingGroup that will delay instance termination until all ECS tasks
        have drained from the instance. Set to 0 to disable task draining.

        Set to 0 to disable task draining.

        :default: Duration.minutes(5)

        :deprecated: The lifecycle draining hook is not configured if using the EC2 Capacity Provider. Enable managed termination protection instead.

        :stability: deprecated
        '''
        result = self._values.get("task_drain_time")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def topic_encryption_key(self) -> typing.Optional[_IKey_36930160]:
        '''(experimental) If {@link AddAutoScalingGroupCapacityOptions.taskDrainTime} is non-zero, then the ECS cluster creates an SNS Topic to as part of a system to drain instances of tasks when the instance is being shut down. If this property is provided, then this key will be used to encrypt the contents of that SNS Topic. See `SNS Data Encryption <https://docs.aws.amazon.com/sns/latest/dg/sns-data-encryption.html>`_ for more information.

        :default: The SNS Topic will not be encrypted.

        :stability: experimental
        '''
        result = self._values.get("topic_encryption_key")
        return typing.cast(typing.Optional[_IKey_36930160], result)

    @builtins.property
    def auto_scaling_group(self) -> _IAutoScalingGroup_352723c6:
        '''(experimental) The autoscaling group to add as a Capacity Provider.

        :stability: experimental
        '''
        result = self._values.get("auto_scaling_group")
        assert result is not None, "Required property 'auto_scaling_group' is missing"
        return typing.cast(_IAutoScalingGroup_352723c6, result)

    @builtins.property
    def capacity_provider_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the capacity provider.

        If a name is specified,
        it cannot start with ``aws``, ``ecs``, or ``fargate``. If no name is specified,
        a default name in the CFNStackName-CFNResourceName-RandomString format is used.

        :default: CloudFormation-generated name

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enable_managed_scaling(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable managed scaling.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("enable_managed_scaling")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_managed_termination_protection(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable managed termination protection.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("enable_managed_termination_protection")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def maximum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Maximum scaling step size.

        In most cases this should be left alone.

        :default: 1000

        :stability: experimental
        '''
        result = self._values.get("maximum_scaling_step_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def minimum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Minimum scaling step size.

        In most cases this should be left alone.

        :default: 1

        :stability: experimental
        '''
        result = self._values.get("minimum_scaling_step_size")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def target_capacity_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Target capacity percent.

        In most cases this should be left alone.

        :default: 100

        :stability: experimental
        '''
        result = self._values.get("target_capacity_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AsgCapacityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AssetImageProps",
    jsii_struct_bases=[_DockerImageAssetOptions_6764a0de],
    name_mapping={
        "exclude": "exclude",
        "follow": "follow",
        "ignore_mode": "ignoreMode",
        "extra_hash": "extraHash",
        "follow_symlinks": "followSymlinks",
        "build_args": "buildArgs",
        "file": "file",
        "invalidation": "invalidation",
        "network_mode": "networkMode",
        "repository_name": "repositoryName",
        "target": "target",
    },
)
class AssetImageProps(_DockerImageAssetOptions_6764a0de):
    def __init__(
        self,
        *,
        exclude: typing.Optional[typing.Sequence[builtins.str]] = None,
        follow: typing.Optional[_FollowMode_98b05cc5] = None,
        ignore_mode: typing.Optional[_IgnoreMode_31d8bf46] = None,
        extra_hash: typing.Optional[builtins.str] = None,
        follow_symlinks: typing.Optional[_SymlinkFollowMode_abf4527a] = None,
        build_args: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        file: typing.Optional[builtins.str] = None,
        invalidation: typing.Optional[_DockerImageAssetInvalidationOptions_e5375707] = None,
        network_mode: typing.Optional[_NetworkMode_2d19607e] = None,
        repository_name: typing.Optional[builtins.str] = None,
        target: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The properties for building an AssetImage.

        :param exclude: (experimental) Glob patterns to exclude from the copy. Default: - nothing is excluded
        :param follow: (deprecated) A strategy for how to handle symlinks. Default: Never
        :param ignore_mode: (experimental) The ignore behavior to use for exclude patterns. Default: IgnoreMode.GLOB
        :param extra_hash: (experimental) Extra information to encode into the fingerprint (e.g. build instructions and other inputs). Default: - hash is only based on source content
        :param follow_symlinks: (experimental) A strategy for how to handle symlinks. Default: SymlinkFollowMode.NEVER
        :param build_args: (experimental) Build args to pass to the ``docker build`` command. Since Docker build arguments are resolved before deployment, keys and values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or ``queue.queueUrl``). Default: - no build args are passed
        :param file: (experimental) Path to the Dockerfile (relative to the directory). Default: 'Dockerfile'
        :param invalidation: (experimental) Options to control which parameters are used to invalidate the asset hash. Default: - hash all parameters
        :param network_mode: (experimental) Networking mode for the RUN commands during build. Support docker API 1.25+. Default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)
        :param repository_name: (deprecated) ECR repository name. Specify this property if you need to statically address the image, e.g. from a Kubernetes Pod. Note, this is only the repository name, without the registry and the tag parts. Default: - the default ECR repository for CDK assets
        :param target: (experimental) Docker target to build to. Default: - no target

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import assets
            from monocdk import aws_ecr_assets as ecr_assets
            from monocdk import aws_ecs as ecs
            
            # network_mode: ecr_assets.NetworkMode
            
            asset_image_props = ecs.AssetImageProps(
                build_args={
                    "build_args_key": "buildArgs"
                },
                exclude=["exclude"],
                extra_hash="extraHash",
                file="file",
                follow=assets.FollowMode.NEVER,
                follow_symlinks=monocdk.SymlinkFollowMode.NEVER,
                ignore_mode=monocdk.IgnoreMode.GLOB,
                invalidation=ecr_assets.DockerImageAssetInvalidationOptions(
                    build_args=False,
                    extra_hash=False,
                    file=False,
                    network_mode=False,
                    repository_name=False,
                    target=False
                ),
                network_mode=network_mode,
                repository_name="repositoryName",
                target="target"
            )
        '''
        if isinstance(invalidation, dict):
            invalidation = _DockerImageAssetInvalidationOptions_e5375707(**invalidation)
        self._values: typing.Dict[str, typing.Any] = {}
        if exclude is not None:
            self._values["exclude"] = exclude
        if follow is not None:
            self._values["follow"] = follow
        if ignore_mode is not None:
            self._values["ignore_mode"] = ignore_mode
        if extra_hash is not None:
            self._values["extra_hash"] = extra_hash
        if follow_symlinks is not None:
            self._values["follow_symlinks"] = follow_symlinks
        if build_args is not None:
            self._values["build_args"] = build_args
        if file is not None:
            self._values["file"] = file
        if invalidation is not None:
            self._values["invalidation"] = invalidation
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if repository_name is not None:
            self._values["repository_name"] = repository_name
        if target is not None:
            self._values["target"] = target

    @builtins.property
    def exclude(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) Glob patterns to exclude from the copy.

        :default: - nothing is excluded

        :stability: experimental
        '''
        result = self._values.get("exclude")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def follow(self) -> typing.Optional[_FollowMode_98b05cc5]:
        '''(deprecated) A strategy for how to handle symlinks.

        :default: Never

        :deprecated: use ``followSymlinks`` instead

        :stability: deprecated
        '''
        result = self._values.get("follow")
        return typing.cast(typing.Optional[_FollowMode_98b05cc5], result)

    @builtins.property
    def ignore_mode(self) -> typing.Optional[_IgnoreMode_31d8bf46]:
        '''(experimental) The ignore behavior to use for exclude patterns.

        :default: IgnoreMode.GLOB

        :stability: experimental
        '''
        result = self._values.get("ignore_mode")
        return typing.cast(typing.Optional[_IgnoreMode_31d8bf46], result)

    @builtins.property
    def extra_hash(self) -> typing.Optional[builtins.str]:
        '''(experimental) Extra information to encode into the fingerprint (e.g. build instructions and other inputs).

        :default: - hash is only based on source content

        :stability: experimental
        '''
        result = self._values.get("extra_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def follow_symlinks(self) -> typing.Optional[_SymlinkFollowMode_abf4527a]:
        '''(experimental) A strategy for how to handle symlinks.

        :default: SymlinkFollowMode.NEVER

        :stability: experimental
        '''
        result = self._values.get("follow_symlinks")
        return typing.cast(typing.Optional[_SymlinkFollowMode_abf4527a], result)

    @builtins.property
    def build_args(self) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) Build args to pass to the ``docker build`` command.

        Since Docker build arguments are resolved before deployment, keys and
        values cannot refer to unresolved tokens (such as ``lambda.functionArn`` or
        ``queue.queueUrl``).

        :default: - no build args are passed

        :stability: experimental
        '''
        result = self._values.get("build_args")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def file(self) -> typing.Optional[builtins.str]:
        '''(experimental) Path to the Dockerfile (relative to the directory).

        :default: 'Dockerfile'

        :stability: experimental
        '''
        result = self._values.get("file")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def invalidation(
        self,
    ) -> typing.Optional[_DockerImageAssetInvalidationOptions_e5375707]:
        '''(experimental) Options to control which parameters are used to invalidate the asset hash.

        :default: - hash all parameters

        :stability: experimental
        '''
        result = self._values.get("invalidation")
        return typing.cast(typing.Optional[_DockerImageAssetInvalidationOptions_e5375707], result)

    @builtins.property
    def network_mode(self) -> typing.Optional[_NetworkMode_2d19607e]:
        '''(experimental) Networking mode for the RUN commands during build.

        Support docker API 1.25+.

        :default: - no networking mode specified (the default networking mode ``NetworkMode.DEFAULT`` will be used)

        :stability: experimental
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[_NetworkMode_2d19607e], result)

    @builtins.property
    def repository_name(self) -> typing.Optional[builtins.str]:
        '''(deprecated) ECR repository name.

        Specify this property if you need to statically address the image, e.g.
        from a Kubernetes Pod. Note, this is only the repository name, without the
        registry and the tag parts.

        :default: - the default ECR repository for CDK assets

        :deprecated:

        to control the location of docker image assets, please override
        ``Stack.addDockerImageAsset``. this feature will be removed in future
        releases.

        :stability: deprecated
        '''
        result = self._values.get("repository_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target(self) -> typing.Optional[builtins.str]:
        '''(experimental) Docker target to build to.

        :default: - no target

        :stability: experimental
        '''
        result = self._values.get("target")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssetImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AssociateCloudMapServiceOptions",
    jsii_struct_bases=[],
    name_mapping={
        "service": "service",
        "container": "container",
        "container_port": "containerPort",
    },
)
class AssociateCloudMapServiceOptions:
    def __init__(
        self,
        *,
        service: _IService_66c1fbd2,
        container: typing.Optional["ContainerDefinition"] = None,
        container_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) The options for using a cloudmap service.

        :param service: (experimental) The cloudmap service to register with.
        :param container: (experimental) The container to point to for a SRV record. Default: - the task definition's default container
        :param container_port: (experimental) The port to point to for a SRV record. Default: - the default port of the task definition's default container

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # cloud_map_service: cloudmap.Service
            # ecs_service: ecs.FargateService
            
            
            ecs_service.associate_cloud_map_service(
                service=cloud_map_service
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "service": service,
        }
        if container is not None:
            self._values["container"] = container
        if container_port is not None:
            self._values["container_port"] = container_port

    @builtins.property
    def service(self) -> _IService_66c1fbd2:
        '''(experimental) The cloudmap service to register with.

        :stability: experimental
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(_IService_66c1fbd2, result)

    @builtins.property
    def container(self) -> typing.Optional["ContainerDefinition"]:
        '''(experimental) The container to point to for a SRV record.

        :default: - the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container")
        return typing.cast(typing.Optional["ContainerDefinition"], result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port to point to for a SRV record.

        :default: - the default port of the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssociateCloudMapServiceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AuthorizationConfig",
    jsii_struct_bases=[],
    name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
)
class AuthorizationConfig:
    def __init__(
        self,
        *,
        access_point_id: typing.Optional[builtins.str] = None,
        iam: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The authorization configuration details for the Amazon EFS file system.

        :param access_point_id: (experimental) The access point ID to use. If an access point is specified, the root directory value will be relative to the directory set for the access point. If specified, transit encryption must be enabled in the EFSVolumeConfiguration. Default: No id
        :param iam: (experimental) Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system. If enabled, transit encryption must be enabled in the EFSVolumeConfiguration. Valid values: ENABLED | DISABLED Default: If this parameter is omitted, the default value of DISABLED is used.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            authorization_config = ecs.AuthorizationConfig(
                access_point_id="accessPointId",
                iam="iam"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if access_point_id is not None:
            self._values["access_point_id"] = access_point_id
        if iam is not None:
            self._values["iam"] = iam

    @builtins.property
    def access_point_id(self) -> typing.Optional[builtins.str]:
        '''(experimental) The access point ID to use.

        If an access point is specified, the root directory value will be
        relative to the directory set for the access point.
        If specified, transit encryption must be enabled in the EFSVolumeConfiguration.

        :default: No id

        :stability: experimental
        '''
        result = self._values.get("access_point_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam(self) -> typing.Optional[builtins.str]:
        '''(experimental) Whether or not to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system.

        If enabled, transit encryption must be enabled in the EFSVolumeConfiguration.

        Valid values: ENABLED | DISABLED

        :default: If this parameter is omitted, the default value of DISABLED is used.

        :stability: experimental
        '''
        result = self._values.get("iam")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AuthorizationConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.AwsLogDriverMode")
class AwsLogDriverMode(enum.Enum):
    '''(experimental) awslogs provides two modes for delivering messages from the container to the log driver.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # cluster: ecs.Cluster
        
        # Create a Task Definition for the container to start
        task_definition = ecs.Ec2TaskDefinition(self, "TaskDef")
        task_definition.add_container("TheContainer",
            image=ecs.ContainerImage.from_asset(path.resolve(__dirname, "..", "eventhandler-image")),
            memory_limit_mi_b=256,
            logging=ecs.AwsLogDriver(stream_prefix="EventDemo", mode=ecs.AwsLogDriverMode.NON_BLOCKING)
        )
        
        # An Rule that describes the event trigger (in this case a scheduled run)
        rule = events.Rule(self, "Rule",
            schedule=events.Schedule.expression("rate(1 min)")
        )
        
        # Pass an environment variable to the container 'TheContainer' in the task
        rule.add_target(targets.EcsTask(
            cluster=cluster,
            task_definition=task_definition,
            task_count=1,
            container_overrides=[autoscaling.aws_events_targets.ContainerOverride(
                container_name="TheContainer",
                environment=[autoscaling.aws_events_targets.TaskEnvironmentVariable(
                    name="I_WAS_TRIGGERED",
                    value="From CloudWatch Events"
                )]
            )]
        ))
    '''

    BLOCKING = "BLOCKING"
    '''(experimental) (default) direct, blocking delivery from container to driver.

    :stability: experimental
    '''
    NON_BLOCKING = "NON_BLOCKING"
    '''(experimental) The non-blocking message delivery mode prevents applications from blocking due to logging back pressure.

    Applications are likely to fail in unexpected ways when STDERR or STDOUT streams block.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.AwsLogDriverProps",
    jsii_struct_bases=[],
    name_mapping={
        "stream_prefix": "streamPrefix",
        "datetime_format": "datetimeFormat",
        "log_group": "logGroup",
        "log_retention": "logRetention",
        "mode": "mode",
        "multiline_pattern": "multilinePattern",
    },
)
class AwsLogDriverProps:
    def __init__(
        self,
        *,
        stream_prefix: builtins.str,
        datetime_format: typing.Optional[builtins.str] = None,
        log_group: typing.Optional[_ILogGroup_846e17a0] = None,
        log_retention: typing.Optional[_RetentionDays_6c560d31] = None,
        mode: typing.Optional[AwsLogDriverMode] = None,
        multiline_pattern: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Specifies the awslogs log driver configuration options.

        :param stream_prefix: (experimental) Prefix for the log streams. The awslogs-stream-prefix option allows you to associate a log stream with the specified prefix, the container name, and the ID of the Amazon ECS task to which the container belongs. If you specify a prefix with this option, then the log stream takes the following format:: prefix-name/container-name/ecs-task-id
        :param datetime_format: (experimental) This option defines a multiline start pattern in Python strftime format. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. Default: - No multiline matching.
        :param log_group: (experimental) The log group to log to. Default: - A log group is automatically created.
        :param log_retention: (experimental) The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct. Default: - Logs never expire.
        :param mode: (experimental) The delivery mode of log messages from the container to awslogs. Default: - AwsLogDriverMode.BLOCKING
        :param multiline_pattern: (experimental) This option defines a multiline start pattern using a regular expression. A log message consists of a line that matches the pattern and any following lines that don’t match the pattern. Thus the matched line is the delimiter between log messages. This option is ignored if datetimeFormat is also configured. Default: - No multiline matching.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # Create a Task Definition for the Windows container to start
            task_definition = ecs.FargateTaskDefinition(self, "TaskDef",
                runtime_platform=autoscaling.aws_ecs.RuntimePlatform(
                    operating_system_family=ecs.OperatingSystemFamily.WINDOWS_SERVER_2019_CORE,
                    cpu_architecture=ecs.CpuArchitecture.X86_64
                ),
                cpu=1024,
                memory_limit_mi_b=2048
            )
            
            task_definition.add_container("windowsservercore",
                logging=ecs.LogDriver.aws_logs(stream_prefix="win-iis-on-fargate"),
                port_mappings=[autoscaling.aws_ecs.PortMapping(container_port=80)],
                image=ecs.ContainerImage.from_registry("mcr.microsoft.com/windows/servercore/iis:windowsservercore-ltsc2019")
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "stream_prefix": stream_prefix,
        }
        if datetime_format is not None:
            self._values["datetime_format"] = datetime_format
        if log_group is not None:
            self._values["log_group"] = log_group
        if log_retention is not None:
            self._values["log_retention"] = log_retention
        if mode is not None:
            self._values["mode"] = mode
        if multiline_pattern is not None:
            self._values["multiline_pattern"] = multiline_pattern

    @builtins.property
    def stream_prefix(self) -> builtins.str:
        '''(experimental) Prefix for the log streams.

        The awslogs-stream-prefix option allows you to associate a log stream
        with the specified prefix, the container name, and the ID of the Amazon
        ECS task to which the container belongs. If you specify a prefix with
        this option, then the log stream takes the following format::

            prefix-name/container-name/ecs-task-id

        :stability: experimental
        '''
        result = self._values.get("stream_prefix")
        assert result is not None, "Required property 'stream_prefix' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def datetime_format(self) -> typing.Optional[builtins.str]:
        '''(experimental) This option defines a multiline start pattern in Python strftime format.

        A log message consists of a line that matches the pattern and any
        following lines that don’t match the pattern. Thus the matched line is
        the delimiter between log messages.

        :default: - No multiline matching.

        :stability: experimental
        '''
        result = self._values.get("datetime_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group(self) -> typing.Optional[_ILogGroup_846e17a0]:
        '''(experimental) The log group to log to.

        :default: - A log group is automatically created.

        :stability: experimental
        '''
        result = self._values.get("log_group")
        return typing.cast(typing.Optional[_ILogGroup_846e17a0], result)

    @builtins.property
    def log_retention(self) -> typing.Optional[_RetentionDays_6c560d31]:
        '''(experimental) The number of days log events are kept in CloudWatch Logs when the log group is automatically created by this construct.

        :default: - Logs never expire.

        :stability: experimental
        '''
        result = self._values.get("log_retention")
        return typing.cast(typing.Optional[_RetentionDays_6c560d31], result)

    @builtins.property
    def mode(self) -> typing.Optional[AwsLogDriverMode]:
        '''(experimental) The delivery mode of log messages from the container to awslogs.

        :default: - AwsLogDriverMode.BLOCKING

        :stability: experimental
        '''
        result = self._values.get("mode")
        return typing.cast(typing.Optional[AwsLogDriverMode], result)

    @builtins.property
    def multiline_pattern(self) -> typing.Optional[builtins.str]:
        '''(experimental) This option defines a multiline start pattern using a regular expression.

        A log message consists of a line that matches the pattern and any
        following lines that don’t match the pattern. Thus the matched line is
        the delimiter between log messages.

        This option is ignored if datetimeFormat is also configured.

        :default: - No multiline matching.

        :stability: experimental
        '''
        result = self._values.get("multiline_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BaseLogDriverProps",
    jsii_struct_bases=[],
    name_mapping={
        "env": "env",
        "env_regex": "envRegex",
        "labels": "labels",
        "tag": "tag",
    },
)
class BaseLogDriverProps:
    def __init__(
        self,
        *,
        env: typing.Optional[typing.Sequence[builtins.str]] = None,
        env_regex: typing.Optional[builtins.str] = None,
        labels: typing.Optional[typing.Sequence[builtins.str]] = None,
        tag: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param env: (experimental) The env option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No env
        :param env_regex: (experimental) The env-regex option is similar to and compatible with env. Its value is a regular expression to match logging-related environment variables. It is used for advanced log tag options. Default: - No envRegex
        :param labels: (experimental) The labels option takes an array of keys. If there is collision between label and env keys, the value of the env takes precedence. Adds additional fields to the extra attributes of a logging message. Default: - No labels
        :param tag: (experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages. Refer to the log tag option documentation for customizing the log tag format. Default: - The first 12 characters of the container ID

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            base_log_driver_props = ecs.BaseLogDriverProps(
                env=["env"],
                env_regex="envRegex",
                labels=["labels"],
                tag="tag"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if env is not None:
            self._values["env"] = env
        if env_regex is not None:
            self._values["env_regex"] = env_regex
        if labels is not None:
            self._values["labels"] = labels
        if tag is not None:
            self._values["tag"] = tag

    @builtins.property
    def env(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The env option takes an array of keys.

        If there is collision between
        label and env keys, the value of the env takes precedence. Adds additional fields
        to the extra attributes of a logging message.

        :default: - No env

        :stability: experimental
        '''
        result = self._values.get("env")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def env_regex(self) -> typing.Optional[builtins.str]:
        '''(experimental) The env-regex option is similar to and compatible with env.

        Its value is a regular
        expression to match logging-related environment variables. It is used for advanced
        log tag options.

        :default: - No envRegex

        :stability: experimental
        '''
        result = self._values.get("env_regex")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def labels(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The labels option takes an array of keys.

        If there is collision
        between label and env keys, the value of the env takes precedence. Adds additional
        fields to the extra attributes of a logging message.

        :default: - No labels

        :stability: experimental
        '''
        result = self._values.get("labels")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tag(self) -> typing.Optional[builtins.str]:
        '''(experimental) By default, Docker uses the first 12 characters of the container ID to tag log messages.

        Refer to the log tag option documentation for customizing the
        log tag format.

        :default: - The first 12 characters of the container ID

        :stability: experimental
        '''
        result = self._values.get("tag")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseLogDriverProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BaseServiceOptions",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "propagate_task_tags_from": "propagateTaskTagsFrom",
        "service_name": "serviceName",
    },
)
class BaseServiceOptions:
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence["CapacityProviderStrategy"]] = None,
        circuit_breaker: typing.Optional["DeploymentCircuitBreaker"] = None,
        cloud_map_options: typing.Optional["CloudMapOptions"] = None,
        deployment_controller: typing.Optional["DeploymentController"] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        enable_execute_command: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_070aa057] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        propagate_task_tags_from: typing.Optional["PropagatedTagSource"] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The properties for the base Ec2Service or FargateService service.

        :param cluster: (experimental) The name of the cluster that hosts the service.
        :param capacity_provider_strategies: (experimental) A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: (experimental) Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: (experimental) The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_controller: (experimental) Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: (experimental) The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: (experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param enable_execute_command: (experimental) Whether to enable the ability to execute into a container. Default: - undefined
        :param health_check_grace_period: (experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: (experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: (experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: (experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param propagate_task_tags_from: (deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Tags can only be propagated to the tasks within the service during service creation. Default: PropagatedTagSource.NONE
        :param service_name: (experimental) The name of the service. Default: - CloudFormation-generated name.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_ecs as ecs
            from monocdk import aws_servicediscovery as servicediscovery
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # duration: monocdk.Duration
            # namespace: servicediscovery.INamespace
            
            base_service_options = ecs.BaseServiceOptions(
                cluster=cluster,
            
                # the properties below are optional
                capacity_provider_strategies=[ecs.CapacityProviderStrategy(
                    capacity_provider="capacityProvider",
            
                    # the properties below are optional
                    base=123,
                    weight=123
                )],
                circuit_breaker=ecs.DeploymentCircuitBreaker(
                    rollback=False
                ),
                cloud_map_options=ecs.CloudMapOptions(
                    cloud_map_namespace=namespace,
                    container=container_definition,
                    container_port=123,
                    dns_record_type=servicediscovery.DnsRecordType.A,
                    dns_ttl=duration,
                    failure_threshold=123,
                    name="name"
                ),
                deployment_controller=ecs.DeploymentController(
                    type=ecs.DeploymentControllerType.ECS
                ),
                desired_count=123,
                enable_eCSManaged_tags=False,
                enable_execute_command=False,
                health_check_grace_period=duration,
                max_healthy_percent=123,
                min_healthy_percent=123,
                propagate_tags=ecs.PropagatedTagSource.SERVICE,
                propagate_task_tags_from=ecs.PropagatedTagSource.SERVICE,
                service_name="serviceName"
            )
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if propagate_task_tags_from is not None:
            self._values["propagate_task_tags_from"] = propagate_task_tags_from
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The name of the cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List["CapacityProviderStrategy"]]:
        '''(experimental) A list of Capacity Provider strategies used to place a service.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List["CapacityProviderStrategy"]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional["DeploymentCircuitBreaker"]:
        '''(experimental) Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled

        :stability: experimental
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional["DeploymentCircuitBreaker"], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional["CloudMapOptions"]:
        '''(experimental) The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.

        :stability: experimental
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional["CloudMapOptions"], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional["DeploymentController"]:
        '''(experimental) Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)

        :stability: experimental
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional["DeploymentController"], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.

        :stability: experimental
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_execute_command(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable the ability to execute into a container.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set

        :stability: experimental
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200

        :stability: experimental
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50

        :stability: experimental
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''(experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE

        :stability: experimental
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def propagate_task_tags_from(self) -> typing.Optional["PropagatedTagSource"]:
        '''(deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Tags can only be propagated to the tasks within the service during service creation.

        :default: PropagatedTagSource.NONE

        :deprecated: Use ``propagateTags`` instead.

        :stability: deprecated
        '''
        result = self._values.get("propagate_task_tags_from")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - CloudFormation-generated name.

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseServiceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BaseServiceProps",
    jsii_struct_bases=[BaseServiceOptions],
    name_mapping={
        "cluster": "cluster",
        "capacity_provider_strategies": "capacityProviderStrategies",
        "circuit_breaker": "circuitBreaker",
        "cloud_map_options": "cloudMapOptions",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableECSManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period": "healthCheckGracePeriod",
        "max_healthy_percent": "maxHealthyPercent",
        "min_healthy_percent": "minHealthyPercent",
        "propagate_tags": "propagateTags",
        "propagate_task_tags_from": "propagateTaskTagsFrom",
        "service_name": "serviceName",
        "launch_type": "launchType",
    },
)
class BaseServiceProps(BaseServiceOptions):
    def __init__(
        self,
        *,
        cluster: "ICluster",
        capacity_provider_strategies: typing.Optional[typing.Sequence["CapacityProviderStrategy"]] = None,
        circuit_breaker: typing.Optional["DeploymentCircuitBreaker"] = None,
        cloud_map_options: typing.Optional["CloudMapOptions"] = None,
        deployment_controller: typing.Optional["DeploymentController"] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[builtins.bool] = None,
        enable_execute_command: typing.Optional[builtins.bool] = None,
        health_check_grace_period: typing.Optional[_Duration_070aa057] = None,
        max_healthy_percent: typing.Optional[jsii.Number] = None,
        min_healthy_percent: typing.Optional[jsii.Number] = None,
        propagate_tags: typing.Optional["PropagatedTagSource"] = None,
        propagate_task_tags_from: typing.Optional["PropagatedTagSource"] = None,
        service_name: typing.Optional[builtins.str] = None,
        launch_type: "LaunchType",
    ) -> None:
        '''(experimental) Complete base service properties that are required to be supplied by the implementation of the BaseService class.

        :param cluster: (experimental) The name of the cluster that hosts the service.
        :param capacity_provider_strategies: (experimental) A list of Capacity Provider strategies used to place a service. Default: - undefined
        :param circuit_breaker: (experimental) Whether to enable the deployment circuit breaker. If this property is defined, circuit breaker will be implicitly enabled. Default: - disabled
        :param cloud_map_options: (experimental) The options for configuring an Amazon ECS service to use service discovery. Default: - AWS Cloud Map service discovery is not enabled.
        :param deployment_controller: (experimental) Specifies which deployment controller to use for the service. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ Default: - Rolling update (ECS)
        :param desired_count: (experimental) The desired number of instantiations of the task definition to keep running on the service. Default: - When creating the service, default is 1; when updating the service, default uses the current task number.
        :param enable_ecs_managed_tags: (experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ Default: false
        :param enable_execute_command: (experimental) Whether to enable the ability to execute into a container. Default: - undefined
        :param health_check_grace_period: (experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. Default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set
        :param max_healthy_percent: (experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment. Default: - 100 if daemon, otherwise 200
        :param min_healthy_percent: (experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment. Default: - 0 if daemon, otherwise 50
        :param propagate_tags: (experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE Default: PropagatedTagSource.NONE
        :param propagate_task_tags_from: (deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. Tags can only be propagated to the tasks within the service during service creation. Default: PropagatedTagSource.NONE
        :param service_name: (experimental) The name of the service. Default: - CloudFormation-generated name.
        :param launch_type: (experimental) The launch type on which to run your service. LaunchType will be omitted if capacity provider strategies are specified on the service.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_ecs as ecs
            from monocdk import aws_servicediscovery as servicediscovery
            
            # cluster: ecs.Cluster
            # container_definition: ecs.ContainerDefinition
            # duration: monocdk.Duration
            # namespace: servicediscovery.INamespace
            
            base_service_props = ecs.BaseServiceProps(
                cluster=cluster,
                launch_type=ecs.LaunchType.EC2,
            
                # the properties below are optional
                capacity_provider_strategies=[ecs.CapacityProviderStrategy(
                    capacity_provider="capacityProvider",
            
                    # the properties below are optional
                    base=123,
                    weight=123
                )],
                circuit_breaker=ecs.DeploymentCircuitBreaker(
                    rollback=False
                ),
                cloud_map_options=ecs.CloudMapOptions(
                    cloud_map_namespace=namespace,
                    container=container_definition,
                    container_port=123,
                    dns_record_type=servicediscovery.DnsRecordType.A,
                    dns_ttl=duration,
                    failure_threshold=123,
                    name="name"
                ),
                deployment_controller=ecs.DeploymentController(
                    type=ecs.DeploymentControllerType.ECS
                ),
                desired_count=123,
                enable_eCSManaged_tags=False,
                enable_execute_command=False,
                health_check_grace_period=duration,
                max_healthy_percent=123,
                min_healthy_percent=123,
                propagate_tags=ecs.PropagatedTagSource.SERVICE,
                propagate_task_tags_from=ecs.PropagatedTagSource.SERVICE,
                service_name="serviceName"
            )
        '''
        if isinstance(circuit_breaker, dict):
            circuit_breaker = DeploymentCircuitBreaker(**circuit_breaker)
        if isinstance(cloud_map_options, dict):
            cloud_map_options = CloudMapOptions(**cloud_map_options)
        if isinstance(deployment_controller, dict):
            deployment_controller = DeploymentController(**deployment_controller)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "launch_type": launch_type,
        }
        if capacity_provider_strategies is not None:
            self._values["capacity_provider_strategies"] = capacity_provider_strategies
        if circuit_breaker is not None:
            self._values["circuit_breaker"] = circuit_breaker
        if cloud_map_options is not None:
            self._values["cloud_map_options"] = cloud_map_options
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period is not None:
            self._values["health_check_grace_period"] = health_check_grace_period
        if max_healthy_percent is not None:
            self._values["max_healthy_percent"] = max_healthy_percent
        if min_healthy_percent is not None:
            self._values["min_healthy_percent"] = min_healthy_percent
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if propagate_task_tags_from is not None:
            self._values["propagate_task_tags_from"] = propagate_task_tags_from
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def cluster(self) -> "ICluster":
        '''(experimental) The name of the cluster that hosts the service.

        :stability: experimental
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast("ICluster", result)

    @builtins.property
    def capacity_provider_strategies(
        self,
    ) -> typing.Optional[typing.List["CapacityProviderStrategy"]]:
        '''(experimental) A list of Capacity Provider strategies used to place a service.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("capacity_provider_strategies")
        return typing.cast(typing.Optional[typing.List["CapacityProviderStrategy"]], result)

    @builtins.property
    def circuit_breaker(self) -> typing.Optional["DeploymentCircuitBreaker"]:
        '''(experimental) Whether to enable the deployment circuit breaker.

        If this property is defined, circuit breaker will be implicitly
        enabled.

        :default: - disabled

        :stability: experimental
        '''
        result = self._values.get("circuit_breaker")
        return typing.cast(typing.Optional["DeploymentCircuitBreaker"], result)

    @builtins.property
    def cloud_map_options(self) -> typing.Optional["CloudMapOptions"]:
        '''(experimental) The options for configuring an Amazon ECS service to use service discovery.

        :default: - AWS Cloud Map service discovery is not enabled.

        :stability: experimental
        '''
        result = self._values.get("cloud_map_options")
        return typing.cast(typing.Optional["CloudMapOptions"], result)

    @builtins.property
    def deployment_controller(self) -> typing.Optional["DeploymentController"]:
        '''(experimental) Specifies which deployment controller to use for the service.

        For more information, see
        `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_

        :default: - Rolling update (ECS)

        :stability: experimental
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional["DeploymentController"], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The desired number of instantiations of the task definition to keep running on the service.

        :default:

        - When creating the service, default is 1; when updating the service, default uses
        the current task number.

        :stability: experimental
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether to enable Amazon ECS managed tags for the tasks within the service.

        For more information, see
        `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def enable_execute_command(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable the ability to execute into a container.

        :default: - undefined

        :stability: experimental
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def health_check_grace_period(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        :default: - defaults to 60 seconds if at least one load balancer is in-use and it is not already set

        :stability: experimental
        '''
        result = self._values.get("health_check_grace_period")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def max_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The maximum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that can run in a service during a deployment.

        :default: - 100 if daemon, otherwise 200

        :stability: experimental
        '''
        result = self._values.get("max_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_healthy_percent(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of tasks, specified as a percentage of the Amazon ECS service's DesiredCount value, that must continue to run and remain healthy during a deployment.

        :default: - 0 if daemon, otherwise 50

        :stability: experimental
        '''
        result = self._values.get("min_healthy_percent")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional["PropagatedTagSource"]:
        '''(experimental) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Valid values are: PropagatedTagSource.SERVICE, PropagatedTagSource.TASK_DEFINITION or PropagatedTagSource.NONE

        :default: PropagatedTagSource.NONE

        :stability: experimental
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def propagate_task_tags_from(self) -> typing.Optional["PropagatedTagSource"]:
        '''(deprecated) Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        Tags can only be propagated to the tasks within the service during service creation.

        :default: PropagatedTagSource.NONE

        :deprecated: Use ``propagateTags`` instead.

        :stability: deprecated
        '''
        result = self._values.get("propagate_task_tags_from")
        return typing.cast(typing.Optional["PropagatedTagSource"], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the service.

        :default: - CloudFormation-generated name.

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_type(self) -> "LaunchType":
        '''(experimental) The launch type on which to run your service.

        LaunchType will be omitted if capacity provider strategies are specified on the service.

        :see:

        - https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy

        Valid values are: LaunchType.ECS or LaunchType.FARGATE or LaunchType.EXTERNAL
        :stability: experimental
        '''
        result = self._values.get("launch_type")
        assert result is not None, "Required property 'launch_type' is missing"
        return typing.cast("LaunchType", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BaseServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.BinPackResource")
class BinPackResource(enum.Enum):
    '''(experimental) Instance resource used for bin packing.

    :stability: experimental
    '''

    CPU = "CPU"
    '''(experimental) Fill up hosts' CPU allocations first.

    :stability: experimental
    '''
    MEMORY = "MEMORY"
    '''(experimental) Fill up hosts' memory allocations first.

    :stability: experimental
    '''


@jsii.implements(_IMachineImage_45d3238d)
class BottleRocketImage(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.BottleRocketImage",
):
    '''(experimental) Construct an Bottlerocket image from the latest AMI published in SSM.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # cluster: ecs.Cluster
        
        
        cluster.add_capacity("bottlerocket-asg",
            min_capacity=2,
            instance_type=ec2.InstanceType("c5.large"),
            machine_image=ecs.BottleRocketImage()
        )
    '''

    def __init__(
        self,
        *,
        architecture: typing.Optional[_InstanceArchitecture_8b47f118] = None,
        cached_in_context: typing.Optional[builtins.bool] = None,
        variant: typing.Optional["BottlerocketEcsVariant"] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the BottleRocketImage class.

        :param architecture: (experimental) The CPU architecture. Default: - x86_64
        :param cached_in_context: (experimental) Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param variant: (experimental) The Amazon ECS variant to use. Only ``aws-ecs-1`` is currently available Default: - BottlerocketEcsVariant.AWS_ECS_1

        :stability: experimental
        '''
        props = BottleRocketImageProps(
            architecture=architecture,
            cached_in_context=cached_in_context,
            variant=variant,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="getImage")
    def get_image(self, scope: _Construct_e78e779f) -> _MachineImageConfig_963798fe:
        '''(experimental) Return the correct image.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast(_MachineImageConfig_963798fe, jsii.invoke(self, "getImage", [scope]))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.BottleRocketImageProps",
    jsii_struct_bases=[],
    name_mapping={
        "architecture": "architecture",
        "cached_in_context": "cachedInContext",
        "variant": "variant",
    },
)
class BottleRocketImageProps:
    def __init__(
        self,
        *,
        architecture: typing.Optional[_InstanceArchitecture_8b47f118] = None,
        cached_in_context: typing.Optional[builtins.bool] = None,
        variant: typing.Optional["BottlerocketEcsVariant"] = None,
    ) -> None:
        '''(experimental) Properties for BottleRocketImage.

        :param architecture: (experimental) The CPU architecture. Default: - x86_64
        :param cached_in_context: (experimental) Whether the AMI ID is cached to be stable between deployments. By default, the newest image is used on each deployment. This will cause instances to be replaced whenever a new version is released, and may cause downtime if there aren't enough running instances in the AutoScalingGroup to reschedule the tasks on. If set to true, the AMI ID will be cached in ``cdk.context.json`` and the same value will be used on future runs. Your instances will not be replaced but your AMI version will grow old over time. To refresh the AMI lookup, you will have to evict the value from the cache using the ``cdk context`` command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information. Can not be set to ``true`` in environment-agnostic stacks. Default: false
        :param variant: (experimental) The Amazon ECS variant to use. Only ``aws-ecs-1`` is currently available Default: - BottlerocketEcsVariant.AWS_ECS_1

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ec2 as ec2
            from monocdk import aws_ecs as ecs
            
            bottle_rocket_image_props = ecs.BottleRocketImageProps(
                architecture=ec2.InstanceArchitecture.ARM_64,
                cached_in_context=False,
                variant=ecs.BottlerocketEcsVariant.AWS_ECS_1
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if architecture is not None:
            self._values["architecture"] = architecture
        if cached_in_context is not None:
            self._values["cached_in_context"] = cached_in_context
        if variant is not None:
            self._values["variant"] = variant

    @builtins.property
    def architecture(self) -> typing.Optional[_InstanceArchitecture_8b47f118]:
        '''(experimental) The CPU architecture.

        :default: - x86_64

        :stability: experimental
        '''
        result = self._values.get("architecture")
        return typing.cast(typing.Optional[_InstanceArchitecture_8b47f118], result)

    @builtins.property
    def cached_in_context(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether the AMI ID is cached to be stable between deployments.

        By default, the newest image is used on each deployment. This will cause
        instances to be replaced whenever a new version is released, and may cause
        downtime if there aren't enough running instances in the AutoScalingGroup
        to reschedule the tasks on.

        If set to true, the AMI ID will be cached in ``cdk.context.json`` and the
        same value will be used on future runs. Your instances will not be replaced
        but your AMI version will grow old over time. To refresh the AMI lookup,
        you will have to evict the value from the cache using the ``cdk context``
        command. See https://docs.aws.amazon.com/cdk/latest/guide/context.html for
        more information.

        Can not be set to ``true`` in environment-agnostic stacks.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("cached_in_context")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def variant(self) -> typing.Optional["BottlerocketEcsVariant"]:
        '''(experimental) The Amazon ECS variant to use.

        Only ``aws-ecs-1`` is currently available

        :default: - BottlerocketEcsVariant.AWS_ECS_1

        :stability: experimental
        '''
        result = self._values.get("variant")
        return typing.cast(typing.Optional["BottlerocketEcsVariant"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BottleRocketImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.BottlerocketEcsVariant")
class BottlerocketEcsVariant(enum.Enum):
    '''(experimental) Amazon ECS variant.

    :stability: experimental
    '''

    AWS_ECS_1 = "AWS_ECS_1"
    '''(experimental) aws-ecs-1 variant.

    :stability: experimental
    '''


class BuiltInAttributes(
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.BuiltInAttributes",
):
    '''(experimental) The built-in container instance attributes.

    :stability: experimental
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        built_in_attributes = ecs.BuiltInAttributes()
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="AMI_ID")
    def AMI_ID(cls) -> builtins.str:
        '''(experimental) The AMI id the instance is using.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "AMI_ID"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="AVAILABILITY_ZONE")
    def AVAILABILITY_ZONE(cls) -> builtins.str:
        '''(experimental) The AvailabilityZone where the instance is running in.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "AVAILABILITY_ZONE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INSTANCE_ID")
    def INSTANCE_ID(cls) -> builtins.str:
        '''(experimental) The id of the instance.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "INSTANCE_ID"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INSTANCE_TYPE")
    def INSTANCE_TYPE(cls) -> builtins.str:
        '''(experimental) The EC2 instance type.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "INSTANCE_TYPE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="OS_TYPE")
    def OS_TYPE(cls) -> builtins.str:
        '''(experimental) The operating system of the instance.

        Either 'linux' or 'windows'.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.sget(cls, "OS_TYPE"))


@jsii.enum(jsii_type="monocdk.aws_ecs.Capability")
class Capability(enum.Enum):
    '''(experimental) A Linux capability.

    :stability: experimental
    '''

    ALL = "ALL"
    '''
    :stability: experimental
    '''
    AUDIT_CONTROL = "AUDIT_CONTROL"
    '''
    :stability: experimental
    '''
    AUDIT_WRITE = "AUDIT_WRITE"
    '''
    :stability: experimental
    '''
    BLOCK_SUSPEND = "BLOCK_SUSPEND"
    '''
    :stability: experimental
    '''
    CHOWN = "CHOWN"
    '''
    :stability: experimental
    '''
    DAC_OVERRIDE = "DAC_OVERRIDE"
    '''
    :stability: experimental
    '''
    DAC_READ_SEARCH = "DAC_READ_SEARCH"
    '''
    :stability: experimental
    '''
    FOWNER = "FOWNER"
    '''
    :stability: experimental
    '''
    FSETID = "FSETID"
    '''
    :stability: experimental
    '''
    IPC_LOCK = "IPC_LOCK"
    '''
    :stability: experimental
    '''
    IPC_OWNER = "IPC_OWNER"
    '''
    :stability: experimental
    '''
    KILL = "KILL"
    '''
    :stability: experimental
    '''
    LEASE = "LEASE"
    '''
    :stability: experimental
    '''
    LINUX_IMMUTABLE = "LINUX_IMMUTABLE"
    '''
    :stability: experimental
    '''
    MAC_ADMIN = "MAC_ADMIN"
    '''
    :stability: experimental
    '''
    MAC_OVERRIDE = "MAC_OVERRIDE"
    '''
    :stability: experimental
    '''
    MKNOD = "MKNOD"
    '''
    :stability: experimental
    '''
    NET_ADMIN = "NET_ADMIN"
    '''
    :stability: experimental
    '''
    NET_BIND_SERVICE = "NET_BIND_SERVICE"
    '''
    :stability: experimental
    '''
    NET_BROADCAST = "NET_BROADCAST"
    '''
    :stability: experimental
    '''
    NET_RAW = "NET_RAW"
    '''
    :stability: experimental
    '''
    SETFCAP = "SETFCAP"
    '''
    :stability: experimental
    '''
    SETGID = "SETGID"
    '''
    :stability: experimental
    '''
    SETPCAP = "SETPCAP"
    '''
    :stability: experimental
    '''
    SETUID = "SETUID"
    '''
    :stability: experimental
    '''
    SYS_ADMIN = "SYS_ADMIN"
    '''
    :stability: experimental
    '''
    SYS_BOOT = "SYS_BOOT"
    '''
    :stability: experimental
    '''
    SYS_CHROOT = "SYS_CHROOT"
    '''
    :stability: experimental
    '''
    SYS_MODULE = "SYS_MODULE"
    '''
    :stability: experimental
    '''
    SYS_NICE = "SYS_NICE"
    '''
    :stability: experimental
    '''
    SYS_PACCT = "SYS_PACCT"
    '''
    :stability: experimental
    '''
    SYS_PTRACE = "SYS_PTRACE"
    '''
    :stability: experimental
    '''
    SYS_RAWIO = "SYS_RAWIO"
    '''
    :stability: experimental
    '''
    SYS_RESOURCE = "SYS_RESOURCE"
    '''
    :stability: experimental
    '''
    SYS_TIME = "SYS_TIME"
    '''
    :stability: experimental
    '''
    SYS_TTY_CONFIG = "SYS_TTY_CONFIG"
    '''
    :stability: experimental
    '''
    SYSLOG = "SYSLOG"
    '''
    :stability: experimental
    '''
    WAKE_ALARM = "WAKE_ALARM"
    '''
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CapacityProviderStrategy",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_provider": "capacityProvider",
        "base": "base",
        "weight": "weight",
    },
)
class CapacityProviderStrategy:
    def __init__(
        self,
        *,
        capacity_provider: builtins.str,
        base: typing.Optional[jsii.Number] = None,
        weight: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) A Capacity Provider strategy to use for the service.

        NOTE: defaultCapacityProviderStrategy on cluster not currently supported.

        :param capacity_provider: (experimental) The name of the capacity provider.
        :param base: (experimental) The base value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default value of 0 is used. Default: - none
        :param weight: (experimental) The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The weight value is taken into consideration after the base value, if defined, is satisfied. Default: - 0

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            capacity_provider_strategy = ecs.CapacityProviderStrategy(
                capacity_provider="capacityProvider",
            
                # the properties below are optional
                base=123,
                weight=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "capacity_provider": capacity_provider,
        }
        if base is not None:
            self._values["base"] = base
        if weight is not None:
            self._values["weight"] = weight

    @builtins.property
    def capacity_provider(self) -> builtins.str:
        '''(experimental) The name of the capacity provider.

        :stability: experimental
        '''
        result = self._values.get("capacity_provider")
        assert result is not None, "Required property 'capacity_provider' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def base(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The base value designates how many tasks, at a minimum, to run on the specified capacity provider.

        Only one
        capacity provider in a capacity provider strategy can have a base defined. If no value is specified, the default
        value of 0 is used.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("base")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def weight(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The weight value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

        The weight value is taken into consideration after the base value, if defined, is satisfied.

        :default: - 0

        :stability: experimental
        '''
        result = self._values.get("weight")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CapacityProviderStrategy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnCapacityProvider(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnCapacityProvider",
):
    '''A CloudFormation ``AWS::ECS::CapacityProvider``.

    The ``AWS::ECS::CapacityProvider`` resource creates an Amazon Elastic Container Service (Amazon ECS) capacity provider. Capacity providers are associated with an Amazon ECS cluster and are used in capacity provider strategies to facilitate cluster auto scaling.

    Only capacity providers using an Auto Scaling group can be created. Amazon ECS tasks on AWS Fargate use the ``FARGATE`` and ``FARGATE_SPOT`` capacity providers which are already created and available to all accounts in Regions supported by AWS Fargate .

    :cloudformationResource: AWS::ECS::CapacityProvider
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_capacity_provider = ecs.CfnCapacityProvider(self, "MyCfnCapacityProvider",
            auto_scaling_group_provider=ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty(
                auto_scaling_group_arn="autoScalingGroupArn",
        
                # the properties below are optional
                managed_scaling=ecs.CfnCapacityProvider.ManagedScalingProperty(
                    instance_warmup_period=123,
                    maximum_scaling_step_size=123,
                    minimum_scaling_step_size=123,
                    status="status",
                    target_capacity=123
                ),
                managed_termination_protection="managedTerminationProtection"
            ),
        
            # the properties below are optional
            name="name",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        auto_scaling_group_provider: typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef],
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::CapacityProvider``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param auto_scaling_group_provider: The Auto Scaling group settings for the capacity provider.
        :param name: The name of the capacity provider. If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.
        :param tags: The metadata that you apply to the capacity provider to help you categorize and organize it. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        '''
        props = CfnCapacityProviderProps(
            auto_scaling_group_provider=auto_scaling_group_provider,
            name=name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''The metadata that you apply to the capacity provider to help you categorize and organize it.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoScalingGroupProvider")
    def auto_scaling_group_provider(
        self,
    ) -> typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef]:
        '''The Auto Scaling group settings for the capacity provider.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider
        '''
        return typing.cast(typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef], jsii.get(self, "autoScalingGroupProvider"))

    @auto_scaling_group_provider.setter
    def auto_scaling_group_provider(
        self,
        value: typing.Union["CfnCapacityProvider.AutoScalingGroupProviderProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "autoScalingGroupProvider", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="name")
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the capacity provider.

        If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-name
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "name"))

    @name.setter
    def name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "name", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "auto_scaling_group_arn": "autoScalingGroupArn",
            "managed_scaling": "managedScaling",
            "managed_termination_protection": "managedTerminationProtection",
        },
    )
    class AutoScalingGroupProviderProperty:
        def __init__(
            self,
            *,
            auto_scaling_group_arn: builtins.str,
            managed_scaling: typing.Optional[typing.Union["CfnCapacityProvider.ManagedScalingProperty", _IResolvable_a771d0ef]] = None,
            managed_termination_protection: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``AutoScalingGroupProvider`` property specifies the details of the Auto Scaling group for the capacity provider.

            :param auto_scaling_group_arn: The Amazon Resource Name (ARN) or short name that identifies the Auto Scaling group.
            :param managed_scaling: The managed scaling settings for the Auto Scaling group capacity provider.
            :param managed_termination_protection: The managed termination protection setting to use for the Auto Scaling group capacity provider. This determines whether the Auto Scaling group has managed termination protection. The default is disabled. .. epigraph:: When using managed termination protection, managed scaling must also be used otherwise managed termination protection doesn't work. When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and each instance in the Auto Scaling group must have instance protection from scale-in actions enabled as well. For more information, see `Instance Protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection>`_ in the *AWS Auto Scaling User Guide* . When managed termination protection is disabled, your Amazon EC2 instances aren't protected from termination when the Auto Scaling group scales in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                auto_scaling_group_provider_property = ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty(
                    auto_scaling_group_arn="autoScalingGroupArn",
                
                    # the properties below are optional
                    managed_scaling=ecs.CfnCapacityProvider.ManagedScalingProperty(
                        instance_warmup_period=123,
                        maximum_scaling_step_size=123,
                        minimum_scaling_step_size=123,
                        status="status",
                        target_capacity=123
                    ),
                    managed_termination_protection="managedTerminationProtection"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "auto_scaling_group_arn": auto_scaling_group_arn,
            }
            if managed_scaling is not None:
                self._values["managed_scaling"] = managed_scaling
            if managed_termination_protection is not None:
                self._values["managed_termination_protection"] = managed_termination_protection

        @builtins.property
        def auto_scaling_group_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) or short name that identifies the Auto Scaling group.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-autoscalinggrouparn
            '''
            result = self._values.get("auto_scaling_group_arn")
            assert result is not None, "Required property 'auto_scaling_group_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def managed_scaling(
            self,
        ) -> typing.Optional[typing.Union["CfnCapacityProvider.ManagedScalingProperty", _IResolvable_a771d0ef]]:
            '''The managed scaling settings for the Auto Scaling group capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-managedscaling
            '''
            result = self._values.get("managed_scaling")
            return typing.cast(typing.Optional[typing.Union["CfnCapacityProvider.ManagedScalingProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def managed_termination_protection(self) -> typing.Optional[builtins.str]:
            '''The managed termination protection setting to use for the Auto Scaling group capacity provider.

            This determines whether the Auto Scaling group has managed termination protection. The default is disabled.
            .. epigraph::

               When using managed termination protection, managed scaling must also be used otherwise managed termination protection doesn't work.

            When managed termination protection is enabled, Amazon ECS prevents the Amazon EC2 instances in an Auto Scaling group that contain tasks from being terminated during a scale-in action. The Auto Scaling group and each instance in the Auto Scaling group must have instance protection from scale-in actions enabled as well. For more information, see `Instance Protection <https://docs.aws.amazon.com/autoscaling/ec2/userguide/as-instance-termination.html#instance-protection>`_ in the *AWS Auto Scaling User Guide* .

            When managed termination protection is disabled, your Amazon EC2 instances aren't protected from termination when the Auto Scaling group scales in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-autoscalinggroupprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider-managedterminationprotection
            '''
            result = self._values.get("managed_termination_protection")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AutoScalingGroupProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCapacityProvider.ManagedScalingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_warmup_period": "instanceWarmupPeriod",
            "maximum_scaling_step_size": "maximumScalingStepSize",
            "minimum_scaling_step_size": "minimumScalingStepSize",
            "status": "status",
            "target_capacity": "targetCapacity",
        },
    )
    class ManagedScalingProperty:
        def __init__(
            self,
            *,
            instance_warmup_period: typing.Optional[jsii.Number] = None,
            maximum_scaling_step_size: typing.Optional[jsii.Number] = None,
            minimum_scaling_step_size: typing.Optional[jsii.Number] = None,
            status: typing.Optional[builtins.str] = None,
            target_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``ManagedScaling`` property specifies the settings for the Auto Scaling group capacity provider.

            When managed scaling is enabled, Amazon ECS manages the scale-in and scale-out actions of the Auto Scaling group. Amazon ECS manages a target tracking scaling policy using an Amazon ECS-managed CloudWatch metric with the specified ``targetCapacity`` value as the target value for the metric. For more information, see `Using Managed Scaling <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/asg-capacity-providers.html#asg-capacity-providers-managed-scaling>`_ in the *Amazon Elastic Container Service Developer Guide* .

            If managed scaling is disabled, the user must manage the scaling of the Auto Scaling group.

            :param instance_warmup_period: The period of time, in seconds, after a newly launched Amazon EC2 instance can contribute to CloudWatch metrics for Auto Scaling group. If this parameter is omitted, the default value of ``300`` seconds is used.
            :param maximum_scaling_step_size: The maximum number of container instances that Amazon ECS scales in or scales out at one time. If this parameter is omitted, the default value of ``10000`` is used.
            :param minimum_scaling_step_size: The minimum number of container instances that Amazon ECS scales in or scales out at one time. If this parameter is omitted, the default value of ``1`` is used.
            :param status: Determines whether to use managed scaling for the capacity provider.
            :param target_capacity: The target capacity value for the capacity provider. The specified value must be greater than ``0`` and less than or equal to ``100`` . A value of ``100`` results in the Amazon EC2 instances in your Auto Scaling group being completely used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                managed_scaling_property = ecs.CfnCapacityProvider.ManagedScalingProperty(
                    instance_warmup_period=123,
                    maximum_scaling_step_size=123,
                    minimum_scaling_step_size=123,
                    status="status",
                    target_capacity=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if instance_warmup_period is not None:
                self._values["instance_warmup_period"] = instance_warmup_period
            if maximum_scaling_step_size is not None:
                self._values["maximum_scaling_step_size"] = maximum_scaling_step_size
            if minimum_scaling_step_size is not None:
                self._values["minimum_scaling_step_size"] = minimum_scaling_step_size
            if status is not None:
                self._values["status"] = status
            if target_capacity is not None:
                self._values["target_capacity"] = target_capacity

        @builtins.property
        def instance_warmup_period(self) -> typing.Optional[jsii.Number]:
            '''The period of time, in seconds, after a newly launched Amazon EC2 instance can contribute to CloudWatch metrics for Auto Scaling group.

            If this parameter is omitted, the default value of ``300`` seconds is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-instancewarmupperiod
            '''
            result = self._values.get("instance_warmup_period")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def maximum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
            '''The maximum number of container instances that Amazon ECS scales in or scales out at one time.

            If this parameter is omitted, the default value of ``10000`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-maximumscalingstepsize
            '''
            result = self._values.get("maximum_scaling_step_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_scaling_step_size(self) -> typing.Optional[jsii.Number]:
            '''The minimum number of container instances that Amazon ECS scales in or scales out at one time.

            If this parameter is omitted, the default value of ``1`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-minimumscalingstepsize
            '''
            result = self._values.get("minimum_scaling_step_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def status(self) -> typing.Optional[builtins.str]:
            '''Determines whether to use managed scaling for the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-status
            '''
            result = self._values.get("status")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_capacity(self) -> typing.Optional[jsii.Number]:
            '''The target capacity value for the capacity provider.

            The specified value must be greater than ``0`` and less than or equal to ``100`` . A value of ``100`` results in the Amazon EC2 instances in your Auto Scaling group being completely used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-capacityprovider-managedscaling.html#cfn-ecs-capacityprovider-managedscaling-targetcapacity
            '''
            result = self._values.get("target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ManagedScalingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnCapacityProviderProps",
    jsii_struct_bases=[],
    name_mapping={
        "auto_scaling_group_provider": "autoScalingGroupProvider",
        "name": "name",
        "tags": "tags",
    },
)
class CfnCapacityProviderProps:
    def __init__(
        self,
        *,
        auto_scaling_group_provider: typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, _IResolvable_a771d0ef],
        name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCapacityProvider``.

        :param auto_scaling_group_provider: The Auto Scaling group settings for the capacity provider.
        :param name: The name of the capacity provider. If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.
        :param tags: The metadata that you apply to the capacity provider to help you categorize and organize it. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_capacity_provider_props = ecs.CfnCapacityProviderProps(
                auto_scaling_group_provider=ecs.CfnCapacityProvider.AutoScalingGroupProviderProperty(
                    auto_scaling_group_arn="autoScalingGroupArn",
            
                    # the properties below are optional
                    managed_scaling=ecs.CfnCapacityProvider.ManagedScalingProperty(
                        instance_warmup_period=123,
                        maximum_scaling_step_size=123,
                        minimum_scaling_step_size=123,
                        status="status",
                        target_capacity=123
                    ),
                    managed_termination_protection="managedTerminationProtection"
                ),
            
                # the properties below are optional
                name="name",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "auto_scaling_group_provider": auto_scaling_group_provider,
        }
        if name is not None:
            self._values["name"] = name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def auto_scaling_group_provider(
        self,
    ) -> typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, _IResolvable_a771d0ef]:
        '''The Auto Scaling group settings for the capacity provider.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-autoscalinggroupprovider
        '''
        result = self._values.get("auto_scaling_group_provider")
        assert result is not None, "Required property 'auto_scaling_group_provider' is missing"
        return typing.cast(typing.Union[CfnCapacityProvider.AutoScalingGroupProviderProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''The name of the capacity provider.

        If a name is specified, it cannot start with ``aws`` , ``ecs`` , or ``fargate`` . If no name is specified, a default name in the ``CFNStackName-CFNResourceName-RandomString`` format is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-name
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''The metadata that you apply to the capacity provider to help you categorize and organize it.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-capacityprovider.html#cfn-ecs-capacityprovider-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityProviderProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnCluster(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnCluster",
):
    '''A CloudFormation ``AWS::ECS::Cluster``.

    The ``AWS::ECS::Cluster`` resource creates an Amazon Elastic Container Service (Amazon ECS) cluster.

    :cloudformationResource: AWS::ECS::Cluster
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_cluster = ecs.CfnCluster(self, "MyCfnCluster",
            capacity_providers=["capacityProviders"],
            cluster_name="clusterName",
            cluster_settings=[ecs.CfnCluster.ClusterSettingsProperty(
                name="name",
                value="value"
            )],
            configuration=ecs.CfnCluster.ClusterConfigurationProperty(
                execute_command_configuration=ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                    kms_key_id="kmsKeyId",
                    log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                        cloud_watch_encryption_enabled=False,
                        cloud_watch_log_group_name="cloudWatchLogGroupName",
                        s3_bucket_name="s3BucketName",
                        s3_encryption_enabled=False,
                        s3_key_prefix="s3KeyPrefix"
                    ),
                    logging="logging"
                )
            ),
            default_capacity_provider_strategy=[ecs.CfnCluster.CapacityProviderStrategyItemProperty(
                base=123,
                capacity_provider="capacityProvider",
                weight=123
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        cluster_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]] = None,
        configuration: typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]] = None,
        default_capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::Cluster``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_providers: The short name of one or more capacity providers to associate with the cluster. A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy. If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created and not already associated with another cluster. To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.
        :param cluster_name: A user-generated string that you use to identify your cluster. If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.
        :param cluster_settings: The setting to use when creating a cluster. This parameter is used to enable CloudWatch Container Insights for a cluster. If this value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .
        :param configuration: The execute command configuration for the cluster.
        :param default_capacity_provider_strategy: The default capacity provider strategy for the cluster. When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.
        :param tags: The metadata that you apply to the cluster to help you categorize and organize them. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        '''
        props = CfnClusterProps(
            capacity_providers=capacity_providers,
            cluster_name=cluster_name,
            cluster_settings=cluster_settings,
            configuration=configuration,
            default_capacity_provider_strategy=default_capacity_provider_strategy,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The Amazon Resource Name (ARN) of the Amazon ECS cluster, such as ``arn:aws:ecs:us-east-2:123456789012:cluster/MyECSCluster`` .

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''The metadata that you apply to the cluster to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviders")
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The short name of one or more capacity providers to associate with the cluster.

        A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy.

        If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created and not already associated with another cluster.

        To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-capacityproviders
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "capacityProviders"))

    @capacity_providers.setter
    def capacity_providers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "capacityProviders", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterName")
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''A user-generated string that you use to identify your cluster.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "clusterName"))

    @cluster_name.setter
    def cluster_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "clusterName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clusterSettings")
    def cluster_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]]:
        '''The setting to use when creating a cluster.

        This parameter is used to enable CloudWatch Container Insights for a cluster. If this value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustersettings
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "clusterSettings"))

    @cluster_settings.setter
    def cluster_settings(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.ClusterSettingsProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "clusterSettings", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="configuration")
    def configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]]:
        '''The execute command configuration for the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-configuration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "configuration"))

    @configuration.setter
    def configuration(
        self,
        value: typing.Optional[typing.Union["CfnCluster.ClusterConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "configuration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultCapacityProviderStrategy")
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]]:
        '''The default capacity provider strategy for the cluster.

        When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-defaultcapacityproviderstrategy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "defaultCapacityProviderStrategy"))

    @default_capacity_provider_strategy.setter
    def default_capacity_provider_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnCluster.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "defaultCapacityProviderStrategy", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.CapacityProviderStrategyItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base": "base",
            "capacity_provider": "capacityProvider",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyItemProperty:
        def __init__(
            self,
            *,
            base: typing.Optional[jsii.Number] = None,
            capacity_provider: typing.Optional[builtins.str] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``CapacityProviderStrategyItem`` property specifies the details of the default capacity provider strategy for the cluster.

            When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

            :param base: The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.
            :param capacity_provider: The short name of the capacity provider.
            :param weight: The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied. If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail. An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                capacity_provider_strategy_item_property = ecs.CfnCluster.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if base is not None:
                self._values["base"] = base
            if capacity_provider is not None:
                self._values["capacity_provider"] = capacity_provider
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider.

            Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def capacity_provider(self) -> typing.Optional[builtins.str]:
            '''The short name of the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

            The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied.

            If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail.

            An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-capacityproviderstrategyitem.html#cfn-ecs-cluster-capacityproviderstrategyitem-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ClusterConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"execute_command_configuration": "executeCommandConfiguration"},
    )
    class ClusterConfigurationProperty:
        def __init__(
            self,
            *,
            execute_command_configuration: typing.Optional[typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The execute command configuration for the cluster.

            :param execute_command_configuration: The details of the execute command configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clusterconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                cluster_configuration_property = ecs.CfnCluster.ClusterConfigurationProperty(
                    execute_command_configuration=ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                        kms_key_id="kmsKeyId",
                        log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                            cloud_watch_encryption_enabled=False,
                            cloud_watch_log_group_name="cloudWatchLogGroupName",
                            s3_bucket_name="s3BucketName",
                            s3_encryption_enabled=False,
                            s3_key_prefix="s3KeyPrefix"
                        ),
                        logging="logging"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if execute_command_configuration is not None:
                self._values["execute_command_configuration"] = execute_command_configuration

        @builtins.property
        def execute_command_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", _IResolvable_a771d0ef]]:
            '''The details of the execute command configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clusterconfiguration.html#cfn-ecs-cluster-clusterconfiguration-executecommandconfiguration
            '''
            result = self._values.get("execute_command_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnCluster.ExecuteCommandConfigurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ClusterSettingsProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class ClusterSettingsProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The settings to use when creating a cluster.

            This parameter is used to turn on CloudWatch Container Insights for a cluster.

            :param name: The name of the cluster setting. The only supported value is ``containerInsights`` .
            :param value: The value to set for the cluster setting. The supported values are ``enabled`` and ``disabled`` . If ``enabled`` is specified, CloudWatch Container Insights will be enabled for the cluster, otherwise it will be disabled unless the ``containerInsights`` account setting is enabled. If a cluster value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                cluster_settings_property = ecs.CfnCluster.ClusterSettingsProperty(
                    name="name",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the cluster setting.

            The only supported value is ``containerInsights`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value to set for the cluster setting.

            The supported values are ``enabled`` and ``disabled`` . If ``enabled`` is specified, CloudWatch Container Insights will be enabled for the cluster, otherwise it will be disabled unless the ``containerInsights`` account setting is enabled. If a cluster value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-clustersettings.html#cfn-ecs-cluster-clustersettings-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClusterSettingsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ExecuteCommandConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "kms_key_id": "kmsKeyId",
            "log_configuration": "logConfiguration",
            "logging": "logging",
        },
    )
    class ExecuteCommandConfigurationProperty:
        def __init__(
            self,
            *,
            kms_key_id: typing.Optional[builtins.str] = None,
            log_configuration: typing.Optional[typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", _IResolvable_a771d0ef]] = None,
            logging: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The details of the execute command configuration.

            :param kms_key_id: Specify an AWS Key Management Service key ID to encrypt the data between the local client and the container.
            :param log_configuration: The log configuration for the results of the execute command actions. The logs can be sent to CloudWatch Logs or an Amazon S3 bucket. When ``logging=OVERRIDE`` is specified, a ``logConfiguration`` must be provided.
            :param logging: The log setting to use for redirecting logs for your execute command results. The following log settings are available. - ``NONE`` : The execute command session is not logged. - ``DEFAULT`` : The ``awslogs`` configuration in the task definition is used. If no logging parameter is specified, it defaults to this value. If no ``awslogs`` log driver is configured in the task definition, the output won't be logged. - ``OVERRIDE`` : Specify the logging details as a part of ``logConfiguration`` . If the ``OVERRIDE`` logging option is specified, the ``logConfiguration`` is required.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                execute_command_configuration_property = ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                    kms_key_id="kmsKeyId",
                    log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                        cloud_watch_encryption_enabled=False,
                        cloud_watch_log_group_name="cloudWatchLogGroupName",
                        s3_bucket_name="s3BucketName",
                        s3_encryption_enabled=False,
                        s3_key_prefix="s3KeyPrefix"
                    ),
                    logging="logging"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if logging is not None:
                self._values["logging"] = logging

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''Specify an AWS Key Management Service key ID to encrypt the data between the local client and the container.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", _IResolvable_a771d0ef]]:
            '''The log configuration for the results of the execute command actions.

            The logs can be sent to CloudWatch Logs or an Amazon S3 bucket. When ``logging=OVERRIDE`` is specified, a ``logConfiguration`` must be provided.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnCluster.ExecuteCommandLogConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def logging(self) -> typing.Optional[builtins.str]:
            '''The log setting to use for redirecting logs for your execute command results. The following log settings are available.

            - ``NONE`` : The execute command session is not logged.
            - ``DEFAULT`` : The ``awslogs`` configuration in the task definition is used. If no logging parameter is specified, it defaults to this value. If no ``awslogs`` log driver is configured in the task definition, the output won't be logged.
            - ``OVERRIDE`` : Specify the logging details as a part of ``logConfiguration`` . If the ``OVERRIDE`` logging option is specified, the ``logConfiguration`` is required.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandconfiguration.html#cfn-ecs-cluster-executecommandconfiguration-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExecuteCommandConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnCluster.ExecuteCommandLogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cloud_watch_encryption_enabled": "cloudWatchEncryptionEnabled",
            "cloud_watch_log_group_name": "cloudWatchLogGroupName",
            "s3_bucket_name": "s3BucketName",
            "s3_encryption_enabled": "s3EncryptionEnabled",
            "s3_key_prefix": "s3KeyPrefix",
        },
    )
    class ExecuteCommandLogConfigurationProperty:
        def __init__(
            self,
            *,
            cloud_watch_encryption_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            cloud_watch_log_group_name: typing.Optional[builtins.str] = None,
            s3_bucket_name: typing.Optional[builtins.str] = None,
            s3_encryption_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            s3_key_prefix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The log configuration for the results of the execute command actions.

            The logs can be sent to CloudWatch Logs or an Amazon S3 bucket.

            :param cloud_watch_encryption_enabled: Determines whether to use encryption on the CloudWatch logs. If not specified, encryption will be disabled.
            :param cloud_watch_log_group_name: The name of the CloudWatch log group to send logs to. .. epigraph:: The CloudWatch log group must already be created.
            :param s3_bucket_name: The name of the S3 bucket to send logs to. .. epigraph:: The S3 bucket must already be created.
            :param s3_encryption_enabled: Determines whether to use encryption on the S3 logs. If not specified, encryption is not used.
            :param s3_key_prefix: An optional folder in the S3 bucket to place logs in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                execute_command_log_configuration_property = ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                    cloud_watch_encryption_enabled=False,
                    cloud_watch_log_group_name="cloudWatchLogGroupName",
                    s3_bucket_name="s3BucketName",
                    s3_encryption_enabled=False,
                    s3_key_prefix="s3KeyPrefix"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cloud_watch_encryption_enabled is not None:
                self._values["cloud_watch_encryption_enabled"] = cloud_watch_encryption_enabled
            if cloud_watch_log_group_name is not None:
                self._values["cloud_watch_log_group_name"] = cloud_watch_log_group_name
            if s3_bucket_name is not None:
                self._values["s3_bucket_name"] = s3_bucket_name
            if s3_encryption_enabled is not None:
                self._values["s3_encryption_enabled"] = s3_encryption_enabled
            if s3_key_prefix is not None:
                self._values["s3_key_prefix"] = s3_key_prefix

        @builtins.property
        def cloud_watch_encryption_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Determines whether to use encryption on the CloudWatch logs.

            If not specified, encryption will be disabled.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-cloudwatchencryptionenabled
            '''
            result = self._values.get("cloud_watch_encryption_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def cloud_watch_log_group_name(self) -> typing.Optional[builtins.str]:
            '''The name of the CloudWatch log group to send logs to.

            .. epigraph::

               The CloudWatch log group must already be created.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-cloudwatchloggroupname
            '''
            result = self._values.get("cloud_watch_log_group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_bucket_name(self) -> typing.Optional[builtins.str]:
            '''The name of the S3 bucket to send logs to.

            .. epigraph::

               The S3 bucket must already be created.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3bucketname
            '''
            result = self._values.get("s3_bucket_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def s3_encryption_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Determines whether to use encryption on the S3 logs.

            If not specified, encryption is not used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3encryptionenabled
            '''
            result = self._values.get("s3_encryption_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def s3_key_prefix(self) -> typing.Optional[builtins.str]:
            '''An optional folder in the S3 bucket to place logs in.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-cluster-executecommandlogconfiguration.html#cfn-ecs-cluster-executecommandlogconfiguration-s3keyprefix
            '''
            result = self._values.get("s3_key_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExecuteCommandLogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(_IInspectable_82c04a63)
class CfnClusterCapacityProviderAssociations(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnClusterCapacityProviderAssociations",
):
    '''A CloudFormation ``AWS::ECS::ClusterCapacityProviderAssociations``.

    The ``AWS::ECS::ClusterCapacityProviderAssociations`` resource associates one or more capacity providers and a default capacity provider strategy with a cluster.

    :cloudformationResource: AWS::ECS::ClusterCapacityProviderAssociations
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_cluster_capacity_provider_associations = ecs.CfnClusterCapacityProviderAssociations(self, "MyCfnClusterCapacityProviderAssociations",
            capacity_providers=["capacityProviders"],
            cluster="cluster",
            default_capacity_provider_strategy=[ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty(
                capacity_provider="capacityProvider",
        
                # the properties below are optional
                base=123,
                weight=123
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        capacity_providers: typing.Sequence[builtins.str],
        cluster: builtins.str,
        default_capacity_provider_strategy: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]],
    ) -> None:
        '''Create a new ``AWS::ECS::ClusterCapacityProviderAssociations``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_providers: The capacity providers to associate with the cluster.
        :param cluster: The cluster the capacity provider association is the target of.
        :param default_capacity_provider_strategy: The default capacity provider strategy to associate with the cluster.
        '''
        props = CfnClusterCapacityProviderAssociationsProps(
            capacity_providers=capacity_providers,
            cluster=cluster,
            default_capacity_provider_strategy=default_capacity_provider_strategy,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviders")
    def capacity_providers(self) -> typing.List[builtins.str]:
        '''The capacity providers to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-capacityproviders
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "capacityProviders"))

    @capacity_providers.setter
    def capacity_providers(self, value: typing.List[builtins.str]) -> None:
        jsii.set(self, "capacityProviders", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''The cluster the capacity provider association is the target of.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultCapacityProviderStrategy")
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]]:
        '''The default capacity provider strategy to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-defaultcapacityproviderstrategy
        '''
        return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]], jsii.get(self, "defaultCapacityProviderStrategy"))

    @default_capacity_provider_strategy.setter
    def default_capacity_provider_strategy(
        self,
        value: typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty", _IResolvable_a771d0ef]]],
    ) -> None:
        jsii.set(self, "defaultCapacityProviderStrategy", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_provider": "capacityProvider",
            "base": "base",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyProperty:
        def __init__(
            self,
            *,
            capacity_provider: builtins.str,
            base: typing.Optional[jsii.Number] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``CapacityProviderStrategy`` property specifies the details of the default capacity provider strategy for the cluster.

            When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

            :param capacity_provider: The short name of the capacity provider.
            :param base: The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.
            :param weight: The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied. If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` will not be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail. An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that is run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                capacity_provider_strategy_property = ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty(
                    capacity_provider="capacityProvider",
                
                    # the properties below are optional
                    base=123,
                    weight=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "capacity_provider": capacity_provider,
            }
            if base is not None:
                self._values["base"] = base
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def capacity_provider(self) -> builtins.str:
            '''The short name of the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            assert result is not None, "Required property 'capacity_provider' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider.

            Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

            The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied.

            If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` will not be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail.

            An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that is run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-clustercapacityproviderassociations-capacityproviderstrategy.html#cfn-ecs-clustercapacityproviderassociations-capacityproviderstrategy-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnClusterCapacityProviderAssociationsProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_providers": "capacityProviders",
        "cluster": "cluster",
        "default_capacity_provider_strategy": "defaultCapacityProviderStrategy",
    },
)
class CfnClusterCapacityProviderAssociationsProps:
    def __init__(
        self,
        *,
        capacity_providers: typing.Sequence[builtins.str],
        cluster: builtins.str,
        default_capacity_provider_strategy: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, _IResolvable_a771d0ef]]],
    ) -> None:
        '''Properties for defining a ``CfnClusterCapacityProviderAssociations``.

        :param capacity_providers: The capacity providers to associate with the cluster.
        :param cluster: The cluster the capacity provider association is the target of.
        :param default_capacity_provider_strategy: The default capacity provider strategy to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_cluster_capacity_provider_associations_props = ecs.CfnClusterCapacityProviderAssociationsProps(
                capacity_providers=["capacityProviders"],
                cluster="cluster",
                default_capacity_provider_strategy=[ecs.CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty(
                    capacity_provider="capacityProvider",
            
                    # the properties below are optional
                    base=123,
                    weight=123
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "capacity_providers": capacity_providers,
            "cluster": cluster,
            "default_capacity_provider_strategy": default_capacity_provider_strategy,
        }

    @builtins.property
    def capacity_providers(self) -> typing.List[builtins.str]:
        '''The capacity providers to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-capacityproviders
        '''
        result = self._values.get("capacity_providers")
        assert result is not None, "Required property 'capacity_providers' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def cluster(self) -> builtins.str:
        '''The cluster the capacity provider association is the target of.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, _IResolvable_a771d0ef]]]:
        '''The default capacity provider strategy to associate with the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-clustercapacityproviderassociations.html#cfn-ecs-clustercapacityproviderassociations-defaultcapacityproviderstrategy
        '''
        result = self._values.get("default_capacity_provider_strategy")
        assert result is not None, "Required property 'default_capacity_provider_strategy' is missing"
        return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnClusterCapacityProviderAssociations.CapacityProviderStrategyProperty, _IResolvable_a771d0ef]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterCapacityProviderAssociationsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_providers": "capacityProviders",
        "cluster_name": "clusterName",
        "cluster_settings": "clusterSettings",
        "configuration": "configuration",
        "default_capacity_provider_strategy": "defaultCapacityProviderStrategy",
        "tags": "tags",
    },
)
class CfnClusterProps:
    def __init__(
        self,
        *,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        cluster_settings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnCluster.ClusterSettingsProperty, _IResolvable_a771d0ef]]]] = None,
        configuration: typing.Optional[typing.Union[CfnCluster.ClusterConfigurationProperty, _IResolvable_a771d0ef]] = None,
        default_capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``CfnCluster``.

        :param capacity_providers: The short name of one or more capacity providers to associate with the cluster. A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy. If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created and not already associated with another cluster. To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.
        :param cluster_name: A user-generated string that you use to identify your cluster. If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.
        :param cluster_settings: The setting to use when creating a cluster. This parameter is used to enable CloudWatch Container Insights for a cluster. If this value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .
        :param configuration: The execute command configuration for the cluster.
        :param default_capacity_provider_strategy: The default capacity provider strategy for the cluster. When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.
        :param tags: The metadata that you apply to the cluster to help you categorize and organize them. Each tag consists of a key and an optional value. You define both. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_cluster_props = ecs.CfnClusterProps(
                capacity_providers=["capacityProviders"],
                cluster_name="clusterName",
                cluster_settings=[ecs.CfnCluster.ClusterSettingsProperty(
                    name="name",
                    value="value"
                )],
                configuration=ecs.CfnCluster.ClusterConfigurationProperty(
                    execute_command_configuration=ecs.CfnCluster.ExecuteCommandConfigurationProperty(
                        kms_key_id="kmsKeyId",
                        log_configuration=ecs.CfnCluster.ExecuteCommandLogConfigurationProperty(
                            cloud_watch_encryption_enabled=False,
                            cloud_watch_log_group_name="cloudWatchLogGroupName",
                            s3_bucket_name="s3BucketName",
                            s3_encryption_enabled=False,
                            s3_key_prefix="s3KeyPrefix"
                        ),
                        logging="logging"
                    )
                ),
                default_capacity_provider_strategy=[ecs.CfnCluster.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if capacity_providers is not None:
            self._values["capacity_providers"] = capacity_providers
        if cluster_name is not None:
            self._values["cluster_name"] = cluster_name
        if cluster_settings is not None:
            self._values["cluster_settings"] = cluster_settings
        if configuration is not None:
            self._values["configuration"] = configuration
        if default_capacity_provider_strategy is not None:
            self._values["default_capacity_provider_strategy"] = default_capacity_provider_strategy
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The short name of one or more capacity providers to associate with the cluster.

        A capacity provider must be associated with a cluster before it can be included as part of the default capacity provider strategy of the cluster or used in a capacity provider strategy.

        If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created and not already associated with another cluster.

        To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-capacityproviders
        '''
        result = self._values.get("capacity_providers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''A user-generated string that you use to identify your cluster.

        If you don't specify a name, AWS CloudFormation generates a unique physical ID for the name.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustername
        '''
        result = self._values.get("cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cluster_settings(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.ClusterSettingsProperty, _IResolvable_a771d0ef]]]]:
        '''The setting to use when creating a cluster.

        This parameter is used to enable CloudWatch Container Insights for a cluster. If this value is specified, it will override the ``containerInsights`` value set with `PutAccountSetting <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSetting.html>`_ or `PutAccountSettingDefault <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_PutAccountSettingDefault.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-clustersettings
        '''
        result = self._values.get("cluster_settings")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.ClusterSettingsProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnCluster.ClusterConfigurationProperty, _IResolvable_a771d0ef]]:
        '''The execute command configuration for the cluster.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-configuration
        '''
        result = self._values.get("configuration")
        return typing.cast(typing.Optional[typing.Union[CfnCluster.ClusterConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def default_capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]]:
        '''The default capacity provider strategy for the cluster.

        When services or tasks are run in the cluster with no launch type or capacity provider strategy specified, the default capacity provider strategy is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-defaultcapacityproviderstrategy
        '''
        result = self._values.get("default_capacity_provider_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnCluster.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''The metadata that you apply to the cluster to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-cluster.html#cfn-ecs-cluster-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnPrimaryTaskSet(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnPrimaryTaskSet",
):
    '''A CloudFormation ``AWS::ECS::PrimaryTaskSet``.

    Specifies which task set in a service is the primary task set. Any parameters that are updated on the primary task set in a service will transition to the service. This is used when a service uses the ``EXTERNAL`` deployment controller type. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

    :cloudformationResource: AWS::ECS::PrimaryTaskSet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_primary_task_set = ecs.CfnPrimaryTaskSet(self, "MyCfnPrimaryTaskSet",
            cluster="cluster",
            service="service",
            task_set_id="taskSetId"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_set_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::ECS::PrimaryTaskSet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.
        :param service: The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.
        :param task_set_id: The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.
        '''
        props = CfnPrimaryTaskSetProps(
            cluster=cluster, service=service, task_set_id=task_set_id
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="service")
    def service(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-service
        '''
        return typing.cast(builtins.str, jsii.get(self, "service"))

    @service.setter
    def service(self, value: builtins.str) -> None:
        jsii.set(self, "service", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskSetId")
    def task_set_id(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-tasksetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "taskSetId"))

    @task_set_id.setter
    def task_set_id(self, value: builtins.str) -> None:
        jsii.set(self, "taskSetId", value)


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnPrimaryTaskSetProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service": "service",
        "task_set_id": "taskSetId",
    },
)
class CfnPrimaryTaskSetProps:
    def __init__(
        self,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_set_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``CfnPrimaryTaskSet``.

        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.
        :param service: The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.
        :param task_set_id: The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_primary_task_set_props = ecs.CfnPrimaryTaskSetProps(
                cluster="cluster",
                service="service",
                task_set_id="taskSetId"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "service": service,
            "task_set_id": task_set_id,
        }

    @builtins.property
    def cluster(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the service that the task set exists in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-service
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def task_set_id(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the task set to set as the primary task set in the deployment.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-primarytaskset.html#cfn-ecs-primarytaskset-tasksetid
        '''
        result = self._values.get("task_set_id")
        assert result is not None, "Required property 'task_set_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPrimaryTaskSetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnService(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnService",
):
    '''A CloudFormation ``AWS::ECS::Service``.

    The ``AWS::ECS::Service`` resource creates an Amazon Elastic Container Service (Amazon ECS) service that runs and maintains the requested number of tasks and associated load balancers.

    :cloudformationResource: AWS::ECS::Service
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_service = ecs.CfnService(self, "MyCfnService",
            capacity_provider_strategy=[ecs.CfnService.CapacityProviderStrategyItemProperty(
                base=123,
                capacity_provider="capacityProvider",
                weight=123
            )],
            cluster="cluster",
            deployment_configuration=ecs.CfnService.DeploymentConfigurationProperty(
                deployment_circuit_breaker=ecs.CfnService.DeploymentCircuitBreakerProperty(
                    enable=False,
                    rollback=False
                ),
                maximum_percent=123,
                minimum_healthy_percent=123
            ),
            deployment_controller=ecs.CfnService.DeploymentControllerProperty(
                type="type"
            ),
            desired_count=123,
            enable_ecs_managed_tags=False,
            enable_execute_command=False,
            health_check_grace_period_seconds=123,
            launch_type="launchType",
            load_balancers=[ecs.CfnService.LoadBalancerProperty(
                container_port=123,
        
                # the properties below are optional
                container_name="containerName",
                load_balancer_name="loadBalancerName",
                target_group_arn="targetGroupArn"
            )],
            network_configuration=ecs.CfnService.NetworkConfigurationProperty(
                awsvpc_configuration=ecs.CfnService.AwsVpcConfigurationProperty(
                    subnets=["subnets"],
        
                    # the properties below are optional
                    assign_public_ip="assignPublicIp",
                    security_groups=["securityGroups"]
                )
            ),
            placement_constraints=[ecs.CfnService.PlacementConstraintProperty(
                type="type",
        
                # the properties below are optional
                expression="expression"
            )],
            placement_strategies=[ecs.CfnService.PlacementStrategyProperty(
                type="type",
        
                # the properties below are optional
                field="field"
            )],
            platform_version="platformVersion",
            propagate_tags="propagateTags",
            role="role",
            scheduling_strategy="schedulingStrategy",
            service_name="serviceName",
            service_registries=[ecs.CfnService.ServiceRegistryProperty(
                container_name="containerName",
                container_port=123,
                port=123,
                registry_arn="registryArn"
            )],
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            task_definition="taskDefinition"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]] = None,
        cluster: typing.Optional[builtins.str] = None,
        deployment_configuration: typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]] = None,
        deployment_controller: typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        enable_execute_command: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        health_check_grace_period_seconds: typing.Optional[jsii.Number] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]] = None,
        placement_strategies: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        propagate_tags: typing.Optional[builtins.str] = None,
        role: typing.Optional[builtins.str] = None,
        scheduling_strategy: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_definition: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::Service``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param capacity_provider_strategy: The capacity provider strategy to use for the service. A capacity provider strategy consists of one or more capacity providers along with the ``base`` and ``weight`` to assign to them. A capacity provider must be associated with the cluster to be used in a capacity provider strategy. The PutClusterCapacityProviders API is used to associate a capacity provider with a cluster. Only capacity providers with an ``ACTIVE`` or ``UPDATING`` status can be used. Review the `Capacity provider considerations <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-capacity-providers.html#capacity-providers-considerations>`_ in the *Amazon Elastic Container Service Developer Guide.* If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used. If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New capacity providers can be created with the CreateCapacityProvider API operation. To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used. The PutClusterCapacityProviders API operation is used to update the list of available capacity providers for a cluster after the cluster is created.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on. If you do not specify a cluster, the default cluster is assumed.
        :param deployment_configuration: Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.
        :param deployment_controller: The deployment controller to use for the service. If no deployment controller is specified, the default value of ``ECS`` is used.
        :param desired_count: The number of instantiations of the specified task definition to place and keep running on your cluster. For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required. For existing services, if a desired count is not specified, it is omitted from the operation.
        :param enable_ecs_managed_tags: Specifies whether to turn on Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param enable_execute_command: Determines whether the execute command functionality is enabled for the service. If ``true`` , the execute command functionality is enabled for all containers in tasks as part of the service.
        :param health_check_grace_period_seconds: The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used. If you do not use an Elastic Load Balancing, we recomend that you use the ``startPeriod`` in the task definition healtch check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ . If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.
        :param launch_type: The launch type on which to run your service. For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param load_balancers: A list of load balancer objects to associate with the service. If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param network_configuration: The network configuration for the service. This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param placement_constraints: An array of placement constraint objects to use for tasks in your service. You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.
        :param placement_strategies: The placement strategy objects to use for tasks in your service. You can specify a maximum of five strategy rules per service. For more information, see `Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param platform_version: The platform version that your tasks in the service are running on. A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param propagate_tags: Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. If no value is specified, the tags are not propagated. Tags can only be propagated to the tasks within the service during service creation. To add tags to a task after service creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action.
        :param role: The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf. This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter. .. epigraph:: If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .
        :param scheduling_strategy: The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ . There are two service scheduler strategies available: - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types. - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies. .. epigraph:: Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.
        :param service_name: The name of your service. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions.
        :param service_registries: The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ . .. epigraph:: Each service may be associated with one service registry. Multiple service registries for each service isn't supported.
        :param tags: The metadata that you apply to the service to help you categorize and organize them. Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_definition: The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service. The ``revision`` is required in order for the resource to stabilize. A task definition must be specified if the service is using either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers.
        '''
        props = CfnServiceProps(
            capacity_provider_strategy=capacity_provider_strategy,
            cluster=cluster,
            deployment_configuration=deployment_configuration,
            deployment_controller=deployment_controller,
            desired_count=desired_count,
            enable_ecs_managed_tags=enable_ecs_managed_tags,
            enable_execute_command=enable_execute_command,
            health_check_grace_period_seconds=health_check_grace_period_seconds,
            launch_type=launch_type,
            load_balancers=load_balancers,
            network_configuration=network_configuration,
            placement_constraints=placement_constraints,
            placement_strategies=placement_strategies,
            platform_version=platform_version,
            propagate_tags=propagate_tags,
            role=role,
            scheduling_strategy=scheduling_strategy,
            service_name=service_name,
            service_registries=service_registries,
            tags=tags,
            task_definition=task_definition,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrName")
    def attr_name(self) -> builtins.str:
        '''The name of the Amazon ECS service, such as ``sample-webapp`` .

        :cloudformationAttribute: Name
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrServiceArn")
    def attr_service_arn(self) -> builtins.str:
        '''Not currently supported in AWS CloudFormation .

        :cloudformationAttribute: ServiceArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrServiceArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''The metadata that you apply to the service to help you categorize and organize them.

        Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capacityProviderStrategy")
    def capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]]:
        '''The capacity provider strategy to use for the service.

        A capacity provider strategy consists of one or more capacity providers along with the ``base`` and ``weight`` to assign to them. A capacity provider must be associated with the cluster to be used in a capacity provider strategy. The PutClusterCapacityProviders API is used to associate a capacity provider with a cluster. Only capacity providers with an ``ACTIVE`` or ``UPDATING`` status can be used.

        Review the `Capacity provider considerations <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-capacity-providers.html#capacity-providers-considerations>`_ in the *Amazon Elastic Container Service Developer Guide.*

        If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used.

        If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New capacity providers can be created with the CreateCapacityProvider API operation.

        To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.

        The PutClusterCapacityProviders API operation is used to update the list of available capacity providers for a cluster after the cluster is created.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "capacityProviderStrategy"))

    @capacity_provider_strategy.setter
    def capacity_provider_strategy(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.CapacityProviderStrategyItemProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "capacityProviderStrategy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on.

        If you do not specify a cluster, the default cluster is assumed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-cluster
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deploymentConfiguration")
    def deployment_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]]:
        '''Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "deploymentConfiguration"))

    @deployment_configuration.setter
    def deployment_configuration(
        self,
        value: typing.Optional[typing.Union["CfnService.DeploymentConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "deploymentConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deploymentController")
    def deployment_controller(
        self,
    ) -> typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]]:
        '''The deployment controller to use for the service.

        If no deployment controller is specified, the default value of ``ECS`` is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentcontroller
        '''
        return typing.cast(typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]], jsii.get(self, "deploymentController"))

    @deployment_controller.setter
    def deployment_controller(
        self,
        value: typing.Optional[typing.Union["CfnService.DeploymentControllerProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "deploymentController", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="desiredCount")
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''The number of instantiations of the specified task definition to place and keep running on your cluster.

        For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required.

        For existing services, if a desired count is not specified, it is omitted from the operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-desiredcount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "desiredCount"))

    @desired_count.setter
    def desired_count(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "desiredCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enableEcsManagedTags")
    def enable_ecs_managed_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether to turn on Amazon ECS managed tags for the tasks within the service.

        For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableecsmanagedtags
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "enableEcsManagedTags"))

    @enable_ecs_managed_tags.setter
    def enable_ecs_managed_tags(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "enableEcsManagedTags", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enableExecuteCommand")
    def enable_execute_command(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Determines whether the execute command functionality is enabled for the service.

        If ``true`` , the execute command functionality is enabled for all containers in tasks as part of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableexecutecommand
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], jsii.get(self, "enableExecuteCommand"))

    @enable_execute_command.setter
    def enable_execute_command(
        self,
        value: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "enableExecuteCommand", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="healthCheckGracePeriodSeconds")
    def health_check_grace_period_seconds(self) -> typing.Optional[jsii.Number]:
        '''The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used.

        If you do not use an Elastic Load Balancing, we recomend that you use the ``startPeriod`` in the task definition healtch check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ .

        If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "healthCheckGracePeriodSeconds"))

    @health_check_grace_period_seconds.setter
    def health_check_grace_period_seconds(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "healthCheckGracePeriodSeconds", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchType")
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''The launch type on which to run your service.

        For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-launchtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchType"))

    @launch_type.setter
    def launch_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "launchType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="loadBalancers")
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]]:
        '''A list of load balancer objects to associate with the service.

        If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-loadbalancers
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "loadBalancers"))

    @load_balancers.setter
    def load_balancers(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.LoadBalancerProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "loadBalancers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkConfiguration")
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]]:
        '''The network configuration for the service.

        This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-networkconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "networkConfiguration"))

    @network_configuration.setter
    def network_configuration(
        self,
        value: typing.Optional[typing.Union["CfnService.NetworkConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "networkConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementConstraints")
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]]:
        '''An array of placement constraint objects to use for tasks in your service.

        You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementconstraints
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "placementConstraints"))

    @placement_constraints.setter
    def placement_constraints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementConstraintProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "placementConstraints", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementStrategies")
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]]:
        '''The placement strategy objects to use for tasks in your service.

        You can specify a maximum of five strategy rules per service. For more information, see `Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementstrategies
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "placementStrategies"))

    @placement_strategies.setter
    def placement_strategies(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.PlacementStrategyProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "placementStrategies", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platformVersion")
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''The platform version that your tasks in the service are running on.

        A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-platformversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "platformVersion"))

    @platform_version.setter
    def platform_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "platformVersion", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="propagateTags")
    def propagate_tags(self) -> typing.Optional[builtins.str]:
        '''Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        If no value is specified, the tags are not propagated. Tags can only be propagated to the tasks within the service during service creation. To add tags to a task after service creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-propagatetags
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "propagateTags"))

    @propagate_tags.setter
    def propagate_tags(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "propagateTags", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="role")
    def role(self) -> typing.Optional[builtins.str]:
        '''The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf.

        This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter.
        .. epigraph::

           If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-role
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "role"))

    @role.setter
    def role(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "role", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="schedulingStrategy")
    def scheduling_strategy(self) -> typing.Optional[builtins.str]:
        '''The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ .

        There are two service scheduler strategies available:

        - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types.
        - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies.

        .. epigraph::

           Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "schedulingStrategy"))

    @scheduling_strategy.setter
    def scheduling_strategy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "schedulingStrategy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceName")
    def service_name(self) -> typing.Optional[builtins.str]:
        '''The name of your service.

        Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-servicename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "serviceName"))

    @service_name.setter
    def service_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "serviceName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceRegistries")
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]]:
        '''The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        .. epigraph::

           Each service may be associated with one service registry. Multiple service registries for each service isn't supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceregistries
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "serviceRegistries"))

    @service_registries.setter
    def service_registries(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnService.ServiceRegistryProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "serviceRegistries", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> typing.Optional[builtins.str]:
        '''The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service.

        The ``revision`` is required in order for the resource to stabilize.

        A task definition must be specified if the service is using either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-taskdefinition
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "taskDefinition"))

    @task_definition.setter
    def task_definition(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "taskDefinition", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.AwsVpcConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subnets": "subnets",
            "assign_public_ip": "assignPublicIp",
            "security_groups": "securityGroups",
        },
    )
    class AwsVpcConfigurationProperty:
        def __init__(
            self,
            *,
            subnets: typing.Sequence[builtins.str],
            assign_public_ip: typing.Optional[builtins.str] = None,
            security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object representing the networking details for a task or service.

            :param subnets: The IDs of the subnets associated with the task or service. There's a limit of 16 subnets that can be specified per ``AwsVpcConfiguration`` . .. epigraph:: All specified subnets must be from the same VPC.
            :param assign_public_ip: Whether the task's elastic network interface receives a public IP address. The default value is ``DISABLED`` .
            :param security_groups: The IDs of the security groups associated with the task or service. If you don't specify a security group, the default security group for the VPC is used. There's a limit of 5 security groups that can be specified per ``AwsVpcConfiguration`` . .. epigraph:: All specified security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                aws_vpc_configuration_property = ecs.CfnService.AwsVpcConfigurationProperty(
                    subnets=["subnets"],
                
                    # the properties below are optional
                    assign_public_ip="assignPublicIp",
                    security_groups=["securityGroups"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "subnets": subnets,
            }
            if assign_public_ip is not None:
                self._values["assign_public_ip"] = assign_public_ip
            if security_groups is not None:
                self._values["security_groups"] = security_groups

        @builtins.property
        def subnets(self) -> typing.List[builtins.str]:
            '''The IDs of the subnets associated with the task or service.

            There's a limit of 16 subnets that can be specified per ``AwsVpcConfiguration`` .
            .. epigraph::

               All specified subnets must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-subnets
            '''
            result = self._values.get("subnets")
            assert result is not None, "Required property 'subnets' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def assign_public_ip(self) -> typing.Optional[builtins.str]:
            '''Whether the task's elastic network interface receives a public IP address.

            The default value is ``DISABLED`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-assignpublicip
            '''
            result = self._values.get("assign_public_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The IDs of the security groups associated with the task or service.

            If you don't specify a security group, the default security group for the VPC is used. There's a limit of 5 security groups that can be specified per ``AwsVpcConfiguration`` .
            .. epigraph::

               All specified security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-awsvpcconfiguration.html#cfn-ecs-service-awsvpcconfiguration-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsVpcConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.CapacityProviderStrategyItemProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base": "base",
            "capacity_provider": "capacityProvider",
            "weight": "weight",
        },
    )
    class CapacityProviderStrategyItemProperty:
        def __init__(
            self,
            *,
            base: typing.Optional[jsii.Number] = None,
            capacity_provider: typing.Optional[builtins.str] = None,
            weight: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The details of a capacity provider strategy.

            A capacity provider strategy can be set when using the ``RunTask`` or ``CreateService`` APIs or as the default capacity provider strategy for a cluster with the ``CreateCluster`` API.

            Only capacity providers that are already associated with a cluster and have an ``ACTIVE`` or ``UPDATING`` status can be used in a capacity provider strategy. The ``PutClusterCapacityProviders`` API is used to associate a capacity provider with a cluster.

            If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New Auto Scaling group capacity providers can be created with the ``CreateCapacityProvider`` API operation.

            To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used in a capacity provider strategy.

            :param base: The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider. Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.
            :param capacity_provider: The short name of the capacity provider.
            :param weight: The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider. The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied. If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail. An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                capacity_provider_strategy_item_property = ecs.CfnService.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if base is not None:
                self._values["base"] = base
            if capacity_provider is not None:
                self._values["capacity_provider"] = capacity_provider
            if weight is not None:
                self._values["weight"] = weight

        @builtins.property
        def base(self) -> typing.Optional[jsii.Number]:
            '''The *base* value designates how many tasks, at a minimum, to run on the specified capacity provider.

            Only one capacity provider in a capacity provider strategy can have a *base* defined. If no value is specified, the default value of ``0`` is used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-base
            '''
            result = self._values.get("base")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def capacity_provider(self) -> typing.Optional[builtins.str]:
            '''The short name of the capacity provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-capacityprovider
            '''
            result = self._values.get("capacity_provider")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weight(self) -> typing.Optional[jsii.Number]:
            '''The *weight* value designates the relative percentage of the total number of tasks launched that should use the specified capacity provider.

            The ``weight`` value is taken into consideration after the ``base`` value, if defined, is satisfied.

            If no ``weight`` value is specified, the default value of ``0`` is used. When multiple capacity providers are specified within a capacity provider strategy, at least one of the capacity providers must have a weight value greater than zero and any capacity providers with a weight of ``0`` can't be used to place tasks. If you specify multiple capacity providers in a strategy that all have a weight of ``0`` , any ``RunTask`` or ``CreateService`` actions using the capacity provider strategy will fail.

            An example scenario for using weights is defining a strategy that contains two capacity providers and both have a weight of ``1`` , then when the ``base`` is satisfied, the tasks will be split evenly across the two capacity providers. Using that same logic, if you specify a weight of ``1`` for *capacityProviderA* and a weight of ``4`` for *capacityProviderB* , then for every one task that's run using *capacityProviderA* , four tasks would use *capacityProviderB* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-capacityproviderstrategyitem.html#cfn-ecs-service-capacityproviderstrategyitem-weight
            '''
            result = self._values.get("weight")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityProviderStrategyItemProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.DeploymentCircuitBreakerProperty",
        jsii_struct_bases=[],
        name_mapping={"enable": "enable", "rollback": "rollback"},
    )
    class DeploymentCircuitBreakerProperty:
        def __init__(
            self,
            *,
            enable: typing.Union[builtins.bool, _IResolvable_a771d0ef],
            rollback: typing.Union[builtins.bool, _IResolvable_a771d0ef],
        ) -> None:
            '''.. epigraph::

   The deployment circuit breaker can only be used for services using the rolling update ( ``ECS`` ) deployment type.

            The ``DeploymentCircuitBreaker`` property determines whether a service deployment will fail if the service can't reach a steady state. If deployment circuit breaker is enabled, a service deployment will transition to a failed state and stop launching new tasks. If rollback is enabled, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :param enable: Determines whether to use the deployment circuit breaker logic for the service.
            :param rollback: Determines whether to configure Amazon ECS to roll back the service if a service deployment fails. If rollback is enabled, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                deployment_circuit_breaker_property = ecs.CfnService.DeploymentCircuitBreakerProperty(
                    enable=False,
                    rollback=False
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "enable": enable,
                "rollback": rollback,
            }

        @builtins.property
        def enable(self) -> typing.Union[builtins.bool, _IResolvable_a771d0ef]:
            '''Determines whether to use the deployment circuit breaker logic for the service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html#cfn-ecs-service-deploymentcircuitbreaker-enable
            '''
            result = self._values.get("enable")
            assert result is not None, "Required property 'enable' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_a771d0ef], result)

        @builtins.property
        def rollback(self) -> typing.Union[builtins.bool, _IResolvable_a771d0ef]:
            '''Determines whether to configure Amazon ECS to roll back the service if a service deployment fails.

            If rollback is enabled, when a service deployment fails, the service is rolled back to the last deployment that completed successfully.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcircuitbreaker.html#cfn-ecs-service-deploymentcircuitbreaker-rollback
            '''
            result = self._values.get("rollback")
            assert result is not None, "Required property 'rollback' is missing"
            return typing.cast(typing.Union[builtins.bool, _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentCircuitBreakerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.DeploymentConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "deployment_circuit_breaker": "deploymentCircuitBreaker",
            "maximum_percent": "maximumPercent",
            "minimum_healthy_percent": "minimumHealthyPercent",
        },
    )
    class DeploymentConfigurationProperty:
        def __init__(
            self,
            *,
            deployment_circuit_breaker: typing.Optional[typing.Union["CfnService.DeploymentCircuitBreakerProperty", _IResolvable_a771d0ef]] = None,
            maximum_percent: typing.Optional[jsii.Number] = None,
            minimum_healthy_percent: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``DeploymentConfiguration`` property specifies optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.

            :param deployment_circuit_breaker: .. epigraph:: The deployment circuit breaker can only be used for services using the rolling update ( ``ECS`` ) deployment type that are not behind a Classic Load Balancer. The *deployment circuit breaker* determines whether a service deployment will fail if the service can't reach a steady state. If enabled, a service deployment will transition to a failed state and stop launching new tasks. You can also enable Amazon ECS to roll back your service to the last completed deployment after a failure. For more information, see `Rolling update <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param maximum_percent: If a service is using the rolling update ( ``ECS`` ) deployment type, the *maximum percent* parameter represents an upper limit on the number of tasks in a service that are allowed in the ``RUNNING`` or ``PENDING`` state during a deployment, as a percentage of the desired number of tasks (rounded down to the nearest integer), and while any container instances are in the ``DRAINING`` state if the service contains tasks using the EC2 launch type. This parameter enables you to define the deployment batch size. For example, if your service has a desired number of four tasks and a maximum percent value of 200%, the scheduler may start four new tasks before stopping the four older tasks (provided that the cluster resources required to do this are available). The default value for maximum percent is 200%. If a service is using the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and tasks that use the EC2 launch type, the *maximum percent* value is set to the default value and is used to define the upper limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If the tasks in the service use the Fargate launch type, the maximum percent value is not used, although it is returned when describing your service.
            :param minimum_healthy_percent: If a service is using the rolling update ( ``ECS`` ) deployment type, the *minimum healthy percent* represents a lower limit on the number of tasks in a service that must remain in the ``RUNNING`` state during a deployment, as a percentage of the desired number of tasks (rounded up to the nearest integer), and while any container instances are in the ``DRAINING`` state if the service contains tasks using the EC2 launch type. This parameter enables you to deploy without using additional cluster capacity. For example, if your service has a desired number of four tasks and a minimum healthy percent of 50%, the scheduler may stop two existing tasks to free up cluster capacity before starting two new tasks. Tasks for services that *do not* use a load balancer are considered healthy if they're in the ``RUNNING`` state; tasks for services that *do* use a load balancer are considered healthy if they're in the ``RUNNING`` state and they're reported as healthy by the load balancer. The default value for minimum healthy percent is 100%. If a service is using the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and tasks that use the EC2 launch type, the *minimum healthy percent* value is set to the default value and is used to define the lower limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If the tasks in the service use the Fargate launch type, the minimum healthy percent value is not used, although it is returned when describing your service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                deployment_configuration_property = ecs.CfnService.DeploymentConfigurationProperty(
                    deployment_circuit_breaker=ecs.CfnService.DeploymentCircuitBreakerProperty(
                        enable=False,
                        rollback=False
                    ),
                    maximum_percent=123,
                    minimum_healthy_percent=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if deployment_circuit_breaker is not None:
                self._values["deployment_circuit_breaker"] = deployment_circuit_breaker
            if maximum_percent is not None:
                self._values["maximum_percent"] = maximum_percent
            if minimum_healthy_percent is not None:
                self._values["minimum_healthy_percent"] = minimum_healthy_percent

        @builtins.property
        def deployment_circuit_breaker(
            self,
        ) -> typing.Optional[typing.Union["CfnService.DeploymentCircuitBreakerProperty", _IResolvable_a771d0ef]]:
            '''.. epigraph::

   The deployment circuit breaker can only be used for services using the rolling update ( ``ECS`` ) deployment type that are not behind a Classic Load Balancer.

            The *deployment circuit breaker* determines whether a service deployment will fail if the service can't reach a steady state. If enabled, a service deployment will transition to a failed state and stop launching new tasks. You can also enable Amazon ECS to roll back your service to the last completed deployment after a failure. For more information, see `Rolling update <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-deploymentcircuitbreaker
            '''
            result = self._values.get("deployment_circuit_breaker")
            return typing.cast(typing.Optional[typing.Union["CfnService.DeploymentCircuitBreakerProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def maximum_percent(self) -> typing.Optional[jsii.Number]:
            '''If a service is using the rolling update ( ``ECS`` ) deployment type, the *maximum percent* parameter represents an upper limit on the number of tasks in a service that are allowed in the ``RUNNING`` or ``PENDING`` state during a deployment, as a percentage of the desired number of tasks (rounded down to the nearest integer), and while any container instances are in the ``DRAINING`` state if the service contains tasks using the EC2 launch type.

            This parameter enables you to define the deployment batch size. For example, if your service has a desired number of four tasks and a maximum percent value of 200%, the scheduler may start four new tasks before stopping the four older tasks (provided that the cluster resources required to do this are available). The default value for maximum percent is 200%.

            If a service is using the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and tasks that use the EC2 launch type, the *maximum percent* value is set to the default value and is used to define the upper limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If the tasks in the service use the Fargate launch type, the maximum percent value is not used, although it is returned when describing your service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-maximumpercent
            '''
            result = self._values.get("maximum_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def minimum_healthy_percent(self) -> typing.Optional[jsii.Number]:
            '''If a service is using the rolling update ( ``ECS`` ) deployment type, the *minimum healthy percent* represents a lower limit on the number of tasks in a service that must remain in the ``RUNNING`` state during a deployment, as a percentage of the desired number of tasks (rounded up to the nearest integer), and while any container instances are in the ``DRAINING`` state if the service contains tasks using the EC2 launch type.

            This parameter enables you to deploy without using additional cluster capacity. For example, if your service has a desired number of four tasks and a minimum healthy percent of 50%, the scheduler may stop two existing tasks to free up cluster capacity before starting two new tasks. Tasks for services that *do not* use a load balancer are considered healthy if they're in the ``RUNNING`` state; tasks for services that *do* use a load balancer are considered healthy if they're in the ``RUNNING`` state and they're reported as healthy by the load balancer. The default value for minimum healthy percent is 100%.

            If a service is using the blue/green ( ``CODE_DEPLOY`` ) or ``EXTERNAL`` deployment types and tasks that use the EC2 launch type, the *minimum healthy percent* value is set to the default value and is used to define the lower limit on the number of the tasks in the service that remain in the ``RUNNING`` state while the container instances are in the ``DRAINING`` state. If the tasks in the service use the Fargate launch type, the minimum healthy percent value is not used, although it is returned when describing your service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentconfiguration.html#cfn-ecs-service-deploymentconfiguration-minimumhealthypercent
            '''
            result = self._values.get("minimum_healthy_percent")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.DeploymentControllerProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class DeploymentControllerProperty:
        def __init__(self, *, type: typing.Optional[builtins.str] = None) -> None:
            '''The deployment controller to use for the service.

            For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The deployment controller type to use. There are three deployment controller types available:. - **ECS** - The rolling update ( ``ECS`` ) deployment type involves replacing the current running version of the container with the latest version. The number of containers Amazon ECS adds or removes from the service during a rolling update is controlled by adjusting the minimum and maximum number of healthy tasks allowed during a service deployment, as specified in the `DeploymentConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeploymentConfiguration.html>`_ . - **CODE_DEPLOY** - The blue/green ( ``CODE_DEPLOY`` ) deployment type uses the blue/green deployment model powered by AWS CodeDeploy , which allows you to verify a new deployment of a service before sending production traffic to it. - **EXTERNAL** - The external ( ``EXTERNAL`` ) deployment type enables you to use any third-party deployment controller for full control over the deployment process for an Amazon ECS service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcontroller.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                deployment_controller_property = ecs.CfnService.DeploymentControllerProperty(
                    type="type"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The deployment controller type to use. There are three deployment controller types available:.

            - **ECS** - The rolling update ( ``ECS`` ) deployment type involves replacing the current running version of the container with the latest version. The number of containers Amazon ECS adds or removes from the service during a rolling update is controlled by adjusting the minimum and maximum number of healthy tasks allowed during a service deployment, as specified in the `DeploymentConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DeploymentConfiguration.html>`_ .
            - **CODE_DEPLOY** - The blue/green ( ``CODE_DEPLOY`` ) deployment type uses the blue/green deployment model powered by AWS CodeDeploy , which allows you to verify a new deployment of a service before sending production traffic to it.
            - **EXTERNAL** - The external ( ``EXTERNAL`` ) deployment type enables you to use any third-party deployment controller for full control over the deployment process for an Amazon ECS service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-deploymentcontroller.html#cfn-ecs-service-deploymentcontroller-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeploymentControllerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.LoadBalancerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_port": "containerPort",
            "container_name": "containerName",
            "load_balancer_name": "loadBalancerName",
            "target_group_arn": "targetGroupArn",
        },
    )
    class LoadBalancerProperty:
        def __init__(
            self,
            *,
            container_port: jsii.Number,
            container_name: typing.Optional[builtins.str] = None,
            load_balancer_name: typing.Optional[builtins.str] = None,
            target_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``LoadBalancer`` property specifies details on a load balancer that is used with a service.

            If the service is using the ``CODE_DEPLOY`` deployment controller, the service is required to use either an Application Load Balancer or Network Load Balancer. When you are creating an AWS CodeDeploy deployment group, you specify two target groups (referred to as a ``targetGroupPair`` ). Each target group binds to a separate task set in the deployment. The load balancer can also have up to two listeners, a required listener for production traffic and an optional listener that allows you to test new revisions of the service before routing production traffic to it.

            Services with tasks that use the ``awsvpc`` network mode (for example, those with the Fargate launch type) only support Application Load Balancers and Network Load Balancers. Classic Load Balancers are not supported. Also, when you create any target groups for these services, you must choose ``ip`` as the target type, not ``instance`` . Tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance.

            :param container_port: The port on the container to associate with the load balancer. This port must correspond to a ``containerPort`` in the task definition the tasks in the service are using. For tasks that use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the ``hostPort`` of the port mapping.
            :param container_name: The name of the container (as it appears in a container definition) to associate with the load balancer.
            :param load_balancer_name: The name of the load balancer to associate with the Amazon ECS service or task set. A load balancer name is only specified when using a Classic Load Balancer. If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should be omitted.
            :param target_group_arn: The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a service or task set. A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer. If you're using a Classic Load Balancer, omit the target group ARN. For services using the ``ECS`` deployment controller, you can specify one or multiple target groups. For more information, see `Registering Multiple Target Groups with a Service <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For services using the ``CODE_DEPLOY`` deployment controller, you're required to define two target groups for the load balancer. For more information, see `Blue/Green Deployment with CodeDeploy <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If your service's task definition uses the ``awsvpc`` network mode, you must choose ``ip`` as the target type, not ``instance`` . Do this when creating your target groups because tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                load_balancer_property = ecs.CfnService.LoadBalancerProperty(
                    container_port=123,
                
                    # the properties below are optional
                    container_name="containerName",
                    load_balancer_name="loadBalancerName",
                    target_group_arn="targetGroupArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "container_port": container_port,
            }
            if container_name is not None:
                self._values["container_name"] = container_name
            if load_balancer_name is not None:
                self._values["load_balancer_name"] = load_balancer_name
            if target_group_arn is not None:
                self._values["target_group_arn"] = target_group_arn

        @builtins.property
        def container_port(self) -> jsii.Number:
            '''The port on the container to associate with the load balancer.

            This port must correspond to a ``containerPort`` in the task definition the tasks in the service are using. For tasks that use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the ``hostPort`` of the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-containerport
            '''
            result = self._values.get("container_port")
            assert result is not None, "Required property 'container_port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The name of the container (as it appears in a container definition) to associate with the load balancer.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def load_balancer_name(self) -> typing.Optional[builtins.str]:
            '''The name of the load balancer to associate with the Amazon ECS service or task set.

            A load balancer name is only specified when using a Classic Load Balancer. If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should be omitted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-loadbalancername
            '''
            result = self._values.get("load_balancer_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_group_arn(self) -> typing.Optional[builtins.str]:
            '''The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a service or task set.

            A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer. If you're using a Classic Load Balancer, omit the target group ARN.

            For services using the ``ECS`` deployment controller, you can specify one or multiple target groups. For more information, see `Registering Multiple Target Groups with a Service <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For services using the ``CODE_DEPLOY`` deployment controller, you're required to define two target groups for the load balancer. For more information, see `Blue/Green Deployment with CodeDeploy <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If your service's task definition uses the ``awsvpc`` network mode, you must choose ``ip`` as the target type, not ``instance`` . Do this when creating your target groups because tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-loadbalancer.html#cfn-ecs-service-loadbalancer-targetgrouparn
            '''
            result = self._values.get("target_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoadBalancerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.NetworkConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"awsvpc_configuration": "awsvpcConfiguration"},
    )
    class NetworkConfigurationProperty:
        def __init__(
            self,
            *,
            awsvpc_configuration: typing.Optional[typing.Union["CfnService.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The ``NetworkConfiguration`` property specifies an object representing the network configuration for a task or service.

            :param awsvpc_configuration: The VPC subnets and security groups that are associated with a task. .. epigraph:: All specified subnets and security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-networkconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                network_configuration_property = ecs.CfnService.NetworkConfigurationProperty(
                    awsvpc_configuration=ecs.CfnService.AwsVpcConfigurationProperty(
                        subnets=["subnets"],
                
                        # the properties below are optional
                        assign_public_ip="assignPublicIp",
                        security_groups=["securityGroups"]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if awsvpc_configuration is not None:
                self._values["awsvpc_configuration"] = awsvpc_configuration

        @builtins.property
        def awsvpc_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnService.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]]:
            '''The VPC subnets and security groups that are associated with a task.

            .. epigraph::

               All specified subnets and security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-networkconfiguration.html#cfn-ecs-service-networkconfiguration-awsvpcconfiguration
            '''
            result = self._values.get("awsvpc_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnService.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.PlacementConstraintProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "expression": "expression"},
    )
    class PlacementConstraintProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``PlacementConstraint`` property specifies an object representing a constraint on task placement in the task definition.

            For more information, see `Task Placement Constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The type of constraint. Use ``distinctInstance`` to ensure that each task in a particular group is running on a different container instance. Use ``memberOf`` to restrict the selection to a group of valid candidates.
            :param expression: A cluster query language expression to apply to the constraint. The expression can have a maximum length of 2000 characters. You can't specify an expression if the constraint type is ``distinctInstance`` . For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                placement_constraint_property = ecs.CfnService.PlacementConstraintProperty(
                    type="type",
                
                    # the properties below are optional
                    expression="expression"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of constraint.

            Use ``distinctInstance`` to ensure that each task in a particular group is running on a different container instance. Use ``memberOf`` to restrict the selection to a group of valid candidates.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html#cfn-ecs-service-placementconstraint-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''A cluster query language expression to apply to the constraint.

            The expression can have a maximum length of 2000 characters. You can't specify an expression if the constraint type is ``distinctInstance`` . For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementconstraint.html#cfn-ecs-service-placementconstraint-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementConstraintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.PlacementStrategyProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "field": "field"},
    )
    class PlacementStrategyProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            field: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``PlacementStrategy`` property specifies the task placement strategy for a task or service.

            For more information, see `Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The type of placement strategy. The ``random`` placement strategy randomly places tasks on available candidates. The ``spread`` placement strategy spreads placement across available candidates evenly based on the ``field`` parameter. The ``binpack`` strategy places tasks on available candidates that have the least available amount of the resource that's specified with the ``field`` parameter. For example, if you binpack on memory, a task is placed on the instance with the least amount of remaining memory but still enough to run the task.
            :param field: The field to apply the placement strategy against. For the ``spread`` placement strategy, valid values are ``instanceId`` (or ``host`` , which has the same effect), or any platform or custom attribute that's applied to a container instance, such as ``attribute:ecs.availability-zone`` . For the ``binpack`` placement strategy, valid values are ``cpu`` and ``memory`` . For the ``random`` placement strategy, this field is not used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                placement_strategy_property = ecs.CfnService.PlacementStrategyProperty(
                    type="type",
                
                    # the properties below are optional
                    field="field"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if field is not None:
                self._values["field"] = field

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of placement strategy.

            The ``random`` placement strategy randomly places tasks on available candidates. The ``spread`` placement strategy spreads placement across available candidates evenly based on the ``field`` parameter. The ``binpack`` strategy places tasks on available candidates that have the least available amount of the resource that's specified with the ``field`` parameter. For example, if you binpack on memory, a task is placed on the instance with the least amount of remaining memory but still enough to run the task.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html#cfn-ecs-service-placementstrategy-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def field(self) -> typing.Optional[builtins.str]:
            '''The field to apply the placement strategy against.

            For the ``spread`` placement strategy, valid values are ``instanceId`` (or ``host`` , which has the same effect), or any platform or custom attribute that's applied to a container instance, such as ``attribute:ecs.availability-zone`` . For the ``binpack`` placement strategy, valid values are ``cpu`` and ``memory`` . For the ``random`` placement strategy, this field is not used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-placementstrategy.html#cfn-ecs-service-placementstrategy-field
            '''
            result = self._values.get("field")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementStrategyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnService.ServiceRegistryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "port": "port",
            "registry_arn": "registryArn",
        },
    )
    class ServiceRegistryProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            port: typing.Optional[jsii.Number] = None,
            registry_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ServiceRegistry`` property specifies details of the service registry.

            For more information, see `Service Discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param container_name: The container name value to be used for your service discovery service. It's already specified in the task definition. If the task definition that your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition that your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.
            :param container_port: The port value to be used for your service discovery service. It's already specified in the task definition. If the task definition your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.
            :param port: The port value used if your service discovery service specified an SRV record. This field might be used if both the ``awsvpc`` network mode and SRV records are used.
            :param registry_arn: The Amazon Resource Name (ARN) of the service registry. The currently supported service registry is AWS Cloud Map . For more information, see `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                service_registry_property = ecs.CfnService.ServiceRegistryProperty(
                    container_name="containerName",
                    container_port=123,
                    port=123,
                    registry_arn="registryArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if port is not None:
                self._values["port"] = port
            if registry_arn is not None:
                self._values["registry_arn"] = registry_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The container name value to be used for your service discovery service.

            It's already specified in the task definition. If the task definition that your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition that your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''The port value to be used for your service discovery service.

            It's already specified in the task definition. If the task definition your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port value used if your service discovery service specified an SRV record.

            This field might be used if both the ``awsvpc`` network mode and SRV records are used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def registry_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the service registry.

            The currently supported service registry is AWS Cloud Map . For more information, see `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-service-serviceregistry.html#cfn-ecs-service-serviceregistry-registryarn
            '''
            result = self._values.get("registry_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceRegistryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnServiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity_provider_strategy": "capacityProviderStrategy",
        "cluster": "cluster",
        "deployment_configuration": "deploymentConfiguration",
        "deployment_controller": "deploymentController",
        "desired_count": "desiredCount",
        "enable_ecs_managed_tags": "enableEcsManagedTags",
        "enable_execute_command": "enableExecuteCommand",
        "health_check_grace_period_seconds": "healthCheckGracePeriodSeconds",
        "launch_type": "launchType",
        "load_balancers": "loadBalancers",
        "network_configuration": "networkConfiguration",
        "placement_constraints": "placementConstraints",
        "placement_strategies": "placementStrategies",
        "platform_version": "platformVersion",
        "propagate_tags": "propagateTags",
        "role": "role",
        "scheduling_strategy": "schedulingStrategy",
        "service_name": "serviceName",
        "service_registries": "serviceRegistries",
        "tags": "tags",
        "task_definition": "taskDefinition",
    },
)
class CfnServiceProps:
    def __init__(
        self,
        *,
        capacity_provider_strategy: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]] = None,
        cluster: typing.Optional[builtins.str] = None,
        deployment_configuration: typing.Optional[typing.Union[CfnService.DeploymentConfigurationProperty, _IResolvable_a771d0ef]] = None,
        deployment_controller: typing.Optional[typing.Union[CfnService.DeploymentControllerProperty, _IResolvable_a771d0ef]] = None,
        desired_count: typing.Optional[jsii.Number] = None,
        enable_ecs_managed_tags: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        enable_execute_command: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
        health_check_grace_period_seconds: typing.Optional[jsii.Number] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.LoadBalancerProperty, _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union[CfnService.NetworkConfigurationProperty, _IResolvable_a771d0ef]] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.PlacementConstraintProperty, _IResolvable_a771d0ef]]]] = None,
        placement_strategies: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.PlacementStrategyProperty, _IResolvable_a771d0ef]]]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        propagate_tags: typing.Optional[builtins.str] = None,
        role: typing.Optional[builtins.str] = None,
        scheduling_strategy: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnService.ServiceRegistryProperty, _IResolvable_a771d0ef]]]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_definition: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnService``.

        :param capacity_provider_strategy: The capacity provider strategy to use for the service. A capacity provider strategy consists of one or more capacity providers along with the ``base`` and ``weight`` to assign to them. A capacity provider must be associated with the cluster to be used in a capacity provider strategy. The PutClusterCapacityProviders API is used to associate a capacity provider with a cluster. Only capacity providers with an ``ACTIVE`` or ``UPDATING`` status can be used. Review the `Capacity provider considerations <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-capacity-providers.html#capacity-providers-considerations>`_ in the *Amazon Elastic Container Service Developer Guide.* If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used. If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New capacity providers can be created with the CreateCapacityProvider API operation. To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used. The PutClusterCapacityProviders API operation is used to update the list of available capacity providers for a cluster after the cluster is created.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on. If you do not specify a cluster, the default cluster is assumed.
        :param deployment_configuration: Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.
        :param deployment_controller: The deployment controller to use for the service. If no deployment controller is specified, the default value of ``ECS`` is used.
        :param desired_count: The number of instantiations of the specified task definition to place and keep running on your cluster. For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required. For existing services, if a desired count is not specified, it is omitted from the operation.
        :param enable_ecs_managed_tags: Specifies whether to turn on Amazon ECS managed tags for the tasks within the service. For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param enable_execute_command: Determines whether the execute command functionality is enabled for the service. If ``true`` , the execute command functionality is enabled for all containers in tasks as part of the service.
        :param health_check_grace_period_seconds: The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started. This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used. If you do not use an Elastic Load Balancing, we recomend that you use the ``startPeriod`` in the task definition healtch check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ . If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.
        :param launch_type: The launch type on which to run your service. For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param load_balancers: A list of load balancer objects to associate with the service. If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param network_configuration: The network configuration for the service. This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param placement_constraints: An array of placement constraint objects to use for tasks in your service. You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.
        :param placement_strategies: The placement strategy objects to use for tasks in your service. You can specify a maximum of five strategy rules per service. For more information, see `Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param platform_version: The platform version that your tasks in the service are running on. A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param propagate_tags: Specifies whether to propagate the tags from the task definition or the service to the tasks in the service. If no value is specified, the tags are not propagated. Tags can only be propagated to the tasks within the service during service creation. To add tags to a task after service creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action.
        :param role: The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf. This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter. .. epigraph:: If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .
        :param scheduling_strategy: The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ . There are two service scheduler strategies available: - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types. - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies. .. epigraph:: Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.
        :param service_name: The name of your service. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions.
        :param service_registries: The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ . .. epigraph:: Each service may be associated with one service registry. Multiple service registries for each service isn't supported.
        :param tags: The metadata that you apply to the service to help you categorize and organize them. Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_definition: The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service. The ``revision`` is required in order for the resource to stabilize. A task definition must be specified if the service is using either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_service_props = ecs.CfnServiceProps(
                capacity_provider_strategy=[ecs.CfnService.CapacityProviderStrategyItemProperty(
                    base=123,
                    capacity_provider="capacityProvider",
                    weight=123
                )],
                cluster="cluster",
                deployment_configuration=ecs.CfnService.DeploymentConfigurationProperty(
                    deployment_circuit_breaker=ecs.CfnService.DeploymentCircuitBreakerProperty(
                        enable=False,
                        rollback=False
                    ),
                    maximum_percent=123,
                    minimum_healthy_percent=123
                ),
                deployment_controller=ecs.CfnService.DeploymentControllerProperty(
                    type="type"
                ),
                desired_count=123,
                enable_ecs_managed_tags=False,
                enable_execute_command=False,
                health_check_grace_period_seconds=123,
                launch_type="launchType",
                load_balancers=[ecs.CfnService.LoadBalancerProperty(
                    container_port=123,
            
                    # the properties below are optional
                    container_name="containerName",
                    load_balancer_name="loadBalancerName",
                    target_group_arn="targetGroupArn"
                )],
                network_configuration=ecs.CfnService.NetworkConfigurationProperty(
                    awsvpc_configuration=ecs.CfnService.AwsVpcConfigurationProperty(
                        subnets=["subnets"],
            
                        # the properties below are optional
                        assign_public_ip="assignPublicIp",
                        security_groups=["securityGroups"]
                    )
                ),
                placement_constraints=[ecs.CfnService.PlacementConstraintProperty(
                    type="type",
            
                    # the properties below are optional
                    expression="expression"
                )],
                placement_strategies=[ecs.CfnService.PlacementStrategyProperty(
                    type="type",
            
                    # the properties below are optional
                    field="field"
                )],
                platform_version="platformVersion",
                propagate_tags="propagateTags",
                role="role",
                scheduling_strategy="schedulingStrategy",
                service_name="serviceName",
                service_registries=[ecs.CfnService.ServiceRegistryProperty(
                    container_name="containerName",
                    container_port=123,
                    port=123,
                    registry_arn="registryArn"
                )],
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                task_definition="taskDefinition"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if capacity_provider_strategy is not None:
            self._values["capacity_provider_strategy"] = capacity_provider_strategy
        if cluster is not None:
            self._values["cluster"] = cluster
        if deployment_configuration is not None:
            self._values["deployment_configuration"] = deployment_configuration
        if deployment_controller is not None:
            self._values["deployment_controller"] = deployment_controller
        if desired_count is not None:
            self._values["desired_count"] = desired_count
        if enable_ecs_managed_tags is not None:
            self._values["enable_ecs_managed_tags"] = enable_ecs_managed_tags
        if enable_execute_command is not None:
            self._values["enable_execute_command"] = enable_execute_command
        if health_check_grace_period_seconds is not None:
            self._values["health_check_grace_period_seconds"] = health_check_grace_period_seconds
        if launch_type is not None:
            self._values["launch_type"] = launch_type
        if load_balancers is not None:
            self._values["load_balancers"] = load_balancers
        if network_configuration is not None:
            self._values["network_configuration"] = network_configuration
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if placement_strategies is not None:
            self._values["placement_strategies"] = placement_strategies
        if platform_version is not None:
            self._values["platform_version"] = platform_version
        if propagate_tags is not None:
            self._values["propagate_tags"] = propagate_tags
        if role is not None:
            self._values["role"] = role
        if scheduling_strategy is not None:
            self._values["scheduling_strategy"] = scheduling_strategy
        if service_name is not None:
            self._values["service_name"] = service_name
        if service_registries is not None:
            self._values["service_registries"] = service_registries
        if tags is not None:
            self._values["tags"] = tags
        if task_definition is not None:
            self._values["task_definition"] = task_definition

    @builtins.property
    def capacity_provider_strategy(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]]:
        '''The capacity provider strategy to use for the service.

        A capacity provider strategy consists of one or more capacity providers along with the ``base`` and ``weight`` to assign to them. A capacity provider must be associated with the cluster to be used in a capacity provider strategy. The PutClusterCapacityProviders API is used to associate a capacity provider with a cluster. Only capacity providers with an ``ACTIVE`` or ``UPDATING`` status can be used.

        Review the `Capacity provider considerations <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-capacity-providers.html#capacity-providers-considerations>`_ in the *Amazon Elastic Container Service Developer Guide.*

        If a ``capacityProviderStrategy`` is specified, the ``launchType`` parameter must be omitted. If no ``capacityProviderStrategy`` or ``launchType`` is specified, the ``defaultCapacityProviderStrategy`` for the cluster is used.

        If specifying a capacity provider that uses an Auto Scaling group, the capacity provider must already be created. New capacity providers can be created with the CreateCapacityProvider API operation.

        To use an AWS Fargate capacity provider, specify either the ``FARGATE`` or ``FARGATE_SPOT`` capacity providers. The AWS Fargate capacity providers are available to all accounts and only need to be associated with a cluster to be used.

        The PutClusterCapacityProviders API operation is used to update the list of available capacity providers for a cluster after the cluster is created.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-capacityproviderstrategy
        '''
        result = self._values.get("capacity_provider_strategy")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.CapacityProviderStrategyItemProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def cluster(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that you run your service on.

        If you do not specify a cluster, the default cluster is assumed.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-cluster
        '''
        result = self._values.get("cluster")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def deployment_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnService.DeploymentConfigurationProperty, _IResolvable_a771d0ef]]:
        '''Optional deployment parameters that control how many tasks run during the deployment and the ordering of stopping and starting tasks.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentconfiguration
        '''
        result = self._values.get("deployment_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnService.DeploymentConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def deployment_controller(
        self,
    ) -> typing.Optional[typing.Union[CfnService.DeploymentControllerProperty, _IResolvable_a771d0ef]]:
        '''The deployment controller to use for the service.

        If no deployment controller is specified, the default value of ``ECS`` is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-deploymentcontroller
        '''
        result = self._values.get("deployment_controller")
        return typing.cast(typing.Optional[typing.Union[CfnService.DeploymentControllerProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def desired_count(self) -> typing.Optional[jsii.Number]:
        '''The number of instantiations of the specified task definition to place and keep running on your cluster.

        For new services, if a desired count is not specified, a default value of ``1`` is used. When using the ``DAEMON`` scheduling strategy, the desired count is not required.

        For existing services, if a desired count is not specified, it is omitted from the operation.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-desiredcount
        '''
        result = self._values.get("desired_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def enable_ecs_managed_tags(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Specifies whether to turn on Amazon ECS managed tags for the tasks within the service.

        For more information, see `Tagging Your Amazon ECS Resources <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableecsmanagedtags
        '''
        result = self._values.get("enable_ecs_managed_tags")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def enable_execute_command(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
        '''Determines whether the execute command functionality is enabled for the service.

        If ``true`` , the execute command functionality is enabled for all containers in tasks as part of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-enableexecutecommand
        '''
        result = self._values.get("enable_execute_command")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

    @builtins.property
    def health_check_grace_period_seconds(self) -> typing.Optional[jsii.Number]:
        '''The period of time, in seconds, that the Amazon ECS service scheduler ignores unhealthy Elastic Load Balancing target health checks after a task has first started.

        This is only used when your service is configured to use a load balancer. If your service has a load balancer defined and you don't specify a health check grace period value, the default value of ``0`` is used.

        If you do not use an Elastic Load Balancing, we recomend that you use the ``startPeriod`` in the task definition healtch check parameters. For more information, see `Health check <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_HealthCheck.html>`_ .

        If your service's tasks take a while to start and respond to Elastic Load Balancing health checks, you can specify a health check grace period of up to 2,147,483,647 seconds (about 69 years). During that time, the Amazon ECS service scheduler ignores health check status. This grace period can prevent the service scheduler from marking tasks as unhealthy and stopping them before they have time to come up.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-healthcheckgraceperiodseconds
        '''
        result = self._values.get("health_check_grace_period_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''The launch type on which to run your service.

        For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-launchtype
        '''
        result = self._values.get("launch_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.LoadBalancerProperty, _IResolvable_a771d0ef]]]]:
        '''A list of load balancer objects to associate with the service.

        If you specify the ``Role`` property, ``LoadBalancers`` must be specified as well. For information about the number of load balancers that you can specify per service, see `Service Load Balancing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-load-balancing.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-loadbalancers
        '''
        result = self._values.get("load_balancers")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.LoadBalancerProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnService.NetworkConfigurationProperty, _IResolvable_a771d0ef]]:
        '''The network configuration for the service.

        This parameter is required for task definitions that use the ``awsvpc`` network mode to receive their own elastic network interface, and it is not supported for other network modes. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-networkconfiguration
        '''
        result = self._values.get("network_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnService.NetworkConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementConstraintProperty, _IResolvable_a771d0ef]]]]:
        '''An array of placement constraint objects to use for tasks in your service.

        You can specify a maximum of 10 constraints for each task. This limit includes constraints in the task definition and those specified at runtime.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementconstraints
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementConstraintProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def placement_strategies(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementStrategyProperty, _IResolvable_a771d0ef]]]]:
        '''The placement strategy objects to use for tasks in your service.

        You can specify a maximum of five strategy rules per service. For more information, see `Task Placement Strategies <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-placementstrategies
        '''
        result = self._values.get("placement_strategies")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.PlacementStrategyProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''The platform version that your tasks in the service are running on.

        A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used. For more information, see `AWS Fargate platform versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-platformversion
        '''
        result = self._values.get("platform_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def propagate_tags(self) -> typing.Optional[builtins.str]:
        '''Specifies whether to propagate the tags from the task definition or the service to the tasks in the service.

        If no value is specified, the tags are not propagated. Tags can only be propagated to the tasks within the service during service creation. To add tags to a task after service creation, use the `TagResource <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TagResource.html>`_ API action.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-propagatetags
        '''
        result = self._values.get("propagate_tags")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def role(self) -> typing.Optional[builtins.str]:
        '''The name or full Amazon Resource Name (ARN) of the IAM role that allows Amazon ECS to make calls to your load balancer on your behalf.

        This parameter is only permitted if you are using a load balancer with your service and your task definition doesn't use the ``awsvpc`` network mode. If you specify the ``role`` parameter, you must also specify a load balancer object with the ``loadBalancers`` parameter.
        .. epigraph::

           If your account has already created the Amazon ECS service-linked role, that role is used for your service unless you specify a role here. The service-linked role is required if your task definition uses the ``awsvpc`` network mode or if the service is configured to use service discovery, an external deployment controller, multiple target groups, or Elastic Inference accelerators in which case you don't specify a role here. For more information, see `Using service-linked roles for Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using-service-linked-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If your specified role has a path other than ``/`` , then you must either specify the full role ARN (this is recommended) or prefix the role name with the path. For example, if a role with the name ``bar`` has a path of ``/foo/`` then you would specify ``/foo/bar`` as the role name. For more information, see `Friendly names and paths <https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-friendly-names>`_ in the *IAM User Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-role
        '''
        result = self._values.get("role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scheduling_strategy(self) -> typing.Optional[builtins.str]:
        '''The scheduling strategy to use for the service. For more information, see `Services <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html>`_ .

        There are two service scheduler strategies available:

        - ``REPLICA`` -The replica scheduling strategy places and maintains the desired number of tasks across your cluster. By default, the service scheduler spreads tasks across Availability Zones. You can use task placement strategies and constraints to customize task placement decisions. This scheduler strategy is required if the service uses the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types.
        - ``DAEMON`` -The daemon scheduling strategy deploys exactly one task on each active container instance that meets all of the task placement constraints that you specify in your cluster. The service scheduler also evaluates the task placement constraints for running tasks and will stop tasks that don't meet the placement constraints. When you're using this strategy, you don't need to specify a desired number of tasks, a task placement strategy, or use Service Auto Scaling policies.

        .. epigraph::

           Tasks using the Fargate launch type or the ``CODE_DEPLOY`` or ``EXTERNAL`` deployment controller types don't support the ``DAEMON`` scheduling strategy.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-schedulingstrategy
        '''
        result = self._values.get("scheduling_strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''The name of your service.

        Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. Service names must be unique within a cluster, but you can have similarly named services in multiple clusters within a Region or across multiple Regions.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-servicename
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.ServiceRegistryProperty, _IResolvable_a771d0ef]]]]:
        '''The details of the service discovery registry to associate with this service. For more information, see `Service discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        .. epigraph::

           Each service may be associated with one service registry. Multiple service registries for each service isn't supported.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-serviceregistries
        '''
        result = self._values.get("service_registries")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnService.ServiceRegistryProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''The metadata that you apply to the service to help you categorize and organize them.

        Each tag consists of a key and an optional value, both of which you define. When a service is deleted, the tags are deleted as well.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def task_definition(self) -> typing.Optional[builtins.str]:
        '''The ``family`` and ``revision`` ( ``family:revision`` ) or full ARN of the task definition to run in your service.

        The ``revision`` is required in order for the resource to stabilize.

        A task definition must be specified if the service is using either the ``ECS`` or ``CODE_DEPLOY`` deployment controllers.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-service.html#cfn-ecs-service-taskdefinition
        '''
        result = self._values.get("task_definition")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnTaskDefinition(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnTaskDefinition",
):
    '''A CloudFormation ``AWS::ECS::TaskDefinition``.

    The ``AWS::ECS::TaskDefinition`` resource describes the container and volume definitions of an Amazon Elastic Container Service (Amazon ECS) task. You can specify which Docker images to use, the required resources, and other configurations related to launching the task definition through an Amazon ECS service or task.

    :cloudformationResource: AWS::ECS::TaskDefinition
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_task_definition = ecs.CfnTaskDefinition(self, "MyCfnTaskDefinition",
            container_definitions=[ecs.CfnTaskDefinition.ContainerDefinitionProperty(
                command=["command"],
                cpu=123,
                depends_on=[ecs.CfnTaskDefinition.ContainerDependencyProperty(
                    condition="condition",
                    container_name="containerName"
                )],
                disable_networking=False,
                dns_search_domains=["dnsSearchDomains"],
                dns_servers=["dnsServers"],
                docker_labels={
                    "docker_labels_key": "dockerLabels"
                },
                docker_security_options=["dockerSecurityOptions"],
                entry_point=["entryPoint"],
                environment=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                    name="name",
                    value="value"
                )],
                environment_files=[ecs.CfnTaskDefinition.EnvironmentFileProperty(
                    type="type",
                    value="value"
                )],
                essential=False,
                extra_hosts=[ecs.CfnTaskDefinition.HostEntryProperty(
                    hostname="hostname",
                    ip_address="ipAddress"
                )],
                firelens_configuration=ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                    options={
                        "options_key": "options"
                    },
                    type="type"
                ),
                health_check=ecs.CfnTaskDefinition.HealthCheckProperty(
                    command=["command"],
                    interval=123,
                    retries=123,
                    start_period=123,
                    timeout=123
                ),
                hostname="hostname",
                image="image",
                interactive=False,
                links=["links"],
                linux_parameters=ecs.CfnTaskDefinition.LinuxParametersProperty(
                    capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                        add=["add"],
                        drop=["drop"]
                    ),
                    devices=[ecs.CfnTaskDefinition.DeviceProperty(
                        container_path="containerPath",
                        host_path="hostPath",
                        permissions=["permissions"]
                    )],
                    init_process_enabled=False,
                    max_swap=123,
                    shared_memory_size=123,
                    swappiness=123,
                    tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                        size=123,
        
                        # the properties below are optional
                        container_path="containerPath",
                        mount_options=["mountOptions"]
                    )]
                ),
                log_configuration=ecs.CfnTaskDefinition.LogConfigurationProperty(
                    log_driver="logDriver",
        
                    # the properties below are optional
                    options={
                        "options_key": "options"
                    },
                    secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                ),
                memory=123,
                memory_reservation=123,
                mount_points=[ecs.CfnTaskDefinition.MountPointProperty(
                    container_path="containerPath",
                    read_only=False,
                    source_volume="sourceVolume"
                )],
                name="name",
                port_mappings=[ecs.CfnTaskDefinition.PortMappingProperty(
                    container_port=123,
                    host_port=123,
                    protocol="protocol"
                )],
                privileged=False,
                pseudo_terminal=False,
                readonly_root_filesystem=False,
                repository_credentials=ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                    credentials_parameter="credentialsParameter"
                ),
                resource_requirements=[ecs.CfnTaskDefinition.ResourceRequirementProperty(
                    type="type",
                    value="value"
                )],
                secrets=[ecs.CfnTaskDefinition.SecretProperty(
                    name="name",
                    value_from="valueFrom"
                )],
                start_timeout=123,
                stop_timeout=123,
                system_controls=[ecs.CfnTaskDefinition.SystemControlProperty(
                    namespace="namespace",
                    value="value"
                )],
                ulimits=[ecs.CfnTaskDefinition.UlimitProperty(
                    hard_limit=123,
                    name="name",
                    soft_limit=123
                )],
                user="user",
                volumes_from=[ecs.CfnTaskDefinition.VolumeFromProperty(
                    read_only=False,
                    source_container="sourceContainer"
                )],
                working_directory="workingDirectory"
            )],
            cpu="cpu",
            ephemeral_storage=ecs.CfnTaskDefinition.EphemeralStorageProperty(
                size_in_gi_b=123
            ),
            execution_role_arn="executionRoleArn",
            family="family",
            inference_accelerators=[ecs.CfnTaskDefinition.InferenceAcceleratorProperty(
                device_name="deviceName",
                device_type="deviceType"
            )],
            ipc_mode="ipcMode",
            memory="memory",
            network_mode="networkMode",
            pid_mode="pidMode",
            placement_constraints=[ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty(
                type="type",
        
                # the properties below are optional
                expression="expression"
            )],
            proxy_configuration=ecs.CfnTaskDefinition.ProxyConfigurationProperty(
                container_name="containerName",
        
                # the properties below are optional
                proxy_configuration_properties=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                    name="name",
                    value="value"
                )],
                type="type"
            ),
            requires_compatibilities=["requiresCompatibilities"],
            runtime_platform=ecs.CfnTaskDefinition.RuntimePlatformProperty(
                cpu_architecture="cpuArchitecture",
                operating_system_family="operatingSystemFamily"
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            task_role_arn="taskRoleArn",
            volumes=[ecs.CfnTaskDefinition.VolumeProperty(
                docker_volume_configuration=ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                    autoprovision=False,
                    driver="driver",
                    driver_opts={
                        "driver_opts_key": "driverOpts"
                    },
                    labels={
                        "labels_key": "labels"
                    },
                    scope="scope"
                ),
                efs_volume_configuration=ecs.CfnTaskDefinition.EfsVolumeConfigurationProperty(
                    file_system_id="fileSystemId",
        
                    # the properties below are optional
                    authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                        access_point_id="accessPointId",
                        iam="iam"
                    ),
                    root_directory="rootDirectory",
                    transit_encryption="transitEncryption",
                    transit_encryption_port=123
                ),
                host=ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                    source_path="sourcePath"
                ),
                name="name"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        container_definitions: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]] = None,
        cpu: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        family: typing.Optional[builtins.str] = None,
        inference_accelerators: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]] = None,
        ipc_mode: typing.Optional[builtins.str] = None,
        memory: typing.Optional[builtins.str] = None,
        network_mode: typing.Optional[builtins.str] = None,
        pid_mode: typing.Optional[builtins.str] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]] = None,
        proxy_configuration: typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]] = None,
        requires_compatibilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        runtime_platform: typing.Optional[typing.Union["CfnTaskDefinition.RuntimePlatformProperty", _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_role_arn: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::TaskDefinition``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param container_definitions: A list of container definitions in JSON format that describe the different containers that make up your task. For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param cpu: The number of ``cpu`` units used by the task. If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter. The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate. - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - 2048 (2 vCPU) - Available ``memory`` values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - 4096 (4 vCPU) - Available ``memory`` values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)
        :param ephemeral_storage: The ephemeral storage settings to use for tasks run with the task definition.
        :param execution_role_arn: The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf. The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param family: The name of a family that this task definition is registered to. Up to 255 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed. A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task definition that you add. .. epigraph:: To use revision numbers when you update a task definition, specify this property. If you don't specify a value, AWS CloudFormation generates a new task definition each time that you update it.
        :param inference_accelerators: The Elastic Inference accelerators to use for the containers in the task.
        :param ipc_mode: The IPC resource namespace to use for the containers in the task. The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same IPC resources. If ``none`` is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the *Docker run reference* . If the ``host`` IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ . If you are setting namespaced kernel parameters using ``systemControls`` for the containers in the task, the following will apply to your IPC resource namespace. For more information, see `System Controls <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_ in the *Amazon Elastic Container Service Developer Guide* . - For tasks that use the ``host`` IPC mode, IPC namespace related ``systemControls`` are not supported. - For tasks that use the ``task`` IPC mode, IPC namespace related ``systemControls`` will apply to all containers within a task. .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
        :param memory: The amount (in MiB) of memory used by the task. If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a container-level memory value. This field is optional and any value can be used. If a task-level memory value is specified, the container-level memory value is optional. For more information regarding container-level memory and memory reservation, see `ContainerDefinition <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html>`_ . If your tasks runs on AWS Fargate , this field is required. You must use one of the following values. The value you choose determines your range of valid values for the ``cpu`` parameter. - 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available ``cpu`` values: 256 (.25 vCPU) - 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available ``cpu`` values: 512 (.5 vCPU) - 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available ``cpu`` values: 1024 (1 vCPU) - Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 2048 (2 vCPU) - Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 4096 (4 vCPU)
        :param network_mode: The Docker networking mode to use for the containers in the task. The valid values are ``none`` , ``bridge`` , ``awsvpc`` , and ``host`` . The default Docker network mode is ``bridge`` . If you are using the Fargate launch type, the ``awsvpc`` network mode is required. If you are using the EC2 launch type, any network mode can be used. If the network mode is set to ``none`` , you cannot specify port mappings in your container definitions, and the tasks containers do not have external connectivity. The ``host`` and ``awsvpc`` network modes offer the highest networking performance for containers because they use the EC2 network stack instead of the virtualized network stack provided by the ``bridge`` mode. With the ``host`` and ``awsvpc`` network modes, exposed container ports are mapped directly to the corresponding host port (for the ``host`` network mode) or the attached elastic network interface port (for the ``awsvpc`` network mode), so you cannot take advantage of dynamic host port mappings. If the network mode is ``awsvpc`` , the task is allocated an elastic network interface, and you must specify a `NetworkConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html>`_ value when you create a service or run a task with the task definition. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: Currently, only Amazon ECS-optimized AMIs, other Amazon Linux variants with the ``ecs-init`` package, or AWS Fargate infrastructure support the ``awsvpc`` network mode. If the network mode is ``host`` , you cannot run multiple instantiations of the same task on a single container instance when port mappings are used. Docker for Windows uses different network modes than Docker for Linux. When you register a task definition with Windows containers, you must not specify a network mode. If you use the console to register a task definition with Windows containers, you must choose the ``<default>`` network mode object. For more information, see `Network settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#network-settings>`_ in the *Docker run reference* .
        :param pid_mode: The process namespace to use for the containers in the task. The valid values are ``host`` or ``task`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the *Docker run reference* . If the ``host`` PID mode is used, be aware that there is a heightened risk of undesired process namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ . .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
        :param placement_constraints: An array of placement constraint objects to use for tasks. .. epigraph:: This parameter isn't supported for tasks run on AWS Fargate .
        :param proxy_configuration: The ``ProxyConfiguration`` property specifies the configuration details for the App Mesh proxy. Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized AMI version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param requires_compatibilities: The task launch types the task definition was validated against. To determine which task launch types the task definition is validated for, see the ``TaskDefinition$compatibilities`` parameter.
        :param runtime_platform: The operating system that your tasks definitions run on. A platform family is specified only for tasks using the Fargate launch type. When you specify a task definition in a service, this value must match the ``runtimePlatform`` value of the service.
        :param tags: The metadata that you apply to the task definition to help you categorize and organize them. Each tag consists of a key and an optional value. You define both of them. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_role_arn: The short name or full Amazon Resource Name (ARN) of the AWS Identity and Access Management role that grants containers in the task permission to call AWS APIs on your behalf. For more information, see `Amazon ECS Task Role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . IAM roles for tasks on Windows require that the ``-EnableTaskIAMRole`` option is set when you launch the Amazon ECS-optimized Windows AMI. Your containers must also run some configuration code to use the feature. For more information, see `Windows IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows_task_IAM_roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param volumes: The list of data volume definitions for the task. For more information, see `Using data volumes in tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: The ``host`` and ``sourcePath`` parameters aren't supported for tasks run on AWS Fargate .
        '''
        props = CfnTaskDefinitionProps(
            container_definitions=container_definitions,
            cpu=cpu,
            ephemeral_storage=ephemeral_storage,
            execution_role_arn=execution_role_arn,
            family=family,
            inference_accelerators=inference_accelerators,
            ipc_mode=ipc_mode,
            memory=memory,
            network_mode=network_mode,
            pid_mode=pid_mode,
            placement_constraints=placement_constraints,
            proxy_configuration=proxy_configuration,
            requires_compatibilities=requires_compatibilities,
            runtime_platform=runtime_platform,
            tags=tags,
            task_role_arn=task_role_arn,
            volumes=volumes,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrTaskDefinitionArn")
    def attr_task_definition_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: TaskDefinitionArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTaskDefinitionArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''The metadata that you apply to the task definition to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both of them.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerDefinitions")
    def container_definitions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]]:
        '''A list of container definitions in JSON format that describe the different containers that make up your task.

        For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-containerdefinitions
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "containerDefinitions"))

    @container_definitions.setter
    def container_definitions(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDefinitionProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "containerDefinitions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cpu")
    def cpu(self) -> typing.Optional[builtins.str]:
        '''The number of ``cpu`` units used by the task.

        If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter.

        The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.

        - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
        - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
        - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
        - 2048 (2 vCPU) - Available ``memory`` values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)
        - 4096 (4 vCPU) - Available ``memory`` values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-cpu
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cpu"))

    @cpu.setter
    def cpu(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cpu", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]]:
        '''The ephemeral storage settings to use for tasks run with the task definition.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ephemeralstorage
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union["CfnTaskDefinition.EphemeralStorageProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "ephemeralStorage", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executionRoleArn")
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf.

        The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-executionrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "executionRoleArn"))

    @execution_role_arn.setter
    def execution_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "executionRoleArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="family")
    def family(self) -> typing.Optional[builtins.str]:
        '''The name of a family that this task definition is registered to.

        Up to 255 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.

        A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task definition that you add.
        .. epigraph::

           To use revision numbers when you update a task definition, specify this property. If you don't specify a value, AWS CloudFormation generates a new task definition each time that you update it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-family
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "family"))

    @family.setter
    def family(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "family", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="inferenceAccelerators")
    def inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]]:
        '''The Elastic Inference accelerators to use for the containers in the task.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-inferenceaccelerators
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "inferenceAccelerators"))

    @inference_accelerators.setter
    def inference_accelerators(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.InferenceAcceleratorProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "inferenceAccelerators", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipcMode")
    def ipc_mode(self) -> typing.Optional[builtins.str]:
        '''The IPC resource namespace to use for the containers in the task.

        The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same IPC resources. If ``none`` is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the *Docker run reference* .

        If the ``host`` IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ .

        If you are setting namespaced kernel parameters using ``systemControls`` for the containers in the task, the following will apply to your IPC resource namespace. For more information, see `System Controls <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        - For tasks that use the ``host`` IPC mode, IPC namespace related ``systemControls`` are not supported.
        - For tasks that use the ``task`` IPC mode, IPC namespace related ``systemControls`` will apply to all containers within a task.

        .. epigraph::

           This parameter is not supported for Windows containers or tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ipcmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ipcMode"))

    @ipc_mode.setter
    def ipc_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ipcMode", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="memory")
    def memory(self) -> typing.Optional[builtins.str]:
        '''The amount (in MiB) of memory used by the task.

        If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a container-level memory value. This field is optional and any value can be used. If a task-level memory value is specified, the container-level memory value is optional. For more information regarding container-level memory and memory reservation, see `ContainerDefinition <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html>`_ .

        If your tasks runs on AWS Fargate , this field is required. You must use one of the following values. The value you choose determines your range of valid values for the ``cpu`` parameter.

        - 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available ``cpu`` values: 256 (.25 vCPU)
        - 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available ``cpu`` values: 512 (.5 vCPU)
        - 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available ``cpu`` values: 1024 (1 vCPU)
        - Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 2048 (2 vCPU)
        - Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 4096 (4 vCPU)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "memory"))

    @memory.setter
    def memory(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "memory", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkMode")
    def network_mode(self) -> typing.Optional[builtins.str]:
        '''The Docker networking mode to use for the containers in the task.

        The valid values are ``none`` , ``bridge`` , ``awsvpc`` , and ``host`` . The default Docker network mode is ``bridge`` . If you are using the Fargate launch type, the ``awsvpc`` network mode is required. If you are using the EC2 launch type, any network mode can be used. If the network mode is set to ``none`` , you cannot specify port mappings in your container definitions, and the tasks containers do not have external connectivity. The ``host`` and ``awsvpc`` network modes offer the highest networking performance for containers because they use the EC2 network stack instead of the virtualized network stack provided by the ``bridge`` mode.

        With the ``host`` and ``awsvpc`` network modes, exposed container ports are mapped directly to the corresponding host port (for the ``host`` network mode) or the attached elastic network interface port (for the ``awsvpc`` network mode), so you cannot take advantage of dynamic host port mappings.

        If the network mode is ``awsvpc`` , the task is allocated an elastic network interface, and you must specify a `NetworkConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html>`_ value when you create a service or run a task with the task definition. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        .. epigraph::

           Currently, only Amazon ECS-optimized AMIs, other Amazon Linux variants with the ``ecs-init`` package, or AWS Fargate infrastructure support the ``awsvpc`` network mode.

        If the network mode is ``host`` , you cannot run multiple instantiations of the same task on a single container instance when port mappings are used.

        Docker for Windows uses different network modes than Docker for Linux. When you register a task definition with Windows containers, you must not specify a network mode. If you use the console to register a task definition with Windows containers, you must choose the ``<default>`` network mode object.

        For more information, see `Network settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#network-settings>`_ in the *Docker run reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-networkmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkMode"))

    @network_mode.setter
    def network_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "networkMode", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="pidMode")
    def pid_mode(self) -> typing.Optional[builtins.str]:
        '''The process namespace to use for the containers in the task.

        The valid values are ``host`` or ``task`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the *Docker run reference* .

        If the ``host`` PID mode is used, be aware that there is a heightened risk of undesired process namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ .
        .. epigraph::

           This parameter is not supported for Windows containers or tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-pidmode
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "pidMode"))

    @pid_mode.setter
    def pid_mode(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "pidMode", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementConstraints")
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]]:
        '''An array of placement constraint objects to use for tasks.

        .. epigraph::

           This parameter isn't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-placementconstraints
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "placementConstraints"))

    @placement_constraints.setter
    def placement_constraints(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "placementConstraints", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="proxyConfiguration")
    def proxy_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]]:
        '''The ``ProxyConfiguration`` property specifies the configuration details for the App Mesh proxy.

        Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized AMI version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-proxyconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "proxyConfiguration"))

    @proxy_configuration.setter
    def proxy_configuration(
        self,
        value: typing.Optional[typing.Union["CfnTaskDefinition.ProxyConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "proxyConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="requiresCompatibilities")
    def requires_compatibilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The task launch types the task definition was validated against.

        To determine which task launch types the task definition is validated for, see the ``TaskDefinition$compatibilities`` parameter.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-requirescompatibilities
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "requiresCompatibilities"))

    @requires_compatibilities.setter
    def requires_compatibilities(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "requiresCompatibilities", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="runtimePlatform")
    def runtime_platform(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskDefinition.RuntimePlatformProperty", _IResolvable_a771d0ef]]:
        '''The operating system that your tasks definitions run on.

        A platform family is specified only for tasks using the Fargate launch type.

        When you specify a task definition in a service, this value must match the ``runtimePlatform`` value of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-runtimeplatform
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.RuntimePlatformProperty", _IResolvable_a771d0ef]], jsii.get(self, "runtimePlatform"))

    @runtime_platform.setter
    def runtime_platform(
        self,
        value: typing.Optional[typing.Union["CfnTaskDefinition.RuntimePlatformProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "runtimePlatform", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskRoleArn")
    def task_role_arn(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the AWS Identity and Access Management role that grants containers in the task permission to call AWS APIs on your behalf.

        For more information, see `Amazon ECS Task Role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        IAM roles for tasks on Windows require that the ``-EnableTaskIAMRole`` option is set when you launch the Amazon ECS-optimized Windows AMI. Your containers must also run some configuration code to use the feature. For more information, see `Windows IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows_task_IAM_roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-taskrolearn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "taskRoleArn"))

    @task_role_arn.setter
    def task_role_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "taskRoleArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volumes")
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]]:
        '''The list of data volume definitions for the task.

        For more information, see `Using data volumes in tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        .. epigraph::

           The ``host`` and ``sourcePath`` parameters aren't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-volumes
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "volumes"))

    @volumes.setter
    def volumes(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "volumes", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.AuthorizationConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"access_point_id": "accessPointId", "iam": "iam"},
    )
    class AuthorizationConfigProperty:
        def __init__(
            self,
            *,
            access_point_id: typing.Optional[builtins.str] = None,
            iam: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The authorization configuration details for the Amazon EFS file system.

            :param access_point_id: The Amazon EFS access point ID to use. If an access point is specified, the root directory value specified in the ``EFSVolumeConfiguration`` must either be omitted or set to ``/`` which will enforce the path set on the EFS access point. If an access point is used, transit encryption must be enabled in the ``EFSVolumeConfiguration`` . For more information, see `Working with Amazon EFS Access Points <https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html>`_ in the *Amazon Elastic File System User Guide* .
            :param iam: Determines whether to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system. If enabled, transit encryption must be enabled in the ``EFSVolumeConfiguration`` . If this parameter is omitted, the default value of ``DISABLED`` is used. For more information, see `Using Amazon EFS Access Points <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/efs-volumes.html#efs-volume-accesspoints>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                authorization_config_property = ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                    access_point_id="accessPointId",
                    iam="iam"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if access_point_id is not None:
                self._values["access_point_id"] = access_point_id
            if iam is not None:
                self._values["iam"] = iam

        @builtins.property
        def access_point_id(self) -> typing.Optional[builtins.str]:
            '''The Amazon EFS access point ID to use.

            If an access point is specified, the root directory value specified in the ``EFSVolumeConfiguration`` must either be omitted or set to ``/`` which will enforce the path set on the EFS access point. If an access point is used, transit encryption must be enabled in the ``EFSVolumeConfiguration`` . For more information, see `Working with Amazon EFS Access Points <https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html>`_ in the *Amazon Elastic File System User Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html#cfn-ecs-taskdefinition-authorizationconfig-accesspointid
            '''
            result = self._values.get("access_point_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def iam(self) -> typing.Optional[builtins.str]:
            '''Determines whether to use the Amazon ECS task IAM role defined in a task definition when mounting the Amazon EFS file system.

            If enabled, transit encryption must be enabled in the ``EFSVolumeConfiguration`` . If this parameter is omitted, the default value of ``DISABLED`` is used. For more information, see `Using Amazon EFS Access Points <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/efs-volumes.html#efs-volume-accesspoints>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-authorizationconfig.html#cfn-ecs-taskdefinition-authorizationconfig-iam
            '''
            result = self._values.get("iam")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AuthorizationConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ContainerDefinitionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "cpu": "cpu",
            "depends_on": "dependsOn",
            "disable_networking": "disableNetworking",
            "dns_search_domains": "dnsSearchDomains",
            "dns_servers": "dnsServers",
            "docker_labels": "dockerLabels",
            "docker_security_options": "dockerSecurityOptions",
            "entry_point": "entryPoint",
            "environment": "environment",
            "environment_files": "environmentFiles",
            "essential": "essential",
            "extra_hosts": "extraHosts",
            "firelens_configuration": "firelensConfiguration",
            "health_check": "healthCheck",
            "hostname": "hostname",
            "image": "image",
            "interactive": "interactive",
            "links": "links",
            "linux_parameters": "linuxParameters",
            "log_configuration": "logConfiguration",
            "memory": "memory",
            "memory_reservation": "memoryReservation",
            "mount_points": "mountPoints",
            "name": "name",
            "port_mappings": "portMappings",
            "privileged": "privileged",
            "pseudo_terminal": "pseudoTerminal",
            "readonly_root_filesystem": "readonlyRootFilesystem",
            "repository_credentials": "repositoryCredentials",
            "resource_requirements": "resourceRequirements",
            "secrets": "secrets",
            "start_timeout": "startTimeout",
            "stop_timeout": "stopTimeout",
            "system_controls": "systemControls",
            "ulimits": "ulimits",
            "user": "user",
            "volumes_from": "volumesFrom",
            "working_directory": "workingDirectory",
        },
    )
    class ContainerDefinitionProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            cpu: typing.Optional[jsii.Number] = None,
            depends_on: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.ContainerDependencyProperty", _IResolvable_a771d0ef]]]] = None,
            disable_networking: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
            dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
            docker_labels: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
            entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
            environment: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]] = None,
            environment_files: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.EnvironmentFileProperty", _IResolvable_a771d0ef]]]] = None,
            essential: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            extra_hosts: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.HostEntryProperty", _IResolvable_a771d0ef]]]] = None,
            firelens_configuration: typing.Optional[typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", _IResolvable_a771d0ef]] = None,
            health_check: typing.Optional[typing.Union["CfnTaskDefinition.HealthCheckProperty", _IResolvable_a771d0ef]] = None,
            hostname: typing.Optional[builtins.str] = None,
            image: typing.Optional[builtins.str] = None,
            interactive: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            links: typing.Optional[typing.Sequence[builtins.str]] = None,
            linux_parameters: typing.Optional[typing.Union["CfnTaskDefinition.LinuxParametersProperty", _IResolvable_a771d0ef]] = None,
            log_configuration: typing.Optional[typing.Union["CfnTaskDefinition.LogConfigurationProperty", _IResolvable_a771d0ef]] = None,
            memory: typing.Optional[jsii.Number] = None,
            memory_reservation: typing.Optional[jsii.Number] = None,
            mount_points: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.MountPointProperty", _IResolvable_a771d0ef]]]] = None,
            name: typing.Optional[builtins.str] = None,
            port_mappings: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.PortMappingProperty", _IResolvable_a771d0ef]]]] = None,
            privileged: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            pseudo_terminal: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            readonly_root_filesystem: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            repository_credentials: typing.Optional[typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", _IResolvable_a771d0ef]] = None,
            resource_requirements: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.ResourceRequirementProperty", _IResolvable_a771d0ef]]]] = None,
            secrets: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]] = None,
            start_timeout: typing.Optional[jsii.Number] = None,
            stop_timeout: typing.Optional[jsii.Number] = None,
            system_controls: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.SystemControlProperty", _IResolvable_a771d0ef]]]] = None,
            ulimits: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.UlimitProperty", _IResolvable_a771d0ef]]]] = None,
            user: typing.Optional[builtins.str] = None,
            volumes_from: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.VolumeFromProperty", _IResolvable_a771d0ef]]]] = None,
            working_directory: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ContainerDefinition`` property specifies a container definition.

            Container definitions are used in task definitions to describe the different containers that are launched as part of a task.

            :param command: The command that's passed to the container. This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#cmd <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ . If there are multiple arguments, each argument is a separated string in the array.
            :param cpu: The number of ``cpu`` units reserved for the container. This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This field is optional for tasks using the Fargate launch type, and the only requirement is that the total amount of CPU reserved for all containers within a task be lower than the task-level ``cpu`` value. .. epigraph:: You can determine the number of CPU units that are available per EC2 instance type by multiplying the vCPUs listed for that instance type on the `Amazon EC2 Instances <https://docs.aws.amazon.com/ec2/instance-types/>`_ detail page by 1,024. Linux containers share unallocated CPU units with other containers on the container instance with the same ratio as their allocated amount. For example, if you run a single-container task on a single-core instance type with 512 CPU units specified for that container, and that's the only task running on the container instance, that container could use the full 1,024 CPU unit share at any given time. However, if you launched another copy of the same task on that container instance, each task is guaranteed a minimum of 512 CPU units when needed. Moreover, each container could float to higher CPU usage if the other container was not using it. If both tasks were 100% active all of the time, they would be limited to 512 CPU units. On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate the relative CPU share ratios for running containers. For more information, see `CPU share constraint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#cpu-share-constraint>`_ in the Docker documentation. The minimum valid CPU share value that the Linux kernel allows is 2. However, the CPU parameter isn't required, and you can use CPU values below 2 in your container definitions. For CPU values below 2 (including null), the behavior varies based on your Amazon ECS container agent version: - *Agent versions less than or equal to 1.1.0:* Null and zero CPU values are passed to Docker as 0, which Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the Linux kernel converts to two CPU shares. - *Agent versions greater than or equal to 1.2.0:* Null, zero, and CPU values of 1 are passed to Docker as 2. On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows containers only have access to the specified amount of CPU that's described in the task definition. A null or zero CPU value is passed to Docker as ``0`` , which Windows interprets as 1% of one CPU.
            :param depends_on: The dependencies defined for container startup and shutdown. A container can contain multiple dependencies. When a dependency is defined for container startup, for container shutdown it is reversed. For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent to turn on container dependencies. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For tasks using the Fargate launch type, the task or service requires the following platforms: - Linux platform version ``1.3.0`` or later. - Windows platform version ``1.0.0`` or later.
            :param disable_networking: When this parameter is true, networking is disabled within the container. This parameter maps to ``NetworkDisabled`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param dns_search_domains: A list of DNS search domains that are presented to the container. This parameter maps to ``DnsSearch`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns-search`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param dns_servers: A list of DNS servers that are presented to the container. This parameter maps to ``Dns`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param docker_labels: A key/value map of labels to add to the container. This parameter maps to ``Labels`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--label`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param docker_security_options: A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. This field isn't valid for containers in tasks using the Fargate launch type. With Windows containers, this parameter can be used to reference a credential spec file when configuring a container for Active Directory authentication. For more information, see `Using gMSAs for Windows Containers <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html>`_ in the *Amazon Elastic Container Service Developer Guide* . This parameter maps to ``SecurityOpt`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--security-opt`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: The Amazon ECS container agent running on a container instance must register with the ``ECS_SELINUX_CAPABLE=true`` or ``ECS_APPARMOR_CAPABLE=true`` environment variables before containers placed on that instance can use these security options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For more information about valid values, see `Docker Run Security Configuration <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" | "credentialspec:CredentialSpecFilePath"
            :param entry_point: .. epigraph:: Early versions of the Amazon ECS container agent don't properly handle ``entryPoint`` parameters. If you have problems using ``entryPoint`` , update your container agent or enter your commands and arguments as ``command`` array items instead. The entry point that's passed to the container. This parameter maps to ``Entrypoint`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--entrypoint`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#entrypoint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#entrypoint>`_ .
            :param environment: The environment variables to pass to a container. This parameter maps to ``Env`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--env`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: We don't recommend that you use plaintext environment variables for sensitive information, such as credential data.
            :param environment_files: A list of files containing the environment variables to pass to a container. This parameter maps to the ``--env-file`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . You can specify up to ten environment files. The file must have a ``.env`` file extension. Each line in an environment file contains an environment variable in ``VARIABLE=VALUE`` format. Lines beginning with ``#`` are treated as comments and are ignored. For more information about the environment variable file syntax, see `Declare default environment variables in file <https://docs.aws.amazon.com/https://docs.docker.com/compose/env-file/>`_ . If there are environment variables specified using the ``environment`` parameter in a container definition, they take precedence over the variables contained within an environment file. If multiple environment files are specified that contain the same variable, they're processed from the top down. We recommend that you use unique variable names. For more information, see `Specifying Environment Variables <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param essential: If the ``essential`` parameter of a container is marked as ``true`` , and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the ``essential`` parameter of a container is marked as ``false`` , its failure doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential. All tasks must have at least one essential container. If you have an application that's composed of multiple containers, group containers that are used for a common purpose into components, and separate the different components into multiple task definitions. For more information, see `Application Architecture <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param extra_hosts: A list of hostnames and IP address mappings to append to the ``/etc/hosts`` file on the container. This parameter maps to ``ExtraHosts`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--add-host`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter isn't supported for Windows containers or tasks that use the ``awsvpc`` network mode.
            :param firelens_configuration: The FireLens configuration for the container. This is used to specify and configure a log router for container logs. For more information, see `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param health_check: The container health check command and associated configuration parameters for the container. This parameter maps to ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``HEALTHCHECK`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param hostname: The hostname to use for your container. This parameter maps to ``Hostname`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--hostname`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: The ``hostname`` parameter is not supported if you're using the ``awsvpc`` network mode.
            :param image: The image used to start a container. This string is passed directly to the Docker daemon. By default, images in the Docker Hub registry are available. Other repositories are specified with either ``*repository-url* / *image* : *tag*`` or ``*repository-url* / *image* @ *digest*`` . Up to 255 letters (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``IMAGE`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . - When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image and tag for the container to use. However, subsequent updates to a repository image aren't propagated to already running tasks. - Images in Amazon ECR repositories can be specified by either using the full ``registry/repository:tag`` or ``registry/repository@digest`` . For example, ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest`` or ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE`` . - Images in official repositories on Docker Hub use a single name (for example, ``ubuntu`` or ``mongo`` ). - Images in other repositories on Docker Hub are qualified with an organization name (for example, ``amazon/amazon-ecs-agent`` ). - Images in other online repositories are qualified further by a domain name (for example, ``quay.io/assemblyline/ubuntu`` ).
            :param interactive: When this parameter is ``true`` , you can deploy containerized applications that require ``stdin`` or a ``tty`` to be allocated. This parameter maps to ``OpenStdin`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--interactive`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param links: The ``links`` parameter allows containers to communicate with each other without the need for port mappings. This parameter is only supported if the network mode of a task definition is ``bridge`` . The ``name:internalName`` construct is analogous to ``name:alias`` in Docker links. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. For more information about linking Docker containers, go to `Legacy container links <https://docs.aws.amazon.com/https://docs.docker.com/network/links/>`_ in the Docker documentation. This parameter maps to ``Links`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--link`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers. > Containers that are collocated on a single container instance may be able to communicate with each other without requiring links or host port mappings. Network isolation is achieved on the container instance using security groups and VPC settings.
            :param linux_parameters: Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param log_configuration: The log configuration specification for the container. This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However, the container may use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information on the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation. .. epigraph:: Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the `LogConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html>`_ data type). Additional log drivers may be available in future releases of the Amazon ECS container agent. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'`` .. epigraph:: The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param memory: The amount (in MiB) of memory to present to the container. If your container attempts to exceed the memory specified here, the container is killed. The total amount of memory reserved for all containers within a task must be lower than the task ``memory`` value, if one is specified. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . If using the Fargate launch type, this parameter is optional. If using the EC2 launch type, you must specify either a task-level memory value or a container-level memory value. If you specify both a container-level ``memory`` and ``memoryReservation`` value, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used. The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container, so you should not specify fewer than 6 MiB of memory for your containers. The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container, so you should not specify fewer than 4 MiB of memory for your containers.
            :param memory_reservation: The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the ``memory`` parameter (if applicable), or all of the available memory on the container instance, whichever comes first. This parameter maps to ``MemoryReservation`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory-reservation`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . If a task-level memory value is not specified, you must specify a non-zero integer for one or both of ``memory`` or ``memoryReservation`` in a container definition. If you specify both, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used. For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of memory for short periods of time, you can set a ``memoryReservation`` of 128 MiB, and a ``memory`` hard limit of 300 MiB. This configuration would allow the container to only reserve 128 MiB of memory from the remaining resources on the container instance, but also allow the container to consume more memory resources when needed. The Docker daemon reserves a minimum of 4 MiB of memory for a container. Therefore, we recommend that you specify fewer than 4 MiB of memory for your containers.
            :param mount_points: The mount points for data volumes in your container. This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volume`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives.
            :param name: The name of a container. If you're linking multiple containers together in a task definition, the ``name`` of one container can be entered in the ``links`` of another container to connect the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This parameter maps to ``name`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--name`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param port_mappings: The list of port mappings for the container. Port mappings allow containers to access ports on the host container instance to send or receive traffic. For task definitions that use the ``awsvpc`` network mode, you should only specify the ``containerPort`` . The ``hostPort`` can be left blank or it must be the same value as the ``containerPort`` . Port mappings on Windows use the ``NetNAT`` gateway address rather than ``localhost`` . There is no loopback for port mappings on Windows, so you cannot access a container's mapped port from the host itself. This parameter maps to ``PortBindings`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--publish`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . If the network mode of a task definition is set to ``none`` , then you can't specify port mappings. If the network mode of a task definition is set to ``host`` , then host ports must either be undefined or they must match the container port in the port mapping. .. epigraph:: After a task reaches the ``RUNNING`` status, manual and automatic host and container port assignments are visible in the *Network Bindings* section of a container description for a selected task in the Amazon ECS console. The assignments are also visible in the ``networkBindings`` section `DescribeTasks <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ responses.
            :param privileged: When this parameter is true, the container is given elevated privileges on the host container instance (similar to the ``root`` user). This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
            :param pseudo_terminal: When this parameter is ``true`` , a TTY is allocated. This parameter maps to ``Tty`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--tty`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param readonly_root_filesystem: When this parameter is true, the container is given read-only access to its root file system. This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--read-only`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: This parameter is not supported for Windows containers.
            :param repository_credentials: The private repository authentication credentials to use.
            :param resource_requirements: The type and amount of a resource to assign to a container. The only supported resource is a GPU.
            :param secrets: The secrets to pass to the container. For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param start_timeout: Time duration (in seconds) to wait before giving up on resolving dependencies for a container. For example, you specify two containers in a task definition with containerA having a dependency on containerB reaching a ``COMPLETE`` , ``SUCCESS`` , or ``HEALTHY`` status. If a ``startTimeout`` value is specified for containerB and it doesn't reach the desired status within that time then containerA gives up and not start. This results in the task transitioning to a ``STOPPED`` state. .. epigraph:: When the ``ECS_CONTAINER_START_TIMEOUT`` container agent configuration variable is used, it's enforced independently from this start timeout value. For tasks using the Fargate launch type, the task or service requires the following platforms: - Linux platform version ``1.3.0`` or later. - Windows platform version ``1.0.0`` or later. For tasks using the EC2 launch type, your container instances require at least version ``1.26.0`` of the container agent to use a container start timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version ``1.26.0-1`` of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param stop_timeout: Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. For tasks using the Fargate launch type, the task or service requires the following platforms: - Linux platform version ``1.3.0`` or later. - Windows platform version ``1.0.0`` or later. The max stop timeout value is 120 seconds and if the parameter is not specified, the default value of 30 seconds is used. For tasks that use the EC2 launch type, if the ``stopTimeout`` parameter isn't specified, the value set for the Amazon ECS container agent configuration variable ``ECS_CONTAINER_STOP_TIMEOUT`` is used. If neither the ``stopTimeout`` parameter or the ``ECS_CONTAINER_STOP_TIMEOUT`` agent configuration variable are set, then the default values of 30 seconds for Linux containers and 30 seconds on Windows containers are used. Your container instances require at least version 1.26.0 of the container agent to use a container stop timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            :param system_controls: A list of namespaced kernel parameters to set in the container. This parameter maps to ``Sysctls`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--sysctl`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: We don't recommended that you specify network-related ``systemControls`` parameters for multiple containers in a single task that also uses either the ``awsvpc`` or ``host`` network modes. For tasks that use the ``awsvpc`` network mode, the container that's started last determines which ``systemControls`` parameters take effect. For tasks that use the ``host`` network mode, it changes the container instance's namespaced kernel parameters as well as the containers.
            :param ulimits: A list of ``ulimits`` to set in the container. This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Valid naming values are displayed in the `Ulimit <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html>`_ data type. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'`` .. epigraph:: This parameter is not supported for Windows containers.
            :param user: The user to use inside the container. This parameter maps to ``User`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--user`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: When running tasks using the ``host`` network mode, don't run containers using the root user (UID 0). We recommend using a non-root user for better security. You can specify the ``user`` using the following formats. If specifying a UID or GID, you must specify it as a positive integer. - ``user`` - ``user:group`` - ``uid`` - ``uid:gid`` - ``user:gid`` - ``uid:group`` .. epigraph:: This parameter is not supported for Windows containers.
            :param volumes_from: Data volumes to mount from another container. This parameter maps to ``VolumesFrom`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volumes-from`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            :param working_directory: The working directory to run commands inside the container in. This parameter maps to ``WorkingDir`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--workdir`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                container_definition_property = ecs.CfnTaskDefinition.ContainerDefinitionProperty(
                    command=["command"],
                    cpu=123,
                    depends_on=[ecs.CfnTaskDefinition.ContainerDependencyProperty(
                        condition="condition",
                        container_name="containerName"
                    )],
                    disable_networking=False,
                    dns_search_domains=["dnsSearchDomains"],
                    dns_servers=["dnsServers"],
                    docker_labels={
                        "docker_labels_key": "dockerLabels"
                    },
                    docker_security_options=["dockerSecurityOptions"],
                    entry_point=["entryPoint"],
                    environment=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                        name="name",
                        value="value"
                    )],
                    environment_files=[ecs.CfnTaskDefinition.EnvironmentFileProperty(
                        type="type",
                        value="value"
                    )],
                    essential=False,
                    extra_hosts=[ecs.CfnTaskDefinition.HostEntryProperty(
                        hostname="hostname",
                        ip_address="ipAddress"
                    )],
                    firelens_configuration=ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                        options={
                            "options_key": "options"
                        },
                        type="type"
                    ),
                    health_check=ecs.CfnTaskDefinition.HealthCheckProperty(
                        command=["command"],
                        interval=123,
                        retries=123,
                        start_period=123,
                        timeout=123
                    ),
                    hostname="hostname",
                    image="image",
                    interactive=False,
                    links=["links"],
                    linux_parameters=ecs.CfnTaskDefinition.LinuxParametersProperty(
                        capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                            add=["add"],
                            drop=["drop"]
                        ),
                        devices=[ecs.CfnTaskDefinition.DeviceProperty(
                            container_path="containerPath",
                            host_path="hostPath",
                            permissions=["permissions"]
                        )],
                        init_process_enabled=False,
                        max_swap=123,
                        shared_memory_size=123,
                        swappiness=123,
                        tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                            size=123,
                
                            # the properties below are optional
                            container_path="containerPath",
                            mount_options=["mountOptions"]
                        )]
                    ),
                    log_configuration=ecs.CfnTaskDefinition.LogConfigurationProperty(
                        log_driver="logDriver",
                
                        # the properties below are optional
                        options={
                            "options_key": "options"
                        },
                        secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    memory=123,
                    memory_reservation=123,
                    mount_points=[ecs.CfnTaskDefinition.MountPointProperty(
                        container_path="containerPath",
                        read_only=False,
                        source_volume="sourceVolume"
                    )],
                    name="name",
                    port_mappings=[ecs.CfnTaskDefinition.PortMappingProperty(
                        container_port=123,
                        host_port=123,
                        protocol="protocol"
                    )],
                    privileged=False,
                    pseudo_terminal=False,
                    readonly_root_filesystem=False,
                    repository_credentials=ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                        credentials_parameter="credentialsParameter"
                    ),
                    resource_requirements=[ecs.CfnTaskDefinition.ResourceRequirementProperty(
                        type="type",
                        value="value"
                    )],
                    secrets=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )],
                    start_timeout=123,
                    stop_timeout=123,
                    system_controls=[ecs.CfnTaskDefinition.SystemControlProperty(
                        namespace="namespace",
                        value="value"
                    )],
                    ulimits=[ecs.CfnTaskDefinition.UlimitProperty(
                        hard_limit=123,
                        name="name",
                        soft_limit=123
                    )],
                    user="user",
                    volumes_from=[ecs.CfnTaskDefinition.VolumeFromProperty(
                        read_only=False,
                        source_container="sourceContainer"
                    )],
                    working_directory="workingDirectory"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if cpu is not None:
                self._values["cpu"] = cpu
            if depends_on is not None:
                self._values["depends_on"] = depends_on
            if disable_networking is not None:
                self._values["disable_networking"] = disable_networking
            if dns_search_domains is not None:
                self._values["dns_search_domains"] = dns_search_domains
            if dns_servers is not None:
                self._values["dns_servers"] = dns_servers
            if docker_labels is not None:
                self._values["docker_labels"] = docker_labels
            if docker_security_options is not None:
                self._values["docker_security_options"] = docker_security_options
            if entry_point is not None:
                self._values["entry_point"] = entry_point
            if environment is not None:
                self._values["environment"] = environment
            if environment_files is not None:
                self._values["environment_files"] = environment_files
            if essential is not None:
                self._values["essential"] = essential
            if extra_hosts is not None:
                self._values["extra_hosts"] = extra_hosts
            if firelens_configuration is not None:
                self._values["firelens_configuration"] = firelens_configuration
            if health_check is not None:
                self._values["health_check"] = health_check
            if hostname is not None:
                self._values["hostname"] = hostname
            if image is not None:
                self._values["image"] = image
            if interactive is not None:
                self._values["interactive"] = interactive
            if links is not None:
                self._values["links"] = links
            if linux_parameters is not None:
                self._values["linux_parameters"] = linux_parameters
            if log_configuration is not None:
                self._values["log_configuration"] = log_configuration
            if memory is not None:
                self._values["memory"] = memory
            if memory_reservation is not None:
                self._values["memory_reservation"] = memory_reservation
            if mount_points is not None:
                self._values["mount_points"] = mount_points
            if name is not None:
                self._values["name"] = name
            if port_mappings is not None:
                self._values["port_mappings"] = port_mappings
            if privileged is not None:
                self._values["privileged"] = privileged
            if pseudo_terminal is not None:
                self._values["pseudo_terminal"] = pseudo_terminal
            if readonly_root_filesystem is not None:
                self._values["readonly_root_filesystem"] = readonly_root_filesystem
            if repository_credentials is not None:
                self._values["repository_credentials"] = repository_credentials
            if resource_requirements is not None:
                self._values["resource_requirements"] = resource_requirements
            if secrets is not None:
                self._values["secrets"] = secrets
            if start_timeout is not None:
                self._values["start_timeout"] = start_timeout
            if stop_timeout is not None:
                self._values["stop_timeout"] = stop_timeout
            if system_controls is not None:
                self._values["system_controls"] = system_controls
            if ulimits is not None:
                self._values["ulimits"] = ulimits
            if user is not None:
                self._values["user"] = user
            if volumes_from is not None:
                self._values["volumes_from"] = volumes_from
            if working_directory is not None:
                self._values["working_directory"] = working_directory

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The command that's passed to the container.

            This parameter maps to ``Cmd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``COMMAND`` parameter to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#cmd <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#cmd>`_ . If there are multiple arguments, each argument is a separated string in the array.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cpu(self) -> typing.Optional[jsii.Number]:
            '''The number of ``cpu`` units reserved for the container.

            This parameter maps to ``CpuShares`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cpu-shares`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            This field is optional for tasks using the Fargate launch type, and the only requirement is that the total amount of CPU reserved for all containers within a task be lower than the task-level ``cpu`` value.
            .. epigraph::

               You can determine the number of CPU units that are available per EC2 instance type by multiplying the vCPUs listed for that instance type on the `Amazon EC2 Instances <https://docs.aws.amazon.com/ec2/instance-types/>`_ detail page by 1,024.

            Linux containers share unallocated CPU units with other containers on the container instance with the same ratio as their allocated amount. For example, if you run a single-container task on a single-core instance type with 512 CPU units specified for that container, and that's the only task running on the container instance, that container could use the full 1,024 CPU unit share at any given time. However, if you launched another copy of the same task on that container instance, each task is guaranteed a minimum of 512 CPU units when needed. Moreover, each container could float to higher CPU usage if the other container was not using it. If both tasks were 100% active all of the time, they would be limited to 512 CPU units.

            On Linux container instances, the Docker daemon on the container instance uses the CPU value to calculate the relative CPU share ratios for running containers. For more information, see `CPU share constraint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#cpu-share-constraint>`_ in the Docker documentation. The minimum valid CPU share value that the Linux kernel allows is 2. However, the CPU parameter isn't required, and you can use CPU values below 2 in your container definitions. For CPU values below 2 (including null), the behavior varies based on your Amazon ECS container agent version:

            - *Agent versions less than or equal to 1.1.0:* Null and zero CPU values are passed to Docker as 0, which Docker then converts to 1,024 CPU shares. CPU values of 1 are passed to Docker as 1, which the Linux kernel converts to two CPU shares.
            - *Agent versions greater than or equal to 1.2.0:* Null, zero, and CPU values of 1 are passed to Docker as 2.

            On Windows container instances, the CPU limit is enforced as an absolute limit, or a quota. Windows containers only have access to the specified amount of CPU that's described in the task definition. A null or zero CPU value is passed to Docker as ``0`` , which Windows interprets as 1% of one CPU.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-cpu
            '''
            result = self._values.get("cpu")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def depends_on(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDependencyProperty", _IResolvable_a771d0ef]]]]:
            '''The dependencies defined for container startup and shutdown.

            A container can contain multiple dependencies. When a dependency is defined for container startup, for container shutdown it is reversed.

            For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent to turn on container dependencies. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For tasks using the Fargate launch type, the task or service requires the following platforms:

            - Linux platform version ``1.3.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dependson
            '''
            result = self._values.get("depends_on")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ContainerDependencyProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def disable_networking(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''When this parameter is true, networking is disabled within the container.

            This parameter maps to ``NetworkDisabled`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-disablenetworking
            '''
            result = self._values.get("disable_networking")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of DNS search domains that are presented to the container.

            This parameter maps to ``DnsSearch`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns-search`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dnssearchdomains
            '''
            result = self._values.get("dns_search_domains")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of DNS servers that are presented to the container.

            This parameter maps to ``Dns`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--dns`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dnsservers
            '''
            result = self._values.get("dns_servers")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def docker_labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''A key/value map of labels to add to the container.

            This parameter maps to ``Labels`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--label`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dockerlabels
            '''
            result = self._values.get("docker_labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems.

            This field isn't valid for containers in tasks using the Fargate launch type.

            With Windows containers, this parameter can be used to reference a credential spec file when configuring a container for Active Directory authentication. For more information, see `Using gMSAs for Windows Containers <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows-gmsa.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            This parameter maps to ``SecurityOpt`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--security-opt`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               The Amazon ECS container agent running on a container instance must register with the ``ECS_SELINUX_CAPABLE=true`` or ``ECS_APPARMOR_CAPABLE=true`` environment variables before containers placed on that instance can use these security options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For more information about valid values, see `Docker Run Security Configuration <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Valid values: "no-new-privileges" | "apparmor:PROFILE" | "label:value" | "credentialspec:CredentialSpecFilePath"

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-dockersecurityoptions
            '''
            result = self._values.get("docker_security_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
            '''.. epigraph::

   Early versions of the Amazon ECS container agent don't properly handle ``entryPoint`` parameters.

            If you have problems using ``entryPoint`` , update your container agent or enter your commands and arguments as ``command`` array items instead.

            The entry point that's passed to the container. This parameter maps to ``Entrypoint`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--entrypoint`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . For more information, see `https://docs.docker.com/engine/reference/builder/#entrypoint <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/builder/#entrypoint>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-entrypoint
            '''
            result = self._values.get("entry_point")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def environment(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]]:
            '''The environment variables to pass to a container.

            This parameter maps to ``Env`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--env`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               We don't recommend that you use plaintext environment variables for sensitive information, such as credential data.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-environment
            '''
            result = self._values.get("environment")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def environment_files(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.EnvironmentFileProperty", _IResolvable_a771d0ef]]]]:
            '''A list of files containing the environment variables to pass to a container.

            This parameter maps to the ``--env-file`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            You can specify up to ten environment files. The file must have a ``.env`` file extension. Each line in an environment file contains an environment variable in ``VARIABLE=VALUE`` format. Lines beginning with ``#`` are treated as comments and are ignored. For more information about the environment variable file syntax, see `Declare default environment variables in file <https://docs.aws.amazon.com/https://docs.docker.com/compose/env-file/>`_ .

            If there are environment variables specified using the ``environment`` parameter in a container definition, they take precedence over the variables contained within an environment file. If multiple environment files are specified that contain the same variable, they're processed from the top down. We recommend that you use unique variable names. For more information, see `Specifying Environment Variables <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-environmentfiles
            '''
            result = self._values.get("environment_files")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.EnvironmentFileProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def essential(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''If the ``essential`` parameter of a container is marked as ``true`` , and that container fails or stops for any reason, all other containers that are part of the task are stopped.

            If the ``essential`` parameter of a container is marked as ``false`` , its failure doesn't affect the rest of the containers in a task. If this parameter is omitted, a container is assumed to be essential.

            All tasks must have at least one essential container. If you have an application that's composed of multiple containers, group containers that are used for a common purpose into components, and separate the different components into multiple task definitions. For more information, see `Application Architecture <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/application_architecture.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-essential
            '''
            result = self._values.get("essential")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def extra_hosts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.HostEntryProperty", _IResolvable_a771d0ef]]]]:
            '''A list of hostnames and IP address mappings to append to the ``/etc/hosts`` file on the container.

            This parameter maps to ``ExtraHosts`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--add-host`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter isn't supported for Windows containers or tasks that use the ``awsvpc`` network mode.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-extrahosts
            '''
            result = self._values.get("extra_hosts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.HostEntryProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def firelens_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", _IResolvable_a771d0ef]]:
            '''The FireLens configuration for the container.

            This is used to specify and configure a log router for container logs. For more information, see `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-firelensconfiguration
            '''
            result = self._values.get("firelens_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.FirelensConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.HealthCheckProperty", _IResolvable_a771d0ef]]:
            '''The container health check command and associated configuration parameters for the container.

            This parameter maps to ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``HEALTHCHECK`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.HealthCheckProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def hostname(self) -> typing.Optional[builtins.str]:
            '''The hostname to use for your container.

            This parameter maps to ``Hostname`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--hostname`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               The ``hostname`` parameter is not supported if you're using the ``awsvpc`` network mode.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def image(self) -> typing.Optional[builtins.str]:
            '''The image used to start a container.

            This string is passed directly to the Docker daemon. By default, images in the Docker Hub registry are available. Other repositories are specified with either ``*repository-url* / *image* : *tag*`` or ``*repository-url* / *image* @ *digest*`` . Up to 255 letters (uppercase and lowercase), numbers, hyphens, underscores, colons, periods, forward slashes, and number signs are allowed. This parameter maps to ``Image`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``IMAGE`` parameter of `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            - When a new task starts, the Amazon ECS container agent pulls the latest version of the specified image and tag for the container to use. However, subsequent updates to a repository image aren't propagated to already running tasks.
            - Images in Amazon ECR repositories can be specified by either using the full ``registry/repository:tag`` or ``registry/repository@digest`` . For example, ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>:latest`` or ``012345678910.dkr.ecr.<region-name>.amazonaws.com/<repository-name>@sha256:94afd1f2e64d908bc90dbca0035a5b567EXAMPLE`` .
            - Images in official repositories on Docker Hub use a single name (for example, ``ubuntu`` or ``mongo`` ).
            - Images in other repositories on Docker Hub are qualified with an organization name (for example, ``amazon/amazon-ecs-agent`` ).
            - Images in other online repositories are qualified further by a domain name (for example, ``quay.io/assemblyline/ubuntu`` ).

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-image
            '''
            result = self._values.get("image")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def interactive(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''When this parameter is ``true`` , you can deploy containerized applications that require ``stdin`` or a ``tty`` to be allocated.

            This parameter maps to ``OpenStdin`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--interactive`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-interactive
            '''
            result = self._values.get("interactive")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def links(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The ``links`` parameter allows containers to communicate with each other without the need for port mappings.

            This parameter is only supported if the network mode of a task definition is ``bridge`` . The ``name:internalName`` construct is analogous to ``name:alias`` in Docker links. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. For more information about linking Docker containers, go to `Legacy container links <https://docs.aws.amazon.com/https://docs.docker.com/network/links/>`_ in the Docker documentation. This parameter maps to ``Links`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--link`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers. > Containers that are collocated on a single container instance may be able to communicate with each other without requiring links or host port mappings. Network isolation is achieved on the container instance using security groups and VPC settings.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-links
            '''
            result = self._values.get("links")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def linux_parameters(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.LinuxParametersProperty", _IResolvable_a771d0ef]]:
            '''Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ .

            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-linuxparameters
            '''
            result = self._values.get("linux_parameters")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.LinuxParametersProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def log_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.LogConfigurationProperty", _IResolvable_a771d0ef]]:
            '''The log configuration specification for the container.

            This parameter maps to ``LogConfig`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--log-driver`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . By default, containers use the same logging driver that the Docker daemon uses. However, the container may use a different logging driver than the Docker daemon by specifying a log driver with this parameter in the container definition. To use a different logging driver for a container, the log system must be configured properly on the container instance (or on a different log server for remote logging options). For more information on the options for different supported log drivers, see `Configure logging drivers <https://docs.aws.amazon.com/https://docs.docker.com/engine/admin/logging/overview/>`_ in the Docker documentation.
            .. epigraph::

               Amazon ECS currently supports a subset of the logging drivers available to the Docker daemon (shown in the `LogConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_LogConfiguration.html>`_ data type). Additional log drivers may be available in future releases of the Amazon ECS container agent.

            This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            .. epigraph::

               The Amazon ECS container agent running on a container instance must register the logging drivers available on that instance with the ``ECS_AVAILABLE_LOGGING_DRIVERS`` environment variable before containers placed on that instance can use these log configuration options. For more information, see `Amazon ECS Container Agent Configuration <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-config.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration
            '''
            result = self._values.get("log_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.LogConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def memory(self) -> typing.Optional[jsii.Number]:
            '''The amount (in MiB) of memory to present to the container.

            If your container attempts to exceed the memory specified here, the container is killed. The total amount of memory reserved for all containers within a task must be lower than the task ``memory`` value, if one is specified. This parameter maps to ``Memory`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            If using the Fargate launch type, this parameter is optional.

            If using the EC2 launch type, you must specify either a task-level memory value or a container-level memory value. If you specify both a container-level ``memory`` and ``memoryReservation`` value, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used.

            The Docker 20.10.0 or later daemon reserves a minimum of 6 MiB of memory for a container, so you should not specify fewer than 6 MiB of memory for your containers.

            The Docker 19.03.13-ce or earlier daemon reserves a minimum of 4 MiB of memory for a container, so you should not specify fewer than 4 MiB of memory for your containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-memory
            '''
            result = self._values.get("memory")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def memory_reservation(self) -> typing.Optional[jsii.Number]:
            '''The soft limit (in MiB) of memory to reserve for the container.

            When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the ``memory`` parameter (if applicable), or all of the available memory on the container instance, whichever comes first. This parameter maps to ``MemoryReservation`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--memory-reservation`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            If a task-level memory value is not specified, you must specify a non-zero integer for one or both of ``memory`` or ``memoryReservation`` in a container definition. If you specify both, ``memory`` must be greater than ``memoryReservation`` . If you specify ``memoryReservation`` , then that value is subtracted from the available memory resources for the container instance where the container is placed. Otherwise, the value of ``memory`` is used.

            For example, if your container normally uses 128 MiB of memory, but occasionally bursts to 256 MiB of memory for short periods of time, you can set a ``memoryReservation`` of 128 MiB, and a ``memory`` hard limit of 300 MiB. This configuration would allow the container to only reserve 128 MiB of memory from the remaining resources on the container instance, but also allow the container to consume more memory resources when needed.

            The Docker daemon reserves a minimum of 4 MiB of memory for a container. Therefore, we recommend that you specify fewer than 4 MiB of memory for your containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-memoryreservation
            '''
            result = self._values.get("memory_reservation")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def mount_points(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.MountPointProperty", _IResolvable_a771d0ef]]]]:
            '''The mount points for data volumes in your container.

            This parameter maps to ``Volumes`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volume`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints
            '''
            result = self._values.get("mount_points")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.MountPointProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of a container.

            If you're linking multiple containers together in a task definition, the ``name`` of one container can be entered in the ``links`` of another container to connect the containers. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This parameter maps to ``name`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--name`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port_mappings(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.PortMappingProperty", _IResolvable_a771d0ef]]]]:
            '''The list of port mappings for the container.

            Port mappings allow containers to access ports on the host container instance to send or receive traffic.

            For task definitions that use the ``awsvpc`` network mode, you should only specify the ``containerPort`` . The ``hostPort`` can be left blank or it must be the same value as the ``containerPort`` .

            Port mappings on Windows use the ``NetNAT`` gateway address rather than ``localhost`` . There is no loopback for port mappings on Windows, so you cannot access a container's mapped port from the host itself.

            This parameter maps to ``PortBindings`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--publish`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . If the network mode of a task definition is set to ``none`` , then you can't specify port mappings. If the network mode of a task definition is set to ``host`` , then host ports must either be undefined or they must match the container port in the port mapping.
            .. epigraph::

               After a task reaches the ``RUNNING`` status, manual and automatic host and container port assignments are visible in the *Network Bindings* section of a container description for a selected task in the Amazon ECS console. The assignments are also visible in the ``networkBindings`` section `DescribeTasks <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ responses.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-portmappings
            '''
            result = self._values.get("port_mappings")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.PortMappingProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def privileged(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''When this parameter is true, the container is given elevated privileges on the host container instance (similar to the ``root`` user).

            This parameter maps to ``Privileged`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--privileged`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers or tasks run on AWS Fargate .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-privileged
            '''
            result = self._values.get("privileged")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def pseudo_terminal(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''When this parameter is ``true`` , a TTY is allocated.

            This parameter maps to ``Tty`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--tty`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-pseudoterminal
            '''
            result = self._values.get("pseudo_terminal")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def readonly_root_filesystem(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''When this parameter is true, the container is given read-only access to its root file system.

            This parameter maps to ``ReadonlyRootfs`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--read-only`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-readonlyrootfilesystem
            '''
            result = self._values.get("readonly_root_filesystem")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def repository_credentials(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", _IResolvable_a771d0ef]]:
            '''The private repository authentication credentials to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-repositorycredentials
            '''
            result = self._values.get("repository_credentials")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.RepositoryCredentialsProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def resource_requirements(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ResourceRequirementProperty", _IResolvable_a771d0ef]]]]:
            '''The type and amount of a resource to assign to a container.

            The only supported resource is a GPU.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-resourcerequirements
            '''
            result = self._values.get("resource_requirements")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.ResourceRequirementProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def secrets(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]]:
            '''The secrets to pass to the container.

            For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-secrets
            '''
            result = self._values.get("secrets")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def start_timeout(self) -> typing.Optional[jsii.Number]:
            '''Time duration (in seconds) to wait before giving up on resolving dependencies for a container.

            For example, you specify two containers in a task definition with containerA having a dependency on containerB reaching a ``COMPLETE`` , ``SUCCESS`` , or ``HEALTHY`` status. If a ``startTimeout`` value is specified for containerB and it doesn't reach the desired status within that time then containerA gives up and not start. This results in the task transitioning to a ``STOPPED`` state.
            .. epigraph::

               When the ``ECS_CONTAINER_START_TIMEOUT`` container agent configuration variable is used, it's enforced independently from this start timeout value.

            For tasks using the Fargate launch type, the task or service requires the following platforms:

            - Linux platform version ``1.3.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            For tasks using the EC2 launch type, your container instances require at least version ``1.26.0`` of the container agent to use a container start timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version ``1.26.0-1`` of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-starttimeout
            '''
            result = self._values.get("start_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def stop_timeout(self) -> typing.Optional[jsii.Number]:
            '''Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own.

            For tasks using the Fargate launch type, the task or service requires the following platforms:

            - Linux platform version ``1.3.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            The max stop timeout value is 120 seconds and if the parameter is not specified, the default value of 30 seconds is used.

            For tasks that use the EC2 launch type, if the ``stopTimeout`` parameter isn't specified, the value set for the Amazon ECS container agent configuration variable ``ECS_CONTAINER_STOP_TIMEOUT`` is used. If neither the ``stopTimeout`` parameter or the ``ECS_CONTAINER_STOP_TIMEOUT`` agent configuration variable are set, then the default values of 30 seconds for Linux containers and 30 seconds on Windows containers are used. Your container instances require at least version 1.26.0 of the container agent to use a container stop timeout value. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you're using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-stoptimeout
            '''
            result = self._values.get("stop_timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def system_controls(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SystemControlProperty", _IResolvable_a771d0ef]]]]:
            '''A list of namespaced kernel parameters to set in the container.

            This parameter maps to ``Sysctls`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--sysctl`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               We don't recommended that you specify network-related ``systemControls`` parameters for multiple containers in a single task that also uses either the ``awsvpc`` or ``host`` network modes. For tasks that use the ``awsvpc`` network mode, the container that's started last determines which ``systemControls`` parameters take effect. For tasks that use the ``host`` network mode, it changes the container instance's namespaced kernel parameters as well as the containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-systemcontrols
            '''
            result = self._values.get("system_controls")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SystemControlProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def ulimits(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.UlimitProperty", _IResolvable_a771d0ef]]]]:
            '''A list of ``ulimits`` to set in the container.

            This parameter maps to ``Ulimits`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--ulimit`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/>`_ . Valid naming values are displayed in the `Ulimit <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_Ulimit.html>`_ data type. This parameter requires version 1.18 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-ulimits
            '''
            result = self._values.get("ulimits")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.UlimitProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def user(self) -> typing.Optional[builtins.str]:
            '''The user to use inside the container.

            This parameter maps to ``User`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--user`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               When running tasks using the ``host`` network mode, don't run containers using the root user (UID 0). We recommend using a non-root user for better security.

            You can specify the ``user`` using the following formats. If specifying a UID or GID, you must specify it as a positive integer.

            - ``user``
            - ``user:group``
            - ``uid``
            - ``uid:gid``
            - ``user:gid``
            - ``uid:group``

            .. epigraph::

               This parameter is not supported for Windows containers.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-user
            '''
            result = self._values.get("user")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volumes_from(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeFromProperty", _IResolvable_a771d0ef]]]]:
            '''Data volumes to mount from another container.

            This parameter maps to ``VolumesFrom`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--volumes-from`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom
            '''
            result = self._values.get("volumes_from")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.VolumeFromProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def working_directory(self) -> typing.Optional[builtins.str]:
            '''The working directory to run commands inside the container in.

            This parameter maps to ``WorkingDir`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--workdir`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions.html#cfn-ecs-taskdefinition-containerdefinition-workingdirectory
            '''
            result = self._values.get("working_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerDefinitionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ContainerDependencyProperty",
        jsii_struct_bases=[],
        name_mapping={"condition": "condition", "container_name": "containerName"},
    )
    class ContainerDependencyProperty:
        def __init__(
            self,
            *,
            condition: typing.Optional[builtins.str] = None,
            container_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ContainerDependency`` property specifies the dependencies defined for container startup and shutdown.

            A container can contain multiple dependencies. When a dependency is defined for container startup, for container shutdown it is reversed.

            Your Amazon ECS container instances require at least version 1.26.0 of the container agent to enable container dependencies. However, we recommend using the latest container agent version. For information about checking your agent version and updating to the latest version, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If you are using an Amazon ECS-optimized Linux AMI, your instance needs at least version 1.26.0-1 of the ``ecs-init`` package. If your container instances are launched from version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               For tasks using the Fargate launch type, this parameter requires that the task or service uses platform version 1.3.0 or later.

            :param condition: The dependency condition of the container. The following are the available conditions and their behavior:. - ``START`` - This condition emulates the behavior of links and volumes today. It validates that a dependent container is started before permitting other containers to start. - ``COMPLETE`` - This condition validates that a dependent container runs to completion (exits) before permitting other containers to start. This can be useful for nonessential containers that run a script and then exit. This condition can't be set on an essential container. - ``SUCCESS`` - This condition is the same as ``COMPLETE`` , but it also requires that the container exits with a ``zero`` status. This condition can't be set on an essential container. - ``HEALTHY`` - This condition validates that the dependent container passes its Docker health check before permitting other containers to start. This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.
            :param container_name: The name of a container.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                container_dependency_property = ecs.CfnTaskDefinition.ContainerDependencyProperty(
                    condition="condition",
                    container_name="containerName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if condition is not None:
                self._values["condition"] = condition
            if container_name is not None:
                self._values["container_name"] = container_name

        @builtins.property
        def condition(self) -> typing.Optional[builtins.str]:
            '''The dependency condition of the container. The following are the available conditions and their behavior:.

            - ``START`` - This condition emulates the behavior of links and volumes today. It validates that a dependent container is started before permitting other containers to start.
            - ``COMPLETE`` - This condition validates that a dependent container runs to completion (exits) before permitting other containers to start. This can be useful for nonessential containers that run a script and then exit. This condition can't be set on an essential container.
            - ``SUCCESS`` - This condition is the same as ``COMPLETE`` , but it also requires that the container exits with a ``zero`` status. This condition can't be set on an essential container.
            - ``HEALTHY`` - This condition validates that the dependent container passes its Docker health check before permitting other containers to start. This requires that the dependent container has health checks configured. This condition is confirmed only at task startup.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html#cfn-ecs-taskdefinition-containerdependency-condition
            '''
            result = self._values.get("condition")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The name of a container.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdependency.html#cfn-ecs-taskdefinition-containerdependency-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ContainerDependencyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.DeviceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "host_path": "hostPath",
            "permissions": "permissions",
        },
    )
    class DeviceProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            host_path: typing.Optional[builtins.str] = None,
            permissions: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``Device`` property specifies an object representing a container instance host device.

            :param container_path: The path inside the container at which to expose the host device.
            :param host_path: The path for the device on the host container instance.
            :param permissions: The explicit permissions to provide to the container for the device. By default, the container has permissions for ``read`` , ``write`` , and ``mknod`` for the device.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                device_property = ecs.CfnTaskDefinition.DeviceProperty(
                    container_path="containerPath",
                    host_path="hostPath",
                    permissions=["permissions"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if host_path is not None:
                self._values["host_path"] = host_path
            if permissions is not None:
                self._values["permissions"] = permissions

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The path inside the container at which to expose the host device.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_path(self) -> typing.Optional[builtins.str]:
            '''The path for the device on the host container instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-hostpath
            '''
            result = self._values.get("host_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def permissions(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The explicit permissions to provide to the container for the device.

            By default, the container has permissions for ``read`` , ``write`` , and ``mknod`` for the device.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-device.html#cfn-ecs-taskdefinition-device-permissions
            '''
            result = self._values.get("permissions")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "autoprovision": "autoprovision",
            "driver": "driver",
            "driver_opts": "driverOpts",
            "labels": "labels",
            "scope": "scope",
        },
    )
    class DockerVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            autoprovision: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            driver: typing.Optional[builtins.str] = None,
            driver_opts: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            labels: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            scope: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``DockerVolumeConfiguration`` property specifies a Docker volume configuration and is used when you use Docker volumes.

            Docker volumes are only supported when you are using the EC2 launch type. Windows containers only support the use of the ``local`` driver. To use bind mounts, specify a ``host`` instead.

            :param autoprovision: If this value is ``true`` , the Docker volume is created if it doesn't already exist. .. epigraph:: This field is only used if the ``scope`` is ``shared`` .
            :param driver: The Docker volume driver to use. The driver value must match the driver name provided by Docker because it is used for task placement. If the driver was installed using the Docker plugin CLI, use ``docker plugin ls`` to retrieve the driver name from your container instance. If the driver was installed using another method, use Docker plugin discovery to retrieve the driver name. For more information, see `Docker plugin discovery <https://docs.aws.amazon.com/https://docs.docker.com/engine/extend/plugin_api/#plugin-discovery>`_ . This parameter maps to ``Driver`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxdriver`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .
            :param driver_opts: A map of Docker driver-specific options passed through. This parameter maps to ``DriverOpts`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxopt`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .
            :param labels: Custom metadata to add to your Docker volume. This parameter maps to ``Labels`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxlabel`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .
            :param scope: The scope for the Docker volume that determines its lifecycle. Docker volumes that are scoped to a ``task`` are automatically provisioned when the task starts and destroyed when the task stops. Docker volumes that are scoped as ``shared`` persist after the task stops.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                docker_volume_configuration_property = ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                    autoprovision=False,
                    driver="driver",
                    driver_opts={
                        "driver_opts_key": "driverOpts"
                    },
                    labels={
                        "labels_key": "labels"
                    },
                    scope="scope"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if autoprovision is not None:
                self._values["autoprovision"] = autoprovision
            if driver is not None:
                self._values["driver"] = driver
            if driver_opts is not None:
                self._values["driver_opts"] = driver_opts
            if labels is not None:
                self._values["labels"] = labels
            if scope is not None:
                self._values["scope"] = scope

        @builtins.property
        def autoprovision(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''If this value is ``true`` , the Docker volume is created if it doesn't already exist.

            .. epigraph::

               This field is only used if the ``scope`` is ``shared`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-autoprovision
            '''
            result = self._values.get("autoprovision")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def driver(self) -> typing.Optional[builtins.str]:
            '''The Docker volume driver to use.

            The driver value must match the driver name provided by Docker because it is used for task placement. If the driver was installed using the Docker plugin CLI, use ``docker plugin ls`` to retrieve the driver name from your container instance. If the driver was installed using another method, use Docker plugin discovery to retrieve the driver name. For more information, see `Docker plugin discovery <https://docs.aws.amazon.com/https://docs.docker.com/engine/extend/plugin_api/#plugin-discovery>`_ . This parameter maps to ``Driver`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxdriver`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-driver
            '''
            result = self._values.get("driver")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def driver_opts(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''A map of Docker driver-specific options passed through.

            This parameter maps to ``DriverOpts`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxopt`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-driveropts
            '''
            result = self._values.get("driver_opts")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def labels(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''Custom metadata to add to your Docker volume.

            This parameter maps to ``Labels`` in the `Create a volume <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/VolumeCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``xxlabel`` option to `docker volume create <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/commandline/volume_create/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-labels
            '''
            result = self._values.get("labels")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def scope(self) -> typing.Optional[builtins.str]:
            '''The scope for the Docker volume that determines its lifecycle.

            Docker volumes that are scoped to a ``task`` are automatically provisioned when the task starts and destroyed when the task stops. Docker volumes that are scoped as ``shared`` persist after the task stops.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-dockervolumeconfiguration.html#cfn-ecs-taskdefinition-dockervolumeconfiguration-scope
            '''
            result = self._values.get("scope")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DockerVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.EfsVolumeConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "file_system_id": "fileSystemId",
            "authorization_config": "authorizationConfig",
            "root_directory": "rootDirectory",
            "transit_encryption": "transitEncryption",
            "transit_encryption_port": "transitEncryptionPort",
        },
    )
    class EfsVolumeConfigurationProperty:
        def __init__(
            self,
            *,
            file_system_id: builtins.str,
            authorization_config: typing.Optional[typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", _IResolvable_a771d0ef]] = None,
            root_directory: typing.Optional[builtins.str] = None,
            transit_encryption: typing.Optional[builtins.str] = None,
            transit_encryption_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param file_system_id: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.FileSystemId``.
            :param authorization_config: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.AuthorizationConfig``.
            :param root_directory: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.RootDirectory``.
            :param transit_encryption: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryption``.
            :param transit_encryption_port: ``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryptionPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                efs_volume_configuration_property = ecs.CfnTaskDefinition.EfsVolumeConfigurationProperty(
                    file_system_id="fileSystemId",
                
                    # the properties below are optional
                    authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                        access_point_id="accessPointId",
                        iam="iam"
                    ),
                    root_directory="rootDirectory",
                    transit_encryption="transitEncryption",
                    transit_encryption_port=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "file_system_id": file_system_id,
            }
            if authorization_config is not None:
                self._values["authorization_config"] = authorization_config
            if root_directory is not None:
                self._values["root_directory"] = root_directory
            if transit_encryption is not None:
                self._values["transit_encryption"] = transit_encryption
            if transit_encryption_port is not None:
                self._values["transit_encryption_port"] = transit_encryption_port

        @builtins.property
        def file_system_id(self) -> builtins.str:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.FileSystemId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-filesystemid
            '''
            result = self._values.get("file_system_id")
            assert result is not None, "Required property 'file_system_id' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def authorization_config(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.AuthorizationConfig``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-authorizationconfig
            '''
            result = self._values.get("authorization_config")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.AuthorizationConfigProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def root_directory(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.RootDirectory``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-rootdirectory
            '''
            result = self._values.get("root_directory")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption(self) -> typing.Optional[builtins.str]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryption``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-transitencryption
            '''
            result = self._values.get("transit_encryption")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_encryption_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnTaskDefinition.EfsVolumeConfigurationProperty.TransitEncryptionPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-efsvolumeconfiguration.html#cfn-ecs-taskdefinition-efsvolumeconfiguration-transitencryptionport
            '''
            result = self._values.get("transit_encryption_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EfsVolumeConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.EnvironmentFileProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class EnvironmentFileProperty:
        def __init__(
            self,
            *,
            type: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A list of files containing the environment variables to pass to a container.

            You can specify up to ten environment files. The file must have a ``.env`` file extension. Each line in an environment file should contain an environment variable in ``VARIABLE=VALUE`` format. Lines beginning with ``#`` are treated as comments and are ignored. For more information about the environment variable file syntax, see `Declare default environment variables in file <https://docs.aws.amazon.com/https://docs.docker.com/compose/env-file/>`_ .

            If there are environment variables specified using the ``environment`` parameter in a container definition, they take precedence over the variables contained within an environment file. If multiple environment files are specified that contain the same variable, they're processed from the top down. We recommend that you use unique variable names. For more information, see `Specifying environment variables <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/taskdef-envfiles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            This parameter is only supported for tasks hosted on Fargate using the following platform versions:

            - Linux platform version ``1.4.0`` or later.
            - Windows platform version ``1.0.0`` or later.

            :param type: The file type to use. The only supported value is ``s3`` .
            :param value: The Amazon Resource Name (ARN) of the Amazon S3 object containing the environment variable file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                environment_file_property = ecs.CfnTaskDefinition.EnvironmentFileProperty(
                    type="type",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The file type to use.

            The only supported value is ``s3`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html#cfn-ecs-taskdefinition-environmentfile-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the Amazon S3 object containing the environment variable file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-environmentfile.html#cfn-ecs-taskdefinition-environmentfile-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnvironmentFileProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.EphemeralStorageProperty",
        jsii_struct_bases=[],
        name_mapping={"size_in_gib": "sizeInGiB"},
    )
    class EphemeralStorageProperty:
        def __init__(self, *, size_in_gib: typing.Optional[jsii.Number] = None) -> None:
            '''The amount of ephemeral storage to allocate for the task.

            This parameter is used to expand the total amount of ephemeral storage available, beyond the default amount, for tasks hosted on AWS Fargate . For more information, see `Fargate task storage <https://docs.aws.amazon.com/AmazonECS/latest/userguide/using_data_volumes.html>`_ in the *Amazon ECS User Guide for AWS Fargate* .
            .. epigraph::

               This parameter is only supported for tasks hosted on Fargate using Linux platform version ``1.4.0`` or later. This parameter is not supported for Windows containers on Fargate.

            :param size_in_gib: The total amount, in GiB, of ephemeral storage to set for the task. The minimum supported value is ``21`` GiB and the maximum supported value is ``200`` GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ephemeralstorage.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                ephemeral_storage_property = ecs.CfnTaskDefinition.EphemeralStorageProperty(
                    size_in_gi_b=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if size_in_gib is not None:
                self._values["size_in_gib"] = size_in_gib

        @builtins.property
        def size_in_gib(self) -> typing.Optional[jsii.Number]:
            '''The total amount, in GiB, of ephemeral storage to set for the task.

            The minimum supported value is ``21`` GiB and the maximum supported value is ``200`` GiB.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-ephemeralstorage.html#cfn-ecs-taskdefinition-ephemeralstorage-sizeingib
            '''
            result = self._values.get("size_in_gib")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EphemeralStorageProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.FirelensConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"options": "options", "type": "type"},
    )
    class FirelensConfigurationProperty:
        def __init__(
            self,
            *,
            options: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The FireLens configuration for the container.

            This is used to specify and configure a log router for container logs. For more information, see `Custom Log Routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param options: The options to use when configuring the log router. This field is optional and can be used to add additional metadata, such as the task, task definition, cluster, and container instance details to the log event. If specified, valid option keys are: - ``enable-ecs-log-metadata`` , which can be ``true`` or ``false`` - ``config-file-type`` , which can be ``s3`` or ``file`` - ``config-file-value`` , which is either an S3 ARN or a file path
            :param type: The log router to use. The valid values are ``fluentd`` or ``fluentbit`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                firelens_configuration_property = ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                    options={
                        "options_key": "options"
                    },
                    type="type"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if options is not None:
                self._values["options"] = options
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''The options to use when configuring the log router.

            This field is optional and can be used to add additional metadata, such as the task, task definition, cluster, and container instance details to the log event.

            If specified, valid option keys are:

            - ``enable-ecs-log-metadata`` , which can be ``true`` or ``false``
            - ``config-file-type`` , which can be ``s3`` or ``file``
            - ``config-file-value`` , which is either an S3 ARN or a file path

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html#cfn-ecs-taskdefinition-firelensconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The log router to use.

            The valid values are ``fluentd`` or ``fluentbit`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-firelensconfiguration.html#cfn-ecs-taskdefinition-firelensconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FirelensConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.HealthCheckProperty",
        jsii_struct_bases=[],
        name_mapping={
            "command": "command",
            "interval": "interval",
            "retries": "retries",
            "start_period": "startPeriod",
            "timeout": "timeout",
        },
    )
    class HealthCheckProperty:
        def __init__(
            self,
            *,
            command: typing.Optional[typing.Sequence[builtins.str]] = None,
            interval: typing.Optional[jsii.Number] = None,
            retries: typing.Optional[jsii.Number] = None,
            start_period: typing.Optional[jsii.Number] = None,
            timeout: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''The ``HealthCheck`` property specifies an object representing a container health check.

            Health check parameters that are specified in a container definition override any Docker health checks that exist in the container image (such as those specified in a parent image or from the image's Dockerfile).

            The following are notes about container health check support:

            - Container health checks require version 1.17.0 or greater of the Amazon ECS container agent. For more information, see `Updating the Amazon ECS Container Agent <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-agent-update.html>`_ .
            - Container health checks are supported for Fargate tasks if you are using platform version 1.1.0 or greater. For more information, see `AWS Fargate Platform Versions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/platform_versions.html>`_ .
            - Container health checks are not supported for tasks that are part of a service that is configured to use a Classic Load Balancer.

            :param command: A string array representing the command that the container runs to determine if it is healthy. The string array must start with ``CMD`` to execute the command arguments directly, or ``CMD-SHELL`` to run the command with the container's default shell. When you use the AWS Management Console JSON panel, the AWS Command Line Interface , or the APIs, enclose the list of commands in brackets. ``[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]`` You don't need to include the brackets when you use the AWS Management Console. ``"CMD-SHELL", "curl -f http://localhost/ || exit 1"`` An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ .
            :param interval: The time period in seconds between each health check execution. You may specify between 5 and 300 seconds. The default value is 30 seconds.
            :param retries: The number of times to retry a failed health check before the container is considered unhealthy. You may specify between 1 and 10 retries. The default value is 3.
            :param start_period: The optional grace period to provide containers time to bootstrap before failed health checks count towards the maximum number of retries. You can specify between 0 and 300 seconds. By default, the ``startPeriod`` is disabled. .. epigraph:: If a health check succeeds within the ``startPeriod`` , then the container is considered healthy and any subsequent failures count toward the maximum number of retries.
            :param timeout: The time period in seconds to wait for a health check to succeed before it is considered a failure. You may specify between 2 and 60 seconds. The default value is 5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                health_check_property = ecs.CfnTaskDefinition.HealthCheckProperty(
                    command=["command"],
                    interval=123,
                    retries=123,
                    start_period=123,
                    timeout=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if command is not None:
                self._values["command"] = command
            if interval is not None:
                self._values["interval"] = interval
            if retries is not None:
                self._values["retries"] = retries
            if start_period is not None:
                self._values["start_period"] = start_period
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def command(self) -> typing.Optional[typing.List[builtins.str]]:
            '''A string array representing the command that the container runs to determine if it is healthy.

            The string array must start with ``CMD`` to execute the command arguments directly, or ``CMD-SHELL`` to run the command with the container's default shell.

            When you use the AWS Management Console JSON panel, the AWS Command Line Interface , or the APIs, enclose the list of commands in brackets.

            ``[ "CMD-SHELL", "curl -f http://localhost/ || exit 1" ]``

            You don't need to include the brackets when you use the AWS Management Console.

            ``"CMD-SHELL", "curl -f http://localhost/ || exit 1"``

            An exit code of 0 indicates success, and non-zero exit code indicates failure. For more information, see ``HealthCheck`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-command
            '''
            result = self._values.get("command")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def interval(self) -> typing.Optional[jsii.Number]:
            '''The time period in seconds between each health check execution.

            You may specify between 5 and 300 seconds. The default value is 30 seconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-interval
            '''
            result = self._values.get("interval")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def retries(self) -> typing.Optional[jsii.Number]:
            '''The number of times to retry a failed health check before the container is considered unhealthy.

            You may specify between 1 and 10 retries. The default value is 3.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-retries
            '''
            result = self._values.get("retries")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def start_period(self) -> typing.Optional[jsii.Number]:
            '''The optional grace period to provide containers time to bootstrap before failed health checks count towards the maximum number of retries.

            You can specify between 0 and 300 seconds. By default, the ``startPeriod`` is disabled.
            .. epigraph::

               If a health check succeeds within the ``startPeriod`` , then the container is considered healthy and any subsequent failures count toward the maximum number of retries.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-startperiod
            '''
            result = self._values.get("start_period")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def timeout(self) -> typing.Optional[jsii.Number]:
            '''The time period in seconds to wait for a health check to succeed before it is considered a failure.

            You may specify between 2 and 60 seconds. The default value is 5.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-healthcheck.html#cfn-ecs-taskdefinition-healthcheck-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HealthCheckProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.HostEntryProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "ip_address": "ipAddress"},
    )
    class HostEntryProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[builtins.str] = None,
            ip_address: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``HostEntry`` property specifies a hostname and an IP address that are added to the ``/etc/hosts`` file of a container through the ``extraHosts`` parameter of its ``ContainerDefinition`` resource.

            :param hostname: The hostname to use in the ``/etc/hosts`` entry.
            :param ip_address: The IP address to use in the ``/etc/hosts`` entry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-hostentry.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                host_entry_property = ecs.CfnTaskDefinition.HostEntryProperty(
                    hostname="hostname",
                    ip_address="ipAddress"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if ip_address is not None:
                self._values["ip_address"] = ip_address

        @builtins.property
        def hostname(self) -> typing.Optional[builtins.str]:
            '''The hostname to use in the ``/etc/hosts`` entry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-hostentry.html#cfn-ecs-taskdefinition-containerdefinition-hostentry-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ip_address(self) -> typing.Optional[builtins.str]:
            '''The IP address to use in the ``/etc/hosts`` entry.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-hostentry.html#cfn-ecs-taskdefinition-containerdefinition-hostentry-ipaddress
            '''
            result = self._values.get("ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostEntryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.HostVolumePropertiesProperty",
        jsii_struct_bases=[],
        name_mapping={"source_path": "sourcePath"},
    )
    class HostVolumePropertiesProperty:
        def __init__(
            self,
            *,
            source_path: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``HostVolumeProperties`` property specifies details on a container instance bind mount host volume.

            :param source_path: When the ``host`` parameter is used, specify a ``sourcePath`` to declare the path on the host container instance that's presented to the container. If this parameter is empty, then the Docker daemon has assigned a host path for you. If the ``host`` parameter contains a ``sourcePath`` file location, then the data volume persists at the specified location on the host container instance until you delete it manually. If the ``sourcePath`` value doesn't exist on the host container instance, the Docker daemon creates it. If the location does exist, the contents of the source path folder are exported. If you're using the Fargate launch type, the ``sourcePath`` parameter is not supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes-host.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                host_volume_properties_property = ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                    source_path="sourcePath"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if source_path is not None:
                self._values["source_path"] = source_path

        @builtins.property
        def source_path(self) -> typing.Optional[builtins.str]:
            '''When the ``host`` parameter is used, specify a ``sourcePath`` to declare the path on the host container instance that's presented to the container.

            If this parameter is empty, then the Docker daemon has assigned a host path for you. If the ``host`` parameter contains a ``sourcePath`` file location, then the data volume persists at the specified location on the host container instance until you delete it manually. If the ``sourcePath`` value doesn't exist on the host container instance, the Docker daemon creates it. If the location does exist, the contents of the source path folder are exported.

            If you're using the Fargate launch type, the ``sourcePath`` parameter is not supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes-host.html#cfn-ecs-taskdefinition-volumes-host-sourcepath
            '''
            result = self._values.get("source_path")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HostVolumePropertiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.InferenceAcceleratorProperty",
        jsii_struct_bases=[],
        name_mapping={"device_name": "deviceName", "device_type": "deviceType"},
    )
    class InferenceAcceleratorProperty:
        def __init__(
            self,
            *,
            device_name: typing.Optional[builtins.str] = None,
            device_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Details on an Elastic Inference accelerator.

            For more information, see `Working with Amazon Elastic Inference on Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-eia.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param device_name: The Elastic Inference accelerator device name. The ``deviceName`` must also be referenced in a container definition as a `ResourceRequirement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html>`_ .
            :param device_type: The Elastic Inference accelerator type to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                inference_accelerator_property = ecs.CfnTaskDefinition.InferenceAcceleratorProperty(
                    device_name="deviceName",
                    device_type="deviceType"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if device_name is not None:
                self._values["device_name"] = device_name
            if device_type is not None:
                self._values["device_type"] = device_type

        @builtins.property
        def device_name(self) -> typing.Optional[builtins.str]:
            '''The Elastic Inference accelerator device name.

            The ``deviceName`` must also be referenced in a container definition as a `ResourceRequirement <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html#cfn-ecs-taskdefinition-inferenceaccelerator-devicename
            '''
            result = self._values.get("device_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def device_type(self) -> typing.Optional[builtins.str]:
            '''The Elastic Inference accelerator type to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html#cfn-ecs-taskdefinition-inferenceaccelerator-devicetype
            '''
            result = self._values.get("device_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InferenceAcceleratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.KernelCapabilitiesProperty",
        jsii_struct_bases=[],
        name_mapping={"add": "add", "drop": "drop"},
    )
    class KernelCapabilitiesProperty:
        def __init__(
            self,
            *,
            add: typing.Optional[typing.Sequence[builtins.str]] = None,
            drop: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``KernelCapabilities`` property specifies the Linux capabilities for the container that are added to or dropped from the default configuration that is provided by Docker.

            For more information on the default capabilities and the non-default available capabilities, see `Runtime privilege and Linux capabilities <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#runtime-privilege-and-linux-capabilities>`_ in the *Docker run reference* . For more detailed information on these Linux capabilities, see the `capabilities(7) <https://docs.aws.amazon.com/http://man7.org/linux/man-pages/man7/capabilities.7.html>`_ Linux manual page.

            :param add: The Linux capabilities for the container that have been added to the default configuration provided by Docker. This parameter maps to ``CapAdd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-add`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: Tasks launched on AWS Fargate only support adding the ``SYS_PTRACE`` kernel capability. Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``
            :param drop: The Linux capabilities for the container that have been removed from the default configuration provided by Docker. This parameter maps to ``CapDrop`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-drop`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                kernel_capabilities_property = ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                    add=["add"],
                    drop=["drop"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if add is not None:
                self._values["add"] = add
            if drop is not None:
                self._values["drop"] = drop

        @builtins.property
        def add(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The Linux capabilities for the container that have been added to the default configuration provided by Docker.

            This parameter maps to ``CapAdd`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-add`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               Tasks launched on AWS Fargate only support adding the ``SYS_PTRACE`` kernel capability.

            Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html#cfn-ecs-taskdefinition-kernelcapabilities-add
            '''
            result = self._values.get("add")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def drop(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The Linux capabilities for the container that have been removed from the default configuration provided by Docker.

            This parameter maps to ``CapDrop`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--cap-drop`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            Valid values: ``"ALL" | "AUDIT_CONTROL" | "AUDIT_WRITE" | "BLOCK_SUSPEND" | "CHOWN" | "DAC_OVERRIDE" | "DAC_READ_SEARCH" | "FOWNER" | "FSETID" | "IPC_LOCK" | "IPC_OWNER" | "KILL" | "LEASE" | "LINUX_IMMUTABLE" | "MAC_ADMIN" | "MAC_OVERRIDE" | "MKNOD" | "NET_ADMIN" | "NET_BIND_SERVICE" | "NET_BROADCAST" | "NET_RAW" | "SETFCAP" | "SETGID" | "SETPCAP" | "SETUID" | "SYS_ADMIN" | "SYS_BOOT" | "SYS_CHROOT" | "SYS_MODULE" | "SYS_NICE" | "SYS_PACCT" | "SYS_PTRACE" | "SYS_RAWIO" | "SYS_RESOURCE" | "SYS_TIME" | "SYS_TTY_CONFIG" | "SYSLOG" | "WAKE_ALARM"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-kernelcapabilities.html#cfn-ecs-taskdefinition-kernelcapabilities-drop
            '''
            result = self._values.get("drop")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KernelCapabilitiesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.KeyValuePairProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value": "value"},
    )
    class KeyValuePairProperty:
        def __init__(
            self,
            *,
            name: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``KeyValuePair`` property specifies a key-value pair object.

            :param name: The name of the key-value pair. For environment variables, this is the name of the environment variable.
            :param value: The value of the key-value pair. For environment variables, this is the value of the environment variable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-environment.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                key_value_pair_property = ecs.CfnTaskDefinition.KeyValuePairProperty(
                    name="name",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if name is not None:
                self._values["name"] = name
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the key-value pair.

            For environment variables, this is the name of the environment variable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-environment.html#cfn-ecs-taskdefinition-containerdefinition-environment-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value of the key-value pair.

            For environment variables, this is the value of the environment variable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-environment.html#cfn-ecs-taskdefinition-containerdefinition-environment-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "KeyValuePairProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.LinuxParametersProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capabilities": "capabilities",
            "devices": "devices",
            "init_process_enabled": "initProcessEnabled",
            "max_swap": "maxSwap",
            "shared_memory_size": "sharedMemorySize",
            "swappiness": "swappiness",
            "tmpfs": "tmpfs",
        },
    )
    class LinuxParametersProperty:
        def __init__(
            self,
            *,
            capabilities: typing.Optional[typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", _IResolvable_a771d0ef]] = None,
            devices: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.DeviceProperty", _IResolvable_a771d0ef]]]] = None,
            init_process_enabled: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            max_swap: typing.Optional[jsii.Number] = None,
            shared_memory_size: typing.Optional[jsii.Number] = None,
            swappiness: typing.Optional[jsii.Number] = None,
            tmpfs: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.TmpfsProperty", _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''The ``LinuxParameters`` property specifies Linux-specific options that are applied to the container, such as Linux `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_ .

            :param capabilities: The Linux capabilities for the container that are added to or dropped from the default configuration provided by Docker. .. epigraph:: For tasks that use the Fargate launch type, ``capabilities`` is supported for all platform versions but the ``add`` parameter is only supported if using platform version 1.4.0 or later.
            :param devices: Any host devices to expose to the container. This parameter maps to ``Devices`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--device`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you're using tasks that use the Fargate launch type, the ``devices`` parameter isn't supported.
            :param init_process_enabled: Run an ``init`` process inside the container that forwards signals and reaps processes. This parameter maps to the ``--init`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param max_swap: The total amount of swap memory (in MiB) a container can use. This parameter will be translated to the ``--memory-swap`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ where the value would be the sum of the container memory plus the ``maxSwap`` value. If a ``maxSwap`` value of ``0`` is specified, the container will not use swap. Accepted values are ``0`` or any positive integer. If the ``maxSwap`` parameter is omitted, the container will use the swap configuration for the container instance it is running on. A ``maxSwap`` value must be set for the ``swappiness`` parameter to be used. .. epigraph:: If you're using tasks that use the Fargate launch type, the ``maxSwap`` parameter isn't supported.
            :param shared_memory_size: The value for the size (in MiB) of the ``/dev/shm`` volume. This parameter maps to the ``--shm-size`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you are using tasks that use the Fargate launch type, the ``sharedMemorySize`` parameter is not supported.
            :param swappiness: This allows you to tune a container's memory swappiness behavior. A ``swappiness`` value of ``0`` will cause swapping to not happen unless absolutely necessary. A ``swappiness`` value of ``100`` will cause pages to be swapped very aggressively. Accepted values are whole numbers between ``0`` and ``100`` . If the ``swappiness`` parameter is not specified, a default value of ``60`` is used. If a value is not specified for ``maxSwap`` then this parameter is ignored. This parameter maps to the ``--memory-swappiness`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you're using tasks that use the Fargate launch type, the ``swappiness`` parameter isn't supported.
            :param tmpfs: The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the ``--tmpfs`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . .. epigraph:: If you're using tasks that use the Fargate launch type, the ``tmpfs`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                linux_parameters_property = ecs.CfnTaskDefinition.LinuxParametersProperty(
                    capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                        add=["add"],
                        drop=["drop"]
                    ),
                    devices=[ecs.CfnTaskDefinition.DeviceProperty(
                        container_path="containerPath",
                        host_path="hostPath",
                        permissions=["permissions"]
                    )],
                    init_process_enabled=False,
                    max_swap=123,
                    shared_memory_size=123,
                    swappiness=123,
                    tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                        size=123,
                
                        # the properties below are optional
                        container_path="containerPath",
                        mount_options=["mountOptions"]
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capabilities is not None:
                self._values["capabilities"] = capabilities
            if devices is not None:
                self._values["devices"] = devices
            if init_process_enabled is not None:
                self._values["init_process_enabled"] = init_process_enabled
            if max_swap is not None:
                self._values["max_swap"] = max_swap
            if shared_memory_size is not None:
                self._values["shared_memory_size"] = shared_memory_size
            if swappiness is not None:
                self._values["swappiness"] = swappiness
            if tmpfs is not None:
                self._values["tmpfs"] = tmpfs

        @builtins.property
        def capabilities(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", _IResolvable_a771d0ef]]:
            '''The Linux capabilities for the container that are added to or dropped from the default configuration provided by Docker.

            .. epigraph::

               For tasks that use the Fargate launch type, ``capabilities`` is supported for all platform versions but the ``add`` parameter is only supported if using platform version 1.4.0 or later.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-capabilities
            '''
            result = self._values.get("capabilities")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.KernelCapabilitiesProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def devices(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.DeviceProperty", _IResolvable_a771d0ef]]]]:
            '''Any host devices to expose to the container.

            This parameter maps to ``Devices`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--device`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``devices`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-devices
            '''
            result = self._values.get("devices")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.DeviceProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def init_process_enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Run an ``init`` process inside the container that forwards signals and reaps processes.

            This parameter maps to the ``--init`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ . This parameter requires version 1.25 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-initprocessenabled
            '''
            result = self._values.get("init_process_enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def max_swap(self) -> typing.Optional[jsii.Number]:
            '''The total amount of swap memory (in MiB) a container can use.

            This parameter will be translated to the ``--memory-swap`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ where the value would be the sum of the container memory plus the ``maxSwap`` value.

            If a ``maxSwap`` value of ``0`` is specified, the container will not use swap. Accepted values are ``0`` or any positive integer. If the ``maxSwap`` parameter is omitted, the container will use the swap configuration for the container instance it is running on. A ``maxSwap`` value must be set for the ``swappiness`` parameter to be used.
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``maxSwap`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-maxswap
            '''
            result = self._values.get("max_swap")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def shared_memory_size(self) -> typing.Optional[jsii.Number]:
            '''The value for the size (in MiB) of the ``/dev/shm`` volume.

            This parameter maps to the ``--shm-size`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you are using tasks that use the Fargate launch type, the ``sharedMemorySize`` parameter is not supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-sharedmemorysize
            '''
            result = self._values.get("shared_memory_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def swappiness(self) -> typing.Optional[jsii.Number]:
            '''This allows you to tune a container's memory swappiness behavior.

            A ``swappiness`` value of ``0`` will cause swapping to not happen unless absolutely necessary. A ``swappiness`` value of ``100`` will cause pages to be swapped very aggressively. Accepted values are whole numbers between ``0`` and ``100`` . If the ``swappiness`` parameter is not specified, a default value of ``60`` is used. If a value is not specified for ``maxSwap`` then this parameter is ignored. This parameter maps to the ``--memory-swappiness`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``swappiness`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-swappiness
            '''
            result = self._values.get("swappiness")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tmpfs(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TmpfsProperty", _IResolvable_a771d0ef]]]]:
            '''The container path, mount options, and size (in MiB) of the tmpfs mount.

            This parameter maps to the ``--tmpfs`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .
            .. epigraph::

               If you're using tasks that use the Fargate launch type, the ``tmpfs`` parameter isn't supported.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-linuxparameters.html#cfn-ecs-taskdefinition-linuxparameters-tmpfs
            '''
            result = self._values.get("tmpfs")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.TmpfsProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LinuxParametersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.LogConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "log_driver": "logDriver",
            "options": "options",
            "secret_options": "secretOptions",
        },
    )
    class LogConfigurationProperty:
        def __init__(
            self,
            *,
            log_driver: builtins.str,
            options: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]] = None,
            secret_options: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''The ``LogConfiguration`` property specifies log configuration options to send to a custom log driver for the container.

            :param log_driver: The log driver to use for the container. For tasks on AWS Fargate , the supported log drivers are ``awslogs`` , ``splunk`` , and ``awsfirelens`` . For tasks hosted on Amazon EC2 instances, the supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , ``splunk`` , and ``awsfirelens`` . For more information about using the ``awslogs`` log driver, see `Using the awslogs log driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For more information about using the ``awsfirelens`` log driver, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you would like to have included. However, we don't currently provide support for running modified copies of this software.
            :param options: The configuration options to send to the log driver. This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``
            :param secret_options: The secrets to pass to the log configuration. For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                log_configuration_property = ecs.CfnTaskDefinition.LogConfigurationProperty(
                    log_driver="logDriver",
                
                    # the properties below are optional
                    options={
                        "options_key": "options"
                    },
                    secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "log_driver": log_driver,
            }
            if options is not None:
                self._values["options"] = options
            if secret_options is not None:
                self._values["secret_options"] = secret_options

        @builtins.property
        def log_driver(self) -> builtins.str:
            '''The log driver to use for the container.

            For tasks on AWS Fargate , the supported log drivers are ``awslogs`` , ``splunk`` , and ``awsfirelens`` .

            For tasks hosted on Amazon EC2 instances, the supported log drivers are ``awslogs`` , ``fluentd`` , ``gelf`` , ``json-file`` , ``journald`` , ``logentries`` , ``syslog`` , ``splunk`` , and ``awsfirelens`` .

            For more information about using the ``awslogs`` log driver, see `Using the awslogs log driver <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_awslogs.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For more information about using the ``awsfirelens`` log driver, see `Custom log routing <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If you have a custom driver that isn't listed, you can fork the Amazon ECS container agent project that's `available on GitHub <https://docs.aws.amazon.com/https://github.com/aws/amazon-ecs-agent>`_ and customize it to work with that driver. We encourage you to submit pull requests for changes that you would like to have included. However, we don't currently provide support for running modified copies of this software.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration-logdriver
            '''
            result = self._values.get("log_driver")
            assert result is not None, "Required property 'log_driver' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]]:
            '''The configuration options to send to the log driver.

            This parameter requires version 1.19 of the Docker Remote API or greater on your container instance. To check the Docker Remote API version on your container instance, log in to your container instance and run the following command: ``sudo docker version --format '{{.Server.APIVersion}}'``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-containerdefinition-logconfiguration-options
            '''
            result = self._values.get("options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Mapping[builtins.str, builtins.str]]], result)

        @builtins.property
        def secret_options(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]]:
            '''The secrets to pass to the log configuration.

            For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-logconfiguration.html#cfn-ecs-taskdefinition-logconfiguration-secretoptions
            '''
            result = self._values.get("secret_options")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.SecretProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LogConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.MountPointProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_path": "containerPath",
            "read_only": "readOnly",
            "source_volume": "sourceVolume",
        },
    )
    class MountPointProperty:
        def __init__(
            self,
            *,
            container_path: typing.Optional[builtins.str] = None,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            source_volume: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``MountPoint`` property specifies details on a volume mount point that is used in a container definition.

            :param container_path: The path on the container to mount the host volume at.
            :param read_only: If this value is ``true`` , the container has read-only access to the volume. If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .
            :param source_volume: The name of the volume to mount. Must be a volume name referenced in the ``name`` parameter of task definition ``volume`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                mount_point_property = ecs.CfnTaskDefinition.MountPointProperty(
                    container_path="containerPath",
                    read_only=False,
                    source_volume="sourceVolume"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_path is not None:
                self._values["container_path"] = container_path
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_volume is not None:
                self._values["source_volume"] = source_volume

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The path on the container to mount the host volume at.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''If this value is ``true`` , the container has read-only access to the volume.

            If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def source_volume(self) -> typing.Optional[builtins.str]:
            '''The name of the volume to mount.

            Must be a volume name referenced in the ``name`` parameter of task definition ``volume`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-mountpoints.html#cfn-ecs-taskdefinition-containerdefinition-mountpoints-sourcevolume
            '''
            result = self._values.get("source_volume")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MountPointProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_port": "containerPort",
            "host_port": "hostPort",
            "protocol": "protocol",
        },
    )
    class PortMappingProperty:
        def __init__(
            self,
            *,
            container_port: typing.Optional[jsii.Number] = None,
            host_port: typing.Optional[jsii.Number] = None,
            protocol: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``PortMapping`` property specifies a port mapping.

            Port mappings allow containers to access ports on the host container instance to send or receive traffic. Port mappings are specified as part of the container definition.

            If you are using containers in a task with the ``awsvpc`` or ``host`` network mode, exposed ports should be specified using ``containerPort`` . The ``hostPort`` can be left blank or it must be the same value as the ``containerPort`` .

            After a task reaches the ``RUNNING`` status, manual and automatic host and container port assignments are visible in the ``networkBindings`` section of `DescribeTasks <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html>`_ API responses.

            :param container_port: The port number on the container that's bound to the user-specified or automatically assigned host port. If you use containers in a task with the ``awsvpc`` or ``host`` network mode, specify the exposed ports using ``containerPort`` . If you use containers in a task with the ``bridge`` network mode and you specify a container port and not a host port, your container automatically receives a host port in the ephemeral port range. For more information, see ``hostPort`` . Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.
            :param host_port: The port number on the container instance to reserve for your container. If you are using containers in a task with the ``awsvpc`` or ``host`` network mode, the ``hostPort`` can either be left blank or set to the same value as the ``containerPort`` . If you are using containers in a task with the ``bridge`` network mode, you can specify a non-reserved host port for your container port mapping, or you can omit the ``hostPort`` (or set it to ``0`` ) while specifying a ``containerPort`` and your container automatically receives a port in the ephemeral port range for your container instance operating system and Docker version. The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under ``/proc/sys/net/ipv4/ip_local_port_range`` . If this kernel parameter is unavailable, the default ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the ephemeral port range as these are reserved for automatic assignment. In general, ports below 32768 are outside of the ephemeral port range. .. epigraph:: The default ephemeral port range from 49153 through 65535 is always used for Docker versions before 1.6.0. The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the task is running (after a task stops, the host port is released). The current reserved ports are displayed in the ``remainingResources`` of `DescribeContainerInstances <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html>`_ output. A container instance can have up to 100 reserved ports at a time, including the default reserved ports. Automatically assigned ports don't count toward the 100 reserved ports limit.
            :param protocol: The protocol used for the port mapping. Valid values are ``tcp`` and ``udp`` . The default is ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                port_mapping_property = ecs.CfnTaskDefinition.PortMappingProperty(
                    container_port=123,
                    host_port=123,
                    protocol="protocol"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_port is not None:
                self._values["container_port"] = container_port
            if host_port is not None:
                self._values["host_port"] = host_port
            if protocol is not None:
                self._values["protocol"] = protocol

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''The port number on the container that's bound to the user-specified or automatically assigned host port.

            If you use containers in a task with the ``awsvpc`` or ``host`` network mode, specify the exposed ports using ``containerPort`` .

            If you use containers in a task with the ``bridge`` network mode and you specify a container port and not a host port, your container automatically receives a host port in the ephemeral port range. For more information, see ``hostPort`` . Port mappings that are automatically assigned in this way do not count toward the 100 reserved ports limit of a container instance.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html#cfn-ecs-taskdefinition-containerdefinition-portmappings-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def host_port(self) -> typing.Optional[jsii.Number]:
            '''The port number on the container instance to reserve for your container.

            If you are using containers in a task with the ``awsvpc`` or ``host`` network mode, the ``hostPort`` can either be left blank or set to the same value as the ``containerPort`` .

            If you are using containers in a task with the ``bridge`` network mode, you can specify a non-reserved host port for your container port mapping, or you can omit the ``hostPort`` (or set it to ``0`` ) while specifying a ``containerPort`` and your container automatically receives a port in the ephemeral port range for your container instance operating system and Docker version.

            The default ephemeral port range for Docker version 1.6.0 and later is listed on the instance under ``/proc/sys/net/ipv4/ip_local_port_range`` . If this kernel parameter is unavailable, the default ephemeral port range from 49153 through 65535 is used. Do not attempt to specify a host port in the ephemeral port range as these are reserved for automatic assignment. In general, ports below 32768 are outside of the ephemeral port range.
            .. epigraph::

               The default ephemeral port range from 49153 through 65535 is always used for Docker versions before 1.6.0.

            The default reserved ports are 22 for SSH, the Docker ports 2375 and 2376, and the Amazon ECS container agent ports 51678-51680. Any host port that was previously specified in a running task is also reserved while the task is running (after a task stops, the host port is released). The current reserved ports are displayed in the ``remainingResources`` of `DescribeContainerInstances <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeContainerInstances.html>`_ output. A container instance can have up to 100 reserved ports at a time, including the default reserved ports. Automatically assigned ports don't count toward the 100 reserved ports limit.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html#cfn-ecs-taskdefinition-containerdefinition-portmappings-readonly
            '''
            result = self._values.get("host_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def protocol(self) -> typing.Optional[builtins.str]:
            '''The protocol used for the port mapping.

            Valid values are ``tcp`` and ``udp`` . The default is ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-portmappings.html#cfn-ecs-taskdefinition-containerdefinition-portmappings-sourcevolume
            '''
            result = self._values.get("protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ProxyConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "proxy_configuration_properties": "proxyConfigurationProperties",
            "type": "type",
        },
    )
    class ProxyConfigurationProperty:
        def __init__(
            self,
            *,
            container_name: builtins.str,
            proxy_configuration_properties: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``ProxyConfiguration`` property specifies the details for the App Mesh proxy.

            For tasks using the EC2 launch type, the container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized AMI version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For tasks using the Fargate launch type, the task or service requires platform version 1.3.0 or later.

            :param container_name: The name of the container that will serve as the App Mesh proxy.
            :param proxy_configuration_properties: The set of network configuration parameters to provide the Container Network Interface (CNI) plugin, specified as key-value pairs. - ``IgnoredUID`` - (Required) The user ID (UID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredGID`` is specified, this field can be empty. - ``IgnoredGID`` - (Required) The group ID (GID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredUID`` is specified, this field can be empty. - ``AppPorts`` - (Required) The list of ports that the application uses. Network traffic to these ports is forwarded to the ``ProxyIngressPort`` and ``ProxyEgressPort`` . - ``ProxyIngressPort`` - (Required) Specifies the port that incoming traffic to the ``AppPorts`` is directed to. - ``ProxyEgressPort`` - (Required) Specifies the port that outgoing traffic from the ``AppPorts`` is directed to. - ``EgressIgnoredPorts`` - (Required) The egress traffic going to the specified ports is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list. - ``EgressIgnoredIPs`` - (Required) The egress traffic going to the specified IP addresses is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list.
            :param type: The proxy type. The only supported value is ``APPMESH`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                proxy_configuration_property = ecs.CfnTaskDefinition.ProxyConfigurationProperty(
                    container_name="containerName",
                
                    # the properties below are optional
                    proxy_configuration_properties=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                        name="name",
                        value="value"
                    )],
                    type="type"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "container_name": container_name,
            }
            if proxy_configuration_properties is not None:
                self._values["proxy_configuration_properties"] = proxy_configuration_properties
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def container_name(self) -> builtins.str:
            '''The name of the container that will serve as the App Mesh proxy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-containername
            '''
            result = self._values.get("container_name")
            assert result is not None, "Required property 'container_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def proxy_configuration_properties(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]]:
            '''The set of network configuration parameters to provide the Container Network Interface (CNI) plugin, specified as key-value pairs.

            - ``IgnoredUID`` - (Required) The user ID (UID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredGID`` is specified, this field can be empty.
            - ``IgnoredGID`` - (Required) The group ID (GID) of the proxy container as defined by the ``user`` parameter in a container definition. This is used to ensure the proxy ignores its own traffic. If ``IgnoredUID`` is specified, this field can be empty.
            - ``AppPorts`` - (Required) The list of ports that the application uses. Network traffic to these ports is forwarded to the ``ProxyIngressPort`` and ``ProxyEgressPort`` .
            - ``ProxyIngressPort`` - (Required) Specifies the port that incoming traffic to the ``AppPorts`` is directed to.
            - ``ProxyEgressPort`` - (Required) Specifies the port that outgoing traffic from the ``AppPorts`` is directed to.
            - ``EgressIgnoredPorts`` - (Required) The egress traffic going to the specified ports is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list.
            - ``EgressIgnoredIPs`` - (Required) The egress traffic going to the specified IP addresses is ignored and not redirected to the ``ProxyEgressPort`` . It can be an empty list.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-proxyconfigurationproperties
            '''
            result = self._values.get("proxy_configuration_properties")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskDefinition.KeyValuePairProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''The proxy type.

            The only supported value is ``APPMESH`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-proxyconfiguration.html#cfn-ecs-taskdefinition-proxyconfiguration-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ProxyConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.RepositoryCredentialsProperty",
        jsii_struct_bases=[],
        name_mapping={"credentials_parameter": "credentialsParameter"},
    )
    class RepositoryCredentialsProperty:
        def __init__(
            self,
            *,
            credentials_parameter: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``RepositoryCredentials`` property specifies the repository credentials for private registry authentication.

            :param credentials_parameter: The Amazon Resource Name (ARN) of the secret containing the private repository credentials. .. epigraph:: When you use the Amazon ECS API, AWS CLI , or AWS SDK, if the secret exists in the same Region as the task that you're launching then you can use either the full ARN or the name of the secret. When you use the AWS Management Console, you must specify the full ARN of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-repositorycredentials.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                repository_credentials_property = ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                    credentials_parameter="credentialsParameter"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if credentials_parameter is not None:
                self._values["credentials_parameter"] = credentials_parameter

        @builtins.property
        def credentials_parameter(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the secret containing the private repository credentials.

            .. epigraph::

               When you use the Amazon ECS API, AWS CLI , or AWS SDK, if the secret exists in the same Region as the task that you're launching then you can use either the full ARN or the name of the secret. When you use the AWS Management Console, you must specify the full ARN of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-repositorycredentials.html#cfn-ecs-taskdefinition-repositorycredentials-credentialsparameter
            '''
            result = self._values.get("credentials_parameter")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RepositoryCredentialsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.ResourceRequirementProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "value": "value"},
    )
    class ResourceRequirementProperty:
        def __init__(self, *, type: builtins.str, value: builtins.str) -> None:
            '''The ``ResourceRequirement`` property specifies the type and amount of a resource to assign to a container.

            The only supported resource is a GPU. For more information, see `Working with GPUs on Amazon ECS <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-gpu.html>`_ in the *Amazon Elastic Container Service Developer Guide*

            :param type: The type of resource to assign to a container. The supported values are ``GPU`` or ``InferenceAccelerator`` .
            :param value: The value for the specified resource type. If the ``GPU`` type is used, the value is the number of physical ``GPUs`` the Amazon ECS container agent will reserve for the container. The number of GPUs reserved for all containers in a task should not exceed the number of available GPUs on the container instance the task is launched on. If the ``InferenceAccelerator`` type is used, the ``value`` should match the ``DeviceName`` for an `InferenceAccelerator <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html>`_ specified in a task definition.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                resource_requirement_property = ecs.CfnTaskDefinition.ResourceRequirementProperty(
                    type="type",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
                "value": value,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of resource to assign to a container.

            The supported values are ``GPU`` or ``InferenceAccelerator`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html#cfn-ecs-taskdefinition-resourcerequirement-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value for the specified resource type.

            If the ``GPU`` type is used, the value is the number of physical ``GPUs`` the Amazon ECS container agent will reserve for the container. The number of GPUs reserved for all containers in a task should not exceed the number of available GPUs on the container instance the task is launched on.

            If the ``InferenceAccelerator`` type is used, the ``value`` should match the ``DeviceName`` for an `InferenceAccelerator <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-inferenceaccelerator.html>`_ specified in a task definition.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-resourcerequirement.html#cfn-ecs-taskdefinition-resourcerequirement-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ResourceRequirementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.RuntimePlatformProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cpu_architecture": "cpuArchitecture",
            "operating_system_family": "operatingSystemFamily",
        },
    )
    class RuntimePlatformProperty:
        def __init__(
            self,
            *,
            cpu_architecture: typing.Optional[builtins.str] = None,
            operating_system_family: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Information about the platform for the Amazon ECS service or task.

            For more informataion about ``RuntimePlatform`` , see `RuntimePlatform <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#runtime-platform>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param cpu_architecture: The CPU architecture. You can run your Linux tasks on an ARM-based platform by setting the value to ``ARM64`` . This option is avaiable for tasks that run on Linuc Amazon EC2 instance or Linux containers on Fargate.
            :param operating_system_family: The operating system.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-runtimeplatform.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                runtime_platform_property = ecs.CfnTaskDefinition.RuntimePlatformProperty(
                    cpu_architecture="cpuArchitecture",
                    operating_system_family="operatingSystemFamily"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cpu_architecture is not None:
                self._values["cpu_architecture"] = cpu_architecture
            if operating_system_family is not None:
                self._values["operating_system_family"] = operating_system_family

        @builtins.property
        def cpu_architecture(self) -> typing.Optional[builtins.str]:
            '''The CPU architecture.

            You can run your Linux tasks on an ARM-based platform by setting the value to ``ARM64`` . This option is avaiable for tasks that run on Linuc Amazon EC2 instance or Linux containers on Fargate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-runtimeplatform.html#cfn-ecs-taskdefinition-runtimeplatform-cpuarchitecture
            '''
            result = self._values.get("cpu_architecture")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def operating_system_family(self) -> typing.Optional[builtins.str]:
            '''The operating system.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-runtimeplatform.html#cfn-ecs-taskdefinition-runtimeplatform-operatingsystemfamily
            '''
            result = self._values.get("operating_system_family")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RuntimePlatformProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.SecretProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "value_from": "valueFrom"},
    )
    class SecretProperty:
        def __init__(self, *, name: builtins.str, value_from: builtins.str) -> None:
            '''The ``Secret`` property specifies an object representing the secret to expose to your container.

            For more information, see `Specifying Sensitive Data <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param name: The name of the secret.
            :param value_from: The secret to expose to the container. The supported values are either the full ARN of the AWS Secrets Manager secret or the full ARN of the parameter in the SSM Parameter Store. For information about the require AWS Identity and Access Management permissions, see `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-secrets.html#secrets-iam>`_ (for Secrets Manager) or `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-parameters.html>`_ (for Systems Manager Parameter store) in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If the SSM Parameter Store parameter exists in the same Region as the task you're launching, then you can use either the full ARN or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                secret_property = ecs.CfnTaskDefinition.SecretProperty(
                    name="name",
                    value_from="valueFrom"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
                "value_from": value_from,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the secret.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html#cfn-ecs-taskdefinition-secret-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value_from(self) -> builtins.str:
            '''The secret to expose to the container.

            The supported values are either the full ARN of the AWS Secrets Manager secret or the full ARN of the parameter in the SSM Parameter Store.

            For information about the require AWS Identity and Access Management permissions, see `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-secrets.html#secrets-iam>`_ (for Secrets Manager) or `Required IAM permissions for Amazon ECS secrets <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/specifying-sensitive-data-parameters.html>`_ (for Systems Manager Parameter store) in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If the SSM Parameter Store parameter exists in the same Region as the task you're launching, then you can use either the full ARN or name of the parameter. If the parameter exists in a different Region, then the full ARN must be specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-secret.html#cfn-ecs-taskdefinition-secret-valuefrom
            '''
            result = self._values.get("value_from")
            assert result is not None, "Required property 'value_from' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SecretProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.SystemControlProperty",
        jsii_struct_bases=[],
        name_mapping={"namespace": "namespace", "value": "value"},
    )
    class SystemControlProperty:
        def __init__(
            self,
            *,
            namespace: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''A list of namespaced kernel parameters to set in the container.

            This parameter maps to ``Sysctls`` in the `Create a container <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/#operation/ContainerCreate>`_ section of the `Docker Remote API <https://docs.aws.amazon.com/https://docs.docker.com/engine/api/v1.35/>`_ and the ``--sysctl`` option to `docker run <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#security-configuration>`_ .

            We don't recommend that you specify network-related ``systemControls`` parameters for multiple containers in a single task. This task also uses either the ``awsvpc`` or ``host`` network mode. It does it for the following reasons.

            - For tasks that use the ``awsvpc`` network mode, if you set ``systemControls`` for any container, it applies to all containers in the task. If you set different ``systemControls`` for multiple containers in a single task, the container that's started last determines which ``systemControls`` take effect.
            - For tasks that use the ``host`` network mode, the ``systemControls`` parameter applies to the container instance's kernel parameter and that of all containers of any tasks running on that container instance.

            :param namespace: The namespaced kernel parameter to set a ``value`` for.
            :param value: The value for the namespaced kernel parameter that's specified in ``namespace`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                system_control_property = ecs.CfnTaskDefinition.SystemControlProperty(
                    namespace="namespace",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if namespace is not None:
                self._values["namespace"] = namespace
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def namespace(self) -> typing.Optional[builtins.str]:
            '''The namespaced kernel parameter to set a ``value`` for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html#cfn-ecs-taskdefinition-systemcontrol-namespace
            '''
            result = self._values.get("namespace")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value for the namespaced kernel parameter that's specified in ``namespace`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-systemcontrol.html#cfn-ecs-taskdefinition-systemcontrol-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SystemControlProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "expression": "expression"},
    )
    class TaskDefinitionPlacementConstraintProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            expression: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``TaskDefinitionPlacementConstraint`` property specifies an object representing a constraint on task placement in the task definition.

            If you are using the Fargate launch type, task placement constraints are not supported.

            For more information, see `Task Placement Constraints <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :param type: The type of constraint. The ``MemberOf`` constraint restricts selection to be from a group of valid candidates.
            :param expression: A cluster query language expression to apply to the constraint. For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                task_definition_placement_constraint_property = ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty(
                    type="type",
                
                    # the properties below are optional
                    expression="expression"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if expression is not None:
                self._values["expression"] = expression

        @builtins.property
        def type(self) -> builtins.str:
            '''The type of constraint.

            The ``MemberOf`` constraint restricts selection to be from a group of valid candidates.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html#cfn-ecs-taskdefinition-taskdefinitionplacementconstraint-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def expression(self) -> typing.Optional[builtins.str]:
            '''A cluster query language expression to apply to the constraint.

            For more information, see `Cluster query language <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-taskdefinitionplacementconstraint.html#cfn-ecs-taskdefinition-taskdefinitionplacementconstraint-expression
            '''
            result = self._values.get("expression")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TaskDefinitionPlacementConstraintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.TmpfsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "size": "size",
            "container_path": "containerPath",
            "mount_options": "mountOptions",
        },
    )
    class TmpfsProperty:
        def __init__(
            self,
            *,
            size: jsii.Number,
            container_path: typing.Optional[builtins.str] = None,
            mount_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The ``Tmpfs`` property specifies the container path, mount options, and size of the tmpfs mount.

            :param size: The maximum size (in MiB) of the tmpfs volume.
            :param container_path: The absolute file path where the tmpfs volume is to be mounted.
            :param mount_options: The list of tmpfs volume mount options. Valid values: ``"defaults" | "ro" | "rw" | "suid" | "nosuid" | "dev" | "nodev" | "exec" | "noexec" | "sync" | "async" | "dirsync" | "remount" | "mand" | "nomand" | "atime" | "noatime" | "diratime" | "nodiratime" | "bind" | "rbind" | "unbindable" | "runbindable" | "private" | "rprivate" | "shared" | "rshared" | "slave" | "rslave" | "relatime" | "norelatime" | "strictatime" | "nostrictatime" | "mode" | "uid" | "gid" | "nr_inodes" | "nr_blocks" | "mpol"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                tmpfs_property = ecs.CfnTaskDefinition.TmpfsProperty(
                    size=123,
                
                    # the properties below are optional
                    container_path="containerPath",
                    mount_options=["mountOptions"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "size": size,
            }
            if container_path is not None:
                self._values["container_path"] = container_path
            if mount_options is not None:
                self._values["mount_options"] = mount_options

        @builtins.property
        def size(self) -> jsii.Number:
            '''The maximum size (in MiB) of the tmpfs volume.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-size
            '''
            result = self._values.get("size")
            assert result is not None, "Required property 'size' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def container_path(self) -> typing.Optional[builtins.str]:
            '''The absolute file path where the tmpfs volume is to be mounted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-containerpath
            '''
            result = self._values.get("container_path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def mount_options(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The list of tmpfs volume mount options.

            Valid values: ``"defaults" | "ro" | "rw" | "suid" | "nosuid" | "dev" | "nodev" | "exec" | "noexec" | "sync" | "async" | "dirsync" | "remount" | "mand" | "nomand" | "atime" | "noatime" | "diratime" | "nodiratime" | "bind" | "rbind" | "unbindable" | "runbindable" | "private" | "rprivate" | "shared" | "rshared" | "slave" | "rslave" | "relatime" | "norelatime" | "strictatime" | "nostrictatime" | "mode" | "uid" | "gid" | "nr_inodes" | "nr_blocks" | "mpol"``

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-tmpfs.html#cfn-ecs-taskdefinition-tmpfs-mountoptions
            '''
            result = self._values.get("mount_options")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TmpfsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.UlimitProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hard_limit": "hardLimit",
            "name": "name",
            "soft_limit": "softLimit",
        },
    )
    class UlimitProperty:
        def __init__(
            self,
            *,
            hard_limit: jsii.Number,
            name: builtins.str,
            soft_limit: jsii.Number,
        ) -> None:
            '''The ``Ulimit`` property specifies the ``ulimit`` settings to pass to the container.

            :param hard_limit: The hard limit for the ulimit type.
            :param name: The ``type`` of the ``ulimit`` .
            :param soft_limit: The soft limit for the ulimit type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                ulimit_property = ecs.CfnTaskDefinition.UlimitProperty(
                    hard_limit=123,
                    name="name",
                    soft_limit=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "hard_limit": hard_limit,
                "name": name,
                "soft_limit": soft_limit,
            }

        @builtins.property
        def hard_limit(self) -> jsii.Number:
            '''The hard limit for the ulimit type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html#cfn-ecs-taskdefinition-containerdefinition-ulimit-hardlimit
            '''
            result = self._values.get("hard_limit")
            assert result is not None, "Required property 'hard_limit' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def name(self) -> builtins.str:
            '''The ``type`` of the ``ulimit`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html#cfn-ecs-taskdefinition-containerdefinition-ulimit-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def soft_limit(self) -> jsii.Number:
            '''The soft limit for the ulimit type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-ulimit.html#cfn-ecs-taskdefinition-containerdefinition-ulimit-softlimit
            '''
            result = self._values.get("soft_limit")
            assert result is not None, "Required property 'soft_limit' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "UlimitProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.VolumeFromProperty",
        jsii_struct_bases=[],
        name_mapping={"read_only": "readOnly", "source_container": "sourceContainer"},
    )
    class VolumeFromProperty:
        def __init__(
            self,
            *,
            read_only: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            source_container: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``VolumeFrom`` property specifies details on a data volume from another container in the same task definition.

            :param read_only: If this value is ``true`` , the container has read-only access to the volume. If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .
            :param source_container: The name of another container within the same task definition to mount volumes from.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-volumesfrom.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                volume_from_property = ecs.CfnTaskDefinition.VolumeFromProperty(
                    read_only=False,
                    source_container="sourceContainer"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if read_only is not None:
                self._values["read_only"] = read_only
            if source_container is not None:
                self._values["source_container"] = source_container

        @builtins.property
        def read_only(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''If this value is ``true`` , the container has read-only access to the volume.

            If this value is ``false`` , then the container can write to the volume. The default value is ``false`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-volumesfrom.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom-readonly
            '''
            result = self._values.get("read_only")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def source_container(self) -> typing.Optional[builtins.str]:
            '''The name of another container within the same task definition to mount volumes from.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-containerdefinitions-volumesfrom.html#cfn-ecs-taskdefinition-containerdefinition-volumesfrom-sourcecontainer
            '''
            result = self._values.get("source_container")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeFromProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskDefinition.VolumeProperty",
        jsii_struct_bases=[],
        name_mapping={
            "docker_volume_configuration": "dockerVolumeConfiguration",
            "efs_volume_configuration": "efsVolumeConfiguration",
            "host": "host",
            "name": "name",
        },
    )
    class VolumeProperty:
        def __init__(
            self,
            *,
            docker_volume_configuration: typing.Optional[typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", _IResolvable_a771d0ef]] = None,
            efs_volume_configuration: typing.Optional[typing.Union["CfnTaskDefinition.EfsVolumeConfigurationProperty", _IResolvable_a771d0ef]] = None,
            host: typing.Optional[typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", _IResolvable_a771d0ef]] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The ``Volume`` property specifies a data volume used in a task definition.

            For tasks that use a Docker volume, specify a ``DockerVolumeConfiguration`` . For tasks that use a bind mount host volume, specify a ``host`` and optional ``sourcePath`` . For more information, see `Using Data Volumes in Tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ .

            :param docker_volume_configuration: This parameter is specified when you use Docker volumes. Windows containers only support the use of the ``local`` driver. To use bind mounts, specify the ``host`` parameter instead. .. epigraph:: Docker volumes aren't supported by tasks run on AWS Fargate .
            :param efs_volume_configuration: ``CfnTaskDefinition.VolumeProperty.EfsVolumeConfiguration``.
            :param host: This parameter is specified when you use bind mount host volumes. The contents of the ``host`` parameter determine whether your bind mount host volume persists on the host container instance and where it's stored. If the ``host`` parameter is empty, then the Docker daemon assigns a host path for your data volume. However, the data isn't guaranteed to persist after the containers that are associated with it stop running. Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives. For example, you can mount ``C:\\my\\path:C:\\my\\path`` and ``D:\\:D:\\`` , but not ``D:\\my\\path:C:\\my\\path`` or ``D:\\:C:\\my\\path`` .
            :param name: The name of the volume. Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This name is referenced in the ``sourceVolume`` parameter of container definition ``mountPoints`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                volume_property = ecs.CfnTaskDefinition.VolumeProperty(
                    docker_volume_configuration=ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                        autoprovision=False,
                        driver="driver",
                        driver_opts={
                            "driver_opts_key": "driverOpts"
                        },
                        labels={
                            "labels_key": "labels"
                        },
                        scope="scope"
                    ),
                    efs_volume_configuration=ecs.CfnTaskDefinition.EfsVolumeConfigurationProperty(
                        file_system_id="fileSystemId",
                
                        # the properties below are optional
                        authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                            access_point_id="accessPointId",
                            iam="iam"
                        ),
                        root_directory="rootDirectory",
                        transit_encryption="transitEncryption",
                        transit_encryption_port=123
                    ),
                    host=ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                        source_path="sourcePath"
                    ),
                    name="name"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if docker_volume_configuration is not None:
                self._values["docker_volume_configuration"] = docker_volume_configuration
            if efs_volume_configuration is not None:
                self._values["efs_volume_configuration"] = efs_volume_configuration
            if host is not None:
                self._values["host"] = host
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def docker_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", _IResolvable_a771d0ef]]:
            '''This parameter is specified when you use Docker volumes.

            Windows containers only support the use of the ``local`` driver. To use bind mounts, specify the ``host`` parameter instead.
            .. epigraph::

               Docker volumes aren't supported by tasks run on AWS Fargate .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volume-dockervolumeconfiguration
            '''
            result = self._values.get("docker_volume_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.DockerVolumeConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def efs_volume_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.EfsVolumeConfigurationProperty", _IResolvable_a771d0ef]]:
            '''``CfnTaskDefinition.VolumeProperty.EfsVolumeConfiguration``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volume-efsvolumeconfiguration
            '''
            result = self._values.get("efs_volume_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.EfsVolumeConfigurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def host(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", _IResolvable_a771d0ef]]:
            '''This parameter is specified when you use bind mount host volumes.

            The contents of the ``host`` parameter determine whether your bind mount host volume persists on the host container instance and where it's stored. If the ``host`` parameter is empty, then the Docker daemon assigns a host path for your data volume. However, the data isn't guaranteed to persist after the containers that are associated with it stop running.

            Windows containers can mount whole directories on the same drive as ``$env:ProgramData`` . Windows containers can't mount directories on a different drive, and mount point can't be across drives. For example, you can mount ``C:\\my\\path:C:\\my\\path`` and ``D:\\:D:\\`` , but not ``D:\\my\\path:C:\\my\\path`` or ``D:\\:C:\\my\\path`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volumes-host
            '''
            result = self._values.get("host")
            return typing.cast(typing.Optional[typing.Union["CfnTaskDefinition.HostVolumePropertiesProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''The name of the volume.

            Up to 255 letters (uppercase and lowercase), numbers, underscores, and hyphens are allowed. This name is referenced in the ``sourceVolume`` parameter of container definition ``mountPoints`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskdefinition-volumes.html#cfn-ecs-taskdefinition-volumes-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnTaskDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "container_definitions": "containerDefinitions",
        "cpu": "cpu",
        "ephemeral_storage": "ephemeralStorage",
        "execution_role_arn": "executionRoleArn",
        "family": "family",
        "inference_accelerators": "inferenceAccelerators",
        "ipc_mode": "ipcMode",
        "memory": "memory",
        "network_mode": "networkMode",
        "pid_mode": "pidMode",
        "placement_constraints": "placementConstraints",
        "proxy_configuration": "proxyConfiguration",
        "requires_compatibilities": "requiresCompatibilities",
        "runtime_platform": "runtimePlatform",
        "tags": "tags",
        "task_role_arn": "taskRoleArn",
        "volumes": "volumes",
    },
)
class CfnTaskDefinitionProps:
    def __init__(
        self,
        *,
        container_definitions: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, _IResolvable_a771d0ef]]]] = None,
        cpu: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[CfnTaskDefinition.EphemeralStorageProperty, _IResolvable_a771d0ef]] = None,
        execution_role_arn: typing.Optional[builtins.str] = None,
        family: typing.Optional[builtins.str] = None,
        inference_accelerators: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, _IResolvable_a771d0ef]]]] = None,
        ipc_mode: typing.Optional[builtins.str] = None,
        memory: typing.Optional[builtins.str] = None,
        network_mode: typing.Optional[builtins.str] = None,
        pid_mode: typing.Optional[builtins.str] = None,
        placement_constraints: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, _IResolvable_a771d0ef]]]] = None,
        proxy_configuration: typing.Optional[typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, _IResolvable_a771d0ef]] = None,
        requires_compatibilities: typing.Optional[typing.Sequence[builtins.str]] = None,
        runtime_platform: typing.Optional[typing.Union[CfnTaskDefinition.RuntimePlatformProperty, _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        task_role_arn: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskDefinition.VolumeProperty, _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnTaskDefinition``.

        :param container_definitions: A list of container definitions in JSON format that describe the different containers that make up your task. For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param cpu: The number of ``cpu`` units used by the task. If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter. The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate. - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - 2048 (2 vCPU) - Available ``memory`` values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - 4096 (4 vCPU) - Available ``memory`` values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)
        :param ephemeral_storage: The ephemeral storage settings to use for tasks run with the task definition.
        :param execution_role_arn: The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf. The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param family: The name of a family that this task definition is registered to. Up to 255 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed. A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task definition that you add. .. epigraph:: To use revision numbers when you update a task definition, specify this property. If you don't specify a value, AWS CloudFormation generates a new task definition each time that you update it.
        :param inference_accelerators: The Elastic Inference accelerators to use for the containers in the task.
        :param ipc_mode: The IPC resource namespace to use for the containers in the task. The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same IPC resources. If ``none`` is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the *Docker run reference* . If the ``host`` IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ . If you are setting namespaced kernel parameters using ``systemControls`` for the containers in the task, the following will apply to your IPC resource namespace. For more information, see `System Controls <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_ in the *Amazon Elastic Container Service Developer Guide* . - For tasks that use the ``host`` IPC mode, IPC namespace related ``systemControls`` are not supported. - For tasks that use the ``task`` IPC mode, IPC namespace related ``systemControls`` will apply to all containers within a task. .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
        :param memory: The amount (in MiB) of memory used by the task. If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a container-level memory value. This field is optional and any value can be used. If a task-level memory value is specified, the container-level memory value is optional. For more information regarding container-level memory and memory reservation, see `ContainerDefinition <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html>`_ . If your tasks runs on AWS Fargate , this field is required. You must use one of the following values. The value you choose determines your range of valid values for the ``cpu`` parameter. - 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available ``cpu`` values: 256 (.25 vCPU) - 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available ``cpu`` values: 512 (.5 vCPU) - 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available ``cpu`` values: 1024 (1 vCPU) - Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 2048 (2 vCPU) - Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 4096 (4 vCPU)
        :param network_mode: The Docker networking mode to use for the containers in the task. The valid values are ``none`` , ``bridge`` , ``awsvpc`` , and ``host`` . The default Docker network mode is ``bridge`` . If you are using the Fargate launch type, the ``awsvpc`` network mode is required. If you are using the EC2 launch type, any network mode can be used. If the network mode is set to ``none`` , you cannot specify port mappings in your container definitions, and the tasks containers do not have external connectivity. The ``host`` and ``awsvpc`` network modes offer the highest networking performance for containers because they use the EC2 network stack instead of the virtualized network stack provided by the ``bridge`` mode. With the ``host`` and ``awsvpc`` network modes, exposed container ports are mapped directly to the corresponding host port (for the ``host`` network mode) or the attached elastic network interface port (for the ``awsvpc`` network mode), so you cannot take advantage of dynamic host port mappings. If the network mode is ``awsvpc`` , the task is allocated an elastic network interface, and you must specify a `NetworkConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html>`_ value when you create a service or run a task with the task definition. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: Currently, only Amazon ECS-optimized AMIs, other Amazon Linux variants with the ``ecs-init`` package, or AWS Fargate infrastructure support the ``awsvpc`` network mode. If the network mode is ``host`` , you cannot run multiple instantiations of the same task on a single container instance when port mappings are used. Docker for Windows uses different network modes than Docker for Linux. When you register a task definition with Windows containers, you must not specify a network mode. If you use the console to register a task definition with Windows containers, you must choose the ``<default>`` network mode object. For more information, see `Network settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#network-settings>`_ in the *Docker run reference* .
        :param pid_mode: The process namespace to use for the containers in the task. The valid values are ``host`` or ``task`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the *Docker run reference* . If the ``host`` PID mode is used, be aware that there is a heightened risk of undesired process namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ . .. epigraph:: This parameter is not supported for Windows containers or tasks run on AWS Fargate .
        :param placement_constraints: An array of placement constraint objects to use for tasks. .. epigraph:: This parameter isn't supported for tasks run on AWS Fargate .
        :param proxy_configuration: The ``ProxyConfiguration`` property specifies the configuration details for the App Mesh proxy. Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized AMI version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param requires_compatibilities: The task launch types the task definition was validated against. To determine which task launch types the task definition is validated for, see the ``TaskDefinition$compatibilities`` parameter.
        :param runtime_platform: The operating system that your tasks definitions run on. A platform family is specified only for tasks using the Fargate launch type. When you specify a task definition in a service, this value must match the ``runtimePlatform`` value of the service.
        :param tags: The metadata that you apply to the task definition to help you categorize and organize them. Each tag consists of a key and an optional value. You define both of them. The following basic restrictions apply to tags: - Maximum number of tags per resource - 50 - For each resource, each tag key must be unique, and each tag key can have only one value. - Maximum key length - 128 Unicode characters in UTF-8 - Maximum value length - 256 Unicode characters in UTF-8 - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @. - Tag keys and values are case-sensitive. - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
        :param task_role_arn: The short name or full Amazon Resource Name (ARN) of the AWS Identity and Access Management role that grants containers in the task permission to call AWS APIs on your behalf. For more information, see `Amazon ECS Task Role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* . IAM roles for tasks on Windows require that the ``-EnableTaskIAMRole`` option is set when you launch the Amazon ECS-optimized Windows AMI. Your containers must also run some configuration code to use the feature. For more information, see `Windows IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows_task_IAM_roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        :param volumes: The list of data volume definitions for the task. For more information, see `Using data volumes in tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: The ``host`` and ``sourcePath`` parameters aren't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_task_definition_props = ecs.CfnTaskDefinitionProps(
                container_definitions=[ecs.CfnTaskDefinition.ContainerDefinitionProperty(
                    command=["command"],
                    cpu=123,
                    depends_on=[ecs.CfnTaskDefinition.ContainerDependencyProperty(
                        condition="condition",
                        container_name="containerName"
                    )],
                    disable_networking=False,
                    dns_search_domains=["dnsSearchDomains"],
                    dns_servers=["dnsServers"],
                    docker_labels={
                        "docker_labels_key": "dockerLabels"
                    },
                    docker_security_options=["dockerSecurityOptions"],
                    entry_point=["entryPoint"],
                    environment=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                        name="name",
                        value="value"
                    )],
                    environment_files=[ecs.CfnTaskDefinition.EnvironmentFileProperty(
                        type="type",
                        value="value"
                    )],
                    essential=False,
                    extra_hosts=[ecs.CfnTaskDefinition.HostEntryProperty(
                        hostname="hostname",
                        ip_address="ipAddress"
                    )],
                    firelens_configuration=ecs.CfnTaskDefinition.FirelensConfigurationProperty(
                        options={
                            "options_key": "options"
                        },
                        type="type"
                    ),
                    health_check=ecs.CfnTaskDefinition.HealthCheckProperty(
                        command=["command"],
                        interval=123,
                        retries=123,
                        start_period=123,
                        timeout=123
                    ),
                    hostname="hostname",
                    image="image",
                    interactive=False,
                    links=["links"],
                    linux_parameters=ecs.CfnTaskDefinition.LinuxParametersProperty(
                        capabilities=ecs.CfnTaskDefinition.KernelCapabilitiesProperty(
                            add=["add"],
                            drop=["drop"]
                        ),
                        devices=[ecs.CfnTaskDefinition.DeviceProperty(
                            container_path="containerPath",
                            host_path="hostPath",
                            permissions=["permissions"]
                        )],
                        init_process_enabled=False,
                        max_swap=123,
                        shared_memory_size=123,
                        swappiness=123,
                        tmpfs=[ecs.CfnTaskDefinition.TmpfsProperty(
                            size=123,
            
                            # the properties below are optional
                            container_path="containerPath",
                            mount_options=["mountOptions"]
                        )]
                    ),
                    log_configuration=ecs.CfnTaskDefinition.LogConfigurationProperty(
                        log_driver="logDriver",
            
                        # the properties below are optional
                        options={
                            "options_key": "options"
                        },
                        secret_options=[ecs.CfnTaskDefinition.SecretProperty(
                            name="name",
                            value_from="valueFrom"
                        )]
                    ),
                    memory=123,
                    memory_reservation=123,
                    mount_points=[ecs.CfnTaskDefinition.MountPointProperty(
                        container_path="containerPath",
                        read_only=False,
                        source_volume="sourceVolume"
                    )],
                    name="name",
                    port_mappings=[ecs.CfnTaskDefinition.PortMappingProperty(
                        container_port=123,
                        host_port=123,
                        protocol="protocol"
                    )],
                    privileged=False,
                    pseudo_terminal=False,
                    readonly_root_filesystem=False,
                    repository_credentials=ecs.CfnTaskDefinition.RepositoryCredentialsProperty(
                        credentials_parameter="credentialsParameter"
                    ),
                    resource_requirements=[ecs.CfnTaskDefinition.ResourceRequirementProperty(
                        type="type",
                        value="value"
                    )],
                    secrets=[ecs.CfnTaskDefinition.SecretProperty(
                        name="name",
                        value_from="valueFrom"
                    )],
                    start_timeout=123,
                    stop_timeout=123,
                    system_controls=[ecs.CfnTaskDefinition.SystemControlProperty(
                        namespace="namespace",
                        value="value"
                    )],
                    ulimits=[ecs.CfnTaskDefinition.UlimitProperty(
                        hard_limit=123,
                        name="name",
                        soft_limit=123
                    )],
                    user="user",
                    volumes_from=[ecs.CfnTaskDefinition.VolumeFromProperty(
                        read_only=False,
                        source_container="sourceContainer"
                    )],
                    working_directory="workingDirectory"
                )],
                cpu="cpu",
                ephemeral_storage=ecs.CfnTaskDefinition.EphemeralStorageProperty(
                    size_in_gi_b=123
                ),
                execution_role_arn="executionRoleArn",
                family="family",
                inference_accelerators=[ecs.CfnTaskDefinition.InferenceAcceleratorProperty(
                    device_name="deviceName",
                    device_type="deviceType"
                )],
                ipc_mode="ipcMode",
                memory="memory",
                network_mode="networkMode",
                pid_mode="pidMode",
                placement_constraints=[ecs.CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty(
                    type="type",
            
                    # the properties below are optional
                    expression="expression"
                )],
                proxy_configuration=ecs.CfnTaskDefinition.ProxyConfigurationProperty(
                    container_name="containerName",
            
                    # the properties below are optional
                    proxy_configuration_properties=[ecs.CfnTaskDefinition.KeyValuePairProperty(
                        name="name",
                        value="value"
                    )],
                    type="type"
                ),
                requires_compatibilities=["requiresCompatibilities"],
                runtime_platform=ecs.CfnTaskDefinition.RuntimePlatformProperty(
                    cpu_architecture="cpuArchitecture",
                    operating_system_family="operatingSystemFamily"
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                task_role_arn="taskRoleArn",
                volumes=[ecs.CfnTaskDefinition.VolumeProperty(
                    docker_volume_configuration=ecs.CfnTaskDefinition.DockerVolumeConfigurationProperty(
                        autoprovision=False,
                        driver="driver",
                        driver_opts={
                            "driver_opts_key": "driverOpts"
                        },
                        labels={
                            "labels_key": "labels"
                        },
                        scope="scope"
                    ),
                    efs_volume_configuration=ecs.CfnTaskDefinition.EfsVolumeConfigurationProperty(
                        file_system_id="fileSystemId",
            
                        # the properties below are optional
                        authorization_config=ecs.CfnTaskDefinition.AuthorizationConfigProperty(
                            access_point_id="accessPointId",
                            iam="iam"
                        ),
                        root_directory="rootDirectory",
                        transit_encryption="transitEncryption",
                        transit_encryption_port=123
                    ),
                    host=ecs.CfnTaskDefinition.HostVolumePropertiesProperty(
                        source_path="sourcePath"
                    ),
                    name="name"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if container_definitions is not None:
            self._values["container_definitions"] = container_definitions
        if cpu is not None:
            self._values["cpu"] = cpu
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if execution_role_arn is not None:
            self._values["execution_role_arn"] = execution_role_arn
        if family is not None:
            self._values["family"] = family
        if inference_accelerators is not None:
            self._values["inference_accelerators"] = inference_accelerators
        if ipc_mode is not None:
            self._values["ipc_mode"] = ipc_mode
        if memory is not None:
            self._values["memory"] = memory
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if pid_mode is not None:
            self._values["pid_mode"] = pid_mode
        if placement_constraints is not None:
            self._values["placement_constraints"] = placement_constraints
        if proxy_configuration is not None:
            self._values["proxy_configuration"] = proxy_configuration
        if requires_compatibilities is not None:
            self._values["requires_compatibilities"] = requires_compatibilities
        if runtime_platform is not None:
            self._values["runtime_platform"] = runtime_platform
        if tags is not None:
            self._values["tags"] = tags
        if task_role_arn is not None:
            self._values["task_role_arn"] = task_role_arn
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def container_definitions(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, _IResolvable_a771d0ef]]]]:
        '''A list of container definitions in JSON format that describe the different containers that make up your task.

        For more information about container definition parameters and defaults, see `Amazon ECS Task Definitions <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_defintions.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-containerdefinitions
        '''
        result = self._values.get("container_definitions")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.ContainerDefinitionProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def cpu(self) -> typing.Optional[builtins.str]:
        '''The number of ``cpu`` units used by the task.

        If you use the EC2 launch type, this field is optional. Any value can be used. If you use the Fargate launch type, this field is required. You must use one of the following values. The value that you choose determines your range of valid values for the ``memory`` parameter.

        The CPU units cannot be less than 1 vCPU when you use Windows containers on Fargate.

        - 256 (.25 vCPU) - Available ``memory`` values: 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB)
        - 512 (.5 vCPU) - Available ``memory`` values: 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB)
        - 1024 (1 vCPU) - Available ``memory`` values: 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB)
        - 2048 (2 vCPU) - Available ``memory`` values: Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB)
        - 4096 (4 vCPU) - Available ``memory`` values: Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-cpu
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskDefinition.EphemeralStorageProperty, _IResolvable_a771d0ef]]:
        '''The ephemeral storage settings to use for tasks run with the task definition.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[CfnTaskDefinition.EphemeralStorageProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def execution_role_arn(self) -> typing.Optional[builtins.str]:
        '''The Amazon Resource Name (ARN) of the task execution role that grants the Amazon ECS container agent permission to make AWS API calls on your behalf.

        The task execution IAM role is required depending on the requirements of your task. For more information, see `Amazon ECS task execution IAM role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_execution_IAM_role.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-executionrolearn
        '''
        result = self._values.get("execution_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def family(self) -> typing.Optional[builtins.str]:
        '''The name of a family that this task definition is registered to.

        Up to 255 letters (uppercase and lowercase), numbers, hyphens, and underscores are allowed.

        A family groups multiple versions of a task definition. Amazon ECS gives the first task definition that you registered to a family a revision number of 1. Amazon ECS gives sequential revision numbers to each task definition that you add.
        .. epigraph::

           To use revision numbers when you update a task definition, specify this property. If you don't specify a value, AWS CloudFormation generates a new task definition each time that you update it.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-family
        '''
        result = self._values.get("family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, _IResolvable_a771d0ef]]]]:
        '''The Elastic Inference accelerators to use for the containers in the task.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-inferenceaccelerators
        '''
        result = self._values.get("inference_accelerators")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.InferenceAcceleratorProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def ipc_mode(self) -> typing.Optional[builtins.str]:
        '''The IPC resource namespace to use for the containers in the task.

        The valid values are ``host`` , ``task`` , or ``none`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` IPC mode on the same container instance share the same IPC resources with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same IPC resources. If ``none`` is specified, then IPC resources within the containers of a task are private and not shared with other containers in a task or on the container instance. If no value is specified, then the IPC resource namespace sharing depends on the Docker daemon setting on the container instance. For more information, see `IPC settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#ipc-settings---ipc>`_ in the *Docker run reference* .

        If the ``host`` IPC mode is used, be aware that there is a heightened risk of undesired IPC namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ .

        If you are setting namespaced kernel parameters using ``systemControls`` for the containers in the task, the following will apply to your IPC resource namespace. For more information, see `System Controls <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        - For tasks that use the ``host`` IPC mode, IPC namespace related ``systemControls`` are not supported.
        - For tasks that use the ``task`` IPC mode, IPC namespace related ``systemControls`` will apply to all containers within a task.

        .. epigraph::

           This parameter is not supported for Windows containers or tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-ipcmode
        '''
        result = self._values.get("ipc_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def memory(self) -> typing.Optional[builtins.str]:
        '''The amount (in MiB) of memory used by the task.

        If your tasks runs on Amazon EC2 instances, you must specify either a task-level memory value or a container-level memory value. This field is optional and any value can be used. If a task-level memory value is specified, the container-level memory value is optional. For more information regarding container-level memory and memory reservation, see `ContainerDefinition <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ContainerDefinition.html>`_ .

        If your tasks runs on AWS Fargate , this field is required. You must use one of the following values. The value you choose determines your range of valid values for the ``cpu`` parameter.

        - 512 (0.5 GB), 1024 (1 GB), 2048 (2 GB) - Available ``cpu`` values: 256 (.25 vCPU)
        - 1024 (1 GB), 2048 (2 GB), 3072 (3 GB), 4096 (4 GB) - Available ``cpu`` values: 512 (.5 vCPU)
        - 2048 (2 GB), 3072 (3 GB), 4096 (4 GB), 5120 (5 GB), 6144 (6 GB), 7168 (7 GB), 8192 (8 GB) - Available ``cpu`` values: 1024 (1 vCPU)
        - Between 4096 (4 GB) and 16384 (16 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 2048 (2 vCPU)
        - Between 8192 (8 GB) and 30720 (30 GB) in increments of 1024 (1 GB) - Available ``cpu`` values: 4096 (4 vCPU)

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-memory
        '''
        result = self._values.get("memory")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_mode(self) -> typing.Optional[builtins.str]:
        '''The Docker networking mode to use for the containers in the task.

        The valid values are ``none`` , ``bridge`` , ``awsvpc`` , and ``host`` . The default Docker network mode is ``bridge`` . If you are using the Fargate launch type, the ``awsvpc`` network mode is required. If you are using the EC2 launch type, any network mode can be used. If the network mode is set to ``none`` , you cannot specify port mappings in your container definitions, and the tasks containers do not have external connectivity. The ``host`` and ``awsvpc`` network modes offer the highest networking performance for containers because they use the EC2 network stack instead of the virtualized network stack provided by the ``bridge`` mode.

        With the ``host`` and ``awsvpc`` network modes, exposed container ports are mapped directly to the corresponding host port (for the ``host`` network mode) or the attached elastic network interface port (for the ``awsvpc`` network mode), so you cannot take advantage of dynamic host port mappings.

        If the network mode is ``awsvpc`` , the task is allocated an elastic network interface, and you must specify a `NetworkConfiguration <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_NetworkConfiguration.html>`_ value when you create a service or run a task with the task definition. For more information, see `Task Networking <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-networking.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        .. epigraph::

           Currently, only Amazon ECS-optimized AMIs, other Amazon Linux variants with the ``ecs-init`` package, or AWS Fargate infrastructure support the ``awsvpc`` network mode.

        If the network mode is ``host`` , you cannot run multiple instantiations of the same task on a single container instance when port mappings are used.

        Docker for Windows uses different network modes than Docker for Linux. When you register a task definition with Windows containers, you must not specify a network mode. If you use the console to register a task definition with Windows containers, you must choose the ``<default>`` network mode object.

        For more information, see `Network settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#network-settings>`_ in the *Docker run reference* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-networkmode
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def pid_mode(self) -> typing.Optional[builtins.str]:
        '''The process namespace to use for the containers in the task.

        The valid values are ``host`` or ``task`` . If ``host`` is specified, then all containers within the tasks that specified the ``host`` PID mode on the same container instance share the same process namespace with the host Amazon EC2 instance. If ``task`` is specified, all containers within the specified task share the same process namespace. If no value is specified, the default is a private namespace. For more information, see `PID settings <https://docs.aws.amazon.com/https://docs.docker.com/engine/reference/run/#pid-settings---pid>`_ in the *Docker run reference* .

        If the ``host`` PID mode is used, be aware that there is a heightened risk of undesired process namespace expose. For more information, see `Docker security <https://docs.aws.amazon.com/https://docs.docker.com/engine/security/security/>`_ .
        .. epigraph::

           This parameter is not supported for Windows containers or tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-pidmode
        '''
        result = self._values.get("pid_mode")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def placement_constraints(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, _IResolvable_a771d0ef]]]]:
        '''An array of placement constraint objects to use for tasks.

        .. epigraph::

           This parameter isn't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-placementconstraints
        '''
        result = self._values.get("placement_constraints")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.TaskDefinitionPlacementConstraintProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def proxy_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, _IResolvable_a771d0ef]]:
        '''The ``ProxyConfiguration`` property specifies the configuration details for the App Mesh proxy.

        Your Amazon ECS container instances require at least version 1.26.0 of the container agent and at least version 1.26.0-1 of the ``ecs-init`` package to enable a proxy configuration. If your container instances are launched from the Amazon ECS-optimized AMI version ``20190301`` or later, then they contain the required versions of the container agent and ``ecs-init`` . For more information, see `Amazon ECS-optimized Linux AMI <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-proxyconfiguration
        '''
        result = self._values.get("proxy_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnTaskDefinition.ProxyConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def requires_compatibilities(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The task launch types the task definition was validated against.

        To determine which task launch types the task definition is validated for, see the ``TaskDefinition$compatibilities`` parameter.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-requirescompatibilities
        '''
        result = self._values.get("requires_compatibilities")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def runtime_platform(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskDefinition.RuntimePlatformProperty, _IResolvable_a771d0ef]]:
        '''The operating system that your tasks definitions run on.

        A platform family is specified only for tasks using the Fargate launch type.

        When you specify a task definition in a service, this value must match the ``runtimePlatform`` value of the service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-runtimeplatform
        '''
        result = self._values.get("runtime_platform")
        return typing.cast(typing.Optional[typing.Union[CfnTaskDefinition.RuntimePlatformProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''The metadata that you apply to the task definition to help you categorize and organize them.

        Each tag consists of a key and an optional value. You define both of them.

        The following basic restrictions apply to tags:

        - Maximum number of tags per resource - 50
        - For each resource, each tag key must be unique, and each tag key can have only one value.
        - Maximum key length - 128 Unicode characters in UTF-8
        - Maximum value length - 256 Unicode characters in UTF-8
        - If your tagging schema is used across multiple services and resources, remember that other services may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable in UTF-8, and the following characters: + - = . _ : / @.
        - Tag keys and values are case-sensitive.
        - Do not use ``aws:`` , ``AWS:`` , or any upper or lowercase combination of such as a prefix for either keys or values as it is reserved for AWS use. You cannot edit or delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource limit.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def task_role_arn(self) -> typing.Optional[builtins.str]:
        '''The short name or full Amazon Resource Name (ARN) of the AWS Identity and Access Management role that grants containers in the task permission to call AWS APIs on your behalf.

        For more information, see `Amazon ECS Task Role <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-iam-roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        IAM roles for tasks on Windows require that the ``-EnableTaskIAMRole`` option is set when you launch the Amazon ECS-optimized Windows AMI. Your containers must also run some configuration code to use the feature. For more information, see `Windows IAM roles for tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/windows_task_IAM_roles.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-taskrolearn
        '''
        result = self._values.get("task_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.VolumeProperty, _IResolvable_a771d0ef]]]]:
        '''The list of data volume definitions for the task.

        For more information, see `Using data volumes in tasks <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_data_volumes.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
        .. epigraph::

           The ``host`` and ``sourcePath`` parameters aren't supported for tasks run on AWS Fargate .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskdefinition.html#cfn-ecs-taskdefinition-volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskDefinition.VolumeProperty, _IResolvable_a771d0ef]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTaskDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnTaskSet(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.CfnTaskSet",
):
    '''A CloudFormation ``AWS::ECS::TaskSet``.

    Create a task set in the specified cluster and service. This is used when a service uses the ``EXTERNAL`` deployment controller type. For more information, see `Amazon ECS Deployment Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

    :cloudformationResource: AWS::ECS::TaskSet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_ecs as ecs
        
        cfn_task_set = ecs.CfnTaskSet(self, "MyCfnTaskSet",
            cluster="cluster",
            service="service",
            task_definition="taskDefinition",
        
            # the properties below are optional
            external_id="externalId",
            launch_type="launchType",
            load_balancers=[ecs.CfnTaskSet.LoadBalancerProperty(
                container_name="containerName",
                container_port=123,
                load_balancer_name="loadBalancerName",
                target_group_arn="targetGroupArn"
            )],
            network_configuration=ecs.CfnTaskSet.NetworkConfigurationProperty(
                aws_vpc_configuration=ecs.CfnTaskSet.AwsVpcConfigurationProperty(
                    subnets=["subnets"],
        
                    # the properties below are optional
                    assign_public_ip="assignPublicIp",
                    security_groups=["securityGroups"]
                )
            ),
            platform_version="platformVersion",
            scale=ecs.CfnTaskSet.ScaleProperty(
                unit="unit",
                value=123
            ),
            service_registries=[ecs.CfnTaskSet.ServiceRegistryProperty(
                container_name="containerName",
                container_port=123,
                port=123,
                registry_arn="registryArn"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_definition: builtins.str,
        external_id: typing.Optional[builtins.str] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        scale: typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Create a new ``AWS::ECS::TaskSet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service to create the task set in.
        :param service: The short name or full Amazon Resource Name (ARN) of the service to create the task set in.
        :param task_definition: The task definition for the tasks in the task set to use.
        :param external_id: An optional non-unique tag that identifies this task set in external systems. If the task set is associated with a service discovery registry, the tasks in this task set will have the ``ECS_TASK_SET_EXTERNAL_ID`` AWS Cloud Map attribute set to the provided value.
        :param launch_type: The launch type that new tasks in the task set uses. For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If a ``launchType`` is specified, the ``capacityProviderStrategy`` parameter must be omitted.
        :param load_balancers: A load balancer object representing the load balancer to use with the task set. The supported load balancer types are either an Application Load Balancer or a Network Load Balancer.
        :param network_configuration: The network configuration for the task set.
        :param platform_version: The platform version that the tasks in the task set uses. A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used.
        :param scale: A floating-point percentage of your desired number of tasks to place and keep running in the task set.
        :param service_registries: The details of the service discovery registries to assign to this task set. For more information, see `Service Discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .
        '''
        props = CfnTaskSetProps(
            cluster=cluster,
            service=service,
            task_definition=task_definition,
            external_id=external_id,
            launch_type=launch_type,
            load_balancers=load_balancers,
            network_configuration=network_configuration,
            platform_version=platform_version,
            scale=scale,
            service_registries=service_registries,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''The ID of the task set.

        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cluster")
    def cluster(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service to create the task set in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-cluster
        '''
        return typing.cast(builtins.str, jsii.get(self, "cluster"))

    @cluster.setter
    def cluster(self, value: builtins.str) -> None:
        jsii.set(self, "cluster", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="service")
    def service(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the service to create the task set in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-service
        '''
        return typing.cast(builtins.str, jsii.get(self, "service"))

    @service.setter
    def service(self, value: builtins.str) -> None:
        jsii.set(self, "service", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> builtins.str:
        '''The task definition for the tasks in the task set to use.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-taskdefinition
        '''
        return typing.cast(builtins.str, jsii.get(self, "taskDefinition"))

    @task_definition.setter
    def task_definition(self, value: builtins.str) -> None:
        jsii.set(self, "taskDefinition", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="externalId")
    def external_id(self) -> typing.Optional[builtins.str]:
        '''An optional non-unique tag that identifies this task set in external systems.

        If the task set is associated with a service discovery registry, the tasks in this task set will have the ``ECS_TASK_SET_EXTERNAL_ID`` AWS Cloud Map attribute set to the provided value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-externalid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "externalId"))

    @external_id.setter
    def external_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "externalId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchType")
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''The launch type that new tasks in the task set uses.

        For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If a ``launchType`` is specified, the ``capacityProviderStrategy`` parameter must be omitted.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-launchtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchType"))

    @launch_type.setter
    def launch_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "launchType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="loadBalancers")
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]]:
        '''A load balancer object representing the load balancer to use with the task set.

        The supported load balancer types are either an Application Load Balancer or a Network Load Balancer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-loadbalancers
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "loadBalancers"))

    @load_balancers.setter
    def load_balancers(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.LoadBalancerProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "loadBalancers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkConfiguration")
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]]:
        '''The network configuration for the task set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-networkconfiguration
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]], jsii.get(self, "networkConfiguration"))

    @network_configuration.setter
    def network_configuration(
        self,
        value: typing.Optional[typing.Union["CfnTaskSet.NetworkConfigurationProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "networkConfiguration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platformVersion")
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''The platform version that the tasks in the task set uses.

        A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-platformversion
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "platformVersion"))

    @platform_version.setter
    def platform_version(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "platformVersion", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="scale")
    def scale(
        self,
    ) -> typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]]:
        '''A floating-point percentage of your desired number of tasks to place and keep running in the task set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-scale
        '''
        return typing.cast(typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]], jsii.get(self, "scale"))

    @scale.setter
    def scale(
        self,
        value: typing.Optional[typing.Union["CfnTaskSet.ScaleProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "scale", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceRegistries")
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]]:
        '''The details of the service discovery registries to assign to this task set.

        For more information, see `Service Discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-serviceregistries
        '''
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]], jsii.get(self, "serviceRegistries"))

    @service_registries.setter
    def service_registries(
        self,
        value: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnTaskSet.ServiceRegistryProperty", _IResolvable_a771d0ef]]]],
    ) -> None:
        jsii.set(self, "serviceRegistries", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.AwsVpcConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "subnets": "subnets",
            "assign_public_ip": "assignPublicIp",
            "security_groups": "securityGroups",
        },
    )
    class AwsVpcConfigurationProperty:
        def __init__(
            self,
            *,
            subnets: typing.Sequence[builtins.str],
            assign_public_ip: typing.Optional[builtins.str] = None,
            security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''The networking details for a task.

            :param subnets: The IDs of the subnets associated with the task or service. There's a limit of 16 subnets that can be specified per ``AwsVpcConfiguration`` . .. epigraph:: All specified subnets must be from the same VPC.
            :param assign_public_ip: Whether the task's elastic network interface receives a public IP address. The default value is ``DISABLED`` .
            :param security_groups: The IDs of the security groups associated with the task or service. If you don't specify a security group, the default security group for the VPC is used. There's a limit of 5 security groups that can be specified per ``AwsVpcConfiguration`` . .. epigraph:: All specified security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                aws_vpc_configuration_property = ecs.CfnTaskSet.AwsVpcConfigurationProperty(
                    subnets=["subnets"],
                
                    # the properties below are optional
                    assign_public_ip="assignPublicIp",
                    security_groups=["securityGroups"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "subnets": subnets,
            }
            if assign_public_ip is not None:
                self._values["assign_public_ip"] = assign_public_ip
            if security_groups is not None:
                self._values["security_groups"] = security_groups

        @builtins.property
        def subnets(self) -> typing.List[builtins.str]:
            '''The IDs of the subnets associated with the task or service.

            There's a limit of 16 subnets that can be specified per ``AwsVpcConfiguration`` .
            .. epigraph::

               All specified subnets must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html#cfn-ecs-taskset-awsvpcconfiguration-subnets
            '''
            result = self._values.get("subnets")
            assert result is not None, "Required property 'subnets' is missing"
            return typing.cast(typing.List[builtins.str], result)

        @builtins.property
        def assign_public_ip(self) -> typing.Optional[builtins.str]:
            '''Whether the task's elastic network interface receives a public IP address.

            The default value is ``DISABLED`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html#cfn-ecs-taskset-awsvpcconfiguration-assignpublicip
            '''
            result = self._values.get("assign_public_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The IDs of the security groups associated with the task or service.

            If you don't specify a security group, the default security group for the VPC is used. There's a limit of 5 security groups that can be specified per ``AwsVpcConfiguration`` .
            .. epigraph::

               All specified security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-awsvpcconfiguration.html#cfn-ecs-taskset-awsvpcconfiguration-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsVpcConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.LoadBalancerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "load_balancer_name": "loadBalancerName",
            "target_group_arn": "targetGroupArn",
        },
    )
    class LoadBalancerProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            load_balancer_name: typing.Optional[builtins.str] = None,
            target_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''Details on the load balancer or load balancers to use with a task set.

            :param container_name: The name of the container (as it appears in a container definition) to associate with the load balancer.
            :param container_port: The port on the container to associate with the load balancer. This port must correspond to a ``containerPort`` in the task definition the tasks in the service are using. For tasks that use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the ``hostPort`` of the port mapping.
            :param load_balancer_name: The name of the load balancer to associate with the Amazon ECS service or task set. A load balancer name is only specified when using a Classic Load Balancer. If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should be omitted.
            :param target_group_arn: The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a service or task set. A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer. If you're using a Classic Load Balancer, omit the target group ARN. For services using the ``ECS`` deployment controller, you can specify one or multiple target groups. For more information, see `Registering Multiple Target Groups with a Service <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html>`_ in the *Amazon Elastic Container Service Developer Guide* . For services using the ``CODE_DEPLOY`` deployment controller, you're required to define two target groups for the load balancer. For more information, see `Blue/Green Deployment with CodeDeploy <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html>`_ in the *Amazon Elastic Container Service Developer Guide* . .. epigraph:: If your service's task definition uses the ``awsvpc`` network mode, you must choose ``ip`` as the target type, not ``instance`` . Do this when creating your target groups because tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                load_balancer_property = ecs.CfnTaskSet.LoadBalancerProperty(
                    container_name="containerName",
                    container_port=123,
                    load_balancer_name="loadBalancerName",
                    target_group_arn="targetGroupArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if load_balancer_name is not None:
                self._values["load_balancer_name"] = load_balancer_name
            if target_group_arn is not None:
                self._values["target_group_arn"] = target_group_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The name of the container (as it appears in a container definition) to associate with the load balancer.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''The port on the container to associate with the load balancer.

            This port must correspond to a ``containerPort`` in the task definition the tasks in the service are using. For tasks that use the EC2 launch type, the container instance they're launched on must allow ingress traffic on the ``hostPort`` of the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def load_balancer_name(self) -> typing.Optional[builtins.str]:
            '''The name of the load balancer to associate with the Amazon ECS service or task set.

            A load balancer name is only specified when using a Classic Load Balancer. If you are using an Application Load Balancer or a Network Load Balancer the load balancer name parameter should be omitted.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-loadbalancername
            '''
            result = self._values.get("load_balancer_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def target_group_arn(self) -> typing.Optional[builtins.str]:
            '''The full Amazon Resource Name (ARN) of the Elastic Load Balancing target group or groups associated with a service or task set.

            A target group ARN is only specified when using an Application Load Balancer or Network Load Balancer. If you're using a Classic Load Balancer, omit the target group ARN.

            For services using the ``ECS`` deployment controller, you can specify one or multiple target groups. For more information, see `Registering Multiple Target Groups with a Service <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/register-multiple-targetgroups.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

            For services using the ``CODE_DEPLOY`` deployment controller, you're required to define two target groups for the load balancer. For more information, see `Blue/Green Deployment with CodeDeploy <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-bluegreen.html>`_ in the *Amazon Elastic Container Service Developer Guide* .
            .. epigraph::

               If your service's task definition uses the ``awsvpc`` network mode, you must choose ``ip`` as the target type, not ``instance`` . Do this when creating your target groups because tasks that use the ``awsvpc`` network mode are associated with an elastic network interface, not an Amazon EC2 instance. This network mode is required for the Fargate launch type.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-loadbalancer.html#cfn-ecs-taskset-loadbalancer-targetgrouparn
            '''
            result = self._values.get("target_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoadBalancerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.NetworkConfigurationProperty",
        jsii_struct_bases=[],
        name_mapping={"aws_vpc_configuration": "awsVpcConfiguration"},
    )
    class NetworkConfigurationProperty:
        def __init__(
            self,
            *,
            aws_vpc_configuration: typing.Optional[typing.Union["CfnTaskSet.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The network configuration for a task.

            :param aws_vpc_configuration: The VPC subnets and security groups that are associated with a task. .. epigraph:: All specified subnets and security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-networkconfiguration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                network_configuration_property = ecs.CfnTaskSet.NetworkConfigurationProperty(
                    aws_vpc_configuration=ecs.CfnTaskSet.AwsVpcConfigurationProperty(
                        subnets=["subnets"],
                
                        # the properties below are optional
                        assign_public_ip="assignPublicIp",
                        security_groups=["securityGroups"]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if aws_vpc_configuration is not None:
                self._values["aws_vpc_configuration"] = aws_vpc_configuration

        @builtins.property
        def aws_vpc_configuration(
            self,
        ) -> typing.Optional[typing.Union["CfnTaskSet.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]]:
            '''The VPC subnets and security groups that are associated with a task.

            .. epigraph::

               All specified subnets and security groups must be from the same VPC.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-networkconfiguration.html#cfn-ecs-taskset-networkconfiguration-awsvpcconfiguration
            '''
            result = self._values.get("aws_vpc_configuration")
            return typing.cast(typing.Optional[typing.Union["CfnTaskSet.AwsVpcConfigurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkConfigurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.ScaleProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class ScaleProperty:
        def __init__(
            self,
            *,
            unit: typing.Optional[builtins.str] = None,
            value: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''A floating-point percentage of the desired number of tasks to place and keep running in the task set.

            :param unit: The unit of measure for the scale value.
            :param value: The value, specified as a percent total of a service's ``desiredCount`` , to scale the task set. Accepted values are numbers between 0 and 100.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-scale.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                scale_property = ecs.CfnTaskSet.ScaleProperty(
                    unit="unit",
                    value=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if unit is not None:
                self._values["unit"] = unit
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def unit(self) -> typing.Optional[builtins.str]:
            '''The unit of measure for the scale value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-scale.html#cfn-ecs-taskset-scale-unit
            '''
            result = self._values.get("unit")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[jsii.Number]:
            '''The value, specified as a percent total of a service's ``desiredCount`` , to scale the task set.

            Accepted values are numbers between 0 and 100.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-scale.html#cfn-ecs-taskset-scale-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ScaleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_ecs.CfnTaskSet.ServiceRegistryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "container_name": "containerName",
            "container_port": "containerPort",
            "port": "port",
            "registry_arn": "registryArn",
        },
    )
    class ServiceRegistryProperty:
        def __init__(
            self,
            *,
            container_name: typing.Optional[builtins.str] = None,
            container_port: typing.Optional[jsii.Number] = None,
            port: typing.Optional[jsii.Number] = None,
            registry_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''The details for the service registry.

            Each service may be associated with one service registry. Multiple service registries for each service are not supported.

            When you add, update, or remove the service registries configuration, Amazon ECS starts a new deployment. New tasks are registered and deregistered to the updated service registry configuration.

            :param container_name: The container name value to be used for your service discovery service. It's already specified in the task definition. If the task definition that your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition that your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.
            :param container_port: The port value to be used for your service discovery service. It's already specified in the task definition. If the task definition your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.
            :param port: The port value used if your service discovery service specified an SRV record. This field might be used if both the ``awsvpc`` network mode and SRV records are used.
            :param registry_arn: The Amazon Resource Name (ARN) of the service registry. The currently supported service registry is AWS Cloud Map . For more information, see `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_ecs as ecs
                
                service_registry_property = ecs.CfnTaskSet.ServiceRegistryProperty(
                    container_name="containerName",
                    container_port=123,
                    port=123,
                    registry_arn="registryArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if container_name is not None:
                self._values["container_name"] = container_name
            if container_port is not None:
                self._values["container_port"] = container_port
            if port is not None:
                self._values["port"] = port
            if registry_arn is not None:
                self._values["registry_arn"] = registry_arn

        @builtins.property
        def container_name(self) -> typing.Optional[builtins.str]:
            '''The container name value to be used for your service discovery service.

            It's already specified in the task definition. If the task definition that your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition that your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-containername
            '''
            result = self._values.get("container_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def container_port(self) -> typing.Optional[jsii.Number]:
            '''The port value to be used for your service discovery service.

            It's already specified in the task definition. If the task definition your service task specifies uses the ``bridge`` or ``host`` network mode, you must specify a ``containerName`` and ``containerPort`` combination from the task definition. If the task definition your service task specifies uses the ``awsvpc`` network mode and a type SRV DNS record is used, you must specify either a ``containerName`` and ``containerPort`` combination or a ``port`` value. However, you can't specify both.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-containerport
            '''
            result = self._values.get("container_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The port value used if your service discovery service specified an SRV record.

            This field might be used if both the ``awsvpc`` network mode and SRV records are used.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def registry_arn(self) -> typing.Optional[builtins.str]:
            '''The Amazon Resource Name (ARN) of the service registry.

            The currently supported service registry is AWS Cloud Map . For more information, see `CreateService <https://docs.aws.amazon.com/cloud-map/latest/api/API_CreateService.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ecs-taskset-serviceregistry.html#cfn-ecs-taskset-serviceregistry-registryarn
            '''
            result = self._values.get("registry_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceRegistryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CfnTaskSetProps",
    jsii_struct_bases=[],
    name_mapping={
        "cluster": "cluster",
        "service": "service",
        "task_definition": "taskDefinition",
        "external_id": "externalId",
        "launch_type": "launchType",
        "load_balancers": "loadBalancers",
        "network_configuration": "networkConfiguration",
        "platform_version": "platformVersion",
        "scale": "scale",
        "service_registries": "serviceRegistries",
    },
)
class CfnTaskSetProps:
    def __init__(
        self,
        *,
        cluster: builtins.str,
        service: builtins.str,
        task_definition: builtins.str,
        external_id: typing.Optional[builtins.str] = None,
        launch_type: typing.Optional[builtins.str] = None,
        load_balancers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskSet.LoadBalancerProperty, _IResolvable_a771d0ef]]]] = None,
        network_configuration: typing.Optional[typing.Union[CfnTaskSet.NetworkConfigurationProperty, _IResolvable_a771d0ef]] = None,
        platform_version: typing.Optional[builtins.str] = None,
        scale: typing.Optional[typing.Union[CfnTaskSet.ScaleProperty, _IResolvable_a771d0ef]] = None,
        service_registries: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union[CfnTaskSet.ServiceRegistryProperty, _IResolvable_a771d0ef]]]] = None,
    ) -> None:
        '''Properties for defining a ``CfnTaskSet``.

        :param cluster: The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service to create the task set in.
        :param service: The short name or full Amazon Resource Name (ARN) of the service to create the task set in.
        :param task_definition: The task definition for the tasks in the task set to use.
        :param external_id: An optional non-unique tag that identifies this task set in external systems. If the task set is associated with a service discovery registry, the tasks in this task set will have the ``ECS_TASK_SET_EXTERNAL_ID`` AWS Cloud Map attribute set to the provided value.
        :param launch_type: The launch type that new tasks in the task set uses. For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* . If a ``launchType`` is specified, the ``capacityProviderStrategy`` parameter must be omitted.
        :param load_balancers: A load balancer object representing the load balancer to use with the task set. The supported load balancer types are either an Application Load Balancer or a Network Load Balancer.
        :param network_configuration: The network configuration for the task set.
        :param platform_version: The platform version that the tasks in the task set uses. A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used.
        :param scale: A floating-point percentage of your desired number of tasks to place and keep running in the task set.
        :param service_registries: The details of the service discovery registries to assign to this task set. For more information, see `Service Discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            
            cfn_task_set_props = ecs.CfnTaskSetProps(
                cluster="cluster",
                service="service",
                task_definition="taskDefinition",
            
                # the properties below are optional
                external_id="externalId",
                launch_type="launchType",
                load_balancers=[ecs.CfnTaskSet.LoadBalancerProperty(
                    container_name="containerName",
                    container_port=123,
                    load_balancer_name="loadBalancerName",
                    target_group_arn="targetGroupArn"
                )],
                network_configuration=ecs.CfnTaskSet.NetworkConfigurationProperty(
                    aws_vpc_configuration=ecs.CfnTaskSet.AwsVpcConfigurationProperty(
                        subnets=["subnets"],
            
                        # the properties below are optional
                        assign_public_ip="assignPublicIp",
                        security_groups=["securityGroups"]
                    )
                ),
                platform_version="platformVersion",
                scale=ecs.CfnTaskSet.ScaleProperty(
                    unit="unit",
                    value=123
                ),
                service_registries=[ecs.CfnTaskSet.ServiceRegistryProperty(
                    container_name="containerName",
                    container_port=123,
                    port=123,
                    registry_arn="registryArn"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "cluster": cluster,
            "service": service,
            "task_definition": task_definition,
        }
        if external_id is not None:
            self._values["external_id"] = external_id
        if launch_type is not None:
            self._values["launch_type"] = launch_type
        if load_balancers is not None:
            self._values["load_balancers"] = load_balancers
        if network_configuration is not None:
            self._values["network_configuration"] = network_configuration
        if platform_version is not None:
            self._values["platform_version"] = platform_version
        if scale is not None:
            self._values["scale"] = scale
        if service_registries is not None:
            self._values["service_registries"] = service_registries

    @builtins.property
    def cluster(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service to create the task set in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-cluster
        '''
        result = self._values.get("cluster")
        assert result is not None, "Required property 'cluster' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''The short name or full Amazon Resource Name (ARN) of the service to create the task set in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-service
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def task_definition(self) -> builtins.str:
        '''The task definition for the tasks in the task set to use.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-taskdefinition
        '''
        result = self._values.get("task_definition")
        assert result is not None, "Required property 'task_definition' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def external_id(self) -> typing.Optional[builtins.str]:
        '''An optional non-unique tag that identifies this task set in external systems.

        If the task set is associated with a service discovery registry, the tasks in this task set will have the ``ECS_TASK_SET_EXTERNAL_ID`` AWS Cloud Map attribute set to the provided value.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-externalid
        '''
        result = self._values.get("external_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_type(self) -> typing.Optional[builtins.str]:
        '''The launch type that new tasks in the task set uses.

        For more information, see `Amazon ECS Launch Types <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/launch_types.html>`_ in the *Amazon Elastic Container Service Developer Guide* .

        If a ``launchType`` is specified, the ``capacityProviderStrategy`` parameter must be omitted.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-launchtype
        '''
        result = self._values.get("launch_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def load_balancers(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.LoadBalancerProperty, _IResolvable_a771d0ef]]]]:
        '''A load balancer object representing the load balancer to use with the task set.

        The supported load balancer types are either an Application Load Balancer or a Network Load Balancer.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-loadbalancers
        '''
        result = self._values.get("load_balancers")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.LoadBalancerProperty, _IResolvable_a771d0ef]]]], result)

    @builtins.property
    def network_configuration(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskSet.NetworkConfigurationProperty, _IResolvable_a771d0ef]]:
        '''The network configuration for the task set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-networkconfiguration
        '''
        result = self._values.get("network_configuration")
        return typing.cast(typing.Optional[typing.Union[CfnTaskSet.NetworkConfigurationProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def platform_version(self) -> typing.Optional[builtins.str]:
        '''The platform version that the tasks in the task set uses.

        A platform version is specified only for tasks using the Fargate launch type. If one isn't specified, the ``LATEST`` platform version is used.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-platformversion
        '''
        result = self._values.get("platform_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def scale(
        self,
    ) -> typing.Optional[typing.Union[CfnTaskSet.ScaleProperty, _IResolvable_a771d0ef]]:
        '''A floating-point percentage of your desired number of tasks to place and keep running in the task set.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-scale
        '''
        result = self._values.get("scale")
        return typing.cast(typing.Optional[typing.Union[CfnTaskSet.ScaleProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def service_registries(
        self,
    ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.ServiceRegistryProperty, _IResolvable_a771d0ef]]]]:
        '''The details of the service discovery registries to assign to this task set.

        For more information, see `Service Discovery <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/service-discovery.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecs-taskset.html#cfn-ecs-taskset-serviceregistries
        '''
        result = self._values.get("service_registries")
        return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union[CfnTaskSet.ServiceRegistryProperty, _IResolvable_a771d0ef]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnTaskSetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CloudMapNamespaceOptions",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "type": "type", "vpc": "vpc"},
)
class CloudMapNamespaceOptions:
    def __init__(
        self,
        *,
        name: builtins.str,
        type: typing.Optional[_NamespaceType_db55a104] = None,
        vpc: typing.Optional[_IVpc_6d1f76c4] = None,
    ) -> None:
        '''(experimental) The options for creating an AWS Cloud Map namespace.

        :param name: (experimental) The name of the namespace, such as example.com.
        :param type: (experimental) The type of CloudMap Namespace to create. Default: PrivateDns
        :param vpc: (experimental) The VPC to associate the namespace with. This property is required for private DNS namespaces. Default: VPC of the cluster for Private DNS Namespace, otherwise none

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ec2 as ec2
            from monocdk import aws_ecs as ecs
            from monocdk import aws_servicediscovery as servicediscovery
            
            # vpc: ec2.Vpc
            
            cloud_map_namespace_options = ecs.CloudMapNamespaceOptions(
                name="name",
            
                # the properties below are optional
                type=servicediscovery.NamespaceType.HTTP,
                vpc=vpc
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
        }
        if type is not None:
            self._values["type"] = type
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def name(self) -> builtins.str:
        '''(experimental) The name of the namespace, such as example.com.

        :stability: experimental
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> typing.Optional[_NamespaceType_db55a104]:
        '''(experimental) The type of CloudMap Namespace to create.

        :default: PrivateDns

        :stability: experimental
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[_NamespaceType_db55a104], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_6d1f76c4]:
        '''(experimental) The VPC to associate the namespace with.

        This property is required for private DNS namespaces.

        :default: VPC of the cluster for Private DNS Namespace, otherwise none

        :stability: experimental
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_6d1f76c4], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CloudMapNamespaceOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CloudMapOptions",
    jsii_struct_bases=[],
    name_mapping={
        "cloud_map_namespace": "cloudMapNamespace",
        "container": "container",
        "container_port": "containerPort",
        "dns_record_type": "dnsRecordType",
        "dns_ttl": "dnsTtl",
        "failure_threshold": "failureThreshold",
        "name": "name",
    },
)
class CloudMapOptions:
    def __init__(
        self,
        *,
        cloud_map_namespace: typing.Optional[_INamespace_eb21e42e] = None,
        container: typing.Optional["ContainerDefinition"] = None,
        container_port: typing.Optional[jsii.Number] = None,
        dns_record_type: typing.Optional[_DnsRecordType_a28443d0] = None,
        dns_ttl: typing.Optional[_Duration_070aa057] = None,
        failure_threshold: typing.Optional[jsii.Number] = None,
        name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The options to enabling AWS Cloud Map for an Amazon ECS service.

        :param cloud_map_namespace: (experimental) The service discovery namespace for the Cloud Map service to attach to the ECS service. Default: - the defaultCloudMapNamespace associated to the cluster
        :param container: (experimental) The container to point to for a SRV record. Default: - the task definition's default container
        :param container_port: (experimental) The port to point to for a SRV record. Default: - the default port of the task definition's default container
        :param dns_record_type: (experimental) The DNS record type that you want AWS Cloud Map to create. The supported record types are A or SRV. Default: - DnsRecordType.A if TaskDefinition.networkMode = AWS_VPC, otherwise DnsRecordType.SRV
        :param dns_ttl: (experimental) The amount of time that you want DNS resolvers to cache the settings for this record. Default: Duration.minutes(1)
        :param failure_threshold: (experimental) The number of 30-second intervals that you want Cloud Map to wait after receiving an UpdateInstanceCustomHealthStatus request before it changes the health status of a service instance. NOTE: This is used for HealthCheckCustomConfig
        :param name: (experimental) The name of the Cloud Map service to attach to the ECS service. Default: CloudFormation-generated name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # task_definition: ecs.TaskDefinition
            # cluster: ecs.Cluster
            
            
            service = ecs.Ec2Service(self, "Service",
                cluster=cluster,
                task_definition=task_definition,
                cloud_map_options=autoscaling.aws_ecs.CloudMapOptions(
                    # Create A records - useful for AWSVPC network mode.
                    dns_record_type=cloudmap.DnsRecordType.A
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if cloud_map_namespace is not None:
            self._values["cloud_map_namespace"] = cloud_map_namespace
        if container is not None:
            self._values["container"] = container
        if container_port is not None:
            self._values["container_port"] = container_port
        if dns_record_type is not None:
            self._values["dns_record_type"] = dns_record_type
        if dns_ttl is not None:
            self._values["dns_ttl"] = dns_ttl
        if failure_threshold is not None:
            self._values["failure_threshold"] = failure_threshold
        if name is not None:
            self._values["name"] = name

    @builtins.property
    def cloud_map_namespace(self) -> typing.Optional[_INamespace_eb21e42e]:
        '''(experimental) The service discovery namespace for the Cloud Map service to attach to the ECS service.

        :default: - the defaultCloudMapNamespace associated to the cluster

        :stability: experimental
        '''
        result = self._values.get("cloud_map_namespace")
        return typing.cast(typing.Optional[_INamespace_eb21e42e], result)

    @builtins.property
    def container(self) -> typing.Optional["ContainerDefinition"]:
        '''(experimental) The container to point to for a SRV record.

        :default: - the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container")
        return typing.cast(typing.Optional["ContainerDefinition"], result)

    @builtins.property
    def container_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The port to point to for a SRV record.

        :default: - the default port of the task definition's default container

        :stability: experimental
        '''
        result = self._values.get("container_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def dns_record_type(self) -> typing.Optional[_DnsRecordType_a28443d0]:
        '''(experimental) The DNS record type that you want AWS Cloud Map to create.

        The supported record types are A or SRV.

        :default: - DnsRecordType.A if TaskDefinition.networkMode = AWS_VPC, otherwise DnsRecordType.SRV

        :stability: experimental
        '''
        result = self._values.get("dns_record_type")
        return typing.cast(typing.Optional[_DnsRecordType_a28443d0], result)

    @builtins.property
    def dns_ttl(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The amount of time that you want DNS resolvers to cache the settings for this record.

        :default: Duration.minutes(1)

        :stability: experimental
        '''
        result = self._values.get("dns_ttl")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def failure_threshold(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of 30-second intervals that you want Cloud Map to wait after receiving an UpdateInstanceCustomHealthStatus request before it changes the health status of a service instance.

        NOTE: This is used for HealthCheckCustomConfig

        :stability: experimental
        '''
        result = self._values.get("failure_threshold")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the Cloud Map service to attach to the ECS service.

        :default: CloudFormation-generated name

        :stability: experimental
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CloudMapOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ClusterAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "cluster_name": "clusterName",
        "security_groups": "securityGroups",
        "vpc": "vpc",
        "autoscaling_group": "autoscalingGroup",
        "cluster_arn": "clusterArn",
        "default_cloud_map_namespace": "defaultCloudMapNamespace",
        "execute_command_configuration": "executeCommandConfiguration",
        "has_ec2_capacity": "hasEc2Capacity",
    },
)
class ClusterAttributes:
    def __init__(
        self,
        *,
        cluster_name: builtins.str,
        security_groups: typing.Sequence[_ISecurityGroup_cdbba9d3],
        vpc: _IVpc_6d1f76c4,
        autoscaling_group: typing.Optional[_IAutoScalingGroup_352723c6] = None,
        cluster_arn: typing.Optional[builtins.str] = None,
        default_cloud_map_namespace: typing.Optional[_INamespace_eb21e42e] = None,
        execute_command_configuration: typing.Optional["ExecuteCommandConfiguration"] = None,
        has_ec2_capacity: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) The properties to import from the ECS cluster.

        :param cluster_name: (experimental) The name of the cluster.
        :param security_groups: (experimental) The security groups associated with the container instances registered to the cluster.
        :param vpc: (experimental) The VPC associated with the cluster.
        :param autoscaling_group: (experimental) Autoscaling group added to the cluster if capacity is added. Default: - No default autoscaling group
        :param cluster_arn: (experimental) The Amazon Resource Name (ARN) that identifies the cluster. Default: Derived from clusterName
        :param default_cloud_map_namespace: (experimental) The AWS Cloud Map namespace to associate with the cluster. Default: - No default namespace
        :param execute_command_configuration: (experimental) The execute command configuration for the cluster. Default: - none.
        :param has_ec2_capacity: (experimental) Specifies whether the cluster has EC2 instance capacity. Default: true

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_autoscaling as autoscaling
            from monocdk import aws_ec2 as ec2
            from monocdk import aws_ecs as ecs
            from monocdk import aws_kms as kms
            from monocdk import aws_logs as logs
            from monocdk import aws_s3 as s3
            from monocdk import aws_servicediscovery as servicediscovery
            
            # auto_scaling_group: autoscaling.AutoScalingGroup
            # bucket: s3.Bucket
            # key: kms.Key
            # log_group: logs.LogGroup
            # namespace: servicediscovery.INamespace
            # security_group: ec2.SecurityGroup
            # vpc: ec2.Vpc
            
            cluster_attributes = ecs.ClusterAttributes(
                cluster_name="clusterName",
                security_groups=[security_group],
                vpc=vpc,
            
                # the properties below are optional
                autoscaling_group=auto_scaling_group,
                cluster_arn="clusterArn",
                default_cloud_map_namespace=namespace,
                execute_command_configuration=ecs.ExecuteCommandConfiguration(
                    kms_key=key,
                    log_configuration=ecs.ExecuteCommandLogConfiguration(
                        cloud_watch_encryption_enabled=False,
                        cloud_watch_log_group=log_group,
                        s3_bucket=bucket,
                        s3_encryption_enabled=False,
                        s3_key_prefix="s3KeyPrefix"
                    ),
                    logging=ecs.ExecuteCommandLogging.NONE
                ),
                has_ec2_capacity=False
            )
        '''
        if isinstance(execute_command_configuration, dict):
            execute_command_configuration = ExecuteCommandConfiguration(**execute_command_configuration)
        self._values: typing.Dict[str, typing.Any] = {
            "cluster_name": cluster_name,
            "security_groups": security_groups,
            "vpc": vpc,
        }
        if autoscaling_group is not None:
            self._values["autoscaling_group"] = autoscaling_group
        if cluster_arn is not None:
            self._values["cluster_arn"] = cluster_arn
        if default_cloud_map_namespace is not None:
            self._values["default_cloud_map_namespace"] = default_cloud_map_namespace
        if execute_command_configuration is not None:
            self._values["execute_command_configuration"] = execute_command_configuration
        if has_ec2_capacity is not None:
            self._values["has_ec2_capacity"] = has_ec2_capacity

    @builtins.property
    def cluster_name(self) -> builtins.str:
        '''(experimental) The name of the cluster.

        :stability: experimental
        '''
        result = self._values.get("cluster_name")
        assert result is not None, "Required property 'cluster_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def security_groups(self) -> typing.List[_ISecurityGroup_cdbba9d3]:
        '''(experimental) The security groups associated with the container instances registered to the cluster.

        :stability: experimental
        '''
        result = self._values.get("security_groups")
        assert result is not None, "Required property 'security_groups' is missing"
        return typing.cast(typing.List[_ISecurityGroup_cdbba9d3], result)

    @builtins.property
    def vpc(self) -> _IVpc_6d1f76c4:
        '''(experimental) The VPC associated with the cluster.

        :stability: experimental
        '''
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return typing.cast(_IVpc_6d1f76c4, result)

    @builtins.property
    def autoscaling_group(self) -> typing.Optional[_IAutoScalingGroup_352723c6]:
        '''(experimental) Autoscaling group added to the cluster if capacity is added.

        :default: - No default autoscaling group

        :stability: experimental
        '''
        result = self._values.get("autoscaling_group")
        return typing.cast(typing.Optional[_IAutoScalingGroup_352723c6], result)

    @builtins.property
    def cluster_arn(self) -> typing.Optional[builtins.str]:
        '''(experimental) The Amazon Resource Name (ARN) that identifies the cluster.

        :default: Derived from clusterName

        :stability: experimental
        '''
        result = self._values.get("cluster_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def default_cloud_map_namespace(self) -> typing.Optional[_INamespace_eb21e42e]:
        '''(experimental) The AWS Cloud Map namespace to associate with the cluster.

        :default: - No default namespace

        :stability: experimental
        '''
        result = self._values.get("default_cloud_map_namespace")
        return typing.cast(typing.Optional[_INamespace_eb21e42e], result)

    @builtins.property
    def execute_command_configuration(
        self,
    ) -> typing.Optional["ExecuteCommandConfiguration"]:
        '''(experimental) The execute command configuration for the cluster.

        :default: - none.

        :stability: experimental
        '''
        result = self._values.get("execute_command_configuration")
        return typing.cast(typing.Optional["ExecuteCommandConfiguration"], result)

    @builtins.property
    def has_ec2_capacity(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether the cluster has EC2 instance capacity.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("has_ec2_capacity")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ClusterProps",
    jsii_struct_bases=[],
    name_mapping={
        "capacity": "capacity",
        "capacity_providers": "capacityProviders",
        "cluster_name": "clusterName",
        "container_insights": "containerInsights",
        "default_cloud_map_namespace": "defaultCloudMapNamespace",
        "enable_fargate_capacity_providers": "enableFargateCapacityProviders",
        "execute_command_configuration": "executeCommandConfiguration",
        "vpc": "vpc",
    },
)
class ClusterProps:
    def __init__(
        self,
        *,
        capacity: typing.Optional[AddCapacityOptions] = None,
        capacity_providers: typing.Optional[typing.Sequence[builtins.str]] = None,
        cluster_name: typing.Optional[builtins.str] = None,
        container_insights: typing.Optional[builtins.bool] = None,
        default_cloud_map_namespace: typing.Optional[CloudMapNamespaceOptions] = None,
        enable_fargate_capacity_providers: typing.Optional[builtins.bool] = None,
        execute_command_configuration: typing.Optional["ExecuteCommandConfiguration"] = None,
        vpc: typing.Optional[_IVpc_6d1f76c4] = None,
    ) -> None:
        '''(experimental) The properties used to define an ECS cluster.

        :param capacity: (experimental) The ec2 capacity to add to the cluster. Default: - no EC2 capacity will be added, you can use ``addCapacity`` to add capacity later.
        :param capacity_providers: (deprecated) The capacity providers to add to the cluster. Default: - None. Currently only FARGATE and FARGATE_SPOT are supported.
        :param cluster_name: (experimental) The name for the cluster. Default: CloudFormation-generated name
        :param container_insights: (experimental) If true CloudWatch Container Insights will be enabled for the cluster. Default: - Container Insights will be disabled for this cluser.
        :param default_cloud_map_namespace: (experimental) The service discovery namespace created in this cluster. Default: - no service discovery namespace created, you can use ``addDefaultCloudMapNamespace`` to add a default service discovery namespace later.
        :param enable_fargate_capacity_providers: (experimental) Whether to enable Fargate Capacity Providers. Default: false
        :param execute_command_configuration: (experimental) The execute command configuration for the cluster. Default: - no configuration will be provided.
        :param vpc: (experimental) The VPC where your ECS instances will be running or your ENIs will be deployed. Default: - creates a new VPC with two AZs

        :stability: experimental
        :exampleMetadata: infused

        Example::

            vpc = ec2.Vpc.from_lookup(self, "Vpc",
                is_default=True
            )
            
            cluster = ecs.Cluster(self, "FargateCluster", vpc=vpc)
            
            task_definition = ecs.TaskDefinition(self, "TD",
                memory_mi_b="512",
                cpu="256",
                compatibility=ecs.Compatibility.FARGATE
            )
            
            container_definition = task_definition.add_container("TheContainer",
                image=ecs.ContainerImage.from_registry("foo/bar"),
                memory_limit_mi_b=256
            )
            
            run_task = tasks.EcsRunTask(self, "RunFargate",
                integration_pattern=sfn.IntegrationPattern.RUN_JOB,
                cluster=cluster,
                task_definition=task_definition,
                assign_public_ip=True,
                container_overrides=[dynamodb.aws_stepfunctions_tasks.ContainerOverride(
                    container_definition=container_definition,
                    environment=[dynamodb.aws_stepfunctions_tasks.TaskEnvironmentVariable(name="SOME_KEY", value=sfn.JsonPath.string_at("$.SomeKey"))]
                )],
                launch_target=tasks.EcsFargateLaunchTarget()
            )
        '''
        if isinstance(capacity, dict):
            capacity = AddCapacityOptions(**capacity)
        if isinstance(default_cloud_map_namespace, dict):
            default_cloud_map_namespace = CloudMapNamespaceOptions(**default_cloud_map_namespace)
        if isinstance(execute_command_configuration, dict):
            execute_command_configuration = ExecuteCommandConfiguration(**execute_command_configuration)
        self._values: typing.Dict[str, typing.Any] = {}
        if capacity is not None:
            self._values["capacity"] = capacity
        if capacity_providers is not None:
            self._values["capacity_providers"] = capacity_providers
        if cluster_name is not None:
            self._values["cluster_name"] = cluster_name
        if container_insights is not None:
            self._values["container_insights"] = container_insights
        if default_cloud_map_namespace is not None:
            self._values["default_cloud_map_namespace"] = default_cloud_map_namespace
        if enable_fargate_capacity_providers is not None:
            self._values["enable_fargate_capacity_providers"] = enable_fargate_capacity_providers
        if execute_command_configuration is not None:
            self._values["execute_command_configuration"] = execute_command_configuration
        if vpc is not None:
            self._values["vpc"] = vpc

    @builtins.property
    def capacity(self) -> typing.Optional[AddCapacityOptions]:
        '''(experimental) The ec2 capacity to add to the cluster.

        :default: - no EC2 capacity will be added, you can use ``addCapacity`` to add capacity later.

        :stability: experimental
        '''
        result = self._values.get("capacity")
        return typing.cast(typing.Optional[AddCapacityOptions], result)

    @builtins.property
    def capacity_providers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(deprecated) The capacity providers to add to the cluster.

        :default: - None. Currently only FARGATE and FARGATE_SPOT are supported.

        :deprecated: Use {@link ClusterProps.enableFargateCapacityProviders} instead.

        :stability: deprecated
        '''
        result = self._values.get("capacity_providers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def cluster_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name for the cluster.

        :default: CloudFormation-generated name

        :stability: experimental
        '''
        result = self._values.get("cluster_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def container_insights(self) -> typing.Optional[builtins.bool]:
        '''(experimental) If true CloudWatch Container Insights will be enabled for the cluster.

        :default: - Container Insights will be disabled for this cluser.

        :stability: experimental
        '''
        result = self._values.get("container_insights")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def default_cloud_map_namespace(self) -> typing.Optional[CloudMapNamespaceOptions]:
        '''(experimental) The service discovery namespace created in this cluster.

        :default:

        - no service discovery namespace created, you can use ``addDefaultCloudMapNamespace`` to add a
        default service discovery namespace later.

        :stability: experimental
        '''
        result = self._values.get("default_cloud_map_namespace")
        return typing.cast(typing.Optional[CloudMapNamespaceOptions], result)

    @builtins.property
    def enable_fargate_capacity_providers(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether to enable Fargate Capacity Providers.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("enable_fargate_capacity_providers")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def execute_command_configuration(
        self,
    ) -> typing.Optional["ExecuteCommandConfiguration"]:
        '''(experimental) The execute command configuration for the cluster.

        :default: - no configuration will be provided.

        :stability: experimental
        '''
        result = self._values.get("execute_command_configuration")
        return typing.cast(typing.Optional["ExecuteCommandConfiguration"], result)

    @builtins.property
    def vpc(self) -> typing.Optional[_IVpc_6d1f76c4]:
        '''(experimental) The VPC where your ECS instances will be running or your ENIs will be deployed.

        :default: - creates a new VPC with two AZs

        :stability: experimental
        '''
        result = self._values.get("vpc")
        return typing.cast(typing.Optional[_IVpc_6d1f76c4], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ClusterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CommonTaskDefinitionAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "task_definition_arn": "taskDefinitionArn",
        "network_mode": "networkMode",
        "task_role": "taskRole",
    },
)
class CommonTaskDefinitionAttributes:
    def __init__(
        self,
        *,
        task_definition_arn: builtins.str,
        network_mode: typing.Optional["NetworkMode"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
    ) -> None:
        '''(experimental) The common task definition attributes used across all types of task definitions.

        :param task_definition_arn: (experimental) The arn of the task definition.
        :param network_mode: (experimental) The networking mode to use for the containers in the task. Default: Network mode cannot be provided to the imported task.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: Permissions cannot be granted to the imported task.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_iam as iam
            
            # role: iam.Role
            
            common_task_definition_attributes = ecs.CommonTaskDefinitionAttributes(
                task_definition_arn="taskDefinitionArn",
            
                # the properties below are optional
                network_mode=ecs.NetworkMode.NONE,
                task_role=role
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "task_definition_arn": task_definition_arn,
        }
        if network_mode is not None:
            self._values["network_mode"] = network_mode
        if task_role is not None:
            self._values["task_role"] = task_role

    @builtins.property
    def task_definition_arn(self) -> builtins.str:
        '''(experimental) The arn of the task definition.

        :stability: experimental
        '''
        result = self._values.get("task_definition_arn")
        assert result is not None, "Required property 'task_definition_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def network_mode(self) -> typing.Optional["NetworkMode"]:
        '''(experimental) The networking mode to use for the containers in the task.

        :default: Network mode cannot be provided to the imported task.

        :stability: experimental
        '''
        result = self._values.get("network_mode")
        return typing.cast(typing.Optional["NetworkMode"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: Permissions cannot be granted to the imported task.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonTaskDefinitionAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.CommonTaskDefinitionProps",
    jsii_struct_bases=[],
    name_mapping={
        "execution_role": "executionRole",
        "family": "family",
        "proxy_configuration": "proxyConfiguration",
        "task_role": "taskRole",
        "volumes": "volumes",
    },
)
class CommonTaskDefinitionProps:
    def __init__(
        self,
        *,
        execution_role: typing.Optional[_IRole_59af6f50] = None,
        family: typing.Optional[builtins.str] = None,
        proxy_configuration: typing.Optional["ProxyConfiguration"] = None,
        task_role: typing.Optional[_IRole_59af6f50] = None,
        volumes: typing.Optional[typing.Sequence["Volume"]] = None,
    ) -> None:
        '''(experimental) The common properties for all task definitions.

        For more information, see
        `Task Definition Parameters <https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html>`_.

        :param execution_role: (experimental) The name of the IAM task execution role that grants the ECS agent permission to call AWS APIs on your behalf. The role will be used to retrieve container images from ECR and create CloudWatch log groups. Default: - An execution role will be automatically created if you use ECR images in your task definition.
        :param family: (experimental) The name of a family that this task definition is registered to. A family groups multiple versions of a task definition. Default: - Automatically generated name.
        :param proxy_configuration: (experimental) The configuration details for the App Mesh proxy. Default: - No proxy configuration.
        :param task_role: (experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf. Default: - A task role is automatically created for you.
        :param volumes: (experimental) The list of volume definitions for the task. For more information, see `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_. Default: - No volumes are passed to the Docker daemon on a container instance.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_ecs as ecs
            from monocdk import aws_iam as iam
            
            # proxy_configuration: ecs.ProxyConfiguration
            # role: iam.Role
            
            common_task_definition_props = ecs.CommonTaskDefinitionProps(
                execution_role=role,
                family="family",
                proxy_configuration=proxy_configuration,
                task_role=role,
                volumes=[ecs.Volume(
                    name="name",
            
                    # the properties below are optional
                    docker_volume_configuration=ecs.DockerVolumeConfiguration(
                        driver="driver",
                        scope=ecs.Scope.TASK,
            
                        # the properties below are optional
                        autoprovision=False,
                        driver_opts={
                            "driver_opts_key": "driverOpts"
                        },
                        labels={
                            "labels_key": "labels"
                        }
                    ),
                    efs_volume_configuration=ecs.EfsVolumeConfiguration(
                        file_system_id="fileSystemId",
            
                        # the properties below are optional
                        authorization_config=ecs.AuthorizationConfig(
                            access_point_id="accessPointId",
                            iam="iam"
                        ),
                        root_directory="rootDirectory",
                        transit_encryption="transitEncryption",
                        transit_encryption_port=123
                    ),
                    host=ecs.Host(
                        source_path="sourcePath"
                    )
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if execution_role is not None:
            self._values["execution_role"] = execution_role
        if family is not None:
            self._values["family"] = family
        if proxy_configuration is not None:
            self._values["proxy_configuration"] = proxy_configuration
        if task_role is not None:
            self._values["task_role"] = task_role
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def execution_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM task execution role that grants the ECS agent permission to call AWS APIs on your behalf.

        The role will be used to retrieve container images from ECR and create CloudWatch log groups.

        :default: - An execution role will be automatically created if you use ECR images in your task definition.

        :stability: experimental
        '''
        result = self._values.get("execution_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def family(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of a family that this task definition is registered to.

        A family groups multiple versions of a task definition.

        :default: - Automatically generated name.

        :stability: experimental
        '''
        result = self._values.get("family")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def proxy_configuration(self) -> typing.Optional["ProxyConfiguration"]:
        '''(experimental) The configuration details for the App Mesh proxy.

        :default: - No proxy configuration.

        :stability: experimental
        '''
        result = self._values.get("proxy_configuration")
        return typing.cast(typing.Optional["ProxyConfiguration"], result)

    @builtins.property
    def task_role(self) -> typing.Optional[_IRole_59af6f50]:
        '''(experimental) The name of the IAM role that grants containers in the task permission to call AWS APIs on your behalf.

        :default: - A task role is automatically created for you.

        :stability: experimental
        '''
        result = self._values.get("task_role")
        return typing.cast(typing.Optional[_IRole_59af6f50], result)

    @builtins.property
    def volumes(self) -> typing.Optional[typing.List["Volume"]]:
        '''(experimental) The list of volume definitions for the task.

        For more information, see
        `Task Definition Parameter Volumes <https://docs.aws.amazon.com/AmazonECS/latest/developerguide//task_definition_parameters.html#volumes>`_.

        :default: - No volumes are passed to the Docker daemon on a container instance.

        :stability: experimental
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.List["Volume"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonTaskDefinitionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_ecs.Compatibility")
class Compatibility(enum.Enum):
    '''(experimental) The task launch type compatibility requirement.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        vpc = ec2.Vpc.from_lookup(self, "Vpc",
            is_default=True
        )
        
        cluster = ecs.Cluster(self, "Ec2Cluster", vpc=vpc)
        cluster.add_capacity("DefaultAutoScalingGroup",
            instance_type=ec2.InstanceType("t2.micro"),
            vpc_subnets=dynamodb.aws_ec2.SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
        )
        
        task_definition = ecs.TaskDefinition(self, "TD",
            compatibility=ecs.Compatibility.EC2
        )
        
        task_definition.add_container("TheContainer",
            image=ecs.ContainerImage.from_registry("foo/bar"),
            memory_limit_mi_b=256
        )
        
        run_task = tasks.EcsRunTask(self, "Run",
            integration_pattern=sfn.IntegrationPattern.RUN_JOB,
            cluster=cluster,
            task_definition=task_definition,
            launch_target=tasks.EcsEc2LaunchTarget(
                placement_strategies=[
                    ecs.PlacementStrategy.spread_across_instances(),
                    ecs.PlacementStrategy.packed_by_cpu(),
                    ecs.PlacementStrategy.randomly()
                ],
                placement_constraints=[
                    ecs.PlacementConstraint.member_of("blieptuut")
                ]
            )
        )
    '''

    EC2 = "EC2"
    '''(experimental) The task should specify the EC2 launch type.

    :stability: experimental
    '''
    FARGATE = "FARGATE"
    '''(experimental) The task should specify the Fargate launch type.

    :stability: experimental
    '''
    EC2_AND_FARGATE = "EC2_AND_FARGATE"
    '''(experimental) The task can specify either the EC2 or Fargate launch types.

    :stability: experimental
    '''
    EXTERNAL = "EXTERNAL"
    '''(experimental) The task should specify the External launch type.

    :stability: experimental
    '''


class ContainerDefinition(
    _Construct_e78e779f,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_ecs.ContainerDefinition",
):
    '''(experimental) A container definition is used in a task definition to describe the containers that are launched as part of a task.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # task_definition: ecs.TaskDefinition
        # cluster: ecs.Cluster
        
        
        # Add a container to the task definition
        specific_container = task_definition.add_container("Container",
            image=ecs.ContainerImage.from_registry("/aws/aws-example-app"),
            memory_limit_mi_b=2048
        )
        
        # Add a port mapping
        specific_container.add_port_mappings(
            container_port=7600,
            protocol=ecs.Protocol.TCP
        )
        
        ecs.Ec2Service(self, "Service",
            cluster=cluster,
            task_definition=task_definition,
            cloud_map_options=autoscaling.aws_ecs.CloudMapOptions(
                # Create SRV records - useful for bridge networking
                dns_record_type=cloudmap.DnsRecordType.SRV,
                # Targets port TCP port 7600 `specificContainer`
                container=specific_container,
                container_port=7600
            )
        )
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        task_definition: "TaskDefinition",
        image: "ContainerImage",
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence["EnvironmentFile"]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        system_controls: typing.Optional[typing.Sequence["SystemControl"]] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Constructs a new instance of the ContainerDefinition class.

        :param scope: -
        :param id: -
        :param task_definition: (experimental) The name of the task definition that includes this container definition. [disable-awslint:ref-via-interface]
        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param system_controls: (experimental) A list of namespaced kernel parameters to set in the container. Default: - No system controls are set.
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /

        :stability: experimental
        '''
        props = ContainerDefinitionProps(
            task_definition=task_definition,
            image=image,
            command=command,
            container_name=container_name,
            cpu=cpu,
            disable_networking=disable_networking,
            dns_search_domains=dns_search_domains,
            dns_servers=dns_servers,
            docker_labels=docker_labels,
            docker_security_options=docker_security_options,
            entry_point=entry_point,
            environment=environment,
            environment_files=environment_files,
            essential=essential,
            extra_hosts=extra_hosts,
            gpu_count=gpu_count,
            health_check=health_check,
            hostname=hostname,
            inference_accelerator_resources=inference_accelerator_resources,
            linux_parameters=linux_parameters,
            logging=logging,
            memory_limit_mib=memory_limit_mib,
            memory_reservation_mib=memory_reservation_mib,
            port_mappings=port_mappings,
            privileged=privileged,
            readonly_root_filesystem=readonly_root_filesystem,
            secrets=secrets,
            start_timeout=start_timeout,
            stop_timeout=stop_timeout,
            system_controls=system_controls,
            user=user,
            working_directory=working_directory,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addContainerDependencies")
    def add_container_dependencies(
        self,
        *container_dependencies: "ContainerDependency",
    ) -> None:
        '''(experimental) This method adds one or more container dependencies to the container.

        :param container_dependencies: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addContainerDependencies", [*container_dependencies]))

    @jsii.member(jsii_name="addEnvironment")
    def add_environment(self, name: builtins.str, value: builtins.str) -> None:
        '''(experimental) This method adds an environment variable to the container.

        :param name: -
        :param value: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addEnvironment", [name, value]))

    @jsii.member(jsii_name="addInferenceAcceleratorResource")
    def add_inference_accelerator_resource(
        self,
        *inference_accelerator_resources: builtins.str,
    ) -> None:
        '''(experimental) This method adds one or more resources to the container.

        :param inference_accelerator_resources: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addInferenceAcceleratorResource", [*inference_accelerator_resources]))

    @jsii.member(jsii_name="addLink")
    def add_link(
        self,
        container: "ContainerDefinition",
        alias: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) This method adds a link which allows containers to communicate with each other without the need for port mappings.

        This parameter is only supported if the task definition is using the bridge network mode.
        Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.

        :param container: -
        :param alias: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addLink", [container, alias]))

    @jsii.member(jsii_name="addMountPoints")
    def add_mount_points(self, *mount_points: "MountPoint") -> None:
        '''(experimental) This method adds one or more mount points for data volumes to the container.

        :param mount_points: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addMountPoints", [*mount_points]))

    @jsii.member(jsii_name="addPortMappings")
    def add_port_mappings(self, *port_mappings: "PortMapping") -> None:
        '''(experimental) This method adds one or more port mappings to the container.

        :param port_mappings: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addPortMappings", [*port_mappings]))

    @jsii.member(jsii_name="addScratch")
    def add_scratch(
        self,
        *,
        container_path: builtins.str,
        name: builtins.str,
        read_only: builtins.bool,
        source_path: builtins.str,
    ) -> None:
        '''(experimental) This method mounts temporary disk space to the container.

        This adds the correct container mountPoint and task definition volume.

        :param container_path: (experimental) The path on the container to mount the scratch volume at.
        :param name: (experimental) The name of the scratch volume to mount. Must be a volume name referenced in the name parameter of task definition volume.
        :param read_only: (experimental) Specifies whether to give the container read-only access to the scratch volume. If this value is true, the container has read-only access to the scratch volume. If this value is false, then the container can write to the scratch volume.
        :param source_path: 

        :stability: experimental
        '''
        scratch = ScratchSpace(
            container_path=container_path,
            name=name,
            read_only=read_only,
            source_path=source_path,
        )

        return typing.cast(None, jsii.invoke(self, "addScratch", [scratch]))

    @jsii.member(jsii_name="addToExecutionPolicy")
    def add_to_execution_policy(self, statement: _PolicyStatement_296fe8a3) -> None:
        '''(experimental) This method adds the specified statement to the IAM task execution policy in the task definition.

        :param statement: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addToExecutionPolicy", [statement]))

    @jsii.member(jsii_name="addUlimits")
    def add_ulimits(self, *ulimits: "Ulimit") -> None:
        '''(experimental) This method adds one or more ulimits to the container.

        :param ulimits: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addUlimits", [*ulimits]))

    @jsii.member(jsii_name="addVolumesFrom")
    def add_volumes_from(self, *volumes_from: "VolumeFrom") -> None:
        '''(experimental) This method adds one or more volumes to the container.

        :param volumes_from: -

        :stability: experimental
        '''
        return typing.cast(None, jsii.invoke(self, "addVolumesFrom", [*volumes_from]))

    @jsii.member(jsii_name="findPortMapping")
    def find_port_mapping(
        self,
        container_port: jsii.Number,
        protocol: "Protocol",
    ) -> typing.Optional["PortMapping"]:
        '''(experimental) Returns the host port for the requested container port if it exists.

        :param container_port: -
        :param protocol: -

        :stability: experimental
        '''
        return typing.cast(typing.Optional["PortMapping"], jsii.invoke(self, "findPortMapping", [container_port, protocol]))

    @jsii.member(jsii_name="renderContainerDefinition")
    def render_container_definition(
        self,
        _task_definition: typing.Optional["TaskDefinition"] = None,
    ) -> CfnTaskDefinition.ContainerDefinitionProperty:
        '''(experimental) Render this container definition to a CloudFormation object.

        :param _task_definition: [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change).

        :stability: experimental
        '''
        return typing.cast(CfnTaskDefinition.ContainerDefinitionProperty, jsii.invoke(self, "renderContainerDefinition", [_task_definition]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerDependencies")
    def container_dependencies(self) -> typing.List["ContainerDependency"]:
        '''(experimental) An array dependencies defined for container startup and shutdown.

        :stability: experimental
        '''
        return typing.cast(typing.List["ContainerDependency"], jsii.get(self, "containerDependencies"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerName")
    def container_name(self) -> builtins.str:
        '''(experimental) The name of this container.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "containerName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerPort")
    def container_port(self) -> jsii.Number:
        '''(experimental) The port the container will listen on.

        :stability: experimental
        '''
        return typing.cast(jsii.Number, jsii.get(self, "containerPort"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="essential")
    def essential(self) -> builtins.bool:
        '''(experimental) Specifies whether the container will be marked essential.

        If the essential parameter of a container is marked as true, and that container
        fails or stops for any reason, all other containers that are part of the task are
        stopped. If the essential parameter of a container is marked as false, then its
        failure does not affect the rest of the containers in a task.

        If this parameter is omitted, a container is assumed to be essential.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "essential"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageName")
    def image_name(self) -> builtins.str:
        '''(experimental) The name of the image referenced by this container.

        :stability: experimental
        '''
        return typing.cast(builtins.str, jsii.get(self, "imageName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ingressPort")
    def ingress_port(self) -> jsii.Number:
        '''(experimental) The inbound rules associated with the security group the task or service will use.

        This property is only used for tasks that use the awsvpc network mode.

        :stability: experimental
        '''
        return typing.cast(jsii.Number, jsii.get(self, "ingressPort"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="memoryLimitSpecified")
    def memory_limit_specified(self) -> builtins.bool:
        '''(experimental) Whether there was at least one memory limit specified in this definition.

        :stability: experimental
        '''
        return typing.cast(builtins.bool, jsii.get(self, "memoryLimitSpecified"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="mountPoints")
    def mount_points(self) -> typing.List["MountPoint"]:
        '''(experimental) The mount points for data volumes in your container.

        :stability: experimental
        '''
        return typing.cast(typing.List["MountPoint"], jsii.get(self, "mountPoints"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="portMappings")
    def port_mappings(self) -> typing.List["PortMapping"]:
        '''(experimental) The list of port mappings for the container.

        Port mappings allow containers to access ports
        on the host container instance to send or receive traffic.

        :stability: experimental
        '''
        return typing.cast(typing.List["PortMapping"], jsii.get(self, "portMappings"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="taskDefinition")
    def task_definition(self) -> "TaskDefinition":
        '''(experimental) The name of the task definition that includes this container definition.

        :stability: experimental
        '''
        return typing.cast("TaskDefinition", jsii.get(self, "taskDefinition"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ulimits")
    def ulimits(self) -> typing.List["Ulimit"]:
        '''(experimental) An array of ulimits to set in the container.

        :stability: experimental
        '''
        return typing.cast(typing.List["Ulimit"], jsii.get(self, "ulimits"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volumesFrom")
    def volumes_from(self) -> typing.List["VolumeFrom"]:
        '''(experimental) The data volumes to mount from another container in the same task definition.

        :stability: experimental
        '''
        return typing.cast(typing.List["VolumeFrom"], jsii.get(self, "volumesFrom"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="environmentFiles")
    def environment_files(
        self,
    ) -> typing.Optional[typing.List["EnvironmentFileConfig"]]:
        '''(experimental) The environment files for this container.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[typing.List["EnvironmentFileConfig"]], jsii.get(self, "environmentFiles"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="linuxParameters")
    def linux_parameters(self) -> typing.Optional["LinuxParameters"]:
        '''(experimental) The Linux-specific modifications that are applied to the container, such as Linux kernel capabilities.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["LinuxParameters"], jsii.get(self, "linuxParameters"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logDriverConfig")
    def log_driver_config(self) -> typing.Optional["LogDriverConfig"]:
        '''(experimental) The log configuration specification for the container.

        :stability: experimental
        '''
        return typing.cast(typing.Optional["LogDriverConfig"], jsii.get(self, "logDriverConfig"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="referencesSecretJsonField")
    def references_secret_json_field(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Whether this container definition references a specific JSON field of a secret stored in Secrets Manager.

        :stability: experimental
        '''
        return typing.cast(typing.Optional[builtins.bool], jsii.get(self, "referencesSecretJsonField"))


@jsii.data_type(
    jsii_type="monocdk.aws_ecs.ContainerDefinitionOptions",
    jsii_struct_bases=[],
    name_mapping={
        "image": "image",
        "command": "command",
        "container_name": "containerName",
        "cpu": "cpu",
        "disable_networking": "disableNetworking",
        "dns_search_domains": "dnsSearchDomains",
        "dns_servers": "dnsServers",
        "docker_labels": "dockerLabels",
        "docker_security_options": "dockerSecurityOptions",
        "entry_point": "entryPoint",
        "environment": "environment",
        "environment_files": "environmentFiles",
        "essential": "essential",
        "extra_hosts": "extraHosts",
        "gpu_count": "gpuCount",
        "health_check": "healthCheck",
        "hostname": "hostname",
        "inference_accelerator_resources": "inferenceAcceleratorResources",
        "linux_parameters": "linuxParameters",
        "logging": "logging",
        "memory_limit_mib": "memoryLimitMiB",
        "memory_reservation_mib": "memoryReservationMiB",
        "port_mappings": "portMappings",
        "privileged": "privileged",
        "readonly_root_filesystem": "readonlyRootFilesystem",
        "secrets": "secrets",
        "start_timeout": "startTimeout",
        "stop_timeout": "stopTimeout",
        "system_controls": "systemControls",
        "user": "user",
        "working_directory": "workingDirectory",
    },
)
class ContainerDefinitionOptions:
    def __init__(
        self,
        *,
        image: "ContainerImage",
        command: typing.Optional[typing.Sequence[builtins.str]] = None,
        container_name: typing.Optional[builtins.str] = None,
        cpu: typing.Optional[jsii.Number] = None,
        disable_networking: typing.Optional[builtins.bool] = None,
        dns_search_domains: typing.Optional[typing.Sequence[builtins.str]] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        docker_labels: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        docker_security_options: typing.Optional[typing.Sequence[builtins.str]] = None,
        entry_point: typing.Optional[typing.Sequence[builtins.str]] = None,
        environment: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        environment_files: typing.Optional[typing.Sequence["EnvironmentFile"]] = None,
        essential: typing.Optional[builtins.bool] = None,
        extra_hosts: typing.Optional[typing.Mapping[builtins.str, builtins.str]] = None,
        gpu_count: typing.Optional[jsii.Number] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        hostname: typing.Optional[builtins.str] = None,
        inference_accelerator_resources: typing.Optional[typing.Sequence[builtins.str]] = None,
        linux_parameters: typing.Optional["LinuxParameters"] = None,
        logging: typing.Optional["LogDriver"] = None,
        memory_limit_mib: typing.Optional[jsii.Number] = None,
        memory_reservation_mib: typing.Optional[jsii.Number] = None,
        port_mappings: typing.Optional[typing.Sequence["PortMapping"]] = None,
        privileged: typing.Optional[builtins.bool] = None,
        readonly_root_filesystem: typing.Optional[builtins.bool] = None,
        secrets: typing.Optional[typing.Mapping[builtins.str, "Secret"]] = None,
        start_timeout: typing.Optional[_Duration_070aa057] = None,
        stop_timeout: typing.Optional[_Duration_070aa057] = None,
        system_controls: typing.Optional[typing.Sequence["SystemControl"]] = None,
        user: typing.Optional[builtins.str] = None,
        working_directory: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param image: (experimental) The image used to start a container. This string is passed directly to the Docker daemon. Images in the Docker Hub registry are available by default. Other repositories are specified with either repository-url/image:tag or repository-url/image@digest. TODO: Update these to specify using classes of IContainerImage
        :param command: (experimental) The command that is passed to the container. If you provide a shell command as a single string, you have to quote command-line arguments. Default: - CMD value built into container image.
        :param container_name: (experimental) The name of the container. Default: - id of node associated with ContainerDefinition.
        :param cpu: (experimental) The minimum number of CPU units to reserve for the container. Default: - No minimum CPU units reserved.
        :param disable_networking: (experimental) Specifies whether networking is disabled within the container. When this parameter is true, networking is disabled within the container. Default: false
        :param dns_search_domains: (experimental) A list of DNS search domains that are presented to the container. Default: - No search domains.
        :param dns_servers: (experimental) A list of DNS servers that are presented to the container. Default: - Default DNS servers.
        :param docker_labels: (experimental) A key/value map of labels to add to the container. Default: - No labels.
        :param docker_security_options: (experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems. Default: - No security labels.
        :param entry_point: (experimental) The ENTRYPOINT value to pass to the container. Default: - Entry point configured in container.
        :param environment: (experimental) The environment variables to pass to the container. Default: - No environment variables.
        :param environment_files: (experimental) The environment files to pass to the container. Default: - No environment files.
        :param essential: (experimental) Specifies whether the container is marked essential. If the essential parameter of a container is marked as true, and that container fails or stops for any reason, all other containers that are part of the task are stopped. If the essential parameter of a container is marked as false, then its failure does not affect the rest of the containers in a task. All tasks must have at least one essential container. If this parameter is omitted, a container is assumed to be essential. Default: true
        :param extra_hosts: (experimental) A list of hostnames and IP address mappings to append to the /etc/hosts file on the container. Default: - No extra hosts.
        :param gpu_count: (experimental) The number of GPUs assigned to the container. Default: - No GPUs assigned.
        :param health_check: (experimental) The health check command and associated configuration parameters for the container. Default: - Health check configuration from container.
        :param hostname: (experimental) The hostname to use for your container. Default: - Automatic hostname.
        :param inference_accelerator_resources: (experimental) The inference accelerators referenced by the container. Default: - No inference accelerators assigned.
        :param linux_parameters: (experimental) Linux-specific modifications that are applied to the container, such as Linux kernel capabilities. For more information see `KernelCapabilities <https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html>`_. Default: - No Linux parameters.
        :param logging: (experimental) The log configuration specification for the container. Default: - Containers use the same logging driver that the Docker daemon uses.
        :param memory_limit_mib: (experimental) The amount (in MiB) of memory to present to the container. If your container attempts to exceed the allocated memory, the container is terminated. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory limit.
        :param memory_reservation_mib: (experimental) The soft limit (in MiB) of memory to reserve for the container. When system memory is under heavy contention, Docker attempts to keep the container memory to this soft limit. However, your container can consume more memory when it needs to, up to either the hard limit specified with the memory parameter (if applicable), or all of the available memory on the container instance, whichever comes first. At least one of memoryLimitMiB and memoryReservationMiB is required for non-Fargate services. Default: - No memory reserved.
        :param port_mappings: (experimental) The port mappings to add to the container definition. Default: - No ports are mapped.
        :param privileged: (experimental) Specifies whether the container is marked as privileged. When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user). Default: false
        :param readonly_root_filesystem: (experimental) When this parameter is true, the container is given read-only access to its root file system. Default: false
        :param secrets: (experimental) The secret environment variables to pass to the container. Default: - No secret environment variables.
        :param start_timeout: (experimental) Time duration (in seconds) to wait before giving up on resolving dependencies for a container. Default: - none
        :param stop_timeout: (experimental) Time duration (in seconds) to wait before the container is forcefully killed if it doesn't exit normally on its own. Default: - none
        :param system_controls: (experimental) A list of namespaced kernel parameters to set in the container. Default: - No system controls are set.
        :param user: (experimental) The user name to use inside the container. Default: root
        :param working_directory: (experimental) The working directory in which to run commands inside the container. Default: /

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # task_definition: ecs.TaskDefinition
            # cluster: ecs.Cluster
            
            
            # Add a container to the task definition
            specific_container = task_definition.add_container("Container",
                image=ecs.ContainerImage.from_registry("/aws/aws-example-app"),
                memory_limit_mi_b=2048
            )
            
            # Add a port mapping
            specific_container.add_port_mappings(
                container_port=7600,
                protocol=ecs.Protocol.TCP
            )
            
            ecs.Ec2Service(self, "Service",
                cluster=cluster,
                task_definition=task_definition,
                cloud_map_options=autoscaling.aws_ecs.CloudMapOptions(
                    # Create SRV records - useful for bridge networking
                    dns_record_type=cloudmap.DnsRecordType.SRV,
                    # Targets port TCP port 7600 `specificContainer`
                    container=specific_container,
                    container_port=7600
                )
            )
        '''
        if isinstance(health_check, dict):
            health_check = HealthCheck(**health_check)
        self._values: typing.Dict[str, typing.Any] = {
            "image": image,
        }
        if command is not None:
            self._values["command"] = command
        if container_name is not None:
            self._values["container_name"] = container_name
        if cpu is not None:
            self._values["cpu"] = cpu
        if disable_networking is not None:
            self._values["disable_networking"] = disable_networking
        if dns_search_domains is not None:
            self._values["dns_search_domains"] = dns_search_domains
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if docker_labels is not None:
            self._values["docker_labels"] = docker_labels
        if docker_security_options is not None:
            self._values["docker_security_options"] = docker_security_options
        if entry_point is not None:
            self._values["entry_point"] = entry_point
        if environment is not None:
            self._values["environment"] = environment
        if environment_files is not None:
            self._values["environment_files"] = environment_files
        if essential is not None:
            self._values["essential"] = essential
        if extra_hosts is not None:
            self._values["extra_hosts"] = extra_hosts
        if gpu_count is not None:
            self._values["gpu_count"] = gpu_count
        if health_check is not None:
            self._values["health_check"] = health_check
        if hostname is not None:
            self._values["hostname"] = hostname
        if inference_accelerator_resources is not None:
            self._values["inference_accelerator_resources"] = inference_accelerator_resources
        if linux_parameters is not None:
            self._values["linux_parameters"] = linux_parameters
        if logging is not None:
            self._values["logging"] = logging
        if memory_limit_mib is not None:
            self._values["memory_limit_mib"] = memory_limit_mib
        if memory_reservation_mib is not None:
            self._values["memory_reservation_mib"] = memory_reservation_mib
        if port_mappings is not None:
            self._values["port_mappings"] = port_mappings
        if privileged is not None:
            self._values["privileged"] = privileged
        if readonly_root_filesystem is not None:
            self._values["readonly_root_filesystem"] = readonly_root_filesystem
        if secrets is not None:
            self._values["secrets"] = secrets
        if start_timeout is not None:
            self._values["start_timeout"] = start_timeout
        if stop_timeout is not None:
            self._values["stop_timeout"] = stop_timeout
        if system_controls is not None:
            self._values["system_controls"] = system_controls
        if user is not None:
            self._values["user"] = user
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def image(self) -> "ContainerImage":
        '''(experimental) The image used to start a container.

        This string is passed directly to the Docker daemon.
        Images in the Docker Hub registry are available by default.
        Other repositories are specified with either repository-url/image:tag or repository-url/image@digest.
        TODO: Update these to specify using classes of IContainerImage

        :stability: experimental
        '''
        result = self._values.get("image")
        assert result is not None, "Required property 'image' is missing"
        return typing.cast("ContainerImage", result)

    @builtins.property
    def command(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The command that is passed to the container.

        If you provide a shell command as a single string, you have to quote command-line arguments.

        :default: - CMD value built into container image.

        :stability: experimental
        '''
        result = self._values.get("command")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def container_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the container.

        :default: - id of node associated with ContainerDefinition.

        :stability: experimental
        '''
        result = self._values.get("container_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cpu(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The minimum number of CPU units to reserve for the container.

        :default: - No minimum CPU units reserved.

        :stability: experimental
        '''
        result = self._values.get("cpu")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def disable_networking(self) -> typing.Optional[builtins.bool]:
        '''(experimental) Specifies whether networking is disabled within the container.

        When this parameter is true, networking is disabled within the container.

        :default: false

        :stability: experimental
        '''
        result = self._values.get("disable_networking")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def dns_search_domains(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS search domains that are presented to the container.

        :default: - No search domains.

        :stability: experimental
        '''
        result = self._values.get("dns_search_domains")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of DNS servers that are presented to the container.

        :default: - Default DNS servers.

        :stability: experimental
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def docker_labels(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) A key/value map of labels to add to the container.

        :default: - No labels.

        :stability: experimental
        '''
        result = self._values.get("docker_labels")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, builtins.str]], result)

    @builtins.property
    def docker_security_options(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) A list of strings to provide custom labels for SELinux and AppArmor multi-level security systems.

        :default: - No security labels.

        :stability: experimental
        '''
        result = self._values.get("docker_security_options")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def entry_point(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(experimental) The ENTRYPOINT value to pass to the container.

        :default: - Entry point configured in container.

        :see: https://docs.docker.com/engine/reference/builder/#entrypoint
        :stability: experimental
        '''
        result = self._values.get("entry_point")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def environment(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, builtins.str]]:
        '''(experimental) The environment variables to pass to the container.

        :default: - No environment variables.

        :stability: experimenta