'''
# AWS App Mesh Construct Library

AWS App Mesh is a service mesh based on the [Envoy](https://www.envoyproxy.io/) proxy that makes it easy to monitor and control microservices. App Mesh standardizes how your microservices communicate, giving you end-to-end visibility and helping to ensure high-availability for your applications.

App Mesh gives you consistent visibility and network traffic controls for every microservice in an application.

App Mesh supports microservice applications that use service discovery naming for their components. To use App Mesh, you must have an existing application running on AWS Fargate, Amazon ECS, Amazon EKS, Kubernetes on AWS, or Amazon EC2.

For further information on **AWS App Mesh**, visit the [AWS App Mesh Documentation](https://docs.aws.amazon.com/app-mesh/index.html).

## Create the App and Stack

```python
app = cdk.App()
stack = cdk.Stack(app, "stack")
```

## Creating the Mesh

A service mesh is a logical boundary for network traffic between the services that reside within it.

After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.

The following example creates the `AppMesh` service mesh with the default egress filter of `DROP_ALL`. See [the AWS CloudFormation `EgressFilter` resource](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html) for more info on egress filters.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh"
)
```

The mesh can instead be created with the `ALLOW_ALL` egress filter by providing the `egressFilter` property.

```python
mesh = appmesh.Mesh(self, "AppMesh",
    mesh_name="myAwsMesh",
    egress_filter=appmesh.MeshFilterType.ALLOW_ALL
)
```

## Adding VirtualRouters

A *mesh* uses  *virtual routers* as logical units to route requests to *virtual nodes*.

Virtual routers handle traffic for one or more virtual services within your mesh.
After you create a virtual router, you can create and associate routes to your virtual router that direct incoming requests to different virtual nodes.

```python
# mesh: appmesh.Mesh

router = mesh.add_virtual_router("router",
    listeners=[appmesh.VirtualRouterListener.http(8080)]
)
```

Note that creating the router using the `addVirtualRouter()` method places it in the same stack as the mesh
(which might be different from the current stack).
The router can also be created using the `VirtualRouter` constructor (passing in the mesh) instead of calling the `addVirtualRouter()` method.
This is particularly useful when splitting your resources between many stacks: for example, defining the mesh itself as part of an infrastructure stack, but defining the other resources, such as routers, in the application stack:

```python
# infra_stack: cdk.Stack
# app_stack: cdk.Stack


mesh = appmesh.Mesh(infra_stack, "AppMesh",
    mesh_name="myAwsMesh",
    egress_filter=appmesh.MeshFilterType.ALLOW_ALL
)

# the VirtualRouter will belong to 'appStack',
# even though the Mesh belongs to 'infraStack'
router = appmesh.VirtualRouter(app_stack, "router",
    mesh=mesh,  # notice that mesh is a required property when creating a router with the 'new' statement
    listeners=[appmesh.VirtualRouterListener.http(8081)]
)
```

The same is true for other `add*()` methods in the App Mesh construct library.

The `VirtualRouterListener` class lets you define protocol-specific listeners.
The `http()`, `http2()`, `grpc()` and `tcp()` methods create listeners for the named protocols.
They accept a single parameter that defines the port to on which requests will be matched.
The port parameter defaults to 8080 if omitted.

## Adding a VirtualService

A *virtual service* is an abstraction of a real service that is provided by a virtual node directly, or indirectly by means of a virtual router. Dependent services call your virtual service by its `virtualServiceName`, and those requests are routed to the virtual node or virtual router specified as the provider for the virtual service.

We recommend that you use the service discovery name of the real service that you're targeting (such as `my-service.default.svc.cluster.local`).

When creating a virtual service:

* If you want the virtual service to spread traffic across multiple virtual nodes, specify a virtual router.
* If you want the virtual service to reach a virtual node directly, without a virtual router, specify a virtual node.

Adding a virtual router as the provider:

```python
# router: appmesh.VirtualRouter


appmesh.VirtualService(self, "virtual-service",
    virtual_service_name="my-service.default.svc.cluster.local",  # optional
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router)
)
```

Adding a virtual node as the provider:

```python
# node: appmesh.VirtualNode


appmesh.VirtualService(self, "virtual-service",
    virtual_service_name="my-service.default.svc.cluster.local",  # optional
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_node(node)
)
```

## Adding a VirtualNode

A *virtual node* acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment.

When you create a virtual node, accept inbound traffic by specifying a *listener*. Outbound traffic that your virtual node expects to send should be specified as a *back end*.

The response metadata for your new virtual node contains the Amazon Resource Name (ARN) that is associated with the virtual node. Set this value (either the full ARN or the truncated resource name) as the `APPMESH_VIRTUAL_NODE_NAME` environment variable for your task group's Envoy proxy container in your task definition or pod spec. For example, the value could be `mesh/default/virtualNode/simpleapp`. This is then mapped to the `node.id` and `node.cluster` Envoy parameters.

> **Note**
> If you require your Envoy stats or tracing to use a different name, you can override the `node.cluster` value that is set by `APPMESH_VIRTUAL_NODE_NAME` with the `APPMESH_VIRTUAL_NODE_CLUSTER` environment variable.

```python
# mesh: appmesh.Mesh
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")
node = mesh.add_virtual_node("virtual-node",
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8081,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),  # minimum
            path="/health-check-path",
            timeout=cdk.Duration.seconds(2),  # minimum
            unhealthy_threshold=2
        )
    )],
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)
```

Create a `VirtualNode` with the constructor and add tags.

```python
# mesh: appmesh.Mesh
# service: cloudmap.Service


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),
            path="/ping",
            timeout=cdk.Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=cdk.aws_appmesh.HttpTimeout(
            idle=cdk.Duration.seconds(5)
        )
    )],
    backend_defaults=cdk.aws_appmesh.BackendDefaults(
        tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
            validation=cdk.aws_appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)

cdk.Tags.of(node).add("Environment", "Dev")
```

Create a `VirtualNode` with the constructor and add backend virtual service.

```python
# mesh: appmesh.Mesh
# router: appmesh.VirtualRouter
# service: cloudmap.Service


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        port=8080,
        health_check=appmesh.HealthCheck.http(
            healthy_threshold=3,
            interval=cdk.Duration.seconds(5),
            path="/ping",
            timeout=cdk.Duration.seconds(2),
            unhealthy_threshold=2
        ),
        timeout=cdk.aws_appmesh.HttpTimeout(
            idle=cdk.Duration.seconds(5)
        )
    )],
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
)

virtual_service = appmesh.VirtualService(self, "service-1",
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router),
    virtual_service_name="service1.domain.local"
)

node.add_backend(appmesh.Backend.virtual_service(virtual_service))
```

The `listeners` property can be left blank and added later with the `node.addListener()` method. The `serviceDiscovery` property must be specified when specifying a listener.

The `backends` property can be added with `node.addBackend()`. In the example, we define a virtual service and add it to the virtual node to allow egress traffic to other nodes.

The `backendDefaults` property is added to the node while creating the virtual node. These are the virtual node's default settings for all backends.

The `VirtualNode.addBackend()` method is especially useful if you want to create a circular traffic flow by having a Virtual Service as a backend whose provider is that same Virtual Node:

```python
# mesh: appmesh.Mesh


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node")
)

virtual_service = appmesh.VirtualService(self, "service-1",
    virtual_service_provider=appmesh.VirtualServiceProvider.virtual_node(node),
    virtual_service_name="service1.domain.local"
)

node.add_backend(appmesh.Backend.virtual_service(virtual_service))
```

### Adding TLS to a listener

The `tls` property specifies TLS configuration when creating a listener for a virtual node or a virtual gateway.
Provide the TLS certificate to the proxy in one of the following ways:

* A certificate from AWS Certificate Manager (ACM).
* A customer-provided certificate (specify a `certificateChain` path file and a `privateKey` file path).
* A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its `secretName`).

```python
# A Virtual Node with listener TLS from an ACM provided certificate
# cert: certificatemanager.Certificate
# mesh: appmesh.Mesh


node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node"),
    listeners=[appmesh.VirtualNodeListener.grpc(
        port=80,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.acm(cert)
        )
    )]
)

# A Virtual Gateway with listener TLS from a customer provided file certificate
gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.grpc(
        port=8080,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
        )
    )],
    virtual_gateway_name="gateway"
)

# A Virtual Gateway with listener TLS from a SDS provided certificate
gateway2 = appmesh.VirtualGateway(self, "gateway2",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.http2(
        port=8080,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.sds("secrete_certificate")
        )
    )],
    virtual_gateway_name="gateway2"
)
```

### Adding mutual TLS authentication

Mutual TLS authentication is an optional component of TLS that offers two-way peer authentication.
To enable mutual TLS authentication, add the `mutualTlsCertificate` property to TLS client policy and/or the `mutualTlsValidation` property to your TLS listener.

`tls.mutualTlsValidation` and `tlsClientPolicy.mutualTlsCertificate` can be sourced from either:

* A customer-provided certificate (specify a `certificateChain` path file and a `privateKey` file path).
* A certificate provided by a Secrets Discovery Service (SDS) endpoint over local Unix Domain Socket (specify its `secretName`).

> **Note**
> Currently, a certificate from AWS Certificate Manager (ACM) cannot be used for mutual TLS authentication.

```python
# mesh: appmesh.Mesh


node1 = appmesh.VirtualNode(self, "node1",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node"),
    listeners=[appmesh.VirtualNodeListener.grpc(
        port=80,
        tls=cdk.aws_appmesh.ListenerTlsOptions(
            mode=appmesh.TlsMode.STRICT,
            certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey"),
            # Validate a file client certificates to enable mutual TLS authentication when a client provides a certificate.
            mutual_tls_validation=cdk.aws_appmesh.MutualTlsValidation(
                trust=appmesh.TlsValidationTrust.file("path-to-certificate")
            )
        )
    )]
)

certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"
node2 = appmesh.VirtualNode(self, "node2",
    mesh=mesh,
    service_discovery=appmesh.ServiceDiscovery.dns("node2"),
    backend_defaults=cdk.aws_appmesh.BackendDefaults(
        tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
            ports=[8080, 8081],
            validation=cdk.aws_appmesh.TlsValidation(
                subject_alternative_names=appmesh.SubjectAlternativeNames.matching_exactly("mesh-endpoint.apps.local"),
                trust=appmesh.TlsValidationTrust.acm([
                    acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                ])
            ),
            # Provide a SDS client certificate when a server requests it and enable mutual TLS authentication.
            mutual_tls_certificate=appmesh.TlsCertificate.sds("secret_certificate")
        )
    )
)
```

### Adding outlier detection to a Virtual Node listener

The `outlierDetection` property adds outlier detection to a Virtual Node listener. The properties
`baseEjectionDuration`, `interval`, `maxEjectionPercent`, and `maxServerErrors` are required.

```python
# mesh: appmesh.Mesh
# Cloud Map service discovery is currently required for host ejection by outlier detection
vpc = ec2.Vpc(self, "vpc")
namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
    vpc=vpc,
    name="domain.local"
)
service = namespace.create_service("Svc")
node = mesh.add_virtual_node("virtual-node",
    service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
    listeners=[appmesh.VirtualNodeListener.http(
        outlier_detection=cdk.aws_appmesh.OutlierDetection(
            base_ejection_duration=cdk.Duration.seconds(10),
            interval=cdk.Duration.seconds(30),
            max_ejection_percent=50,
            max_server_errors=5
        )
    )]
)
```

### Adding a connection pool to a listener

The `connectionPool` property can be added to a Virtual Node listener or Virtual Gateway listener to add a request connection pool. Each listener protocol type has its own connection pool properties.

```python
# A Virtual Node with a gRPC listener with a connection pool set
# mesh: appmesh.Mesh

node = appmesh.VirtualNode(self, "node",
    mesh=mesh,
    # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
    # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
    # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
    # By default, the response type is assumed to be LOAD_BALANCER
    service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
    listeners=[appmesh.VirtualNodeListener.http(
        port=80,
        connection_pool=cdk.aws_appmesh.HttpConnectionPool(
            max_connections=100,
            max_pending_requests=10
        )
    )]
)

# A Virtual Gateway with a gRPC listener with a connection pool set
gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.grpc(
        port=8080,
        connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
            max_requests=10
        )
    )],
    virtual_gateway_name="gateway"
)
```

## Adding a Route

A *route* matches requests with an associated virtual router and distributes traffic to its associated virtual nodes.
The route distributes matching requests to one or more target virtual nodes with relative weighting.

The `RouteSpec` class lets you define protocol-specific route specifications.
The `tcp()`, `http()`, `http2()`, and `grpc()` methods create a specification for the named protocols.

For HTTP-based routes, the match field can match on path (prefix, exact, or regex), HTTP method, scheme,
HTTP headers, and query parameters. By default, HTTP-based routes match all requests.

For gRPC-based routes, the match field can  match on service name, method name, and metadata.
When specifying the method name, the service name must also be specified.

For example, here's how to add an HTTP route that matches based on a prefix of the URL path:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.http(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=cdk.aws_appmesh.HttpRouteMatch(
            # Path that is passed to this method must start with '/'.
            path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
        )
    )
)
```

Add an HTTP2 route that matches based on exact path, method, scheme, headers, and query parameters:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http2",
    route_spec=appmesh.RouteSpec.http2(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=cdk.aws_appmesh.HttpRouteMatch(
            path=appmesh.HttpRoutePathMatch.exactly("/exact"),
            method=appmesh.HttpRouteMethod.POST,
            protocol=appmesh.HttpRouteProtocol.HTTPS,
            headers=[
                # All specified headers must match for the route to match.
                appmesh.HeaderMatch.value_is("Content-Type", "application/json"),
                appmesh.HeaderMatch.value_is_not("Content-Type", "application/json")
            ],
            query_parameters=[
                # All specified query parameters must match for the route to match.
                appmesh.QueryParameterMatch.value_is("query-field", "value")
            ]
        )
    )
)
```

Add a single route with two targets and split traffic 50/50:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.http(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node,
            weight=50
        ), cdk.aws_appmesh.WeightedTarget(
            virtual_node=node,
            weight=50
        )
        ],
        match=cdk.aws_appmesh.HttpRouteMatch(
            path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
        )
    )
)
```

Add an http2 route with retries:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http2-retry",
    route_spec=appmesh.RouteSpec.http2(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
        retry_policy=cdk.aws_appmesh.HttpRetryPolicy(
            # Retry if the connection failed
            tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
            # Retry if HTTP responds with a gateway error (502, 503, 504)
            http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
            # Retry five times
            retry_attempts=5,
            # Use a 1 second timeout per retry
            retry_timeout=cdk.Duration.seconds(1)
        )
    )
)
```

Add a gRPC route with retries:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-retry",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
        match=cdk.aws_appmesh.GrpcRouteMatch(service_name="servicename"),
        retry_policy=cdk.aws_appmesh.GrpcRetryPolicy(
            tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
            http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
            # Retry if gRPC responds that the request was cancelled, a resource
            # was exhausted, or if the service is unavailable
            grpc_retry_events=[appmesh.GrpcRetryEvent.CANCELLED, appmesh.GrpcRetryEvent.RESOURCE_EXHAUSTED, appmesh.GrpcRetryEvent.UNAVAILABLE
            ],
            retry_attempts=5,
            retry_timeout=cdk.Duration.seconds(1)
        )
    )
)
```

Add an gRPC route that matches based on method name and metadata:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-grpc-retry",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
        match=cdk.aws_appmesh.GrpcRouteMatch(
            # When method name is specified, service name must be also specified.
            method_name="methodname",
            service_name="servicename",
            metadata=[
                # All specified metadata must match for the route to match.
                appmesh.HeaderMatch.value_starts_with("Content-Type", "application/"),
                appmesh.HeaderMatch.value_does_not_start_with("Content-Type", "text/")
            ]
        )
    )
)
```

Add a gRPC route with timeout:

```python
# router: appmesh.VirtualRouter
# node: appmesh.VirtualNode


router.add_route("route-http",
    route_spec=appmesh.RouteSpec.grpc(
        weighted_targets=[cdk.aws_appmesh.WeightedTarget(
            virtual_node=node
        )
        ],
        match=cdk.aws_appmesh.GrpcRouteMatch(
            service_name="my-service.default.svc.cluster.local"
        ),
        timeout=cdk.aws_appmesh.GrpcTimeout(
            idle=cdk.Duration.seconds(2),
            per_request=cdk.Duration.seconds(1)
        )
    )
)
```

## Adding a Virtual Gateway

A *virtual gateway* allows resources outside your mesh to communicate with resources inside your mesh.
The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance.
Unlike a virtual node, which represents Envoy running with an application, a virtual gateway represents Envoy deployed by itself.

A virtual gateway is similar to a virtual node in that it has a listener that accepts traffic for a particular port and protocol (HTTP, HTTP2, gRPC).
Traffic received by the virtual gateway is directed to other services in your mesh
using rules defined in gateway routes which can be added to your virtual gateway.

Create a virtual gateway with the constructor:

```python
# mesh: appmesh.Mesh

certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"

gateway = appmesh.VirtualGateway(self, "gateway",
    mesh=mesh,
    listeners=[appmesh.VirtualGatewayListener.http(
        port=443,
        health_check=appmesh.HealthCheck.http(
            interval=cdk.Duration.seconds(10)
        )
    )],
    backend_defaults=cdk.aws_appmesh.BackendDefaults(
        tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
            ports=[8080, 8081],
            validation=cdk.aws_appmesh.TlsValidation(
                trust=appmesh.TlsValidationTrust.acm([
                    acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                ])
            )
        )
    ),
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
    virtual_gateway_name="virtualGateway"
)
```

Add a virtual gateway directly to the mesh:

```python
# mesh: appmesh.Mesh


gateway = mesh.add_virtual_gateway("gateway",
    access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
    virtual_gateway_name="virtualGateway",
    listeners=[appmesh.VirtualGatewayListener.http(
        port=443,
        health_check=appmesh.HealthCheck.http(
            interval=cdk.Duration.seconds(10)
        )
    )]
)
```

The `listeners` field defaults to an HTTP Listener on port 8080 if omitted.
A gateway route can be added using the `gateway.addGatewayRoute()` method.

The `backendDefaults` property, provided when creating the virtual gateway, specifies the virtual gateway's default settings for all backends.

## Adding a Gateway Route

A *gateway route* is attached to a virtual gateway and routes matching traffic to an existing virtual service.

For HTTP-based gateway routes, the `match` field can be used to match on
path (prefix, exact, or regex), HTTP method, host name, HTTP headers, and query parameters.
By default, HTTP-based gateway routes match all requests.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            path=appmesh.HttpGatewayRoutePathMatch.regex("regex")
        )
    )
)
```

For gRPC-based gateway routes, the `match` field can be used to match on service name, host name, and metadata.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-grpc",
    route_spec=appmesh.GatewayRouteSpec.grpc(
        route_target=virtual_service,
        match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
            hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
        )
    )
)
```

For HTTP based gateway routes, App Mesh automatically rewrites the matched prefix path in Gateway Route to “/”.
This automatic rewrite configuration can be overwritten in following ways:

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            # This disables the default rewrite to '/', and retains original path.
            path=appmesh.HttpGatewayRoutePathMatch.starts_with("/path-to-app/", "")
        )
    )
)

gateway.add_gateway_route("gateway-route-http-1",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            # If the request full path is '/path-to-app/xxxxx', this rewrites the path to '/rewrittenUri/xxxxx'.
            # Please note both `prefixPathMatch` and `rewriteTo` must start and end with the `/` character.
            path=appmesh.HttpGatewayRoutePathMatch.starts_with("/path-to-app/", "/rewrittenUri/")
        )
    )
)
```

If matching other path (exact or regex), only specific rewrite path can be specified.
Unlike `startsWith()` method above, no default rewrite is performed.

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-http-2",
    route_spec=appmesh.GatewayRouteSpec.http(
        route_target=virtual_service,
        match=cdk.aws_appmesh.HttpGatewayRouteMatch(
            # This rewrites the path from '/test' to '/rewrittenPath'.
            path=appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath")
        )
    )
)
```

For HTTP/gRPC based routes, App Mesh automatically rewrites
the original request received at the Virtual Gateway to the destination Virtual Service name.
This default host name rewrite can be configured by specifying the rewrite rule as one of the `match` property:

```python
# gateway: appmesh.VirtualGateway
# virtual_service: appmesh.VirtualService


gateway.add_gateway_route("gateway-route-grpc",
    route_spec=appmesh.GatewayRouteSpec.grpc(
        route_target=virtual_service,
        match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
            hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
            # This disables the default rewrite to virtual service name and retain original request.
            rewrite_request_hostname=False
        )
    )
)
```

## Importing Resources

Each App Mesh resource class comes with two static methods, `from<Resource>Arn` and `from<Resource>Attributes` (where `<Resource>` is replaced with the resource name, such as `VirtualNode`) for importing a reference to an existing App Mesh resource.
These imported resources can be used with other resources in your mesh as if they were defined directly in your CDK application.

```python
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh/virtualNode/testNode"
appmesh.VirtualNode.from_virtual_node_arn(self, "importedVirtualNode", arn)
```

```python
virtual_node_name = "my-virtual-node"
appmesh.VirtualNode.from_virtual_node_attributes(self, "imported-virtual-node",
    mesh=appmesh.Mesh.from_mesh_name(self, "Mesh", "testMesh"),
    virtual_node_name=virtual_node_name
)
```

To import a mesh, again there are two static methods, `fromMeshArn` and `fromMeshName`.

```python
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh"
appmesh.Mesh.from_mesh_arn(self, "imported-mesh", arn)
```

```python
appmesh.Mesh.from_mesh_name(self, "imported-mesh", "abc")
```

## IAM Grants

`VirtualNode` and `VirtualGateway` provide `grantStreamAggregatedResources` methods that grant identities that are running
Envoy access to stream generated config from App Mesh.

```python
# mesh: appmesh.Mesh

gateway = appmesh.VirtualGateway(self, "testGateway", mesh=mesh)
envoy_user = iam.User(self, "envoyUser")

#
# This will grant `grantStreamAggregatedResources` ONLY for this gateway.
#
gateway.grant_stream_aggregated_resources(envoy_user)
```

## Adding Resources to shared meshes

A shared mesh allows resources created by different accounts to communicate with each other in the same mesh:

```python
# This is the ARN for the mesh from different AWS IAM account ID.
# Ensure mesh is properly shared with your account. For more details, see: https://github.com/aws/aws-cdk/issues/15404
arn = "arn:aws:appmesh:us-east-1:123456789012:mesh/testMesh"
shared_mesh = appmesh.Mesh.from_mesh_arn(self, "imported-mesh", arn)

# This VirtualNode resource can communicate with the resources in the mesh from different AWS IAM account ID.
appmesh.VirtualNode(self, "test-node",
    mesh=shared_mesh
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from .._jsii import *

import constructs
from .. import (
    CfnResource as _CfnResource_e0a482dc,
    CfnTag as _CfnTag_95fbdc29,
    Construct as _Construct_e78e779f,
    Duration as _Duration_070aa057,
    IInspectable as _IInspectable_82c04a63,
    IResolvable as _IResolvable_a771d0ef,
    IResource as _IResource_8c1dbbbd,
    Resource as _Resource_abff4495,
    TagManager as _TagManager_0b7ab120,
    TreeInspector as _TreeInspector_1cd1894e,
)
from ..aws_acmpca import ICertificateAuthority as _ICertificateAuthority_7f5d51a5
from ..aws_certificatemanager import ICertificate as _ICertificate_c7bbdc16
from ..aws_iam import Grant as _Grant_bcb5eae7, IGrantable as _IGrantable_4c5a91d1
from ..aws_servicediscovery import IService as _IService_66c1fbd2


class AccessLog(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.AccessLog",
):
    '''(experimental) Configuration for Envoy Access logs for mesh endpoints.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        vpc = ec2.Vpc(self, "vpc")
        namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
            vpc=vpc,
            name="domain.local"
        )
        service = namespace.create_service("Svc")
        node = mesh.add_virtual_node("virtual-node",
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8081,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=cdk.Duration.seconds(5),  # minimum
                    path="/health-check-path",
                    timeout=cdk.Duration.seconds(2),  # minimum
                    unhealthy_threshold=2
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFilePath") # type: ignore[misc]
    @builtins.classmethod
    def from_file_path(cls, file_path: builtins.str) -> "AccessLog":
        '''(experimental) Path to a file to write access logs to.

        :param file_path: -

        :default: - no file based access logging

        :stability: experimental
        '''
        return typing.cast("AccessLog", jsii.sinvoke(cls, "fromFilePath", [file_path]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "AccessLogConfig":
        '''(experimental) Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        ...


class _AccessLogProxy(AccessLog):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "AccessLogConfig":
        '''(experimental) Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        return typing.cast("AccessLogConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AccessLog).__jsii_proxy_class__ = lambda : _AccessLogProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.AccessLogConfig",
    jsii_struct_bases=[],
    name_mapping={
        "virtual_gateway_access_log": "virtualGatewayAccessLog",
        "virtual_node_access_log": "virtualNodeAccessLog",
    },
)
class AccessLogConfig:
    def __init__(
        self,
        *,
        virtual_gateway_access_log: typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"] = None,
        virtual_node_access_log: typing.Optional["CfnVirtualNode.AccessLogProperty"] = None,
    ) -> None:
        '''(experimental) All Properties for Envoy Access logs for mesh endpoints.

        :param virtual_gateway_access_log: (experimental) VirtualGateway CFN configuration for Access Logging. Default: - no access logging
        :param virtual_node_access_log: (experimental) VirtualNode CFN configuration for Access Logging. Default: - no access logging

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            access_log_config = appmesh.AccessLogConfig(
                virtual_gateway_access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                        path="path"
                    )
                ),
                virtual_node_access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                    file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                        path="path"
                    )
                )
            )
        '''
        if isinstance(virtual_gateway_access_log, dict):
            virtual_gateway_access_log = CfnVirtualGateway.VirtualGatewayAccessLogProperty(**virtual_gateway_access_log)
        if isinstance(virtual_node_access_log, dict):
            virtual_node_access_log = CfnVirtualNode.AccessLogProperty(**virtual_node_access_log)
        self._values: typing.Dict[str, typing.Any] = {}
        if virtual_gateway_access_log is not None:
            self._values["virtual_gateway_access_log"] = virtual_gateway_access_log
        if virtual_node_access_log is not None:
            self._values["virtual_node_access_log"] = virtual_node_access_log

    @builtins.property
    def virtual_gateway_access_log(
        self,
    ) -> typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"]:
        '''(experimental) VirtualGateway CFN configuration for Access Logging.

        :default: - no access logging

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway_access_log")
        return typing.cast(typing.Optional["CfnVirtualGateway.VirtualGatewayAccessLogProperty"], result)

    @builtins.property
    def virtual_node_access_log(
        self,
    ) -> typing.Optional["CfnVirtualNode.AccessLogProperty"]:
        '''(experimental) VirtualNode CFN configuration for Access Logging.

        :default: - no access logging

        :stability: experimental
        '''
        result = self._values.get("virtual_node_access_log")
        return typing.cast(typing.Optional["CfnVirtualNode.AccessLogProperty"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AccessLogConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class Backend(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.Backend",
):
    '''(experimental) Contains static factory methods to create backends.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        # router: appmesh.VirtualRouter
        # service: cloudmap.Service
        
        
        node = appmesh.VirtualNode(self, "node",
            mesh=mesh,
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8080,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=cdk.Duration.seconds(5),
                    path="/ping",
                    timeout=cdk.Duration.seconds(2),
                    unhealthy_threshold=2
                ),
                timeout=cdk.aws_appmesh.HttpTimeout(
                    idle=cdk.Duration.seconds(5)
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
        
        virtual_service = appmesh.VirtualService(self, "service-1",
            virtual_service_provider=appmesh.VirtualServiceProvider.virtual_router(router),
            virtual_service_name="service1.domain.local"
        )
        
        node.add_backend(appmesh.Backend.virtual_service(virtual_service))
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="virtualService") # type: ignore[misc]
    @builtins.classmethod
    def virtual_service(
        cls,
        virtual_service: "IVirtualService",
        *,
        tls_client_policy: typing.Optional["TlsClientPolicy"] = None,
    ) -> "Backend":
        '''(experimental) Construct a Virtual Service backend.

        :param virtual_service: -
        :param tls_client_policy: (experimental) TLS properties for Client policy for the backend. Default: - none

        :stability: experimental
        '''
        props = VirtualServiceBackendOptions(tls_client_policy=tls_client_policy)

        return typing.cast("Backend", jsii.sinvoke(cls, "virtualService", [virtual_service, props]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, _scope: _Construct_e78e779f) -> "BackendConfig":
        '''(experimental) Return backend config.

        :param _scope: -

        :stability: experimental
        '''
        ...


class _BackendProxy(Backend):
    @jsii.member(jsii_name="bind")
    def bind(self, _scope: _Construct_e78e779f) -> "BackendConfig":
        '''(experimental) Return backend config.

        :param _scope: -

        :stability: experimental
        '''
        return typing.cast("BackendConfig", jsii.invoke(self, "bind", [_scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Backend).__jsii_proxy_class__ = lambda : _BackendProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.BackendConfig",
    jsii_struct_bases=[],
    name_mapping={"virtual_service_backend": "virtualServiceBackend"},
)
class BackendConfig:
    def __init__(
        self,
        *,
        virtual_service_backend: "CfnVirtualNode.BackendProperty",
    ) -> None:
        '''(experimental) Properties for a backend.

        :param virtual_service_backend: (experimental) Config for a Virtual Service backend.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            backend_config = appmesh.BackendConfig(
                virtual_service_backend=appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
            
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )
            )
        '''
        if isinstance(virtual_service_backend, dict):
            virtual_service_backend = CfnVirtualNode.BackendProperty(**virtual_service_backend)
        self._values: typing.Dict[str, typing.Any] = {
            "virtual_service_backend": virtual_service_backend,
        }

    @builtins.property
    def virtual_service_backend(self) -> "CfnVirtualNode.BackendProperty":
        '''(experimental) Config for a Virtual Service backend.

        :stability: experimental
        '''
        result = self._values.get("virtual_service_backend")
        assert result is not None, "Required property 'virtual_service_backend' is missing"
        return typing.cast("CfnVirtualNode.BackendProperty", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackendConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.BackendDefaults",
    jsii_struct_bases=[],
    name_mapping={"tls_client_policy": "tlsClientPolicy"},
)
class BackendDefaults:
    def __init__(
        self,
        *,
        tls_client_policy: typing.Optional["TlsClientPolicy"] = None,
    ) -> None:
        '''(experimental) Represents the properties needed to define backend defaults.

        :param tls_client_policy: (experimental) TLS properties for Client policy for backend defaults. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # mesh: appmesh.Mesh
            # service: cloudmap.Service
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=8080,
                    health_check=appmesh.HealthCheck.http(
                        healthy_threshold=3,
                        interval=cdk.Duration.seconds(5),
                        path="/ping",
                        timeout=cdk.Duration.seconds(2),
                        unhealthy_threshold=2
                    ),
                    timeout=cdk.aws_appmesh.HttpTimeout(
                        idle=cdk.Duration.seconds(5)
                    )
                )],
                backend_defaults=cdk.aws_appmesh.BackendDefaults(
                    tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
                        validation=cdk.aws_appmesh.TlsValidation(
                            trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
                        )
                    )
                ),
                access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
            )
            
            cdk.Tags.of(node).add("Environment", "Dev")
        '''
        if isinstance(tls_client_policy, dict):
            tls_client_policy = TlsClientPolicy(**tls_client_policy)
        self._values: typing.Dict[str, typing.Any] = {}
        if tls_client_policy is not None:
            self._values["tls_client_policy"] = tls_client_policy

    @builtins.property
    def tls_client_policy(self) -> typing.Optional["TlsClientPolicy"]:
        '''(experimental) TLS properties for Client policy for backend defaults.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls_client_policy")
        return typing.cast(typing.Optional["TlsClientPolicy"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BackendDefaults(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnGatewayRoute(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnGatewayRoute",
):
    '''A CloudFormation ``AWS::AppMesh::GatewayRoute``.

    Creates a gateway route.

    A gateway route is attached to a virtual gateway and routes traffic to an existing virtual service. If a route matches a request, it can distribute traffic to a target virtual service.

    For more information about gateway routes, see `Gateway routes <https://docs.aws.amazon.com/app-mesh/latest/userguide/gateway-routes.html>`_ .

    :cloudformationResource: AWS::AppMesh::GatewayRoute
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_gateway_route = appmesh.CfnGatewayRoute(self, "MyCfnGatewayRoute",
            mesh_name="meshName",
            spec=appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        service_name="serviceName"
                    )
                ),
                http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
        
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                priority=123
            ),
            virtual_gateway_name="virtualGatewayName",
        
            # the properties below are optional
            gateway_route_name="gatewayRouteName",
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef],
        virtual_gateway_name: builtins.str,
        gateway_route_name: typing.Optional[builtins.str] = None,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::GatewayRoute``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh that the resource resides in.
        :param spec: The specifications of the gateway route.
        :param virtual_gateway_name: The virtual gateway that the gateway route is associated with.
        :param gateway_route_name: The name of the gateway route.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the gateway route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        props = CfnGatewayRouteProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_gateway_name=virtual_gateway_name,
            gateway_route_name=gateway_route_name,
            mesh_owner=mesh_owner,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the gateway route.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrGatewayRouteName")
    def attr_gateway_route_name(self) -> builtins.str:
        '''The name of the gateway route.

        :cloudformationAttribute: GatewayRouteName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrGatewayRouteName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the gateway route resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the gateway route.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVirtualGatewayName")
    def attr_virtual_gateway_name(self) -> builtins.str:
        '''The name of the virtual gateway that the gateway route is associated with.

        :cloudformationAttribute: VirtualGatewayName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualGatewayName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the gateway route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the resource resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef]:
        '''The specifications of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-spec
        '''
        return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnGatewayRoute.GatewayRouteSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "spec", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualGatewayName")
    def virtual_gateway_name(self) -> builtins.str:
        '''The virtual gateway that the gateway route is associated with.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-virtualgatewayname
        '''
        return typing.cast(builtins.str, jsii.get(self, "virtualGatewayName"))

    @virtual_gateway_name.setter
    def virtual_gateway_name(self, value: builtins.str) -> None:
        jsii.set(self, "virtualGatewayName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gatewayRouteName")
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-gatewayroutename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "gatewayRouteName"))

    @gateway_route_name.setter
    def gateway_route_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "gatewayRouteName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshOwner", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "suffix": "suffix"},
    )
    class GatewayRouteHostnameMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the gateway route host name to match.

            :param exact: The exact host name to match on.
            :param suffix: The specified ending characters of the host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_hostname_match_property = appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                    exact="exact",
                    suffix="suffix"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html#cfn-appmesh-gatewayroute-gatewayroutehostnamematch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The specified ending characters of the host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamematch.html#cfn-appmesh-gatewayroute-gatewayroutehostnamematch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteHostnameMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"default_target_hostname": "defaultTargetHostname"},
    )
    class GatewayRouteHostnameRewriteProperty:
        def __init__(
            self,
            *,
            default_target_hostname: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the gateway route host name to rewrite.

            :param default_target_hostname: The default target host name to write to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamerewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_hostname_rewrite_property = appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                    default_target_hostname="defaultTargetHostname"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if default_target_hostname is not None:
                self._values["default_target_hostname"] = default_target_hostname

        @builtins.property
        def default_target_hostname(self) -> typing.Optional[builtins.str]:
            '''The default target host name to write to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutehostnamerewrite.html#cfn-appmesh-gatewayroute-gatewayroutehostnamerewrite-defaulttargethostname
            '''
            result = self._values.get("default_target_hostname")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteHostnameRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class GatewayRouteMetadataMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the method header to be matched.

            :param exact: The exact method header to be matched on.
            :param prefix: The specified beginning characters of the method header to be matched on.
            :param range: An object that represents the range of values to match on.
            :param regex: The regex used to match the method header.
            :param suffix: The specified ending characters of the method header to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_metadata_match_property = appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact method header to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The specified beginning characters of the method header to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the method header.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The specified ending characters of the method header to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutemetadatamatch.html#cfn-appmesh-gatewayroute-gatewayroutemetadatamatch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteMetadataMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"end": "end", "start": "start"},
    )
    class GatewayRouteRangeMatchProperty:
        def __init__(self, *, end: jsii.Number, start: jsii.Number) -> None:
            '''An object that represents the range of values to match on.

            The first character of the range is included in the range, though the last character is not. For example, if the range specified were 1-100, only values 1-99 would be matched.

            :param end: The end of the range.
            :param start: The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_range_match_property = appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                    end=123,
                    start=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "end": end,
                "start": start,
            }

        @builtins.property
        def end(self) -> jsii.Number:
            '''The end of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html#cfn-appmesh-gatewayroute-gatewayrouterangematch-end
            '''
            result = self._values.get("end")
            assert result is not None, "Required property 'end' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def start(self) -> jsii.Number:
            '''The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayrouterangematch.html#cfn-appmesh-gatewayroute-gatewayrouterangematch-start
            '''
            result = self._values.get("start")
            assert result is not None, "Required property 'start' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteRangeMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "grpc_route": "grpcRoute",
            "http2_route": "http2Route",
            "http_route": "httpRoute",
            "priority": "priority",
        },
    )
    class GatewayRouteSpecProperty:
        def __init__(
            self,
            *,
            grpc_route: typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", _IResolvable_a771d0ef]] = None,
            http2_route: typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]] = None,
            http_route: typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]] = None,
            priority: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a gateway route specification.

            Specify one gateway route type.

            :param grpc_route: An object that represents the specification of a gRPC gateway route.
            :param http2_route: An object that represents the specification of an HTTP/2 gateway route.
            :param http_route: An object that represents the specification of an HTTP gateway route.
            :param priority: The ordering of the gateway routes spec.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_spec_property = appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                    grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                )
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            service_name="serviceName"
                        )
                    ),
                    http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                )
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                )
                            ),
                
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    priority=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if grpc_route is not None:
                self._values["grpc_route"] = grpc_route
            if http2_route is not None:
                self._values["http2_route"] = http2_route
            if http_route is not None:
                self._values["http_route"] = http_route
            if priority is not None:
                self._values["priority"] = priority

        @builtins.property
        def grpc_route(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of a gRPC gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-grpcroute
            '''
            result = self._values.get("grpc_route")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2_route(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP/2 gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-http2route
            '''
            result = self._values.get("http2_route")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http_route(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-httproute
            '''
            result = self._values.get("http_route")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The ordering of the gateway routes spec.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutespec.html#cfn-appmesh-gatewayroute-gatewayroutespec-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteTargetProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service": "virtualService"},
    )
    class GatewayRouteTargetProperty:
        def __init__(
            self,
            *,
            virtual_service: typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents a gateway route target.

            :param virtual_service: An object that represents a virtual service gateway route target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_target_property = appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                    virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                        virtual_service_name="virtualServiceName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "virtual_service": virtual_service,
            }

        @builtins.property
        def virtual_service(
            self,
        ) -> typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", _IResolvable_a771d0ef]:
            '''An object that represents a virtual service gateway route target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutetarget.html#cfn-appmesh-gatewayroute-gatewayroutetarget-virtualservice
            '''
            result = self._values.get("virtual_service")
            assert result is not None, "Required property 'virtual_service' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteVirtualServiceProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service_name": "virtualServiceName"},
    )
    class GatewayRouteVirtualServiceProperty:
        def __init__(self, *, virtual_service_name: builtins.str) -> None:
            '''An object that represents the virtual service that traffic is routed to.

            :param virtual_service_name: The name of the virtual service that traffic is routed to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutevirtualservice.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                gateway_route_virtual_service_property = appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                    virtual_service_name="virtualServiceName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "virtual_service_name": virtual_service_name,
            }

        @builtins.property
        def virtual_service_name(self) -> builtins.str:
            '''The name of the virtual service that traffic is routed to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-gatewayroutevirtualservice.html#cfn-appmesh-gatewayroute-gatewayroutevirtualservice-virtualservicename
            '''
            result = self._values.get("virtual_service_name")
            assert result is not None, "Required property 'virtual_service_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GatewayRouteVirtualServiceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"target": "target", "rewrite": "rewrite"},
    )
    class GrpcGatewayRouteActionProperty:
        def __init__(
            self,
            *,
            target: typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef],
            rewrite: typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param target: An object that represents the target that traffic is routed to when a request matches the gateway route.
            :param rewrite: The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_action_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                    target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                        virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                            virtual_service_name="virtualServiceName"
                        )
                    ),
                
                    # the properties below are optional
                    rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                            default_target_hostname="defaultTargetHostname"
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "target": target,
            }
            if rewrite is not None:
                self._values["rewrite"] = rewrite

        @builtins.property
        def target(
            self,
        ) -> typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef]:
            '''An object that represents the target that traffic is routed to when a request matches the gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html#cfn-appmesh-gatewayroute-grpcgatewayrouteaction-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def rewrite(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", _IResolvable_a771d0ef]]:
            '''The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouteaction.html#cfn-appmesh-gatewayroute-grpcgatewayrouteaction-rewrite
            '''
            result = self._values.get("rewrite")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GrpcGatewayRouteRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "hostname": "hostname",
            "metadata": "metadata",
            "service_name": "serviceName",
        },
    )
    class GrpcGatewayRouteMatchProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]] = None,
            metadata: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", _IResolvable_a771d0ef]]]] = None,
            service_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param hostname: The gateway route host name to be matched on.
            :param metadata: The gateway route metadata to be matched on.
            :param service_name: The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_match_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                        exact="exact",
                        suffix="suffix"
                    ),
                    metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    service_name="serviceName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if metadata is not None:
                self._values["metadata"] = metadata
            if service_name is not None:
                self._values["service_name"] = service_name

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]]:
            '''The gateway route host name to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", _IResolvable_a771d0ef]]]]:
            '''The gateway route metadata to be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.GrpcGatewayRouteMetadataProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def service_name(self) -> typing.Optional[builtins.str]:
            '''The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutematch.html#cfn-appmesh-gatewayroute-grpcgatewayroutematch-servicename
            '''
            result = self._values.get("service_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class GrpcGatewayRouteMetadataProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object representing the metadata of the gateway route.

            :param name: A name for the gateway route metadata.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: The criteria for determining a metadata match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_metadata_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the gateway route metadata.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", _IResolvable_a771d0ef]]:
            '''The criteria for determining a metadata match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroutemetadata.html#cfn-appmesh-gatewayroute-grpcgatewayroutemetadata-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteMetadataMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match"},
    )
    class GrpcGatewayRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", _IResolvable_a771d0ef],
            match: typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents a gRPC gateway route.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
                
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        service_name="serviceName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "action": action,
                "match": match,
            }

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html#cfn-appmesh-gatewayroute-grpcgatewayroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GrpcGatewayRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayroute.html#cfn-appmesh-gatewayroute-grpcgatewayroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GrpcGatewayRouteMatchProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname"},
    )
    class GrpcGatewayRouteRewriteProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the gateway route to rewrite.

            :param hostname: The host name of the gateway route to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouterewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_gateway_route_rewrite_property = appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                        default_target_hostname="defaultTargetHostname"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]]:
            '''The host name of the gateway route to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-grpcgatewayrouterewrite.html#cfn-appmesh-gatewayroute-grpcgatewayrouterewrite-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcGatewayRouteRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"target": "target", "rewrite": "rewrite"},
    )
    class HttpGatewayRouteActionProperty:
        def __init__(
            self,
            *,
            target: typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef],
            rewrite: typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param target: An object that represents the target that traffic is routed to when a request matches the gateway route.
            :param rewrite: The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_action_property = appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                    target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                        virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                            virtual_service_name="virtualServiceName"
                        )
                    ),
                
                    # the properties below are optional
                    rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                            default_target_hostname="defaultTargetHostname"
                        ),
                        path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                            exact="exact"
                        ),
                        prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                            default_prefix="defaultPrefix",
                            value="value"
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "target": target,
            }
            if rewrite is not None:
                self._values["rewrite"] = rewrite

        @builtins.property
        def target(
            self,
        ) -> typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef]:
            '''An object that represents the target that traffic is routed to when a request matches the gateway route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html#cfn-appmesh-gatewayroute-httpgatewayrouteaction-target
            '''
            result = self._values.get("target")
            assert result is not None, "Required property 'target' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.GatewayRouteTargetProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def rewrite(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", _IResolvable_a771d0ef]]:
            '''The gateway route action to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteaction.html#cfn-appmesh-gatewayroute-httpgatewayrouteaction-rewrite
            '''
            result = self._values.get("rewrite")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class HttpGatewayRouteHeaderMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_header_match_property = appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteRangeMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheadermatch.html#cfn-appmesh-gatewayroute-httpgatewayrouteheadermatch-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteHeaderMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class HttpGatewayRouteHeaderProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the HTTP header in the gateway route.

            :param name: A name for the HTTP header in the gateway route that will be matched on.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: An object that represents the method and value to match with the header value sent in a request. Specify one match method.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_header_property = appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the HTTP header in the gateway route that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteheader.html#cfn-appmesh-gatewayroute-httpgatewayrouteheader-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "headers": "headers",
            "hostname": "hostname",
            "method": "method",
            "path": "path",
            "prefix": "prefix",
            "query_parameters": "queryParameters",
        },
    )
    class HttpGatewayRouteMatchProperty:
        def __init__(
            self,
            *,
            headers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", _IResolvable_a771d0ef]]]] = None,
            hostname: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]] = None,
            method: typing.Optional[builtins.str] = None,
            path: typing.Optional[typing.Union["CfnGatewayRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]] = None,
            prefix: typing.Optional[builtins.str] = None,
            query_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnGatewayRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param headers: The client request headers to match on.
            :param hostname: The host name to match on.
            :param method: The method to match on.
            :param path: The path to match on.
            :param prefix: Specifies the path to match requests with. This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .
            :param query_parameters: The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_match_property = appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                    headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                        exact="exact",
                        suffix="suffix"
                    ),
                    method="method",
                    path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                        exact="exact",
                        regex="regex"
                    ),
                    prefix="prefix",
                    query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                        name="name",
                
                        # the properties below are optional
                        match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                            exact="exact"
                        )
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if headers is not None:
                self._values["headers"] = headers
            if hostname is not None:
                self._values["hostname"] = hostname
            if method is not None:
                self._values["method"] = method
            if path is not None:
                self._values["path"] = path
            if prefix is not None:
                self._values["prefix"] = prefix
            if query_parameters is not None:
                self._values["query_parameters"] = query_parameters

        @builtins.property
        def headers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", _IResolvable_a771d0ef]]]]:
            '''The client request headers to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.HttpGatewayRouteHeaderProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]]:
            '''The host name to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The method to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]]:
            '''The path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''Specifies the path to match requests with.

            This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def query_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]]:
            '''The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutematch.html#cfn-appmesh-gatewayroute-httpgatewayroutematch-queryparameters
            '''
            result = self._values.get("query_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnGatewayRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpGatewayRoutePathRewriteProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object that represents the path to rewrite.

            :param exact: The exact path to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutepathrewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_path_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                    exact="exact"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroutepathrewrite.html#cfn-appmesh-gatewayroute-httpgatewayroutepathrewrite-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRoutePathRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"default_prefix": "defaultPrefix", "value": "value"},
    )
    class HttpGatewayRoutePrefixRewriteProperty:
        def __init__(
            self,
            *,
            default_prefix: typing.Optional[builtins.str] = None,
            value: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the beginning characters of the route to rewrite.

            :param default_prefix: The default prefix used to replace the incoming route prefix when rewritten.
            :param value: The value used to replace the incoming route prefix when rewritten.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_prefix_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                    default_prefix="defaultPrefix",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if default_prefix is not None:
                self._values["default_prefix"] = default_prefix
            if value is not None:
                self._values["value"] = value

        @builtins.property
        def default_prefix(self) -> typing.Optional[builtins.str]:
            '''The default prefix used to replace the incoming route prefix when rewritten.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouteprefixrewrite-defaultprefix
            '''
            result = self._values.get("default_prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def value(self) -> typing.Optional[builtins.str]:
            '''The value used to replace the incoming route prefix when rewritten.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouteprefixrewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouteprefixrewrite-value
            '''
            result = self._values.get("value")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRoutePrefixRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "match": "match"},
    )
    class HttpGatewayRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", _IResolvable_a771d0ef],
            match: typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents an HTTP gateway route.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_property = appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
                
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
                
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "action": action,
                "match": match,
            }

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html#cfn-appmesh-gatewayroute-httpgatewayroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.HttpGatewayRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayroute.html#cfn-appmesh-gatewayroute-httpgatewayroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnGatewayRoute.HttpGatewayRouteMatchProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "path": "path", "prefix": "prefix"},
    )
    class HttpGatewayRouteRewriteProperty:
        def __init__(
            self,
            *,
            hostname: typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]] = None,
            path: typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", _IResolvable_a771d0ef]] = None,
            prefix: typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object representing the gateway route to rewrite.

            :param hostname: The host name to rewrite.
            :param path: The path to rewrite.
            :param prefix: The specified beginning characters to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_gateway_route_rewrite_property = appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                    hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                        default_target_hostname="defaultTargetHostname"
                    ),
                    path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                        exact="exact"
                    ),
                    prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                        default_prefix="defaultPrefix",
                        value="value"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if hostname is not None:
                self._values["hostname"] = hostname
            if path is not None:
                self._values["path"] = path
            if prefix is not None:
                self._values["prefix"] = prefix

        @builtins.property
        def hostname(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]]:
            '''The host name to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-hostname
            '''
            result = self._values.get("hostname")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.GatewayRouteHostnameRewriteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", _IResolvable_a771d0ef]]:
            '''The path to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def prefix(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", _IResolvable_a771d0ef]]:
            '''The specified beginning characters to rewrite.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpgatewayrouterewrite.html#cfn-appmesh-gatewayroute-httpgatewayrouterewrite-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpGatewayRouteRewriteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpPathMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "regex": "regex"},
    )
    class HttpPathMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            regex: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the path to match in the request.

            :param exact: The exact path to match on.
            :param regex: The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_path_match_property = appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if regex is not None:
                self._values["regex"] = regex

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html#cfn-appmesh-gatewayroute-httppathmatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httppathmatch.html#cfn-appmesh-gatewayroute-httppathmatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpPathMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpQueryParameterMatchProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object representing the query parameter to match.

            :param exact: The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpqueryparametermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_query_parameter_match_property = appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                    exact="exact"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-httpqueryparametermatch.html#cfn-appmesh-gatewayroute-httpqueryparametermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpQueryParameterMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnGatewayRoute.QueryParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "match": "match"},
    )
    class QueryParameterProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            match: typing.Optional[typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the query parameter in the request.

            :param name: A name for the query parameter that will be matched on.
            :param match: The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                query_parameter_property = appmesh.CfnGatewayRoute.QueryParameterProperty(
                    name="name",
                
                    # the properties below are optional
                    match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                        exact="exact"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the query parameter that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html#cfn-appmesh-gatewayroute-queryparameter-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]]:
            '''The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-gatewayroute-queryparameter.html#cfn-appmesh-gatewayroute-queryparameter-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnGatewayRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnGatewayRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_gateway_name": "virtualGatewayName",
        "gateway_route_name": "gatewayRouteName",
        "mesh_owner": "meshOwner",
        "tags": "tags",
    },
)
class CfnGatewayRouteProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, _IResolvable_a771d0ef],
        virtual_gateway_name: builtins.str,
        gateway_route_name: typing.Optional[builtins.str] = None,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``CfnGatewayRoute``.

        :param mesh_name: The name of the service mesh that the resource resides in.
        :param spec: The specifications of the gateway route.
        :param virtual_gateway_name: The virtual gateway that the gateway route is associated with.
        :param gateway_route_name: The name of the gateway route.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the gateway route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_gateway_route_props = appmesh.CfnGatewayRouteProps(
                mesh_name="meshName",
                spec=appmesh.CfnGatewayRoute.GatewayRouteSpecProperty(
                    grpc_route=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                )
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            service_name="serviceName"
                        )
                    ),
                    http2_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                )
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    http_route=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                        action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                            target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                                virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                    virtual_service_name="virtualServiceName"
                                )
                            ),
            
                            # the properties below are optional
                            rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                                hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                    default_target_hostname="defaultTargetHostname"
                                ),
                                path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                    exact="exact"
                                ),
                                prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                    default_prefix="defaultPrefix",
                                    value="value"
                                )
                            )
                        ),
                        match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                            headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                                exact="exact",
                                suffix="suffix"
                            ),
                            method="method",
                            path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )]
                        )
                    ),
                    priority=123
                ),
                virtual_gateway_name="virtualGatewayName",
            
                # the properties below are optional
                gateway_route_name="gatewayRouteName",
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_gateway_name": virtual_gateway_name,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the resource resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, _IResolvable_a771d0ef]:
        '''The specifications of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnGatewayRoute.GatewayRouteSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def virtual_gateway_name(self) -> builtins.str:
        '''The virtual gateway that the gateway route is associated with.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-virtualgatewayname
        '''
        result = self._values.get("virtual_gateway_name")
        assert result is not None, "Required property 'virtual_gateway_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''The name of the gateway route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-gatewayroutename
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the gateway route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-gatewayroute.html#cfn-appmesh-gatewayroute-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnMesh(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnMesh",
):
    '''A CloudFormation ``AWS::AppMesh::Mesh``.

    Creates a service mesh.

    A service mesh is a logical boundary for network traffic between services that are represented by resources within the mesh. After you create your service mesh, you can create virtual services, virtual nodes, virtual routers, and routes to distribute traffic between the applications in your mesh.

    For more information about service meshes, see `Service meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html>`_ .

    :cloudformationResource: AWS::AppMesh::Mesh
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_mesh = appmesh.CfnMesh(self, "MyCfnMesh",
            mesh_name="meshName",
            spec=appmesh.CfnMesh.MeshSpecProperty(
                egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                    type="type"
                )
            ),
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: typing.Optional[builtins.str] = None,
        spec: typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::Mesh``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name to use for the service mesh.
        :param spec: The service mesh specification to apply.
        :param tags: Optional metadata that you can apply to the service mesh to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        props = CfnMeshProps(mesh_name=mesh_name, spec=spec, tags=tags)

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the mesh.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the mesh.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the service mesh to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the service mesh.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-meshname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]]:
        '''The service mesh specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-spec
        '''
        return typing.cast(typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Optional[typing.Union["CfnMesh.MeshSpecProperty", _IResolvable_a771d0ef]],
    ) -> None:
        jsii.set(self, "spec", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnMesh.EgressFilterProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class EgressFilterProperty:
        def __init__(self, *, type: builtins.str) -> None:
            '''An object that represents the egress filter rules for a service mesh.

            :param type: The egress filter type. By default, the type is ``DROP_ALL`` , which allows egress only from virtual nodes to other defined resources in the service mesh (and any traffic to ``*.amazonaws.com`` for AWS API calls). You can set the egress filter type to ``ALLOW_ALL`` to allow egress to any endpoint inside or outside of the service mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                egress_filter_property = appmesh.CfnMesh.EgressFilterProperty(
                    type="type"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''The egress filter type.

            By default, the type is ``DROP_ALL`` , which allows egress only from virtual nodes to other defined resources in the service mesh (and any traffic to ``*.amazonaws.com`` for AWS API calls). You can set the egress filter type to ``ALLOW_ALL`` to allow egress to any endpoint inside or outside of the service mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-egressfilter.html#cfn-appmesh-mesh-egressfilter-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EgressFilterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnMesh.MeshServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={},
    )
    class MeshServiceDiscoveryProperty:
        def __init__(self) -> None:
            '''
            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                mesh_service_discovery_property = appmesh.CfnMesh.MeshServiceDiscoveryProperty()
            '''
            self._values: typing.Dict[str, typing.Any] = {}

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MeshServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnMesh.MeshSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"egress_filter": "egressFilter"},
    )
    class MeshSpecProperty:
        def __init__(
            self,
            *,
            egress_filter: typing.Optional[typing.Union["CfnMesh.EgressFilterProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a service mesh.

            :param egress_filter: The egress filter rules for the service mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                mesh_spec_property = appmesh.CfnMesh.MeshSpecProperty(
                    egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                        type="type"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if egress_filter is not None:
                self._values["egress_filter"] = egress_filter

        @builtins.property
        def egress_filter(
            self,
        ) -> typing.Optional[typing.Union["CfnMesh.EgressFilterProperty", _IResolvable_a771d0ef]]:
            '''The egress filter rules for the service mesh.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-mesh-meshspec.html#cfn-appmesh-mesh-meshspec-egressfilter
            '''
            result = self._values.get("egress_filter")
            return typing.cast(typing.Optional[typing.Union["CfnMesh.EgressFilterProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MeshSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnMeshProps",
    jsii_struct_bases=[],
    name_mapping={"mesh_name": "meshName", "spec": "spec", "tags": "tags"},
)
class CfnMeshProps:
    def __init__(
        self,
        *,
        mesh_name: typing.Optional[builtins.str] = None,
        spec: typing.Optional[typing.Union[CfnMesh.MeshSpecProperty, _IResolvable_a771d0ef]] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``CfnMesh``.

        :param mesh_name: The name to use for the service mesh.
        :param spec: The service mesh specification to apply.
        :param tags: Optional metadata that you can apply to the service mesh to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_mesh_props = appmesh.CfnMeshProps(
                mesh_name="meshName",
                spec=appmesh.CfnMesh.MeshSpecProperty(
                    egress_filter=appmesh.CfnMesh.EgressFilterProperty(
                        type="type"
                    )
                ),
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if mesh_name is not None:
            self._values["mesh_name"] = mesh_name
        if spec is not None:
            self._values["spec"] = spec
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the service mesh.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-meshname
        '''
        result = self._values.get("mesh_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Optional[typing.Union[CfnMesh.MeshSpecProperty, _IResolvable_a771d0ef]]:
        '''The service mesh specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-spec
        '''
        result = self._values.get("spec")
        return typing.cast(typing.Optional[typing.Union[CfnMesh.MeshSpecProperty, _IResolvable_a771d0ef]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the service mesh to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-mesh.html#cfn-appmesh-mesh-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnMeshProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnRoute(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnRoute",
):
    '''A CloudFormation ``AWS::AppMesh::Route``.

    Creates a route that is associated with a virtual router.

    You can route several different protocols and define a retry policy for a route. Traffic can be routed to one or more virtual nodes.

    For more information about routes, see `Routes <https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html>`_ .

    :cloudformationResource: AWS::AppMesh::Route
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_route = appmesh.CfnRoute(self, "MyCfnRoute",
            mesh_name="meshName",
            spec=appmesh.CfnRoute.RouteSpecProperty(
                grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                    action=appmesh.CfnRoute.GrpcRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
                    match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                        metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method_name="methodName",
                        service_name="serviceName"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        grpc_retry_events=["grpcRetryEvents"],
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                http2_route=appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                http_route=appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
        
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
        
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
        
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
        
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                ),
                priority=123,
                tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                    action=appmesh.CfnRoute.TcpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
        
                    # the properties below are optional
                    timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            ),
            virtual_router_name="virtualRouterName",
        
            # the properties below are optional
            mesh_owner="meshOwner",
            route_name="routeName",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef],
        virtual_router_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        route_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::Route``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the route in.
        :param spec: The route specification to apply.
        :param virtual_router_name: The name of the virtual router in which to create the route. If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param route_name: The name to use for the route.
        :param tags: Optional metadata that you can apply to the route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        props = CfnRouteProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_router_name=virtual_router_name,
            mesh_owner=mesh_owner,
            route_name=route_name,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the route.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the route resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrRouteName")
    def attr_route_name(self) -> builtins.str:
        '''The name of the route.

        :cloudformationAttribute: RouteName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrRouteName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the route.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVirtualRouterName")
    def attr_virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router that the route is associated with.

        :cloudformationAttribute: VirtualRouterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualRouterName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the route in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(self) -> typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef]:
        '''The route specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-spec
        '''
        return typing.cast(typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnRoute.RouteSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "spec", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualRouterName")
    def virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router in which to create the route.

        If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-virtualroutername
        '''
        return typing.cast(builtins.str, jsii.get(self, "virtualRouterName"))

    @virtual_router_name.setter
    def virtual_router_name(self, value: builtins.str) -> None:
        jsii.set(self, "virtualRouterName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshOwner", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="routeName")
    def route_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-routename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "routeName"))

    @route_name.setter
    def route_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "routeName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.DurationProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class DurationProperty:
        def __init__(self, *, unit: builtins.str, value: jsii.Number) -> None:
            '''An object that represents a duration of time.

            :param unit: A unit of time.
            :param value: A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                duration_property = appmesh.CfnRoute.DurationProperty(
                    unit="unit",
                    value=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "unit": unit,
                "value": value,
            }

        @builtins.property
        def unit(self) -> builtins.str:
            '''A unit of time.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html#cfn-appmesh-route-duration-unit
            '''
            result = self._values.get("unit")
            assert result is not None, "Required property 'unit' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> jsii.Number:
            '''A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-duration.html#cfn-appmesh-route-duration-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRetryPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_retries": "maxRetries",
            "per_retry_timeout": "perRetryTimeout",
            "grpc_retry_events": "grpcRetryEvents",
            "http_retry_events": "httpRetryEvents",
            "tcp_retry_events": "tcpRetryEvents",
        },
    )
    class GrpcRetryPolicyProperty:
        def __init__(
            self,
            *,
            max_retries: jsii.Number,
            per_retry_timeout: typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef],
            grpc_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            http_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            tcp_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a retry policy.

            Specify at least one value for at least one of the types of ``RetryEvents`` , a value for ``maxRetries`` , and a value for ``perRetryTimeout`` . Both ``server-error`` and ``gateway-error`` under ``httpRetryEvents`` include the Envoy ``reset`` policy. For more information on the ``reset`` policy, see the `Envoy documentation <https://docs.aws.amazon.com/https://www.envoyproxy.io/docs/envoy/latest/configuration/http/http_filters/router_filter#x-envoy-retry-on>`_ .

            :param max_retries: The maximum number of retry attempts.
            :param per_retry_timeout: The timeout for each retry attempt.
            :param grpc_retry_events: Specify at least one of the valid values.
            :param http_retry_events: Specify at least one of the following values. - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511 - *gateway-error* – HTTP status codes 502, 503, and 504 - *client-error* – HTTP status code 409 - *stream-error* – Retry on refused stream
            :param tcp_retry_events: Specify a valid value. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_retry_policy_property = appmesh.CfnRoute.GrpcRetryPolicyProperty(
                    max_retries=123,
                    per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                
                    # the properties below are optional
                    grpc_retry_events=["grpcRetryEvents"],
                    http_retry_events=["httpRetryEvents"],
                    tcp_retry_events=["tcpRetryEvents"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_retries": max_retries,
                "per_retry_timeout": per_retry_timeout,
            }
            if grpc_retry_events is not None:
                self._values["grpc_retry_events"] = grpc_retry_events
            if http_retry_events is not None:
                self._values["http_retry_events"] = http_retry_events
            if tcp_retry_events is not None:
                self._values["tcp_retry_events"] = tcp_retry_events

        @builtins.property
        def max_retries(self) -> jsii.Number:
            '''The maximum number of retry attempts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-maxretries
            '''
            result = self._values.get("max_retries")
            assert result is not None, "Required property 'max_retries' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def per_retry_timeout(
            self,
        ) -> typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]:
            '''The timeout for each retry attempt.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-perretrytimeout
            '''
            result = self._values.get("per_retry_timeout")
            assert result is not None, "Required property 'per_retry_timeout' is missing"
            return typing.cast(typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def grpc_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the valid values.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-grpcretryevents
            '''
            result = self._values.get("grpc_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def http_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the following values.

            - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
            - *gateway-error* – HTTP status codes 502, 503, and 504
            - *client-error* – HTTP status code 409
            - *stream-error* – Retry on refused stream

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-httpretryevents
            '''
            result = self._values.get("http_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tcp_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify a valid value.

            The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcretrypolicy.html#cfn-appmesh-route-grpcretrypolicy-tcpretryevents
            '''
            result = self._values.get("tcp_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRetryPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class GrpcRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcrouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_action_property = appmesh.CfnRoute.GrpcRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcrouteaction.html#cfn-appmesh-route-grpcrouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "metadata": "metadata",
            "method_name": "methodName",
            "service_name": "serviceName",
        },
    )
    class GrpcRouteMatchProperty:
        def __init__(
            self,
            *,
            metadata: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnRoute.GrpcRouteMetadataProperty", _IResolvable_a771d0ef]]]] = None,
            method_name: typing.Optional[builtins.str] = None,
            service_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the criteria for determining a request match.

            :param metadata: An object that represents the data to match from the request.
            :param method_name: The method name to match from the request. If you specify a name, you must also specify a ``serviceName`` .
            :param service_name: The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_match_property = appmesh.CfnRoute.GrpcRouteMatchProperty(
                    metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnRoute.MatchRangeProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    method_name="methodName",
                    service_name="serviceName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if metadata is not None:
                self._values["metadata"] = metadata
            if method_name is not None:
                self._values["method_name"] = method_name
            if service_name is not None:
                self._values["service_name"] = service_name

        @builtins.property
        def metadata(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.GrpcRouteMetadataProperty", _IResolvable_a771d0ef]]]]:
            '''An object that represents the data to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-metadata
            '''
            result = self._values.get("metadata")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.GrpcRouteMetadataProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def method_name(self) -> typing.Optional[builtins.str]:
            '''The method name to match from the request.

            If you specify a name, you must also specify a ``serviceName`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-methodname
            '''
            result = self._values.get("method_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def service_name(self) -> typing.Optional[builtins.str]:
            '''The fully qualified domain name for the service to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutematch.html#cfn-appmesh-route-grpcroutematch-servicename
            '''
            result = self._values.get("service_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class GrpcRouteMetadataMatchMethodProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the match method.

            Specify one of the match values.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_metadata_match_method_property = appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnRoute.MatchRangeProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadatamatchmethod.html#cfn-appmesh-route-grpcroutemetadatamatchmethod-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMetadataMatchMethodProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteMetadataProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class GrpcRouteMetadataProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the match metadata for the route.

            :param name: The name of the route.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: An object that represents the data to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_metadata_property = appmesh.CfnRoute.GrpcRouteMetadataProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''The name of the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the data to match from the request.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroutemetadata.html#cfn-appmesh-route-grpcroutemetadata-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcRouteMetadataMatchMethodProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteMetadataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "match": "match",
            "retry_policy": "retryPolicy",
            "timeout": "timeout",
        },
    )
    class GrpcRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union["CfnRoute.GrpcRouteActionProperty", _IResolvable_a771d0ef],
            match: typing.Union["CfnRoute.GrpcRouteMatchProperty", _IResolvable_a771d0ef],
            retry_policy: typing.Optional[typing.Union["CfnRoute.GrpcRetryPolicyProperty", _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union["CfnRoute.GrpcTimeoutProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a gRPC route type.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param retry_policy: An object that represents a retry policy.
            :param timeout: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_route_property = appmesh.CfnRoute.GrpcRouteProperty(
                    action=appmesh.CfnRoute.GrpcRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
                    match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                        metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method_name="methodName",
                        service_name="serviceName"
                    ),
                
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                
                        # the properties below are optional
                        grpc_retry_events=["grpcRetryEvents"],
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "action": action,
                "match": match,
            }
            if retry_policy is not None:
                self._values["retry_policy"] = retry_policy
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnRoute.GrpcRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnRoute.GrpcRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnRoute.GrpcRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnRoute.GrpcRouteMatchProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def retry_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcRetryPolicyProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a retry policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-retrypolicy
            '''
            result = self._values.get("retry_policy")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcRetryPolicyProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpcroute.html#cfn-appmesh-route-grpcroute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.GrpcTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class GrpcTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_timeout_property = appmesh.CfnRoute.GrpcTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html#cfn-appmesh-route-grpctimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-grpctimeout.html#cfn-appmesh-route-grpctimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HeaderMatchMethodProperty",
        jsii_struct_bases=[],
        name_mapping={
            "exact": "exact",
            "prefix": "prefix",
            "range": "range",
            "regex": "regex",
            "suffix": "suffix",
        },
    )
    class HeaderMatchMethodProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            prefix: typing.Optional[builtins.str] = None,
            range: typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]] = None,
            regex: typing.Optional[builtins.str] = None,
            suffix: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the method and value to match with the header value sent in a request.

            Specify one match method.

            :param exact: The value sent by the client must match the specified value exactly.
            :param prefix: The value sent by the client must begin with the specified characters.
            :param range: An object that represents the range of values to match on.
            :param regex: The value sent by the client must include the specified characters.
            :param suffix: The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                header_match_method_property = appmesh.CfnRoute.HeaderMatchMethodProperty(
                    exact="exact",
                    prefix="prefix",
                    range=appmesh.CfnRoute.MatchRangeProperty(
                        end=123,
                        start=123
                    ),
                    regex="regex",
                    suffix="suffix"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if prefix is not None:
                self._values["prefix"] = prefix
            if range is not None:
                self._values["range"] = range
            if regex is not None:
                self._values["regex"] = regex
            if suffix is not None:
                self._values["suffix"] = suffix

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must match the specified value exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must begin with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def range(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the range of values to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-range
            '''
            result = self._values.get("range")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.MatchRangeProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must include the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def suffix(self) -> typing.Optional[builtins.str]:
            '''The value sent by the client must end with the specified characters.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-headermatchmethod.html#cfn-appmesh-route-headermatchmethod-suffix
            '''
            result = self._values.get("suffix")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HeaderMatchMethodProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpPathMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact", "regex": "regex"},
    )
    class HttpPathMatchProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[builtins.str] = None,
            regex: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object representing the path to match in the request.

            :param exact: The exact path to match on.
            :param regex: The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_path_match_property = appmesh.CfnRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact
            if regex is not None:
                self._values["regex"] = regex

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html#cfn-appmesh-route-httppathmatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def regex(self) -> typing.Optional[builtins.str]:
            '''The regex used to match the path.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httppathmatch.html#cfn-appmesh-route-httppathmatch-regex
            '''
            result = self._values.get("regex")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpPathMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpQueryParameterMatchProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class HttpQueryParameterMatchProperty:
        def __init__(self, *, exact: typing.Optional[builtins.str] = None) -> None:
            '''An object representing the query parameter to match.

            :param exact: The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpqueryparametermatch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_query_parameter_match_property = appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                    exact="exact"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[builtins.str]:
            '''The exact query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpqueryparametermatch.html#cfn-appmesh-route-httpqueryparametermatch-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpQueryParameterMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRetryPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_retries": "maxRetries",
            "per_retry_timeout": "perRetryTimeout",
            "http_retry_events": "httpRetryEvents",
            "tcp_retry_events": "tcpRetryEvents",
        },
    )
    class HttpRetryPolicyProperty:
        def __init__(
            self,
            *,
            max_retries: jsii.Number,
            per_retry_timeout: typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef],
            http_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
            tcp_retry_events: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents a retry policy.

            Specify at least one value for at least one of the types of ``RetryEvents`` , a value for ``maxRetries`` , and a value for ``perRetryTimeout`` . Both ``server-error`` and ``gateway-error`` under ``httpRetryEvents`` include the Envoy ``reset`` policy. For more information on the ``reset`` policy, see the `Envoy documentation <https://docs.aws.amazon.com/https://www.envoyproxy.io/docs/envoy/latest/configuration/http/http_filters/router_filter#x-envoy-retry-on>`_ .

            :param max_retries: The maximum number of retry attempts.
            :param per_retry_timeout: The timeout for each retry attempt.
            :param http_retry_events: Specify at least one of the following values. - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511 - *gateway-error* – HTTP status codes 502, 503, and 504 - *client-error* – HTTP status code 409 - *stream-error* – Retry on refused stream
            :param tcp_retry_events: Specify a valid value. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_retry_policy_property = appmesh.CfnRoute.HttpRetryPolicyProperty(
                    max_retries=123,
                    per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                
                    # the properties below are optional
                    http_retry_events=["httpRetryEvents"],
                    tcp_retry_events=["tcpRetryEvents"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_retries": max_retries,
                "per_retry_timeout": per_retry_timeout,
            }
            if http_retry_events is not None:
                self._values["http_retry_events"] = http_retry_events
            if tcp_retry_events is not None:
                self._values["tcp_retry_events"] = tcp_retry_events

        @builtins.property
        def max_retries(self) -> jsii.Number:
            '''The maximum number of retry attempts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-maxretries
            '''
            result = self._values.get("max_retries")
            assert result is not None, "Required property 'max_retries' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def per_retry_timeout(
            self,
        ) -> typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]:
            '''The timeout for each retry attempt.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-perretrytimeout
            '''
            result = self._values.get("per_retry_timeout")
            assert result is not None, "Required property 'per_retry_timeout' is missing"
            return typing.cast(typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def http_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify at least one of the following values.

            - *server-error* – HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511
            - *gateway-error* – HTTP status codes 502, 503, and 504
            - *client-error* – HTTP status code 409
            - *stream-error* – Retry on refused stream

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-httpretryevents
            '''
            result = self._values.get("http_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tcp_retry_events(self) -> typing.Optional[typing.List[builtins.str]]:
            '''Specify a valid value.

            The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httpretrypolicy.html#cfn-appmesh-route-httpretrypolicy-tcpretryevents
            '''
            result = self._values.get("tcp_retry_events")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRetryPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class HttpRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_action_property = appmesh.CfnRoute.HttpRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteaction.html#cfn-appmesh-route-httprouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "invert": "invert", "match": "match"},
    )
    class HttpRouteHeaderProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            invert: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            match: typing.Optional[typing.Union["CfnRoute.HeaderMatchMethodProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the HTTP header in the request.

            :param name: A name for the HTTP header in the client request that will be matched on.
            :param invert: Specify ``True`` to match anything except the match criteria. The default value is ``False`` .
            :param match: The ``HeaderMatchMethod`` object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_header_property = appmesh.CfnRoute.HttpRouteHeaderProperty(
                    name="name",
                
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if invert is not None:
                self._values["invert"] = invert
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the HTTP header in the client request that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def invert(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Specify ``True`` to match anything except the match criteria.

            The default value is ``False`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-invert
            '''
            result = self._values.get("invert")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HeaderMatchMethodProperty", _IResolvable_a771d0ef]]:
            '''The ``HeaderMatchMethod`` object.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httprouteheader.html#cfn-appmesh-route-httprouteheader-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HeaderMatchMethodProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteMatchProperty",
        jsii_struct_bases=[],
        name_mapping={
            "headers": "headers",
            "method": "method",
            "path": "path",
            "prefix": "prefix",
            "query_parameters": "queryParameters",
            "scheme": "scheme",
        },
    )
    class HttpRouteMatchProperty:
        def __init__(
            self,
            *,
            headers: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnRoute.HttpRouteHeaderProperty", _IResolvable_a771d0ef]]]] = None,
            method: typing.Optional[builtins.str] = None,
            path: typing.Optional[typing.Union["CfnRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]] = None,
            prefix: typing.Optional[builtins.str] = None,
            query_parameters: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]] = None,
            scheme: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the requirements for a route to match HTTP requests for a virtual router.

            :param headers: The client request headers to match on.
            :param method: The client request method to match on. Specify only one.
            :param path: The client request path to match on.
            :param prefix: Specifies the path to match requests with. This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .
            :param query_parameters: The client request query parameters to match on.
            :param scheme: The client request scheme to match on. Specify only one. Applicable only for HTTP2 routes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_match_property = appmesh.CfnRoute.HttpRouteMatchProperty(
                    headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                        name="name",
                
                        # the properties below are optional
                        invert=False,
                        match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                            exact="exact",
                            prefix="prefix",
                            range=appmesh.CfnRoute.MatchRangeProperty(
                                end=123,
                                start=123
                            ),
                            regex="regex",
                            suffix="suffix"
                        )
                    )],
                    method="method",
                    path=appmesh.CfnRoute.HttpPathMatchProperty(
                        exact="exact",
                        regex="regex"
                    ),
                    prefix="prefix",
                    query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                        name="name",
                
                        # the properties below are optional
                        match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                            exact="exact"
                        )
                    )],
                    scheme="scheme"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if headers is not None:
                self._values["headers"] = headers
            if method is not None:
                self._values["method"] = method
            if path is not None:
                self._values["path"] = path
            if prefix is not None:
                self._values["prefix"] = prefix
            if query_parameters is not None:
                self._values["query_parameters"] = query_parameters
            if scheme is not None:
                self._values["scheme"] = scheme

        @builtins.property
        def headers(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.HttpRouteHeaderProperty", _IResolvable_a771d0ef]]]]:
            '''The client request headers to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-headers
            '''
            result = self._values.get("headers")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.HttpRouteHeaderProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def method(self) -> typing.Optional[builtins.str]:
            '''The client request method to match on.

            Specify only one.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-method
            '''
            result = self._values.get("method")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def path(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]]:
            '''The client request path to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpPathMatchProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def prefix(self) -> typing.Optional[builtins.str]:
            '''Specifies the path to match requests with.

            This parameter must always start with ``/`` , which by itself matches all requests to the virtual service name. You can also match for path-based routing of requests. For example, if your virtual service name is ``my-service.local`` and you want the route to match requests to ``my-service.local/metrics`` , your prefix should be ``/metrics`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-prefix
            '''
            result = self._values.get("prefix")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def query_parameters(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]]:
            '''The client request query parameters to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-queryparameters
            '''
            result = self._values.get("query_parameters")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.QueryParameterProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def scheme(self) -> typing.Optional[builtins.str]:
            '''The client request scheme to match on.

            Specify only one. Applicable only for HTTP2 routes.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproutematch.html#cfn-appmesh-route-httproutematch-scheme
            '''
            result = self._values.get("scheme")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteMatchProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "action": "action",
            "match": "match",
            "retry_policy": "retryPolicy",
            "timeout": "timeout",
        },
    )
    class HttpRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union["CfnRoute.HttpRouteActionProperty", _IResolvable_a771d0ef],
            match: typing.Union["CfnRoute.HttpRouteMatchProperty", _IResolvable_a771d0ef],
            retry_policy: typing.Optional[typing.Union["CfnRoute.HttpRetryPolicyProperty", _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union["CfnRoute.HttpTimeoutProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents an HTTP or HTTP/2 route type.

            :param action: An object that represents the action to take if a match is determined.
            :param match: An object that represents the criteria for determining a request match.
            :param retry_policy: An object that represents a retry policy.
            :param timeout: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_route_property = appmesh.CfnRoute.HttpRouteProperty(
                    action=appmesh.CfnRoute.HttpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
                    match=appmesh.CfnRoute.HttpRouteMatchProperty(
                        headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                            name="name",
                
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnRoute.MatchRangeProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        method="method",
                        path=appmesh.CfnRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                            name="name",
                
                            # the properties below are optional
                            match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )],
                        scheme="scheme"
                    ),
                
                    # the properties below are optional
                    retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                        max_retries=123,
                        per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                
                        # the properties below are optional
                        http_retry_events=["httpRetryEvents"],
                        tcp_retry_events=["tcpRetryEvents"]
                    ),
                    timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "action": action,
                "match": match,
            }
            if retry_policy is not None:
                self._values["retry_policy"] = retry_policy
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnRoute.HttpRouteActionProperty", _IResolvable_a771d0ef]:
            '''An object that represents the action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnRoute.HttpRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnRoute.HttpRouteMatchProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a request match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnRoute.HttpRouteMatchProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def retry_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpRetryPolicyProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a retry policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-retrypolicy
            '''
            result = self._values.get("retry_policy")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpRetryPolicyProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httproute.html#cfn-appmesh-route-httproute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.HttpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class HttpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_timeout_property = appmesh.CfnRoute.HttpTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html#cfn-appmesh-route-httptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-httptimeout.html#cfn-appmesh-route-httptimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.MatchRangeProperty",
        jsii_struct_bases=[],
        name_mapping={"end": "end", "start": "start"},
    )
    class MatchRangeProperty:
        def __init__(self, *, end: jsii.Number, start: jsii.Number) -> None:
            '''An object that represents the range of values to match on.

            The first character of the range is included in the range, though the last character is not. For example, if the range specified were 1-100, only values 1-99 would be matched.

            :param end: The end of the range.
            :param start: The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                match_range_property = appmesh.CfnRoute.MatchRangeProperty(
                    end=123,
                    start=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "end": end,
                "start": start,
            }

        @builtins.property
        def end(self) -> jsii.Number:
            '''The end of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html#cfn-appmesh-route-matchrange-end
            '''
            result = self._values.get("end")
            assert result is not None, "Required property 'end' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def start(self) -> jsii.Number:
            '''The start of the range.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-matchrange.html#cfn-appmesh-route-matchrange-start
            '''
            result = self._values.get("start")
            assert result is not None, "Required property 'start' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MatchRangeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.QueryParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name", "match": "match"},
    )
    class QueryParameterProperty:
        def __init__(
            self,
            *,
            name: builtins.str,
            match: typing.Optional[typing.Union["CfnRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the query parameter in the request.

            :param name: A name for the query parameter that will be matched on.
            :param match: The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                query_parameter_property = appmesh.CfnRoute.QueryParameterProperty(
                    name="name",
                
                    # the properties below are optional
                    match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                        exact="exact"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }
            if match is not None:
                self._values["match"] = match

        @builtins.property
        def name(self) -> builtins.str:
            '''A name for the query parameter that will be matched on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html#cfn-appmesh-route-queryparameter-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def match(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]]:
            '''The query parameter to match on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-queryparameter.html#cfn-appmesh-route-queryparameter-match
            '''
            result = self._values.get("match")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpQueryParameterMatchProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "QueryParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.RouteSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "grpc_route": "grpcRoute",
            "http2_route": "http2Route",
            "http_route": "httpRoute",
            "priority": "priority",
            "tcp_route": "tcpRoute",
        },
    )
    class RouteSpecProperty:
        def __init__(
            self,
            *,
            grpc_route: typing.Optional[typing.Union["CfnRoute.GrpcRouteProperty", _IResolvable_a771d0ef]] = None,
            http2_route: typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]] = None,
            http_route: typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]] = None,
            priority: typing.Optional[jsii.Number] = None,
            tcp_route: typing.Optional[typing.Union["CfnRoute.TcpRouteProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a route specification.

            Specify one route type.

            :param grpc_route: An object that represents the specification of a gRPC route.
            :param http2_route: An object that represents the specification of an HTTP/2 route.
            :param http_route: An object that represents the specification of an HTTP route.
            :param priority: The priority for the route. Routes are matched based on the specified value, where 0 is the highest priority.
            :param tcp_route: An object that represents the specification of a TCP route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                route_spec_property = appmesh.CfnRoute.RouteSpecProperty(
                    grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                        action=appmesh.CfnRoute.GrpcRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                        match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                            metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method_name="methodName",
                            service_name="serviceName"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            grpc_retry_events=["grpcRetryEvents"],
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http2_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
                
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
                
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
                
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    priority=123,
                    tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                        action=appmesh.CfnRoute.TcpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                
                        # the properties below are optional
                        timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if grpc_route is not None:
                self._values["grpc_route"] = grpc_route
            if http2_route is not None:
                self._values["http2_route"] = http2_route
            if http_route is not None:
                self._values["http_route"] = http_route
            if priority is not None:
                self._values["priority"] = priority
            if tcp_route is not None:
                self._values["tcp_route"] = tcp_route

        @builtins.property
        def grpc_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.GrpcRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of a gRPC route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-grpcroute
            '''
            result = self._values.get("grpc_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.GrpcRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP/2 route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-http2route
            '''
            result = self._values.get("http2_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of an HTTP route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-httproute
            '''
            result = self._values.get("http_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.HttpRouteProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''The priority for the route.

            Routes are matched based on the specified value, where 0 is the highest priority.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def tcp_route(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.TcpRouteProperty", _IResolvable_a771d0ef]]:
            '''An object that represents the specification of a TCP route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-routespec.html#cfn-appmesh-route-routespec-tcproute
            '''
            result = self._values.get("tcp_route")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.TcpRouteProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "RouteSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpRouteActionProperty",
        jsii_struct_bases=[],
        name_mapping={"weighted_targets": "weightedTargets"},
    )
    class TcpRouteActionProperty:
        def __init__(
            self,
            *,
            weighted_targets: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the action to take if a match is determined.

            :param weighted_targets: An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcprouteaction.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_route_action_property = appmesh.CfnRoute.TcpRouteActionProperty(
                    weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                        virtual_node="virtualNode",
                        weight=123
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "weighted_targets": weighted_targets,
            }

        @builtins.property
        def weighted_targets(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]]:
            '''An object that represents the targets that traffic is routed to when a request matches the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcprouteaction.html#cfn-appmesh-route-tcprouteaction-weightedtargets
            '''
            result = self._values.get("weighted_targets")
            assert result is not None, "Required property 'weighted_targets' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnRoute.WeightedTargetProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteActionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpRouteProperty",
        jsii_struct_bases=[],
        name_mapping={"action": "action", "timeout": "timeout"},
    )
    class TcpRouteProperty:
        def __init__(
            self,
            *,
            action: typing.Union["CfnRoute.TcpRouteActionProperty", _IResolvable_a771d0ef],
            timeout: typing.Optional[typing.Union["CfnRoute.TcpTimeoutProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a TCP route type.

            :param action: The action to take if a match is determined.
            :param timeout: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_route_property = appmesh.CfnRoute.TcpRouteProperty(
                    action=appmesh.CfnRoute.TcpRouteActionProperty(
                        weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                            virtual_node="virtualNode",
                            weight=123
                        )]
                    ),
                
                    # the properties below are optional
                    timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                        idle=appmesh.CfnRoute.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "action": action,
            }
            if timeout is not None:
                self._values["timeout"] = timeout

        @builtins.property
        def action(
            self,
        ) -> typing.Union["CfnRoute.TcpRouteActionProperty", _IResolvable_a771d0ef]:
            '''The action to take if a match is determined.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-action
            '''
            result = self._values.get("action")
            assert result is not None, "Required property 'action' is missing"
            return typing.cast(typing.Union["CfnRoute.TcpRouteActionProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.TcpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcproute.html#cfn-appmesh-route-tcproute-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.TcpTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.TcpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle"},
    )
    class TcpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_timeout_property = appmesh.CfnRoute.TcpTimeoutProperty(
                    idle=appmesh.CfnRoute.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-tcptimeout.html#cfn-appmesh-route-tcptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnRoute.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnRoute.WeightedTargetProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_node": "virtualNode", "weight": "weight"},
    )
    class WeightedTargetProperty:
        def __init__(self, *, virtual_node: builtins.str, weight: jsii.Number) -> None:
            '''An object that represents a target and its relative weight.

            Traffic is distributed across targets according to their relative weight. For example, a weighted target with a relative weight of 50 receives five times as much traffic as one with a relative weight of 10. The total weight for all targets combined must be less than or equal to 100.

            :param virtual_node: The virtual node to associate with the weighted target.
            :param weight: The relative weight of the weighted target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                weighted_target_property = appmesh.CfnRoute.WeightedTargetProperty(
                    virtual_node="virtualNode",
                    weight=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "virtual_node": virtual_node,
                "weight": weight,
            }

        @builtins.property
        def virtual_node(self) -> builtins.str:
            '''The virtual node to associate with the weighted target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-virtualnode
            '''
            result = self._values.get("virtual_node")
            assert result is not None, "Required property 'virtual_node' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def weight(self) -> jsii.Number:
            '''The relative weight of the weighted target.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-route-weightedtarget.html#cfn-appmesh-route-weightedtarget-weight
            '''
            result = self._values.get("weight")
            assert result is not None, "Required property 'weight' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "WeightedTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_router_name": "virtualRouterName",
        "mesh_owner": "meshOwner",
        "route_name": "routeName",
        "tags": "tags",
    },
)
class CfnRouteProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[CfnRoute.RouteSpecProperty, _IResolvable_a771d0ef],
        virtual_router_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        route_name: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``CfnRoute``.

        :param mesh_name: The name of the service mesh to create the route in.
        :param spec: The route specification to apply.
        :param virtual_router_name: The name of the virtual router in which to create the route. If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param route_name: The name to use for the route.
        :param tags: Optional metadata that you can apply to the route to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_route_props = appmesh.CfnRouteProps(
                mesh_name="meshName",
                spec=appmesh.CfnRoute.RouteSpecProperty(
                    grpc_route=appmesh.CfnRoute.GrpcRouteProperty(
                        action=appmesh.CfnRoute.GrpcRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                        match=appmesh.CfnRoute.GrpcRouteMatchProperty(
                            metadata=[appmesh.CfnRoute.GrpcRouteMetadataProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.GrpcRouteMetadataMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method_name="methodName",
                            service_name="serviceName"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.GrpcRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            grpc_retry_events=["grpcRetryEvents"],
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.GrpcTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http2_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    http_route=appmesh.CfnRoute.HttpRouteProperty(
                        action=appmesh.CfnRoute.HttpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
                        match=appmesh.CfnRoute.HttpRouteMatchProperty(
                            headers=[appmesh.CfnRoute.HttpRouteHeaderProperty(
                                name="name",
            
                                # the properties below are optional
                                invert=False,
                                match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                                    exact="exact",
                                    prefix="prefix",
                                    range=appmesh.CfnRoute.MatchRangeProperty(
                                        end=123,
                                        start=123
                                    ),
                                    regex="regex",
                                    suffix="suffix"
                                )
                            )],
                            method="method",
                            path=appmesh.CfnRoute.HttpPathMatchProperty(
                                exact="exact",
                                regex="regex"
                            ),
                            prefix="prefix",
                            query_parameters=[appmesh.CfnRoute.QueryParameterProperty(
                                name="name",
            
                                # the properties below are optional
                                match=appmesh.CfnRoute.HttpQueryParameterMatchProperty(
                                    exact="exact"
                                )
                            )],
                            scheme="scheme"
                        ),
            
                        # the properties below are optional
                        retry_policy=appmesh.CfnRoute.HttpRetryPolicyProperty(
                            max_retries=123,
                            per_retry_timeout=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
            
                            # the properties below are optional
                            http_retry_events=["httpRetryEvents"],
                            tcp_retry_events=["tcpRetryEvents"]
                        ),
                        timeout=appmesh.CfnRoute.HttpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    priority=123,
                    tcp_route=appmesh.CfnRoute.TcpRouteProperty(
                        action=appmesh.CfnRoute.TcpRouteActionProperty(
                            weighted_targets=[appmesh.CfnRoute.WeightedTargetProperty(
                                virtual_node="virtualNode",
                                weight=123
                            )]
                        ),
            
                        # the properties below are optional
                        timeout=appmesh.CfnRoute.TcpTimeoutProperty(
                            idle=appmesh.CfnRoute.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    )
                ),
                virtual_router_name="virtualRouterName",
            
                # the properties below are optional
                mesh_owner="meshOwner",
                route_name="routeName",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_router_name": virtual_router_name,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if route_name is not None:
            self._values["route_name"] = route_name
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the route in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(self) -> typing.Union[CfnRoute.RouteSpecProperty, _IResolvable_a771d0ef]:
        '''The route specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnRoute.RouteSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router in which to create the route.

        If the virtual router is in a shared mesh, then you must be the owner of the virtual router resource.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-virtualroutername
        '''
        result = self._values.get("virtual_router_name")
        assert result is not None, "Required property 'virtual_router_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def route_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the route.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-routename
        '''
        result = self._values.get("route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the route to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-route.html#cfn-appmesh-route-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualGateway(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualGateway",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualGateway``.

    Creates a virtual gateway.

    A virtual gateway allows resources outside your mesh to communicate to resources that are inside your mesh. The virtual gateway represents an Envoy proxy running in an Amazon ECS task, in a Kubernetes service, or on an Amazon EC2 instance. Unlike a virtual node, which represents an Envoy running with an application, a virtual gateway represents Envoy deployed by itself.

    For more information about virtual gateways, see `Virtual gateways <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_gateways.html>`_ .

    :cloudformationResource: AWS::AppMesh::VirtualGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_gateway = appmesh.CfnVirtualGateway(self, "MyCfnVirtualGateway",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                    port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
        
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                            max_connections=123,
        
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                            max_requests=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
        
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
        
                        # the properties below are optional
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
        
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )],
        
                # the properties below are optional
                backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
        
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
        
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                ),
                logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                    access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                            path="path"
                        )
                    )
                )
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_gateway_name="virtualGatewayName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        virtual_gateway_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh that the virtual gateway resides in.
        :param spec: The specifications of the virtual gateway.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual gateway to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_gateway_name: The name of the virtual gateway.
        '''
        props = CfnVirtualGatewayProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_gateway_name=virtual_gateway_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual gateway.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual gateway.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVirtualGatewayName")
    def attr_virtual_gateway_name(self) -> builtins.str:
        '''The name of the virtual gateway.

        :cloudformationAttribute: VirtualGatewayName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualGatewayName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual gateway to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef]:
        '''The specifications of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-spec
        '''
        return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualGateway.VirtualGatewaySpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "spec", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshOwner", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualGatewayName")
    def virtual_gateway_name(self) -> typing.Optional[builtins.str]:
        '''The name of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-virtualgatewayname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualGatewayName"))

    @virtual_gateway_name.setter
    def virtual_gateway_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "virtualGatewayName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class SubjectAlternativeNameMatchersProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents the methods by which a subject alternative name on a peer Transport Layer Security (TLS) certificate can be matched.

            :param exact: The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenamematchers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_name_matchers_property = appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                    exact=["exact"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenamematchers.html#cfn-appmesh-virtualgateway-subjectalternativenamematchers-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNameMatchersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty",
        jsii_struct_bases=[],
        name_mapping={"match": "match"},
    )
    class SubjectAlternativeNamesProperty:
        def __init__(
            self,
            *,
            match: typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents the subject alternative names secured by the certificate.

            :param match: An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenames.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_names_property = appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                        exact=["exact"]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "match": match,
            }

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-subjectalternativenames.html#cfn-appmesh-virtualgateway-subjectalternativenames-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNamesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file"},
    )
    class VirtualGatewayAccessLogProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''The access log configuration for a virtual gateway.

            :param file: The file object to send virtual gateway access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_access_log_property = appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                        path="path"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", _IResolvable_a771d0ef]]:
            '''The file object to send virtual gateway access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayaccesslog-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayFileAccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"client_policy": "clientPolicy"},
    )
    class VirtualGatewayBackendDefaultsProperty:
        def __init__(
            self,
            *,
            client_policy: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the default properties for a backend.

            :param client_policy: A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaybackenddefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_backend_defaults_property = appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaybackenddefaults.html#cfn-appmesh-virtualgateway-virtualgatewaybackenddefaults-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayBackendDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"tls": "tls"},
    )
    class VirtualGatewayClientPolicyProperty:
        def __init__(
            self,
            *,
            tls: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a client policy.

            :param tls: A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_client_policy_property = appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                    certificate_authority_arns=["certificateAuthorityArns"]
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        ),
                
                        # the properties below are optional
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        enforce=False,
                        ports=[123]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicy-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "validation": "validation",
            "certificate": "certificate",
            "enforce": "enforce",
            "ports": "ports",
        },
    )
    class VirtualGatewayClientPolicyTlsProperty:
        def __init__(
            self,
            *,
            validation: typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", _IResolvable_a771d0ef],
            certificate: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", _IResolvable_a771d0ef]] = None,
            enforce: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            ports: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[jsii.Number]]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) client policy.

            :param validation: A reference to an object that represents a Transport Layer Security (TLS) validation context.
            :param certificate: A reference to an object that represents a virtual gateway's client's Transport Layer Security (TLS) certificate.
            :param enforce: Whether the policy is enforced. The default is ``True`` , if a value isn't specified.
            :param ports: One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_client_policy_tls_property = appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                    validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                certificate_authority_arns=["certificateAuthorityArns"]
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    ),
                
                    # the properties below are optional
                    certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    enforce=False,
                    ports=[123]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "validation": validation,
            }
            if certificate is not None:
                self._values["certificate"] = certificate
            if enforce is not None:
                self._values["enforce"] = enforce
            if ports is not None:
                self._values["ports"] = ports

        @builtins.property
        def validation(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", _IResolvable_a771d0ef]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-validation
            '''
            result = self._values.get("validation")
            assert result is not None, "Required property 'validation' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def certificate(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's client's Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-certificate
            '''
            result = self._values.get("certificate")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def enforce(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the policy is enforced.

            The default is ``True`` , if a value isn't specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-enforce
            '''
            result = self._values.get("enforce")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def ports(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]]:
            '''One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclientpolicytls.html#cfn-appmesh-virtualgateway-virtualgatewayclientpolicytls-ports
            '''
            result = self._values.get("ports")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientPolicyTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class VirtualGatewayClientTlsCertificateProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the virtual gateway's client's Transport Layer Security (TLS) certificate.

            :param file: An object that represents a local file certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .
            :param sds: A reference to an object that represents a virtual gateway's client's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_client_tls_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewayclienttlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's client's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayclienttlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewayclienttlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayClientTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2"},
    )
    class VirtualGatewayConnectionPoolProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            http: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            http2: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the type of virtual gateway connection pool.

            Only one protocol is used at a time and should be the same protocol as the one chosen under port mapping.

            If not present the default value for ``maxPendingRequests`` is ``2147483647`` .

            :param grpc: An object that represents a type of connection pool.
            :param http: An object that represents a type of connection pool.
            :param http2: An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                    grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                        max_requests=123
                    ),
                    http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                        max_connections=123,
                
                        # the properties below are optional
                        max_pending_requests=123
                    ),
                    http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                        max_requests=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayconnectionpool-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path"},
    )
    class VirtualGatewayFileAccessLogProperty:
        def __init__(self, *, path: builtins.str) -> None:
            '''An object that represents an access log file.

            :param path: The file path to write access logs to. You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_file_access_log_property = appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                    path="path"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "path": path,
            }

        @builtins.property
        def path(self) -> builtins.str:
            '''The file path to write access logs to.

            You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayfileaccesslog.html#cfn-appmesh-virtualgateway-virtualgatewayfileaccesslog-path
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayFileAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualGatewayGrpcConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_grpc_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                    max_requests=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewaygrpcconnectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayGrpcConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={
            "healthy_threshold": "healthyThreshold",
            "interval_millis": "intervalMillis",
            "protocol": "protocol",
            "timeout_millis": "timeoutMillis",
            "unhealthy_threshold": "unhealthyThreshold",
            "path": "path",
            "port": "port",
        },
    )
    class VirtualGatewayHealthCheckPolicyProperty:
        def __init__(
            self,
            *,
            healthy_threshold: jsii.Number,
            interval_millis: jsii.Number,
            protocol: builtins.str,
            timeout_millis: jsii.Number,
            unhealthy_threshold: jsii.Number,
            path: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents the health check policy for a virtual gateway's listener.

            :param healthy_threshold: The number of consecutive successful health checks that must occur before declaring the listener healthy.
            :param interval_millis: The time period in milliseconds between each health check execution.
            :param protocol: The protocol for the health check request. If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .
            :param timeout_millis: The amount of time to wait when receiving a response from the health check, in milliseconds.
            :param unhealthy_threshold: The number of consecutive failed health checks that must occur before declaring a virtual gateway unhealthy.
            :param path: The destination path for the health check request. This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.
            :param port: The destination port for the health check request. This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_health_check_policy_property = appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
                
                    # the properties below are optional
                    path="path",
                    port=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "healthy_threshold": healthy_threshold,
                "interval_millis": interval_millis,
                "protocol": protocol,
                "timeout_millis": timeout_millis,
                "unhealthy_threshold": unhealthy_threshold,
            }
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def healthy_threshold(self) -> jsii.Number:
            '''The number of consecutive successful health checks that must occur before declaring the listener healthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-healthythreshold
            '''
            result = self._values.get("healthy_threshold")
            assert result is not None, "Required property 'healthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def interval_millis(self) -> jsii.Number:
            '''The time period in milliseconds between each health check execution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-intervalmillis
            '''
            result = self._values.get("interval_millis")
            assert result is not None, "Required property 'interval_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol for the health check request.

            If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def timeout_millis(self) -> jsii.Number:
            '''The amount of time to wait when receiving a response from the health check, in milliseconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-timeoutmillis
            '''
            result = self._values.get("timeout_millis")
            assert result is not None, "Required property 'timeout_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def unhealthy_threshold(self) -> jsii.Number:
            '''The number of consecutive failed health checks that must occur before declaring a virtual gateway unhealthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-unhealthythreshold
            '''
            result = self._values.get("unhealthy_threshold")
            assert result is not None, "Required property 'unhealthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The destination path for the health check request.

            This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The destination port for the health check request.

            This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy.html#cfn-appmesh-virtualgateway-virtualgatewayhealthcheckpolicy-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHealthCheckPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualGatewayHttp2ConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttp2connectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_http2_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                    max_requests=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttp2connectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttp2connectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHttp2ConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_connections": "maxConnections",
            "max_pending_requests": "maxPendingRequests",
        },
    )
    class VirtualGatewayHttpConnectionPoolProperty:
        def __init__(
            self,
            *,
            max_connections: jsii.Number,
            max_pending_requests: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.
            :param max_pending_requests: Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_http_connection_pool_property = appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                    max_connections=123,
                
                    # the properties below are optional
                    max_pending_requests=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_connections": max_connections,
            }
            if max_pending_requests is not None:
                self._values["max_pending_requests"] = max_pending_requests

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_pending_requests(self) -> typing.Optional[jsii.Number]:
            '''Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayhttpconnectionpool.html#cfn-appmesh-virtualgateway-virtualgatewayhttpconnectionpool-maxpendingrequests
            '''
            result = self._values.get("max_pending_requests")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayHttpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_mapping": "portMapping",
            "connection_pool": "connectionPool",
            "health_check": "healthCheck",
            "tls": "tls",
        },
    )
    class VirtualGatewayListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", _IResolvable_a771d0ef],
            connection_pool: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            health_check: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", _IResolvable_a771d0ef]] = None,
            tls: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener for a virtual gateway.

            :param port_mapping: The port mapping information for the listener.
            :param connection_pool: The connection pool information for the listener.
            :param health_check: The health check information for the listener.
            :param tls: A reference to an object that represents the Transport Layer Security (TLS) properties for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                    port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
                
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                            max_connections=123,
                
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                            max_requests=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
                
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                        certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
                
                        # the properties below are optional
                        validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "port_mapping": port_mapping,
            }
            if connection_pool is not None:
                self._values["connection_pool"] = connection_pool
            if health_check is not None:
                self._values["health_check"] = health_check
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", _IResolvable_a771d0ef]:
            '''The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayPortMappingProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def connection_pool(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''The connection pool information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-connectionpool
            '''
            result = self._values.get("connection_pool")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", _IResolvable_a771d0ef]]:
            '''The health check information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the Transport Layer Security (TLS) properties for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistener.html#cfn-appmesh-virtualgateway-virtualgatewaylistener-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class VirtualGatewayListenerTlsAcmCertificateProperty:
        def __init__(self, *, certificate_arn: builtins.str) -> None:
            '''An object that represents an AWS Certificate Manager certificate.

            :param certificate_arn: The Amazon Resource Name (ARN) for the certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_acm_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                    certificate_arn="certificateArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_arn": certificate_arn,
            }

        @builtins.property
        def certificate_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) for the certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsacmcertificate-certificatearn
            '''
            result = self._values.get("certificate_arn")
            assert result is not None, "Required property 'certificate_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsAcmCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class VirtualGatewayListenerTlsCertificateProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) certificate.

            :param acm: A reference to an object that represents an AWS Certificate Manager certificate.
            :param file: A reference to an object that represents a local file certificate.
            :param sds: A reference to an object that represents a virtual gateway's listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                    acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                        certificate_arn="certificateArn"
                    ),
                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a local file certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_chain": "certificateChain",
            "private_key": "privateKey",
        },
    )
    class VirtualGatewayListenerTlsFileCertificateProperty:
        def __init__(
            self,
            *,
            certificate_chain: builtins.str,
            private_key: builtins.str,
        ) -> None:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :param certificate_chain: The certificate chain for the certificate.
            :param private_key: The private key for a certificate stored on the file system of the mesh endpoint that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_file_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                    certificate_chain="certificateChain",
                    private_key="privateKey"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_chain": certificate_chain,
                "private_key": private_key,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate chain for the certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def private_key(self) -> builtins.str:
            '''The private key for a certificate stored on the file system of the mesh endpoint that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsfilecertificate-privatekey
            '''
            result = self._values.get("private_key")
            assert result is not None, "Required property 'private_key' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsFileCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate": "certificate",
            "mode": "mode",
            "validation": "validation",
        },
    )
    class VirtualGatewayListenerTlsProperty:
        def __init__(
            self,
            *,
            certificate: typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", _IResolvable_a771d0ef],
            mode: builtins.str,
            validation: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the Transport Layer Security (TLS) properties for a listener.

            :param certificate: An object that represents a Transport Layer Security (TLS) certificate.
            :param mode: Specify one of the following modes. - ** STRICT – Listener only accepts connections with TLS enabled. - ** PERMISSIVE – Listener accepts connections with or without TLS enabled. - ** DISABLED – Listener only accepts connections without TLS.
            :param validation: A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                    certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                        acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                            certificate_arn="certificateArn"
                        ),
                        file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    mode="mode",
                
                    # the properties below are optional
                    validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate": certificate,
                "mode": mode,
            }
            if validation is not None:
                self._values["validation"] = validation

        @builtins.property
        def certificate(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", _IResolvable_a771d0ef]:
            '''An object that represents a Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-certificate
            '''
            result = self._values.get("certificate")
            assert result is not None, "Required property 'certificate' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def mode(self) -> builtins.str:
            '''Specify one of the following modes.

            - ** STRICT – Listener only accepts connections with TLS enabled.
            - ** PERMISSIVE – Listener accepts connections with or without TLS enabled.
            - ** DISABLED – Listener only accepts connections without TLS.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-mode
            '''
            result = self._values.get("mode")
            assert result is not None, "Required property 'mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def validation(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertls.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertls-validation
            '''
            result = self._values.get("validation")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class VirtualGatewayListenerTlsSdsCertificateProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents the virtual gateway's listener's Secret Discovery Service certificate.The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_sds_certificate_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                    secret_name="secretName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlssdscertificate-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsSdsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class VirtualGatewayListenerTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a virtual gateway listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_validation_context_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a virtual gateway listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class VirtualGatewayListenerTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) validation context trust.

            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_listener_tls_validation_context_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaylistenertlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayListenerTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"access_log": "accessLog"},
    )
    class VirtualGatewayLoggingProperty:
        def __init__(
            self,
            *,
            access_log: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents logging information.

            :param access_log: The access log configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylogging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_logging_property = appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                    access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                        file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                            path="path"
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if access_log is not None:
                self._values["access_log"] = access_log

        @builtins.property
        def access_log(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", _IResolvable_a771d0ef]]:
            '''The access log configuration.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaylogging.html#cfn-appmesh-virtualgateway-virtualgatewaylogging-accesslog
            '''
            result = self._values.get("access_log")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayAccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayLoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class VirtualGatewayPortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object that represents a port mapping.

            :param port: The port used for the port mapping. Specify one protocol.
            :param protocol: The protocol used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_port_mapping_property = appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            Specify one protocol.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html#cfn-appmesh-virtualgateway-virtualgatewayportmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayportmapping.html#cfn-appmesh-virtualgateway-virtualgatewayportmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayPortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "listeners": "listeners",
            "backend_defaults": "backendDefaults",
            "logging": "logging",
        },
    )
    class VirtualGatewaySpecProperty:
        def __init__(
            self,
            *,
            listeners: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", _IResolvable_a771d0ef]]],
            backend_defaults: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", _IResolvable_a771d0ef]] = None,
            logging: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a service mesh resource.

            :param listeners: The listeners that the mesh endpoint is expected to receive inbound traffic from. You can specify one listener.
            :param backend_defaults: A reference to an object that represents the defaults for backends.
            :param logging: An object that represents logging information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_spec_property = appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                    listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                        port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
                
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                                max_connections=123,
                
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                                max_requests=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
                
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
                
                            # the properties below are optional
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                
                    # the properties below are optional
                    backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                            tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                        access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                                path="path"
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "listeners": listeners,
            }
            if backend_defaults is not None:
                self._values["backend_defaults"] = backend_defaults
            if logging is not None:
                self._values["logging"] = logging

        @builtins.property
        def listeners(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", _IResolvable_a771d0ef]]]:
            '''The listeners that the mesh endpoint is expected to receive inbound traffic from.

            You can specify one listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-listeners
            '''
            result = self._values.get("listeners")
            assert result is not None, "Required property 'listeners' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualGateway.VirtualGatewayListenerProperty", _IResolvable_a771d0ef]]], result)

        @builtins.property
        def backend_defaults(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the defaults for backends.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-backenddefaults
            '''
            result = self._values.get("backend_defaults")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", _IResolvable_a771d0ef]]:
            '''An object that represents logging information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewayspec.html#cfn-appmesh-virtualgateway-virtualgatewayspec-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayLoggingProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewaySpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_authority_arns": "certificateAuthorityArns"},
    )
    class VirtualGatewayTlsValidationContextAcmTrustProperty:
        def __init__(
            self,
            *,
            certificate_authority_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :param certificate_authority_arns: One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_acm_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                    certificate_authority_arns=["certificateAuthorityArns"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_authority_arns": certificate_authority_arns,
            }

        @builtins.property
        def certificate_authority_arns(self) -> typing.List[builtins.str]:
            '''One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextacmtrust-certificateauthorityarns
            '''
            result = self._values.get("certificate_authority_arns")
            assert result is not None, "Required property 'certificate_authority_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextAcmTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_chain": "certificateChain"},
    )
    class VirtualGatewayTlsValidationContextFileTrustProperty:
        def __init__(self, *, certificate_chain: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :param certificate_chain: The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_file_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                    certificate_chain="certificateChain"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_chain": certificate_chain,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextfiletrust-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextFileTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class VirtualGatewayTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                            certificate_authority_arns=["certificateAuthorityArns"]
                        ),
                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a virtual gateway's listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class VirtualGatewayTlsValidationContextSdsTrustProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents a virtual gateway's listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret for a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_sds_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                    secret_name="secretName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret for a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontextsdstrust-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextSdsTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class VirtualGatewayTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust.

            :param acm: A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.
            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_gateway_tls_validation_context_trust_property = appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                    acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                        certificate_authority_arns=["certificateAuthorityArns"]
                    ),
                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a virtual gateway's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust.html#cfn-appmesh-virtualgateway-virtualgatewaytlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualGatewayTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_gateway_name": "virtualGatewayName",
    },
)
class CfnVirtualGatewayProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        virtual_gateway_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualGateway``.

        :param mesh_name: The name of the service mesh that the virtual gateway resides in.
        :param spec: The specifications of the virtual gateway.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual gateway to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_gateway_name: The name of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_gateway_props = appmesh.CfnVirtualGatewayProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualGateway.VirtualGatewaySpecProperty(
                    listeners=[appmesh.CfnVirtualGateway.VirtualGatewayListenerProperty(
                        port_mapping=appmesh.CfnVirtualGateway.VirtualGatewayPortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
            
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualGateway.VirtualGatewayConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualGateway.VirtualGatewayGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualGateway.VirtualGatewayHttpConnectionPoolProperty(
                                max_connections=123,
            
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualGateway.VirtualGatewayHttp2ConnectionPoolProperty(
                                max_requests=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
            
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        tls=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsProperty(
                            certificate=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
            
                            # the properties below are optional
                            validation=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
            
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
            
                    # the properties below are optional
                    backend_defaults=appmesh.CfnVirtualGateway.VirtualGatewayBackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyProperty(
                            tls=appmesh.CfnVirtualGateway.VirtualGatewayClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualGateway.VirtualGatewayTlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualGateway.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualGateway.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualGateway.VirtualGatewayClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualGateway.VirtualGatewayListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    logging=appmesh.CfnVirtualGateway.VirtualGatewayLoggingProperty(
                        access_log=appmesh.CfnVirtualGateway.VirtualGatewayAccessLogProperty(
                            file=appmesh.CfnVirtualGateway.VirtualGatewayFileAccessLogProperty(
                                path="path"
                            )
                        )
                    )
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_gateway_name="virtualGatewayName"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_gateway_name is not None:
            self._values["virtual_gateway_name"] = virtual_gateway_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual gateway resides in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, _IResolvable_a771d0ef]:
        '''The specifications of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualGateway.VirtualGatewaySpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual gateway to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def virtual_gateway_name(self) -> typing.Optional[builtins.str]:
        '''The name of the virtual gateway.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualgateway.html#cfn-appmesh-virtualgateway-virtualgatewayname
        '''
        result = self._values.get("virtual_gateway_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualNode(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualNode",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualNode``.

    Creates a virtual node within a service mesh.

    A virtual node acts as a logical pointer to a particular task group, such as an Amazon ECS service or a Kubernetes deployment. When you create a virtual node, you can specify the service discovery information for your task group, and whether the proxy running in a task group will communicate with other proxies using Transport Layer Security (TLS).

    You define a ``listener`` for any inbound traffic that your virtual node expects. Any virtual service that your virtual node expects to communicate to is specified as a ``backend`` .

    The response metadata for your new virtual node contains the ``arn`` that is associated with the virtual node. Set this value to the full ARN; for example, ``arn:aws:appmesh:us-west-2:123456789012:myMesh/default/virtualNode/myApp`` ) as the ``APPMESH_RESOURCE_ARN`` environment variable for your task group's Envoy proxy container in your task definition or pod spec. This is then mapped to the ``node.id`` and ``node.cluster`` Envoy parameters.
    .. epigraph::

       By default, App Mesh uses the name of the resource you specified in ``APPMESH_RESOURCE_ARN`` when Envoy is referring to itself in metrics and traces. You can override this behavior by setting the ``APPMESH_RESOURCE_CLUSTER`` environment variable with your own name.

    For more information about virtual nodes, see `Virtual nodes <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html>`_ . You must be using ``1.15.0`` or later of the Envoy image when setting these variables. For more information about App Mesh Envoy variables, see `Envoy image <https://docs.aws.amazon.com/app-mesh/latest/userguide/envoy.html>`_ in the AWS App Mesh User Guide.

    :cloudformationResource: AWS::AppMesh::VirtualNode
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_node = appmesh.CfnVirtualNode(self, "MyCfnVirtualNode",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
        
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
        
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                ),
                backends=[appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
        
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
        
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
        
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )],
                listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                    port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
        
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                            max_connections=123,
        
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                            max_requests=123
                        ),
                        tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                            max_connections=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
        
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                        base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        interval=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        max_ejection_percent=123,
                        max_server_errors=123
                    ),
                    timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                        grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                        certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
        
                        # the properties below are optional
                        validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
        
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )],
                logging=appmesh.CfnVirtualNode.LoggingProperty(
                    access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                        file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                            path="path"
                        )
                    )
                ),
                service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                    aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                        namespace_name="namespaceName",
                        service_name="serviceName",
        
                        # the properties below are optional
                        attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                            key="key",
                            value="value"
                        )]
                    ),
                    dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                        hostname="hostname",
        
                        # the properties below are optional
                        response_type="responseType"
                    )
                )
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_node_name="virtualNodeName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        virtual_node_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualNode``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the virtual node in.
        :param spec: The virtual node specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual node to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_node_name: The name to use for the virtual node.
        '''
        props = CfnVirtualNodeProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_node_name=virtual_node_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual node.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual node resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual node.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVirtualNodeName")
    def attr_virtual_node_name(self) -> builtins.str:
        '''The name of the virtual node.

        :cloudformationAttribute: VirtualNodeName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualNodeName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual node to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual node in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef]:
        '''The virtual node specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-spec
        '''
        return typing.cast(typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualNode.VirtualNodeSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "spec", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshOwner", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualNodeName")
    def virtual_node_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual node.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-virtualnodename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualNodeName"))

    @virtual_node_name.setter
    def virtual_node_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "virtualNodeName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.AccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file"},
    )
    class AccessLogProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union["CfnVirtualNode.FileAccessLogProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the access logging information for a virtual node.

            :param file: The file object to send virtual node access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-accesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                access_log_property = appmesh.CfnVirtualNode.AccessLogProperty(
                    file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                        path="path"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.FileAccessLogProperty", _IResolvable_a771d0ef]]:
            '''The file object to send virtual node access logs to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-accesslog.html#cfn-appmesh-virtualnode-accesslog-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.FileAccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class AwsCloudMapInstanceAttributeProperty:
        def __init__(self, *, key: builtins.str, value: builtins.str) -> None:
            '''An object that represents the AWS Cloud Map attribute information for your virtual node.

            .. epigraph::

               AWS Cloud Map is not available in the eu-south-1 Region.

            :param key: The name of an AWS Cloud Map service instance attribute key. Any AWS Cloud Map service instance that contains the specified key and value is returned.
            :param value: The value of an AWS Cloud Map service instance attribute key. Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                aws_cloud_map_instance_attribute_property = appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                    key="key",
                    value="value"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''The name of an AWS Cloud Map service instance attribute key.

            Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html#cfn-appmesh-virtualnode-awscloudmapinstanceattribute-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> builtins.str:
            '''The value of an AWS Cloud Map service instance attribute key.

            Any AWS Cloud Map service instance that contains the specified key and value is returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapinstanceattribute.html#cfn-appmesh-virtualnode-awscloudmapinstanceattribute-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsCloudMapInstanceAttributeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={
            "namespace_name": "namespaceName",
            "service_name": "serviceName",
            "attributes": "attributes",
        },
    )
    class AwsCloudMapServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            namespace_name: builtins.str,
            service_name: builtins.str,
            attributes: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", _IResolvable_a771d0ef]]]] = None,
        ) -> None:
            '''An object that represents the AWS Cloud Map service discovery information for your virtual node.

            .. epigraph::

               AWS Cloud Map is not available in the eu-south-1 Region.

            :param namespace_name: The name of the AWS Cloud Map namespace to use.
            :param service_name: The name of the AWS Cloud Map service to use.
            :param attributes: A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance. Only instances that match all of the specified key/value pairs will be returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                aws_cloud_map_service_discovery_property = appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                    namespace_name="namespaceName",
                    service_name="serviceName",
                
                    # the properties below are optional
                    attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                        key="key",
                        value="value"
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "namespace_name": namespace_name,
                "service_name": service_name,
            }
            if attributes is not None:
                self._values["attributes"] = attributes

        @builtins.property
        def namespace_name(self) -> builtins.str:
            '''The name of the AWS Cloud Map namespace to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-namespacename
            '''
            result = self._values.get("namespace_name")
            assert result is not None, "Required property 'namespace_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def service_name(self) -> builtins.str:
            '''The name of the AWS Cloud Map service to use.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-servicename
            '''
            result = self._values.get("service_name")
            assert result is not None, "Required property 'service_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def attributes(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", _IResolvable_a771d0ef]]]]:
            '''A string map that contains attributes with values that you can use to filter instances by any custom attribute that you specified when you registered the instance.

            Only instances that match all of the specified key/value pairs will be returned.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-awscloudmapservicediscovery.html#cfn-appmesh-virtualnode-awscloudmapservicediscovery-attributes
            '''
            result = self._values.get("attributes")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.AwsCloudMapInstanceAttributeProperty", _IResolvable_a771d0ef]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AwsCloudMapServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.BackendDefaultsProperty",
        jsii_struct_bases=[],
        name_mapping={"client_policy": "clientPolicy"},
    )
    class BackendDefaultsProperty:
        def __init__(
            self,
            *,
            client_policy: typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the default properties for a backend.

            :param client_policy: A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backenddefaults.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                backend_defaults_property = appmesh.CfnVirtualNode.BackendDefaultsProperty(
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backenddefaults.html#cfn-appmesh-virtualnode-backenddefaults-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BackendDefaultsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.BackendProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_service": "virtualService"},
    )
    class BackendProperty:
        def __init__(
            self,
            *,
            virtual_service: typing.Optional[typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the backends that a virtual node is expected to send outbound traffic to.

            :param virtual_service: Specifies a virtual service to use as a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backend.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                backend_property = appmesh.CfnVirtualNode.BackendProperty(
                    virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                        virtual_service_name="virtualServiceName",
                
                        # the properties below are optional
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if virtual_service is not None:
                self._values["virtual_service"] = virtual_service

        @builtins.property
        def virtual_service(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", _IResolvable_a771d0ef]]:
            '''Specifies a virtual service to use as a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-backend.html#cfn-appmesh-virtualnode-backend-virtualservice
            '''
            result = self._values.get("virtual_service")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualServiceBackendProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BackendProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ClientPolicyProperty",
        jsii_struct_bases=[],
        name_mapping={"tls": "tls"},
    )
    class ClientPolicyProperty:
        def __init__(
            self,
            *,
            tls: typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a client policy.

            :param tls: A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicy.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                client_policy_property = appmesh.CfnVirtualNode.ClientPolicyProperty(
                    tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                        validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                    certificate_authority_arns=["certificateAuthorityArns"]
                                ),
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        ),
                
                        # the properties below are optional
                        certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        enforce=False,
                        ports=[123]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicy.html#cfn-appmesh-virtualnode-clientpolicy-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientPolicyProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ClientPolicyTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "validation": "validation",
            "certificate": "certificate",
            "enforce": "enforce",
            "ports": "ports",
        },
    )
    class ClientPolicyTlsProperty:
        def __init__(
            self,
            *,
            validation: typing.Union["CfnVirtualNode.TlsValidationContextProperty", _IResolvable_a771d0ef],
            certificate: typing.Optional[typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", _IResolvable_a771d0ef]] = None,
            enforce: typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]] = None,
            ports: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[jsii.Number]]] = None,
        ) -> None:
            '''A reference to an object that represents a Transport Layer Security (TLS) client policy.

            :param validation: A reference to an object that represents a TLS validation context.
            :param certificate: A reference to an object that represents a client's TLS certificate.
            :param enforce: Whether the policy is enforced. The default is ``True`` , if a value isn't specified.
            :param ports: One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                client_policy_tls_property = appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                certificate_authority_arns=["certificateAuthorityArns"]
                            ),
                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    ),
                
                    # the properties below are optional
                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    enforce=False,
                    ports=[123]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "validation": validation,
            }
            if certificate is not None:
                self._values["certificate"] = certificate
            if enforce is not None:
                self._values["enforce"] = enforce
            if ports is not None:
                self._values["ports"] = ports

        @builtins.property
        def validation(
            self,
        ) -> typing.Union["CfnVirtualNode.TlsValidationContextProperty", _IResolvable_a771d0ef]:
            '''A reference to an object that represents a TLS validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-validation
            '''
            result = self._values.get("validation")
            assert result is not None, "Required property 'validation' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.TlsValidationContextProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def certificate(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client's TLS certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-certificate
            '''
            result = self._values.get("certificate")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientTlsCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def enforce(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]]:
            '''Whether the policy is enforced.

            The default is ``True`` , if a value isn't specified.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-enforce
            '''
            result = self._values.get("enforce")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, _IResolvable_a771d0ef]], result)

        @builtins.property
        def ports(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]]:
            '''One or more ports that the policy is enforced for.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clientpolicytls.html#cfn-appmesh-virtualnode-clientpolicytls-ports
            '''
            result = self._values.get("ports")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[jsii.Number]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientPolicyTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ClientTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class ClientTlsCertificateProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the client's certificate.

            :param file: An object that represents a local file certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .
            :param sds: A reference to an object that represents a client's TLS Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                client_tls_certificate_property = appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html#cfn-appmesh-virtualnode-clienttlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a client's TLS Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-clienttlscertificate.html#cfn-appmesh-virtualnode-clienttlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={"hostname": "hostname", "response_type": "responseType"},
    )
    class DnsServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            hostname: builtins.str,
            response_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''An object that represents the DNS service discovery information for your virtual node.

            :param hostname: Specifies the DNS service discovery hostname for the virtual node.
            :param response_type: Specifies the DNS response type for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                dns_service_discovery_property = appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                    hostname="hostname",
                
                    # the properties below are optional
                    response_type="responseType"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "hostname": hostname,
            }
            if response_type is not None:
                self._values["response_type"] = response_type

        @builtins.property
        def hostname(self) -> builtins.str:
            '''Specifies the DNS service discovery hostname for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-hostname
            '''
            result = self._values.get("hostname")
            assert result is not None, "Required property 'hostname' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def response_type(self) -> typing.Optional[builtins.str]:
            '''Specifies the DNS response type for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-dnsservicediscovery.html#cfn-appmesh-virtualnode-dnsservicediscovery-responsetype
            '''
            result = self._values.get("response_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DnsServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.DurationProperty",
        jsii_struct_bases=[],
        name_mapping={"unit": "unit", "value": "value"},
    )
    class DurationProperty:
        def __init__(self, *, unit: builtins.str, value: jsii.Number) -> None:
            '''An object that represents a duration of time.

            :param unit: A unit of time.
            :param value: A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                duration_property = appmesh.CfnVirtualNode.DurationProperty(
                    unit="unit",
                    value=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "unit": unit,
                "value": value,
            }

        @builtins.property
        def unit(self) -> builtins.str:
            '''A unit of time.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html#cfn-appmesh-virtualnode-duration-unit
            '''
            result = self._values.get("unit")
            assert result is not None, "Required property 'unit' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> jsii.Number:
            '''A number of time units.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-duration.html#cfn-appmesh-virtualnode-duration-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DurationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.FileAccessLogProperty",
        jsii_struct_bases=[],
        name_mapping={"path": "path"},
    )
    class FileAccessLogProperty:
        def __init__(self, *, path: builtins.str) -> None:
            '''An object that represents an access log file.

            :param path: The file path to write access logs to. You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk. .. epigraph:: The Envoy process must have write permissions to the path that you specify here. Otherwise, Envoy fails to bootstrap properly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                file_access_log_property = appmesh.CfnVirtualNode.FileAccessLogProperty(
                    path="path"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "path": path,
            }

        @builtins.property
        def path(self) -> builtins.str:
            '''The file path to write access logs to.

            You can use ``/dev/stdout`` to send access logs to standard out and configure your Envoy container to use a log driver, such as ``awslogs`` , to export the access logs to a log storage service such as Amazon CloudWatch Logs. You can also specify a path in the Envoy container's file system to write the files to disk.
            .. epigraph::

               The Envoy process must have write permissions to the path that you specify here. Otherwise, Envoy fails to bootstrap properly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-fileaccesslog.html#cfn-appmesh-virtualnode-fileaccesslog-path
            '''
            result = self._values.get("path")
            assert result is not None, "Required property 'path' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FileAccessLogProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.GrpcTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class GrpcTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                grpc_timeout_property = appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html#cfn-appmesh-virtualnode-grpctimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-grpctimeout.html#cfn-appmesh-virtualnode-grpctimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GrpcTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.HealthCheckProperty",
        jsii_struct_bases=[],
        name_mapping={
            "healthy_threshold": "healthyThreshold",
            "interval_millis": "intervalMillis",
            "protocol": "protocol",
            "timeout_millis": "timeoutMillis",
            "unhealthy_threshold": "unhealthyThreshold",
            "path": "path",
            "port": "port",
        },
    )
    class HealthCheckProperty:
        def __init__(
            self,
            *,
            healthy_threshold: jsii.Number,
            interval_millis: jsii.Number,
            protocol: builtins.str,
            timeout_millis: jsii.Number,
            unhealthy_threshold: jsii.Number,
            path: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents the health check policy for a virtual node's listener.

            :param healthy_threshold: The number of consecutive successful health checks that must occur before declaring listener healthy.
            :param interval_millis: The time period in milliseconds between each health check execution.
            :param protocol: The protocol for the health check request. If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .
            :param timeout_millis: The amount of time to wait when receiving a response from the health check, in milliseconds.
            :param unhealthy_threshold: The number of consecutive failed health checks that must occur before declaring a virtual node unhealthy.
            :param path: The destination path for the health check request. This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.
            :param port: The destination port for the health check request. This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                health_check_property = appmesh.CfnVirtualNode.HealthCheckProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
                
                    # the properties below are optional
                    path="path",
                    port=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "healthy_threshold": healthy_threshold,
                "interval_millis": interval_millis,
                "protocol": protocol,
                "timeout_millis": timeout_millis,
                "unhealthy_threshold": unhealthy_threshold,
            }
            if path is not None:
                self._values["path"] = path
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def healthy_threshold(self) -> jsii.Number:
            '''The number of consecutive successful health checks that must occur before declaring listener healthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-healthythreshold
            '''
            result = self._values.get("healthy_threshold")
            assert result is not None, "Required property 'healthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def interval_millis(self) -> jsii.Number:
            '''The time period in milliseconds between each health check execution.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-intervalmillis
            '''
            result = self._values.get("interval_millis")
            assert result is not None, "Required property 'interval_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol for the health check request.

            If you specify ``grpc`` , then your service must conform to the `GRPC Health Checking Protocol <https://docs.aws.amazon.com/https://github.com/grpc/grpc/blob/master/doc/health-checking.md>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def timeout_millis(self) -> jsii.Number:
            '''The amount of time to wait when receiving a response from the health check, in milliseconds.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-timeoutmillis
            '''
            result = self._values.get("timeout_millis")
            assert result is not None, "Required property 'timeout_millis' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def unhealthy_threshold(self) -> jsii.Number:
            '''The number of consecutive failed health checks that must occur before declaring a virtual node unhealthy.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-unhealthythreshold
            '''
            result = self._values.get("unhealthy_threshold")
            assert result is not None, "Required property 'unhealthy_threshold' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def path(self) -> typing.Optional[builtins.str]:
            '''The destination path for the health check request.

            This value is only used if the specified protocol is HTTP or HTTP/2. For any other protocol, this value is ignored.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-path
            '''
            result = self._values.get("path")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''The destination port for the health check request.

            This port must match the port defined in the ``PortMapping`` for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-healthcheck.html#cfn-appmesh-virtualnode-healthcheck-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HealthCheckProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.HttpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle", "per_request": "perRequest"},
    )
    class HttpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]] = None,
            per_request: typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.
            :param per_request: An object that represents a per request timeout. The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                http_timeout_property = appmesh.CfnVirtualNode.HttpTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    per_request=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle
            if per_request is not None:
                self._values["per_request"] = per_request

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html#cfn-appmesh-virtualnode-httptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def per_request(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a per request timeout.

            The default value is 15 seconds. If you set a higher timeout, then make sure that the higher value is set for each App Mesh resource in a conversation. For example, if a virtual node backend uses a virtual router provider to route to another virtual node, then the timeout should be greater than 15 seconds for the source and destination virtual node and the route.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-httptimeout.html#cfn-appmesh-virtualnode-httptimeout-perrequest
            '''
            result = self._values.get("per_request")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HttpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "port_mapping": "portMapping",
            "connection_pool": "connectionPool",
            "health_check": "healthCheck",
            "outlier_detection": "outlierDetection",
            "timeout": "timeout",
            "tls": "tls",
        },
    )
    class ListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union["CfnVirtualNode.PortMappingProperty", _IResolvable_a771d0ef],
            connection_pool: typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            health_check: typing.Optional[typing.Union["CfnVirtualNode.HealthCheckProperty", _IResolvable_a771d0ef]] = None,
            outlier_detection: typing.Optional[typing.Union["CfnVirtualNode.OutlierDetectionProperty", _IResolvable_a771d0ef]] = None,
            timeout: typing.Optional[typing.Union["CfnVirtualNode.ListenerTimeoutProperty", _IResolvable_a771d0ef]] = None,
            tls: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener for a virtual node.

            :param port_mapping: The port mapping information for the listener.
            :param connection_pool: The connection pool information for the listener.
            :param health_check: The health check information for the listener.
            :param outlier_detection: The outlier detection information for the listener.
            :param timeout: An object that represents timeouts for different protocols.
            :param tls: A reference to an object that represents the Transport Layer Security (TLS) properties for a listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_property = appmesh.CfnVirtualNode.ListenerProperty(
                    port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    ),
                
                    # the properties below are optional
                    connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                        grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                            max_requests=123
                        ),
                        http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                            max_connections=123,
                
                            # the properties below are optional
                            max_pending_requests=123
                        ),
                        http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                            max_requests=123
                        ),
                        tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                            max_connections=123
                        )
                    ),
                    health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                        healthy_threshold=123,
                        interval_millis=123,
                        protocol="protocol",
                        timeout_millis=123,
                        unhealthy_threshold=123,
                
                        # the properties below are optional
                        path="path",
                        port=123
                    ),
                    outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                        base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        interval=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        max_ejection_percent=123,
                        max_server_errors=123
                    ),
                    timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                        grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            per_request=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        ),
                        tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                            idle=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            )
                        )
                    ),
                    tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                        certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                            acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                certificate_arn="certificateArn"
                            ),
                            file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                certificate_chain="certificateChain",
                                private_key="privateKey"
                            ),
                            sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                secret_name="secretName"
                            )
                        ),
                        mode="mode",
                
                        # the properties below are optional
                        validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                            trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                    certificate_chain="certificateChain"
                                ),
                                sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                    secret_name="secretName"
                                )
                            ),
                
                            # the properties below are optional
                            subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                    exact=["exact"]
                                )
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "port_mapping": port_mapping,
            }
            if connection_pool is not None:
                self._values["connection_pool"] = connection_pool
            if health_check is not None:
                self._values["health_check"] = health_check
            if outlier_detection is not None:
                self._values["outlier_detection"] = outlier_detection
            if timeout is not None:
                self._values["timeout"] = timeout
            if tls is not None:
                self._values["tls"] = tls

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union["CfnVirtualNode.PortMappingProperty", _IResolvable_a771d0ef]:
            '''The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.PortMappingProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def connection_pool(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''The connection pool information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-connectionpool
            '''
            result = self._values.get("connection_pool")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def health_check(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.HealthCheckProperty", _IResolvable_a771d0ef]]:
            '''The health check information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-healthcheck
            '''
            result = self._values.get("health_check")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.HealthCheckProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def outlier_detection(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.OutlierDetectionProperty", _IResolvable_a771d0ef]]:
            '''The outlier detection information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-outlierdetection
            '''
            result = self._values.get("outlier_detection")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.OutlierDetectionProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def timeout(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents timeouts for different protocols.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-timeout
            '''
            result = self._values.get("timeout")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tls(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the Transport Layer Security (TLS) properties for a listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listener.html#cfn-appmesh-virtualnode-listener-tls
            '''
            result = self._values.get("tls")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2", "tcp": "tcp"},
    )
    class ListenerTimeoutProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union["CfnVirtualNode.GrpcTimeoutProperty", _IResolvable_a771d0ef]] = None,
            http: typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]] = None,
            http2: typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]] = None,
            tcp: typing.Optional[typing.Union["CfnVirtualNode.TcpTimeoutProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents timeouts for different protocols.

            :param grpc: An object that represents types of timeouts.
            :param http: An object that represents types of timeouts.
            :param http2: An object that represents types of timeouts.
            :param tcp: An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_timeout_property = appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                    grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        ),
                        per_request=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    ),
                    tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                        idle=appmesh.CfnVirtualNode.DurationProperty(
                            unit="unit",
                            value=123
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2
            if tcp is not None:
                self._values["tcp"] = tcp

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.GrpcTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.GrpcTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.HttpTimeoutProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tcp(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TcpTimeoutProperty", _IResolvable_a771d0ef]]:
            '''An object that represents types of timeouts.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertimeout.html#cfn-appmesh-virtualnode-listenertimeout-tcp
            '''
            result = self._values.get("tcp")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TcpTimeoutProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_arn": "certificateArn"},
    )
    class ListenerTlsAcmCertificateProperty:
        def __init__(self, *, certificate_arn: builtins.str) -> None:
            '''An object that represents an AWS Certificate Manager certificate.

            :param certificate_arn: The Amazon Resource Name (ARN) for the certificate. The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsacmcertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_acm_certificate_property = appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                    certificate_arn="certificateArn"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_arn": certificate_arn,
            }

        @builtins.property
        def certificate_arn(self) -> builtins.str:
            '''The Amazon Resource Name (ARN) for the certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsacmcertificate.html#cfn-appmesh-virtualnode-listenertlsacmcertificate-certificatearn
            '''
            result = self._values.get("certificate_arn")
            assert result is not None, "Required property 'certificate_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsAcmCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class ListenerTlsCertificateProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) certificate.

            :param acm: A reference to an object that represents an AWS Certificate Manager certificate.
            :param file: A reference to an object that represents a local file certificate.
            :param sds: A reference to an object that represents a listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_certificate_property = appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                    acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                        certificate_arn="certificateArn"
                    ),
                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                        certificate_chain="certificateChain",
                        private_key="privateKey"
                    ),
                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsAcmCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a local file certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsFileCertificateProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a listener's Secret Discovery Service certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlscertificate.html#cfn-appmesh-virtualnode-listenertlscertificate-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsSdsCertificateProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate_chain": "certificateChain",
            "private_key": "privateKey",
        },
    )
    class ListenerTlsFileCertificateProperty:
        def __init__(
            self,
            *,
            certificate_chain: builtins.str,
            private_key: builtins.str,
        ) -> None:
            '''An object that represents a local file certificate.

            The certificate must meet specific requirements and you must have proxy authorization enabled. For more information, see `Transport Layer Security (TLS) <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html#virtual-node-tls-prerequisites>`_ .

            :param certificate_chain: The certificate chain for the certificate.
            :param private_key: The private key for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_file_certificate_property = appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                    certificate_chain="certificateChain",
                    private_key="privateKey"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_chain": certificate_chain,
                "private_key": private_key,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate chain for the certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html#cfn-appmesh-virtualnode-listenertlsfilecertificate-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def private_key(self) -> builtins.str:
            '''The private key for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsfilecertificate.html#cfn-appmesh-virtualnode-listenertlsfilecertificate-privatekey
            '''
            result = self._values.get("private_key")
            assert result is not None, "Required property 'private_key' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsFileCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "certificate": "certificate",
            "mode": "mode",
            "validation": "validation",
        },
    )
    class ListenerTlsProperty:
        def __init__(
            self,
            *,
            certificate: typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", _IResolvable_a771d0ef],
            mode: builtins.str,
            validation: typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the Transport Layer Security (TLS) properties for a listener.

            :param certificate: A reference to an object that represents a listener's Transport Layer Security (TLS) certificate.
            :param mode: Specify one of the following modes. - ** STRICT – Listener only accepts connections with TLS enabled. - ** PERMISSIVE – Listener accepts connections with or without TLS enabled. - ** DISABLED – Listener only accepts connections without TLS.
            :param validation: A reference to an object that represents a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_property = appmesh.CfnVirtualNode.ListenerTlsProperty(
                    certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                        acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                            certificate_arn="certificateArn"
                        ),
                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                            certificate_chain="certificateChain",
                            private_key="privateKey"
                        ),
                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                            secret_name="secretName"
                        )
                    ),
                    mode="mode",
                
                    # the properties below are optional
                    validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                        trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                certificate_chain="certificateChain"
                            ),
                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                secret_name="secretName"
                            )
                        ),
                
                        # the properties below are optional
                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                exact=["exact"]
                            )
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate": certificate,
                "mode": mode,
            }
            if validation is not None:
                self._values["validation"] = validation

        @builtins.property
        def certificate(
            self,
        ) -> typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", _IResolvable_a771d0ef]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-certificate
            '''
            result = self._values.get("certificate")
            assert result is not None, "Required property 'certificate' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.ListenerTlsCertificateProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def mode(self) -> builtins.str:
            '''Specify one of the following modes.

            - ** STRICT – Listener only accepts connections with TLS enabled.
            - ** PERMISSIVE – Listener accepts connections with or without TLS enabled.
            - ** DISABLED – Listener only accepts connections without TLS.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-mode
            '''
            result = self._values.get("mode")
            assert result is not None, "Required property 'mode' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def validation(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertls.html#cfn-appmesh-virtualnode-listenertls-validation
            '''
            result = self._values.get("validation")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ListenerTlsValidationContextProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class ListenerTlsSdsCertificateProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents the listener's Secret Discovery Service certificate.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlssdscertificate.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_sds_certificate_property = appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                    secret_name="secretName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret requested from the Secret Discovery Service provider representing Transport Layer Security (TLS) materials like a certificate or certificate chain.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlssdscertificate.html#cfn-appmesh-virtualnode-listenertlssdscertificate-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsSdsCertificateProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class ListenerTlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) validation context.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_validation_context_property = appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                    trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html#cfn-appmesh-virtualnode-listenertlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.ListenerTlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a listener's Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontext.html#cfn-appmesh-virtualnode-listenertlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"file": "file", "sds": "sds"},
    )
    class ListenerTlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            file: typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a listener's Transport Layer Security (TLS) validation context trust.

            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                listener_tls_validation_context_trust_property = appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-listenertlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a listener's Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-listenertlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-listenertlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ListenerTlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.LoggingProperty",
        jsii_struct_bases=[],
        name_mapping={"access_log": "accessLog"},
    )
    class LoggingProperty:
        def __init__(
            self,
            *,
            access_log: typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the logging information for a virtual node.

            :param access_log: The access log configuration for a virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-logging.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                logging_property = appmesh.CfnVirtualNode.LoggingProperty(
                    access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                        file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                            path="path"
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if access_log is not None:
                self._values["access_log"] = access_log

        @builtins.property
        def access_log(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", _IResolvable_a771d0ef]]:
            '''The access log configuration for a virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-logging.html#cfn-appmesh-virtualnode-logging-accesslog
            '''
            result = self._values.get("access_log")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.AccessLogProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LoggingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.OutlierDetectionProperty",
        jsii_struct_bases=[],
        name_mapping={
            "base_ejection_duration": "baseEjectionDuration",
            "interval": "interval",
            "max_ejection_percent": "maxEjectionPercent",
            "max_server_errors": "maxServerErrors",
        },
    )
    class OutlierDetectionProperty:
        def __init__(
            self,
            *,
            base_ejection_duration: typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef],
            interval: typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef],
            max_ejection_percent: jsii.Number,
            max_server_errors: jsii.Number,
        ) -> None:
            '''An object that represents the outlier detection for a virtual node's listener.

            :param base_ejection_duration: The base amount of time for which a host is ejected.
            :param interval: The time interval between ejection sweep analysis.
            :param max_ejection_percent: Maximum percentage of hosts in load balancing pool for upstream service that can be ejected. Will eject at least one host regardless of the value.
            :param max_server_errors: Number of consecutive ``5xx`` errors required for ejection.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                outlier_detection_property = appmesh.CfnVirtualNode.OutlierDetectionProperty(
                    base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    interval=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    ),
                    max_ejection_percent=123,
                    max_server_errors=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "base_ejection_duration": base_ejection_duration,
                "interval": interval,
                "max_ejection_percent": max_ejection_percent,
                "max_server_errors": max_server_errors,
            }

        @builtins.property
        def base_ejection_duration(
            self,
        ) -> typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]:
            '''The base amount of time for which a host is ejected.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-baseejectionduration
            '''
            result = self._values.get("base_ejection_duration")
            assert result is not None, "Required property 'base_ejection_duration' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def interval(
            self,
        ) -> typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]:
            '''The time interval between ejection sweep analysis.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-interval
            '''
            result = self._values.get("interval")
            assert result is not None, "Required property 'interval' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def max_ejection_percent(self) -> jsii.Number:
            '''Maximum percentage of hosts in load balancing pool for upstream service that can be ejected.

            Will eject at least one host regardless of the value.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-maxejectionpercent
            '''
            result = self._values.get("max_ejection_percent")
            assert result is not None, "Required property 'max_ejection_percent' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_server_errors(self) -> jsii.Number:
            '''Number of consecutive ``5xx`` errors required for ejection.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-outlierdetection.html#cfn-appmesh-virtualnode-outlierdetection-maxservererrors
            '''
            result = self._values.get("max_server_errors")
            assert result is not None, "Required property 'max_server_errors' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OutlierDetectionProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class PortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object representing a virtual node or virtual router listener port mapping.

            :param port: The port used for the port mapping.
            :param protocol: The protocol used for the port mapping. Specify ``http`` , ``http2`` , ``grpc`` , or ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                port_mapping_property = appmesh.CfnVirtualNode.PortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html#cfn-appmesh-virtualnode-portmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            Specify ``http`` , ``http2`` , ``grpc`` , or ``tcp`` .

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-portmapping.html#cfn-appmesh-virtualnode-portmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.ServiceDiscoveryProperty",
        jsii_struct_bases=[],
        name_mapping={"aws_cloud_map": "awsCloudMap", "dns": "dns"},
    )
    class ServiceDiscoveryProperty:
        def __init__(
            self,
            *,
            aws_cloud_map: typing.Optional[typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", _IResolvable_a771d0ef]] = None,
            dns: typing.Optional[typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the service discovery information for a virtual node.

            :param aws_cloud_map: Specifies any AWS Cloud Map information for the virtual node.
            :param dns: Specifies the DNS information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                service_discovery_property = appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                    aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                        namespace_name="namespaceName",
                        service_name="serviceName",
                
                        # the properties below are optional
                        attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                            key="key",
                            value="value"
                        )]
                    ),
                    dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                        hostname="hostname",
                
                        # the properties below are optional
                        response_type="responseType"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if aws_cloud_map is not None:
                self._values["aws_cloud_map"] = aws_cloud_map
            if dns is not None:
                self._values["dns"] = dns

        @builtins.property
        def aws_cloud_map(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''Specifies any AWS Cloud Map information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html#cfn-appmesh-virtualnode-servicediscovery-awscloudmap
            '''
            result = self._values.get("aws_cloud_map")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def dns(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''Specifies the DNS information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-servicediscovery.html#cfn-appmesh-virtualnode-servicediscovery-dns
            '''
            result = self._values.get("dns")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DnsServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ServiceDiscoveryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty",
        jsii_struct_bases=[],
        name_mapping={"exact": "exact"},
    )
    class SubjectAlternativeNameMatchersProperty:
        def __init__(
            self,
            *,
            exact: typing.Optional[typing.Sequence[builtins.str]] = None,
        ) -> None:
            '''An object that represents the methods by which a subject alternative name on a peer Transport Layer Security (TLS) certificate can be matched.

            :param exact: The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenamematchers.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_name_matchers_property = appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                    exact=["exact"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if exact is not None:
                self._values["exact"] = exact

        @builtins.property
        def exact(self) -> typing.Optional[typing.List[builtins.str]]:
            '''The values sent must match the specified values exactly.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenamematchers.html#cfn-appmesh-virtualnode-subjectalternativenamematchers-exact
            '''
            result = self._values.get("exact")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNameMatchersProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty",
        jsii_struct_bases=[],
        name_mapping={"match": "match"},
    )
    class SubjectAlternativeNamesProperty:
        def __init__(
            self,
            *,
            match: typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents the subject alternative names secured by the certificate.

            :param match: An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenames.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                subject_alternative_names_property = appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                        exact=["exact"]
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "match": match,
            }

        @builtins.property
        def match(
            self,
        ) -> typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef]:
            '''An object that represents the criteria for determining a SANs match.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-subjectalternativenames.html#cfn-appmesh-virtualnode-subjectalternativenames-match
            '''
            result = self._values.get("match")
            assert result is not None, "Required property 'match' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.SubjectAlternativeNameMatchersProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SubjectAlternativeNamesProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TcpTimeoutProperty",
        jsii_struct_bases=[],
        name_mapping={"idle": "idle"},
    )
    class TcpTimeoutProperty:
        def __init__(
            self,
            *,
            idle: typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents types of timeouts.

            :param idle: An object that represents an idle timeout. An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tcptimeout.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tcp_timeout_property = appmesh.CfnVirtualNode.TcpTimeoutProperty(
                    idle=appmesh.CfnVirtualNode.DurationProperty(
                        unit="unit",
                        value=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if idle is not None:
                self._values["idle"] = idle

        @builtins.property
        def idle(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]]:
            '''An object that represents an idle timeout.

            An idle timeout bounds the amount of time that a connection may be idle. The default value is none.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tcptimeout.html#cfn-appmesh-virtualnode-tcptimeout-idle
            '''
            result = self._values.get("idle")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.DurationProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TcpTimeoutProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_authority_arns": "certificateAuthorityArns"},
    )
    class TlsValidationContextAcmTrustProperty:
        def __init__(
            self,
            *,
            certificate_authority_arns: typing.Sequence[builtins.str],
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :param certificate_authority_arns: One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextacmtrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_acm_trust_property = appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                    certificate_authority_arns=["certificateAuthorityArns"]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_authority_arns": certificate_authority_arns,
            }

        @builtins.property
        def certificate_authority_arns(self) -> typing.List[builtins.str]:
            '''One or more ACM Amazon Resource Name (ARN)s.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextacmtrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextacmtrust-certificateauthorityarns
            '''
            result = self._values.get("certificate_authority_arns")
            assert result is not None, "Required property 'certificate_authority_arns' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextAcmTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"certificate_chain": "certificateChain"},
    )
    class TlsValidationContextFileTrustProperty:
        def __init__(self, *, certificate_chain: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :param certificate_chain: The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextfiletrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_file_trust_property = appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                    certificate_chain="certificateChain"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "certificate_chain": certificate_chain,
            }

        @builtins.property
        def certificate_chain(self) -> builtins.str:
            '''The certificate trust chain for a certificate stored on the file system of the virtual node that the proxy is running on.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextfiletrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextfiletrust-certificatechain
            '''
            result = self._values.get("certificate_chain")
            assert result is not None, "Required property 'certificate_chain' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextFileTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextProperty",
        jsii_struct_bases=[],
        name_mapping={
            "trust": "trust",
            "subject_alternative_names": "subjectAlternativeNames",
        },
    )
    class TlsValidationContextProperty:
        def __init__(
            self,
            *,
            trust: typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", _IResolvable_a771d0ef],
            subject_alternative_names: typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents how the proxy will validate its peer during Transport Layer Security (TLS) negotiation.

            :param trust: A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.
            :param subject_alternative_names: A reference to an object that represents the SANs for a Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_property = appmesh.CfnVirtualNode.TlsValidationContextProperty(
                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                            certificate_authority_arns=["certificateAuthorityArns"]
                        ),
                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                            certificate_chain="certificateChain"
                        ),
                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                            secret_name="secretName"
                        )
                    ),
                
                    # the properties below are optional
                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                            exact=["exact"]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "trust": trust,
            }
            if subject_alternative_names is not None:
                self._values["subject_alternative_names"] = subject_alternative_names

        @builtins.property
        def trust(
            self,
        ) -> typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", _IResolvable_a771d0ef]:
            '''A reference to where to retrieve the trust chain when validating a peer’s Transport Layer Security (TLS) certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html#cfn-appmesh-virtualnode-tlsvalidationcontext-trust
            '''
            result = self._values.get("trust")
            assert result is not None, "Required property 'trust' is missing"
            return typing.cast(typing.Union["CfnVirtualNode.TlsValidationContextTrustProperty", _IResolvable_a771d0ef], result)

        @builtins.property
        def subject_alternative_names(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the SANs for a Transport Layer Security (TLS) validation context.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontext.html#cfn-appmesh-virtualnode-tlsvalidationcontext-subjectalternativenames
            '''
            result = self._values.get("subject_alternative_names")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.SubjectAlternativeNamesProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"secret_name": "secretName"},
    )
    class TlsValidationContextSdsTrustProperty:
        def __init__(self, *, secret_name: builtins.str) -> None:
            '''An object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            The proxy must be configured with a local SDS provider via a Unix Domain Socket. See App Mesh `TLS documentation <https://docs.aws.amazon.com/app-mesh/latest/userguide/tls.html>`_ for more info.

            :param secret_name: A reference to an object that represents the name of the secret for a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextsdstrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_sds_trust_property = appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                    secret_name="secretName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "secret_name": secret_name,
            }

        @builtins.property
        def secret_name(self) -> builtins.str:
            '''A reference to an object that represents the name of the secret for a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontextsdstrust.html#cfn-appmesh-virtualnode-tlsvalidationcontextsdstrust-secretname
            '''
            result = self._values.get("secret_name")
            assert result is not None, "Required property 'secret_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextSdsTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.TlsValidationContextTrustProperty",
        jsii_struct_bases=[],
        name_mapping={"acm": "acm", "file": "file", "sds": "sds"},
    )
    class TlsValidationContextTrustProperty:
        def __init__(
            self,
            *,
            acm: typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]] = None,
            file: typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]] = None,
            sds: typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a Transport Layer Security (TLS) validation context trust.

            :param acm: A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.
            :param file: An object that represents a Transport Layer Security (TLS) validation context trust for a local file.
            :param sds: A reference to an object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                tls_validation_context_trust_property = appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                        certificate_authority_arns=["certificateAuthorityArns"]
                    ),
                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                        certificate_chain="certificateChain"
                    ),
                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                        secret_name="secretName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if acm is not None:
                self._values["acm"] = acm
            if file is not None:
                self._values["file"] = file
            if sds is not None:
                self._values["sds"] = sds

        @builtins.property
        def acm(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) validation context trust for an AWS Certificate Manager certificate.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-acm
            '''
            result = self._values.get("acm")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextAcmTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def file(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a Transport Layer Security (TLS) validation context trust for a local file.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-file
            '''
            result = self._values.get("file")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextFileTrustProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def sds(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents a Transport Layer Security (TLS) Secret Discovery Service validation context trust.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-tlsvalidationcontexttrust.html#cfn-appmesh-virtualnode-tlsvalidationcontexttrust-sds
            '''
            result = self._values.get("sds")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.TlsValidationContextSdsTrustProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TlsValidationContextTrustProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"grpc": "grpc", "http": "http", "http2": "http2", "tcp": "tcp"},
    )
    class VirtualNodeConnectionPoolProperty:
        def __init__(
            self,
            *,
            grpc: typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            http: typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            http2: typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
            tcp: typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the type of virtual node connection pool.

            Only one protocol is used at a time and should be the same protocol as the one chosen under port mapping.

            If not present the default value for ``maxPendingRequests`` is ``2147483647`` .

            :param grpc: An object that represents a type of connection pool.
            :param http: An object that represents a type of connection pool.
            :param http2: An object that represents a type of connection pool.
            :param tcp: An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                    grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                        max_requests=123
                    ),
                    http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                        max_connections=123,
                
                        # the properties below are optional
                        max_pending_requests=123
                    ),
                    http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                        max_requests=123
                    ),
                    tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                        max_connections=123
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if grpc is not None:
                self._values["grpc"] = grpc
            if http is not None:
                self._values["http"] = http
            if http2 is not None:
                self._values["http2"] = http2
            if tcp is not None:
                self._values["tcp"] = tcp

        @builtins.property
        def grpc(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-grpc
            '''
            result = self._values.get("grpc")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-http
            '''
            result = self._values.get("http")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def http2(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-http2
            '''
            result = self._values.get("http2")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def tcp(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", _IResolvable_a771d0ef]]:
            '''An object that represents a type of connection pool.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodeconnectionpool.html#cfn-appmesh-virtualnode-virtualnodeconnectionpool-tcp
            '''
            result = self._values.get("tcp")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualNodeGrpcConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodegrpcconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_grpc_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                    max_requests=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodegrpcconnectionpool.html#cfn-appmesh-virtualnode-virtualnodegrpcconnectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeGrpcConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_requests": "maxRequests"},
    )
    class VirtualNodeHttp2ConnectionPoolProperty:
        def __init__(self, *, max_requests: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_requests: Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttp2connectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_http2_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                    max_requests=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_requests": max_requests,
            }

        @builtins.property
        def max_requests(self) -> jsii.Number:
            '''Maximum number of inflight requests Envoy can concurrently support across hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttp2connectionpool.html#cfn-appmesh-virtualnode-virtualnodehttp2connectionpool-maxrequests
            '''
            result = self._values.get("max_requests")
            assert result is not None, "Required property 'max_requests' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeHttp2ConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={
            "max_connections": "maxConnections",
            "max_pending_requests": "maxPendingRequests",
        },
    )
    class VirtualNodeHttpConnectionPoolProperty:
        def __init__(
            self,
            *,
            max_connections: jsii.Number,
            max_pending_requests: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.
            :param max_pending_requests: Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_http_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                    max_connections=123,
                
                    # the properties below are optional
                    max_pending_requests=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_connections": max_connections,
            }
            if max_pending_requests is not None:
                self._values["max_pending_requests"] = max_pending_requests

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodehttpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def max_pending_requests(self) -> typing.Optional[jsii.Number]:
            '''Number of overflowing requests after ``max_connections`` Envoy will queue to upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodehttpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodehttpconnectionpool-maxpendingrequests
            '''
            result = self._values.get("max_pending_requests")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeHttpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeSpecProperty",
        jsii_struct_bases=[],
        name_mapping={
            "backend_defaults": "backendDefaults",
            "backends": "backends",
            "listeners": "listeners",
            "logging": "logging",
            "service_discovery": "serviceDiscovery",
        },
    )
    class VirtualNodeSpecProperty:
        def __init__(
            self,
            *,
            backend_defaults: typing.Optional[typing.Union["CfnVirtualNode.BackendDefaultsProperty", _IResolvable_a771d0ef]] = None,
            backends: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnVirtualNode.BackendProperty", _IResolvable_a771d0ef]]]] = None,
            listeners: typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnVirtualNode.ListenerProperty", _IResolvable_a771d0ef]]]] = None,
            logging: typing.Optional[typing.Union["CfnVirtualNode.LoggingProperty", _IResolvable_a771d0ef]] = None,
            service_discovery: typing.Optional[typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a virtual node.

            :param backend_defaults: A reference to an object that represents the defaults for backends.
            :param backends: The backends that the virtual node is expected to send outbound traffic to.
            :param listeners: The listener that the virtual node is expected to receive inbound traffic from. You can specify one listener.
            :param logging: The inbound and outbound access logging information for the virtual node.
            :param service_discovery: The service discovery information for the virtual node. If your virtual node does not expect ingress traffic, you can omit this parameter. If you specify a ``listener`` , then you must specify service discovery information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_spec_property = appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                    backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
                
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    backends=[appmesh.CfnVirtualNode.BackendProperty(
                        virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                            virtual_service_name="virtualServiceName",
                
                            # the properties below are optional
                            client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                                tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                                certificate_authority_arns=["certificateAuthorityArns"]
                                            ),
                                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                                certificate_chain="certificateChain"
                                            ),
                                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                                secret_name="secretName"
                                            )
                                        ),
                
                                        # the properties below are optional
                                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                                exact=["exact"]
                                            )
                                        )
                                    ),
                
                                    # the properties below are optional
                                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                            certificate_chain="certificateChain",
                                            private_key="privateKey"
                                        ),
                                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                                    enforce=False,
                                    ports=[123]
                                )
                            )
                        )
                    )],
                    listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                        port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
                
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                                max_connections=123,
                
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                                max_requests=123
                            ),
                            tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                                max_connections=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
                
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                            base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            interval=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            max_ejection_percent=123,
                            max_server_errors=123
                        ),
                        timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                            grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            )
                        ),
                        tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                            certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
                
                            # the properties below are optional
                            validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                    logging=appmesh.CfnVirtualNode.LoggingProperty(
                        access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                            file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                                path="path"
                            )
                        )
                    ),
                    service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                        aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                            namespace_name="namespaceName",
                            service_name="serviceName",
                
                            # the properties below are optional
                            attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                                key="key",
                                value="value"
                            )]
                        ),
                        dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                            hostname="hostname",
                
                            # the properties below are optional
                            response_type="responseType"
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if backend_defaults is not None:
                self._values["backend_defaults"] = backend_defaults
            if backends is not None:
                self._values["backends"] = backends
            if listeners is not None:
                self._values["listeners"] = listeners
            if logging is not None:
                self._values["logging"] = logging
            if service_discovery is not None:
                self._values["service_discovery"] = service_discovery

        @builtins.property
        def backend_defaults(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.BackendDefaultsProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the defaults for backends.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-backenddefaults
            '''
            result = self._values.get("backend_defaults")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.BackendDefaultsProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def backends(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.BackendProperty", _IResolvable_a771d0ef]]]]:
            '''The backends that the virtual node is expected to send outbound traffic to.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-backends
            '''
            result = self._values.get("backends")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.BackendProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def listeners(
            self,
        ) -> typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.ListenerProperty", _IResolvable_a771d0ef]]]]:
            '''The listener that the virtual node is expected to receive inbound traffic from.

            You can specify one listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-listeners
            '''
            result = self._values.get("listeners")
            return typing.cast(typing.Optional[typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualNode.ListenerProperty", _IResolvable_a771d0ef]]]], result)

        @builtins.property
        def logging(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.LoggingProperty", _IResolvable_a771d0ef]]:
            '''The inbound and outbound access logging information for the virtual node.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-logging
            '''
            result = self._values.get("logging")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.LoggingProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def service_discovery(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", _IResolvable_a771d0ef]]:
            '''The service discovery information for the virtual node.

            If your virtual node does not expect ingress traffic, you can omit this parameter. If you specify a ``listener`` , then you must specify service discovery information.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodespec.html#cfn-appmesh-virtualnode-virtualnodespec-servicediscovery
            '''
            result = self._values.get("service_discovery")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ServiceDiscoveryProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty",
        jsii_struct_bases=[],
        name_mapping={"max_connections": "maxConnections"},
    )
    class VirtualNodeTcpConnectionPoolProperty:
        def __init__(self, *, max_connections: jsii.Number) -> None:
            '''An object that represents a type of connection pool.

            :param max_connections: Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodetcpconnectionpool.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_tcp_connection_pool_property = appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                    max_connections=123
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "max_connections": max_connections,
            }

        @builtins.property
        def max_connections(self) -> jsii.Number:
            '''Maximum number of outbound TCP connections Envoy can establish concurrently with all hosts in upstream cluster.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualnodetcpconnectionpool.html#cfn-appmesh-virtualnode-virtualnodetcpconnectionpool-maxconnections
            '''
            result = self._values.get("max_connections")
            assert result is not None, "Required property 'max_connections' is missing"
            return typing.cast(jsii.Number, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeTcpConnectionPoolProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualNode.VirtualServiceBackendProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_service_name": "virtualServiceName",
            "client_policy": "clientPolicy",
        },
    )
    class VirtualServiceBackendProperty:
        def __init__(
            self,
            *,
            virtual_service_name: builtins.str,
            client_policy: typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents a virtual service backend for a virtual node.

            :param virtual_service_name: The name of the virtual service that is acting as a virtual node backend.
            :param client_policy: A reference to an object that represents the client policy for a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_service_backend_property = appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                    virtual_service_name="virtualServiceName",
                
                    # the properties below are optional
                    client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                        tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                            validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                    acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                        certificate_authority_arns=["certificateAuthorityArns"]
                                    ),
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
                
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            ),
                
                            # the properties below are optional
                            certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            enforce=False,
                            ports=[123]
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "virtual_service_name": virtual_service_name,
            }
            if client_policy is not None:
                self._values["client_policy"] = client_policy

        @builtins.property
        def virtual_service_name(self) -> builtins.str:
            '''The name of the virtual service that is acting as a virtual node backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html#cfn-appmesh-virtualnode-virtualservicebackend-virtualservicename
            '''
            result = self._values.get("virtual_service_name")
            assert result is not None, "Required property 'virtual_service_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def client_policy(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]]:
            '''A reference to an object that represents the client policy for a backend.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualnode-virtualservicebackend.html#cfn-appmesh-virtualnode-virtualservicebackend-clientpolicy
            '''
            result = self._values.get("client_policy")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualNode.ClientPolicyProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceBackendProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualNodeProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_node_name": "virtualNodeName",
    },
)
class CfnVirtualNodeProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        virtual_node_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualNode``.

        :param mesh_name: The name of the service mesh to create the virtual node in.
        :param spec: The virtual node specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual node to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_node_name: The name to use for the virtual node.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_node_props = appmesh.CfnVirtualNodeProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualNode.VirtualNodeSpecProperty(
                    backend_defaults=appmesh.CfnVirtualNode.BackendDefaultsProperty(
                        client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                            tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                    trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                        acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                            certificate_authority_arns=["certificateAuthorityArns"]
                                        ),
                                        file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                            certificate_chain="certificateChain"
                                        ),
                                        sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                            secret_name="secretName"
                                        )
                                    ),
            
                                    # the properties below are optional
                                    subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                        match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                            exact=["exact"]
                                        )
                                    )
                                ),
            
                                # the properties below are optional
                                certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                    file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                        certificate_chain="certificateChain",
                                        private_key="privateKey"
                                    ),
                                    sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                        secret_name="secretName"
                                    )
                                ),
                                enforce=False,
                                ports=[123]
                            )
                        )
                    ),
                    backends=[appmesh.CfnVirtualNode.BackendProperty(
                        virtual_service=appmesh.CfnVirtualNode.VirtualServiceBackendProperty(
                            virtual_service_name="virtualServiceName",
            
                            # the properties below are optional
                            client_policy=appmesh.CfnVirtualNode.ClientPolicyProperty(
                                tls=appmesh.CfnVirtualNode.ClientPolicyTlsProperty(
                                    validation=appmesh.CfnVirtualNode.TlsValidationContextProperty(
                                        trust=appmesh.CfnVirtualNode.TlsValidationContextTrustProperty(
                                            acm=appmesh.CfnVirtualNode.TlsValidationContextAcmTrustProperty(
                                                certificate_authority_arns=["certificateAuthorityArns"]
                                            ),
                                            file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                                certificate_chain="certificateChain"
                                            ),
                                            sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                                secret_name="secretName"
                                            )
                                        ),
            
                                        # the properties below are optional
                                        subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                            match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                                exact=["exact"]
                                            )
                                        )
                                    ),
            
                                    # the properties below are optional
                                    certificate=appmesh.CfnVirtualNode.ClientTlsCertificateProperty(
                                        file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                            certificate_chain="certificateChain",
                                            private_key="privateKey"
                                        ),
                                        sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                            secret_name="secretName"
                                        )
                                    ),
                                    enforce=False,
                                    ports=[123]
                                )
                            )
                        )
                    )],
                    listeners=[appmesh.CfnVirtualNode.ListenerProperty(
                        port_mapping=appmesh.CfnVirtualNode.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        ),
            
                        # the properties below are optional
                        connection_pool=appmesh.CfnVirtualNode.VirtualNodeConnectionPoolProperty(
                            grpc=appmesh.CfnVirtualNode.VirtualNodeGrpcConnectionPoolProperty(
                                max_requests=123
                            ),
                            http=appmesh.CfnVirtualNode.VirtualNodeHttpConnectionPoolProperty(
                                max_connections=123,
            
                                # the properties below are optional
                                max_pending_requests=123
                            ),
                            http2=appmesh.CfnVirtualNode.VirtualNodeHttp2ConnectionPoolProperty(
                                max_requests=123
                            ),
                            tcp=appmesh.CfnVirtualNode.VirtualNodeTcpConnectionPoolProperty(
                                max_connections=123
                            )
                        ),
                        health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                            healthy_threshold=123,
                            interval_millis=123,
                            protocol="protocol",
                            timeout_millis=123,
                            unhealthy_threshold=123,
            
                            # the properties below are optional
                            path="path",
                            port=123
                        ),
                        outlier_detection=appmesh.CfnVirtualNode.OutlierDetectionProperty(
                            base_ejection_duration=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            interval=appmesh.CfnVirtualNode.DurationProperty(
                                unit="unit",
                                value=123
                            ),
                            max_ejection_percent=123,
                            max_server_errors=123
                        ),
                        timeout=appmesh.CfnVirtualNode.ListenerTimeoutProperty(
                            grpc=appmesh.CfnVirtualNode.GrpcTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            http2=appmesh.CfnVirtualNode.HttpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                ),
                                per_request=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            ),
                            tcp=appmesh.CfnVirtualNode.TcpTimeoutProperty(
                                idle=appmesh.CfnVirtualNode.DurationProperty(
                                    unit="unit",
                                    value=123
                                )
                            )
                        ),
                        tls=appmesh.CfnVirtualNode.ListenerTlsProperty(
                            certificate=appmesh.CfnVirtualNode.ListenerTlsCertificateProperty(
                                acm=appmesh.CfnVirtualNode.ListenerTlsAcmCertificateProperty(
                                    certificate_arn="certificateArn"
                                ),
                                file=appmesh.CfnVirtualNode.ListenerTlsFileCertificateProperty(
                                    certificate_chain="certificateChain",
                                    private_key="privateKey"
                                ),
                                sds=appmesh.CfnVirtualNode.ListenerTlsSdsCertificateProperty(
                                    secret_name="secretName"
                                )
                            ),
                            mode="mode",
            
                            # the properties below are optional
                            validation=appmesh.CfnVirtualNode.ListenerTlsValidationContextProperty(
                                trust=appmesh.CfnVirtualNode.ListenerTlsValidationContextTrustProperty(
                                    file=appmesh.CfnVirtualNode.TlsValidationContextFileTrustProperty(
                                        certificate_chain="certificateChain"
                                    ),
                                    sds=appmesh.CfnVirtualNode.TlsValidationContextSdsTrustProperty(
                                        secret_name="secretName"
                                    )
                                ),
            
                                # the properties below are optional
                                subject_alternative_names=appmesh.CfnVirtualNode.SubjectAlternativeNamesProperty(
                                    match=appmesh.CfnVirtualNode.SubjectAlternativeNameMatchersProperty(
                                        exact=["exact"]
                                    )
                                )
                            )
                        )
                    )],
                    logging=appmesh.CfnVirtualNode.LoggingProperty(
                        access_log=appmesh.CfnVirtualNode.AccessLogProperty(
                            file=appmesh.CfnVirtualNode.FileAccessLogProperty(
                                path="path"
                            )
                        )
                    ),
                    service_discovery=appmesh.CfnVirtualNode.ServiceDiscoveryProperty(
                        aws_cloud_map=appmesh.CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty(
                            namespace_name="namespaceName",
                            service_name="serviceName",
            
                            # the properties below are optional
                            attributes=[appmesh.CfnVirtualNode.AwsCloudMapInstanceAttributeProperty(
                                key="key",
                                value="value"
                            )]
                        ),
                        dns=appmesh.CfnVirtualNode.DnsServiceDiscoveryProperty(
                            hostname="hostname",
            
                            # the properties below are optional
                            response_type="responseType"
                        )
                    )
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_node_name="virtualNodeName"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_node_name is not None:
            self._values["virtual_node_name"] = virtual_node_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual node in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, _IResolvable_a771d0ef]:
        '''The virtual node specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualNode.VirtualNodeSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual node to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def virtual_node_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual node.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualnode.html#cfn-appmesh-virtualnode-virtualnodename
        '''
        result = self._values.get("virtual_node_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualNodeProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualRouter(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualRouter",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualRouter``.

    Creates a virtual router within a service mesh.

    Specify a ``listener`` for any inbound traffic that your virtual router receives. Create a virtual router for each protocol and port that you need to route. Virtual routers handle traffic for one or more virtual services within your mesh. After you create your virtual router, create and associate routes for your virtual router that direct incoming requests to different virtual nodes.

    For more information about virtual routers, see `Virtual routers <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_routers.html>`_ .

    :cloudformationResource: AWS::AppMesh::VirtualRouter
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_router = appmesh.CfnVirtualRouter(self, "MyCfnVirtualRouter",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                    port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    )
                )]
            ),
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )],
            virtual_router_name="virtualRouterName"
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        virtual_router_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualRouter``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the virtual router in.
        :param spec: The virtual router specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual router to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_router_name: The name to use for the virtual router.
        '''
        props = CfnVirtualRouterProps(
            mesh_name=mesh_name,
            spec=spec,
            mesh_owner=mesh_owner,
            tags=tags,
            virtual_router_name=virtual_router_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual router.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual router resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual router.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVirtualRouterName")
    def attr_virtual_router_name(self) -> builtins.str:
        '''The name of the virtual router.

        :cloudformationAttribute: VirtualRouterName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualRouterName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual router to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual router in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef]:
        '''The virtual router specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-spec
        '''
        return typing.cast(typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualRouter.VirtualRouterSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "spec", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshOwner", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualRouterName")
    def virtual_router_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual router.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-virtualroutername
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualRouterName"))

    @virtual_router_name.setter
    def virtual_router_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "virtualRouterName", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualRouter.PortMappingProperty",
        jsii_struct_bases=[],
        name_mapping={"port": "port", "protocol": "protocol"},
    )
    class PortMappingProperty:
        def __init__(self, *, port: jsii.Number, protocol: builtins.str) -> None:
            '''An object representing a virtual router listener port mapping.

            :param port: The port used for the port mapping.
            :param protocol: The protocol used for the port mapping. Specify one protocol.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                port_mapping_property = appmesh.CfnVirtualRouter.PortMappingProperty(
                    port=123,
                    protocol="protocol"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "port": port,
                "protocol": protocol,
            }

        @builtins.property
        def port(self) -> jsii.Number:
            '''The port used for the port mapping.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html#cfn-appmesh-virtualrouter-portmapping-port
            '''
            result = self._values.get("port")
            assert result is not None, "Required property 'port' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def protocol(self) -> builtins.str:
            '''The protocol used for the port mapping.

            Specify one protocol.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-portmapping.html#cfn-appmesh-virtualrouter-portmapping-protocol
            '''
            result = self._values.get("protocol")
            assert result is not None, "Required property 'protocol' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualRouter.VirtualRouterListenerProperty",
        jsii_struct_bases=[],
        name_mapping={"port_mapping": "portMapping"},
    )
    class VirtualRouterListenerProperty:
        def __init__(
            self,
            *,
            port_mapping: typing.Union["CfnVirtualRouter.PortMappingProperty", _IResolvable_a771d0ef],
        ) -> None:
            '''An object that represents a virtual router listener.

            :param port_mapping: The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterlistener.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_router_listener_property = appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                    port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                        port=123,
                        protocol="protocol"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "port_mapping": port_mapping,
            }

        @builtins.property
        def port_mapping(
            self,
        ) -> typing.Union["CfnVirtualRouter.PortMappingProperty", _IResolvable_a771d0ef]:
            '''The port mapping information for the listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterlistener.html#cfn-appmesh-virtualrouter-virtualrouterlistener-portmapping
            '''
            result = self._values.get("port_mapping")
            assert result is not None, "Required property 'port_mapping' is missing"
            return typing.cast(typing.Union["CfnVirtualRouter.PortMappingProperty", _IResolvable_a771d0ef], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualRouter.VirtualRouterSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"listeners": "listeners"},
    )
    class VirtualRouterSpecProperty:
        def __init__(
            self,
            *,
            listeners: typing.Union[_IResolvable_a771d0ef, typing.Sequence[typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", _IResolvable_a771d0ef]]],
        ) -> None:
            '''An object that represents the specification of a virtual router.

            :param listeners: The listeners that the virtual router is expected to receive inbound traffic from. You can specify one listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterspec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_router_spec_property = appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                    listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                        port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        )
                    )]
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "listeners": listeners,
            }

        @builtins.property
        def listeners(
            self,
        ) -> typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", _IResolvable_a771d0ef]]]:
            '''The listeners that the virtual router is expected to receive inbound traffic from.

            You can specify one listener.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualrouter-virtualrouterspec.html#cfn-appmesh-virtualrouter-virtualrouterspec-listeners
            '''
            result = self._values.get("listeners")
            assert result is not None, "Required property 'listeners' is missing"
            return typing.cast(typing.Union[_IResolvable_a771d0ef, typing.List[typing.Union["CfnVirtualRouter.VirtualRouterListenerProperty", _IResolvable_a771d0ef]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualRouterProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "mesh_owner": "meshOwner",
        "tags": "tags",
        "virtual_router_name": "virtualRouterName",
    },
)
class CfnVirtualRouterProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, _IResolvable_a771d0ef],
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
        virtual_router_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualRouter``.

        :param mesh_name: The name of the service mesh to create the virtual router in.
        :param spec: The virtual router specification to apply.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual router to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        :param virtual_router_name: The name to use for the virtual router.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_router_props = appmesh.CfnVirtualRouterProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualRouter.VirtualRouterSpecProperty(
                    listeners=[appmesh.CfnVirtualRouter.VirtualRouterListenerProperty(
                        port_mapping=appmesh.CfnVirtualRouter.PortMappingProperty(
                            port=123,
                            protocol="protocol"
                        )
                    )]
                ),
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )],
                virtual_router_name="virtualRouterName"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags
        if virtual_router_name is not None:
            self._values["virtual_router_name"] = virtual_router_name

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual router in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, _IResolvable_a771d0ef]:
        '''The virtual router specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualRouter.VirtualRouterSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual router to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    @builtins.property
    def virtual_router_name(self) -> typing.Optional[builtins.str]:
        '''The name to use for the virtual router.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualrouter.html#cfn-appmesh-virtualrouter-virtualroutername
        '''
        result = self._values.get("virtual_router_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualRouterProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(_IInspectable_82c04a63)
class CfnVirtualService(
    _CfnResource_e0a482dc,
    metaclass=jsii.JSIIMeta,
    jsii_type="monocdk.aws_appmesh.CfnVirtualService",
):
    '''A CloudFormation ``AWS::AppMesh::VirtualService``.

    Creates a virtual service within a service mesh.

    A virtual service is an abstraction of a real service that is provided by a virtual node directly or indirectly by means of a virtual router. Dependent services call your virtual service by its ``virtualServiceName`` , and those requests are routed to the virtual node or virtual router that is specified as the provider for the virtual service.

    For more information about virtual services, see `Virtual services <https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html>`_ .

    :cloudformationResource: AWS::AppMesh::VirtualService
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html
    :exampleMetadata: fixture=_generated

    Example::

        # The code below shows an example of how to instantiate this type.
        # The values are placeholders you should change.
        from monocdk import aws_appmesh as appmesh
        
        cfn_virtual_service = appmesh.CfnVirtualService(self, "MyCfnVirtualService",
            mesh_name="meshName",
            spec=appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                    virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                        virtual_node_name="virtualNodeName"
                    ),
                    virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                        virtual_router_name="virtualRouterName"
                    )
                )
            ),
            virtual_service_name="virtualServiceName",
        
            # the properties below are optional
            mesh_owner="meshOwner",
            tags=[CfnTag(
                key="key",
                value="value"
            )]
        )
    '''

    def __init__(
        self,
        scope: _Construct_e78e779f,
        id: builtins.str,
        *,
        mesh_name: builtins.str,
        spec: typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef],
        virtual_service_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Create a new ``AWS::AppMesh::VirtualService``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param mesh_name: The name of the service mesh to create the virtual service in.
        :param spec: The virtual service specification to apply.
        :param virtual_service_name: The name to use for the virtual service.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual service to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.
        '''
        props = CfnVirtualServiceProps(
            mesh_name=mesh_name,
            spec=spec,
            virtual_service_name=virtual_service_name,
            mesh_owner=mesh_owner,
            tags=tags,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: _TreeInspector_1cd1894e) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''The full Amazon Resource Name (ARN) for the virtual service.

        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshName")
    def attr_mesh_name(self) -> builtins.str:
        '''The name of the service mesh that the virtual service resides in.

        :cloudformationAttribute: MeshName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrMeshOwner")
    def attr_mesh_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then it's the ID of the account that shared the mesh with your account. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: MeshOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrMeshOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrResourceOwner")
    def attr_resource_owner(self) -> builtins.str:
        '''The AWS IAM account ID of the resource owner.

        If the account ID is not your own, then it's the ID of the mesh owner or of another account that the mesh is shared with. For more information about mesh sharing, see `Working with Shared Meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :cloudformationAttribute: ResourceOwner
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrResourceOwner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrUid")
    def attr_uid(self) -> builtins.str:
        '''The unique identifier for the virtual service.

        :cloudformationAttribute: Uid
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrUid"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVirtualServiceName")
    def attr_virtual_service_name(self) -> builtins.str:
        '''The name of the virtual service.

        :cloudformationAttribute: VirtualServiceName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVirtualServiceName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> _TagManager_0b7ab120:
        '''Optional metadata that you can apply to the virtual service to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-tags
        '''
        return typing.cast(_TagManager_0b7ab120, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshName")
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual service in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshname
        '''
        return typing.cast(builtins.str, jsii.get(self, "meshName"))

    @mesh_name.setter
    def mesh_name(self, value: builtins.str) -> None:
        jsii.set(self, "meshName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spec")
    def spec(
        self,
    ) -> typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef]:
        '''The virtual service specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-spec
        '''
        return typing.cast(typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef], jsii.get(self, "spec"))

    @spec.setter
    def spec(
        self,
        value: typing.Union["CfnVirtualService.VirtualServiceSpecProperty", _IResolvable_a771d0ef],
    ) -> None:
        jsii.set(self, "spec", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualServiceName")
    def virtual_service_name(self) -> builtins.str:
        '''The name to use for the virtual service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-virtualservicename
        '''
        return typing.cast(builtins.str, jsii.get(self, "virtualServiceName"))

    @virtual_service_name.setter
    def virtual_service_name(self, value: builtins.str) -> None:
        jsii.set(self, "virtualServiceName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="meshOwner")
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshowner
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "meshOwner"))

    @mesh_owner.setter
    def mesh_owner(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "meshOwner", value)

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_node_name": "virtualNodeName"},
    )
    class VirtualNodeServiceProviderProperty:
        def __init__(self, *, virtual_node_name: builtins.str) -> None:
            '''An object that represents a virtual node service provider.

            :param virtual_node_name: The name of the virtual node that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualnodeserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_node_service_provider_property = appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                    virtual_node_name="virtualNodeName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "virtual_node_name": virtual_node_name,
            }

        @builtins.property
        def virtual_node_name(self) -> builtins.str:
            '''The name of the virtual node that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualnodeserviceprovider.html#cfn-appmesh-virtualservice-virtualnodeserviceprovider-virtualnodename
            '''
            result = self._values.get("virtual_node_name")
            assert result is not None, "Required property 'virtual_node_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualNodeServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={"virtual_router_name": "virtualRouterName"},
    )
    class VirtualRouterServiceProviderProperty:
        def __init__(self, *, virtual_router_name: builtins.str) -> None:
            '''An object that represents a virtual node service provider.

            :param virtual_router_name: The name of the virtual router that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualrouterserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_router_service_provider_property = appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                    virtual_router_name="virtualRouterName"
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "virtual_router_name": virtual_router_name,
            }

        @builtins.property
        def virtual_router_name(self) -> builtins.str:
            '''The name of the virtual router that is acting as a service provider.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualrouterserviceprovider.html#cfn-appmesh-virtualservice-virtualrouterserviceprovider-virtualroutername
            '''
            result = self._values.get("virtual_router_name")
            assert result is not None, "Required property 'virtual_router_name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualRouterServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualServiceProviderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "virtual_node": "virtualNode",
            "virtual_router": "virtualRouter",
        },
    )
    class VirtualServiceProviderProperty:
        def __init__(
            self,
            *,
            virtual_node: typing.Optional[typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", _IResolvable_a771d0ef]] = None,
            virtual_router: typing.Optional[typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the provider for a virtual service.

            :param virtual_node: The virtual node associated with a virtual service.
            :param virtual_router: The virtual router associated with a virtual service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_service_provider_property = appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                    virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                        virtual_node_name="virtualNodeName"
                    ),
                    virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                        virtual_router_name="virtualRouterName"
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if virtual_node is not None:
                self._values["virtual_node"] = virtual_node
            if virtual_router is not None:
                self._values["virtual_router"] = virtual_router

        @builtins.property
        def virtual_node(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", _IResolvable_a771d0ef]]:
            '''The virtual node associated with a virtual service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html#cfn-appmesh-virtualservice-virtualserviceprovider-virtualnode
            '''
            result = self._values.get("virtual_node")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualService.VirtualNodeServiceProviderProperty", _IResolvable_a771d0ef]], result)

        @builtins.property
        def virtual_router(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", _IResolvable_a771d0ef]]:
            '''The virtual router associated with a virtual service.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualserviceprovider.html#cfn-appmesh-virtualservice-virtualserviceprovider-virtualrouter
            '''
            result = self._values.get("virtual_router")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualService.VirtualRouterServiceProviderProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceProviderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="monocdk.aws_appmesh.CfnVirtualService.VirtualServiceSpecProperty",
        jsii_struct_bases=[],
        name_mapping={"provider": "provider"},
    )
    class VirtualServiceSpecProperty:
        def __init__(
            self,
            *,
            provider: typing.Optional[typing.Union["CfnVirtualService.VirtualServiceProviderProperty", _IResolvable_a771d0ef]] = None,
        ) -> None:
            '''An object that represents the specification of a virtual service.

            :param provider: The App Mesh object that is acting as the provider for a virtual service. You can specify a single virtual node or virtual router.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualservicespec.html
            :exampleMetadata: fixture=_generated

            Example::

                # The code below shows an example of how to instantiate this type.
                # The values are placeholders you should change.
                from monocdk import aws_appmesh as appmesh
                
                virtual_service_spec_property = appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                    provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                        virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                            virtual_node_name="virtualNodeName"
                        ),
                        virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                            virtual_router_name="virtualRouterName"
                        )
                    )
                )
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if provider is not None:
                self._values["provider"] = provider

        @builtins.property
        def provider(
            self,
        ) -> typing.Optional[typing.Union["CfnVirtualService.VirtualServiceProviderProperty", _IResolvable_a771d0ef]]:
            '''The App Mesh object that is acting as the provider for a virtual service.

            You can specify a single virtual node or virtual router.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appmesh-virtualservice-virtualservicespec.html#cfn-appmesh-virtualservice-virtualservicespec-provider
            '''
            result = self._values.get("provider")
            return typing.cast(typing.Optional[typing.Union["CfnVirtualService.VirtualServiceProviderProperty", _IResolvable_a771d0ef]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VirtualServiceSpecProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CfnVirtualServiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "mesh_name": "meshName",
        "spec": "spec",
        "virtual_service_name": "virtualServiceName",
        "mesh_owner": "meshOwner",
        "tags": "tags",
    },
)
class CfnVirtualServiceProps:
    def __init__(
        self,
        *,
        mesh_name: builtins.str,
        spec: typing.Union[CfnVirtualService.VirtualServiceSpecProperty, _IResolvable_a771d0ef],
        virtual_service_name: builtins.str,
        mesh_owner: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[_CfnTag_95fbdc29]] = None,
    ) -> None:
        '''Properties for defining a ``CfnVirtualService``.

        :param mesh_name: The name of the service mesh to create the virtual service in.
        :param spec: The virtual service specification to apply.
        :param virtual_service_name: The name to use for the virtual service.
        :param mesh_owner: The AWS IAM account ID of the service mesh owner. If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .
        :param tags: Optional metadata that you can apply to the virtual service to assist with categorization and organization. Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            cfn_virtual_service_props = appmesh.CfnVirtualServiceProps(
                mesh_name="meshName",
                spec=appmesh.CfnVirtualService.VirtualServiceSpecProperty(
                    provider=appmesh.CfnVirtualService.VirtualServiceProviderProperty(
                        virtual_node=appmesh.CfnVirtualService.VirtualNodeServiceProviderProperty(
                            virtual_node_name="virtualNodeName"
                        ),
                        virtual_router=appmesh.CfnVirtualService.VirtualRouterServiceProviderProperty(
                            virtual_router_name="virtualRouterName"
                        )
                    )
                ),
                virtual_service_name="virtualServiceName",
            
                # the properties below are optional
                mesh_owner="meshOwner",
                tags=[CfnTag(
                    key="key",
                    value="value"
                )]
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "mesh_name": mesh_name,
            "spec": spec,
            "virtual_service_name": virtual_service_name,
        }
        if mesh_owner is not None:
            self._values["mesh_owner"] = mesh_owner
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def mesh_name(self) -> builtins.str:
        '''The name of the service mesh to create the virtual service in.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshname
        '''
        result = self._values.get("mesh_name")
        assert result is not None, "Required property 'mesh_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def spec(
        self,
    ) -> typing.Union[CfnVirtualService.VirtualServiceSpecProperty, _IResolvable_a771d0ef]:
        '''The virtual service specification to apply.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-spec
        '''
        result = self._values.get("spec")
        assert result is not None, "Required property 'spec' is missing"
        return typing.cast(typing.Union[CfnVirtualService.VirtualServiceSpecProperty, _IResolvable_a771d0ef], result)

    @builtins.property
    def virtual_service_name(self) -> builtins.str:
        '''The name to use for the virtual service.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-virtualservicename
        '''
        result = self._values.get("virtual_service_name")
        assert result is not None, "Required property 'virtual_service_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def mesh_owner(self) -> typing.Optional[builtins.str]:
        '''The AWS IAM account ID of the service mesh owner.

        If the account ID is not your own, then the account that you specify must share the mesh with your account before you can create the resource in the service mesh. For more information about mesh sharing, see `Working with shared meshes <https://docs.aws.amazon.com/app-mesh/latest/userguide/sharing.html>`_ .

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-meshowner
        '''
        result = self._values.get("mesh_owner")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[_CfnTag_95fbdc29]]:
        '''Optional metadata that you can apply to the virtual service to assist with categorization and organization.

        Each tag consists of a key and an optional value, both of which you define. Tag keys can have a maximum character length of 128 characters, and tag values can have a maximum length of 256 characters.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-appmesh-virtualservice.html#cfn-appmesh-virtualservice-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[_CfnTag_95fbdc29]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnVirtualServiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.CommonGatewayRouteSpecOptions",
    jsii_struct_bases=[],
    name_mapping={"priority": "priority"},
)
class CommonGatewayRouteSpecOptions:
    def __init__(self, *, priority: typing.Optional[jsii.Number] = None) -> None:
        '''(experimental) Base options for all gateway route specs.

        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            common_gateway_route_spec_options = appmesh.CommonGatewayRouteSpecOptions(
                priority=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority

        :stability: experimental
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommonGatewayRouteSpecOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_appmesh.DnsResponseType")
class DnsResponseType(enum.Enum):
    '''(experimental) Enum of DNS service discovery response type.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # A Virtual Node with a gRPC listener with a connection pool set
        # mesh: appmesh.Mesh
        
        node = appmesh.VirtualNode(self, "node",
            mesh=mesh,
            # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
            # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
            # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
            # By default, the response type is assumed to be LOAD_BALANCER
            service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
            listeners=[appmesh.VirtualNodeListener.http(
                port=80,
                connection_pool=cdk.aws_appmesh.HttpConnectionPool(
                    max_connections=100,
                    max_pending_requests=10
                )
            )]
        )
        
        # A Virtual Gateway with a gRPC listener with a connection pool set
        gateway = appmesh.VirtualGateway(self, "gateway",
            mesh=mesh,
            listeners=[appmesh.VirtualGatewayListener.grpc(
                port=8080,
                connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
                    max_requests=10
                )
            )],
            virtual_gateway_name="gateway"
        )
    '''

    LOAD_BALANCER = "LOAD_BALANCER"
    '''(experimental) DNS resolver returns a loadbalanced set of endpoints and the traffic would be sent to the given endpoints.

    It would not drain existing connections to other endpoints that are not part of this list.

    :stability: experimental
    '''
    ENDPOINTS = "ENDPOINTS"
    '''(experimental) DNS resolver is returning all the endpoints.

    This also means that if an endpoint is missing, it would drain the current connections to the missing endpoint.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteAttributes",
    jsii_struct_bases=[],
    name_mapping={
        "gateway_route_name": "gatewayRouteName",
        "virtual_gateway": "virtualGateway",
    },
)
class GatewayRouteAttributes:
    def __init__(
        self,
        *,
        gateway_route_name: builtins.str,
        virtual_gateway: "IVirtualGateway",
    ) -> None:
        '''(experimental) Interface with properties necessary to import a reusable GatewayRoute.

        :param gateway_route_name: (experimental) The name of the GatewayRoute.
        :param virtual_gateway: (experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            # virtual_gateway: appmesh.VirtualGateway
            
            gateway_route_attributes = appmesh.GatewayRouteAttributes(
                gateway_route_name="gatewayRouteName",
                virtual_gateway=virtual_gateway
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "gateway_route_name": gateway_route_name,
            "virtual_gateway": virtual_gateway,
        }

    @builtins.property
    def gateway_route_name(self) -> builtins.str:
        '''(experimental) The name of the GatewayRoute.

        :stability: experimental
        '''
        result = self._values.get("gateway_route_name")
        assert result is not None, "Required property 'gateway_route_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def virtual_gateway(self) -> "IVirtualGateway":
        '''(experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway")
        assert result is not None, "Required property 'virtual_gateway' is missing"
        return typing.cast("IVirtualGateway", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteAttributes(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteBaseProps",
    jsii_struct_bases=[],
    name_mapping={"route_spec": "routeSpec", "gateway_route_name": "gatewayRouteName"},
)
class GatewayRouteBaseProps:
    def __init__(
        self,
        *,
        route_spec: "GatewayRouteSpec",
        gateway_route_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) Basic configuration properties for a GatewayRoute.

        :param route_spec: (experimental) What protocol the route uses.
        :param gateway_route_name: (experimental) The name of the GatewayRoute. Default: - an automatically generated name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-grpc",
                route_spec=appmesh.GatewayRouteSpec.grpc(
                    route_target=virtual_service,
                    match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                        hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
                        # This disables the default rewrite to virtual service name and retain original request.
                        rewrite_request_hostname=False
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "route_spec": route_spec,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name

    @builtins.property
    def route_spec(self) -> "GatewayRouteSpec":
        '''(experimental) What protocol the route uses.

        :stability: experimental
        '''
        result = self._values.get("route_spec")
        assert result is not None, "Required property 'route_spec' is missing"
        return typing.cast("GatewayRouteSpec", result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the GatewayRoute.

        :default: - an automatically generated name

        :stability: experimental
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteBaseProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GatewayRouteHostnameMatch(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.GatewayRouteHostnameMatch",
):
    '''(experimental) Used to generate host name matching methods.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-grpc",
            route_spec=appmesh.GatewayRouteSpec.grpc(
                route_target=virtual_service,
                match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                    hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="endsWith") # type: ignore[misc]
    @builtins.classmethod
    def ends_with(cls, suffix: builtins.str) -> "GatewayRouteHostnameMatch":
        '''(experimental) The value of the host name with the given name must end with the specified characters.

        :param suffix: The specified ending characters of the host name to match on.

        :stability: experimental
        '''
        return typing.cast("GatewayRouteHostnameMatch", jsii.sinvoke(cls, "endsWith", [suffix]))

    @jsii.member(jsii_name="exactly") # type: ignore[misc]
    @builtins.classmethod
    def exactly(cls, name: builtins.str) -> "GatewayRouteHostnameMatch":
        '''(experimental) The value of the host name must match the specified value exactly.

        :param name: The exact host name to match on.

        :stability: experimental
        '''
        return typing.cast("GatewayRouteHostnameMatch", jsii.sinvoke(cls, "exactly", [name]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteHostnameMatchConfig":
        '''(experimental) Returns the gateway route host name match configuration.

        :param scope: -

        :stability: experimental
        '''
        ...


class _GatewayRouteHostnameMatchProxy(GatewayRouteHostnameMatch):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteHostnameMatchConfig":
        '''(experimental) Returns the gateway route host name match configuration.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast("GatewayRouteHostnameMatchConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GatewayRouteHostnameMatch).__jsii_proxy_class__ = lambda : _GatewayRouteHostnameMatchProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteHostnameMatchConfig",
    jsii_struct_bases=[],
    name_mapping={"hostname_match": "hostnameMatch"},
)
class GatewayRouteHostnameMatchConfig:
    def __init__(
        self,
        *,
        hostname_match: CfnGatewayRoute.GatewayRouteHostnameMatchProperty,
    ) -> None:
        '''(experimental) Configuration for gateway route host name match.

        :param hostname_match: (experimental) GatewayRoute CFN configuration for host name match.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            gateway_route_hostname_match_config = appmesh.GatewayRouteHostnameMatchConfig(
                hostname_match=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                    exact="exact",
                    suffix="suffix"
                )
            )
        '''
        if isinstance(hostname_match, dict):
            hostname_match = CfnGatewayRoute.GatewayRouteHostnameMatchProperty(**hostname_match)
        self._values: typing.Dict[str, typing.Any] = {
            "hostname_match": hostname_match,
        }

    @builtins.property
    def hostname_match(self) -> CfnGatewayRoute.GatewayRouteHostnameMatchProperty:
        '''(experimental) GatewayRoute CFN configuration for host name match.

        :stability: experimental
        '''
        result = self._values.get("hostname_match")
        assert result is not None, "Required property 'hostname_match' is missing"
        return typing.cast(CfnGatewayRoute.GatewayRouteHostnameMatchProperty, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteHostnameMatchConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteProps",
    jsii_struct_bases=[GatewayRouteBaseProps],
    name_mapping={
        "route_spec": "routeSpec",
        "gateway_route_name": "gatewayRouteName",
        "virtual_gateway": "virtualGateway",
    },
)
class GatewayRouteProps(GatewayRouteBaseProps):
    def __init__(
        self,
        *,
        route_spec: "GatewayRouteSpec",
        gateway_route_name: typing.Optional[builtins.str] = None,
        virtual_gateway: "IVirtualGateway",
    ) -> None:
        '''(experimental) Properties to define a new GatewayRoute.

        :param route_spec: (experimental) What protocol the route uses.
        :param gateway_route_name: (experimental) The name of the GatewayRoute. Default: - an automatically generated name
        :param virtual_gateway: (experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            # gateway_route_spec: appmesh.GatewayRouteSpec
            # virtual_gateway: appmesh.VirtualGateway
            
            gateway_route_props = appmesh.GatewayRouteProps(
                route_spec=gateway_route_spec,
                virtual_gateway=virtual_gateway,
            
                # the properties below are optional
                gateway_route_name="gatewayRouteName"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "route_spec": route_spec,
            "virtual_gateway": virtual_gateway,
        }
        if gateway_route_name is not None:
            self._values["gateway_route_name"] = gateway_route_name

    @builtins.property
    def route_spec(self) -> "GatewayRouteSpec":
        '''(experimental) What protocol the route uses.

        :stability: experimental
        '''
        result = self._values.get("route_spec")
        assert result is not None, "Required property 'route_spec' is missing"
        return typing.cast("GatewayRouteSpec", result)

    @builtins.property
    def gateway_route_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The name of the GatewayRoute.

        :default: - an automatically generated name

        :stability: experimental
        '''
        result = self._values.get("gateway_route_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def virtual_gateway(self) -> "IVirtualGateway":
        '''(experimental) The VirtualGateway this GatewayRoute is associated with.

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway")
        assert result is not None, "Required property 'virtual_gateway' is missing"
        return typing.cast("IVirtualGateway", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class GatewayRouteSpec(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.GatewayRouteSpec",
):
    '''(experimental) Used to generate specs with different protocols for a GatewayRoute.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-grpc",
            route_spec=appmesh.GatewayRouteSpec.grpc(
                route_target=virtual_service,
                match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                    hostname=appmesh.GatewayRouteHostnameMatch.exactly("example.com"),
                    # This disables the default rewrite to virtual service name and retain original request.
                    rewrite_request_hostname=False
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="grpc") # type: ignore[misc]
    @builtins.classmethod
    def grpc(
        cls,
        *,
        match: "GrpcGatewayRouteMatch",
        route_target: "IVirtualService",
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''(experimental) Creates an gRPC Based GatewayRoute.

        :param match: (experimental) The criterion for determining a request match for this GatewayRoute.
        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        '''
        options = GrpcGatewayRouteSpecOptions(
            match=match, route_target=route_target, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "grpc", [options]))

    @jsii.member(jsii_name="http") # type: ignore[misc]
    @builtins.classmethod
    def http(
        cls,
        *,
        route_target: "IVirtualService",
        match: typing.Optional["HttpGatewayRouteMatch"] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''(experimental) Creates an HTTP Based GatewayRoute.

        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param match: (experimental) The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        '''
        options = HttpGatewayRouteSpecOptions(
            route_target=route_target, match=match, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "http", [options]))

    @jsii.member(jsii_name="http2") # type: ignore[misc]
    @builtins.classmethod
    def http2(
        cls,
        *,
        route_target: "IVirtualService",
        match: typing.Optional["HttpGatewayRouteMatch"] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> "GatewayRouteSpec":
        '''(experimental) Creates an HTTP2 Based GatewayRoute.

        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param match: (experimental) The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        '''
        options = HttpGatewayRouteSpecOptions(
            route_target=route_target, match=match, priority=priority
        )

        return typing.cast("GatewayRouteSpec", jsii.sinvoke(cls, "http2", [options]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteSpecConfig":
        '''(experimental) Called when the GatewayRouteSpec type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        ...


class _GatewayRouteSpecProxy(GatewayRouteSpec):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "GatewayRouteSpecConfig":
        '''(experimental) Called when the GatewayRouteSpec type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -

        :stability: experimental
        '''
        return typing.cast("GatewayRouteSpecConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GatewayRouteSpec).__jsii_proxy_class__ = lambda : _GatewayRouteSpecProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GatewayRouteSpecConfig",
    jsii_struct_bases=[],
    name_mapping={
        "grpc_spec_config": "grpcSpecConfig",
        "http2_spec_config": "http2SpecConfig",
        "http_spec_config": "httpSpecConfig",
        "priority": "priority",
    },
)
class GatewayRouteSpecConfig:
    def __init__(
        self,
        *,
        grpc_spec_config: typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty] = None,
        http2_spec_config: typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty] = None,
        http_spec_config: typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty] = None,
        priority: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) All Properties for GatewayRoute Specs.

        :param grpc_spec_config: (experimental) The spec for a grpc gateway route. Default: - no grpc spec
        :param http2_spec_config: (experimental) The spec for an http2 gateway route. Default: - no http2 spec
        :param http_spec_config: (experimental) The spec for an http gateway route. Default: - no http spec
        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            gateway_route_spec_config = appmesh.GatewayRouteSpecConfig(
                grpc_spec_config=appmesh.CfnGatewayRoute.GrpcGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.GrpcGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.GrpcGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.GrpcGatewayRouteMatchProperty(
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        metadata=[appmesh.CfnGatewayRoute.GrpcGatewayRouteMetadataProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.GatewayRouteMetadataMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        service_name="serviceName"
                    )
                ),
                http2_spec_config=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
            
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                http_spec_config=appmesh.CfnGatewayRoute.HttpGatewayRouteProperty(
                    action=appmesh.CfnGatewayRoute.HttpGatewayRouteActionProperty(
                        target=appmesh.CfnGatewayRoute.GatewayRouteTargetProperty(
                            virtual_service=appmesh.CfnGatewayRoute.GatewayRouteVirtualServiceProperty(
                                virtual_service_name="virtualServiceName"
                            )
                        ),
            
                        # the properties below are optional
                        rewrite=appmesh.CfnGatewayRoute.HttpGatewayRouteRewriteProperty(
                            hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameRewriteProperty(
                                default_target_hostname="defaultTargetHostname"
                            ),
                            path=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                                exact="exact"
                            ),
                            prefix=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                                default_prefix="defaultPrefix",
                                value="value"
                            )
                        )
                    ),
                    match=appmesh.CfnGatewayRoute.HttpGatewayRouteMatchProperty(
                        headers=[appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderProperty(
                            name="name",
            
                            # the properties below are optional
                            invert=False,
                            match=appmesh.CfnGatewayRoute.HttpGatewayRouteHeaderMatchProperty(
                                exact="exact",
                                prefix="prefix",
                                range=appmesh.CfnGatewayRoute.GatewayRouteRangeMatchProperty(
                                    end=123,
                                    start=123
                                ),
                                regex="regex",
                                suffix="suffix"
                            )
                        )],
                        hostname=appmesh.CfnGatewayRoute.GatewayRouteHostnameMatchProperty(
                            exact="exact",
                            suffix="suffix"
                        ),
                        method="method",
                        path=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                            exact="exact",
                            regex="regex"
                        ),
                        prefix="prefix",
                        query_parameters=[appmesh.CfnGatewayRoute.QueryParameterProperty(
                            name="name",
            
                            # the properties below are optional
                            match=appmesh.CfnGatewayRoute.HttpQueryParameterMatchProperty(
                                exact="exact"
                            )
                        )]
                    )
                ),
                priority=123
            )
        '''
        if isinstance(grpc_spec_config, dict):
            grpc_spec_config = CfnGatewayRoute.GrpcGatewayRouteProperty(**grpc_spec_config)
        if isinstance(http2_spec_config, dict):
            http2_spec_config = CfnGatewayRoute.HttpGatewayRouteProperty(**http2_spec_config)
        if isinstance(http_spec_config, dict):
            http_spec_config = CfnGatewayRoute.HttpGatewayRouteProperty(**http_spec_config)
        self._values: typing.Dict[str, typing.Any] = {}
        if grpc_spec_config is not None:
            self._values["grpc_spec_config"] = grpc_spec_config
        if http2_spec_config is not None:
            self._values["http2_spec_config"] = http2_spec_config
        if http_spec_config is not None:
            self._values["http_spec_config"] = http_spec_config
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def grpc_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty]:
        '''(experimental) The spec for a grpc gateway route.

        :default: - no grpc spec

        :stability: experimental
        '''
        result = self._values.get("grpc_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.GrpcGatewayRouteProperty], result)

    @builtins.property
    def http2_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty]:
        '''(experimental) The spec for an http2 gateway route.

        :default: - no http2 spec

        :stability: experimental
        '''
        result = self._values.get("http2_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty], result)

    @builtins.property
    def http_spec_config(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty]:
        '''(experimental) The spec for an http gateway route.

        :default: - no http spec

        :stability: experimental
        '''
        result = self._values.get("http_spec_config")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRouteProperty], result)

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority

        :stability: experimental
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GatewayRouteSpecConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcConnectionPool",
    jsii_struct_bases=[],
    name_mapping={"max_requests": "maxRequests"},
)
class GrpcConnectionPool:
    def __init__(self, *, max_requests: jsii.Number) -> None:
        '''(experimental) Connection pool properties for gRPC listeners.

        :param max_requests: (experimental) The maximum requests in the pool. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with a gRPC listener with a connection pool set
            # mesh: appmesh.Mesh
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
                # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
                # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
                # By default, the response type is assumed to be LOAD_BALANCER
                service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=80,
                    connection_pool=cdk.aws_appmesh.HttpConnectionPool(
                        max_connections=100,
                        max_pending_requests=10
                    )
                )]
            )
            
            # A Virtual Gateway with a gRPC listener with a connection pool set
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
                        max_requests=10
                    )
                )],
                virtual_gateway_name="gateway"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "max_requests": max_requests,
        }

    @builtins.property
    def max_requests(self) -> jsii.Number:
        '''(experimental) The maximum requests in the pool.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("max_requests")
        assert result is not None, "Required property 'max_requests' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcConnectionPool(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcGatewayListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "port": "port",
        "tls": "tls",
    },
)
class GrpcGatewayListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[GrpcConnectionPool] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        port: typing.Optional[jsii.Number] = None,
        tls: typing.Optional["ListenerTlsOptions"] = None,
    ) -> None:
        '''(experimental) Represents the properties needed to define GRPC Listeners for a VirtualGateway.

        :param connection_pool: (experimental) Connection pool for http listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param tls: (experimental) Represents the configuration for enabling TLS on a listener. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with listener TLS from an ACM provided certificate
            # cert: certificatemanager.Certificate
            # mesh: appmesh.Mesh
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.dns("node"),
                listeners=[appmesh.VirtualNodeListener.grpc(
                    port=80,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.acm(cert)
                    )
                )]
            )
            
            # A Virtual Gateway with listener TLS from a customer provided file certificate
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
                    )
                )],
                virtual_gateway_name="gateway"
            )
            
            # A Virtual Gateway with listener TLS from a SDS provided certificate
            gateway2 = appmesh.VirtualGateway(self, "gateway2",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.http2(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.sds("secrete_certificate")
                    )
                )],
                virtual_gateway_name="gateway2"
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = GrpcConnectionPool(**connection_pool)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        self._values: typing.Dict[str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if port is not None:
            self._values["port"] = port
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[GrpcConnectionPool]:
        '''(experimental) Connection pool for http listeners.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[GrpcConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check information for the listener.

        :default: - no healthcheck

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port to listen for connections on.

        :default: - 8080

        :stability: experimental
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''(experimental) Represents the configuration for enabling TLS on a listener.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls")
        return typing.cast(typing.Optional["ListenerTlsOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcGatewayListenerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcGatewayRouteMatch",
    jsii_struct_bases=[],
    name_mapping={
        "hostname": "hostname",
        "metadata": "metadata",
        "rewrite_request_hostname": "rewriteRequestHostname",
        "service_name": "serviceName",
    },
)
class GrpcGatewayRouteMatch:
    def __init__(
        self,
        *,
        hostname: typing.Optional[GatewayRouteHostnameMatch] = None,
        metadata: typing.Optional[typing.Sequence["HeaderMatch"]] = None,
        rewrite_request_hostname: typing.Optional[builtins.bool] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The criterion for determining a request match for this GatewayRoute.

        :param hostname: (experimental) Create host name based gRPC gateway route match. Default: - no matching on host name
        :param metadata: (experimental) Create metadata based gRPC gateway route match. All specified metadata must match for the route to match. Default: - no matching on metadata
        :param rewrite_request_hostname: (experimental) When ``true``, rewrites the original request received at the Virtual Gateway to the destination Virtual Service name. When ``false``, retains the original hostname from the request. Default: true
        :param service_name: (experimental) Create service name based gRPC gateway route match. Default: - no matching on service name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-grpc",
                route_spec=appmesh.GatewayRouteSpec.grpc(
                    route_target=virtual_service,
                    match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                        hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if hostname is not None:
            self._values["hostname"] = hostname
        if metadata is not None:
            self._values["metadata"] = metadata
        if rewrite_request_hostname is not None:
            self._values["rewrite_request_hostname"] = rewrite_request_hostname
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def hostname(self) -> typing.Optional[GatewayRouteHostnameMatch]:
        '''(experimental) Create host name based gRPC gateway route match.

        :default: - no matching on host name

        :stability: experimental
        '''
        result = self._values.get("hostname")
        return typing.cast(typing.Optional[GatewayRouteHostnameMatch], result)

    @builtins.property
    def metadata(self) -> typing.Optional[typing.List["HeaderMatch"]]:
        '''(experimental) Create metadata based gRPC gateway route match.

        All specified metadata must match for the route to match.

        :default: - no matching on metadata

        :stability: experimental
        '''
        result = self._values.get("metadata")
        return typing.cast(typing.Optional[typing.List["HeaderMatch"]], result)

    @builtins.property
    def rewrite_request_hostname(self) -> typing.Optional[builtins.bool]:
        '''(experimental) When ``true``, rewrites the original request received at the Virtual Gateway to the destination Virtual Service name.

        When ``false``, retains the original hostname from the request.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("rewrite_request_hostname")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Create service name based gRPC gateway route match.

        :default: - no matching on service name

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcGatewayRouteMatch(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcGatewayRouteSpecOptions",
    jsii_struct_bases=[CommonGatewayRouteSpecOptions],
    name_mapping={
        "priority": "priority",
        "match": "match",
        "route_target": "routeTarget",
    },
)
class GrpcGatewayRouteSpecOptions(CommonGatewayRouteSpecOptions):
    def __init__(
        self,
        *,
        priority: typing.Optional[jsii.Number] = None,
        match: GrpcGatewayRouteMatch,
        route_target: "IVirtualService",
    ) -> None:
        '''(experimental) Properties specific for a gRPC GatewayRoute.

        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority
        :param match: (experimental) The criterion for determining a request match for this GatewayRoute.
        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-grpc",
                route_spec=appmesh.GatewayRouteSpec.grpc(
                    route_target=virtual_service,
                    match=cdk.aws_appmesh.GrpcGatewayRouteMatch(
                        hostname=appmesh.GatewayRouteHostnameMatch.ends_with(".example.com")
                    )
                )
            )
        '''
        if isinstance(match, dict):
            match = GrpcGatewayRouteMatch(**match)
        self._values: typing.Dict[str, typing.Any] = {
            "match": match,
            "route_target": route_target,
        }
        if priority is not None:
            self._values["priority"] = priority

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority

        :stability: experimental
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def match(self) -> GrpcGatewayRouteMatch:
        '''(experimental) The criterion for determining a request match for this GatewayRoute.

        :stability: experimental
        '''
        result = self._values.get("match")
        assert result is not None, "Required property 'match' is missing"
        return typing.cast(GrpcGatewayRouteMatch, result)

    @builtins.property
    def route_target(self) -> "IVirtualService":
        '''(experimental) The VirtualService this GatewayRoute directs traffic to.

        :stability: experimental
        '''
        result = self._values.get("route_target")
        assert result is not None, "Required property 'route_target' is missing"
        return typing.cast("IVirtualService", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcGatewayRouteSpecOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcHealthCheckOptions",
    jsii_struct_bases=[],
    name_mapping={
        "healthy_threshold": "healthyThreshold",
        "interval": "interval",
        "timeout": "timeout",
        "unhealthy_threshold": "unhealthyThreshold",
    },
)
class GrpcHealthCheckOptions:
    def __init__(
        self,
        *,
        healthy_threshold: typing.Optional[jsii.Number] = None,
        interval: typing.Optional[_Duration_070aa057] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        unhealthy_threshold: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Properties used to define GRPC Based healthchecks.

        :param healthy_threshold: (experimental) The number of consecutive successful health checks that must occur before declaring listener healthy. Default: 2
        :param interval: (experimental) The time period between each health check execution. Default: Duration.seconds(5)
        :param timeout: (experimental) The amount of time to wait when receiving a response from the health check. Default: Duration.seconds(2)
        :param unhealthy_threshold: (experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy. Default: - 2

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_appmesh as appmesh
            
            # duration: monocdk.Duration
            
            grpc_health_check_options = appmesh.GrpcHealthCheckOptions(
                healthy_threshold=123,
                interval=duration,
                timeout=duration,
                unhealthy_threshold=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if healthy_threshold is not None:
            self._values["healthy_threshold"] = healthy_threshold
        if interval is not None:
            self._values["interval"] = interval
        if timeout is not None:
            self._values["timeout"] = timeout
        if unhealthy_threshold is not None:
            self._values["unhealthy_threshold"] = unhealthy_threshold

    @builtins.property
    def healthy_threshold(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of consecutive successful health checks that must occur before declaring listener healthy.

        :default: 2

        :stability: experimental
        '''
        result = self._values.get("healthy_threshold")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def interval(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The time period between each health check execution.

        :default: Duration.seconds(5)

        :stability: experimental
        '''
        result = self._values.get("interval")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The amount of time to wait when receiving a response from the health check.

        :default: Duration.seconds(2)

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def unhealthy_threshold(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy.

        :default: - 2

        :stability: experimental
        '''
        result = self._values.get("unhealthy_threshold")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcHealthCheckOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_appmesh.GrpcRetryEvent")
class GrpcRetryEvent(enum.Enum):
    '''(experimental) gRPC events.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # router: appmesh.VirtualRouter
        # node: appmesh.VirtualNode
        
        
        router.add_route("route-grpc-retry",
            route_spec=appmesh.RouteSpec.grpc(
                weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
                match=cdk.aws_appmesh.GrpcRouteMatch(service_name="servicename"),
                retry_policy=cdk.aws_appmesh.GrpcRetryPolicy(
                    tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
                    http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
                    # Retry if gRPC responds that the request was cancelled, a resource
                    # was exhausted, or if the service is unavailable
                    grpc_retry_events=[appmesh.GrpcRetryEvent.CANCELLED, appmesh.GrpcRetryEvent.RESOURCE_EXHAUSTED, appmesh.GrpcRetryEvent.UNAVAILABLE
                    ],
                    retry_attempts=5,
                    retry_timeout=cdk.Duration.seconds(1)
                )
            )
        )
    '''

    CANCELLED = "CANCELLED"
    '''(experimental) Request was cancelled.

    :see: https://grpc.github.io/grpc/core/md_doc_statuscodes.html
    :stability: experimental
    '''
    DEADLINE_EXCEEDED = "DEADLINE_EXCEEDED"
    '''(experimental) The deadline was exceeded.

    :see: https://grpc.github.io/grpc/core/md_doc_statuscodes.html
    :stability: experimental
    '''
    INTERNAL_ERROR = "INTERNAL_ERROR"
    '''(experimental) Internal error.

    :see: https://grpc.github.io/grpc/core/md_doc_statuscodes.html
    :stability: experimental
    '''
    RESOURCE_EXHAUSTED = "RESOURCE_EXHAUSTED"
    '''(experimental) A resource was exhausted.

    :see: https://grpc.github.io/grpc/core/md_doc_statuscodes.html
    :stability: experimental
    '''
    UNAVAILABLE = "UNAVAILABLE"
    '''(experimental) The service is unavailable.

    :see: https://grpc.github.io/grpc/core/md_doc_statuscodes.html
    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcRouteMatch",
    jsii_struct_bases=[],
    name_mapping={
        "metadata": "metadata",
        "method_name": "methodName",
        "service_name": "serviceName",
    },
)
class GrpcRouteMatch:
    def __init__(
        self,
        *,
        metadata: typing.Optional[typing.Sequence["HeaderMatch"]] = None,
        method_name: typing.Optional[builtins.str] = None,
        service_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''(experimental) The criterion for determining a request match for this Route.

        At least one match type must be selected.

        :param metadata: (experimental) Create metadata based gRPC route match. All specified metadata must match for the route to match. Default: - do not match on metadata
        :param method_name: (experimental) The method name to match from the request. If the method name is specified, service name must be also provided. Default: - do not match on method name
        :param service_name: (experimental) Create service name based gRPC route match. Default: - do not match on service name

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # router: appmesh.VirtualRouter
            # node: appmesh.VirtualNode
            
            
            router.add_route("route-http",
                route_spec=appmesh.RouteSpec.grpc(
                    weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                        virtual_node=node
                    )
                    ],
                    match=cdk.aws_appmesh.GrpcRouteMatch(
                        service_name="my-service.default.svc.cluster.local"
                    ),
                    timeout=cdk.aws_appmesh.GrpcTimeout(
                        idle=cdk.Duration.seconds(2),
                        per_request=cdk.Duration.seconds(1)
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if metadata is not None:
            self._values["metadata"] = metadata
        if method_name is not None:
            self._values["method_name"] = method_name
        if service_name is not None:
            self._values["service_name"] = service_name

    @builtins.property
    def metadata(self) -> typing.Optional[typing.List["HeaderMatch"]]:
        '''(experimental) Create metadata based gRPC route match.

        All specified metadata must match for the route to match.

        :default: - do not match on metadata

        :stability: experimental
        '''
        result = self._values.get("metadata")
        return typing.cast(typing.Optional[typing.List["HeaderMatch"]], result)

    @builtins.property
    def method_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) The method name to match from the request.

        If the method name is specified, service name must be also provided.

        :default: - do not match on method name

        :stability: experimental
        '''
        result = self._values.get("method_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def service_name(self) -> typing.Optional[builtins.str]:
        '''(experimental) Create service name based gRPC route match.

        :default: - do not match on service name

        :stability: experimental
        '''
        result = self._values.get("service_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcRouteMatch(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcTimeout",
    jsii_struct_bases=[],
    name_mapping={"idle": "idle", "per_request": "perRequest"},
)
class GrpcTimeout:
    def __init__(
        self,
        *,
        idle: typing.Optional[_Duration_070aa057] = None,
        per_request: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Represents timeouts for GRPC protocols.

        :param idle: (experimental) Represents an idle timeout. The amount of time that a connection may be idle. Default: - none
        :param per_request: (experimental) Represents per request timeout. Default: - 15 s

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # router: appmesh.VirtualRouter
            # node: appmesh.VirtualNode
            
            
            router.add_route("route-http",
                route_spec=appmesh.RouteSpec.grpc(
                    weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                        virtual_node=node
                    )
                    ],
                    match=cdk.aws_appmesh.GrpcRouteMatch(
                        service_name="my-service.default.svc.cluster.local"
                    ),
                    timeout=cdk.aws_appmesh.GrpcTimeout(
                        idle=cdk.Duration.seconds(2),
                        per_request=cdk.Duration.seconds(1)
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if idle is not None:
            self._values["idle"] = idle
        if per_request is not None:
            self._values["per_request"] = per_request

    @builtins.property
    def idle(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Represents an idle timeout.

        The amount of time that a connection may be idle.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("idle")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def per_request(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Represents per request timeout.

        :default: - 15 s

        :stability: experimental
        '''
        result = self._values.get("per_request")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcTimeout(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.GrpcVirtualNodeListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "outlier_detection": "outlierDetection",
        "port": "port",
        "timeout": "timeout",
        "tls": "tls",
    },
)
class GrpcVirtualNodeListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[GrpcConnectionPool] = None,
        health_check: typing.Optional["HealthCheck"] = None,
        outlier_detection: typing.Optional["OutlierDetection"] = None,
        port: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[GrpcTimeout] = None,
        tls: typing.Optional["ListenerTlsOptions"] = None,
    ) -> None:
        '''(experimental) Represent the GRPC Node Listener prorperty.

        :param connection_pool: (experimental) Connection pool for http listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param outlier_detection: (experimental) Represents the configuration for enabling outlier detection. Default: - none
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param timeout: (experimental) Timeout for GRPC protocol. Default: - None
        :param tls: (experimental) Represents the configuration for enabling TLS on a listener. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with listener TLS from an ACM provided certificate
            # cert: certificatemanager.Certificate
            # mesh: appmesh.Mesh
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.dns("node"),
                listeners=[appmesh.VirtualNodeListener.grpc(
                    port=80,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.acm(cert)
                    )
                )]
            )
            
            # A Virtual Gateway with listener TLS from a customer provided file certificate
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
                    )
                )],
                virtual_gateway_name="gateway"
            )
            
            # A Virtual Gateway with listener TLS from a SDS provided certificate
            gateway2 = appmesh.VirtualGateway(self, "gateway2",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.http2(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.sds("secrete_certificate")
                    )
                )],
                virtual_gateway_name="gateway2"
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = GrpcConnectionPool(**connection_pool)
        if isinstance(outlier_detection, dict):
            outlier_detection = OutlierDetection(**outlier_detection)
        if isinstance(timeout, dict):
            timeout = GrpcTimeout(**timeout)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        self._values: typing.Dict[str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if outlier_detection is not None:
            self._values["outlier_detection"] = outlier_detection
        if port is not None:
            self._values["port"] = port
        if timeout is not None:
            self._values["timeout"] = timeout
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[GrpcConnectionPool]:
        '''(experimental) Connection pool for http listeners.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[GrpcConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional["HealthCheck"]:
        '''(experimental) The health check information for the listener.

        :default: - no healthcheck

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional["HealthCheck"], result)

    @builtins.property
    def outlier_detection(self) -> typing.Optional["OutlierDetection"]:
        '''(experimental) Represents the configuration for enabling outlier detection.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("outlier_detection")
        return typing.cast(typing.Optional["OutlierDetection"], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port to listen for connections on.

        :default: - 8080

        :stability: experimental
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def timeout(self) -> typing.Optional[GrpcTimeout]:
        '''(experimental) Timeout for GRPC protocol.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[GrpcTimeout], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''(experimental) Represents the configuration for enabling TLS on a listener.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls")
        return typing.cast(typing.Optional["ListenerTlsOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GrpcVirtualNodeListenerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HeaderMatch(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.HeaderMatch",
):
    '''(experimental) Used to generate header matching methods.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # router: appmesh.VirtualRouter
        # node: appmesh.VirtualNode
        
        
        router.add_route("route-http2",
            route_spec=appmesh.RouteSpec.http2(
                weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                    virtual_node=node
                )
                ],
                match=cdk.aws_appmesh.HttpRouteMatch(
                    path=appmesh.HttpRoutePathMatch.exactly("/exact"),
                    method=appmesh.HttpRouteMethod.POST,
                    protocol=appmesh.HttpRouteProtocol.HTTPS,
                    headers=[
                        # All specified headers must match for the route to match.
                        appmesh.HeaderMatch.value_is("Content-Type", "application/json"),
                        appmesh.HeaderMatch.value_is_not("Content-Type", "application/json")
                    ],
                    query_parameters=[
                        # All specified query parameters must match for the route to match.
                        appmesh.QueryParameterMatch.value_is("query-field", "value")
                    ]
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="valueDoesNotEndWith") # type: ignore[misc]
    @builtins.classmethod
    def value_does_not_end_with(
        cls,
        header_name: builtins.str,
        suffix: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must not end with the specified characters.

        :param header_name: the name of the header to match against.
        :param suffix: The suffix to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueDoesNotEndWith", [header_name, suffix]))

    @jsii.member(jsii_name="valueDoesNotMatchRegex") # type: ignore[misc]
    @builtins.classmethod
    def value_does_not_match_regex(
        cls,
        header_name: builtins.str,
        regex: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must not include the specified characters.

        :param header_name: the name of the header to match against.
        :param regex: The regex to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueDoesNotMatchRegex", [header_name, regex]))

    @jsii.member(jsii_name="valueDoesNotStartWith") # type: ignore[misc]
    @builtins.classmethod
    def value_does_not_start_with(
        cls,
        header_name: builtins.str,
        prefix: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must not start with the specified characters.

        :param header_name: the name of the header to match against.
        :param prefix: The prefix to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueDoesNotStartWith", [header_name, prefix]))

    @jsii.member(jsii_name="valueEndsWith") # type: ignore[misc]
    @builtins.classmethod
    def value_ends_with(
        cls,
        header_name: builtins.str,
        suffix: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must end with the specified characters.

        :param header_name: the name of the header to match against.
        :param suffix: The suffix to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueEndsWith", [header_name, suffix]))

    @jsii.member(jsii_name="valueIs") # type: ignore[misc]
    @builtins.classmethod
    def value_is(
        cls,
        header_name: builtins.str,
        header_value: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must match the specified value exactly.

        :param header_name: the name of the header to match against.
        :param header_value: The exact value to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueIs", [header_name, header_value]))

    @jsii.member(jsii_name="valueIsNot") # type: ignore[misc]
    @builtins.classmethod
    def value_is_not(
        cls,
        header_name: builtins.str,
        header_value: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must not match the specified value exactly.

        :param header_name: the name of the header to match against.
        :param header_value: The exact value to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueIsNot", [header_name, header_value]))

    @jsii.member(jsii_name="valueMatchesRegex") # type: ignore[misc]
    @builtins.classmethod
    def value_matches_regex(
        cls,
        header_name: builtins.str,
        regex: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must include the specified characters.

        :param header_name: the name of the header to match against.
        :param regex: The regex to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueMatchesRegex", [header_name, regex]))

    @jsii.member(jsii_name="valuesIsInRange") # type: ignore[misc]
    @builtins.classmethod
    def values_is_in_range(
        cls,
        header_name: builtins.str,
        start: jsii.Number,
        end: jsii.Number,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must be in a range of values.

        :param header_name: the name of the header to match against.
        :param start: Match on values starting at and including this value.
        :param end: Match on values up to but not including this value.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valuesIsInRange", [header_name, start, end]))

    @jsii.member(jsii_name="valuesIsNotInRange") # type: ignore[misc]
    @builtins.classmethod
    def values_is_not_in_range(
        cls,
        header_name: builtins.str,
        start: jsii.Number,
        end: jsii.Number,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must not be in a range of values.

        :param header_name: the name of the header to match against.
        :param start: Match on values starting at and including this value.
        :param end: Match on values up to but not including this value.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valuesIsNotInRange", [header_name, start, end]))

    @jsii.member(jsii_name="valueStartsWith") # type: ignore[misc]
    @builtins.classmethod
    def value_starts_with(
        cls,
        header_name: builtins.str,
        prefix: builtins.str,
    ) -> "HeaderMatch":
        '''(experimental) The value of the header with the given name in the request must start with the specified characters.

        :param header_name: the name of the header to match against.
        :param prefix: The prefix to test against.

        :stability: experimental
        '''
        return typing.cast("HeaderMatch", jsii.sinvoke(cls, "valueStartsWith", [header_name, prefix]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "HeaderMatchConfig":
        '''(experimental) Returns the header match configuration.

        :param scope: -

        :stability: experimental
        '''
        ...


class _HeaderMatchProxy(HeaderMatch):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "HeaderMatchConfig":
        '''(experimental) Returns the header match configuration.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast("HeaderMatchConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, HeaderMatch).__jsii_proxy_class__ = lambda : _HeaderMatchProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HeaderMatchConfig",
    jsii_struct_bases=[],
    name_mapping={"header_match": "headerMatch"},
)
class HeaderMatchConfig:
    def __init__(self, *, header_match: CfnRoute.HttpRouteHeaderProperty) -> None:
        '''(experimental) Configuration for ``HeaderMatch``.

        :param header_match: (experimental) Route CFN configuration for the route header match.

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            header_match_config = appmesh.HeaderMatchConfig(
                header_match=appmesh.CfnRoute.HttpRouteHeaderProperty(
                    name="name",
            
                    # the properties below are optional
                    invert=False,
                    match=appmesh.CfnRoute.HeaderMatchMethodProperty(
                        exact="exact",
                        prefix="prefix",
                        range=appmesh.CfnRoute.MatchRangeProperty(
                            end=123,
                            start=123
                        ),
                        regex="regex",
                        suffix="suffix"
                    )
                )
            )
        '''
        if isinstance(header_match, dict):
            header_match = CfnRoute.HttpRouteHeaderProperty(**header_match)
        self._values: typing.Dict[str, typing.Any] = {
            "header_match": header_match,
        }

    @builtins.property
    def header_match(self) -> CfnRoute.HttpRouteHeaderProperty:
        '''(experimental) Route CFN configuration for the route header match.

        :stability: experimental
        '''
        result = self._values.get("header_match")
        assert result is not None, "Required property 'header_match' is missing"
        return typing.cast(CfnRoute.HttpRouteHeaderProperty, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HeaderMatchConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HealthCheck(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.HealthCheck",
):
    '''(experimental) Contains static factory methods for creating health checks for different protocols.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # mesh: appmesh.Mesh
        vpc = ec2.Vpc(self, "vpc")
        namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
            vpc=vpc,
            name="domain.local"
        )
        service = namespace.create_service("Svc")
        node = mesh.add_virtual_node("virtual-node",
            service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
            listeners=[appmesh.VirtualNodeListener.http(
                port=8081,
                health_check=appmesh.HealthCheck.http(
                    healthy_threshold=3,
                    interval=cdk.Duration.seconds(5),  # minimum
                    path="/health-check-path",
                    timeout=cdk.Duration.seconds(2),  # minimum
                    unhealthy_threshold=2
                )
            )],
            access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="grpc") # type: ignore[misc]
    @builtins.classmethod
    def grpc(
        cls,
        *,
        healthy_threshold: typing.Optional[jsii.Number] = None,
        interval: typing.Optional[_Duration_070aa057] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        unhealthy_threshold: typing.Optional[jsii.Number] = None,
    ) -> "HealthCheck":
        '''(experimental) Construct a GRPC health check.

        :param healthy_threshold: (experimental) The number of consecutive successful health checks that must occur before declaring listener healthy. Default: 2
        :param interval: (experimental) The time period between each health check execution. Default: Duration.seconds(5)
        :param timeout: (experimental) The amount of time to wait when receiving a response from the health check. Default: Duration.seconds(2)
        :param unhealthy_threshold: (experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy. Default: - 2

        :stability: experimental
        '''
        options = GrpcHealthCheckOptions(
            healthy_threshold=healthy_threshold,
            interval=interval,
            timeout=timeout,
            unhealthy_threshold=unhealthy_threshold,
        )

        return typing.cast("HealthCheck", jsii.sinvoke(cls, "grpc", [options]))

    @jsii.member(jsii_name="http") # type: ignore[misc]
    @builtins.classmethod
    def http(
        cls,
        *,
        healthy_threshold: typing.Optional[jsii.Number] = None,
        interval: typing.Optional[_Duration_070aa057] = None,
        path: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        unhealthy_threshold: typing.Optional[jsii.Number] = None,
    ) -> "HealthCheck":
        '''(experimental) Construct a HTTP health check.

        :param healthy_threshold: (experimental) The number of consecutive successful health checks that must occur before declaring listener healthy. Default: 2
        :param interval: (experimental) The time period between each health check execution. Default: Duration.seconds(5)
        :param path: (experimental) The destination path for the health check request. Default: /
        :param timeout: (experimental) The amount of time to wait when receiving a response from the health check. Default: Duration.seconds(2)
        :param unhealthy_threshold: (experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy. Default: - 2

        :stability: experimental
        '''
        options = HttpHealthCheckOptions(
            healthy_threshold=healthy_threshold,
            interval=interval,
            path=path,
            timeout=timeout,
            unhealthy_threshold=unhealthy_threshold,
        )

        return typing.cast("HealthCheck", jsii.sinvoke(cls, "http", [options]))

    @jsii.member(jsii_name="http2") # type: ignore[misc]
    @builtins.classmethod
    def http2(
        cls,
        *,
        healthy_threshold: typing.Optional[jsii.Number] = None,
        interval: typing.Optional[_Duration_070aa057] = None,
        path: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        unhealthy_threshold: typing.Optional[jsii.Number] = None,
    ) -> "HealthCheck":
        '''(experimental) Construct a HTTP2 health check.

        :param healthy_threshold: (experimental) The number of consecutive successful health checks that must occur before declaring listener healthy. Default: 2
        :param interval: (experimental) The time period between each health check execution. Default: Duration.seconds(5)
        :param path: (experimental) The destination path for the health check request. Default: /
        :param timeout: (experimental) The amount of time to wait when receiving a response from the health check. Default: Duration.seconds(2)
        :param unhealthy_threshold: (experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy. Default: - 2

        :stability: experimental
        '''
        options = HttpHealthCheckOptions(
            healthy_threshold=healthy_threshold,
            interval=interval,
            path=path,
            timeout=timeout,
            unhealthy_threshold=unhealthy_threshold,
        )

        return typing.cast("HealthCheck", jsii.sinvoke(cls, "http2", [options]))

    @jsii.member(jsii_name="tcp") # type: ignore[misc]
    @builtins.classmethod
    def tcp(
        cls,
        *,
        healthy_threshold: typing.Optional[jsii.Number] = None,
        interval: typing.Optional[_Duration_070aa057] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        unhealthy_threshold: typing.Optional[jsii.Number] = None,
    ) -> "HealthCheck":
        '''(experimental) Construct a TCP health check.

        :param healthy_threshold: (experimental) The number of consecutive successful health checks that must occur before declaring listener healthy. Default: 2
        :param interval: (experimental) The time period between each health check execution. Default: Duration.seconds(5)
        :param timeout: (experimental) The amount of time to wait when receiving a response from the health check. Default: Duration.seconds(2)
        :param unhealthy_threshold: (experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy. Default: - 2

        :stability: experimental
        '''
        options = TcpHealthCheckOptions(
            healthy_threshold=healthy_threshold,
            interval=interval,
            timeout=timeout,
            unhealthy_threshold=unhealthy_threshold,
        )

        return typing.cast("HealthCheck", jsii.sinvoke(cls, "tcp", [options]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(
        self,
        scope: _Construct_e78e779f,
        *,
        default_port: typing.Optional[jsii.Number] = None,
    ) -> "HealthCheckConfig":
        '''(experimental) Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -
        :param default_port: (experimental) Port for Health Check interface. Default: - no default port is provided

        :stability: experimental
        '''
        ...


class _HealthCheckProxy(HealthCheck):
    @jsii.member(jsii_name="bind")
    def bind(
        self,
        scope: _Construct_e78e779f,
        *,
        default_port: typing.Optional[jsii.Number] = None,
    ) -> "HealthCheckConfig":
        '''(experimental) Called when the AccessLog type is initialized.

        Can be used to enforce
        mutual exclusivity with future properties

        :param scope: -
        :param default_port: (experimental) Port for Health Check interface. Default: - no default port is provided

        :stability: experimental
        '''
        options = HealthCheckBindOptions(default_port=default_port)

        return typing.cast("HealthCheckConfig", jsii.invoke(self, "bind", [scope, options]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, HealthCheck).__jsii_proxy_class__ = lambda : _HealthCheckProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HealthCheckBindOptions",
    jsii_struct_bases=[],
    name_mapping={"default_port": "defaultPort"},
)
class HealthCheckBindOptions:
    def __init__(self, *, default_port: typing.Optional[jsii.Number] = None) -> None:
        '''(experimental) Options used for creating the Health Check object.

        :param default_port: (experimental) Port for Health Check interface. Default: - no default port is provided

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            health_check_bind_options = appmesh.HealthCheckBindOptions(
                default_port=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if default_port is not None:
            self._values["default_port"] = default_port

    @builtins.property
    def default_port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port for Health Check interface.

        :default: - no default port is provided

        :stability: experimental
        '''
        result = self._values.get("default_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HealthCheckBindOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HealthCheckConfig",
    jsii_struct_bases=[],
    name_mapping={
        "virtual_gateway_health_check": "virtualGatewayHealthCheck",
        "virtual_node_health_check": "virtualNodeHealthCheck",
    },
)
class HealthCheckConfig:
    def __init__(
        self,
        *,
        virtual_gateway_health_check: typing.Optional[CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty] = None,
        virtual_node_health_check: typing.Optional[CfnVirtualNode.HealthCheckProperty] = None,
    ) -> None:
        '''(experimental) All Properties for Health Checks for mesh endpoints.

        :param virtual_gateway_health_check: (experimental) VirtualGateway CFN configuration for Health Checks. Default: - no health checks
        :param virtual_node_health_check: (experimental) VirtualNode CFN configuration for Health Checks. Default: - no health checks

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            health_check_config = appmesh.HealthCheckConfig(
                virtual_gateway_health_check=appmesh.CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
            
                    # the properties below are optional
                    path="path",
                    port=123
                ),
                virtual_node_health_check=appmesh.CfnVirtualNode.HealthCheckProperty(
                    healthy_threshold=123,
                    interval_millis=123,
                    protocol="protocol",
                    timeout_millis=123,
                    unhealthy_threshold=123,
            
                    # the properties below are optional
                    path="path",
                    port=123
                )
            )
        '''
        if isinstance(virtual_gateway_health_check, dict):
            virtual_gateway_health_check = CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty(**virtual_gateway_health_check)
        if isinstance(virtual_node_health_check, dict):
            virtual_node_health_check = CfnVirtualNode.HealthCheckProperty(**virtual_node_health_check)
        self._values: typing.Dict[str, typing.Any] = {}
        if virtual_gateway_health_check is not None:
            self._values["virtual_gateway_health_check"] = virtual_gateway_health_check
        if virtual_node_health_check is not None:
            self._values["virtual_node_health_check"] = virtual_node_health_check

    @builtins.property
    def virtual_gateway_health_check(
        self,
    ) -> typing.Optional[CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty]:
        '''(experimental) VirtualGateway CFN configuration for Health Checks.

        :default: - no health checks

        :stability: experimental
        '''
        result = self._values.get("virtual_gateway_health_check")
        return typing.cast(typing.Optional[CfnVirtualGateway.VirtualGatewayHealthCheckPolicyProperty], result)

    @builtins.property
    def virtual_node_health_check(
        self,
    ) -> typing.Optional[CfnVirtualNode.HealthCheckProperty]:
        '''(experimental) VirtualNode CFN configuration for Health Checks.

        :default: - no health checks

        :stability: experimental
        '''
        result = self._values.get("virtual_node_health_check")
        return typing.cast(typing.Optional[CfnVirtualNode.HealthCheckProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HealthCheckConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.Http2ConnectionPool",
    jsii_struct_bases=[],
    name_mapping={"max_requests": "maxRequests"},
)
class Http2ConnectionPool:
    def __init__(self, *, max_requests: jsii.Number) -> None:
        '''(experimental) Connection pool properties for HTTP2 listeners.

        :param max_requests: (experimental) The maximum requests in the pool. Default: - none

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            http2_connection_pool = appmesh.Http2ConnectionPool(
                max_requests=123
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "max_requests": max_requests,
        }

    @builtins.property
    def max_requests(self) -> jsii.Number:
        '''(experimental) The maximum requests in the pool.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("max_requests")
        assert result is not None, "Required property 'max_requests' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Http2ConnectionPool(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.Http2GatewayListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "port": "port",
        "tls": "tls",
    },
)
class Http2GatewayListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[Http2ConnectionPool] = None,
        health_check: typing.Optional[HealthCheck] = None,
        port: typing.Optional[jsii.Number] = None,
        tls: typing.Optional["ListenerTlsOptions"] = None,
    ) -> None:
        '''(experimental) Represents the properties needed to define HTTP2 Listeners for a VirtualGateway.

        :param connection_pool: (experimental) Connection pool for http listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param tls: (experimental) Represents the configuration for enabling TLS on a listener. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with listener TLS from an ACM provided certificate
            # cert: certificatemanager.Certificate
            # mesh: appmesh.Mesh
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.dns("node"),
                listeners=[appmesh.VirtualNodeListener.grpc(
                    port=80,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.acm(cert)
                    )
                )]
            )
            
            # A Virtual Gateway with listener TLS from a customer provided file certificate
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.file("path/to/certChain", "path/to/privateKey")
                    )
                )],
                virtual_gateway_name="gateway"
            )
            
            # A Virtual Gateway with listener TLS from a SDS provided certificate
            gateway2 = appmesh.VirtualGateway(self, "gateway2",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.http2(
                    port=8080,
                    tls=cdk.aws_appmesh.ListenerTlsOptions(
                        mode=appmesh.TlsMode.STRICT,
                        certificate=appmesh.TlsCertificate.sds("secrete_certificate")
                    )
                )],
                virtual_gateway_name="gateway2"
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = Http2ConnectionPool(**connection_pool)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        self._values: typing.Dict[str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if port is not None:
            self._values["port"] = port
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[Http2ConnectionPool]:
        '''(experimental) Connection pool for http listeners.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[Http2ConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional[HealthCheck]:
        '''(experimental) The health check information for the listener.

        :default: - no healthcheck

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[HealthCheck], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port to listen for connections on.

        :default: - 8080

        :stability: experimental
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''(experimental) Represents the configuration for enabling TLS on a listener.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls")
        return typing.cast(typing.Optional["ListenerTlsOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Http2GatewayListenerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.Http2VirtualNodeListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "outlier_detection": "outlierDetection",
        "port": "port",
        "timeout": "timeout",
        "tls": "tls",
    },
)
class Http2VirtualNodeListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[Http2ConnectionPool] = None,
        health_check: typing.Optional[HealthCheck] = None,
        outlier_detection: typing.Optional["OutlierDetection"] = None,
        port: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional["HttpTimeout"] = None,
        tls: typing.Optional["ListenerTlsOptions"] = None,
    ) -> None:
        '''(experimental) Represent the HTTP2 Node Listener prorperty.

        :param connection_pool: (experimental) Connection pool for http2 listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param outlier_detection: (experimental) Represents the configuration for enabling outlier detection. Default: - none
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param timeout: (experimental) Timeout for HTTP protocol. Default: - None
        :param tls: (experimental) Represents the configuration for enabling TLS on a listener. Default: - none

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            import monocdk as monocdk
            from monocdk import aws_appmesh as appmesh
            
            # duration: monocdk.Duration
            # health_check: appmesh.HealthCheck
            # mutual_tls_validation_trust: appmesh.MutualTlsValidationTrust
            # subject_alternative_names: appmesh.SubjectAlternativeNames
            # tls_certificate: appmesh.TlsCertificate
            
            http2_virtual_node_listener_options = appmesh.Http2VirtualNodeListenerOptions(
                connection_pool=appmesh.Http2ConnectionPool(
                    max_requests=123
                ),
                health_check=health_check,
                outlier_detection=appmesh.OutlierDetection(
                    base_ejection_duration=duration,
                    interval=duration,
                    max_ejection_percent=123,
                    max_server_errors=123
                ),
                port=123,
                timeout=appmesh.HttpTimeout(
                    idle=duration,
                    per_request=duration
                ),
                tls=appmesh.ListenerTlsOptions(
                    certificate=tls_certificate,
                    mode=appmesh.TlsMode.STRICT,
            
                    # the properties below are optional
                    mutual_tls_validation=appmesh.MutualTlsValidation(
                        trust=mutual_tls_validation_trust,
            
                        # the properties below are optional
                        subject_alternative_names=subject_alternative_names
                    )
                )
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = Http2ConnectionPool(**connection_pool)
        if isinstance(outlier_detection, dict):
            outlier_detection = OutlierDetection(**outlier_detection)
        if isinstance(timeout, dict):
            timeout = HttpTimeout(**timeout)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        self._values: typing.Dict[str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if outlier_detection is not None:
            self._values["outlier_detection"] = outlier_detection
        if port is not None:
            self._values["port"] = port
        if timeout is not None:
            self._values["timeout"] = timeout
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[Http2ConnectionPool]:
        '''(experimental) Connection pool for http2 listeners.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[Http2ConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional[HealthCheck]:
        '''(experimental) The health check information for the listener.

        :default: - no healthcheck

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[HealthCheck], result)

    @builtins.property
    def outlier_detection(self) -> typing.Optional["OutlierDetection"]:
        '''(experimental) Represents the configuration for enabling outlier detection.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("outlier_detection")
        return typing.cast(typing.Optional["OutlierDetection"], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port to listen for connections on.

        :default: - 8080

        :stability: experimental
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def timeout(self) -> typing.Optional["HttpTimeout"]:
        '''(experimental) Timeout for HTTP protocol.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional["HttpTimeout"], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''(experimental) Represents the configuration for enabling TLS on a listener.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls")
        return typing.cast(typing.Optional["ListenerTlsOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Http2VirtualNodeListenerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpConnectionPool",
    jsii_struct_bases=[],
    name_mapping={
        "max_connections": "maxConnections",
        "max_pending_requests": "maxPendingRequests",
    },
)
class HttpConnectionPool:
    def __init__(
        self,
        *,
        max_connections: jsii.Number,
        max_pending_requests: jsii.Number,
    ) -> None:
        '''(experimental) Connection pool properties for HTTP listeners.

        :param max_connections: (experimental) The maximum connections in the pool. Default: - none
        :param max_pending_requests: (experimental) The maximum pending requests in the pool. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # A Virtual Node with a gRPC listener with a connection pool set
            # mesh: appmesh.Mesh
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                # DNS service discovery can optionally specify the DNS response type as either LOAD_BALANCER or ENDPOINTS.
                # LOAD_BALANCER means that the DNS resolver returns a loadbalanced set of endpoints,
                # whereas ENDPOINTS means that the DNS resolver is returning all the endpoints.
                # By default, the response type is assumed to be LOAD_BALANCER
                service_discovery=appmesh.ServiceDiscovery.dns("node", appmesh.DnsResponseType.ENDPOINTS),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=80,
                    connection_pool=cdk.aws_appmesh.HttpConnectionPool(
                        max_connections=100,
                        max_pending_requests=10
                    )
                )]
            )
            
            # A Virtual Gateway with a gRPC listener with a connection pool set
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.grpc(
                    port=8080,
                    connection_pool=cdk.aws_appmesh.GrpcConnectionPool(
                        max_requests=10
                    )
                )],
                virtual_gateway_name="gateway"
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "max_connections": max_connections,
            "max_pending_requests": max_pending_requests,
        }

    @builtins.property
    def max_connections(self) -> jsii.Number:
        '''(experimental) The maximum connections in the pool.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("max_connections")
        assert result is not None, "Required property 'max_connections' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def max_pending_requests(self) -> jsii.Number:
        '''(experimental) The maximum pending requests in the pool.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("max_pending_requests")
        assert result is not None, "Required property 'max_pending_requests' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpConnectionPool(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpGatewayListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "port": "port",
        "tls": "tls",
    },
)
class HttpGatewayListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[HttpConnectionPool] = None,
        health_check: typing.Optional[HealthCheck] = None,
        port: typing.Optional[jsii.Number] = None,
        tls: typing.Optional["ListenerTlsOptions"] = None,
    ) -> None:
        '''(experimental) Represents the properties needed to define HTTP Listeners for a VirtualGateway.

        :param connection_pool: (experimental) Connection pool for http listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param tls: (experimental) Represents the configuration for enabling TLS on a listener. Default: - none

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # mesh: appmesh.Mesh
            
            certificate_authority_arn = "arn:aws:acm-pca:us-east-1:123456789012:certificate-authority/12345678-1234-1234-1234-123456789012"
            
            gateway = appmesh.VirtualGateway(self, "gateway",
                mesh=mesh,
                listeners=[appmesh.VirtualGatewayListener.http(
                    port=443,
                    health_check=appmesh.HealthCheck.http(
                        interval=cdk.Duration.seconds(10)
                    )
                )],
                backend_defaults=cdk.aws_appmesh.BackendDefaults(
                    tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
                        ports=[8080, 8081],
                        validation=cdk.aws_appmesh.TlsValidation(
                            trust=appmesh.TlsValidationTrust.acm([
                                acmpca.CertificateAuthority.from_certificate_authority_arn(self, "certificate", certificate_authority_arn)
                            ])
                        )
                    )
                ),
                access_log=appmesh.AccessLog.from_file_path("/dev/stdout"),
                virtual_gateway_name="virtualGateway"
            )
        '''
        if isinstance(connection_pool, dict):
            connection_pool = HttpConnectionPool(**connection_pool)
        if isinstance(tls, dict):
            tls = ListenerTlsOptions(**tls)
        self._values: typing.Dict[str, typing.Any] = {}
        if connection_pool is not None:
            self._values["connection_pool"] = connection_pool
        if health_check is not None:
            self._values["health_check"] = health_check
        if port is not None:
            self._values["port"] = port
        if tls is not None:
            self._values["tls"] = tls

    @builtins.property
    def connection_pool(self) -> typing.Optional[HttpConnectionPool]:
        '''(experimental) Connection pool for http listeners.

        :default: - None

        :stability: experimental
        '''
        result = self._values.get("connection_pool")
        return typing.cast(typing.Optional[HttpConnectionPool], result)

    @builtins.property
    def health_check(self) -> typing.Optional[HealthCheck]:
        '''(experimental) The health check information for the listener.

        :default: - no healthcheck

        :stability: experimental
        '''
        result = self._values.get("health_check")
        return typing.cast(typing.Optional[HealthCheck], result)

    @builtins.property
    def port(self) -> typing.Optional[jsii.Number]:
        '''(experimental) Port to listen for connections on.

        :default: - 8080

        :stability: experimental
        '''
        result = self._values.get("port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tls(self) -> typing.Optional["ListenerTlsOptions"]:
        '''(experimental) Represents the configuration for enabling TLS on a listener.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("tls")
        return typing.cast(typing.Optional["ListenerTlsOptions"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpGatewayListenerOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpGatewayRouteMatch",
    jsii_struct_bases=[],
    name_mapping={
        "headers": "headers",
        "hostname": "hostname",
        "method": "method",
        "path": "path",
        "query_parameters": "queryParameters",
        "rewrite_request_hostname": "rewriteRequestHostname",
    },
)
class HttpGatewayRouteMatch:
    def __init__(
        self,
        *,
        headers: typing.Optional[typing.Sequence[HeaderMatch]] = None,
        hostname: typing.Optional[GatewayRouteHostnameMatch] = None,
        method: typing.Optional["HttpRouteMethod"] = None,
        path: typing.Optional["HttpGatewayRoutePathMatch"] = None,
        query_parameters: typing.Optional[typing.Sequence["QueryParameterMatch"]] = None,
        rewrite_request_hostname: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''(experimental) The criterion for determining a request match for this GatewayRoute.

        :param headers: (experimental) Specifies the client request headers to match on. All specified headers must match for the gateway route to match. Default: - do not match on headers
        :param hostname: (experimental) The gateway route host name to be matched on. Default: - do not match on host name
        :param method: (experimental) The method to match on. Default: - do not match on method
        :param path: (experimental) Specify how to match requests based on the 'path' part of their URL. Default: - matches requests with any path
        :param query_parameters: (experimental) The query parameters to match on. All specified query parameters must match for the route to match. Default: - do not match on query parameters
        :param rewrite_request_hostname: (experimental) When ``true``, rewrites the original request received at the Virtual Gateway to the destination Virtual Service name. When ``false``, retains the original hostname from the request. Default: true

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-http-2",
                route_spec=appmesh.GatewayRouteSpec.http(
                    route_target=virtual_service,
                    match=cdk.aws_appmesh.HttpGatewayRouteMatch(
                        # This rewrites the path from '/test' to '/rewrittenPath'.
                        path=appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath")
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if headers is not None:
            self._values["headers"] = headers
        if hostname is not None:
            self._values["hostname"] = hostname
        if method is not None:
            self._values["method"] = method
        if path is not None:
            self._values["path"] = path
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters
        if rewrite_request_hostname is not None:
            self._values["rewrite_request_hostname"] = rewrite_request_hostname

    @builtins.property
    def headers(self) -> typing.Optional[typing.List[HeaderMatch]]:
        '''(experimental) Specifies the client request headers to match on.

        All specified headers
        must match for the gateway route to match.

        :default: - do not match on headers

        :stability: experimental
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[typing.List[HeaderMatch]], result)

    @builtins.property
    def hostname(self) -> typing.Optional[GatewayRouteHostnameMatch]:
        '''(experimental) The gateway route host name to be matched on.

        :default: - do not match on host name

        :stability: experimental
        '''
        result = self._values.get("hostname")
        return typing.cast(typing.Optional[GatewayRouteHostnameMatch], result)

    @builtins.property
    def method(self) -> typing.Optional["HttpRouteMethod"]:
        '''(experimental) The method to match on.

        :default: - do not match on method

        :stability: experimental
        '''
        result = self._values.get("method")
        return typing.cast(typing.Optional["HttpRouteMethod"], result)

    @builtins.property
    def path(self) -> typing.Optional["HttpGatewayRoutePathMatch"]:
        '''(experimental) Specify how to match requests based on the 'path' part of their URL.

        :default: - matches requests with any path

        :stability: experimental
        '''
        result = self._values.get("path")
        return typing.cast(typing.Optional["HttpGatewayRoutePathMatch"], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[typing.List["QueryParameterMatch"]]:
        '''(experimental) The query parameters to match on.

        All specified query parameters must match for the route to match.

        :default: - do not match on query parameters

        :stability: experimental
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[typing.List["QueryParameterMatch"]], result)

    @builtins.property
    def rewrite_request_hostname(self) -> typing.Optional[builtins.bool]:
        '''(experimental) When ``true``, rewrites the original request received at the Virtual Gateway to the destination Virtual Service name.

        When ``false``, retains the original hostname from the request.

        :default: true

        :stability: experimental
        '''
        result = self._values.get("rewrite_request_hostname")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpGatewayRouteMatch(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HttpGatewayRoutePathMatch(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.HttpGatewayRoutePathMatch",
):
    '''(experimental) Defines HTTP gateway route matching based on the URL path of the request.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # gateway: appmesh.VirtualGateway
        # virtual_service: appmesh.VirtualService
        
        
        gateway.add_gateway_route("gateway-route-http-2",
            route_spec=appmesh.GatewayRouteSpec.http(
                route_target=virtual_service,
                match=cdk.aws_appmesh.HttpGatewayRouteMatch(
                    # This rewrites the path from '/test' to '/rewrittenPath'.
                    path=appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath")
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="exactly") # type: ignore[misc]
    @builtins.classmethod
    def exactly(
        cls,
        path: builtins.str,
        rewrite_to: typing.Optional[builtins.str] = None,
    ) -> "HttpGatewayRoutePathMatch":
        '''(experimental) The value of the path must match the specified value exactly.

        The provided ``path`` must start with the '/' character.

        :param path: the exact path to match on.
        :param rewrite_to: the value to substitute for the matched part of the path of the gateway request URL As a default, retains original request's URL path.

        :stability: experimental
        '''
        return typing.cast("HttpGatewayRoutePathMatch", jsii.sinvoke(cls, "exactly", [path, rewrite_to]))

    @jsii.member(jsii_name="regex") # type: ignore[misc]
    @builtins.classmethod
    def regex(
        cls,
        regex: builtins.str,
        rewrite_to: typing.Optional[builtins.str] = None,
    ) -> "HttpGatewayRoutePathMatch":
        '''(experimental) The value of the path must match the specified regex.

        :param regex: the regex used to match the path.
        :param rewrite_to: the value to substitute for the matched part of the path of the gateway request URL As a default, retains original request's URL path.

        :stability: experimental
        '''
        return typing.cast("HttpGatewayRoutePathMatch", jsii.sinvoke(cls, "regex", [regex, rewrite_to]))

    @jsii.member(jsii_name="startsWith") # type: ignore[misc]
    @builtins.classmethod
    def starts_with(
        cls,
        prefix: builtins.str,
        rewrite_to: typing.Optional[builtins.str] = None,
    ) -> "HttpGatewayRoutePathMatch":
        '''(experimental) The value of the path must match the specified prefix.

        :param prefix: the value to use to match the beginning of the path part of the URL of the request. It must start with the '/' character. When ``rewriteTo`` is provided, it must also end with the '/' character. If provided as "/", matches all requests. For example, if your virtual service name is "my-service.local" and you want the route to match requests to "my-service.local/metrics", your prefix should be "/metrics".
        :param rewrite_to: Specify either disabling automatic rewrite or rewriting to specified prefix path. To disable automatic rewrite, provide ``''``. As a default, request's URL path is automatically rewritten to '/'.

        :stability: experimental
        '''
        return typing.cast("HttpGatewayRoutePathMatch", jsii.sinvoke(cls, "startsWith", [prefix, rewrite_to]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "HttpGatewayRoutePathMatchConfig":
        '''(experimental) Returns the gateway route path match configuration.

        :param scope: -

        :stability: experimental
        '''
        ...


class _HttpGatewayRoutePathMatchProxy(HttpGatewayRoutePathMatch):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "HttpGatewayRoutePathMatchConfig":
        '''(experimental) Returns the gateway route path match configuration.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast("HttpGatewayRoutePathMatchConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, HttpGatewayRoutePathMatch).__jsii_proxy_class__ = lambda : _HttpGatewayRoutePathMatchProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpGatewayRoutePathMatchConfig",
    jsii_struct_bases=[],
    name_mapping={
        "prefix_path_match": "prefixPathMatch",
        "prefix_path_rewrite": "prefixPathRewrite",
        "whole_path_match": "wholePathMatch",
        "whole_path_rewrite": "wholePathRewrite",
    },
)
class HttpGatewayRoutePathMatchConfig:
    def __init__(
        self,
        *,
        prefix_path_match: typing.Optional[builtins.str] = None,
        prefix_path_rewrite: typing.Optional[CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty] = None,
        whole_path_match: typing.Optional[CfnGatewayRoute.HttpPathMatchProperty] = None,
        whole_path_rewrite: typing.Optional[CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty] = None,
    ) -> None:
        '''(experimental) The type returned from the ``bind()`` method in {@link HttpGatewayRoutePathMatch}.

        :param prefix_path_match: (experimental) Gateway route configuration for matching on the prefix of the URL path of the request. Default: - no matching will be performed on the prefix of the URL path
        :param prefix_path_rewrite: (experimental) Gateway route configuration for rewriting the prefix of the URL path of the request. Default: - rewrites the request's URL path to '/'
        :param whole_path_match: (experimental) Gateway route configuration for matching on the complete URL path of the request. Default: - no matching will be performed on the complete URL path
        :param whole_path_rewrite: (experimental) Gateway route configuration for rewriting the complete URL path of the request.. Default: - no rewrite will be performed on the request's complete URL path

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            http_gateway_route_path_match_config = appmesh.HttpGatewayRoutePathMatchConfig(
                prefix_path_match="prefixPathMatch",
                prefix_path_rewrite=appmesh.CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(
                    default_prefix="defaultPrefix",
                    value="value"
                ),
                whole_path_match=appmesh.CfnGatewayRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                ),
                whole_path_rewrite=appmesh.CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(
                    exact="exact"
                )
            )
        '''
        if isinstance(prefix_path_rewrite, dict):
            prefix_path_rewrite = CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty(**prefix_path_rewrite)
        if isinstance(whole_path_match, dict):
            whole_path_match = CfnGatewayRoute.HttpPathMatchProperty(**whole_path_match)
        if isinstance(whole_path_rewrite, dict):
            whole_path_rewrite = CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty(**whole_path_rewrite)
        self._values: typing.Dict[str, typing.Any] = {}
        if prefix_path_match is not None:
            self._values["prefix_path_match"] = prefix_path_match
        if prefix_path_rewrite is not None:
            self._values["prefix_path_rewrite"] = prefix_path_rewrite
        if whole_path_match is not None:
            self._values["whole_path_match"] = whole_path_match
        if whole_path_rewrite is not None:
            self._values["whole_path_rewrite"] = whole_path_rewrite

    @builtins.property
    def prefix_path_match(self) -> typing.Optional[builtins.str]:
        '''(experimental) Gateway route configuration for matching on the prefix of the URL path of the request.

        :default: - no matching will be performed on the prefix of the URL path

        :stability: experimental
        '''
        result = self._values.get("prefix_path_match")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def prefix_path_rewrite(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty]:
        '''(experimental) Gateway route configuration for rewriting the prefix of the URL path of the request.

        :default: - rewrites the request's URL path to '/'

        :stability: experimental
        '''
        result = self._values.get("prefix_path_rewrite")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRoutePrefixRewriteProperty], result)

    @builtins.property
    def whole_path_match(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpPathMatchProperty]:
        '''(experimental) Gateway route configuration for matching on the complete URL path of the request.

        :default: - no matching will be performed on the complete URL path

        :stability: experimental
        '''
        result = self._values.get("whole_path_match")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpPathMatchProperty], result)

    @builtins.property
    def whole_path_rewrite(
        self,
    ) -> typing.Optional[CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty]:
        '''(experimental) Gateway route configuration for rewriting the complete URL path of the request..

        :default: - no rewrite will be performed on the request's complete URL path

        :stability: experimental
        '''
        result = self._values.get("whole_path_rewrite")
        return typing.cast(typing.Optional[CfnGatewayRoute.HttpGatewayRoutePathRewriteProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpGatewayRoutePathMatchConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpGatewayRouteSpecOptions",
    jsii_struct_bases=[CommonGatewayRouteSpecOptions],
    name_mapping={
        "priority": "priority",
        "route_target": "routeTarget",
        "match": "match",
    },
)
class HttpGatewayRouteSpecOptions(CommonGatewayRouteSpecOptions):
    def __init__(
        self,
        *,
        priority: typing.Optional[jsii.Number] = None,
        route_target: "IVirtualService",
        match: typing.Optional[HttpGatewayRouteMatch] = None,
    ) -> None:
        '''(experimental) Properties specific for HTTP Based GatewayRoutes.

        :param priority: (experimental) The priority for the gateway route. When a Virtual Gateway has multiple gateway routes, gateway route match is performed in the order of specified value, where 0 is the highest priority, and first matched gateway route is selected. Default: - no particular priority
        :param route_target: (experimental) The VirtualService this GatewayRoute directs traffic to.
        :param match: (experimental) The criterion for determining a request match for this GatewayRoute. When path match is defined, this may optionally determine the path rewrite configuration. Default: - matches any path and automatically rewrites the path to '/'

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # gateway: appmesh.VirtualGateway
            # virtual_service: appmesh.VirtualService
            
            
            gateway.add_gateway_route("gateway-route-http-2",
                route_spec=appmesh.GatewayRouteSpec.http(
                    route_target=virtual_service,
                    match=cdk.aws_appmesh.HttpGatewayRouteMatch(
                        # This rewrites the path from '/test' to '/rewrittenPath'.
                        path=appmesh.HttpGatewayRoutePathMatch.exactly("/test", "/rewrittenPath")
                    )
                )
            )
        '''
        if isinstance(match, dict):
            match = HttpGatewayRouteMatch(**match)
        self._values: typing.Dict[str, typing.Any] = {
            "route_target": route_target,
        }
        if priority is not None:
            self._values["priority"] = priority
        if match is not None:
            self._values["match"] = match

    @builtins.property
    def priority(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The priority for the gateway route.

        When a Virtual Gateway has multiple gateway routes, gateway route match
        is performed in the order of specified value, where 0 is the highest priority,
        and first matched gateway route is selected.

        :default: - no particular priority

        :stability: experimental
        '''
        result = self._values.get("priority")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def route_target(self) -> "IVirtualService":
        '''(experimental) The VirtualService this GatewayRoute directs traffic to.

        :stability: experimental
        '''
        result = self._values.get("route_target")
        assert result is not None, "Required property 'route_target' is missing"
        return typing.cast("IVirtualService", result)

    @builtins.property
    def match(self) -> typing.Optional[HttpGatewayRouteMatch]:
        '''(experimental) The criterion for determining a request match for this GatewayRoute.

        When path match is defined, this may optionally determine the path rewrite configuration.

        :default: - matches any path and automatically rewrites the path to '/'

        :stability: experimental
        '''
        result = self._values.get("match")
        return typing.cast(typing.Optional[HttpGatewayRouteMatch], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpGatewayRouteSpecOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpHealthCheckOptions",
    jsii_struct_bases=[],
    name_mapping={
        "healthy_threshold": "healthyThreshold",
        "interval": "interval",
        "path": "path",
        "timeout": "timeout",
        "unhealthy_threshold": "unhealthyThreshold",
    },
)
class HttpHealthCheckOptions:
    def __init__(
        self,
        *,
        healthy_threshold: typing.Optional[jsii.Number] = None,
        interval: typing.Optional[_Duration_070aa057] = None,
        path: typing.Optional[builtins.str] = None,
        timeout: typing.Optional[_Duration_070aa057] = None,
        unhealthy_threshold: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''(experimental) Properties used to define HTTP Based healthchecks.

        :param healthy_threshold: (experimental) The number of consecutive successful health checks that must occur before declaring listener healthy. Default: 2
        :param interval: (experimental) The time period between each health check execution. Default: Duration.seconds(5)
        :param path: (experimental) The destination path for the health check request. Default: /
        :param timeout: (experimental) The amount of time to wait when receiving a response from the health check. Default: Duration.seconds(2)
        :param unhealthy_threshold: (experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy. Default: - 2

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # mesh: appmesh.Mesh
            vpc = ec2.Vpc(self, "vpc")
            namespace = cloudmap.PrivateDnsNamespace(self, "test-namespace",
                vpc=vpc,
                name="domain.local"
            )
            service = namespace.create_service("Svc")
            node = mesh.add_virtual_node("virtual-node",
                service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=8081,
                    health_check=appmesh.HealthCheck.http(
                        healthy_threshold=3,
                        interval=cdk.Duration.seconds(5),  # minimum
                        path="/health-check-path",
                        timeout=cdk.Duration.seconds(2),  # minimum
                        unhealthy_threshold=2
                    )
                )],
                access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if healthy_threshold is not None:
            self._values["healthy_threshold"] = healthy_threshold
        if interval is not None:
            self._values["interval"] = interval
        if path is not None:
            self._values["path"] = path
        if timeout is not None:
            self._values["timeout"] = timeout
        if unhealthy_threshold is not None:
            self._values["unhealthy_threshold"] = unhealthy_threshold

    @builtins.property
    def healthy_threshold(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of consecutive successful health checks that must occur before declaring listener healthy.

        :default: 2

        :stability: experimental
        '''
        result = self._values.get("healthy_threshold")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def interval(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The time period between each health check execution.

        :default: Duration.seconds(5)

        :stability: experimental
        '''
        result = self._values.get("interval")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def path(self) -> typing.Optional[builtins.str]:
        '''(experimental) The destination path for the health check request.

        :default: /

        :stability: experimental
        '''
        result = self._values.get("path")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) The amount of time to wait when receiving a response from the health check.

        :default: Duration.seconds(2)

        :stability: experimental
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def unhealthy_threshold(self) -> typing.Optional[jsii.Number]:
        '''(experimental) The number of consecutive failed health checks that must occur before declaring a listener unhealthy.

        :default: - 2

        :stability: experimental
        '''
        result = self._values.get("unhealthy_threshold")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpHealthCheckOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_appmesh.HttpRetryEvent")
class HttpRetryEvent(enum.Enum):
    '''(experimental) HTTP events on which to retry.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # router: appmesh.VirtualRouter
        # node: appmesh.VirtualNode
        
        
        router.add_route("route-http2-retry",
            route_spec=appmesh.RouteSpec.http2(
                weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
                retry_policy=cdk.aws_appmesh.HttpRetryPolicy(
                    # Retry if the connection failed
                    tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
                    # Retry if HTTP responds with a gateway error (502, 503, 504)
                    http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
                    # Retry five times
                    retry_attempts=5,
                    # Use a 1 second timeout per retry
                    retry_timeout=cdk.Duration.seconds(1)
                )
            )
        )
    '''

    SERVER_ERROR = "SERVER_ERROR"
    '''(experimental) HTTP status codes 500, 501, 502, 503, 504, 505, 506, 507, 508, 510, and 511.

    :stability: experimental
    '''
    GATEWAY_ERROR = "GATEWAY_ERROR"
    '''(experimental) HTTP status codes 502, 503, and 504.

    :stability: experimental
    '''
    CLIENT_ERROR = "CLIENT_ERROR"
    '''(experimental) HTTP status code 409.

    :stability: experimental
    '''
    STREAM_ERROR = "STREAM_ERROR"
    '''(experimental) Retry on refused stream.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpRetryPolicy",
    jsii_struct_bases=[],
    name_mapping={
        "retry_attempts": "retryAttempts",
        "retry_timeout": "retryTimeout",
        "http_retry_events": "httpRetryEvents",
        "tcp_retry_events": "tcpRetryEvents",
    },
)
class HttpRetryPolicy:
    def __init__(
        self,
        *,
        retry_attempts: jsii.Number,
        retry_timeout: _Duration_070aa057,
        http_retry_events: typing.Optional[typing.Sequence[HttpRetryEvent]] = None,
        tcp_retry_events: typing.Optional[typing.Sequence["TcpRetryEvent"]] = None,
    ) -> None:
        '''(experimental) HTTP retry policy.

        :param retry_attempts: (experimental) The maximum number of retry attempts.
        :param retry_timeout: (experimental) The timeout for each retry attempt.
        :param http_retry_events: (experimental) Specify HTTP events on which to retry. You must specify at least one value for at least one types of retry events. Default: - no retries for http events
        :param tcp_retry_events: (experimental) TCP events on which to retry. The event occurs before any processing of a request has started and is encountered when the upstream is temporarily or permanently unavailable. You must specify at least one value for at least one types of retry events. Default: - no retries for tcp events

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # router: appmesh.VirtualRouter
            # node: appmesh.VirtualNode
            
            
            router.add_route("route-http2-retry",
                route_spec=appmesh.RouteSpec.http2(
                    weighted_targets=[cdk.aws_appmesh.WeightedTarget(virtual_node=node)],
                    retry_policy=cdk.aws_appmesh.HttpRetryPolicy(
                        # Retry if the connection failed
                        tcp_retry_events=[appmesh.TcpRetryEvent.CONNECTION_ERROR],
                        # Retry if HTTP responds with a gateway error (502, 503, 504)
                        http_retry_events=[appmesh.HttpRetryEvent.GATEWAY_ERROR],
                        # Retry five times
                        retry_attempts=5,
                        # Use a 1 second timeout per retry
                        retry_timeout=cdk.Duration.seconds(1)
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "retry_attempts": retry_attempts,
            "retry_timeout": retry_timeout,
        }
        if http_retry_events is not None:
            self._values["http_retry_events"] = http_retry_events
        if tcp_retry_events is not None:
            self._values["tcp_retry_events"] = tcp_retry_events

    @builtins.property
    def retry_attempts(self) -> jsii.Number:
        '''(experimental) The maximum number of retry attempts.

        :stability: experimental
        '''
        result = self._values.get("retry_attempts")
        assert result is not None, "Required property 'retry_attempts' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def retry_timeout(self) -> _Duration_070aa057:
        '''(experimental) The timeout for each retry attempt.

        :stability: experimental
        '''
        result = self._values.get("retry_timeout")
        assert result is not None, "Required property 'retry_timeout' is missing"
        return typing.cast(_Duration_070aa057, result)

    @builtins.property
    def http_retry_events(self) -> typing.Optional[typing.List[HttpRetryEvent]]:
        '''(experimental) Specify HTTP events on which to retry.

        You must specify at least one value
        for at least one types of retry events.

        :default: - no retries for http events

        :stability: experimental
        '''
        result = self._values.get("http_retry_events")
        return typing.cast(typing.Optional[typing.List[HttpRetryEvent]], result)

    @builtins.property
    def tcp_retry_events(self) -> typing.Optional[typing.List["TcpRetryEvent"]]:
        '''(experimental) TCP events on which to retry.

        The event occurs before any processing of a
        request has started and is encountered when the upstream is temporarily or
        permanently unavailable. You must specify at least one value for at least
        one types of retry events.

        :default: - no retries for tcp events

        :stability: experimental
        '''
        result = self._values.get("tcp_retry_events")
        return typing.cast(typing.Optional[typing.List["TcpRetryEvent"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpRetryPolicy(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpRouteMatch",
    jsii_struct_bases=[],
    name_mapping={
        "headers": "headers",
        "method": "method",
        "path": "path",
        "protocol": "protocol",
        "query_parameters": "queryParameters",
    },
)
class HttpRouteMatch:
    def __init__(
        self,
        *,
        headers: typing.Optional[typing.Sequence[HeaderMatch]] = None,
        method: typing.Optional["HttpRouteMethod"] = None,
        path: typing.Optional["HttpRoutePathMatch"] = None,
        protocol: typing.Optional["HttpRouteProtocol"] = None,
        query_parameters: typing.Optional[typing.Sequence["QueryParameterMatch"]] = None,
    ) -> None:
        '''(experimental) The criterion for determining a request match for this Route.

        :param headers: (experimental) Specifies the client request headers to match on. All specified headers must match for the route to match. Default: - do not match on headers
        :param method: (experimental) The HTTP client request method to match on. Default: - do not match on request method
        :param path: (experimental) Specifies how is the request matched based on the path part of its URL. Default: - matches requests with all paths
        :param protocol: (experimental) The client request protocol to match on. Applicable only for HTTP2 routes. Default: - do not match on HTTP2 request protocol
        :param query_parameters: (experimental) The query parameters to match on. All specified query parameters must match for the route to match. Default: - do not match on query parameters

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # router: appmesh.VirtualRouter
            # node: appmesh.VirtualNode
            
            
            router.add_route("route-http",
                route_spec=appmesh.RouteSpec.http(
                    weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                        virtual_node=node,
                        weight=50
                    ), cdk.aws_appmesh.WeightedTarget(
                        virtual_node=node,
                        weight=50
                    )
                    ],
                    match=cdk.aws_appmesh.HttpRouteMatch(
                        path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
                    )
                )
            )
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if headers is not None:
            self._values["headers"] = headers
        if method is not None:
            self._values["method"] = method
        if path is not None:
            self._values["path"] = path
        if protocol is not None:
            self._values["protocol"] = protocol
        if query_parameters is not None:
            self._values["query_parameters"] = query_parameters

    @builtins.property
    def headers(self) -> typing.Optional[typing.List[HeaderMatch]]:
        '''(experimental) Specifies the client request headers to match on.

        All specified headers
        must match for the route to match.

        :default: - do not match on headers

        :stability: experimental
        '''
        result = self._values.get("headers")
        return typing.cast(typing.Optional[typing.List[HeaderMatch]], result)

    @builtins.property
    def method(self) -> typing.Optional["HttpRouteMethod"]:
        '''(experimental) The HTTP client request method to match on.

        :default: - do not match on request method

        :stability: experimental
        '''
        result = self._values.get("method")
        return typing.cast(typing.Optional["HttpRouteMethod"], result)

    @builtins.property
    def path(self) -> typing.Optional["HttpRoutePathMatch"]:
        '''(experimental) Specifies how is the request matched based on the path part of its URL.

        :default: - matches requests with all paths

        :stability: experimental
        '''
        result = self._values.get("path")
        return typing.cast(typing.Optional["HttpRoutePathMatch"], result)

    @builtins.property
    def protocol(self) -> typing.Optional["HttpRouteProtocol"]:
        '''(experimental) The client request protocol to match on.

        Applicable only for HTTP2 routes.

        :default: - do not match on HTTP2 request protocol

        :stability: experimental
        '''
        result = self._values.get("protocol")
        return typing.cast(typing.Optional["HttpRouteProtocol"], result)

    @builtins.property
    def query_parameters(self) -> typing.Optional[typing.List["QueryParameterMatch"]]:
        '''(experimental) The query parameters to match on.

        All specified query parameters must match for the route to match.

        :default: - do not match on query parameters

        :stability: experimental
        '''
        result = self._values.get("query_parameters")
        return typing.cast(typing.Optional[typing.List["QueryParameterMatch"]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpRouteMatch(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_appmesh.HttpRouteMethod")
class HttpRouteMethod(enum.Enum):
    '''(experimental) Supported values for matching routes based on the HTTP request method.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # router: appmesh.VirtualRouter
        # node: appmesh.VirtualNode
        
        
        router.add_route("route-http2",
            route_spec=appmesh.RouteSpec.http2(
                weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                    virtual_node=node
                )
                ],
                match=cdk.aws_appmesh.HttpRouteMatch(
                    path=appmesh.HttpRoutePathMatch.exactly("/exact"),
                    method=appmesh.HttpRouteMethod.POST,
                    protocol=appmesh.HttpRouteProtocol.HTTPS,
                    headers=[
                        # All specified headers must match for the route to match.
                        appmesh.HeaderMatch.value_is("Content-Type", "application/json"),
                        appmesh.HeaderMatch.value_is_not("Content-Type", "application/json")
                    ],
                    query_parameters=[
                        # All specified query parameters must match for the route to match.
                        appmesh.QueryParameterMatch.value_is("query-field", "value")
                    ]
                )
            )
        )
    '''

    GET = "GET"
    '''(experimental) GET request.

    :stability: experimental
    '''
    HEAD = "HEAD"
    '''(experimental) HEAD request.

    :stability: experimental
    '''
    POST = "POST"
    '''(experimental) POST request.

    :stability: experimental
    '''
    PUT = "PUT"
    '''(experimental) PUT request.

    :stability: experimental
    '''
    DELETE = "DELETE"
    '''(experimental) DELETE request.

    :stability: experimental
    '''
    CONNECT = "CONNECT"
    '''(experimental) CONNECT request.

    :stability: experimental
    '''
    OPTIONS = "OPTIONS"
    '''(experimental) OPTIONS request.

    :stability: experimental
    '''
    TRACE = "TRACE"
    '''(experimental) TRACE request.

    :stability: experimental
    '''
    PATCH = "PATCH"
    '''(experimental) PATCH request.

    :stability: experimental
    '''


class HttpRoutePathMatch(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="monocdk.aws_appmesh.HttpRoutePathMatch",
):
    '''(experimental) Defines HTTP route matching based on the URL path of the request.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # router: appmesh.VirtualRouter
        # node: appmesh.VirtualNode
        
        
        router.add_route("route-http",
            route_spec=appmesh.RouteSpec.http(
                weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                    virtual_node=node,
                    weight=50
                ), cdk.aws_appmesh.WeightedTarget(
                    virtual_node=node,
                    weight=50
                )
                ],
                match=cdk.aws_appmesh.HttpRouteMatch(
                    path=appmesh.HttpRoutePathMatch.starts_with("/path-to-app")
                )
            )
        )
    '''

    def __init__(self) -> None:
        '''
        :stability: experimental
        '''
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="exactly") # type: ignore[misc]
    @builtins.classmethod
    def exactly(cls, path: builtins.str) -> "HttpRoutePathMatch":
        '''(experimental) The value of the path must match the specified value exactly.

        The provided ``path`` must start with the '/' character.

        :param path: the exact path to match on.

        :stability: experimental
        '''
        return typing.cast("HttpRoutePathMatch", jsii.sinvoke(cls, "exactly", [path]))

    @jsii.member(jsii_name="regex") # type: ignore[misc]
    @builtins.classmethod
    def regex(cls, regex: builtins.str) -> "HttpRoutePathMatch":
        '''(experimental) The value of the path must match the specified regex.

        :param regex: the regex used to match the path.

        :stability: experimental
        '''
        return typing.cast("HttpRoutePathMatch", jsii.sinvoke(cls, "regex", [regex]))

    @jsii.member(jsii_name="startsWith") # type: ignore[misc]
    @builtins.classmethod
    def starts_with(cls, prefix: builtins.str) -> "HttpRoutePathMatch":
        '''(experimental) The value of the path must match the specified prefix.

        :param prefix: the value to use to match the beginning of the path part of the URL of the request. It must start with the '/' character. If provided as "/", matches all requests. For example, if your virtual service name is "my-service.local" and you want the route to match requests to "my-service.local/metrics", your prefix should be "/metrics".

        :stability: experimental
        '''
        return typing.cast("HttpRoutePathMatch", jsii.sinvoke(cls, "startsWith", [prefix]))

    @jsii.member(jsii_name="bind") # type: ignore[misc]
    @abc.abstractmethod
    def bind(self, scope: _Construct_e78e779f) -> "HttpRoutePathMatchConfig":
        '''(experimental) Returns the route path match configuration.

        :param scope: -

        :stability: experimental
        '''
        ...


class _HttpRoutePathMatchProxy(HttpRoutePathMatch):
    @jsii.member(jsii_name="bind")
    def bind(self, scope: _Construct_e78e779f) -> "HttpRoutePathMatchConfig":
        '''(experimental) Returns the route path match configuration.

        :param scope: -

        :stability: experimental
        '''
        return typing.cast("HttpRoutePathMatchConfig", jsii.invoke(self, "bind", [scope]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, HttpRoutePathMatch).__jsii_proxy_class__ = lambda : _HttpRoutePathMatchProxy


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpRoutePathMatchConfig",
    jsii_struct_bases=[],
    name_mapping={
        "prefix_path_match": "prefixPathMatch",
        "whole_path_match": "wholePathMatch",
    },
)
class HttpRoutePathMatchConfig:
    def __init__(
        self,
        *,
        prefix_path_match: typing.Optional[builtins.str] = None,
        whole_path_match: typing.Optional[CfnRoute.HttpPathMatchProperty] = None,
    ) -> None:
        '''(experimental) The type returned from the ``bind()`` method in {@link HttpRoutePathMatch}.

        :param prefix_path_match: (experimental) Route configuration for matching on the prefix of the URL path of the request. Default: - no matching will be performed on the prefix of the URL path
        :param whole_path_match: (experimental) Route configuration for matching on the complete URL path of the request. Default: - no matching will be performed on the complete URL path

        :stability: experimental
        :exampleMetadata: fixture=_generated

        Example::

            # The code below shows an example of how to instantiate this type.
            # The values are placeholders you should change.
            from monocdk import aws_appmesh as appmesh
            
            http_route_path_match_config = appmesh.HttpRoutePathMatchConfig(
                prefix_path_match="prefixPathMatch",
                whole_path_match=appmesh.CfnRoute.HttpPathMatchProperty(
                    exact="exact",
                    regex="regex"
                )
            )
        '''
        if isinstance(whole_path_match, dict):
            whole_path_match = CfnRoute.HttpPathMatchProperty(**whole_path_match)
        self._values: typing.Dict[str, typing.Any] = {}
        if prefix_path_match is not None:
            self._values["prefix_path_match"] = prefix_path_match
        if whole_path_match is not None:
            self._values["whole_path_match"] = whole_path_match

    @builtins.property
    def prefix_path_match(self) -> typing.Optional[builtins.str]:
        '''(experimental) Route configuration for matching on the prefix of the URL path of the request.

        :default: - no matching will be performed on the prefix of the URL path

        :stability: experimental
        '''
        result = self._values.get("prefix_path_match")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def whole_path_match(self) -> typing.Optional[CfnRoute.HttpPathMatchProperty]:
        '''(experimental) Route configuration for matching on the complete URL path of the request.

        :default: - no matching will be performed on the complete URL path

        :stability: experimental
        '''
        result = self._values.get("whole_path_match")
        return typing.cast(typing.Optional[CfnRoute.HttpPathMatchProperty], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpRoutePathMatchConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="monocdk.aws_appmesh.HttpRouteProtocol")
class HttpRouteProtocol(enum.Enum):
    '''(experimental) Supported :scheme options for HTTP2.

    :stability: experimental
    :exampleMetadata: infused

    Example::

        # router: appmesh.VirtualRouter
        # node: appmesh.VirtualNode
        
        
        router.add_route("route-http2",
            route_spec=appmesh.RouteSpec.http2(
                weighted_targets=[cdk.aws_appmesh.WeightedTarget(
                    virtual_node=node
                )
                ],
                match=cdk.aws_appmesh.HttpRouteMatch(
                    path=appmesh.HttpRoutePathMatch.exactly("/exact"),
                    method=appmesh.HttpRouteMethod.POST,
                    protocol=appmesh.HttpRouteProtocol.HTTPS,
                    headers=[
                        # All specified headers must match for the route to match.
                        appmesh.HeaderMatch.value_is("Content-Type", "application/json"),
                        appmesh.HeaderMatch.value_is_not("Content-Type", "application/json")
                    ],
                    query_parameters=[
                        # All specified query parameters must match for the route to match.
                        appmesh.QueryParameterMatch.value_is("query-field", "value")
                    ]
                )
            )
        )
    '''

    HTTP = "HTTP"
    '''(experimental) Match HTTP requests.

    :stability: experimental
    '''
    HTTPS = "HTTPS"
    '''(experimental) Match HTTPS requests.

    :stability: experimental
    '''


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpTimeout",
    jsii_struct_bases=[],
    name_mapping={"idle": "idle", "per_request": "perRequest"},
)
class HttpTimeout:
    def __init__(
        self,
        *,
        idle: typing.Optional[_Duration_070aa057] = None,
        per_request: typing.Optional[_Duration_070aa057] = None,
    ) -> None:
        '''(experimental) Represents timeouts for HTTP protocols.

        :param idle: (experimental) Represents an idle timeout. The amount of time that a connection may be idle. Default: - none
        :param per_request: (experimental) Represents per request timeout. Default: - 15 s

        :stability: experimental
        :exampleMetadata: infused

        Example::

            # mesh: appmesh.Mesh
            # service: cloudmap.Service
            
            
            node = appmesh.VirtualNode(self, "node",
                mesh=mesh,
                service_discovery=appmesh.ServiceDiscovery.cloud_map(service),
                listeners=[appmesh.VirtualNodeListener.http(
                    port=8080,
                    health_check=appmesh.HealthCheck.http(
                        healthy_threshold=3,
                        interval=cdk.Duration.seconds(5),
                        path="/ping",
                        timeout=cdk.Duration.seconds(2),
                        unhealthy_threshold=2
                    ),
                    timeout=cdk.aws_appmesh.HttpTimeout(
                        idle=cdk.Duration.seconds(5)
                    )
                )],
                backend_defaults=cdk.aws_appmesh.BackendDefaults(
                    tls_client_policy=cdk.aws_appmesh.TlsClientPolicy(
                        validation=cdk.aws_appmesh.TlsValidation(
                            trust=appmesh.TlsValidationTrust.file("/keys/local_cert_chain.pem")
                        )
                    )
                ),
                access_log=appmesh.AccessLog.from_file_path("/dev/stdout")
            )
            
            cdk.Tags.of(node).add("Environment", "Dev")
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if idle is not None:
            self._values["idle"] = idle
        if per_request is not None:
            self._values["per_request"] = per_request

    @builtins.property
    def idle(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Represents an idle timeout.

        The amount of time that a connection may be idle.

        :default: - none

        :stability: experimental
        '''
        result = self._values.get("idle")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    @builtins.property
    def per_request(self) -> typing.Optional[_Duration_070aa057]:
        '''(experimental) Represents per request timeout.

        :default: - 15 s

        :stability: experimental
        '''
        result = self._values.get("per_request")
        return typing.cast(typing.Optional[_Duration_070aa057], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpTimeout(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="monocdk.aws_appmesh.HttpVirtualNodeListenerOptions",
    jsii_struct_bases=[],
    name_mapping={
        "connection_pool": "connectionPool",
        "health_check": "healthCheck",
        "outlier_detection": "outlierDetection",
        "port": "port",
        "timeout": "timeout",
        "tls": "tls",
    },
)
class HttpVirtualNodeListenerOptions:
    def __init__(
        self,
        *,
        connection_pool: typing.Optional[HttpConnectionPool] = None,
        health_check: typing.Optional[HealthCheck] = None,
        outlier_detection: typing.Optional["OutlierDetection"] = None,
        port: typing.Optional[jsii.Number] = None,
        timeout: typing.Optional[HttpTimeout] = None,
        tls: typing.Optional["ListenerTlsOptions"] = None,
    ) -> None:
        '''(experimental) Represent the HTTP Node Listener prorperty.

        :param connection_pool: (experimental) Connection pool for http listeners. Default: - None
        :param health_check: (experimental) The health check information for the listener. Default: - no healthcheck
        :param outlier_detection: (experimental) Represents the configuration for enabling outlier detection. Default: - none
        :param port: (experimental) Port to listen for connections on. Default: - 8080
        :param timeout: (experimental) Timeout for HTTP protocol. Default: - None
        :param tls: (experimental) Represents the configuration for 