"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DestinationPattern = exports.DestinationIdentifier = exports.DockerImageManifestEntry = exports.FileManifestEntry = exports.AssetManifestReader = void 0;
// FIXME: copied from `ckd-assets`, because this tool needs to read the asset manifest aswell.
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("../../../cloud-assembly-schema");
/**
 * A manifest of assets
 */
class AssetManifestReader {
    constructor(directory, manifest) {
        this.manifest = manifest;
        this.directory = directory;
    }
    /**
     * Load an asset manifest from the given file
     */
    static fromFile(fileName) {
        try {
            const obj = cloud_assembly_schema_1.Manifest.loadAssetManifest(fileName);
            return new AssetManifestReader(path.dirname(fileName), obj);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest '${fileName}': ${e.message}`);
        }
    }
    /**
     * Load an asset manifest from the given file or directory
     *
     * If the argument given is a directoy, the default asset file name will be used.
     */
    static fromPath(filePath) {
        let st;
        try {
            st = fs.statSync(filePath);
        }
        catch (e) {
            throw new Error(`Cannot read asset manifest at '${filePath}': ${e.message}`);
        }
        if (st.isDirectory()) {
            return AssetManifestReader.fromFile(path.join(filePath, AssetManifestReader.DEFAULT_FILENAME));
        }
        return AssetManifestReader.fromFile(filePath);
    }
    /**
     * Select a subset of assets and destinations from this manifest.
     *
     * Only assets with at least 1 selected destination are retained.
     *
     * If selection is not given, everything is returned.
     */
    select(selection) {
        if (selection === undefined) {
            return this;
        }
        const ret = { version: this.manifest.version, dockerImages: {}, files: {} };
        for (const assetType of ASSET_TYPES) {
            for (const [assetId, asset] of Object.entries(this.manifest[assetType] || {})) {
                const filteredDestinations = filterDict(asset.destinations, (_, destId) => selection.some(sel => sel.matches(new DestinationIdentifier(assetId, destId))));
                if (Object.keys(filteredDestinations).length > 0) {
                    ret[assetType][assetId] = {
                        ...asset,
                        destinations: filteredDestinations,
                    };
                }
            }
        }
        return new AssetManifestReader(this.directory, ret);
    }
    /**
     * Describe the asset manifest as a list of strings
     */
    list() {
        return [
            ...describeAssets('file', this.manifest.files || {}),
            ...describeAssets('docker-image', this.manifest.dockerImages || {}),
        ];
        function describeAssets(type, assets) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets || {})) {
                ret.push(`${assetId} ${type} ${JSON.stringify(asset.source)}`);
                const destStrings = Object.entries(asset.destinations).map(([destId, dest]) => ` ${assetId}:${destId} ${JSON.stringify(dest)}`);
                ret.push(...prefixTreeChars(destStrings, '  '));
            }
            return ret;
        }
    }
    /**
     * List of assets, splat out to destinations
     */
    get entries() {
        return [
            ...makeEntries(this.manifest.files || {}, FileManifestEntry),
            ...makeEntries(this.manifest.dockerImages || {}, DockerImageManifestEntry),
        ];
        function makeEntries(assets, ctor) {
            const ret = new Array();
            for (const [assetId, asset] of Object.entries(assets)) {
                for (const [destId, destination] of Object.entries(asset.destinations)) {
                    ret.push(new ctor(new DestinationIdentifier(assetId, destId), asset.source, destination));
                }
            }
            return ret;
        }
    }
}
exports.AssetManifestReader = AssetManifestReader;
/**
 * The default name of the asset manifest in a cdk.out directory
 */
AssetManifestReader.DEFAULT_FILENAME = 'assets.json';
const ASSET_TYPES = ['files', 'dockerImages'];
/**
 * A manifest entry for a file asset
 */
class FileManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'file';
        this.genericSource = source;
    }
}
exports.FileManifestEntry = FileManifestEntry;
/**
 * A manifest entry for a docker image asset
 */
class DockerImageManifestEntry {
    constructor(
    /** Identifier for this asset */
    id, 
    /** Source of the file asset */
    source, 
    /** Destination for the file asset */
    destination) {
        this.id = id;
        this.source = source;
        this.destination = destination;
        this.type = 'docker-image';
        this.genericSource = source;
    }
}
exports.DockerImageManifestEntry = DockerImageManifestEntry;
/**
 * Identify an asset destination in an asset manifest
 */
class DestinationIdentifier {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        return this.destinationId ? `${this.assetId}:${this.destinationId}` : this.assetId;
    }
}
exports.DestinationIdentifier = DestinationIdentifier;
function filterDict(xs, pred) {
    const ret = {};
    for (const [key, value] of Object.entries(xs)) {
        if (pred(value, key)) {
            ret[key] = value;
        }
    }
    return ret;
}
/**
 * A filter pattern for an destination identifier
 */
class DestinationPattern {
    constructor(assetId, destinationId) {
        this.assetId = assetId;
        this.destinationId = destinationId;
    }
    /**
     * Parse a ':'-separated string into an asset/destination identifier
     */
    static parse(s) {
        if (!s) {
            throw new Error('Empty string is not a valid destination identifier');
        }
        const parts = s.split(':').map(x => x !== '*' ? x : undefined);
        if (parts.length === 1) {
            return new DestinationPattern(parts[0]);
        }
        if (parts.length === 2) {
            return new DestinationPattern(parts[0] || undefined, parts[1] || undefined);
        }
        throw new Error(`Asset identifier must contain at most 2 ':'-separated parts, got '${s}'`);
    }
    /**
     * Whether or not this pattern matches the given identifier
     */
    matches(id) {
        return (this.assetId === undefined || this.assetId === id.assetId)
            && (this.destinationId === undefined || this.destinationId === id.destinationId);
    }
    /**
     * Return a string representation for this asset identifier
     */
    toString() {
        var _a, _b;
        return `${(_a = this.assetId) !== null && _a !== void 0 ? _a : '*'}:${(_b = this.destinationId) !== null && _b !== void 0 ? _b : '*'}`;
    }
}
exports.DestinationPattern = DestinationPattern;
/**
 * Prefix box-drawing characters to make lines look like a hanging tree
 */
function prefixTreeChars(xs, prefix = '') {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        const isLast = i === xs.length - 1;
        const boxChar = isLast ? '└' : '├';
        ret.push(`${prefix}${boxChar}${xs[i]}`);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXNzZXQtbWFuaWZlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhc3NldC1tYW5pZmVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw4RkFBOEY7QUFDOUYseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QiwwRUFBaUs7QUFFaks7O0dBRUc7QUFDSCxNQUFhLG1CQUFtQjtJQTBDOUIsWUFBWSxTQUFpQixFQUFtQixRQUF1QjtRQUF2QixhQUFRLEdBQVIsUUFBUSxDQUFlO1FBQ3JFLElBQUksQ0FBQyxTQUFTLEdBQUcsU0FBUyxDQUFDO0tBQzVCO0lBdENEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFnQjtRQUNyQyxJQUFJO1lBQ0YsTUFBTSxHQUFHLEdBQUcsZ0NBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUVqRCxPQUFPLElBQUksbUJBQW1CLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztTQUM3RDtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQywrQkFBK0IsUUFBUSxNQUFNLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQzNFO0tBQ0Y7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFnQjtRQUNyQyxJQUFJLEVBQUUsQ0FBQztRQUNQLElBQUk7WUFDRixFQUFFLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUM1QjtRQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ1YsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsUUFBUSxNQUFNLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQzlFO1FBQ0QsSUFBSSxFQUFFLENBQUMsV0FBVyxFQUFFLEVBQUU7WUFDcEIsT0FBTyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsbUJBQW1CLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1NBQ2hHO1FBQ0QsT0FBTyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7S0FDL0M7SUFXRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsU0FBZ0M7UUFDNUMsSUFBSSxTQUFTLEtBQUssU0FBUyxFQUFFO1lBQUUsT0FBTyxJQUFJLENBQUM7U0FBRTtRQUU3QyxNQUFNLEdBQUcsR0FDTCxFQUFFLE9BQU8sRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxZQUFZLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxFQUFFLEVBQUUsQ0FBQztRQUVwRSxLQUFLLE1BQU0sU0FBUyxJQUFJLFdBQVcsRUFBRTtZQUNuQyxLQUFLLE1BQU0sQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxFQUFFO2dCQUM3RSxNQUFNLG9CQUFvQixHQUFHLFVBQVUsQ0FDckMsS0FBSyxDQUFDLFlBQVksRUFDbEIsQ0FBQyxDQUFDLEVBQUUsTUFBTSxFQUFFLEVBQUUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLHFCQUFxQixDQUFDLE9BQU8sRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFFakcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtvQkFDaEQsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHO3dCQUN4QixHQUFHLEtBQUs7d0JBQ1IsWUFBWSxFQUFFLG9CQUFvQjtxQkFDbkMsQ0FBQztpQkFDSDthQUNGO1NBQ0Y7UUFFRCxPQUFPLElBQUksbUJBQW1CLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsQ0FBQztLQUNyRDtJQUVEOztPQUVHO0lBQ0ksSUFBSTtRQUNULE9BQU87WUFDTCxHQUFHLGNBQWMsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLElBQUksRUFBRSxDQUFDO1lBQ3BELEdBQUcsY0FBYyxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksSUFBSSxFQUFFLENBQUM7U0FDcEUsQ0FBQztRQUVGLFNBQVMsY0FBYyxDQUFDLElBQVksRUFBRSxNQUEwRTtZQUM5RyxNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1lBQ2hDLEtBQUssTUFBTSxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUMsRUFBRTtnQkFDM0QsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLE9BQU8sSUFBSSxJQUFJLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2dCQUUvRCxNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsSUFBSSxPQUFPLElBQUksTUFBTSxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2dCQUNoSSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsZUFBZSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO2FBQ2pEO1lBQ0QsT0FBTyxHQUFHLENBQUM7UUFDYixDQUFDO0tBQ0Y7SUFFRDs7T0FFRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPO1lBQ0wsR0FBRyxXQUFXLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLElBQUksRUFBRSxFQUFFLGlCQUFpQixDQUFDO1lBQzVELEdBQUcsV0FBVyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxJQUFJLEVBQUUsRUFBRSx3QkFBd0IsQ0FBQztTQUMzRSxDQUFDO1FBRUYsU0FBUyxXQUFXLENBQ2xCLE1BQXNFLEVBQ3RFLElBQXFFO1lBRXJFLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFLLENBQUM7WUFDM0IsS0FBSyxNQUFNLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ3JELEtBQUssTUFBTSxDQUFDLE1BQU0sRUFBRSxXQUFXLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRTtvQkFDdEUsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJLHFCQUFxQixDQUFDLE9BQU8sRUFBRSxNQUFNLENBQUMsRUFBRSxLQUFLLENBQUMsTUFBTSxFQUFFLFdBQVcsQ0FBQyxDQUFDLENBQUM7aUJBQzNGO2FBQ0Y7WUFDRCxPQUFPLEdBQUcsQ0FBQztRQUNiLENBQUM7S0FDRjs7QUF2SEgsa0RBd0hDO0FBdkhDOztHQUVHO0FBQ29CLG9DQUFnQixHQUFHLGFBQWEsQ0FBQztBQXdIMUQsTUFBTSxXQUFXLEdBQWdCLENBQUMsT0FBTyxFQUFFLGNBQWMsQ0FBQyxDQUFDO0FBMkIzRDs7R0FFRztBQUNILE1BQWEsaUJBQWlCO0lBSTVCO0lBQ0UsZ0NBQWdDO0lBQ2hCLEVBQXlCO0lBQ3pDLCtCQUErQjtJQUNmLE1BQWtCO0lBQ2xDLHFDQUFxQztJQUNyQixXQUE0QjtRQUo1QixPQUFFLEdBQUYsRUFBRSxDQUF1QjtRQUV6QixXQUFNLEdBQU4sTUFBTSxDQUFZO1FBRWxCLGdCQUFXLEdBQVgsV0FBVyxDQUFpQjtRQVI5QixTQUFJLEdBQUcsTUFBTSxDQUFDO1FBVTVCLElBQUksQ0FBQyxhQUFhLEdBQUcsTUFBTSxDQUFDO0tBQzdCO0NBQ0Y7QUFkRCw4Q0FjQztBQUVEOztHQUVHO0FBQ0gsTUFBYSx3QkFBd0I7SUFJbkM7SUFDRSxnQ0FBZ0M7SUFDaEIsRUFBeUI7SUFDekMsK0JBQStCO0lBQ2YsTUFBeUI7SUFDekMscUNBQXFDO0lBQ3JCLFdBQW1DO1FBSm5DLE9BQUUsR0FBRixFQUFFLENBQXVCO1FBRXpCLFdBQU0sR0FBTixNQUFNLENBQW1CO1FBRXpCLGdCQUFXLEdBQVgsV0FBVyxDQUF3QjtRQVJyQyxTQUFJLEdBQUcsY0FBYyxDQUFDO1FBVXBDLElBQUksQ0FBQyxhQUFhLEdBQUcsTUFBTSxDQUFDO0tBQzdCO0NBQ0Y7QUFkRCw0REFjQztBQUVEOztHQUVHO0FBQ0gsTUFBYSxxQkFBcUI7SUFXaEMsWUFBWSxPQUFlLEVBQUUsYUFBcUI7UUFDaEQsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7UUFDdkIsSUFBSSxDQUFDLGFBQWEsR0FBRyxhQUFhLENBQUM7S0FDcEM7SUFFRDs7T0FFRztJQUNJLFFBQVE7UUFDYixPQUFPLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUM7S0FDcEY7Q0FDRjtBQXRCRCxzREFzQkM7QUFFRCxTQUFTLFVBQVUsQ0FBSSxFQUFxQixFQUFFLElBQW9DO0lBQ2hGLE1BQU0sR0FBRyxHQUFzQixFQUFFLENBQUM7SUFDbEMsS0FBSyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUU7UUFDN0MsSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxFQUFFO1lBQ3BCLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7U0FDbEI7S0FDRjtJQUNELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQUVEOztHQUVHO0FBQ0gsTUFBYSxrQkFBa0I7SUFzQjdCLFlBQVksT0FBZ0IsRUFBRSxhQUFzQjtRQUNsRCxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsYUFBYSxHQUFHLGFBQWEsQ0FBQztLQUNwQztJQXhCRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBUztRQUMzQixJQUFJLENBQUMsQ0FBQyxFQUFFO1lBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1NBQUU7UUFDbEYsTUFBTSxLQUFLLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQy9ELElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLElBQUksa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FBRTtRQUNwRSxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQUUsT0FBTyxJQUFJLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLFNBQVMsQ0FBQyxDQUFDO1NBQUU7UUFDeEcsTUFBTSxJQUFJLEtBQUssQ0FBQyxxRUFBcUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUM1RjtJQWlCRDs7T0FFRztJQUNJLE9BQU8sQ0FBQyxFQUF5QjtRQUN0QyxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLE9BQU8sS0FBSyxFQUFFLENBQUMsT0FBTyxDQUFDO2VBQzdELENBQUMsSUFBSSxDQUFDLGFBQWEsS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLGFBQWEsS0FBSyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7S0FDcEY7SUFFRDs7T0FFRztJQUNJLFFBQVE7O1FBQ2IsT0FBTyxHQUFHLE1BQUEsSUFBSSxDQUFDLE9BQU8sbUNBQUksR0FBRyxJQUFJLE1BQUEsSUFBSSxDQUFDLGFBQWEsbUNBQUksR0FBRyxFQUFFLENBQUM7S0FDOUQ7Q0FDRjtBQXpDRCxnREF5Q0M7QUFFRDs7R0FFRztBQUNILFNBQVMsZUFBZSxDQUFDLEVBQVksRUFBRSxNQUFNLEdBQUcsRUFBRTtJQUNoRCxNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO0lBQ2hDLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxFQUFFLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQ2xDLE1BQU0sTUFBTSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztRQUNuQyxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO1FBQ25DLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLEdBQUcsT0FBTyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7S0FDekM7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBGSVhNRTogY29waWVkIGZyb20gYGNrZC1hc3NldHNgLCBiZWNhdXNlIHRoaXMgdG9vbCBuZWVkcyB0byByZWFkIHRoZSBhc3NldCBtYW5pZmVzdCBhc3dlbGwuXG5pbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQXNzZXRNYW5pZmVzdCwgQXdzRGVzdGluYXRpb24sIERvY2tlckltYWdlRGVzdGluYXRpb24sIERvY2tlckltYWdlU291cmNlLCBGaWxlRGVzdGluYXRpb24sIEZpbGVTb3VyY2UsIE1hbmlmZXN0IH0gZnJvbSAnLi4vLi4vLi4vY2xvdWQtYXNzZW1ibHktc2NoZW1hJztcblxuLyoqXG4gKiBBIG1hbmlmZXN0IG9mIGFzc2V0c1xuICovXG5leHBvcnQgY2xhc3MgQXNzZXRNYW5pZmVzdFJlYWRlciB7XG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBuYW1lIG9mIHRoZSBhc3NldCBtYW5pZmVzdCBpbiBhIGNkay5vdXQgZGlyZWN0b3J5XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfRklMRU5BTUUgPSAnYXNzZXRzLmpzb24nO1xuXG4gIC8qKlxuICAgKiBMb2FkIGFuIGFzc2V0IG1hbmlmZXN0IGZyb20gdGhlIGdpdmVuIGZpbGVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUZpbGUoZmlsZU5hbWU6IHN0cmluZykge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCBvYmogPSBNYW5pZmVzdC5sb2FkQXNzZXRNYW5pZmVzdChmaWxlTmFtZSk7XG5cbiAgICAgIHJldHVybiBuZXcgQXNzZXRNYW5pZmVzdFJlYWRlcihwYXRoLmRpcm5hbWUoZmlsZU5hbWUpLCBvYmopO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IHJlYWQgYXNzZXQgbWFuaWZlc3QgJyR7ZmlsZU5hbWV9JzogJHtlLm1lc3NhZ2V9YCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIExvYWQgYW4gYXNzZXQgbWFuaWZlc3QgZnJvbSB0aGUgZ2l2ZW4gZmlsZSBvciBkaXJlY3RvcnlcbiAgICpcbiAgICogSWYgdGhlIGFyZ3VtZW50IGdpdmVuIGlzIGEgZGlyZWN0b3ksIHRoZSBkZWZhdWx0IGFzc2V0IGZpbGUgbmFtZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21QYXRoKGZpbGVQYXRoOiBzdHJpbmcpIHtcbiAgICBsZXQgc3Q7XG4gICAgdHJ5IHtcbiAgICAgIHN0ID0gZnMuc3RhdFN5bmMoZmlsZVBhdGgpO1xuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IHJlYWQgYXNzZXQgbWFuaWZlc3QgYXQgJyR7ZmlsZVBhdGh9JzogJHtlLm1lc3NhZ2V9YCk7XG4gICAgfVxuICAgIGlmIChzdC5pc0RpcmVjdG9yeSgpKSB7XG4gICAgICByZXR1cm4gQXNzZXRNYW5pZmVzdFJlYWRlci5mcm9tRmlsZShwYXRoLmpvaW4oZmlsZVBhdGgsIEFzc2V0TWFuaWZlc3RSZWFkZXIuREVGQVVMVF9GSUxFTkFNRSkpO1xuICAgIH1cbiAgICByZXR1cm4gQXNzZXRNYW5pZmVzdFJlYWRlci5mcm9tRmlsZShmaWxlUGF0aCk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGRpcmVjdG9yeSB3aGVyZSB0aGUgbWFuaWZlc3Qgd2FzIGZvdW5kXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGlyZWN0b3J5OiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IoZGlyZWN0b3J5OiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgbWFuaWZlc3Q6IEFzc2V0TWFuaWZlc3QpIHtcbiAgICB0aGlzLmRpcmVjdG9yeSA9IGRpcmVjdG9yeTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZWxlY3QgYSBzdWJzZXQgb2YgYXNzZXRzIGFuZCBkZXN0aW5hdGlvbnMgZnJvbSB0aGlzIG1hbmlmZXN0LlxuICAgKlxuICAgKiBPbmx5IGFzc2V0cyB3aXRoIGF0IGxlYXN0IDEgc2VsZWN0ZWQgZGVzdGluYXRpb24gYXJlIHJldGFpbmVkLlxuICAgKlxuICAgKiBJZiBzZWxlY3Rpb24gaXMgbm90IGdpdmVuLCBldmVyeXRoaW5nIGlzIHJldHVybmVkLlxuICAgKi9cbiAgcHVibGljIHNlbGVjdChzZWxlY3Rpb24/OiBEZXN0aW5hdGlvblBhdHRlcm5bXSk6IEFzc2V0TWFuaWZlc3RSZWFkZXIge1xuICAgIGlmIChzZWxlY3Rpb24gPT09IHVuZGVmaW5lZCkgeyByZXR1cm4gdGhpczsgfVxuXG4gICAgY29uc3QgcmV0OiBBc3NldE1hbmlmZXN0ICYgUmVxdWlyZWQ8UGljazxBc3NldE1hbmlmZXN0LCBBc3NldFR5cGU+PlxuICAgICAgPSB7IHZlcnNpb246IHRoaXMubWFuaWZlc3QudmVyc2lvbiwgZG9ja2VySW1hZ2VzOiB7fSwgZmlsZXM6IHt9IH07XG5cbiAgICBmb3IgKGNvbnN0IGFzc2V0VHlwZSBvZiBBU1NFVF9UWVBFUykge1xuICAgICAgZm9yIChjb25zdCBbYXNzZXRJZCwgYXNzZXRdIG9mIE9iamVjdC5lbnRyaWVzKHRoaXMubWFuaWZlc3RbYXNzZXRUeXBlXSB8fCB7fSkpIHtcbiAgICAgICAgY29uc3QgZmlsdGVyZWREZXN0aW5hdGlvbnMgPSBmaWx0ZXJEaWN0KFxuICAgICAgICAgIGFzc2V0LmRlc3RpbmF0aW9ucyxcbiAgICAgICAgICAoXywgZGVzdElkKSA9PiBzZWxlY3Rpb24uc29tZShzZWwgPT4gc2VsLm1hdGNoZXMobmV3IERlc3RpbmF0aW9uSWRlbnRpZmllcihhc3NldElkLCBkZXN0SWQpKSkpO1xuXG4gICAgICAgIGlmIChPYmplY3Qua2V5cyhmaWx0ZXJlZERlc3RpbmF0aW9ucykubGVuZ3RoID4gMCkge1xuICAgICAgICAgIHJldFthc3NldFR5cGVdW2Fzc2V0SWRdID0ge1xuICAgICAgICAgICAgLi4uYXNzZXQsXG4gICAgICAgICAgICBkZXN0aW5hdGlvbnM6IGZpbHRlcmVkRGVzdGluYXRpb25zLFxuICAgICAgICAgIH07XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEFzc2V0TWFuaWZlc3RSZWFkZXIodGhpcy5kaXJlY3RvcnksIHJldCk7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmUgdGhlIGFzc2V0IG1hbmlmZXN0IGFzIGEgbGlzdCBvZiBzdHJpbmdzXG4gICAqL1xuICBwdWJsaWMgbGlzdCgpIHtcbiAgICByZXR1cm4gW1xuICAgICAgLi4uZGVzY3JpYmVBc3NldHMoJ2ZpbGUnLCB0aGlzLm1hbmlmZXN0LmZpbGVzIHx8IHt9KSxcbiAgICAgIC4uLmRlc2NyaWJlQXNzZXRzKCdkb2NrZXItaW1hZ2UnLCB0aGlzLm1hbmlmZXN0LmRvY2tlckltYWdlcyB8fCB7fSksXG4gICAgXTtcblxuICAgIGZ1bmN0aW9uIGRlc2NyaWJlQXNzZXRzKHR5cGU6IHN0cmluZywgYXNzZXRzOiBSZWNvcmQ8c3RyaW5nLCB7IHNvdXJjZTogYW55LCBkZXN0aW5hdGlvbnM6IFJlY29yZDxzdHJpbmcsIGFueT4gfT4pIHtcbiAgICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gICAgICBmb3IgKGNvbnN0IFthc3NldElkLCBhc3NldF0gb2YgT2JqZWN0LmVudHJpZXMoYXNzZXRzIHx8IHt9KSkge1xuICAgICAgICByZXQucHVzaChgJHthc3NldElkfSAke3R5cGV9ICR7SlNPTi5zdHJpbmdpZnkoYXNzZXQuc291cmNlKX1gKTtcblxuICAgICAgICBjb25zdCBkZXN0U3RyaW5ncyA9IE9iamVjdC5lbnRyaWVzKGFzc2V0LmRlc3RpbmF0aW9ucykubWFwKChbZGVzdElkLCBkZXN0XSkgPT4gYCAke2Fzc2V0SWR9OiR7ZGVzdElkfSAke0pTT04uc3RyaW5naWZ5KGRlc3QpfWApO1xuICAgICAgICByZXQucHVzaCguLi5wcmVmaXhUcmVlQ2hhcnMoZGVzdFN0cmluZ3MsICcgICcpKTtcbiAgICAgIH1cbiAgICAgIHJldHVybiByZXQ7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgYXNzZXRzLCBzcGxhdCBvdXQgdG8gZGVzdGluYXRpb25zXG4gICAqL1xuICBwdWJsaWMgZ2V0IGVudHJpZXMoKTogSU1hbmlmZXN0RW50cnlbXSB7XG4gICAgcmV0dXJuIFtcbiAgICAgIC4uLm1ha2VFbnRyaWVzKHRoaXMubWFuaWZlc3QuZmlsZXMgfHwge30sIEZpbGVNYW5pZmVzdEVudHJ5KSxcbiAgICAgIC4uLm1ha2VFbnRyaWVzKHRoaXMubWFuaWZlc3QuZG9ja2VySW1hZ2VzIHx8IHt9LCBEb2NrZXJJbWFnZU1hbmlmZXN0RW50cnkpLFxuICAgIF07XG5cbiAgICBmdW5jdGlvbiBtYWtlRW50cmllczxBLCBCLCBDPihcbiAgICAgIGFzc2V0czogUmVjb3JkPHN0cmluZywgeyBzb3VyY2U6IEEsIGRlc3RpbmF0aW9uczogUmVjb3JkPHN0cmluZywgQj4gfT4sXG4gICAgICBjdG9yOiBuZXcgKGlkOiBEZXN0aW5hdGlvbklkZW50aWZpZXIsIHNvdXJjZTogQSwgZGVzdGluYXRpb246IEIpID0+IEMpOiBDW10ge1xuXG4gICAgICBjb25zdCByZXQgPSBuZXcgQXJyYXk8Qz4oKTtcbiAgICAgIGZvciAoY29uc3QgW2Fzc2V0SWQsIGFzc2V0XSBvZiBPYmplY3QuZW50cmllcyhhc3NldHMpKSB7XG4gICAgICAgIGZvciAoY29uc3QgW2Rlc3RJZCwgZGVzdGluYXRpb25dIG9mIE9iamVjdC5lbnRyaWVzKGFzc2V0LmRlc3RpbmF0aW9ucykpIHtcbiAgICAgICAgICByZXQucHVzaChuZXcgY3RvcihuZXcgRGVzdGluYXRpb25JZGVudGlmaWVyKGFzc2V0SWQsIGRlc3RJZCksIGFzc2V0LnNvdXJjZSwgZGVzdGluYXRpb24pKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gIH1cbn1cblxudHlwZSBBc3NldFR5cGUgPSAnZmlsZXMnIHwgJ2RvY2tlckltYWdlcyc7XG5cbmNvbnN0IEFTU0VUX1RZUEVTOiBBc3NldFR5cGVbXSA9IFsnZmlsZXMnLCAnZG9ja2VySW1hZ2VzJ107XG5cbi8qKlxuICogQSBzaW5nbGUgYXNzZXQgZnJvbSBhbiBhc3NldCBtYW5pZmVzdCdcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJTWFuaWZlc3RFbnRyeSB7XG4gIC8qKlxuICAgKiBUaGUgaWRlbnRpZmllciBvZiB0aGUgYXNzZXRcbiAgICovXG4gIHJlYWRvbmx5IGlkOiBEZXN0aW5hdGlvbklkZW50aWZpZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIGFzc2V0XG4gICAqL1xuICByZWFkb25seSB0eXBlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFR5cGUtZGVwZW5kZW50IHNvdXJjZSBkYXRhXG4gICAqL1xuICByZWFkb25seSBnZW5lcmljU291cmNlOiB1bmtub3duO1xuXG4gIC8qKlxuICAgKiBUeXBlLWRlcGVuZGVudCBkZXN0aW5hdGlvbiBkYXRhXG4gICAqL1xuICByZWFkb25seSBkZXN0aW5hdGlvbjogQXdzRGVzdGluYXRpb247XG59XG5cbi8qKlxuICogQSBtYW5pZmVzdCBlbnRyeSBmb3IgYSBmaWxlIGFzc2V0XG4gKi9cbmV4cG9ydCBjbGFzcyBGaWxlTWFuaWZlc3RFbnRyeSBpbXBsZW1lbnRzIElNYW5pZmVzdEVudHJ5IHtcbiAgcHVibGljIHJlYWRvbmx5IGdlbmVyaWNTb3VyY2U6IHVua25vd247XG4gIHB1YmxpYyByZWFkb25seSB0eXBlID0gJ2ZpbGUnO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIC8qKiBJZGVudGlmaWVyIGZvciB0aGlzIGFzc2V0ICovXG4gICAgcHVibGljIHJlYWRvbmx5IGlkOiBEZXN0aW5hdGlvbklkZW50aWZpZXIsXG4gICAgLyoqIFNvdXJjZSBvZiB0aGUgZmlsZSBhc3NldCAqL1xuICAgIHB1YmxpYyByZWFkb25seSBzb3VyY2U6IEZpbGVTb3VyY2UsXG4gICAgLyoqIERlc3RpbmF0aW9uIGZvciB0aGUgZmlsZSBhc3NldCAqL1xuICAgIHB1YmxpYyByZWFkb25seSBkZXN0aW5hdGlvbjogRmlsZURlc3RpbmF0aW9uLFxuICApIHtcbiAgICB0aGlzLmdlbmVyaWNTb3VyY2UgPSBzb3VyY2U7XG4gIH1cbn1cblxuLyoqXG4gKiBBIG1hbmlmZXN0IGVudHJ5IGZvciBhIGRvY2tlciBpbWFnZSBhc3NldFxuICovXG5leHBvcnQgY2xhc3MgRG9ja2VySW1hZ2VNYW5pZmVzdEVudHJ5IGltcGxlbWVudHMgSU1hbmlmZXN0RW50cnkge1xuICBwdWJsaWMgcmVhZG9ubHkgZ2VuZXJpY1NvdXJjZTogdW5rbm93bjtcbiAgcHVibGljIHJlYWRvbmx5IHR5cGUgPSAnZG9ja2VyLWltYWdlJztcblxuICBjb25zdHJ1Y3RvcihcbiAgICAvKiogSWRlbnRpZmllciBmb3IgdGhpcyBhc3NldCAqL1xuICAgIHB1YmxpYyByZWFkb25seSBpZDogRGVzdGluYXRpb25JZGVudGlmaWVyLFxuICAgIC8qKiBTb3VyY2Ugb2YgdGhlIGZpbGUgYXNzZXQgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgc291cmNlOiBEb2NrZXJJbWFnZVNvdXJjZSxcbiAgICAvKiogRGVzdGluYXRpb24gZm9yIHRoZSBmaWxlIGFzc2V0ICovXG4gICAgcHVibGljIHJlYWRvbmx5IGRlc3RpbmF0aW9uOiBEb2NrZXJJbWFnZURlc3RpbmF0aW9uLFxuICApIHtcbiAgICB0aGlzLmdlbmVyaWNTb3VyY2UgPSBzb3VyY2U7XG4gIH1cbn1cblxuLyoqXG4gKiBJZGVudGlmeSBhbiBhc3NldCBkZXN0aW5hdGlvbiBpbiBhbiBhc3NldCBtYW5pZmVzdFxuICovXG5leHBvcnQgY2xhc3MgRGVzdGluYXRpb25JZGVudGlmaWVyIHtcbiAgLyoqXG4gICAqIElkZW50aWZpZXMgdGhlIGFzc2V0LCBieSBzb3VyY2UuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXNzZXRJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJZGVudGlmaWVzIHRoZSBkZXN0aW5hdGlvbiB3aGVyZSB0aGlzIGFzc2V0IHdpbGwgYmUgcHVibGlzaGVkXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVzdGluYXRpb25JZDogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKGFzc2V0SWQ6IHN0cmluZywgZGVzdGluYXRpb25JZDogc3RyaW5nKSB7XG4gICAgdGhpcy5hc3NldElkID0gYXNzZXRJZDtcbiAgICB0aGlzLmRlc3RpbmF0aW9uSWQgPSBkZXN0aW5hdGlvbklkO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIHN0cmluZyByZXByZXNlbnRhdGlvbiBmb3IgdGhpcyBhc3NldCBpZGVudGlmaWVyXG4gICAqL1xuICBwdWJsaWMgdG9TdHJpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMuZGVzdGluYXRpb25JZCA/IGAke3RoaXMuYXNzZXRJZH06JHt0aGlzLmRlc3RpbmF0aW9uSWR9YCA6IHRoaXMuYXNzZXRJZDtcbiAgfVxufVxuXG5mdW5jdGlvbiBmaWx0ZXJEaWN0PEE+KHhzOiBSZWNvcmQ8c3RyaW5nLCBBPiwgcHJlZDogKHg6IEEsIGtleTogc3RyaW5nKSA9PiBib29sZWFuKTogUmVjb3JkPHN0cmluZywgQT4ge1xuICBjb25zdCByZXQ6IFJlY29yZDxzdHJpbmcsIEE+ID0ge307XG4gIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHhzKSkge1xuICAgIGlmIChwcmVkKHZhbHVlLCBrZXkpKSB7XG4gICAgICByZXRba2V5XSA9IHZhbHVlO1xuICAgIH1cbiAgfVxuICByZXR1cm4gcmV0O1xufVxuXG4vKipcbiAqIEEgZmlsdGVyIHBhdHRlcm4gZm9yIGFuIGRlc3RpbmF0aW9uIGlkZW50aWZpZXJcbiAqL1xuZXhwb3J0IGNsYXNzIERlc3RpbmF0aW9uUGF0dGVybiB7XG4gIC8qKlxuICAgKiBQYXJzZSBhICc6Jy1zZXBhcmF0ZWQgc3RyaW5nIGludG8gYW4gYXNzZXQvZGVzdGluYXRpb24gaWRlbnRpZmllclxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBwYXJzZShzOiBzdHJpbmcpIHtcbiAgICBpZiAoIXMpIHsgdGhyb3cgbmV3IEVycm9yKCdFbXB0eSBzdHJpbmcgaXMgbm90IGEgdmFsaWQgZGVzdGluYXRpb24gaWRlbnRpZmllcicpOyB9XG4gICAgY29uc3QgcGFydHMgPSBzLnNwbGl0KCc6JykubWFwKHggPT4geCAhPT0gJyonID8geCA6IHVuZGVmaW5lZCk7XG4gICAgaWYgKHBhcnRzLmxlbmd0aCA9PT0gMSkgeyByZXR1cm4gbmV3IERlc3RpbmF0aW9uUGF0dGVybihwYXJ0c1swXSk7IH1cbiAgICBpZiAocGFydHMubGVuZ3RoID09PSAyKSB7IHJldHVybiBuZXcgRGVzdGluYXRpb25QYXR0ZXJuKHBhcnRzWzBdIHx8IHVuZGVmaW5lZCwgcGFydHNbMV0gfHwgdW5kZWZpbmVkKTsgfVxuICAgIHRocm93IG5ldyBFcnJvcihgQXNzZXQgaWRlbnRpZmllciBtdXN0IGNvbnRhaW4gYXQgbW9zdCAyICc6Jy1zZXBhcmF0ZWQgcGFydHMsIGdvdCAnJHtzfSdgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJZGVudGlmaWVzIHRoZSBhc3NldCwgYnkgc291cmNlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGFzc2V0SWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIElkZW50aWZpZXMgdGhlIGRlc3RpbmF0aW9uIHdoZXJlIHRoaXMgYXNzZXQgd2lsbCBiZSBwdWJsaXNoZWRcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZXN0aW5hdGlvbklkPzogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKGFzc2V0SWQ/OiBzdHJpbmcsIGRlc3RpbmF0aW9uSWQ/OiBzdHJpbmcpIHtcbiAgICB0aGlzLmFzc2V0SWQgPSBhc3NldElkO1xuICAgIHRoaXMuZGVzdGluYXRpb25JZCA9IGRlc3RpbmF0aW9uSWQ7XG4gIH1cblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhpcyBwYXR0ZXJuIG1hdGNoZXMgdGhlIGdpdmVuIGlkZW50aWZpZXJcbiAgICovXG4gIHB1YmxpYyBtYXRjaGVzKGlkOiBEZXN0aW5hdGlvbklkZW50aWZpZXIpIHtcbiAgICByZXR1cm4gKHRoaXMuYXNzZXRJZCA9PT0gdW5kZWZpbmVkIHx8IHRoaXMuYXNzZXRJZCA9PT0gaWQuYXNzZXRJZClcbiAgICAgICYmICh0aGlzLmRlc3RpbmF0aW9uSWQgPT09IHVuZGVmaW5lZCB8fCB0aGlzLmRlc3RpbmF0aW9uSWQgPT09IGlkLmRlc3RpbmF0aW9uSWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiBhIHN0cmluZyByZXByZXNlbnRhdGlvbiBmb3IgdGhpcyBhc3NldCBpZGVudGlmaWVyXG4gICAqL1xuICBwdWJsaWMgdG9TdHJpbmcoKSB7XG4gICAgcmV0dXJuIGAke3RoaXMuYXNzZXRJZCA/PyAnKid9OiR7dGhpcy5kZXN0aW5hdGlvbklkID8/ICcqJ31gO1xuICB9XG59XG5cbi8qKlxuICogUHJlZml4IGJveC1kcmF3aW5nIGNoYXJhY3RlcnMgdG8gbWFrZSBsaW5lcyBsb29rIGxpa2UgYSBoYW5naW5nIHRyZWVcbiAqL1xuZnVuY3Rpb24gcHJlZml4VHJlZUNoYXJzKHhzOiBzdHJpbmdbXSwgcHJlZml4ID0gJycpIHtcbiAgY29uc3QgcmV0ID0gbmV3IEFycmF5PHN0cmluZz4oKTtcbiAgZm9yIChsZXQgaSA9IDA7IGkgPCB4cy5sZW5ndGg7IGkrKykge1xuICAgIGNvbnN0IGlzTGFzdCA9IGkgPT09IHhzLmxlbmd0aCAtIDE7XG4gICAgY29uc3QgYm94Q2hhciA9IGlzTGFzdCA/ICfilJQnIDogJ+KUnCc7XG4gICAgcmV0LnB1c2goYCR7cHJlZml4fSR7Ym94Q2hhcn0ke3hzW2ldfWApO1xuICB9XG4gIHJldHVybiByZXQ7XG59XG4iXX0=