"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Files = void 0;
const s3assets = require("../../../../aws-s3-assets");
/**
 * Additional files to use in a shell script
 */
class Files {
    /**
     * Use the files from a CodePipeline artifact
     */
    static fromArtifact(artifact) {
        if (!artifact) {
            // Typechecking may mess up
            throw new Error('Files.fromArtifact(): input artifact is required, got undefined');
        }
        return {
            bind: () => ({ artifact }),
            grantRead: () => { },
        };
    }
    /**
     * Create a new asset to bundle up the files in a directory on disk
     */
    static fromDirectory(directoryPath) {
        let realFiles;
        return {
            bind(scope) {
                realFiles = Files.fromAsset(new s3assets.Asset(scope, directoryPath, {
                    path: directoryPath,
                }));
                return realFiles.bind(scope);
            },
            grantRead(grantee) {
                if (!realFiles) {
                    throw new Error('bind() must be called first');
                }
                realFiles.grantRead(grantee);
            },
        };
    }
    /**
     * Use an existing asset as a file source
     */
    static fromAsset(asset) {
        return {
            bind: () => ({
                commands: [
                    `echo "Downloading additional files from ${asset.s3ObjectUrl}"`,
                    `aws s3 cp ${asset.s3ObjectUrl} /tmp/dl.zip`,
                    'unzip /tmp/dl.zip -d .',
                ],
            }),
            grantRead: (grantee) => asset.grantRead(grantee),
        };
    }
    constructor() {
    }
}
exports.Files = Files;
//# sourceMappingURL=data:application/json;base64,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