"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepOutput = void 0;
const core_1 = require("../../../core");
const STEP_OUTPUT_SYM = Symbol.for('@aws-cdk/pipelines.StepOutput');
const PRODUCED_OUTPUTS_SYM = Symbol.for('@aws-cdk/pipelines.outputs');
/**
 * A symbolic reference to a value produced by another step
 *
 * Generating and consuming outputs is engine-specific. Many engines will be
 * able to support a feature like "outputs", but it's not guaranteed that
 * all of them will.
 *
 * Outputs can only be generated by engine-specific steps (CodeBuildStep instead
 * of ShellStep, etc), but can (currently) be consumed anywhere(*). When
 * an engine-specific step generates an Output, it should put a well-known
 * string and arbitrary data that is useful to the engine into the engine-specific
 * fields on the StepOutput.
 *
 * The graph blueprint will take care of dependencies and ordering, the engine
 * is responsible interpreting and rendering StepOutputs. The engine should call
 * `defineResolution()` on all outputs.
 *
 * StepOutputs currently purposely aren't part of the public API because users
 * shouldn't see the innards poking out. So, instead of keeping state on `Step`,
 * we keep side-state here in a WeakMap which can be accessed via static members
 * on `StepOutput`.
 *
 * (*) If we need to restrict this, we add the checking and erroring in the engine.
 */
class StepOutput {
    constructor(step, engineName, engineSpecificInformation) {
        this.creationStack = [];
        this.resolution = undefined;
        this.step = step;
        this.engineName = engineName;
        this.engineSpecificInformation = engineSpecificInformation;
        Object.defineProperty(this, STEP_OUTPUT_SYM, { value: true });
    }
    /**
     * Return true if the given IResolvable is a StepOutput
     */
    static isStepOutput(resolvable) {
        return !!resolvable[STEP_OUTPUT_SYM];
    }
    /**
     * Find all StepOutputs referenced in the given structure
     */
    static findAll(structure) {
        return findAllStepOutputs(structure);
    }
    /**
     * Return the produced outputs for the given step
     */
    static producedStepOutputs(step) {
        var _a;
        return (_a = step[PRODUCED_OUTPUTS_SYM]) !== null && _a !== void 0 ? _a : [];
    }
    /**
     * Add produced outputs for the given step
     */
    static recordProducer(...outputs) {
        for (const output of outputs) {
            const step = output.step;
            let list = step[PRODUCED_OUTPUTS_SYM];
            if (!list) {
                list = [];
                step[PRODUCED_OUTPUTS_SYM] = list;
            }
            list.push(...outputs);
        }
    }
    /**
     * Define the resolved value for this StepOutput.
     *
     * Should be called by the engine.
     */
    defineResolution(value) {
        this.resolution = value;
    }
    resolve(_context) {
        if (this.resolution === undefined) {
            throw new Error(`Output for step ${this.step} not configured. Either the step is not in the pipeline, or this engine does not support Outputs for this step.`);
        }
        return this.resolution;
    }
    toString() {
        return core_1.Token.asString(this);
    }
}
exports.StepOutput = StepOutput;
function findAllStepOutputs(structure) {
    const ret = new Set();
    recurse(structure);
    return Array.from(ret);
    function checkToken(x) {
        if (x && StepOutput.isStepOutput(x)) {
            ret.add(x);
            return true;
        }
        // Return false if it wasn't a Token in the first place (in which case we recurse)
        return x !== undefined;
    }
    function recurse(x) {
        if (!x) {
            return;
        }
        if (core_1.Tokenization.isResolvable(x)) {
            checkToken(x);
            return;
        }
        if (Array.isArray(x)) {
            if (!checkToken(core_1.Tokenization.reverseList(x))) {
                x.forEach(recurse);
            }
            return;
        }
        if (typeof x === 'number') {
            checkToken(core_1.Tokenization.reverseNumber(x));
            return;
        }
        if (typeof x === 'string') {
            core_1.Tokenization.reverseString(x).tokens.forEach(checkToken);
            return;
        }
        if (typeof x === 'object') {
            for (const [k, v] of Object.entries(x)) {
                recurse(k);
                recurse(v);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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