import { IResolvable, IResolveContext } from '../../../core';
import { Step } from '../blueprint/step';
/**
 * A symbolic reference to a value produced by another step
 *
 * Generating and consuming outputs is engine-specific. Many engines will be
 * able to support a feature like "outputs", but it's not guaranteed that
 * all of them will.
 *
 * Outputs can only be generated by engine-specific steps (CodeBuildStep instead
 * of ShellStep, etc), but can (currently) be consumed anywhere(*). When
 * an engine-specific step generates an Output, it should put a well-known
 * string and arbitrary data that is useful to the engine into the engine-specific
 * fields on the StepOutput.
 *
 * The graph blueprint will take care of dependencies and ordering, the engine
 * is responsible interpreting and rendering StepOutputs. The engine should call
 * `defineResolution()` on all outputs.
 *
 * StepOutputs currently purposely aren't part of the public API because users
 * shouldn't see the innards poking out. So, instead of keeping state on `Step`,
 * we keep side-state here in a WeakMap which can be accessed via static members
 * on `StepOutput`.
 *
 * (*) If we need to restrict this, we add the checking and erroring in the engine.
 */
export declare class StepOutput implements IResolvable {
    /**
     * Return true if the given IResolvable is a StepOutput
     */
    static isStepOutput(resolvable: IResolvable): resolvable is StepOutput;
    /**
     * Find all StepOutputs referenced in the given structure
     */
    static findAll(structure: any): StepOutput[];
    /**
     * Return the produced outputs for the given step
     */
    static producedStepOutputs(step: Step): StepOutput[];
    /**
     * Add produced outputs for the given step
     */
    static recordProducer(...outputs: StepOutput[]): void;
    /**
     * The step that produces this output
     */
    readonly step: Step;
    /**
     * Name of the engine for which this output is intended
     */
    readonly engineName: string;
    /**
     * Additional data on the output, to be interpreted by the engine
     */
    readonly engineSpecificInformation: any;
    readonly creationStack: string[];
    private resolution;
    constructor(step: Step, engineName: string, engineSpecificInformation: any);
    /**
     * Define the resolved value for this StepOutput.
     *
     * Should be called by the engine.
     */
    defineResolution(value: any): void;
    resolve(_context: IResolveContext): any;
    toString(): string;
}
