"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineGraph = void 0;
const blueprint_1 = require("../blueprint");
const graph_1 = require("./graph");
const pipeline_queries_1 = require("./pipeline-queries");
/**
 * Logic to turn the deployment blueprint into a graph
 *
 * This code makes all the decisions on how to lay out the CodePipeline
 */
class PipelineGraph {
    constructor(pipeline, props = {}) {
        var _a, _b, _c, _d, _e, _f;
        this.pipeline = pipeline;
        this.graph = graph_1.Graph.of('', { type: 'group' });
        this.added = new Map();
        this.assetNodes = new Map();
        this.assetNodesByType = new Map();
        this.stackOutputDependencies = new graph_1.DependencyBuilders();
        this._fileAssetCtr = 0;
        this._dockerAssetCtr = 0;
        this.publishTemplate = (_a = props.publishTemplate) !== null && _a !== void 0 ? _a : false;
        this.prepareStep = (_b = props.prepareStep) !== null && _b !== void 0 ? _b : true;
        this.singlePublisher = (_c = props.singlePublisherPerAssetType) !== null && _c !== void 0 ? _c : false;
        this.queries = new pipeline_queries_1.PipelineQueries(pipeline);
        if (pipeline.synth instanceof blueprint_1.Step) {
            this.synthNode = this.addBuildStep(pipeline.synth);
            if (((_e = (_d = this.synthNode) === null || _d === void 0 ? void 0 : _d.data) === null || _e === void 0 ? void 0 : _e.type) === 'step') {
                this.synthNode.data.isBuildStep = true;
            }
        }
        this.lastPreparationNode = this.synthNode;
        const cloudAssembly = (_f = pipeline.synth.primaryOutput) === null || _f === void 0 ? void 0 : _f.primaryOutput;
        if (!cloudAssembly) {
            throw new Error(`The synth step must produce the cloud assembly artifact, but doesn't: ${pipeline.synth}`);
        }
        this.cloudAssemblyFileSet = cloudAssembly;
        if (props.selfMutation) {
            const stage = graph_1.Graph.of('UpdatePipeline', { type: 'group' });
            this.graph.add(stage);
            this.selfMutateNode = graph_1.GraphNode.of('SelfMutate', { type: 'self-update' });
            stage.add(this.selfMutateNode);
            this.selfMutateNode.dependOn(this.synthNode);
            this.lastPreparationNode = this.selfMutateNode;
        }
        const waves = pipeline.waves.map(w => this.addWave(w));
        // Make sure the waves deploy sequentially
        for (let i = 1; i < waves.length; i++) {
            waves[i].dependOn(waves[i - 1]);
        }
    }
    isSynthNode(node) {
        return this.synthNode === node;
    }
    addBuildStep(step) {
        return this.addAndRecurse(step, this.topLevelGraph('Build'));
    }
    addWave(wave) {
        // If the wave only has one Stage in it, don't add an additional Graph around it
        const retGraph = wave.stages.length === 1
            ? this.addStage(wave.stages[0])
            : graph_1.Graph.of(wave.id, { type: 'group' }, wave.stages.map(s => this.addStage(s)));
        this.addPrePost(wave.pre, wave.post, retGraph);
        retGraph.dependOn(this.lastPreparationNode);
        this.graph.add(retGraph);
        return retGraph;
    }
    addStage(stage) {
        const retGraph = graph_1.Graph.of(stage.stageName, { type: 'group' });
        const stackGraphs = new Map();
        for (const stack of stage.stacks) {
            const stackGraph = graph_1.Graph.of(this.simpleStackName(stack.stackName, stage.stageName), { type: 'stack-group', stack });
            const prepareNode = this.prepareStep ? graph_1.GraphNode.of('Prepare', { type: 'prepare', stack }) : undefined;
            const deployNode = graph_1.GraphNode.of('Deploy', {
                type: 'execute',
                stack,
                captureOutputs: this.queries.stackOutputsReferenced(stack).length > 0,
            });
            retGraph.add(stackGraph);
            stackGraph.add(deployNode);
            // node or node collection that represents first point of contact in each stack
            let firstDeployNode;
            if (prepareNode) {
                stackGraph.add(prepareNode);
                deployNode.dependOn(prepareNode);
                firstDeployNode = prepareNode;
            }
            else {
                firstDeployNode = deployNode;
            }
            // add changeset steps at the stack level
            if (stack.changeSet.length > 0) {
                if (prepareNode) {
                    this.addChangeSet(stack.changeSet, prepareNode, deployNode, stackGraph);
                }
                else {
                    throw new Error('Your pipeline engine does not support changeSet steps');
                }
            }
            // add pre and post steps at the stack level
            const preNodes = this.addPrePost(stack.pre, stack.post, stackGraph);
            if (preNodes.nodes.length > 0) {
                firstDeployNode = preNodes;
            }
            stackGraphs.set(stack, stackGraph);
            const cloudAssembly = this.cloudAssemblyFileSet;
            firstDeployNode.dependOn(this.addAndRecurse(cloudAssembly.producer, retGraph));
            // add the template asset
            if (this.publishTemplate) {
                if (!stack.templateAsset) {
                    throw new Error(`"publishTemplate" is enabled, but stack ${stack.stackArtifactId} does not have a template asset`);
                }
                firstDeployNode.dependOn(this.publishAsset(stack.templateAsset));
            }
            // Depend on Assets
            // FIXME: Custom Cloud Assembly currently doesn't actually help separating
            // out templates from assets!!!
            for (const asset of stack.assets) {
                const assetNode = this.publishAsset(asset);
                firstDeployNode.dependOn(assetNode);
            }
            // Add stack output synchronization point
            if (this.queries.stackOutputsReferenced(stack).length > 0) {
                this.stackOutputDependencies.get(stack).dependOn(deployNode);
            }
        }
        for (const stack of stage.stacks) {
            for (const dep of stack.stackDependencies) {
                const stackNode = stackGraphs.get(stack);
                const depNode = stackGraphs.get(dep);
                if (!stackNode) {
                    throw new Error(`cannot find node for ${stack.stackName}`);
                }
                if (!depNode) {
                    throw new Error(`cannot find node for ${dep.stackName}`);
                }
                stackNode.dependOn(depNode);
            }
        }
        this.addPrePost(stage.pre, stage.post, retGraph);
        return retGraph;
    }
    addChangeSet(changeSet, prepareNode, deployNode, graph) {
        for (const c of changeSet) {
            const changeSetNode = this.addAndRecurse(c, graph);
            changeSetNode === null || changeSetNode === void 0 ? void 0 : changeSetNode.dependOn(prepareNode);
            deployNode.dependOn(changeSetNode);
        }
    }
    addPrePost(pre, post, parent) {
        const currentNodes = new graph_1.GraphNodeCollection(parent.nodes);
        const preNodes = new graph_1.GraphNodeCollection(new Array());
        for (const p of pre) {
            const preNode = this.addAndRecurse(p, parent);
            currentNodes.dependOn(preNode);
            preNodes.nodes.push(preNode);
        }
        for (const p of post) {
            const postNode = this.addAndRecurse(p, parent);
            postNode === null || postNode === void 0 ? void 0 : postNode.dependOn(...currentNodes.nodes);
        }
        return preNodes;
    }
    topLevelGraph(name) {
        let ret = this.graph.tryGetChild(name);
        if (!ret) {
            ret = new graph_1.Graph(name);
            this.graph.add(ret);
        }
        return ret;
    }
    addAndRecurse(step, parent) {
        if (step === PipelineGraph.NO_STEP) {
            return undefined;
        }
        const previous = this.added.get(step);
        if (previous) {
            return previous;
        }
        const node = graph_1.GraphNode.of(step.id, { type: 'step', step });
        // If the step is a source step, change the parent to a special "Source" stage
        // (CodePipeline wants it that way)
        if (step.isSource) {
            parent = this.topLevelGraph('Source');
        }
        parent.add(node);
        this.added.set(step, node);
        for (const dep of step.dependencies) {
            const producerNode = this.addAndRecurse(dep, parent);
            node.dependOn(producerNode);
        }
        // Add stack dependencies (by use of the dependency builder this also works
        // if we encounter the Step before the Stack has been properly added yet)
        if (step instanceof blueprint_1.ShellStep) {
            for (const output of Object.values(step.envFromCfnOutputs)) {
                const stack = this.queries.producingStack(output);
                this.stackOutputDependencies.get(stack).dependBy(node);
            }
        }
        return node;
    }
    publishAsset(stackAsset) {
        const assetsGraph = this.topLevelGraph('Assets');
        let assetNode = this.assetNodes.get(stackAsset.assetId);
        if (assetNode) {
            // If there's already a node publishing this asset, add as a new publishing
            // destination to the same node.
        }
        else if (this.singlePublisher && this.assetNodesByType.has(stackAsset.assetType)) {
            // If we're doing a single node per type, lookup by that
            assetNode = this.assetNodesByType.get(stackAsset.assetType);
        }
        else {
            // Otherwise add a new one
            const id = stackAsset.assetType === blueprint_1.AssetType.FILE
                ? (this.singlePublisher ? 'FileAsset' : `FileAsset${++this._fileAssetCtr}`)
                : (this.singlePublisher ? 'DockerAsset' : `DockerAsset${++this._dockerAssetCtr}`);
            assetNode = graph_1.GraphNode.of(id, { type: 'publish-assets', assets: [] });
            assetsGraph.add(assetNode);
            assetNode.dependOn(this.lastPreparationNode);
            this.assetNodesByType.set(stackAsset.assetType, assetNode);
            this.assetNodes.set(stackAsset.assetId, assetNode);
        }
        const data = assetNode.data;
        if ((data === null || data === void 0 ? void 0 : data.type) !== 'publish-assets') {
            throw new Error(`${assetNode} has the wrong data.type: ${data === null || data === void 0 ? void 0 : data.type}`);
        }
        if (!data.assets.some(a => a.assetSelector === stackAsset.assetSelector)) {
            data.assets.push(stackAsset);
        }
        return assetNode;
    }
    /**
     * Simplify the stack name by removing the `Stage-` prefix if it exists.
     */
    simpleStackName(stackName, stageName) {
        return stripPrefix(stackName, `${stageName}-`);
    }
}
exports.PipelineGraph = PipelineGraph;
/**
 * A Step object that may be used as the producer of FileSets that should not be represented in the graph
 */
PipelineGraph.NO_STEP = new class extends blueprint_1.Step {
}('NO_STEP');
function stripPrefix(s, prefix) {
    return s.startsWith(prefix) ? s.substr(prefix.length) : s;
}
//# sourceMappingURL=data:application/json;base64,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