"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGraph = exports.GraphNodeCollection = exports.Graph = exports.DependencyBuilders = exports.DependencyBuilder = exports.GraphNode = void 0;
/**
 * A library for nested graphs
 */
const javascript_1 = require("../private/javascript");
const toposort_1 = require("./toposort");
class GraphNode {
    constructor(id, props = {}) {
        this.id = id;
        this.dependencies = [];
        this.data = props.data;
    }
    static of(id, data) {
        return new GraphNode(id, { data });
    }
    /**
     * A graph-wide unique identifier for this node. Rendered by joining the IDs
     * of all ancestors with hyphens.
     */
    get uniqueId() {
        return this.ancestorPath(this.root).map(x => x.id).join('-');
    }
    /**
     * The union of all dependencies of this node and the dependencies of all
     * parent graphs.
     */
    get allDeps() {
        var _a, _b;
        const fromParent = (_b = (_a = this.parentGraph) === null || _a === void 0 ? void 0 : _a.allDeps) !== null && _b !== void 0 ? _b : [];
        return Array.from(new Set([...this.dependencies, ...fromParent]));
    }
    dependOn(...dependencies) {
        if (dependencies.includes(this)) {
            throw new Error(`Cannot add dependency on self: ${this}`);
        }
        this.dependencies.push(...dependencies.filter(javascript_1.isDefined));
    }
    ancestorPath(upTo) {
        let x = this;
        const ret = [x];
        while (x.parentGraph && x.parentGraph !== upTo) {
            x = x.parentGraph;
            ret.unshift(x);
        }
        return ret;
    }
    rootPath() {
        let x = this;
        const ret = [x];
        while (x.parentGraph) {
            x = x.parentGraph;
            ret.unshift(x);
        }
        return ret;
    }
    get root() {
        let x = this;
        while (x.parentGraph) {
            x = x.parentGraph;
        }
        return x;
    }
    get parentGraph() {
        return this._parentGraph;
    }
    /**
     * @internal
     */
    _setParentGraph(parentGraph) {
        if (this._parentGraph) {
            throw new Error('Node already has a parent');
        }
        this._parentGraph = parentGraph;
    }
    toString() {
        return `${this.constructor.name}(${this.id})`;
    }
}
exports.GraphNode = GraphNode;
/**
 * A dependency set that can be constructed partially and later finished
 *
 * It doesn't matter in what order sources and targets for the dependency
 * relationship(s) get added. This class can serve as a synchronization
 * point if the order in which graph nodes get added to the graph is not
 * well-defined.
 *
 * Useful utility during graph building.
 */
class DependencyBuilder {
    constructor() {
        this.targets = [];
        this.sources = [];
    }
    dependOn(...targets) {
        for (const target of targets) {
            for (const source of this.sources) {
                source.dependOn(target);
            }
            this.targets.push(target);
        }
        return this;
    }
    dependBy(...sources) {
        for (const source of sources) {
            for (const target of this.targets) {
                source.dependOn(target);
            }
            this.sources.push(source);
        }
        return this;
    }
}
exports.DependencyBuilder = DependencyBuilder;
class DependencyBuilders {
    constructor() {
        this.builders = new Map();
    }
    get(key) {
        const b = this.builders.get(key);
        if (b) {
            return b;
        }
        const ret = new DependencyBuilder();
        this.builders.set(key, ret);
        return ret;
    }
}
exports.DependencyBuilders = DependencyBuilders;
class Graph extends GraphNode {
    constructor(name, props = {}) {
        super(name, props);
        this.children = new Map();
        if (props.nodes) {
            this.add(...props.nodes);
        }
    }
    static of(id, data, nodes) {
        return new Graph(id, { data, nodes });
    }
    get nodes() {
        return new Set(this.children.values());
    }
    tryGetChild(name) {
        return this.children.get(name);
    }
    contains(node) {
        return this.nodes.has(node);
    }
    add(...nodes) {
        for (const node of nodes) {
            node._setParentGraph(this);
            if (this.children.has(node.id)) {
                throw new Error(`Node with duplicate id: ${node.id}`);
            }
            this.children.set(node.id, node);
        }
    }
    absorb(other) {
        this.add(...other.nodes);
    }
    /**
     * Return topologically sorted tranches of nodes at this graph level
     */
    sortedChildren() {
        // Project dependencies to current children
        const nodes = this.nodes;
        const projectedDependencies = projectDependencies(this.deepDependencies(), (node) => {
            while (!nodes.has(node) && node.parentGraph) {
                node = node.parentGraph;
            }
            return nodes.has(node) ? [node] : [];
        });
        return toposort_1.topoSort(nodes, projectedDependencies);
    }
    /**
     * Return a topologically sorted list of non-Graph nodes in the entire subgraph
     */
    sortedLeaves() {
        // Project dependencies to leaf nodes
        const descendantsMap = new Map();
        findDescendants(this);
        function findDescendants(node) {
            const ret = [];
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    ret.push(...findDescendants(child));
                }
            }
            else {
                ret.push(node);
            }
            descendantsMap.set(node, ret);
            return ret;
        }
        const projectedDependencies = projectDependencies(this.deepDependencies(), (node) => { var _a; return (_a = descendantsMap.get(node)) !== null && _a !== void 0 ? _a : []; });
        return toposort_1.topoSort(new Set(projectedDependencies.keys()), projectedDependencies);
    }
    consoleLog(indent = 0) {
        process.stdout.write(' '.repeat(indent) + this + depString(this) + '\n');
        for (const node of this.nodes) {
            if (node instanceof Graph) {
                node.consoleLog(indent + 2);
            }
            else {
                process.stdout.write(' '.repeat(indent + 2) + node + depString(node) + '\n');
            }
        }
        function depString(node) {
            if (node.dependencies.length > 0) {
                return ` -> ${Array.from(node.dependencies).join(', ')}`;
            }
            return '';
        }
    }
    /**
     * Return the union of all dependencies of the descendants of this graph
     */
    deepDependencies() {
        const ret = new Map();
        for (const node of this.nodes) {
            recurse(node);
        }
        return ret;
        function recurse(node) {
            let deps = ret.get(node);
            if (!deps) {
                ret.set(node, deps = new Set());
            }
            for (let dep of node.dependencies) {
                deps.add(dep);
            }
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    recurse(child);
                }
            }
        }
    }
    /**
     * Return all non-Graph nodes
     */
    allLeaves() {
        const ret = [];
        recurse(this);
        return new GraphNodeCollection(ret);
        function recurse(node) {
            if (node instanceof Graph) {
                for (const child of node.nodes) {
                    recurse(child);
                }
            }
            else {
                ret.push(node);
            }
        }
    }
}
exports.Graph = Graph;
/**
 * A collection of graph nodes
 */
class GraphNodeCollection {
    constructor(nodes) {
        this.nodes = Array.from(nodes);
    }
    dependOn(...dependencies) {
        for (const node of this.nodes) {
            node.dependOn(...dependencies.filter(javascript_1.isDefined));
        }
    }
    /**
    * Returns the graph node that's shared between these nodes
    */
    commonAncestor() {
        const paths = new Array();
        for (const x of this.nodes) {
            paths.push(x.rootPath());
        }
        if (paths.length === 0) {
            throw new Error('Cannot find common ancestor between an empty set of nodes');
        }
        if (paths.length === 1) {
            const path = paths[0];
            if (path.length < 2) {
                throw new Error(`Cannot find ancestor of node without ancestor: ${path[0]}`);
            }
            return path[path.length - 2];
        }
        const originalPaths = [...paths];
        // Remove the first element of every path as long as the 2nd elements are all
        // the same -- this leaves the shared element in first place.
        //
        //   A, B, C, 1, 2    }---> C
        //   A, B, C, 3       }
        while (paths.every(path => paths[0].length >= 2 && path.length >= 2 && path[1] === paths[0][1])) {
            for (const path of paths) {
                path.shift();
            }
        }
        // If any of the paths are left with 1 element, there's no shared parent.
        if (paths.some(path => path.length < 2)) {
            throw new Error(`Could not determine a shared parent between nodes: ${originalPaths.map(nodes => nodes.map(n => n.id).join('/'))}`);
        }
        return paths[0][0];
    }
}
exports.GraphNodeCollection = GraphNodeCollection;
/**
 * Dependency map of nodes in this graph, taking into account dependencies between nodes in subgraphs
 *
 * Guaranteed to return an entry in the map for every node in the current graph.
 */
function projectDependencies(dependencies, project) {
    // Project keys
    for (const node of dependencies.keys()) {
        const projectedNodes = project(node);
        if (projectedNodes.length === 1 && projectedNodes[0] === node) {
            continue;
        } // Nothing to do, just for efficiency
        const deps = javascript_1.extract(dependencies, node);
        for (const projectedNode of projectedNodes) {
            javascript_1.addAll(dependencies.get(projectedNode), deps);
        }
    }
    // Project values. Ignore self-dependencies, they were just between nodes that were collapsed into the same node.
    for (const [node, deps] of dependencies.entries()) {
        const depset = new Set(javascript_1.flatMap(deps, project));
        depset.delete(node);
        dependencies.set(node, depset);
    }
    return dependencies;
}
function isGraph(x) {
    return x instanceof Graph;
}
exports.isGraph = isGraph;
//# sourceMappingURL=data:application/json;base64,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