"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.dockerCredentialsInstallCommands = exports.DockerCredentialUsage = exports.DockerCredential = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
/**
 * Represents credentials used to access a Docker registry.
 */
class DockerCredential {
    constructor(usages) {
        this.usages = usages;
    }
    /**
     * Creates a DockerCredential for DockerHub.
     * Convenience method for `customRegistry('https://index.docker.io/v1/', opts)`.
     */
    static dockerHub(secret, opts = {}) {
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_ISecret(secret);
        jsiiDeprecationWarnings.monocdk_pipelines_ExternalDockerCredentialOptions(opts);
        return new ExternalDockerCredential('https://index.docker.io/v1/', secret, opts);
    }
    /**
     * Creates a DockerCredential for a registry, based on its domain name (e.g., 'www.example.com').
     */
    static customRegistry(registryDomain, secret, opts = {}) {
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_ISecret(secret);
        jsiiDeprecationWarnings.monocdk_pipelines_ExternalDockerCredentialOptions(opts);
        return new ExternalDockerCredential(registryDomain, secret, opts);
    }
    /**
     * Creates a DockerCredential for one or more ECR repositories.
     *
     * NOTE - All ECR repositories in the same account and region share a domain name
     * (e.g., 0123456789012.dkr.ecr.eu-west-1.amazonaws.com), and can only have one associated
     * set of credentials (and DockerCredential). Attempting to associate one set of credentials
     * with one ECR repo and another with another ECR repo in the same account and region will
     * result in failures when using these credentials in the pipeline.
     */
    static ecr(repositories, opts) {
        jsiiDeprecationWarnings.monocdk_pipelines_EcrDockerCredentialOptions(opts);
        return new EcrDockerCredential(repositories, opts !== null && opts !== void 0 ? opts : {});
    }
    /**
     * Determines if this credential is relevant to the input usage.
     * @internal
     */
    _applicableForUsage(usage) {
        return !this.usages || this.usages.includes(usage);
    }
}
exports.DockerCredential = DockerCredential;
_a = JSII_RTTI_SYMBOL_1;
DockerCredential[_a] = { fqn: "monocdk.pipelines.DockerCredential", version: "1.149.0" };
/** Defines which stages of a pipeline require the specified credentials */
var DockerCredentialUsage;
(function (DockerCredentialUsage) {
    /** Synth/Build */
    DockerCredentialUsage["SYNTH"] = "SYNTH";
    /** Self-update */
    DockerCredentialUsage["SELF_UPDATE"] = "SELF_UPDATE";
    /** Asset publishing */
    DockerCredentialUsage["ASSET_PUBLISHING"] = "ASSET_PUBLISHING";
})(DockerCredentialUsage = exports.DockerCredentialUsage || (exports.DockerCredentialUsage = {}));
;
/** DockerCredential defined by registry domain and a secret */
class ExternalDockerCredential extends DockerCredential {
    constructor(registryDomain, secret, opts) {
        super(opts.usages);
        this.registryDomain = registryDomain;
        this.secret = secret;
        this.opts = opts;
    }
    grantRead(grantee, usage) {
        var _b;
        if (!this._applicableForUsage(usage)) {
            return;
        }
        if (this.opts.assumeRole) {
            grantee.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [this.opts.assumeRole.roleArn],
            }));
        }
        const role = (_b = this.opts.assumeRole) !== null && _b !== void 0 ? _b : grantee;
        this.secret.grantRead(role);
    }
    _renderCdkAssetsConfig() {
        var _b;
        return {
            [this.registryDomain]: {
                secretsManagerSecretId: this.secret.secretArn,
                secretsUsernameField: this.opts.secretUsernameField,
                secretsPasswordField: this.opts.secretPasswordField,
                assumeRoleArn: (_b = this.opts.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            },
        };
    }
}
/** DockerCredential defined by a set of ECR repositories in the same account & region */
class EcrDockerCredential extends DockerCredential {
    constructor(repositories, opts) {
        super(opts.usages);
        this.repositories = repositories;
        this.opts = opts;
        if (repositories.length === 0) {
            throw new Error('must supply at least one `ecr.IRepository` to create an `EcrDockerCredential`');
        }
        this.registryDomain = core_1.Fn.select(0, core_1.Fn.split('/', repositories[0].repositoryUri));
    }
    grantRead(grantee, usage) {
        var _b;
        if (!this._applicableForUsage(usage)) {
            return;
        }
        if (this.opts.assumeRole) {
            grantee.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [this.opts.assumeRole.roleArn],
            }));
        }
        const role = (_b = this.opts.assumeRole) !== null && _b !== void 0 ? _b : grantee;
        this.repositories.forEach(repo => repo.grantPull(role));
    }
    _renderCdkAssetsConfig() {
        var _b;
        return {
            [this.registryDomain]: {
                ecrRepository: true,
                assumeRoleArn: (_b = this.opts.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            },
        };
    }
}
/**
 * Creates a set of OS-specific buildspec installation commands for setting up the given
 * registries and associated credentials.
 *
 * @param registries - Registries to configure credentials for. It is an error to provide
 * multiple registries for the same domain.
 * @param osType - (optional) Defaults to Linux.
 * @returns An array of commands to configure cdk-assets to use these credentials.
 */
function dockerCredentialsInstallCommands(usage, registries, osType) {
    const relevantRegistries = (registries !== null && registries !== void 0 ? registries : []).filter(reg => reg._applicableForUsage(usage));
    if (!relevantRegistries || relevantRegistries.length === 0) {
        return [];
    }
    const domainCredentials = relevantRegistries.reduce(function (map, registry) {
        Object.assign(map, registry._renderCdkAssetsConfig());
        return map;
    }, {});
    const cdkAssetsConfigFile = {
        version: '1.0',
        domainCredentials,
    };
    const windowsCommands = [
        'mkdir %USERPROFILE%\\.cdk',
        `echo '${JSON.stringify(cdkAssetsConfigFile)}' > %USERPROFILE%\\.cdk\\cdk-docker-creds.json`,
    ];
    const linuxCommands = [
        'mkdir $HOME/.cdk',
        `echo '${JSON.stringify(cdkAssetsConfigFile)}' > $HOME/.cdk/cdk-docker-creds.json`,
    ];
    if (osType === 'both') {
        return [
            // These tags are magic and will be stripped when rendering the project
            ...windowsCommands.map(c => `!WINDOWS!${c}`),
            ...linuxCommands.map(c => `!LINUX!${c}`),
        ];
    }
    else if (osType === ec2.OperatingSystemType.WINDOWS) {
        return windowsCommands;
    }
    else {
        return linuxCommands;
    }
}
exports.dockerCredentialsInstallCommands = dockerCredentialsInstallCommands;
//# sourceMappingURL=data:application/json;base64,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