"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeCodeBuildOptions = exports.CodeBuildFactory = void 0;
const fs = require("fs");
const path = require("path");
const codebuild = require("../../../../aws-codebuild");
const codepipeline_actions = require("../../../../aws-codepipeline-actions");
const ec2 = require("../../../../aws-ec2");
const core_1 = require("../../../../core");
const constructs_1 = require("constructs");
const pipeline_queries_1 = require("../../helpers-internal/pipeline-queries");
const step_output_1 = require("../../helpers-internal/step-output");
const construct_internals_1 = require("../../private/construct-internals");
const identifiers_1 = require("../../private/identifiers");
const javascript_1 = require("../../private/javascript");
const buildspecs_1 = require("./buildspecs");
/**
 * Produce a CodeBuild project from a ShellStep and some CodeBuild-specific customizations
 *
 * The functionality here is shared between the `CodePipeline` translating a `ShellStep` into
 * a CodeBuild project, as well as the `CodeBuildStep` straight up.
 */
class CodeBuildFactory {
    constructor(constructId, props) {
        var _a;
        this.constructId = constructId;
        this.props = props;
        this.stepId = (_a = props.stepId) !== null && _a !== void 0 ? _a : constructId;
    }
    // eslint-disable-next-line max-len
    static fromShellStep(constructId, shellStep, additional) {
        return new CodeBuildFactory(constructId, {
            commands: shellStep.commands,
            env: shellStep.env,
            envFromCfnOutputs: shellStep.envFromCfnOutputs,
            inputs: shellStep.inputs,
            outputs: shellStep.outputs,
            stepId: shellStep.id,
            installCommands: shellStep.installCommands,
            producedStepOutputs: step_output_1.StepOutput.producedStepOutputs(shellStep),
            ...additional,
        });
    }
    static fromCodeBuildStep(constructId, step, additional) {
        const factory = CodeBuildFactory.fromShellStep(constructId, step, {
            projectName: step.projectName,
            role: step.role,
            ...additional,
            projectOptions: mergeCodeBuildOptions(additional === null || additional === void 0 ? void 0 : additional.projectOptions, {
                buildEnvironment: step.buildEnvironment,
                rolePolicy: step.rolePolicyStatements,
                securityGroups: step.securityGroups,
                partialBuildSpec: step.partialBuildSpec,
                vpc: step.vpc,
                subnetSelection: step.subnetSelection,
                timeout: step.timeout,
            }),
        });
        return {
            produceAction: (stage, options) => {
                const result = factory.produceAction(stage, options);
                if (result.project) {
                    step._setProject(result.project);
                }
                return result;
            },
        };
    }
    get project() {
        if (!this._project) {
            throw new Error('Project becomes available after produce() has been called');
        }
        return this._project;
    }
    produceAction(stage, options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        const projectOptions = mergeCodeBuildOptions(options.codeBuildDefaults, this.props.projectOptions);
        const inputs = (_a = this.props.inputs) !== null && _a !== void 0 ? _a : [];
        const outputs = (_b = this.props.outputs) !== null && _b !== void 0 ? _b : [];
        const mainInput = inputs.find(x => x.directory === '.');
        const extraInputs = inputs.filter(x => x.directory !== '.');
        const inputArtifact = mainInput
            ? options.artifacts.toCodePipeline(mainInput.fileSet)
            : options.fallbackArtifact;
        const extraInputArtifacts = extraInputs.map(x => options.artifacts.toCodePipeline(x.fileSet));
        const outputArtifacts = outputs.map(x => options.artifacts.toCodePipeline(x.fileSet));
        if (!inputArtifact) {
            // This should actually never happen because CodeBuild projects shouldn't be added before the
            // Source, which always produces at least an artifact.
            throw new Error(`CodeBuild action '${this.stepId}' requires an input (and the pipeline doesn't have a Source to fall back to). Add an input or a pipeline source.`);
        }
        const installCommands = [
            ...generateInputArtifactLinkCommands(options.artifacts, extraInputs),
            ...(_c = this.props.installCommands) !== null && _c !== void 0 ? _c : [],
        ];
        const buildSpecHere = codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                install: ((_d = installCommands.length) !== null && _d !== void 0 ? _d : 0) > 0 ? { commands: installCommands } : undefined,
                build: this.props.commands.length > 0 ? { commands: this.props.commands } : undefined,
            },
            artifacts: javascript_1.noEmptyObject(renderArtifactsBuildSpec(options.artifacts, (_e = this.props.outputs) !== null && _e !== void 0 ? _e : [])),
        });
        // Partition environment variables into environment variables that can go on the project
        // and environment variables that MUST go in the pipeline (those that reference CodePipeline variables)
        const env = javascript_1.noUndefined((_f = this.props.env) !== null && _f !== void 0 ? _f : {});
        const [actionEnvs, projectEnvs] = javascript_1.partition(Object.entries(env !== null && env !== void 0 ? env : {}), ([, v]) => containsPipelineVariable(v));
        const environment = mergeBuildEnvironments((_g = projectOptions === null || projectOptions === void 0 ? void 0 : projectOptions.buildEnvironment) !== null && _g !== void 0 ? _g : {}, {
            environmentVariables: javascript_1.noEmptyObject(javascript_1.mapValues(javascript_1.mkdict(projectEnvs), value => ({ value }))),
        });
        const fullBuildSpec = (projectOptions === null || projectOptions === void 0 ? void 0 : projectOptions.partialBuildSpec) ? codebuild.mergeBuildSpecs(projectOptions.partialBuildSpec, buildSpecHere)
            : buildSpecHere;
        const osFromEnvironment = environment.buildImage && environment.buildImage instanceof codebuild.WindowsBuildImage
            ? ec2.OperatingSystemType.WINDOWS
            : ec2.OperatingSystemType.LINUX;
        const actualBuildSpec = filterBuildSpecCommands(fullBuildSpec, osFromEnvironment);
        const scope = (_h = this.props.scope) !== null && _h !== void 0 ? _h : options.scope;
        let projectBuildSpec;
        if (this.props.passBuildSpecViaCloudAssembly) {
            // Write to disk and replace with a reference
            const relativeSpecFile = `buildspec-${constructs_1.Node.of(scope).addr}-${this.constructId}.yaml`;
            const absSpecFile = path.join(construct_internals_1.cloudAssemblyBuildSpecDir(scope), relativeSpecFile);
            // This should resolve to a pure JSON string. If it resolves to an object, it's a CFN
            // expression, and we can't support that yet. Maybe someday if we think really hard about it.
            const fileContents = core_1.Stack.of(scope).resolve(actualBuildSpec.toBuildSpec());
            if (typeof fileContents !== 'string') {
                throw new Error(`This BuildSpec contains CloudFormation references and is supported by publishInParallel=false: ${JSON.stringify(fileContents, undefined, 2)}`);
            }
            fs.writeFileSync(absSpecFile, fileContents, { encoding: 'utf-8' });
            projectBuildSpec = codebuild.BuildSpec.fromSourceFilename(relativeSpecFile);
        }
        else {
            projectBuildSpec = actualBuildSpec;
        }
        // A hash over the values that make the CodeBuild Project unique (and necessary
        // to restart the pipeline if one of them changes). projectName is not necessary to include
        // here because the pipeline will definitely restart if projectName changes.
        // (Resolve tokens)
        const projectConfigHash = identifiers_1.hash(core_1.Stack.of(scope).resolve({
            environment: serializeBuildEnvironment(environment),
            buildSpecString: actualBuildSpec.toBuildSpec(),
        }));
        const actionName = (_j = options.actionName) !== null && _j !== void 0 ? _j : this.stepId;
        let projectScope = scope;
        if ((_k = this.props.additionalConstructLevel) !== null && _k !== void 0 ? _k : true) {
            projectScope = construct_internals_1.obtainScope(scope, actionName);
        }
        const safePipelineName = core_1.Token.isUnresolved(options.pipeline.pipeline.pipelineName)
            ? `${core_1.Stack.of(options.pipeline).stackName}/${constructs_1.Node.of(options.pipeline.pipeline).id}`
            : options.pipeline.pipeline.pipelineName;
        const project = new codebuild.PipelineProject(projectScope, this.constructId, {
            projectName: this.props.projectName,
            description: `Pipeline step ${safePipelineName}/${stage.stageName}/${actionName}`.substring(0, 255),
            environment,
            vpc: projectOptions.vpc,
            subnetSelection: projectOptions.subnetSelection,
            securityGroups: projectOptions.securityGroups,
            buildSpec: projectBuildSpec,
            role: this.props.role,
            timeout: projectOptions.timeout,
        });
        if (this.props.additionalDependable) {
            project.node.addDependency(this.props.additionalDependable);
        }
        if (projectOptions.rolePolicy !== undefined) {
            projectOptions.rolePolicy.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        const queries = new pipeline_queries_1.PipelineQueries(options.pipeline);
        const stackOutputEnv = javascript_1.mapValues((_l = this.props.envFromCfnOutputs) !== null && _l !== void 0 ? _l : {}, outputRef => `#{${identifiers_1.stackVariableNamespace(queries.producingStack(outputRef))}.${outputRef.outputName}}`);
        const configHashEnv = options.beforeSelfMutation
            ? { _PROJECT_CONFIG_HASH: projectConfigHash }
            : {};
        stage.addAction(new codepipeline_actions.CodeBuildAction({
            actionName: actionName,
            input: inputArtifact,
            extraInputs: extraInputArtifacts,
            outputs: outputArtifacts,
            project,
            runOrder: options.runOrder,
            variablesNamespace: options.variablesNamespace,
            // Inclusion of the hash here will lead to the pipeline structure for any changes
            // made the config of the underlying CodeBuild Project.
            // Hence, the pipeline will be restarted. This is necessary if the users
            // adds (for example) build or test commands to the buildspec.
            environmentVariables: javascript_1.noEmptyObject(cbEnv({
                ...javascript_1.mkdict(actionEnvs),
                ...configHashEnv,
                ...stackOutputEnv,
            })),
        }));
        this._project = project;
        return { runOrdersConsumed: 1, project };
    }
}
exports.CodeBuildFactory = CodeBuildFactory;
/**
 * Generate commands to move additional input artifacts into the right place
 */
function generateInputArtifactLinkCommands(artifacts, inputs) {
    return inputs.map(input => {
        const fragments = [];
        fragments.push(`[ ! -d "${input.directory}" ] || { echo 'additionalInputs: "${input.directory}" must not exist yet. If you want to merge multiple artifacts, use a "cp" command.'; exit 1; }`);
        const parentDirectory = path.dirname(input.directory);
        if (!['.', '..'].includes(parentDirectory)) {
            fragments.push(`mkdir -p -- "${parentDirectory}"`);
        }
        const artifact = artifacts.toCodePipeline(input.fileSet);
        fragments.push(`ln -s -- "$CODEBUILD_SRC_DIR_${artifact.artifactName}" "${input.directory}"`);
        return fragments.join(' && ');
    });
}
function renderArtifactsBuildSpec(artifactMap, outputs) {
    // save the generated files in the output artifact
    // This part of the buildspec has to look completely different depending on whether we're
    // using secondary artifacts or not.
    if (outputs.length === 0) {
        return {};
    }
    if (outputs.length === 1) {
        return {
            'base-directory': outputs[0].directory,
            'files': '**/*',
        };
    }
    const secondary = {};
    for (const output of outputs) {
        const art = artifactMap.toCodePipeline(output.fileSet);
        if (!art.artifactName) {
            throw new Error('You must give the output artifact a name');
        }
        secondary[art.artifactName] = {
            'base-directory': output.directory,
            'files': '**/*',
        };
    }
    return { 'secondary-artifacts': secondary };
}
function mergeCodeBuildOptions(...opts) {
    const xs = [{}, ...opts.filter(isDefined)];
    while (xs.length > 1) {
        const [a, b] = xs.splice(xs.length - 2, 2);
        xs.push(merge2(a, b));
    }
    return xs[0];
    function merge2(a, b) {
        var _a, _b, _c, _d, _e, _f, _g;
        return {
            buildEnvironment: mergeBuildEnvironments(a.buildEnvironment, b.buildEnvironment),
            rolePolicy: definedArray([...(_a = a.rolePolicy) !== null && _a !== void 0 ? _a : [], ...(_b = b.rolePolicy) !== null && _b !== void 0 ? _b : []]),
            securityGroups: definedArray([...(_c = a.securityGroups) !== null && _c !== void 0 ? _c : [], ...(_d = b.securityGroups) !== null && _d !== void 0 ? _d : []]),
            partialBuildSpec: buildspecs_1.mergeBuildSpecs(a.partialBuildSpec, b.partialBuildSpec),
            vpc: (_e = b.vpc) !== null && _e !== void 0 ? _e : a.vpc,
            subnetSelection: (_f = b.subnetSelection) !== null && _f !== void 0 ? _f : a.subnetSelection,
            timeout: (_g = b.timeout) !== null && _g !== void 0 ? _g : a.timeout,
        };
    }
}
exports.mergeCodeBuildOptions = mergeCodeBuildOptions;
function mergeBuildEnvironments(a, b) {
    var _a, _b, _c;
    if (!a || !b) {
        return a !== null && a !== void 0 ? a : b;
    }
    return {
        buildImage: (_a = b.buildImage) !== null && _a !== void 0 ? _a : a.buildImage,
        computeType: (_b = b.computeType) !== null && _b !== void 0 ? _b : a.computeType,
        environmentVariables: {
            ...a.environmentVariables,
            ...b.environmentVariables,
        },
        privileged: (_c = b.privileged) !== null && _c !== void 0 ? _c : a.privileged,
    };
}
function isDefined(x) {
    return x !== undefined;
}
/**
 * Serialize a build environment to data (get rid of constructs & objects), so we can JSON.stringify it
 */
function serializeBuildEnvironment(env) {
    var _a, _b, _c, _d, _e;
    return {
        privileged: env.privileged,
        environmentVariables: env.environmentVariables,
        type: (_a = env.buildImage) === null || _a === void 0 ? void 0 : _a.type,
        imageId: (_b = env.buildImage) === null || _b === void 0 ? void 0 : _b.imageId,
        computeType: env.computeType,
        imagePullPrincipalType: (_c = env.buildImage) === null || _c === void 0 ? void 0 : _c.imagePullPrincipalType,
        secretsManagerArn: (_e = (_d = env.buildImage) === null || _d === void 0 ? void 0 : _d.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.secretArn,
    };
}
/**
 * Whether the given string contains a reference to a CodePipeline variable
 */
function containsPipelineVariable(s) {
    return !!s.match(/#\{[^}]+\}/) || step_output_1.StepOutput.findAll(s).length > 0;
}
/**
 * Turn a collection into a collection of CodePipeline environment variables
 */
function cbEnv(xs) {
    return javascript_1.mkdict(Object.entries(xs)
        .filter(([, v]) => v !== undefined)
        .map(([k, v]) => [k, { value: v }]));
}
function definedArray(xs) {
    return xs.length > 0 ? xs : undefined;
}
/**
 * If lines in the buildspec start with '!WINDOWS!' or '!LINUX!', only render them on that platform.
 *
 * Very private protocol for now, but may come in handy in other libraries as well.
 */
function filterBuildSpecCommands(buildSpec, osType) {
    if (!buildSpec.isImmediate) {
        return buildSpec;
    }
    const spec = buildSpec.spec;
    const winTag = '!WINDOWS!';
    const linuxTag = '!LINUX!';
    const expectedTag = osType === ec2.OperatingSystemType.WINDOWS ? winTag : linuxTag;
    return codebuild.BuildSpec.fromObject(recurse(spec));
    function recurse(x) {
        if (Array.isArray(x)) {
            const ret = [];
            for (const el of x) {
                const [tag, payload] = extractTag(el);
                if (tag === undefined || tag === expectedTag) {
                    ret.push(payload);
                }
            }
            return ret;
        }
        if (x && typeof x === 'object') {
            return javascript_1.mapValues(x, recurse);
        }
        return x;
    }
    function extractTag(x) {
        if (typeof x !== 'string') {
            return [undefined, x];
        }
        for (const tag of [winTag, linuxTag]) {
            if (x.startsWith(tag)) {
                return [tag, x.substr(tag.length)];
            }
        }
        return [undefined, x];
    }
}
//# sourceMappingURL=data:application/json;base64,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