"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipeline = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const cb = require("../../../aws-codebuild");
const cp = require("../../../aws-codepipeline");
const cpa = require("../../../aws-codepipeline-actions");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const cxapi = require("../../../cx-api");
const constructs_1 = require("constructs");
const blueprint_1 = require("../blueprint");
const docker_credentials_1 = require("../docker-credentials");
const helpers_internal_1 = require("../helpers-internal");
const main_1 = require("../main");
const asset_singleton_role_1 = require("../private/asset-singleton-role");
const cli_version_1 = require("../private/cli-version");
const construct_internals_1 = require("../private/construct-internals");
const fs_1 = require("../private/fs");
const identifiers_1 = require("../private/identifiers");
const javascript_1 = require("../private/javascript");
const template_configuration_1 = require("../private/template-configuration");
const artifact_map_1 = require("./artifact-map");
const codebuild_step_1 = require("./codebuild-step");
const codebuild_factory_1 = require("./private/codebuild-factory");
const outputs_1 = require("./private/outputs");
/**
 * A CDK Pipeline that uses CodePipeline to deploy CDK apps
 *
 * This is a `Pipeline` with its `engine` property set to
 * `CodePipelineEngine`, and exists for nicer ergonomics for
 * users that don't need to switch out engines.
 */
class CodePipeline extends main_1.PipelineBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, props);
        this.props = props;
        this.artifacts = new artifact_map_1.ArtifactMap();
        /**
         * Asset roles shared for publishing
         */
        this.assetCodeBuildRoles = {};
        /**
         * Per asset type, the target role ARNs that need to be assumed
         */
        this.assetPublishingRoles = {};
        jsiiDeprecationWarnings.monocdk_pipelines_CodePipelineProps(props);
        this.selfMutation = (_b = props.selfMutation) !== null && _b !== void 0 ? _b : true;
        this.dockerCredentials = (_c = props.dockerCredentials) !== null && _c !== void 0 ? _c : [];
        this.singlePublisherPerAssetType = !((_d = props.publishAssetsInParallel) !== null && _d !== void 0 ? _d : true);
        this.cliVersion = (_e = props.cliVersion) !== null && _e !== void 0 ? _e : cli_version_1.preferredCliVersion();
    }
    /**
     * The CodeBuild project that performs the Synth
     *
     * Only available after the pipeline has been built.
     */
    get synthProject() {
        if (!this._synthProject) {
            throw new Error('Call pipeline.buildPipeline() before reading this property');
        }
        return this._synthProject;
    }
    /**
     * The CodePipeline pipeline that deploys the CDK app
     *
     * Only available after the pipeline has been built.
     */
    get pipeline() {
        if (!this._pipeline) {
            throw new Error('Pipeline not created yet');
        }
        return this._pipeline;
    }
    doBuildPipeline() {
        var _b;
        if (this._pipeline) {
            throw new Error('Pipeline already created');
        }
        this._myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        if (this.props.codePipeline) {
            if (this.props.pipelineName) {
                throw new Error('Cannot set \'pipelineName\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            if (this.props.crossAccountKeys !== undefined) {
                throw new Error('Cannot set \'crossAccountKeys\' if an existing CodePipeline is given using \'codePipeline\'');
            }
            this._pipeline = this.props.codePipeline;
        }
        else {
            this._pipeline = new cp.Pipeline(this, 'Pipeline', {
                pipelineName: this.props.pipelineName,
                crossAccountKeys: (_b = this.props.crossAccountKeys) !== null && _b !== void 0 ? _b : false,
                reuseCrossRegionSupportStacks: this.props.reuseCrossRegionSupportStacks,
                // This is necessary to make self-mutation work (deployments are guaranteed
                // to happen only after the builds of the latest pipeline definition).
                restartExecutionOnUpdate: true,
            });
        }
        const graphFromBp = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: this.selfMutation,
            singlePublisherPerAssetType: this.singlePublisherPerAssetType,
        });
        this._cloudAssemblyFileSet = graphFromBp.cloudAssemblyFileSet;
        this.pipelineStagesAndActionsFromGraph(graphFromBp);
    }
    get myCxAsmRoot() {
        if (!this._myCxAsmRoot) {
            throw new Error('Can\'t read \'myCxAsmRoot\' if build deployment not called yet');
        }
        return this._myCxAsmRoot;
    }
    /**
     * Scope for Assets-related resources.
     *
     * Purely exists for construct tree backwards compatibility with legacy pipelines
     */
    get assetsScope() {
        return construct_internals_1.obtainScope(this, 'Assets');
    }
    pipelineStagesAndActionsFromGraph(structure) {
        var _b, _c;
        // Translate graph into Pipeline Stages and Actions
        let beforeSelfMutation = this.selfMutation;
        for (const stageNode of javascript_1.flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            // Group our ordered tranches into blocks of 50.
            // We can map these onto stages without exceeding the capacity of a Stage.
            const chunks = chunkTranches(50, stageNode.sortedLeaves());
            const actionsOverflowStage = chunks.length > 1;
            for (const [i, tranches] of javascript_1.enumerate(chunks)) {
                const stageName = actionsOverflowStage ? `${stageNode.id}.${i + 1}` : stageNode.id;
                const pipelineStage = this.pipeline.addStage({ stageName });
                const sharedParent = new helpers_internal_1.GraphNodeCollection(javascript_1.flatten(tranches)).commonAncestor();
                let runOrder = 1;
                for (const tranche of tranches) {
                    const runOrdersConsumed = [0];
                    for (const node of tranche) {
                        const factory = this.actionFromNode(node);
                        const nodeType = this.nodeTypeFromNode(node);
                        const name = identifiers_1.actionName(node, sharedParent);
                        const variablesNamespace = ((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) === 'step'
                            ? outputs_1.namespaceStepOutputs(node.data.step, pipelineStage, name)
                            : undefined;
                        const result = factory.produceAction(pipelineStage, {
                            actionName: name,
                            runOrder,
                            artifacts: this.artifacts,
                            scope: construct_internals_1.obtainScope(this.pipeline, stageName),
                            fallbackArtifact: this._fallbackArtifact,
                            pipeline: this,
                            // If this step happens to produce a CodeBuild job, set the default options
                            codeBuildDefaults: nodeType ? this.codeBuildDefaultsFor(nodeType) : undefined,
                            beforeSelfMutation,
                            variablesNamespace,
                        });
                        if (((_c = node.data) === null || _c === void 0 ? void 0 : _c.type) === 'self-update') {
                            beforeSelfMutation = false;
                        }
                        this.postProcessNode(node, result);
                        runOrdersConsumed.push(result.runOrdersConsumed);
                    }
                    runOrder += Math.max(...runOrdersConsumed);
                }
            }
        }
    }
    /**
     * Do additional things after the action got added to the pipeline
     *
     * Some minor state manipulation of CodeBuild projects and pipeline
     * artifacts.
     */
    postProcessNode(node, result) {
        var _b, _c, _d;
        const nodeType = this.nodeTypeFromNode(node);
        if (result.project) {
            const dockerUsage = dockerUsageFromCodeBuild(nodeType !== null && nodeType !== void 0 ? nodeType : CodeBuildProjectType.STEP);
            if (dockerUsage) {
                for (const c of this.dockerCredentials) {
                    c.grantRead(result.project, dockerUsage);
                }
            }
            if (nodeType === CodeBuildProjectType.SYNTH) {
                this._synthProject = result.project;
            }
        }
        if (((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) === 'step' && ((_c = node.data.step.primaryOutput) === null || _c === void 0 ? void 0 : _c.primaryOutput) && !this._fallbackArtifact) {
            this._fallbackArtifact = this.artifacts.toCodePipeline((_d = node.data.step.primaryOutput) === null || _d === void 0 ? void 0 : _d.primaryOutput);
        }
    }
    /**
     * Make an action from the given node and/or step
     */
    actionFromNode(node) {
        var _b, _c;
        switch ((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`actionFromNode: did not expect to get group nodes: ${(_c = node.data) === null || _c === void 0 ? void 0 : _c.type}`);
            case 'self-update':
                return this.selfMutateAction();
            case 'publish-assets':
                return this.publishAssetsAction(node, node.data.assets);
            case 'prepare':
                return this.createChangeSetAction(node.data.stack);
            case 'execute':
                return this.executeChangeSetAction(node.data.stack, node.data.captureOutputs);
            case 'step':
                return this.actionFromStep(node, node.data.step);
        }
    }
    /**
     * Take a Step and turn it into a CodePipeline Action
     *
     * There are only 3 types of Steps we need to support:
     *
     * - Shell (generic)
     * - ManualApproval (generic)
     * - CodePipelineActionFactory (CodePipeline-specific)
     *
     * The rest is expressed in terms of these 3, or in terms of graph nodes
     * which are handled elsewhere.
     */
    actionFromStep(node, step) {
        const nodeType = this.nodeTypeFromNode(node);
        // CodePipeline-specific steps first -- this includes Sources
        if (isCodePipelineActionFactory(step)) {
            return step;
        }
        // Now built-in steps
        if (step instanceof blueprint_1.ShellStep || step instanceof codebuild_step_1.CodeBuildStep) {
            // The 'CdkBuildProject' will be the construct ID of the CodeBuild project, necessary for backwards compat
            let constructId = nodeType === CodeBuildProjectType.SYNTH
                ? 'CdkBuildProject'
                : step.id;
            return step instanceof codebuild_step_1.CodeBuildStep
                ? codebuild_factory_1.CodeBuildFactory.fromCodeBuildStep(constructId, step)
                : codebuild_factory_1.CodeBuildFactory.fromShellStep(constructId, step);
        }
        if (step instanceof blueprint_1.ManualApprovalStep) {
            return {
                produceAction: (stage, options) => {
                    stage.addAction(new cpa.ManualApprovalAction({
                        actionName: options.actionName,
                        runOrder: options.runOrder,
                        additionalInformation: step.comment,
                    }));
                    return { runOrdersConsumed: 1 };
                },
            };
        }
        throw new Error(`Deployment step '${step}' is not supported for CodePipeline-backed pipelines`);
    }
    createChangeSetAction(stack) {
        const changeSetName = 'PipelineChange';
        const templateArtifact = this.artifacts.toCodePipeline(this._cloudAssemblyFileSet);
        const templateConfigurationPath = this.writeTemplateConfiguration(stack);
        const region = stack.region !== core_1.Stack.of(this).region ? stack.region : undefined;
        const account = stack.account !== core_1.Stack.of(this).account ? stack.account : undefined;
        const relativeTemplatePath = path.relative(this.myCxAsmRoot, stack.absoluteTemplatePath);
        return {
            produceAction: (stage, options) => {
                stage.addAction(new cpa.CloudFormationCreateReplaceChangeSetAction({
                    actionName: options.actionName,
                    runOrder: options.runOrder,
                    changeSetName,
                    stackName: stack.stackName,
                    templatePath: templateArtifact.atPath(fs_1.toPosixPath(relativeTemplatePath)),
                    adminPermissions: true,
                    role: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.assumeRoleArn),
                    deploymentRole: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.executionRoleArn),
                    region: region,
                    templateConfiguration: templateConfigurationPath
                        ? templateArtifact.atPath(fs_1.toPosixPath(templateConfigurationPath))
                        : undefined,
                    cfnCapabilities: [core_1.CfnCapabilities.NAMED_IAM, core_1.CfnCapabilities.AUTO_EXPAND],
                }));
                return { runOrdersConsumed: 1 };
            },
        };
    }
    executeChangeSetAction(stack, captureOutputs) {
        const changeSetName = 'PipelineChange';
        const region = stack.region !== core_1.Stack.of(this).region ? stack.region : undefined;
        const account = stack.account !== core_1.Stack.of(this).account ? stack.account : undefined;
        return {
            produceAction: (stage, options) => {
                stage.addAction(new cpa.CloudFormationExecuteChangeSetAction({
                    actionName: options.actionName,
                    runOrder: options.runOrder,
                    changeSetName,
                    stackName: stack.stackName,
                    role: this.roleFromPlaceholderArn(this.pipeline, region, account, stack.assumeRoleArn),
                    region: region,
                    variablesNamespace: captureOutputs ? identifiers_1.stackVariableNamespace(stack) : undefined,
                }));
                return { runOrdersConsumed: 1 };
            },
        };
    }
    selfMutateAction() {
        var _b;
        const installSuffix = this.cliVersion ? `@${this.cliVersion}` : '';
        const pipelineStack = core_1.Stack.of(this.pipeline);
        const pipelineStackIdentifier = (_b = pipelineStack.node.path) !== null && _b !== void 0 ? _b : pipelineStack.stackName;
        const step = new codebuild_step_1.CodeBuildStep('SelfMutate', {
            projectName: javascript_1.maybeSuffix(this.props.pipelineName, '-selfupdate'),
            input: this._cloudAssemblyFileSet,
            installCommands: [
                `npm install -g aws-cdk${installSuffix}`,
            ],
            commands: [
                `cdk -a ${fs_1.toPosixPath(construct_internals_1.embeddedAsmPath(this.pipeline))} deploy ${pipelineStackIdentifier} --require-approval=never --verbose`,
            ],
            rolePolicyStatements: [
                // allow the self-mutating project permissions to assume the bootstrap Action role
                new iam.PolicyStatement({
                    actions: ['sts:AssumeRole'],
                    resources: [`arn:*:iam::${core_1.Stack.of(this.pipeline).account}:role/*`],
                    conditions: {
                        'ForAnyValue:StringEquals': {
                            'iam:ResourceTag/aws-cdk:bootstrap-role': ['image-publishing', 'file-publishing', 'deploy'],
                        },
                    },
                }),
                new iam.PolicyStatement({
                    actions: ['cloudformation:DescribeStacks'],
                    resources: ['*'],
                }),
                // S3 checks for the presence of the ListBucket permission
                new iam.PolicyStatement({
                    actions: ['s3:ListBucket'],
                    resources: ['*'],
                }),
            ],
        });
        // Different on purpose -- id needed for backwards compatible LogicalID
        return codebuild_factory_1.CodeBuildFactory.fromCodeBuildStep('SelfMutation', step, {
            additionalConstructLevel: false,
            scope: construct_internals_1.obtainScope(this, 'UpdatePipeline'),
        });
    }
    publishAssetsAction(node, assets) {
        var _b;
        const installSuffix = this.cliVersion ? `@${this.cliVersion}` : '';
        const commands = assets.map(asset => {
            const relativeAssetManifestPath = path.relative(this.myCxAsmRoot, asset.assetManifestPath);
            return `cdk-assets --path "${fs_1.toPosixPath(relativeAssetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        });
        const assetType = assets[0].assetType;
        if (assets.some(a => a.assetType !== assetType)) {
            throw new Error('All assets in a single publishing step must be of the same type');
        }
        const publishingRoles = this.assetPublishingRoles[assetType] = ((_b = this.assetPublishingRoles[assetType]) !== null && _b !== void 0 ? _b : new Set());
        for (const asset of assets) {
            if (asset.assetPublishingRoleArn) {
                publishingRoles.add(asset.assetPublishingRoleArn);
            }
        }
        const role = this.obtainAssetCodeBuildRole(assets[0].assetType);
        // The base commands that need to be run
        const script = new codebuild_step_1.CodeBuildStep(node.id, {
            commands,
            installCommands: [
                `npm install -g cdk-assets${installSuffix}`,
            ],
            input: this._cloudAssemblyFileSet,
            buildEnvironment: {
                privileged: assets.some(asset => asset.assetType === blueprint_1.AssetType.DOCKER_IMAGE),
            },
            role,
        });
        // Customizations that are not accessible to regular users
        return codebuild_factory_1.CodeBuildFactory.fromCodeBuildStep(node.id, script, {
            additionalConstructLevel: false,
            // If we use a single publisher, pass buildspec via file otherwise it'll
            // grow too big.
            passBuildSpecViaCloudAssembly: this.singlePublisherPerAssetType,
            scope: this.assetsScope,
        });
    }
    nodeTypeFromNode(node) {
        var _b, _c, _d, _e;
        if (((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) === 'step') {
            return !!((_c = node.data) === null || _c === void 0 ? void 0 : _c.isBuildStep) ? CodeBuildProjectType.SYNTH : CodeBuildProjectType.STEP;
        }
        if (((_d = node.data) === null || _d === void 0 ? void 0 : _d.type) === 'publish-assets') {
            return CodeBuildProjectType.ASSETS;
        }
        if (((_e = node.data) === null || _e === void 0 ? void 0 : _e.type) === 'self-update') {
            return CodeBuildProjectType.SELF_MUTATE;
        }
        return undefined;
    }
    codeBuildDefaultsFor(nodeType) {
        const defaultOptions = {
            buildEnvironment: {
                buildImage: cb.LinuxBuildImage.STANDARD_5_0,
                computeType: cb.ComputeType.SMALL,
            },
        };
        const typeBasedCustomizations = {
            [CodeBuildProjectType.SYNTH]: this.props.dockerEnabledForSynth
                ? codebuild_factory_1.mergeCodeBuildOptions(this.props.synthCodeBuildDefaults, { buildEnvironment: { privileged: true } })
                : this.props.synthCodeBuildDefaults,
            [CodeBuildProjectType.ASSETS]: this.props.assetPublishingCodeBuildDefaults,
            [CodeBuildProjectType.SELF_MUTATE]: this.props.dockerEnabledForSelfMutation
                ? codebuild_factory_1.mergeCodeBuildOptions(this.props.selfMutationCodeBuildDefaults, { buildEnvironment: { privileged: true } })
                : this.props.selfMutationCodeBuildDefaults,
            [CodeBuildProjectType.STEP]: {},
        };
        const dockerUsage = dockerUsageFromCodeBuild(nodeType);
        const dockerCommands = dockerUsage !== undefined
            ? docker_credentials_1.dockerCredentialsInstallCommands(dockerUsage, this.dockerCredentials, 'both')
            : [];
        const typeBasedDockerCommands = dockerCommands.length > 0 ? {
            partialBuildSpec: cb.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: dockerCommands,
                    },
                },
            }),
        } : {};
        return codebuild_factory_1.mergeCodeBuildOptions(defaultOptions, this.props.codeBuildDefaults, typeBasedCustomizations[nodeType], typeBasedDockerCommands);
    }
    roleFromPlaceholderArn(scope, region, account, arn) {
        if (!arn) {
            return undefined;
        }
        // Use placeholder arn as construct ID.
        const id = arn;
        // https://github.com/aws/aws-cdk/issues/7255
        let existingRole = constructs_1.Node.of(scope).tryFindChild(`ImmutableRole${id}`);
        if (existingRole) {
            return existingRole;
        }
        // For when #7255 is fixed.
        existingRole = constructs_1.Node.of(scope).tryFindChild(id);
        if (existingRole) {
            return existingRole;
        }
        const arnToImport = cxapi.EnvironmentPlaceholders.replace(arn, {
            region: region !== null && region !== void 0 ? region : core_1.Aws.REGION,
            accountId: account !== null && account !== void 0 ? account : core_1.Aws.ACCOUNT_ID,
            partition: core_1.Aws.PARTITION,
        });
        return iam.Role.fromRoleArn(scope, id, arnToImport, { mutable: false, addGrantsToResources: true });
    }
    /**
     * Non-template config files for CodePipeline actions
     *
     * Currently only supports tags.
     */
    writeTemplateConfiguration(stack) {
        if (Object.keys(stack.tags).length === 0) {
            return undefined;
        }
        const absConfigPath = `${stack.absoluteTemplatePath}.config.json`;
        const relativeConfigPath = path.relative(this.myCxAsmRoot, absConfigPath);
        // Write the template configuration file (for parameters into CreateChangeSet call that
        // cannot be configured any other way). They must come from a file, and there's unfortunately
        // no better hook to write this file (`construct.onSynthesize()` would have been the prime candidate
        // but that is being deprecated--and DeployCdkStackAction isn't even a construct).
        template_configuration_1.writeTemplateConfiguration(absConfigPath, {
            Tags: javascript_1.noUndefined(stack.tags),
        });
        return relativeConfigPath;
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    obtainAssetCodeBuildRole(assetType) {
        if (this.assetCodeBuildRoles[assetType]) {
            return this.assetCodeBuildRoles[assetType];
        }
        const stack = core_1.Stack.of(this);
        const rolePrefix = assetType === blueprint_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new asset_singleton_role_1.AssetSingletonRole(this.assetsScope, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(stack.account)),
        });
        // Publishing role access
        // The ARNs include raw AWS pseudo parameters (e.g., ${AWS::Partition}), which need to be substituted.
        // Lazy-evaluated so all asset publishing roles are included.
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: core_1.Lazy.list({ produce: () => { var _b; return Array.from((_b = this.assetPublishingRoles[assetType]) !== null && _b !== void 0 ? _b : []).map(arn => core_1.Fn.sub(arn)); } }),
        }));
        // Grant pull access for any ECR registries and secrets that exist
        if (assetType === blueprint_1.AssetType.DOCKER_IMAGE) {
            this.dockerCredentials.forEach(reg => reg.grantRead(assetRole, docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING));
        }
        this.assetCodeBuildRoles[assetType] = assetRole;
        return assetRole;
    }
}
exports.CodePipeline = CodePipeline;
_a = JSII_RTTI_SYMBOL_1;
CodePipeline[_a] = { fqn: "monocdk.pipelines.CodePipeline", version: "1.149.0" };
function dockerUsageFromCodeBuild(cbt) {
    switch (cbt) {
        case CodeBuildProjectType.ASSETS: return docker_credentials_1.DockerCredentialUsage.ASSET_PUBLISHING;
        case CodeBuildProjectType.SELF_MUTATE: return docker_credentials_1.DockerCredentialUsage.SELF_UPDATE;
        case CodeBuildProjectType.SYNTH: return docker_credentials_1.DockerCredentialUsage.SYNTH;
        case CodeBuildProjectType.STEP: return undefined;
    }
}
var CodeBuildProjectType;
(function (CodeBuildProjectType) {
    CodeBuildProjectType["SYNTH"] = "SYNTH";
    CodeBuildProjectType["ASSETS"] = "ASSETS";
    CodeBuildProjectType["SELF_MUTATE"] = "SELF_MUTATE";
    CodeBuildProjectType["STEP"] = "STEP";
})(CodeBuildProjectType || (CodeBuildProjectType = {}));
/**
 * Take a set of tranches and split them up into groups so
 * that no set of tranches has more than n items total
 */
function chunkTranches(n, xss) {
    const ret = [];
    while (xss.length > 0) {
        const tranches = [];
        let count = 0;
        while (xss.length > 0) {
            const xs = xss[0];
            const spaceRemaining = n - count;
            if (xs.length <= spaceRemaining) {
                tranches.push(xs);
                count += xs.length;
                xss.shift();
            }
            else {
                tranches.push(xs.splice(0, spaceRemaining));
                count = n;
                break;
            }
        }
        ret.push(tranches);
    }
    return ret;
}
function isCodePipelineActionFactory(x) {
    return !!x.produceAction;
}
//# sourceMappingURL=data:application/json;base64,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