"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineSource = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cp_actions = require("../../../aws-codepipeline-actions");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const blueprint_1 = require("../blueprint");
const outputs_1 = require("./private/outputs");
/**
 * Factory for CodePipeline source steps
 *
 * This class contains a number of factory methods for the different types
 * of sources that CodePipeline supports.
 */
class CodePipelineSource extends blueprint_1.Step {
    constructor() {
        super(...arguments);
        // tells `PipelineGraph` to hoist a "Source" step
        this.isSource = true;
    }
    /**
     * Returns a GitHub source, using OAuth tokens to authenticate with
     * GitHub and a separate webhook to detect changes. This is no longer
     * the recommended method. Please consider using `connection()`
     * instead.
     *
     * Pass in the owner and repository in a single string, like this:
     *
     * ```ts
     * pipelines.CodePipelineSource.gitHub('owner/repo', 'main');
     * ```
     *
     * Authentication will be done by a secret called `github-token` in AWS
     * Secrets Manager (unless specified otherwise).
     *
     * The token should have these permissions:
     *
     * * **repo** - to read the repository
     * * **admin:repo_hook** - if you plan to use webhooks (true by default)
     */
    static gitHub(repoString, branch, props = {}) {
        jsiiDeprecationWarnings.monocdk_pipelines_GitHubSourceOptions(props);
        return new GitHubSource(repoString, branch, props);
    }
    /**
     * Returns an S3 source.
     *
     * @param bucket The bucket where the source code is located.
     * @param props The options, which include the key that identifies the source code file and
     * and how the pipeline should be triggered.
     *
     * @example
     * declare const bucket: s3.Bucket;
     * pipelines.CodePipelineSource.s3(bucket, 'path/to/file.zip');
     */
    static s3(bucket, objectKey, props = {}) {
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        jsiiDeprecationWarnings.monocdk_pipelines_S3SourceOptions(props);
        return new S3Source(bucket, objectKey, props);
    }
    /**
     * Returns an ECR source.
     *
     * @param repository The repository that will be watched for changes.
     * @param props The options, which include the image tag to be checked for changes.
     *
     * @example
     * declare const repository: ecr.IRepository;
     * pipelines.CodePipelineSource.ecr(repository, {
     *   imageTag: 'latest',
     * });
     */
    static ecr(repository, props = {}) {
        jsiiDeprecationWarnings.monocdk_aws_ecr_IRepository(repository);
        jsiiDeprecationWarnings.monocdk_pipelines_ECRSourceOptions(props);
        return new ECRSource(repository, props);
    }
    /**
     * Returns a CodeStar connection source. A CodeStar connection allows AWS CodePipeline to
     * access external resources, such as repositories in GitHub, GitHub Enterprise or
     * BitBucket.
     *
     * To use this method, you first need to create a CodeStar connection
     * using the AWS console. In the process, you may have to sign in to the external provider
     * -- GitHub, for example -- to authorize AWS to read and modify your repository.
     * Once you have done this, copy the connection ARN and use it to create the source.
     *
     * Example:
     *
     * ```ts
     * pipelines.CodePipelineSource.connection('owner/repo', 'main', {
     *   connectionArn: 'arn:aws:codestar-connections:us-east-1:222222222222:connection/7d2469ff-514a-4e4f-9003-5ca4a43cdc41', // Created using the AWS console
     * });
     * ```
     *
     * @param repoString A string that encodes owner and repository separated by a slash (e.g. 'owner/repo').
     * @param branch The branch to use.
     * @param props The source properties, including the connection ARN.
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/welcome-connections.html
     */
    static connection(repoString, branch, props) {
        jsiiDeprecationWarnings.monocdk_pipelines_ConnectionSourceOptions(props);
        return new CodeStarConnectionSource(repoString, branch, props);
    }
    /**
     * Returns a CodeCommit source.
     *
     * @param repository The CodeCommit repository.
     * @param branch The branch to use.
     * @param props The source properties.
     *
     * @example
     * declare const repository: codecommit.IRepository;
     * pipelines.CodePipelineSource.codeCommit(repository, 'main');
     */
    static codeCommit(repository, branch, props = {}) {
        jsiiDeprecationWarnings.monocdk_aws_codecommit_IRepository(repository);
        jsiiDeprecationWarnings.monocdk_pipelines_CodeCommitSourceOptions(props);
        return new CodeCommitSource(repository, branch, props);
    }
    produceAction(stage, options) {
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
        jsiiDeprecationWarnings.monocdk_pipelines_ProduceActionOptions(options);
        const output = options.artifacts.toCodePipeline(this.primaryOutput);
        const action = this.getAction(output, options.actionName, options.runOrder, options.variablesNamespace);
        stage.addAction(action);
        return { runOrdersConsumed: 1 };
    }
    /**
     * Return an attribute of the current source revision
     *
     * These values can be passed into the environment variables of pipeline steps,
     * so your steps can access information about the source revision.
     *
     * What attributes are available depends on the type of source. These attributes
     * are supported:
     *
     * - GitHub, CodeCommit, and CodeStarSourceConnection
     *   - `AuthorDate`
     *   - `BranchName`
     *   - `CommitId`
     *   - `CommitMessage`
     * - GitHub, CodeCommit and ECR
     *   - `RepositoryName`
     * - GitHub and CodeCommit
     *   - `CommitterDate`
     * - GitHub
     *   - `CommitUrl`
     * - CodeStarSourceConnection
     *   - `FullRepositoryName`
     * - S3
     *   - `ETag`
     *   - `VersionId`
     * - ECR
     *   - `ImageDigest`
     *   - `ImageTag`
     *   - `ImageURI`
     *   - `RegistryId`
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-variables.html#reference-variables-list
     * @example
     * // Access the CommitId of a GitHub source in the synth
     * const source = pipelines.CodePipelineSource.gitHub('owner/repo', 'main');
     *
     * const pipeline = new pipelines.CodePipeline(scope, 'MyPipeline', {
     *   synth: new pipelines.ShellStep('Synth', {
     *     input: source,
     *     commands: [],
     *     env: {
     *       'COMMIT_ID': source.sourceAttribute('CommitId'),
     *     }
     *   })
     * });
     */
    sourceAttribute(name) {
        return outputs_1.makeCodePipelineOutput(this, name);
    }
}
exports.CodePipelineSource = CodePipelineSource;
_a = JSII_RTTI_SYMBOL_1;
CodePipelineSource[_a] = { fqn: "monocdk.pipelines.CodePipelineSource", version: "1.149.0" };
/**
 * Extend CodePipelineSource so we can type-test in the CodePipelineEngine.
 */
class GitHubSource extends CodePipelineSource {
    constructor(repoString, branch, props) {
        var _b;
        super(repoString);
        this.branch = branch;
        this.props = props;
        const parts = repoString.split('/');
        if (core_1.Token.isUnresolved(repoString) || parts.length !== 2) {
            throw new Error(`GitHub repository name should be a resolved string like '<owner>/<repo>', got '${repoString}'`);
        }
        this.owner = parts[0];
        this.repo = parts[1];
        this.authentication = (_b = props.authentication) !== null && _b !== void 0 ? _b : core_1.SecretValue.secretsManager('github-token');
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, actionName, runOrder, variablesNamespace) {
        return new cp_actions.GitHubSourceAction({
            output,
            actionName,
            runOrder,
            oauthToken: this.authentication,
            owner: this.owner,
            repo: this.repo,
            branch: this.branch,
            trigger: this.props.trigger,
            variablesNamespace,
        });
    }
}
class S3Source extends CodePipelineSource {
    constructor(bucket, objectKey, props) {
        super(constructs_1.Node.of(bucket).addr);
        this.bucket = bucket;
        this.objectKey = objectKey;
        this.props = props;
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, _actionName, runOrder, variablesNamespace) {
        var _b;
        return new cp_actions.S3SourceAction({
            output,
            // Bucket names are guaranteed to conform to ActionName restrictions
            actionName: (_b = this.props.actionName) !== null && _b !== void 0 ? _b : this.bucket.bucketName,
            runOrder,
            bucketKey: this.objectKey,
            trigger: this.props.trigger,
            bucket: this.bucket,
            variablesNamespace,
        });
    }
}
class ECRSource extends CodePipelineSource {
    constructor(repository, props) {
        super(constructs_1.Node.of(repository).addr);
        this.repository = repository;
        this.props = props;
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, _actionName, runOrder, variablesNamespace) {
        var _b;
        // RepositoryName can contain '/' that is not a valid ActionName character, use '_' instead
        const formattedRepositoryName = core_1.Fn.join('_', core_1.Fn.split('/', this.repository.repositoryName));
        return new cp_actions.EcrSourceAction({
            output,
            actionName: (_b = this.props.actionName) !== null && _b !== void 0 ? _b : formattedRepositoryName,
            runOrder,
            repository: this.repository,
            imageTag: this.props.imageTag,
            variablesNamespace,
        });
    }
}
class CodeStarConnectionSource extends CodePipelineSource {
    constructor(repoString, branch, props) {
        super(repoString);
        this.branch = branch;
        this.props = props;
        const parts = repoString.split('/');
        if (core_1.Token.isUnresolved(repoString) || parts.length !== 2) {
            throw new Error(`CodeStar repository name should be a resolved string like '<owner>/<repo>', got '${repoString}'`);
        }
        this.owner = parts[0];
        this.repo = parts[1];
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, actionName, runOrder, variablesNamespace) {
        return new cp_actions.CodeStarConnectionsSourceAction({
            output,
            actionName,
            runOrder,
            connectionArn: this.props.connectionArn,
            owner: this.owner,
            repo: this.repo,
            branch: this.branch,
            codeBuildCloneOutput: this.props.codeBuildCloneOutput,
            triggerOnPush: this.props.triggerOnPush,
            variablesNamespace,
        });
    }
}
class CodeCommitSource extends CodePipelineSource {
    constructor(repository, branch, props) {
        super(core_1.Token.isUnresolved(repository.repositoryName)
            ? constructs_1.Node.of(repository).addr
            : repository.repositoryName);
        this.repository = repository;
        this.branch = branch;
        this.props = props;
        this.configurePrimaryOutput(new blueprint_1.FileSet('Source', this));
    }
    getAction(output, _actionName, runOrder, variablesNamespace) {
        return new cp_actions.CodeCommitSourceAction({
            output,
            // Guaranteed to be okay as action name
            actionName: this.repository.repositoryName,
            runOrder,
            branch: this.branch,
            trigger: this.props.trigger,
            repository: this.repository,
            eventRole: this.props.eventRole,
            codeBuildCloneOutput: this.props.codeBuildCloneOutput,
            variablesNamespace,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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