"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineFileSet = exports.ArtifactMap = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cp = require("../../../aws-codepipeline");
const blueprint_1 = require("../blueprint");
const helpers_internal_1 = require("../helpers-internal");
/**
 * Translate FileSets to CodePipeline Artifacts
 */
class ArtifactMap {
    constructor() {
        this.artifacts = new Map();
        this.usedNames = new Set();
    }
    /**
     * Return the matching CodePipeline artifact for a FileSet
     */
    toCodePipeline(x) {
        jsiiDeprecationWarnings.monocdk_pipelines_FileSet(x);
        if (x instanceof CodePipelineFileSet) {
            return x._artifact;
        }
        let ret = this.artifacts.get(x);
        if (!ret) {
            // They all need a name
            const artifactName = this.makeUniqueName(`${x.producer.id}.${x.id}`);
            this.usedNames.add(artifactName);
            this.artifacts.set(x, ret = new cp.Artifact(artifactName));
        }
        return ret;
    }
    makeUniqueName(baseName) {
        let i = 1;
        baseName = sanitizeArtifactName(baseName);
        let name = baseName;
        while (this.usedNames.has(name)) {
            name = `${baseName}${++i}`;
        }
        return name;
    }
}
exports.ArtifactMap = ArtifactMap;
_a = JSII_RTTI_SYMBOL_1;
ArtifactMap[_a] = { fqn: "monocdk.pipelines.ArtifactMap", version: "1.149.0" };
/**
 * Sanitize a string to be a valid artifact name
 *
 * This must comport to both the rules of artifacts in CodePipeline, as well
 * as the names of Source Identifiers in CodeBuild.
 *
 * Artifact Name limits aren't documented.
 *
 * Source Identifier limits are documented here:
 * https://docs.aws.amazon.com/codebuild/latest/APIReference/API_ProjectSource.html#CodeBuild-Type-ProjectSource-sourceIdentifier
 */
function sanitizeArtifactName(x) {
    let sani = x.replace(/[^A-Za-z0-9_]/g, '_'); // Charset requirement is imposed by CodeBuild
    const maxLength = 100; // Max length of 100 is imposed by CodePipeline library
    if (sani.length > maxLength) {
        const fingerprint = crypto.createHash('sha256').update(sani).digest('hex').substr(0, 8);
        sani = sani.substr(0, maxLength - fingerprint.length) + fingerprint;
    }
    return sani;
}
/**
 * A FileSet created from a CodePipeline artifact
 *
 * You only need to use this if you want to add CDK Pipeline stages
 * add the end of an existing CodePipeline, which should be very rare.
 */
class CodePipelineFileSet extends blueprint_1.FileSet {
    constructor(artifact) {
        var _c;
        super((_c = artifact.artifactName) !== null && _c !== void 0 ? _c : 'Imported', helpers_internal_1.PipelineGraph.NO_STEP);
        this._artifact = artifact;
    }
    /**
     * Turn a CodePipeline Artifact into a FileSet
     */
    static fromArtifact(artifact) {
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_Artifact(artifact);
        return new CodePipelineFileSet(artifact);
    }
}
exports.CodePipelineFileSet = CodePipelineFileSet;
_b = JSII_RTTI_SYMBOL_1;
CodePipelineFileSet[_b] = { fqn: "monocdk.pipelines.CodePipelineFileSet", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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